#!/usr/bin/env python3
# Definir _ como fallback si load_translations() falla
def _(text):
    return text  # Fallback que devuelve el texto original si no hay traducciones

try:
    load_translations()
except NameError:
    pass  # Ignorar si load_translations() no está disponible

from qt.core import Qt, QDialog, QVBoxLayout, QPushButton, QMessageBox, QLabel, QListWidget, QFileDialog
import os
import ast
import reprlib  # Para serializar diccionarios de manera segura

from calibre.utils.config import config_dir
from calibre_plugins.theme_switcher.config import prefs
from calibre.gui2 import info_dialog, error_dialog
from PyQt5.QtWidgets import QApplication

class Dialog(QDialog):

    def __init__(self, gui, icon, do_user_config):
        QDialog.__init__(self, gui)
        self.gui = gui
        self.do_user_config = do_user_config

        self.palettes_dir = os.path.join(config_dir, prefs['palettes_subdir'])
        print(f"Palettes directory: {self.palettes_dir}")  # Depuración
        if not os.path.exists(self.palettes_dir):
            os.makedirs(self.palettes_dir)
            print(f"Created palettes directory: {self.palettes_dir}")

        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.label = QLabel(_('Theme Switcher'))
        self.l.addWidget(self.label)

        self.list_widget = QListWidget(self)
        self.l.addWidget(self.list_widget)
        self.refresh_list()

        self.apply_button = QPushButton(_('Apply Selected Theme'), self)
        self.apply_button.clicked.connect(self.apply_theme)
        self.l.addWidget(self.apply_button)

        self.export_button = QPushButton(_('Export Selected Palette'), self)
        self.export_button.clicked.connect(self.export_palette)
        self.l.addWidget(self.export_button)

        self.import_button = QPushButton(_('Import New Palette'), self)
        self.import_button.clicked.connect(self.import_palette)
        self.l.addWidget(self.import_button)

        self.conf_button = QPushButton(_('Configure this plugin'), self)
        self.conf_button.clicked.connect(self.config)
        self.l.addWidget(self.conf_button)

        self.about_button = QPushButton(_('About'), self)
        self.about_button.clicked.connect(self.about)
        self.l.addWidget(self.about_button)

        self.setWindowTitle(_('Theme Switcher'))
        self.setWindowIcon(icon)

        self.resize(self.sizeHint())

    def refresh_list(self):
        self.list_widget.clear()
        print(f"Refreshing list from: {self.palettes_dir}")  # Depuración
        try:
            for file in os.listdir(self.palettes_dir):
                if file.lower().endswith('.calibre-palette'):
                    print(f"Found file: {file}")  # Depuración
                    self.list_widget.addItem(file)
            if not self.list_widget.count():
                print("No .calibre-palette files found in the directory.")
                self.list_widget.addItem("No palettes found")
        except Exception as e:
            print(f"Error reading directory: {e}")
            self.list_widget.addItem(f"Error: {str(e)}")

    def about(self):
        text = get_resources('about.txt')
        QMessageBox.about(self, _('About Theme Switcher'),
                          text.decode('utf-8') if text else _('No about text available'))

    def apply_theme(self):
        selected = self.list_widget.currentItem()
        if not selected:
            return error_dialog(self, _('No theme selected'), _('Please select a theme first.'), show=True)
        
        palette_path = os.path.join(self.palettes_dir, selected.text())
        try:
            with open(palette_path, 'r', encoding='utf-8') as f:
                content = f.read()
                print(f"Raw content of {selected.text()}: {content}")  # Depuración
                palette_data = ast.literal_eval(content)
            
            from calibre.utils.config_base import prefs as base_prefs
            current_theme = base_prefs['ui_theme'] if 'ui_theme' in base_prefs else 'light'
            if current_theme not in ['dark', 'light']:
                current_theme = 'light'  # Fallback
            
            if isinstance(palette_data, dict) and 'dark' in palette_data and 'light' in palette_data:
                if current_theme in palette_data and 'palette' in palette_data[current_theme]:
                    palette = palette_data[current_theme]['palette']
                else:
                    raise ValueError(_(f"Invalid theme '{current_theme}' in {selected.text()}. Missing 'palette' key."))
            else:
                palette = palette_data  # Asumir paleta simple
            
            if not isinstance(palette, dict):
                raise ValueError(_(f"Invalid palette format in {selected.text()}. Must be a dictionary."))
            
            print(f"Extracted palette for {current_theme}: {palette}")  # Depuración adicional
            base_prefs['global_color_palette'] = palette
            
            app = QApplication.instance()
            print(f"Applying palette with set_palette_from_spec: {palette}")  # Depuración
            app.set_palette_from_spec(palette)
            print(f"Palette applied successfully")  # Depuración
            
            print(f"Requesting restart")  # Depuración
            self.gui.quit(restart=True)
            info_dialog(self, _('Theme Applied'), _('The theme has been applied. Restart Calibre if changes are not visible.'), show=True)
        except SyntaxError as e:
            error_dialog(self, _('Error Applying Theme'), _(f"Invalid syntax in {selected.text()} on line {e.lineno}: {str(e)}. Please check the file format."), show=True)
        except ValueError as e:
            error_dialog(self, _('Error Applying Theme'), str(e), show=True)
        except Exception as e:
            error_dialog(self, _('Error Applying Theme'), str(e), show=True)

    def export_palette(self):
        selected = self.list_widget.currentItem()
        if not selected:
            return error_dialog(self, _('No palette selected'), _('Please select a palette to export.'), show=True)
        
        palette_path = os.path.join(self.palettes_dir, selected.text())
        try:
            with open(palette_path, 'r', encoding='utf-8') as f:
                palette_data = ast.literal_eval(f.read())
            
            export_path, _ = QFileDialog.getSaveFileName(self, _('Export Palette'), '', _('Calibre Palette (*.calibre-palette)'))
            if export_path:
                with open(export_path, 'w', encoding='utf-8') as f:
                    f.write(repr(palette_data))
                info_dialog(self, _('Palette Exported'), _('The palette has been exported successfully.'), show=True)
        except Exception as e:
            error_dialog(self, _('Error Exporting Palette'), str(e), show=True)

    def import_palette(self):
        file_path, _ = QFileDialog.getOpenFileName(self, _('Select Palette File'), '', _('Calibre Palette (*.calibre-palette)'))
        if file_path:
            dest_path = os.path.join(self.palettes_dir, os.path.basename(file_path))
            with open(file_path, 'rb') as src, open(dest_path, 'wb') as dest:
                dest.write(src.read())
            self.refresh_list()
            info_dialog(self, _('Palette Imported'), _('The palette has been imported successfully.'), show=True)

    def config(self):
        self.do_user_config(parent=self)
