#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys, os, json, socket
from subprocess import Popen, PIPE
from urllib.request import urlopen, urlretrieve

# code provided by DiapDealer
def is_connected():
    ''' tests Internet connectivity '''
    try:
        sock = socket.create_connection(('8.8.8.8', 53), 1)
        sock.close()
        return True
    except:
        pass

# encode/escape text to make it xml safe (from navprocessor.py)
def xmlencode(data):
    ''' escapes text to make it xml safe '''
    if data is None:
        return ''
    newdata = data
    newdata = newdata.replace('&', '&amp;')
    newdata = newdata.replace('<', '&lt;')
    newdata = newdata.replace('>', '&gt;')
    newdata = newdata.replace('"', '&quot;')
    return newdata


def jar_wrapper(*args):
    ''' simple wrapper for executing css-validator.jar '''
    process = Popen(args, stdout=PIPE, stderr=PIPE, shell=False)
    ret = process.communicate()
    return ret
    
def run(bk):
    ''' main routine'''

    # get CSSValidator plugin path
    plugin_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name)
    css_val_path = os.path.join(plugin_path, 'css-validator.jar')

    # (re-)download css-validator.jar, if it's missing
    if not os.path.exists(css_val_path):
        if is_connected():
            response = urlopen('https://api.github.com/repos/w3c/css-validator/releases').read().decode('utf-8')
            parsed_json = json.loads(response)
            latest_version = parsed_json[0]['tag_name']
            browser_download_url = parsed_json[0]['assets'][0]['browser_download_url']
            print('css-validator.jar not found, downloading latest version ({}) from Github'.format(latest_version))
            urlretrieve(browser_download_url, css_val_path)
            if not os.path.exists(css_val_path):
                print('css-validator.jar download failed. Please reinstall the plugin.')
                return -1
        else:
            print('css-validator.jar not found. Please reinstall the plugin.')
            return -1

    #------------------------
    # get preference
    #------------------------
    prefs = bk.getPrefs()

    # write initial JSON file
    if prefs == {}:
        prefs['java_path'] = 'java'
        prefs['css_epub2_validation_spec'] = 'css21'
        prefs['css_epub3_validation_spec'] = 'css3'
        prefs['warning'] = '2'
        prefs['ignore_message'] = []
        bk.savePrefs(prefs)

    #----------------
    # get values
    #----------------
    debug = prefs.get('debug', False)

    # alternative Java executable path
    java_path = 'java'
    if 'java_path' in prefs:
        java_path = prefs.get('java_path', 'java').replace('\\\\', '/').replace('\\', '/')

    # get the epub2 profile value
    epubversion = '2.0'
    if bk.launcher_version() >= 20160102:
        epubversion = bk.epub_version()
    
    if epubversion.startswith('2'):
        profile = 'css21'
        if 'css_epub2_validation_spec' in prefs:
            profile = prefs.get('css_epub2_validation_spec', profile)
        else:
            prefs['css_epub2_validation_spec'] = profile
            bk.savePrefs(prefs)

    # get the epub3 profile value
    else:
        profile = 'css3'
        if 'css_epub3_validation_spec' in prefs:
            profile = prefs.get('css_epub3_validation_spec', profile)
        else:
            prefs['css_epub3_validation_spec'] = profile
            bk.savePrefs(prefs)

    # get the lang value
    lang = 'en'
    if 'lang' in prefs:
        lang = prefs.get('lang', lang)
    else:
        # set validator message language to GUI language, if available
        try:
            sigil_lang = bk.sigil_ui_lang[0:2]
            if sigil_lang in ['bg', 'cs', 'de', 'el', 'en', 'es', 'fa',
                'fr', 'hi', 'hu', 'it', 'ja', 'ko', 'nl', 'pl', 'pt', 
                'ro', 'ru', 'sv', 'uk', 'zh']:
                lang = sigil_lang
        except:
            pass
        prefs['lang'] = lang
        bk.savePrefs(prefs)
    
    # get & validate warning level (the value MUST BE A STRING)
    warning = '2'
    if 'warning' in prefs:
        warning = str(prefs.get('warning', warning))
    else:
        prefs['warning'] = warning
        bk.savePrefs(prefs)

    # ignore certain warnings (values must be internal error types, 
    # e.g. ["vendor-extension"])
    ignore_message = prefs.get('ignore_message', [])

    #----------------------------------------------------------------------
    # define css-validator.jar command line parameters
    #----------------------------------------------------------------------
    args = [java_path, '-Dfile.encoding=UTF8', '-jar', css_val_path, '-profile', 
        profile, '-warning', warning, '-output', 'json', 
        '-lang', lang, '']

    #--------------------
    # get stylesheets
    #--------------------
    for manifest_id, OPF_href in bk.css_iter():

        # get base name
        base_name = os.path.basename(OPF_href)

        # get book path
        if bk.launcher_version() >= 20190927:
            book_path = bk.id_to_bookpath(manifest_id)
        else:
            book_path = os.path.join('OEBPS/Styles/', base_name)

        # read CSS file contents
        css_contents = bk.readfile(manifest_id)

        # split file for context information
        css_lines = css_contents.splitlines()

        # display status message
        print('Checking {}... please wait.'.format(base_name))
        if debug: bk.add_result('info', book_path, None, 'Profile: {} Warning: {} Lang: {}'\
        .format(profile, warning, lang))

        #---------------------------
        # run css validator
        css_file_path = os.path.join(bk._w.ebook_root, book_path)
        args[-1] = 'file:///' + css_file_path
        if debug: print('\nargs:', *args)
        try:
            result = jar_wrapper(*args)
            stdout = result[0].decode("utf-8")
            stderr = result[1].decode("utf-8")
            if debug: print(stdout, stderr)

        except Exception as ex:
            if type(ex).__name__ == 'FileNotFoundError':
                if debug: print('Java Runtime not found!!!')
                bk.add_result('error', book_path, None, 'Java Runtime not found!!!')
            else:
                if debug: print('An exception of type {0} occurred.'.format(type(ex).__name__))
                bk.add_result('error', book_path, None, \
                'Python error: an exception of type {0} occurred.'.format(type(ex).__name__))
            return 0

        # check for StackOverflowError/Exception errors
        if stderr.find('java.lang.StackOverflowError') != -1 or stderr.find('Exception in thread') != -1:
            if debug: print('CSS Valdidator Java error.\n', stderr)
            bk.add_result('error', book_path, None, \
            'CSS Valdidator Java overflow or exception error.')
            return 0

        #---------------------------------
        # parse JSON output
        #---------------------------------
        
        # older Sigil versions don't require the full file path
        if bk.launcher_version() < 20190927:
            book_path = base_name

        # load json contents
        try:
            parsed_json = json.loads(stdout)
        except Exception as ex:
            # sometimes the validator generates invalid JSON ouput,
            # if the warn level is '-1'
            print('An exception of type {0} occurred.'.format(type(ex).__name__))
            bk.add_result('error', book_path, None, \
            'Python error: an exception of type {0} occurred.'.format(type(ex).__name__))
            if debug: print(stdout)
            return 0

        # parse the json file
        if 'cssvalidation' in parsed_json:
            valid = parsed_json['cssvalidation']['validity']
            csslevel = parsed_json['cssvalidation']['csslevel']
            error_count = parsed_json['cssvalidation']['result']['errorcount']
            warning_count = parsed_json['cssvalidation']['result']['warningcount']

            # check validity 
            if not valid:

                #------------------
                # parse errors 
                #------------------
                if error_count > 0:
                    for err in parsed_json['cssvalidation']['errors']:
                        line = context = line_context = None 
                        context = line_context = ''
                        if debug: print('\n', err)
                        context = xmlencode(err['context'])
                        _type = err['type']
                        if _type not in ignore_message:
                            message = _type + ' : ' + xmlencode(err['message'])
                            if 'line' in err: # some error messages don't have line numbers
                                line = err['line']
                                try:
                                    line_context = xmlencode(css_lines[line -1])
                                except:
                                    # necessary for older Sigil versions
                                    line_context = xmlencode(css_lines[line -1].decode('utf-8'))
                            if context: message = context + ': ' + message
                            if line_context: message = message + ' : ' + line_context
                            if debug: print('\n', book_path, line, message.replace(':  :', ':'))
                            bk.add_result('error', book_path, line, message.replace(':  :', ':'))

                #-----------------
                # parse warnings
                #-----------------
                if warning_count > 0 and warning != '-1':
                    for warn in parsed_json['cssvalidation']['warnings']:
                        line = context = line_context = None
                        _type = warn['type']
                        if _type not in ignore_message:
                            message = _type + ' : ' + xmlencode(warn['message'])
                            if debug: print(warn)
                            if 'line' in warn: # some error messages don't have line numbers
                                line = warn['line']
                                try:
                                    line_context = xmlencode(css_lines[line -1])
                                except:
                                    # necessary for older Sigil versions
                                    line_context = xmlencode(css_lines[line -1].decode('utf-8'))
                            if line_context: message = message + ' : ' + line_context
                            if debug: print(book_path, line, message.replace(':  :', ':'))
                            bk.add_result('warning', book_path, line, message.replace(':  :', ':'))
            else:
                if debug: bk.add_result('info', book_path, None, 'Valid; CSS level (detected): ' + csslevel)

        else:
            bk.add_result('error', None, None, 'JSON output couldn\'t be parsed!')
            return 0

    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
