#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import os
import re
import zipfile

def extract_version_from_content(content):
    """Extract version from file content using various patterns"""
    if not content:
        return None

    # First check for version reference
    version_ref_match = re.search(r'^\s*version\s*=\s*([A-Za-z_][A-Za-z0-9_]*)\s*$', content, re.MULTILINE)
    if version_ref_match:
        ref_name = version_ref_match.group(1)
        # Try tuple format first
        ref_pattern = rf'^\s*{ref_name}\s*=\s*\(([0-9]+(?:\s*,\s*[0-9]+)*)\)\s*(?:#[^\n]*)?$'
        ref_match = re.search(ref_pattern, content, re.MULTILINE)
        if ref_match:
            try:
                version_nums = [int(x.strip()) for x in ref_match.group(1).split(',')]
                return tuple(version_nums)
            except ValueError:
                pass
        # Try string format
        ref_str_pattern = rf'^\s*{ref_name}\s*=\s*["\']([0-9]+(?:\.[0-9]+)*)["\']'
        ref_str_match = re.search(ref_str_pattern, content, re.MULTILINE)
        if ref_str_match:
            try:
                return tuple(int(x) for x in ref_str_match.group(1).split('.'))
            except ValueError:
                pass

    # List of version patterns to try, in order of preference
    version_patterns = [
        # Standard version tuple with optional comment (module level or class level)
        r'^\s*version\s*=\s*\(([0-9]+(?:\s*,\s*[0-9]+)*)\)\s*(?:#[^\n]*)?$',
        # Class-level version with more whitespace (for plugins like Action Chains)
        r'^\s+version\s+=\s+\(([0-9]+(?:\s*,\s*[0-9]+)*)\)\s*(?:#[^\n]*)?$',
        # Plugin version variations with optional comment
        r'^\s*(?:PLUGIN(?:VER)?|AUDIOBOOKREADER)_VERSION(?:_TUPLE)?\s*=\s*\(([0-9]+(?:\s*,\s*[0-9]+)*)\)\s*(?:#[^\n]*)?$',
        # Plain PLUGINVER
        r'^\s*PLUGINVER\s*=\s*\(([0-9]+(?:\s*,\s*[0-9]+)*)\)\s*(?:#[^\n]*)?$',
        # Version string format
        r'^\s*__version__\s*=\s*["\']([0-9]+(?:\.[0-9]+)*)["\']'
    ]

    for pattern in version_patterns:
        match = re.search(pattern, content, re.MULTILINE)
        if match:
            version_str = match.group(1)
            try:
                if ',' in version_str:
                    version_nums = [int(x.strip()) for x in version_str.split(',')]
                    return tuple(version_nums)
                else:
                    return tuple(int(x) for x in version_str.split('.'))
            except ValueError:
                continue

    return None

def extract_plugin_version(plugin_path):
    """Extract version from a plugin (zip file or directory)"""
    if not plugin_path or not os.path.exists(plugin_path):
        return None

    # Files to check for version info, in order of preference
    files_to_check = [
        '__init__.py',
        '__version__.py',
        'common.py',
        'reader.py',
        'device.py',
        'main.py',
        'plugin.py'
    ]

    try:
        if zipfile.is_zipfile(plugin_path):
            with zipfile.ZipFile(plugin_path, 'r') as zf:
                # Try each potential file
                for check_file in files_to_check:
                    # Look for the file in root or subdirectories
                    matches = [f for f in zf.namelist() if f.endswith('/' + check_file) or f == check_file]
                    if matches:
                        # Prioritize root-level files over subdirectory files
                        root_matches = [f for f in matches if f == check_file]
                        if root_matches:
                            matches = root_matches + [f for f in matches if f != check_file]

                        for match in matches:
                            try:
                                with zf.open(match) as f:
                                    content = f.read().decode('utf-8', 'ignore')
                                    version = extract_version_from_content(content)
                                    if version:
                                        return version
                            except Exception:
                                continue
        else:
            # Handle extracted plugin folder
            for check_file in files_to_check:
                file_path = os.path.join(plugin_path, check_file)
                if os.path.exists(file_path):
                    with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
                        content = f.read()
                        version = extract_version_from_content(content)
                        if version:
                            return version

    except Exception as e:
        try:
            prints(f"Error extracting version from {plugin_path}: {str(e)}")
        except Exception:
            pass

    return None

def normalize_version(version):
    """Convert version tuple/string to normalized tuple, handling edge cases"""
    if not version:
        return None

    if isinstance(version, str):
        # Handle version strings (e.g. "1.32.0" or "1.32")
        try:
            nums = [int(x) for x in version.split('.')]
            while nums and nums[-1] == 0:  # Remove trailing zeros
                nums.pop()
            return tuple(nums)
        except (ValueError, AttributeError):
            return None

    if isinstance(version, (tuple, list)):
        # Handle version tuples, removing trailing zeros
        nums = list(version)
        while nums and nums[-1] == 0:
            nums.pop()
        return tuple(nums)

    return None

def version_tuple_to_string(version_tuple):
    """Convert a version tuple to string format"""
    if not version_tuple:
        return None
    normalized = normalize_version(version_tuple)
    return '.'.join(map(str, normalized)) if normalized else None

def compare_versions(ver1, ver2):
    """Compare two version tuples, returns -1 if ver1 < ver2, 0 if equal, 1 if ver1 > ver2"""
    if ver1 is None and ver2 is None:
        return 0
    if ver1 is None:
        return -1
    if ver2 is None:
        return 1

    # Pad shorter version with zeros
    max_length = max(len(ver1), len(ver2))
    ver1_padded = ver1 + (0,) * (max_length - len(ver1))
    ver2_padded = ver2 + (0,) * (max_length - len(ver2))

    if ver1_padded < ver2_padded:
        return -1
    elif ver1_padded > ver2_padded:
        return 1
    return 0