#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import json
import os
from qt.core import (
    QAbstractItemView, QApplication, QDialog,
    QHBoxLayout, QHeaderView, QLabel, QLineEdit,
    QMenu, QPushButton, QTableWidget, QTableWidgetItem,
    QVBoxLayout, QWidget, Qt, QToolButton, QIcon
)

from calibre.constants import config_dir, isportable, get_portable_base, DEBUG, iswindows
from calibre.utils.config import JSONConfig
from calibre import prints
from calibre.utils.localization import _


class SettingsTab(QWidget):
    """Class to handle the Calibre Settings tab functionality"""

    def __init__(self, parent):
        QWidget.__init__(self, parent)
        self.parent = parent
        self.gui = parent.gui

        # Setup the UI
        self.setup_ui()

        # Populate the table
        self.populate_settings_table()


    def setup_ui(self):
        """Setup the tab UI elements"""
        layout = QVBoxLayout(self)

        # Add filter bar
        filter_layout = QHBoxLayout()
        filter_label = QLabel(_('Filter:'))
        self.settings_filter_edit = QLineEdit()
        self.settings_filter_edit.setPlaceholderText(_('Search settings...'))
        self.settings_filter_edit.textChanged.connect(self.apply_settings_filter)
        filter_layout.addWidget(filter_label)

        # Add clear button visually inside the QLineEdit (like plugins tab)
        self.settings_filter_clear_btn = QToolButton(self.settings_filter_edit)
        self.settings_filter_clear_btn.setIcon(QIcon.ic('clear_left.png'))
        self.settings_filter_clear_btn.setToolTip(_('Clear filter'))
        self.settings_filter_clear_btn.setCursor(Qt.CursorShape.ArrowCursor)
        self.settings_filter_clear_btn.setVisible(False)
        self.settings_filter_clear_btn.setStyleSheet('QToolButton { border: none; padding: 0px; }')
        self.settings_filter_clear_btn.setFixedSize(18, 18)
        self.settings_filter_clear_btn.clicked.connect(self.settings_filter_edit.clear)
        self.settings_filter_edit.textChanged.connect(
            lambda text: self.settings_filter_clear_btn.setVisible(bool(text)))
        from PyQt5.QtWidgets import QStyle  # or PyQt6.QtWidgets if using PyQt6
        frame = self.settings_filter_edit.style().pixelMetric(QStyle.PixelMetric.PM_DefaultFrameWidth)
        self.settings_filter_edit.setTextMargins(0, 0, self.settings_filter_clear_btn.width() + frame + 2, 0)
        def move_clear_btn():
            self.settings_filter_clear_btn.move(
                self.settings_filter_edit.rect().right() - self.settings_filter_clear_btn.width() - frame,
                (self.settings_filter_edit.rect().height() - self.settings_filter_clear_btn.height()) // 2)
        self.settings_filter_edit.resizeEvent = lambda event: move_clear_btn()
        move_clear_btn()
        # Prevent Enter from triggering default buttons; handle return explicitly
        try:
            self.settings_filter_edit.returnPressed.connect(lambda: self.apply_settings_filter(self.settings_filter_edit.text()))
        except Exception:
            pass
        filter_layout.addWidget(self.settings_filter_edit)

        # Add column visibility toggle
        self.settings_column_button = QPushButton(_('&Manage Columns'))
        self.settings_column_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        self.settings_column_button.setToolTip(_('Show/hide columns'))
        self.settings_column_button.setAutoDefault(False)
        self.settings_column_button.setDefault(False)
        self.settings_column_button.clicked.connect(self.show_settings_column_menu)
        filter_layout.addWidget(self.settings_column_button)

        # Add export buttons with save icon and tooltips (consistent with other tabs)
        settings_csv_export_button = QPushButton(_('Export &CSV'))
        settings_csv_export_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        settings_csv_export_button.setIcon(QIcon.ic('save.png'))
        settings_csv_export_button.setToolTip(_('Export visible settings to CSV'))
        settings_csv_export_button.setAutoDefault(False)
        settings_csv_export_button.setDefault(False)
        settings_csv_export_button.clicked.connect(lambda: self.parent.export_settings('csv'))
        settings_xlsx_export_button = QPushButton(_('Export &XLSX'))
        settings_xlsx_export_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        settings_xlsx_export_button.setIcon(QIcon.ic('save.png'))
        settings_xlsx_export_button.setToolTip(_('Export visible settings to XLSX'))
        settings_xlsx_export_button.setAutoDefault(False)
        settings_xlsx_export_button.setDefault(False)
        settings_xlsx_export_button.clicked.connect(lambda: self.parent.export_settings('xlsx'))
        # Restore / Reset Columns action and button (mirror Plugins tab)
        try:
            # Try to import QAction/QKeySequence from qt.core (works with qt.py wrapper)
            from qt.core import QAction, QKeySequence
        except Exception:
            # Fallback to PyQt imports if needed
            try:
                from PyQt5.QtWidgets import QAction
                from PyQt5.QtGui import QKeySequence
            except Exception:
                QAction = None
                QKeySequence = None

        if QAction is not None:
            try:
                restore_action = QAction(_('&Reset Columns'), self)
                try:
                    if QKeySequence is not None:
                        restore_action.setShortcut(QKeySequence(_('Alt+R')))
                except Exception:
                    pass
                restore_action.setShortcutContext(Qt.WidgetShortcut)
                restore_action.setToolTip(_('Restore default column visibility and ordering'))
                try:
                    # Register on parent dialog so the shortcut is active immediately
                    self.parent.addAction(restore_action)
                except Exception:
                    pass
                try:
                    self.restore_action = restore_action
                except Exception:
                    pass
                restore_action.triggered.connect(self.restore_default_columns)
            except Exception:
                restore_action = None

        restore_defaults_button = QPushButton('\u00A0' + _('&Reset Columns'))
        restore_defaults_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        try:
            restore_defaults_button.setIcon(QIcon.ic('view-refresh.png'))
        except Exception:
            pass
        restore_defaults_button.setToolTip(_('Restore default column visibility and ordering'))
        restore_defaults_button.setAutoDefault(False)
        restore_defaults_button.setDefault(False)
        restore_defaults_button.clicked.connect(self.restore_default_columns)

        filter_layout.addWidget(settings_csv_export_button)
        filter_layout.addWidget(settings_xlsx_export_button)
        filter_layout.addWidget(restore_defaults_button)

        layout.addLayout(filter_layout)

        # Create settings table
        self.settings_table = QTableWidget()
        self.setup_settings_table()
        layout.addWidget(self.settings_table)

    def setup_settings_table(self):
        """Setup the settings table structure"""
        self.settings_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.settings_table.setAlternatingRowColors(True)
        self.settings_table.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)
        self.settings_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

        # Setup columns
        self.settings_table.setColumnCount(5)  # Updated column count to include UI Location
        headers = [_('Category'), _('Setting'), _('Value'), _('UI Location'), _('Source')]  # Added UI Location column
        self.settings_table.setHorizontalHeaderLabels(headers)

        header = self.settings_table.horizontalHeader()
        # Set minimum and initial width for UI Location column (column 3)
        header.setSectionResizeMode(3, QHeaderView.ResizeMode.Interactive)
        # Set global minimum for all columns to 50
        header.setMinimumSectionSize(50)
        # Set per-column minimum for UI Location (if supported)
        try:
            header.setMinimumSectionSize(50)  # fallback for all
            header.setSectionResizeMode(3, QHeaderView.ResizeMode.Interactive)
            header.setMinimumSectionSize(270)  # enforce 270px minimum for UI Location
        except Exception:
            pass
        # Set initial width for UI Location
        header.resizeSection(3, 270)
        self.settings_table.setColumnWidth(3, 270)

        # Set minimum width for Category column (column 0)
        header.setSectionResizeMode(0, QHeaderView.ResizeMode.Interactive)
        header.setMinimumSectionSize(150)
        self.settings_table.setColumnWidth(0, 150)

        # Enable sorting
        self.settings_table.setSortingEnabled(True)

        # Enable context menu
        self.settings_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.settings_table.customContextMenuRequested.connect(self.show_settings_context_menu)

        # Enable double-click to open main Preferences dialog
        self.settings_table.cellDoubleClicked.connect(self.open_main_preferences_dialog)

        # Enable column reordering
        header = self.settings_table.horizontalHeader()
        header.setSectionsMovable(True)
        header.sectionMoved.connect(lambda: self.parent.save_column_state('settings'))

    def _is_portable_install(self):
        """Check if this is a portable installation"""
        return isportable

    def _get_library_paths(self):
        """Get all configured library paths"""
        from calibre.gui2 import gprefs
        library_usage = gprefs.get('library_usage_stats', {})
        return sorted(library_usage.keys())

    def _get_setting_description(self, setting_name):
        """Get the description/tooltip for a setting"""
        descriptions = {
            'Default Output Format': _('The default format to convert e-books to'),
            'Input Format Order': _('Order of formats to prefer when getting formats from input books'),
            'Filename Pattern': _('Pattern to match filenames when adding books from the filesystem'),
            'Case Sensitive': _('Make searching case sensitive when searching your library'),
            'Limit Search Columns': _('Restrict searching to specific columns to improve search performance'),
            'Limited Search Columns': _('List of columns to restrict searching to when Limit Search Columns is enabled'),
            'Read File Metadata': _('Read metadata from e-book files when adding books'),
            'Mark New Books': _('Mark newly added books until calibre is restarted'),
            'New Book Tags': _('Tags to automatically apply to newly added books'),
            'Numeric Collation': _('Sort numbers based on numerical value instead of as text'),
            'Network Timeout': _('Timeout in seconds for network operations'),
            'Worker Process Priority': _('Priority level for background processes'),
            'Manage Device Metadata': _('How to handle metadata when sending books to devices'),
            'Swap Author Names': _('Swap author first and last names when reading metadata'),
            'User Categories': _('User-created categories for organizing books'),
            'GUI Layout': _('Layout of the main calibre window (wide/narrow)'),
            'Show System Tray Icon': _('Show calibre icon in system tray/notification area'),
            'Disable Tray Notifications': _('Turn off popup notifications in system tray'),
            'Highlight Search Matches': _('Highlight all search matches in book list'),
            'Separate Cover Flow': _('Show cover flow in separate window'),
            'Worker Limit': _('Maximum number of simultaneous conversion/download jobs'),
            'Database Path': _('Location of the calibre library database file'),
            'Auto Download Cover': _('Automatically download covers from internet'),
            'Get Social Metadata': _('Download social metadata like tags and ratings'),
            'Search As You Type': _('Update search results while typing'),
            'Enforce CPU Limit': _('Limit CPU cores used for background jobs'),
            'Grid View Visible': _('Show books in a grid of covers'),
            'Sort Tags By': _('How to sort tags in Tag browser'),
            'Format Priorities': _('Order of format preference for device'),
            'Save Template': _('Template for filenames when saving to device'),
            'Use Author Sort': _('Use author sort value on device'),
            'Use Subdirectories': _('Create subdirectories on device'),
            'Quick View Open at Shutdown': _('Keep Quick View window open on restart'),
            'Color Theme': _('Light/Dark/System theme setting'),
            'Cover Flow Queue Length': _('Number of covers to show in flow'),
            'Tag Browser Hidden Categories': _('Categories hidden from Tag browser'),
            'Plugin Search History': _('Recent plugin searches'),
            'Oldest News': _('Days to keep downloaded news'),
            'Toolbar Actions': _('Buttons shown on toolbar'),
            'Auto Convert Added Books': _('When enabled, automatically converts added books to the preferred output format configured in Calibre Settings'),
            'Prefer Metadata from': _('Source to prefer when metadata exists in both file and Calibre database'),
            'Save Cover in Book File': _('Whether to save cover image inside e-book files during conversion'),
            'Auto-Merge Metadata': _('Automatically merge metadata from different sources'),
            'Download Tags from Download': _('Download tags when downloading metadata'),
            'Preferred Author Sort': _('How to format author names in the author sort field'),
            'Author Name Disambiguation': _('How to handle authors with the same name'),
            'Ignore Duplicates on Import': _('Skip duplicates when importing multiple books'),
            'Title Case Conversion': _('Rules for converting titles to title case'),
            'Keep Aspect Ratio on Resize': _('Maintain aspect ratio when resizing covers'),
            'ISBN Provider': _('Service to use for ISBN lookups'),
            'Default Cover Image Size': _('Default size for generated cover images'),
            'Preserve Title Case': _('Keep original title case when downloading metadata')
        }
        return descriptions.get(setting_name, '')

    def apply_settings_filter(self, text):
        """Filter the settings table based on search text"""
        filter_text = text.lower()
        for row in range(self.settings_table.rowCount()):
            show = False
            for col in range(self.settings_table.columnCount()):
                item = self.settings_table.item(row, col)
                if item and filter_text in item.text().lower():
                    show = True
                    break
            self.settings_table.setRowHidden(row, not show)

        # Update statistics
        self.parent.update_statistics()

    def show_settings_column_menu(self):
        """Show menu to toggle settings columns visibility"""
        headers = [self.settings_table.horizontalHeaderItem(i).text()
                  for i in range(self.settings_table.columnCount())]

        menu = QMenu(self)
        for i, header in enumerate(headers):
            action = menu.addAction(header)
            action.setCheckable(True)
            action.setChecked(not self.settings_table.isColumnHidden(i))
            action.triggered.connect(lambda checked, col=i:
                self.parent.toggle_column_to_state(self.settings_table, col, checked))

        menu.popup(self.settings_column_button.mapToGlobal(
            self.settings_column_button.rect().bottomLeft()))

    def restore_default_columns(self):
        """Restore default column visibility, ordering, and widths for Settings tab"""
        from calibre.gui2 import question_dialog, info_dialog

        if not question_dialog(self, _('Restore Default Columns'),
                               _('This will restore the default column visibility and ordering for the Settings tab. Continue?')):
            return

        # Determine desired default columns by logical index based on current header labels
        # Default: show all columns but ensure UI Location (index 3) is visible and reasonably wide
        header = self.settings_table.horizontalHeader()
        count = self.settings_table.columnCount()
        desired_columns = list(range(count))

        # Reset visibility: show only desired columns
        for col in range(count):
            try:
                self.settings_table.setColumnHidden(col, col not in desired_columns)
            except Exception:
                pass

        # Reset ordering to natural order (0..n-1)
        try:
            for visual_index, logical_index in enumerate(desired_columns):
                current_visual = header.visualIndex(logical_index)
                if current_visual != visual_index:
                    header.moveSection(current_visual, visual_index)
        except Exception:
            pass

        # Restore default widths for sensible columns
        try:
            default_widths = {
                0: 150,  # Category
                1: 230,  # Setting
                2: 280,  # Value
                3: 280,  # UI Location
                4: 120   # Source
            }
            for col, w in default_widths.items():
                if col < self.settings_table.columnCount():
                    try:
                        self.settings_table.setColumnWidth(col, w)
                    except Exception:
                        pass
        except Exception:
            pass

        # Ensure header does not use any stretch mode and explicitly set
        # interactive resize on each column so widths we just applied stick.
        try:
            header.setStretchLastSection(False)
        except Exception:
            pass
        try:
            for col in range(count):
                try:
                    header.setSectionResizeMode(col, QHeaderView.ResizeMode.Interactive)
                except Exception:
                    try:
                        # Fallback for PyQt5 naming
                        header.setSectionResizeMode(col, QHeaderView.Interactive)
                    except Exception:
                        pass
                # Re-apply the width explicitly to avoid automatic stretching
                w = default_widths.get(col, 120) if 'default_widths' in locals() else 120
                try:
                    header.resizeSection(col, w)
                except Exception:
                    pass
                try:
                    self.settings_table.setColumnWidth(col, w)
                except Exception:
                    pass
        except Exception:
            pass

        # Clear saved preferences keys so defaults persist
        from calibre.gui2 import gprefs
        # Remove any saved hidden columns, column order, and column widths so
        # previously persisted (and possibly oversized) widths do not reapply.
        gprefs.pop('calibre_config_reports_settings_hidden_columns', None)
        gprefs.pop('calibre_config_reports_settings_column_order', None)
        gprefs.pop('calibre_config_reports_settings_column_order_labels', None)
        gprefs.pop('calibre_config_reports_settings_column_widths', None)
        # Persist gprefs change immediately to avoid race where another restore
        # routine reads the old values before we save defaults.
        try:
            gprefs.save()
        except Exception:
            pass

        # Save the default state immediately so it persists
        try:
            self.parent.save_column_state('settings')
        except Exception:
            pass

        info_dialog(self, _('Columns Restored'),
                    _('Default column visibility, ordering, and widths have been restored.'), show=True)

    def show_settings_context_menu(self, pos):
        """Show context menu for settings table"""
        index = self.settings_table.indexAt(pos)
        menu = QMenu(self)

        if index.isValid():
            # Copy row data action
            copy_action = menu.addAction(self.parent.copy_icon, _('Copy row data'))
            copy_action.triggered.connect(lambda: self.copy_settings_row(index.row()))

            # Add separator
            menu.addSeparator()

            # Add column visibility submenu
            columns_menu = menu.addMenu(_('Show/Hide Columns'))
            for i in range(self.settings_table.columnCount()):
                col_name = self.settings_table.horizontalHeaderItem(i).text()
                action = columns_menu.addAction(col_name)
                action.setCheckable(True)
                action.setChecked(not self.settings_table.isColumnHidden(i))
                action.triggered.connect(lambda checked, col=i:
                    self.parent.toggle_column_to_state(self.settings_table, col, checked))

        menu.popup(self.settings_table.viewport().mapToGlobal(pos))

    def copy_settings_row(self, row):
        """Copy all data for a settings row to clipboard"""
        data = []
        for col in range(self.settings_table.columnCount()):
            header = self.settings_table.horizontalHeaderItem(col).text()
            cell_text = self.settings_table.item(row, col).text()
            data.append(f"{header}: {cell_text}")

        QApplication.clipboard().setText('\n'.join(data))
        self.parent.show_copy_feedback(row, 0, self.settings_table)

    def open_main_preferences_dialog(self, row, col):
        """Open the main Calibre Preferences dialog on double-click in settings table"""
        self.gui.iactions['Preferences'].do_config()

    def populate_settings_table(self):
        """Populate the settings table with configuration data"""
        import os
        import json
        import csv
        from calibre.utils.localization import _ as _t  # Import with a different name to avoid conflicts
        global _
        _ = _t  # Make it available in the local scope

        settings_data = []        # Load config files from the actual Calibre config directory
        metadata_sources_dir = os.path.join(config_dir, 'metadata_sources')

        def load_json(path):
            try:
                with open(path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    # Ensure we return a dictionary even if the file contained something else
                    return data if isinstance(data, dict) else {}
            except Exception:
                return {}
        global_py = load_json(os.path.join(config_dir, 'global.py.json'))
        gui_json = load_json(os.path.join(config_dir, 'gui.json'))
        gui_py = load_json(os.path.join(config_dir, 'gui.py.json'))
        tweaks_json = load_json(os.path.join(config_dir, 'tweaks.json'))
        meta_global = load_json(os.path.join(metadata_sources_dir, 'global.json'))

        # Try loading device settings if available
        device_settings_path = os.path.join(config_dir, 'device_drivers_SMART_DEVICE_APP.py.json')
        device_settings = load_json(device_settings_path) if os.path.exists(device_settings_path) else {}

        # Load TTS settings if available
        tts_path = os.path.join(config_dir, 'tts.json')
        tts_settings = load_json(tts_path) if os.path.exists(tts_path) else {}
          # Helper for value lookup
        def get_val(dct, key, default=''):
            if not isinstance(dct, dict):
                return default
            v = dct.get(key, default)
            if isinstance(v, list):
                return ', '.join(map(str, v))
            if isinstance(v, dict):
                return json.dumps(v)
            return str(v)        # Add installation type and libraries info
        is_portable = isportable
        settings_data.append((_('Installation'), _('Type'), _('Portable') if is_portable else _('Standard'), _('N/A (internal setting)'), 'global.py.json'))
        # Get all library paths and usage stats
        from calibre.gui2 import gprefs
        library_usage = gprefs.get('library_usage_stats', {})
        for i, path in enumerate(sorted(library_usage.keys()), 1):
            times_used = library_usage[path]
            # Use OS-native path separator for display
            native_path = path.replace('/', os.sep).replace('\\', os.sep)
            value = f"{native_path}; Usage count: {times_used}"
            # UI location for Libraries: Main toolbar ⇒ Choose Library button
            settings_data.append((_('Libraries'), _('Library %d') % i, value, _('Main toolbar ⇒ Choose Library button'), 'gui.json'))

        # UI locations mapping - IMPORTANT! This ensures the UI locations are shown in the table
        ui_locations = {
            'Language': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Draw a grid in the book list': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Show row numbers in book list': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Show tooltips in book list': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Show splash screen': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Enable system tray icon (needs restart)': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Disable notifications on job completion': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Disable animations': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Show layout buttons': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Status bar buttons'),
            'Show actions button': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Status bar buttons'),
            'Show text under icons': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Toolbar'),
            'Icon size': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Toolbar'),
            'Use two lines for the text under the icons (needs restart)': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Toolbar'),
            'Round the corners of covers': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Cover corner radius': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Interface font': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Extra spacing to add between rows in the book list (can be negative)': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Allow using drag and drop to merge books': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Show cover': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'Show cover size': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'Use Roman numerals for series': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'Show comments heading': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'Link and note icon size': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'Text styling': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'Grid View Visible': _('Preferences ⇒ Look & feel ⇒ Cover grid'),
            'Highlight Search Matches': _('Preferences ⇒ Searching ⇒ General'),
            'Search As You Type': _('Preferences ⇒ Searching ⇒ General'),
            'Show Highlight Toggle Button': _('Preferences ⇒ Searching ⇒ General'),
            'Auto Convert Added Books': _('Preferences ⇒ Adding books ⇒ Adding actions'),
            'Auto-Merge Added Books': _('Preferences ⇒ Adding books ⇒ Adding actions'),
            'Case Sensitive': _('Preferences ⇒ Searching ⇒ General'),
            'Use Author Sort': _('Preferences ⇒ Sending to device'),
            'Use Subdirectories': _('Preferences ⇒ Sending to device'),
            'Format Priorities': _('Preferences ⇒ Sending to device'),
            'author_sort_copy_method': _('Preferences ⇒ Tweaks'),
            'Book list split': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Cover corner radius': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Show layout buttons': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Status bar buttons'),
            'Show actions button': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Status bar buttons'),
            'Show text under icons': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Toolbar'),
            'Toolbar icon size': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Toolbar'),
            'Wrap toolbar text': _('Preferences ⇒ Look & feel ⇒ Main interface ⇒ Toolbar'),
            'BD overlay cover size': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'Comments heading position': _('Preferences ⇒ Look & feel ⇒ Book details'),
            'booklist_grid': _('Preferences ⇒ Look & feel ⇒ Main interface'),
            'Tag browser search box visible': _('Preferences ⇒ Look & feel ⇒ Tag browser'),
            'Tag browser hide empty categories': _('Preferences ⇒ Look & feel ⇒ Tag browser'),
            'Tag browser old look': _('Preferences ⇒ Look & feel ⇒ Tag browser'),
            'Tag browser show counts': _('Preferences ⇒ Look & feel ⇒ Tag browser'),
            'Tags browser partition method': _('Preferences ⇒ Look & feel ⇒ Tag browser'),
            'Quick View Open at Shutdown': _('Preferences ⇒ Look & feel ⇒ Quick view'),
            'Quick View Respects VLS': _('Preferences ⇒ Look & feel ⇒ Quick view'),
        }

        # Add settings from global.py.json
        settings_data.extend([
            (_('System'), _('Language'), get_val(global_py, 'language', 'en'), _('Preferences ⇒ Look & feel ⇒ Main interface'), 'global.py.json'),
            (_('Library Management'), _('Primary Library Path'), get_val(global_py, 'library_path', ''), _('Main toolbar ⇒ Choose Library button'), 'global.py.json'),
            (_('Library Management'), _('Read File Metadata'), get_val(global_py, 'read_file_metadata', 'True'), _('Preferences ⇒ Adding books ⇒ Adding actions'), 'global.py.json'),
            (_('Library Management'), _('Check Duplicates on Copy/Move'), get_val(global_py, 'check_for_dupes_on_ctl', 'False'), _('Preferences ⇒ Library ⇒ Splitting'), 'global.py.json'),
            (_('Library Management'), _('Mark New Books'), get_val(global_py, 'mark_new_books', 'False'), _('Preferences ⇒ Adding books ⇒ Adding actions'), 'global.py.json'),
            (_('Library Management'), _('New Book Tags'), get_val(global_py, 'new_book_tags', '[]'), _('Preferences ⇒ Adding books ⇒ Adding actions'), 'global.py.json'),
            (_('Library Management'), _('Worker Process Priority'), get_val(global_py, 'worker_process_priority', 'normal'), _('Preferences ⇒ Advanced ⇒ Tweaks'), 'global.py.json'),
            (_('Library Management'), _('Auto-Merge Added Books'), get_val(global_py, 'add_formats_to_existing', 'False'), _('Preferences ⇒ Adding books ⇒ Adding actions'), 'global.py.json'),
            (_('File Formats'), _('Default Output Format'), get_val(global_py, 'output_format', 'epub'), _('Preferences ⇒ Behavior ⇒ Default actions'), 'global.py.json'),
            (_('File Formats'), _('Input Format Order'),
                (lambda v: (
                    (', '.join(json.loads(v)[:19]) + ',+more') if isinstance(v, str) and v.startswith('[') and len(json.loads(v)) > 19
                    else (', '.join(json.loads(v))) if isinstance(v, str) and v.startswith('[')
                    else (', '.join(v[:19]) + ',+more') if isinstance(v, list) and len(v) > 19
                    else (', '.join(v)) if isinstance(v, list)
                    else v
                ))(get_val(global_py, 'input_format_order', '')),
                _('Preferences ⇒ Behavior'), 'global.py.json'),
            (_('File Patterns'), _('Filename Pattern'), get_val(global_py, 'filename_pattern', ''), _('Preferences ⇒ Adding books'), 'global.py.json'),
            (_('Network'), _('Network Timeout'), get_val(global_py, 'network_timeout', '5') + ' ' + _('seconds'), _('Preferences ⇒ Sharing ⇒ Sharing over the net'), 'global.py.json'),
            (_('Search'), _('Case Sensitive'), get_val(global_py, 'case_sensitive', 'False'), _('Preferences ⇒ Searching'), 'global.py.json'),
            (_('Search'), _('Use Primary Find in Search'), get_val(global_py, 'use_primary_find_in_search', 'True'), _('Preferences ⇒ Searching'), 'global.py.json'),
            (_('Search'), _('Limit Search Columns'), get_val(global_py, 'limit_search_columns', 'False'), _('Preferences ⇒ Searching'), 'global.py.json'),
            (_('Search'), _('Limited Search Columns'), get_val(global_py, 'limit_search_columns_to', '[]'), _('Preferences ⇒ Searching'), 'global.py.json'),
            (_('Search'), _('Numeric Collation'), get_val(global_py, 'numeric_collation', 'False'), _('Preferences ⇒ Searching'), 'global.py.json'),
            (_('Author Handling'), _('Swap Author Names'), get_val(global_py, 'swap_author_names', 'False'), _('Preferences ⇒ Advanced ⇒ Tweaks'), 'global.py.json'),
            (_('Categories'), _('User Categories'), str(len(global_py.get('user_categories', {}) if isinstance(global_py.get('user_categories'), dict) else {})), _('Preferences ⇒ Look & feel ⇒ Tag browser'), 'global.py.json'),
            (_('Device'), _('Manage Device Metadata'), get_val(global_py, 'manage_device_metadata', 'manual'), _('Preferences ⇒ Sending to device'), 'global.py.json')])        # Add TTS settings if available
        if tts_settings and isinstance(tts_settings, dict) and 'engines' in tts_settings and isinstance(tts_settings['engines'], dict):
            for engine, engine_data in tts_settings['engines'].items():
                if 'default_voice' in engine_data:
                    settings_data.append(
                        (_('TTS'), _('Default Voice (%s)') % engine, engine_data['default_voice'], _('Preferences ⇒ Text-to-Speech'), 'tts.json')
                    )

        # Add grouped window geometry information if available
        geom = gui_json.get('geometry-of-calibre_main_window_geometry', {}) if isinstance(gui_json, dict) else {}
        if geom and isinstance(geom, dict):
            try:
                pos = geom.get('pos', [0, 0]) if isinstance(geom, dict) else [0, 0]
                size = geom.get('size', [0, 0]) if isinstance(geom, dict) else [0, 0]
                screen = geom.get('screen', {}) if isinstance(geom, dict) else {}

                # Compose a single value string for all geometry info
                geom_parts = []
                if isinstance(pos, list) and len(pos) >= 2:
                    geom_parts.append(f"Pos: X={pos[0]}, Y={pos[1]}")
                if isinstance(size, list) and len(size) >= 2:
                    geom_parts.append(f"Size: {size[0]}x{size[1]}")
                if isinstance(screen, dict) and 'size' in screen and isinstance(screen['size'], list) and len(screen['size']) >= 2:
                    geom_parts.append(f"Screen: {screen['size'][0]}x{screen['size'][1]}")
                if isinstance(screen, dict) and 'name' in screen:
                    geom_parts.append(f"Monitor: {screen['name']}")
                if geom_parts:
                    settings_data.append((
                        _('Window Geometry'),
                        _('Main Window Geometry'),
                        '; '.join(geom_parts),
                        'gui.json',
                        'gui.json'
                    ))
            except Exception:
                pass

        # Add interface settings from gui.json and gui.py.json
        # First from gui.json
        settings_data.extend([
            (_('Interface'), _('Draw a grid in the book list'), get_val(gui_json, 'booklist_grid', 'False'), ui_locations.get('Draw a grid in the book list', ''), 'gui.json'),
            (_('Interface'), _('Show row numbers in book list'), get_val(gui_json, 'row_numbers_in_book_list', 'True'), ui_locations.get('Show row numbers in book list', ''), 'gui.json'),
            (_('Interface'), _('Show tooltips in book list'), get_val(gui_json, 'book_list_tooltips', 'True'), ui_locations.get('Show tooltips in book list', ''), 'gui.json'),
            (_('Interface'), _('Show splash screen'), get_val(gui_json, 'show_splash_screen', 'True'), ui_locations.get('Show splash screen', ''), 'gui.json'),
            (_('Interface'), _('Book list split'), get_val(gui_json, 'book_list_split', 'False'), ui_locations.get('Book list split', ''), 'gui.json'),
            (_('Interface'), _('Cover corner radius'), get_val(gui_json, 'cover_corner_radius', '0'), ui_locations.get('Cover corner radius', ''), 'gui.json'),
            (_('Interface'), _('Show layout buttons'), get_val(gui_json, 'show_layout_buttons', 'True'), ui_locations.get('Show layout buttons', ''), 'gui.json'),
            (_('Interface'), _('Show actions button'), get_val(gui_json, 'show_sb_all_actions_button', 'True'), ui_locations.get('Show actions button', ''), 'gui.json'),
            (_('Interface'), _('Show text under icons'), get_val(gui_json, 'toolbar_text', 'always'), ui_locations.get('Show text under icons', ''), 'gui.json'),
            (_('Interface'), _('Toolbar icon size'), get_val(gui_json, 'toolbar_icon_size', 'medium'), ui_locations.get('Toolbar icon size', ''), 'gui.json'),
            (_('Interface'), _('Wrap toolbar text'), get_val(gui_json, 'wrap_toolbar_text', 'False'), ui_locations.get('Wrap toolbar text', ''), 'gui.json')
        ])        # Add Book Details settings from gui.json
        # Add Book Details settings from gui.json, grouping splitter states
        splitter_h = get_val(gui_json, 'book_details_splitter_horizontal_state', '')
        splitter_v = get_val(gui_json, 'book_details_splitter_vertical_state', '')
        splitter_value = f"Horizontal: {splitter_h}, Vertical: {splitter_v}" if splitter_h or splitter_v else ''
        settings_data.extend([
            (_('Book Details'), _('Show cover'), get_val(gui_json, 'bd_show_cover', 'True'), ui_locations.get('Show cover', ''), 'gui.json'),
            (_('Book Details'), _('Show cover size'), get_val(gui_json, 'bd_overlay_cover_size', 'False'), ui_locations.get('Show cover size', ''), 'gui.json'),
            (_('Book Details'), _('Comments heading position'), get_val(gui_json, 'book_details_comments_heading_pos', 'above'), ui_locations.get('Comments heading position', ''), 'gui.json'),
            (_('Book Details'), _('Link and note icon size'), get_val(gui_json, 'book_details_note_link_icon_width', '0'), ui_locations.get('Link and note icon size', ''), 'gui.json'),
        ])
        if splitter_value:
            settings_data.append((
                _('Book Details'),
                _('Book details splitter state'),
                splitter_value,
                'gui.json',
                'gui.json'
            ))

        # Add tag browser settings from gui.json
        settings_data.extend([
            (_('Tag Browser'), _('Tag browser search box visible'), get_val(gui_json, 'tag browser search box visible', 'False'), ui_locations.get('Tag browser search box visible', ''), 'gui.json'),
            (_('Tag Browser'), _('Tag browser hide empty categories'), get_val(gui_json, 'tag_browser_hide_empty_categories', 'False'), ui_locations.get('Tag browser hide empty categories', ''), 'gui.json'),
            (_('Tag Browser'), _('Tag browser old look'), get_val(gui_json, 'tag_browser_old_look', 'True'), ui_locations.get('Tag browser old look', ''), 'gui.json'),
            (_('Tag Browser'), _('Tag browser show counts'), get_val(gui_json, 'tag_browser_show_counts', 'True'), ui_locations.get('Tag browser show counts', ''), 'gui.json'),
            (_('Tag Browser'), _('Tag browser splitter horizontal state'), get_val(gui_json, 'tag_browser_splitter_horizontal_state', ''), 'gui.json', 'gui.json'),
            (_('Tag Browser'), _('Tags browser partition method'), get_val(gui_json, 'tags_browser_partition_method', 'disable'), ui_locations.get('Tags browser partition method', ''), 'gui.json')
        ])        # Then from gui.py.json
        settings_data.extend([
            (_('Interface'), _('GUI Layout'), get_val(gui_py, 'gui_layout', 'wide'), _('Preferences ⇒ Look & feel ⇒ Main interface'), 'gui.py.json'),
            (_('Interface'), _('Color Theme'), get_val(gui_py, 'color_palette', 'system'), _('Preferences ⇒ Look & feel ⇒ Main interface'), 'gui.py.json'),
            (_('Interface'), _('Grid View Visible'), get_val(gui_py, 'grid view visible', 'False'), ui_locations.get('Grid View Visible', ''), 'gui.py.json'),
            (_('Interface'), _('Cover Flow Queue Length'), get_val(gui_py, 'cover_flow_queue_length', '6'), _('Preferences ⇒ Look & feel ⇒ Cover flow'), 'gui.py.json'),
            (_('Interface'), _('Separate Cover Flow'), get_val(gui_py, 'separate_cover_flow', 'False'), _('Preferences ⇒ Look & feel ⇒ Cover flow'), 'gui.py.json'),
            (_('Interface'), _('Sort Tags By'), get_val(gui_py, 'sort_tags_by', 'name'), _('Preferences ⇒ Look & feel ⇒ Tag browser'), 'gui.py.json'),
            (_('Interface'), _('Match Tags Type'), get_val(gui_py, 'match_tags_type', 'any'), _('Preferences ⇒ Look & feel ⇒ Tag browser'), 'gui.py.json'),
            (_('Interface'), _('Use Roman Numerals for Series'), get_val(gui_py, 'use_roman_numerals_for_series_number', 'True'), ui_locations.get('Use Roman numerals for series', ''), 'gui.py.json'),
            (_('Interface'), _('Tag Browser Hidden Categories'), get_val(gui_py, 'tag_browser_hidden_categories', ''), _('Preferences ⇒ Look & feel ⇒ Tag browser'), 'gui.py.json'),
            (_('Interface'), _('Quick View Open at Shutdown'), get_val(gui_json, 'qv_open_at_shutdown', 'False'), ui_locations.get('Quick View Open at Shutdown', ''), 'gui.json'),
            (_('Interface'), _('Quick View Respects VLS'), get_val(gui_json, 'qv_respects_vls', 'False'), ui_locations.get('Quick View Respects VLS', ''), 'gui.json')
        ])        # Add performance settings
        settings_data.extend([
            (_('Performance'), _('Search As You Type'), get_val(gui_py, 'search_as_you_type', 'False'), ui_locations.get('Search As You Type', ''), 'gui.py.json'),
            (_('Performance'), _('Highlight Search Matches'), get_val(gui_py, 'highlight_search_matches', 'True'), ui_locations.get('Highlight Search Matches', ''), 'gui.py.json'),
            (_('Performance'), _('Disable Animations'), get_val(gui_py, 'disable_animations', 'False'), ui_locations.get('Disable animations', ''), 'gui.py.json'),
            (_('Performance'), _('Enforce CPU Limit'), get_val(gui_py, 'enforce_cpu_limit', 'True'), _('Preferences ⇒ Advanced ⇒ Tweaks'), 'gui.py.json'),
            (_('Performance'), _('Worker Limit'), get_val(gui_py, 'worker_limit', '6'), _('Preferences ⇒ Advanced ⇒ Tweaks'), 'gui.py.json'),
            (_('Performance'), _('Show Highlight Toggle Button'), get_val(gui_json, 'show_highlight_toggle_button', 'False'), ui_locations.get('Show Highlight Toggle Button', ''), 'gui.json')
        ])        # Add system tray settings
        settings_data.extend([
            (_('System Tray'), _('Show System Tray Icon'), get_val(gui_py, 'systray_icon', 'False'), _('Preferences ⇒ Look & feel ⇒ Main interface'), 'gui.py.json'),
            (_('System Tray'), _('Disable Tray Notification'), get_val(gui_py, 'disable_tray_notification', 'False'), _('Preferences ⇒ Look & feel ⇒ Main interface'), 'gui.py.json'),
            (_('System Tray'), _('Autolaunch Server'), get_val(gui_py, 'autolaunch_server', 'False'), _('Preferences ⇒ Sharing ⇒ Sharing over the net'), 'gui.py.json')
        ])

        # Add update settings
        settings_data.extend([
            (_('Updates'), _('New Version Notification'), get_val(gui_py, 'new_version_notification', 'False'), _('Preferences ⇒ Miscellaneous'), 'gui.py.json'),
            (_('Updates'), _('Auto Download Cover'), get_val(gui_py, 'auto_download_cover', 'False'), _('Preferences ⇒ Metadata download'), 'gui.py.json'),
            (_('Updates'), _('Get Social Metadata'), get_val(gui_py, 'get_social_metadata', 'True'), _('Preferences ⇒ Metadata download'), 'gui.py.json'),
            (_('Updates'), _('Oldest News'), get_val(gui_py, 'oldest_news', '60'), _('Preferences ⇒ Behavior ⇒ News'), 'gui.py.json')
        ])        # Add adding books settings
        settings_data.extend([
            (_('Adding Books'), _('Auto Convert Added Books'), get_val(gui_json, 'manual_add_auto_convert', 'False'), ui_locations.get('Auto Convert Added Books', ''), 'gui.json'),
            (_('Adding Books'), _('Auto Convert Same Format'), get_val(gui_json, 'auto_convert_same_fmt', 'False'), _('Preferences ⇒ Adding books ⇒ Adding actions'), 'gui.json'),
            (_('Adding Books'), _('Automatically Add All Files'), get_val(gui_json, 'auto_add_everything', 'False'), _('Preferences ⇒ Adding books'), 'gui.json'),
            (_('Adding Books'), _('Check Duplicates for Auto-Add'), get_val(gui_json, 'auto_add_check_for_duplicates', 'True'), _('Preferences ⇒ Adding books'), 'gui.json'),
            (_('Adding Books'), _('Auto-Add Auto Convert'), get_val(gui_json, 'auto_add_auto_convert', 'False'), _('Preferences ⇒ Adding books'), 'gui.json')
        ])        # Add device settings if available
        if device_settings:            # Format priorities
            if 'format_map' in device_settings and isinstance(device_settings['format_map'], dict):
                format_priorities = device_settings['format_map'].get('format_priorities', [])
                settings_data.append(
                    (_('Device'), _('Format Priorities'), ', '.join(format_priorities), ui_locations.get('Format Priorities', ''), 'device_drivers_SMART_DEVICE_APP.py.json')
                )

            # Save template
            if 'save_template' in device_settings:
                settings_data.append(
                    (_('Device'), _('Save Template'), device_settings['save_template'], _('Preferences ⇒ Sending to device'), 'device_drivers_SMART_DEVICE_APP.py.json')
                )

            # Read metadata
            if 'read_metadata' in device_settings:
                settings_data.append(
                    (_('Device'), _('Read Metadata'), str(device_settings['read_metadata']), _('Preferences ⇒ Sending to device'), 'device_drivers_SMART_DEVICE_APP.py.json')
                )

            # Use author sort
            if 'use_author_sort' in device_settings:
                settings_data.append(
                    (_('Device'), _('Use Author Sort'), str(device_settings['use_author_sort']), ui_locations.get('Use Author Sort', ''), 'device_drivers_SMART_DEVICE_APP.py.json')
                )

            # Use subdirectories
            if 'use_subdirs' in device_settings:
                settings_data.append(
                    (_('Device'), _('Use Subdirectories'), str(device_settings['use_subdirs']), ui_locations.get('Use Subdirectories', ''), 'device_drivers_SMART_DEVICE_APP.py.json')                )

            # Supported formats
            if 'format_map' in device_settings and isinstance(device_settings['format_map'], dict) and 'formats' in device_settings['format_map']:
                formats = device_settings['format_map']['formats']
                settings_data.append(
                    (_('Device Formats'), _('Supported Formats'), ', '.join(formats), _('Preferences ⇒ Sending to device ⇒ Formats'), 'device_drivers_SMART_DEVICE_APP.py.json')
                )

            # Port number if available
            if 'settings' in device_settings and 'autostart_port' in device_settings['settings']:
                settings_data.append(
                    (_('Device Connection'), _('Port Number'), str(device_settings['settings']['autostart_port']), _('Preferences ⇒ Sharing ⇒ Wireless device connection'), 'device_drivers_SMART_DEVICE_APP.py.json')
                )        # Add table columns configuration
        columns = gui_py.get('column_map', [])
        if columns:
            settings_data.append((_('Tables'), _('Columns'), ', '.join(columns), _('Preferences ⇒ Look & feel ⇒ Book list'), 'gui.py.json'))

        # Add toolbar actions configuration
        toolbar_actions = gui_json.get('action-layout-toolbar', []) if isinstance(gui_json, dict) else []
        if toolbar_actions and isinstance(toolbar_actions, list):
            settings_data.append((_('Toolbar'), _('Main Toolbar Actions'), ', '.join(filter(None, toolbar_actions)), _('Preferences ⇒ Look & feel ⇒ Toolbars'), 'gui.json'))

        child_actions = gui_json.get('action-layout-toolbar-child', []) if isinstance(gui_json, dict) else []
        if child_actions and isinstance(child_actions, list):
            settings_data.append((_('Toolbar'), _('Child Toolbar Actions'), ', '.join(filter(None, child_actions)), _('Preferences ⇒ Look & feel ⇒ Toolbars'), 'gui.json'))

        device_actions = gui_json.get('action-layout-toolbar-device', []) if isinstance(gui_json, dict) else []
        if device_actions and isinstance(device_actions, list):
            settings_data.append((_('Toolbar'), _('Device Toolbar Actions'), ', '.join(filter(None, device_actions)), _('Preferences ⇒ Look & feel ⇒ Toolbars'), 'gui.json'))

        # Add tweaks configuration
        try:
            # Categorize tweaks for better organization
            tweak_categories = {
                'author': [],
                'cover': [],
                'interface': [],
                'metadata': [],
                'network': [],                'device': [],
                'other': []
            }

            # Only iterate if tweaks_json is actually a dict
            if isinstance(tweaks_json, dict):
                for key, value in tweaks_json.items():
                    # Determine category based on key name
                    category = 'other'  # default
                    if 'author' in key.lower():
                        category = 'author'
                    elif 'cover' in key.lower():
                        category = 'cover'
                    elif any(term in key.lower() for term in ['gui', 'interface', 'display']):
                        category = 'interface'
                    elif 'metadata' in key.lower():
                        category = 'metadata'
                    elif any(term in key.lower() for term in ['network', 'server', 'smtp']):
                        category = 'network'
                    elif 'device' in key.lower():
                        category = 'device'

                    # Convert value to string representation
                    if isinstance(value, (list, dict)):
                        value_str = json.dumps(value)
                    else:
                        value_str = str(value)

                    # Add to appropriate category
                    tweak_categories[category].append((key, value_str))            # Add tweaks to settings data
            for category, tweaks in tweak_categories.items():
                for key, value in tweaks:
                    display_category = _(f"{category.capitalize()} Tweaks")
                    # Special handling for author_sort_copy_method
                    if key == 'author_sort_copy_method':
                        settings_data.append((display_category, key, tweaks_json.get('author_sort_copy_method', 'comma'), ui_locations.get('author_sort_copy_method', ''), 'tweaks.json'))
                    else:
                        settings_data.append((display_category, key, value, _('Preferences ⇒ Advanced ⇒ Tweaks'), 'tweaks.json'))
        except Exception as e:
            try:
                prints(f"Error processing tweaks: {e}")
            except Exception:
                pass
        # Add metadata sources settings
        # Only process if meta_global is a dict
        if isinstance(meta_global, dict):
            for k, v in meta_global.items():
                if isinstance(v, (list, dict)):
                    v_str = json.dumps(v)
                else:
                    v_str = str(v)
                settings_data.append((
                    _('Metadata download'), k, v_str, _('Preferences ⇒ Metadata download'), 'metadata_sources/global.json'))

        # --- Add all settings from screened_calibre_settings.csv ---
        csv_path = os.path.join(config_dir, 'screened_calibre_settings.csv')
        if os.path.exists(csv_path):
            with open(csv_path, encoding='utf-8') as csvfile:
                reader = csv.DictReader(csvfile)
                for row in reader:
                    key = row['key']
                    value = row['value']
                    source = row['source']

                    # Skip if this key is already in our settings_data
                    if any(setting[1] == key for setting in settings_data):
                        continue

                    # Try to get value from the correct config dict if possible
                    if source == 'gui.json':
                        val = get_val(gui_json, key, value)
                    elif source == 'gui.py.json':
                        val = get_val(gui_py, key, value)
                    elif source == 'global.py.json':
                        val = get_val(global_py, key, value)
                    elif source == 'tweaks.json':
                        # Special handling for author_sort_copy_method
                        if key == 'author_sort_copy_method':
                            val = tweaks_json.get('author_sort_copy_method', 'comma')
                        else:
                            val = get_val(tweaks_json, key, value)
                    elif source == 'metadata_sources\\global.json' or source == 'metadata_sources/global.json':
                        val = get_val(meta_global, key, value)
                    else:
                        val = value

                    # Try to categorize the setting
                    cat = _('(Other)')

                    # Attempt to find a suitable category based on the key
                    if any(term in key.lower() for term in ['author', 'name']):
                        cat = _('Author Handling')
                    elif any(term in key.lower() for term in ['book', 'cover', 'bd_']):
                        cat = _('Book Details')
                    elif any(term in key.lower() for term in ['tag', 'category']):
                        cat = _('Categories')
                    elif any(term in key.lower() for term in ['device', 'format']):
                        cat = _('Device')
                    elif any(term in key.lower() for term in ['interface', 'gui', 'display', 'show']):
                        cat = _('Interface')
                    elif any(term in key.lower() for term in ['search', 'find']):
                        cat = _('Search')
                    elif any(term in key.lower() for term in ['toolbar', 'action']):
                        cat = _('Toolbar')
                    elif any(term in key.lower() for term in ['worker', 'cpu', 'performance']):
                        cat = _('Performance')
                    elif any(term in key.lower() for term in ['add', 'import']):
                        cat = _('Adding Books')                    # Try to find a UI location
                    ui_loc = ''
                    if key in ui_locations:
                        ui_loc = ui_locations[key]

                    settings_data.append((cat, key, val, ui_loc, source))# Populate the table with all collected settings data
        self.settings_table.setRowCount(len(settings_data))
        for row, data_tuple in enumerate(settings_data):
            # Handle different tuple lengths
            if len(data_tuple) == 5:
                category, setting, value, ui_location, source = data_tuple
            elif len(data_tuple) == 4:
                category, setting, value, ui_location = data_tuple
                source = ui_location if ui_location.endswith('.json') else ''
            else:
                category, setting, value = data_tuple[:3]
                ui_location = ''
                source = ''

            # Set the category
            self.settings_table.setItem(row, 0, QTableWidgetItem(category))

            # Set the setting name
            self.settings_table.setItem(row, 1, QTableWidgetItem(setting))

            # Set the value, with tooltip for full content
            value_item = QTableWidgetItem(value)
            value_item.setToolTip(value)
            self.settings_table.setItem(row, 2, value_item)

            # Set the UI location if available, otherwise empty
            ui_location_item = QTableWidgetItem(ui_location)
            if ui_location:
                ui_location_item.setToolTip(ui_location)
            self.settings_table.setItem(row, 3, ui_location_item)

            # Set the source file
            self.settings_table.setItem(row, 4, QTableWidgetItem(source))

            # Add tooltip with description if available
            description = self._get_setting_description(setting)
            if description:
                self.settings_table.item(row, 1).setToolTip(description)

        # Adjust column widths
        # Always set desired column widths after resizeColumnsToContents to enforce them
        self.settings_table.resizeColumnsToContents()
        self.settings_table.setColumnWidth(0, 150)  # Category
        self.settings_table.setColumnWidth(1, 230)  # Setting
        self.settings_table.setColumnWidth(2, 280)  # Value
        self.settings_table.setColumnWidth(3, 280)  # UI Location
        self.settings_table.setColumnWidth(4, 120)  # Source