#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import os
import json
import datetime
import threading
import shutil
import time
from calibre.constants import config_dir, __version__ as calibre_version
from calibre.gui2 import gprefs
from calibre import prints
from .version_utils import extract_plugin_version, compare_versions, version_tuple_to_string, normalize_version

# Debug controlled by environment variable
import os as _ccr_os
CCR_DEBUG_ENABLED = _ccr_os.environ.get('CCR_DEBUG_ENABLED', '0').lower() in ('1', 'true', 'yes')

# Silence debug prints in production by overriding prints when debugging is disabled.
if not CCR_DEBUG_ENABLED:
    def prints(*args, **kwargs):
        return None
else:
    try:
        from calibre.utils.logging import prints as prints
    except Exception:
        def prints(*args, **kwargs):
            __import__('builtins').print(*args, **kwargs)

HISTORY_FILE = os.path.join(config_dir, 'plugins', 'calibre_config_reports', 'ccr_history.json')

def get_setting_value(setting_key, loaded_configs=None, setting_config=None):
    """
    Get the value of a setting from the correct source file.

    Args:
        setting_key (str): The setting key to look up
        loaded_configs (dict, optional): Pre-loaded config files
        setting_config (dict, optional): Pre-loaded setting configuration

    Returns:
        tuple: (value, source_file) or (None, None) if not found
    """
    if loaded_configs is None:
        loaded_configs = load_all_config_files()

    if setting_config is None:
        setting_config = get_setting_config_info()

    if setting_key not in setting_config:
        return None, None

    config_info = setting_config[setting_key]
    source = config_info['source']

    if source == 'db':
        # Special handling for database settings
        if setting_key == 'categories_using_hierarchy':
            try:
                from calibre.gui2.ui import get_gui
                gui = get_gui()
                if gui is not None and hasattr(gui, 'current_db'):
                    db = gui.current_db
                    if db is not None:
                        return db.prefs.get('categories_using_hierarchy', None), 'db'
            except Exception:
                pass
        return None, source
    elif source == 'gprefs':
        # Special handling for gprefs
        from calibre.gui2 import gprefs
        return gprefs.get(setting_key, None), source
    elif source in loaded_configs:
        # Regular config file
        value = loaded_configs[source].get(setting_key, None)
        return value, source

    return None, None


def load_all_config_files():
    """
    Load all configuration files used by CCR.

    Returns:
        dict: Dictionary with config file contents
    """
    import json

    config_files = {
        'gui.py.json': os.path.join(config_dir, 'gui.py.json'),
        'gui.json': os.path.join(config_dir, 'gui.json'),
        'global.py.json': os.path.join(config_dir, 'global.py.json'),
        'tweaks.json': os.path.join(config_dir, 'tweaks.json'),
        'tts.json': os.path.join(config_dir, 'tts.json'),
    }

    loaded_configs = {}

    for config_name, config_path in config_files.items():
        try:
            with open(config_path, 'r', encoding='utf-8') as f:
                loaded_configs[config_name] = json.load(f)
        except Exception:
            loaded_configs[config_name] = {}

    return loaded_configs


def get_setting_config_info():
    """
    Return a dictionary mapping setting keys to their configuration information.
    This centralizes the knowledge of which file each setting comes from.

    Returns:
        dict: {setting_key: {'category': str, 'display_name': str, 'source': str}}
    """
    settings_config = {}

    # Settings tracked in GUI preferences (from monitor_gui_preferences)
    gui_settings = [
        ('search_as_you_type', 'Search', 'Search as you type', 'gui.py.json'),
        ('highlight_search_matches', 'Search', 'Highlight search matches instead of restricting the book list to the results', 'gui.py.json'),
        ('show_highlight_toggle_button', 'Search', 'Show a quick toggle button to switch between highlighting and restricting results next to the Search bar', 'gui.json'),
        ('case_sensitive', 'Search', 'Case sensitive searching', 'global.py.json'),
        ('use_primary_find_in_search', 'Search', 'Unaccented characters match accented characters and punctuation is ignored', 'global.py.json'),
        ('limit_search_columns', 'Search', 'Limit the searched metadata', 'global.py.json'),
        ('limit_search_columns_to', 'Search', 'Columns that non-prefixed searches are limited to', 'global.py.json'),
        ('search_tool_bar_shows_text', 'Search', 'Show text next to buttons in the search bar', 'gui.py.json'),
        ('allow_keyboard_search_in_library_views', 'Search', 'Use keyboard searching in the book list', 'gui.py.json'),
        ('categories_using_hierarchy', 'Search', 'Show saved searches as a hierarchy', 'db'),
        ('systray_icon', 'System Tray', 'Show System Tray Icon', 'gui.py.json'),
        ('separate_cover_flow', 'Interface', 'Separate Cover Flow', 'gui.py.json'),
        ('grid_view_visible', 'Interface', 'Grid View Visible', 'gui.py.json'),
        ('disable_animations', 'Performance', 'Disable Animations', 'gui.py.json'),
        ('enforce_cpu_limit', 'Performance', 'Enforce CPU Limit', 'gui.py.json'),
        ('auto_download_cover', 'Metadata', 'Auto Download Cover', 'gui.py.json'),
        ('autolaunch_server', 'Content Server', 'Auto Launch Server', 'gui.py.json'),
        ('read_file_metadata', 'Adding books', 'Read file metadata', 'global.py.json'),
        ('swap_author_names', 'Adding books', 'Swap author names', 'global.py.json'),
    ]

    # Settings tracked in extra settings monitoring
    extra_gui_settings = [
        ('send_to_storage_card_by_default', 'Preferences', 'Send To Storage Card By Default', 'gui.json'),
        ('confirm_delete', 'Preferences', 'Confirm Delete', 'gui.json'),
        ('main_window_geometry', 'Window Geometry', 'Main Window Geometry', 'gui.json'),
        ('new_version_notification', 'GUI Preferences', 'New Version Notification', 'gui.json'),
        ('use_roman_numerals_for_series_number', 'GUI Preferences', 'Use Roman Numerals For Series Number', 'gui.json'),
        ('sort_tags_by', 'GUI Preferences', 'Sort Tags By', 'gui.json'),
        ('match_tags_type', 'GUI Preferences', 'Match Tags Type', 'gui.json'),
        ('cover_flow_queue_length', 'GUI Preferences', 'Cover Flow Queue Length', 'gui.json'),
        ('oldest_news', 'GUI Preferences', 'Oldest News', 'gui.json'),
        ('upload_news_to_device', 'GUI Preferences', 'Upload News To Device', 'gui.json'),
        ('delete_news_from_library_on_upload', 'GUI Preferences', 'Delete News From Library On Upload', 'gui.json'),
        ('default_send_to_device_action', 'GUI Preferences', 'Default Send To Device Action', 'gui.json'),
        ('worker_limit', 'GUI Preferences', 'Worker Limit', 'gui.json'),
        ('overwrite_author_title_metadata', 'GUI Preferences', 'Overwrite Author Title Metadata', 'gui.json'),
        ('gui_layout', 'Interface', 'GUI Layout', 'gui.json'),
        ('toolbar_icon_size', 'Interface', 'Icon Size', 'gui.json'),
        ('show_splash_screen', 'Interface', 'Show the splash screen on startup', 'gui.json'),
        ('show_tooltips_in_book_list', 'Interface', 'Show Tooltips In Book List', 'gui.json'),
        ('icon_size', 'Interface', 'Icon Size', 'gui.json'),
        ('color_palette', 'Interface', 'Color Theme', 'gui.json'),
        ('dark_palette_name', 'Interface', 'Dark Theme', 'gui.json'),
        ('light_palette_name', 'Interface', 'Light Theme', 'gui.json'),
        ('book_list_tooltips', 'Interface', 'Book List Tooltips', 'gui.json'),
        ('show_sb_all_actions_button', 'Interface', 'Show All Actions Button', 'gui.json'),
        ('tag browser search box visible', 'Interface', 'Tag Browser Search Box Visible', 'gui.json'),
        ('search bar visible', 'Interface', 'Search Bar Visible', 'gui.json'),
        ('font', 'Interface', 'Font', 'gui.json'),
        ('show_layout_buttons', 'Interface', 'Show Layout Buttons', 'gui.json'),
        ('show_actions_button', 'Interface', 'Show Actions Button', 'gui.json'),
        ('disable_tray_notification', 'System Tray', 'Disable Tray Notification', 'gui.py.json'),
    ]

    # Global settings
    global_settings = [
        ('output_format', 'File Formats', 'Default Output Format', 'global.py.json'),
        ('input_format_order', 'File Formats', 'Input Format Order', 'global.py.json'),
        ('language', 'System', 'UI Language', 'global.py.json'),
    ]

    # Tweaks settings
    tweaks_settings = [
        ('author_sort_copy_method', 'Metadata Processing', 'Author Sort Copy Method', 'tweaks.json'),
    ]

    # Combine all settings
    all_settings = gui_settings + extra_gui_settings + global_settings + tweaks_settings

    for setting_key, category, display_name, source in all_settings:
        settings_config[setting_key] = {
            'category': category,
            'display_name': display_name,
            'source': source
        }

    return settings_config


def migrate_history_file():
    """Migrate old history.json to ccr_history.json to preserve existing user data"""
    old_history_file = os.path.join(config_dir, 'plugins', 'calibre_config_reports', 'history.json')

    # Only migrate if old file exists and new file doesn't exist
    if os.path.exists(old_history_file) and not os.path.exists(HISTORY_FILE):
        try:
            shutil.move(old_history_file, HISTORY_FILE)
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Migrated history file from {old_history_file} to {HISTORY_FILE}")
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Failed to migrate history file: {e}")# Perform migration on module load
migrate_history_file()

def get_ccr_version():
    """Return the current CCR version as a string (e.g., '1.1.1') without hardcoding.

    Strategy:
    - Prefer PluginCatalogPlugin.version (the single source of truth).
    - If import fails for any reason, try parsing __init__.py in this package.
    - If still unavailable, return None (caller must handle gracefully).
    """
    # 1) Try the class attribute provided by the plugin
    try:
        from . import PluginCatalogPlugin
        return ".".join(str(x) for x in PluginCatalogPlugin.version)
    except Exception:
        pass

    # 2) Parse __init__.py in this package as a fallback
    try:
        init_path = os.path.join(os.path.dirname(__file__), '__init__.py')
        if os.path.exists(init_path):
            import re
            with open(init_path, 'r', encoding='utf-8') as f:
                src = f.read()
            m = re.search(r"version\s*=\s*\(([^)]+)\)", src)
            if m:
                parts = [p.strip() for p in m.group(1).split(',')]
                # Keep only digits
                parts = [re.sub(r"[^0-9]", "", p) for p in parts]
                parts = [p for p in parts if p != ""]
                if parts:
                    return ".".join(parts)
    except Exception:
        pass

    # 3) Give up — let caller decide what to do
    return None

def _commit_gprefs_safely():
    """Attempt to flush gprefs to disk immediately (best-effort)."""
    try:
        import calibre.gui2.gprefs as gprefs_module
        if hasattr(gprefs_module, 'gprefs') and hasattr(gprefs_module.gprefs, 'commit'):
            gprefs_module.gprefs.commit()
    except Exception:
        # Non-fatal — normal Calibre save cycle will eventually persist
        pass

class HistoryLogger:
    _lock = threading.Lock()

    @staticmethod
    def log_event(event_type, details):
        entry = {
            "timestamp": datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            "event_type": event_type,
            "details": details
        }
        with HistoryLogger._lock:
            try:
                history = []
                if os.path.exists(HISTORY_FILE):
                    with open(HISTORY_FILE, 'r', encoding='utf-8') as f:
                        history = json.load(f)
                history.append(entry)
                with open(HISTORY_FILE, 'w', encoding='utf-8') as f:
                    json.dump(history, f, indent=2)
            except Exception as e:
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: Error logging event: {str(e)}")

    @staticmethod
    def get_history():
        with HistoryLogger._lock:
            try:
                if os.path.exists(HISTORY_FILE): #dont add extra paren
                    with open(HISTORY_FILE, 'r', encoding='utf-8') as f:
                        history = json.load(f)
                        return history
            except Exception as e:
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: Error reading history: {str(e)}")
            return []

def _initialize_tracking_silently():
    """Initialize all tracking variables on first install without logging existing settings as changes"""
    import json

    # Initialize Calibre version tracking
    gprefs['calibre_config_reports_last_calibre_version'] = calibre_version

    # Initialize plugin version tracking
    customize_path = os.path.join(config_dir, 'customize.py.json')
    try:
        current_versions = {}
        if (os.path.exists(customize_path)):
            with open(customize_path, 'r', encoding='utf-8') as f:
                customize = json.load(f)
                for name, path in customize.get('plugins', {}).items():
                    version = extract_plugin_version(path)
                    if version:
                        normalized_version = normalize_version(version)
                        current_versions[name] = (normalized_version, path)
        gprefs['calibre_config_reports_last_versions'] = current_versions
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Error initializing plugin tracking: {str(e)}")

    # Initialize GUI preferences tracking
    _initialize_gui_preferences_silently()

    # Initialize extra settings tracking
    _initialize_extra_settings_silently()

    if CCR_DEBUG_ENABLED:
        prints("CCR: All tracking variables initialized silently")

def _initialize_gui_preferences_silently():
    """Initialize GUI preferences tracking without logging"""
    import json

    settings_to_track = [
        ('search_as_you_type', 'Search', 'Search as you type', 'gui.py.json', None),
        ('highlight_search_matches', 'Search', 'Highlight search matches instead of restricting the book list to the results', 'gui.py.json', None),
        ('show_highlight_toggle_button', 'Search', 'Show a quick toggle button to switch between highlighting and restricting results next to the Search bar', 'gui.json', None),
        ('case_sensitive', 'Search', 'Case sensitive searching', 'global.py.json', None),
        ('use_primary_find_in_search', 'Search', 'Unaccented characters match accented characters and punctuation is ignored', 'global.py.json', None),
        ('limit_search_columns', 'Search', 'Limit the searched metadata', 'global.py.json', None),
        ('limit_search_columns_to', 'Search', 'Columns that non-prefixed searches are limited to', 'global.py.json', None),
        ('search_tool_bar_shows_text', 'Search', 'Show text next to buttons in the search bar', 'gui.py.json', None),
        ('allow_keyboard_search_in_library_views', 'Search', 'Use keyboard searching in the book list', 'gui.py.json', None),
        ('categories_using_hierarchy', 'Search', 'Show saved searches as a hierarchy', 'db', None),
        ('systray_icon', 'System Tray', 'Show System Tray Icon', 'gui.py.json', None),
        ('separate_cover_flow', 'Interface', 'Separate Cover Flow', 'gui.py.json', None),
        ('grid_view_visible', 'Interface', 'Grid View Visible', 'gui.py.json', None),
        ('disable_animations', 'Performance', 'Disable Animations', 'gui.py.json', None),
        ('enforce_cpu_limit', 'Performance', 'Enforce CPU Limit', 'gui.py.json', None),
        ('auto_download_cover', 'Metadata', 'Auto Download Cover', 'gui.py.json', None),
        ('autolaunch_server', 'Content Server', 'Auto Launch Server', 'gui.py.json', None),
        ('read_file_metadata', 'Adding books', 'Read file metadata', 'global.py.json', None),
        ('swap_author_names', 'Adding books', 'Swap author names', 'global.py.json', None),
    ]

    # Load config files
    config_files = {
        'gui.py.json': os.path.join(config_dir, 'gui.py.json'),
        'gui.json': os.path.join(config_dir, 'gui.json'),
        'global.py.json': os.path.join(config_dir, 'global.py.json'),
    }

    def load_json(path):
        try:
            with open(path, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception:
            return {}

    loaded_json = {k: load_json(v) for k, v in config_files.items()}

    # Initialize DB cache for categories_using_hierarchy
    def get_db_value(key):
        if key == 'categories_using_hierarchy':
            try:
                from calibre.gui2.ui import get_gui
                gui = get_gui()
                if gui is not None and hasattr(gui, 'current_db'):
                    db = gui.current_db
                    if db is not None:
                        return db.prefs.get('categories_using_hierarchy', None)
            except Exception:
                pass
        return None

    # Store current values without logging
    last_gui_states = {}
    for setting, category, display_name, source, extra in settings_to_track:
        try:
            if source in loaded_json:
                current_value = loaded_json[source].get(setting, None)
            elif source == 'gprefs':
                current_value = gprefs.get(setting, None)
            elif source == 'db':
                current_value = get_db_value(setting)
            else:
                current_value = None

            # Handle special cases
            if isinstance(current_value, set):
                current_value = list(current_value)
            if setting == 'show_highlight_toggle_button' and current_value is None:
                current_value = False

            last_gui_states[setting] = current_value
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"CCR: Error initializing setting {setting}: {str(e)}")

    gprefs['calibre_config_reports_last_gui_states'] = last_gui_states

def _initialize_extra_settings_silently():
    """Initialize extra settings tracking without logging"""
    import json

    # Initialize tracking for all monitored extra settings
    last_extra_states = {}
    last_global_states = {}

    # Load config files
    gui_json_path = os.path.join(config_dir, 'gui.json')
    global_json_path = os.path.join(config_dir, 'global.py.json')
    gui_py_json_path = os.path.join(config_dir, 'gui.py.json')
    tweaks_json_path = os.path.join(config_dir, 'tweaks.json')

    try:
        with open(gui_json_path, 'r', encoding='utf-8') as f:
            gui_json = json.load(f)
    except Exception:
        gui_json = {}

    try:
        with open(global_json_path, 'r', encoding='utf-8') as f:
            global_json = json.load(f)
    except Exception:
        global_json = {}

    try:
        with open(gui_py_json_path, 'r', encoding='utf-8') as f:
            gui_py_json = json.load(f)
    except Exception:
        gui_py_json = {}

    try:
        with open(tweaks_json_path, 'r', encoding='utf-8') as f:
            tweaks_json = json.load(f)
    except Exception:
        tweaks_json = {}

    # Store all current values from the various tracked settings lists
    # (This duplicates the settings lists from monitor_extra_settings for initialization)

    # Global settings
    for key in ['output_format', 'input_format_order', 'language']:
        last_global_states[key] = global_json.get(key, None)

    # GUI settings from gui.json and gui.py.json
    gui_settings_keys = [
        'send_to_storage_card_by_default', 'confirm_delete', 'main_window_geometry',
        'new_version_notification', 'use_roman_numerals_for_series_number',
        'sort_tags_by', 'match_tags_type', 'cover_flow_queue_length',
        'autolaunch_server', 'oldest_news', 'upload_news_to_device',
        'delete_news_from_library_on_upload', 'default_send_to_device_action',
        'worker_limit', 'overwrite_author_title_metadata', 'auto_download_cover',
        'enforce_cpu_limit', 'gui_layout', 'toolbar_icon_size', 'show_splash_screen',
        'show_tooltips_in_book_list', 'icon_size', 'color_palette',
        'dark_palette_name', 'light_palette_name', 'book_list_tooltips',
        'show_sb_all_actions_button', 'tag browser search box visible',
        'search bar visible', 'font', 'show_layout_buttons', 'show_actions_button',
        'disable_tray_notification'
    ]

    for key in gui_settings_keys:
        if key == 'main_window_geometry':
            geom = gui_json.get('geometry-of-calibre_main_window_geometry', {})
            last_extra_states[key] = geom.get('frame_geometry', None)
        elif key == 'gui_layout':
            widget_state = gui_json.get('main_window_central_widget_state', {})
            last_extra_states[key] = widget_state.get('layout', gui_json.get('gui_layout', None))
        elif key == 'disable_tray_notification':
            last_extra_states[key] = gui_py_json.get(key, None)
        else:
            last_extra_states[key] = gui_json.get(key, None)

    # Handle tweaks.json settings
    last_extra_states['author_sort_copy_method'] = tweaks_json.get('author_sort_copy_method', None)

    # Store initialized states
    gprefs['calibre_config_reports_last_global_states'] = last_global_states
    gprefs['calibre_config_reports_last_extra_states'] = last_extra_states
    gprefs['calibre_config_reports_last_language'] = global_json.get('language', None)

def monitor_and_log_config_events():
    """Monitor configuration changes and log events to the history

    Note: This function is called when the CCR dialog opens. For plugins installed
    through Plugin Updater without restarting Calibre, changes will only be detected
    the next time CCR opens. To catch all plugin installations immediately, restart
    Calibre after installing plugins, or open CCR periodically to refresh monitoring.
    """
    if CCR_DEBUG_ENABLED:
        prints("CCR: Starting monitor_and_log_config_events")
        pass

    # Clean up any standalone plugin removals from hooks
    try:
        from .plugin_install_hook import cleanup_standalone_removals
        cleanup_standalone_removals()
    except Exception:
        pass  # Non-critical if hooks aren't available

    # --- First-install detection and CCR reinstall detection ---
    is_first_install = gprefs.get('calibre_config_reports_first_install', True)
    last_ccr_version = gprefs.get('calibre_config_reports_last_ccr_version', None)

    # Get current CCR version dynamically (no hardcoded fallback)
    current_ccr_version = get_ccr_version()

    if CCR_DEBUG_ENABLED:
        prints(f"CCR: Current version: {current_ccr_version}, Last stored: {last_ccr_version}, First install: {is_first_install}")

    if is_first_install:
        # On first install, don't log all existing settings as "changes"
        # Just log that CCR was installed and initialize tracking
        HistoryLogger.log_event('plugin_installed', {
            'event_type': 'Plugin installed',
            'name': 'Calibre Config Reports',
            'version': current_ccr_version or 'Unknown',
            'path': 'First install - monitoring configuration changes'
        })

        # Initialize all tracking without logging changes
        _initialize_tracking_silently()

        # Mark that we've completed the first install and store version
        gprefs['calibre_config_reports_first_install'] = False
        if current_ccr_version:
            gprefs['calibre_config_reports_last_ccr_version'] = current_ccr_version
        _commit_gprefs_safely()

        if CCR_DEBUG_ENABLED:
            prints("CCR: First install completed - tracking initialized without logging existing settings")
        return

    # Same CCR version as last time: do NOT log anything here.
    # Reinstalls are only logged by real plugin operations (hooks) or
    # by customize.py.json diff detection. Simply opening CCR must not
    # create a spurious "reinstalled" entry.
    elif last_ccr_version and current_ccr_version and last_ccr_version == current_ccr_version:
        current_time = time.time()
        gprefs['calibre_config_reports_last_ccr_version'] = current_ccr_version
        gprefs['calibre_config_reports_last_check_time'] = current_time
        _commit_gprefs_safely()
        # Continue with normal monitoring without logging a reinstall

    # Check for CCR upgrade/downgrade
    elif last_ccr_version and current_ccr_version and last_ccr_version != current_ccr_version:
        # This is a version change
        comparison = compare_versions(
            normalize_version(current_ccr_version),
            normalize_version(last_ccr_version)
        )

        if comparison > 0:
            event_type = 'plugin_updated'
            event_label = 'Plugin updated'
        else:
            event_type = 'plugin_downgraded'
            event_label = 'Plugin downgraded'

        HistoryLogger.log_event(event_type, {
            'event_type': event_label,
            'name': 'Calibre Config Reports',
            'old_version': last_ccr_version,
            'new_version': current_ccr_version,
            'path': f'Plugin {event_label.lower()} from v{last_ccr_version} to v{current_ccr_version}'
        })

        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Version change detected: {last_ccr_version} -> {current_ccr_version}")

        # Update stored version
        gprefs['calibre_config_reports_last_ccr_version'] = current_ccr_version
        _commit_gprefs_safely()
        # Continue with normal monitoring

    # --- Calibre version update detection ---
    last_version = gprefs.get('calibre_config_reports_last_calibre_version', None)
    if last_version != calibre_version:
        if last_version is not None:
            cmp = compare_versions(
                normalize_version(last_version),
                normalize_version(calibre_version)
            )
            if cmp > 0:
                # Calibre reverted/downgraded
                HistoryLogger.log_event('calibre_reverted', {
                    'event_type': 'Calibre reverted',
                    'old_version': last_version,
                    'new_version': calibre_version
                })
            else:
                # Calibre updated
                HistoryLogger.log_event('calibre_updated', {
                    'event_type': 'Calibre updated',
                    'old_version': last_version,
                    'new_version': calibre_version
                })
        gprefs['calibre_config_reports_last_calibre_version'] = calibre_version

    # --- Plugin installation/update detection ---
    # Always run robust customize.py.json monitoring; hooks only supplement via dedup
    try:
        _monitor_plugin_changes_via_customize_json()
    except Exception:
        # Non-fatal; continue with settings monitoring
        pass

def _get_last_versions_from_history(current_versions=None):
    """Get the last logged version for each plugin from history.

    If current_versions is provided (dict name -> (version_tuple, path)), skip the most recent
    history entry for a plugin when it matches the current version. This allows us to find the
    prior version even if hooks already logged the new version before this monitor runs.
    """
    try:
        history = HistoryLogger.get_history()
        last_versions = {}
        current_version_map = {}
        if isinstance(current_versions, dict):
            # Flatten to name -> version_tuple for easy comparison
            for n, (v, _p) in current_versions.items():
                current_version_map[n] = v

        # Go through history from newest to oldest to find last (prior) version for each plugin
        for entry in reversed(history):
            event_type = entry.get('event_type', '')
            details = entry.get('details', {})

            if not (isinstance(details, dict) and event_type.startswith('plugin_')):
                continue

            name = details.get('name', '')
            if not name or name in last_versions:
                continue

            # Extract version recorded by this entry
            if event_type in ['plugin_updated', 'plugin_downgraded']:
                version = details.get('new_version', '')
            else:
                version = details.get('version', '')

            if not version or version == 'Unknown':
                continue

            try:
                normalized_version = normalize_version(version)
            except Exception:
                continue

            # If we know the current version and this entry matches it, skip to find the prior one
            if name in current_version_map and normalized_version == current_version_map[name]:
                # Keep looking for an older entry for this plugin
                continue

            # Otherwise, record this as the last known prior version
            last_versions[name] = normalized_version

        return last_versions
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Error reading history versions: {e}")
        return {}

def _check_recent_hook_log(plugin_name, event_type, version_str):
    """Check if this event was recently logged by plugin hooks to prevent duplicates"""
    try:
        # Check recent history for identical events from hooks
        history = HistoryLogger.get_history()
        cutoff_time = datetime.datetime.now() - datetime.timedelta(seconds=300)  # 5 minutes

        for entry in reversed(history[-20:]):  # Check last 20 entries
            # Parse timestamp safely
            try:
                entry_time = datetime.datetime.strptime(entry.get('timestamp', ''), '%Y-%m-%d %H:%M:%S')
            except Exception:
                # Malformed or missing timestamp; skip this entry
                continue

            if entry_time < cutoff_time:
                continue

            try:
                details = entry.get('details', {})
                entry_name = details.get('name', '')
                entry_event = entry.get('event_type', '')

                # Only consider entries that match the same plugin and event type
                if entry_name != plugin_name or entry_event != event_type:
                    continue

                # Extract the version string depending on event type
                if event_type in ['plugin_updated', 'plugin_downgraded']:
                    entry_version = details.get('new_version', '')
                else:
                    entry_version = details.get('version', '')

                # Try normalized comparison first
                entry_ver_norm = None
                check_ver_norm = None
                try:
                    entry_ver_norm = normalize_version(entry_version)
                    check_ver_norm = normalize_version(version_str)
                except Exception:
                    # Normalization failed for one of the strings; fall back
                    entry_ver_norm = None
                    check_ver_norm = None

                if entry_ver_norm is not None and check_ver_norm is not None:
                    if entry_ver_norm == check_ver_norm:
                        if CCR_DEBUG_ENABLED:
                            prints(f"CCR: DUPLICATE DETECTED - skipping {plugin_name} {event_type} (recently logged by hooks)")
                        return True
                else:
                    # Fallback string comparison handling common formatting differences
                    try:
                        if (entry_version == version_str or
                            (entry_version == "1" and version_str == "1.0.0") or
                            (entry_version == "1.0.0" and version_str == "1")):
                            if CCR_DEBUG_ENABLED:
                                prints(f"CCR: DUPLICATE DETECTED (string fallback) - skipping {plugin_name} {event_type}")
                            return True
                    except Exception:
                        # If fallback comparison fails, skip this entry
                        continue
            except Exception:
                # If there's any error processing this history entry, skip it
                continue

        return False
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Error checking recent hook logs: {e}")
        return False

def _monitor_plugin_changes_via_customize_json():
    """Monitor plugin changes via customize.py.json using 1.03 robust logic"""
    customize_path = os.path.join(config_dir, 'customize.py.json')
    try:
        # Get current plugin versions from customize.py.json
        current_versions = {}
        if os.path.exists(customize_path):
            with open(customize_path, 'r', encoding='utf-8') as f:
                customize = json.load(f)
                for name, path in customize.get('plugins', {}).items():
                    version = extract_plugin_version(path)
                    if CCR_DEBUG_ENABLED:
                        prints(f"CCR: Plugin {name} at {path} -> version {version}")
                    if version:
                        normalized_version = normalize_version(version)
                        current_versions[name] = (normalized_version, path)
                    elif CCR_DEBUG_ENABLED:
                        prints(f"CCR: SKIPPED {name} - version extraction failed")

        # Get last tracked plugin versions from gprefs
        last_versions = {}
        stored_versions = gprefs.get('calibre_config_reports_last_versions', {})

        # Validate stored versions against current reality - if a plugin exists
        # but the stored version is wildly different, it's likely stale data
        for name, (version, path) in stored_versions.items():
            if version:
                normalized_version = normalize_version(version)

                # Check for unrealistic version differences that suggest stale data
                if name in current_versions:
                    current_ver = current_versions[name][0]

                    if normalized_version and current_ver:
                        # If major version differs by more than 5, treat as stale data
                        major_diff = abs(current_ver[0] - normalized_version[0]) if len(current_ver) > 0 and len(normalized_version) > 0 else 0
                        if major_diff > 5:
                            if CCR_DEBUG_ENABLED:
                                prints(f"CCR: STALE DATA detected for {name}: stored {normalized_version} vs current {current_ver}, skipping version tracking")
                            continue

                last_versions[name] = (normalized_version, path)

        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Found {len(current_versions)} current plugins, {len(last_versions)} tracked plugins")

        # Detect if we have widespread stale data (many plugins with suspicious version differences)
        stale_count = 0
        total_comparable = 0

        for name in current_versions:
            if name in last_versions:
                total_comparable += 1
                current_ver = current_versions[name][0]
                last_ver = last_versions[name][0]

                if current_ver and last_ver:
                    major_diff = abs(current_ver[0] - last_ver[0]) if len(current_ver) > 0 and len(last_ver) > 0 else 0
                    if major_diff > 3:  # Conservative threshold for stale detection
                        stale_count += 1

        # If more than 50% of comparable plugins have suspicious version differences,
        # this suggests the stored data is entirely stale - reset tracking
        if total_comparable > 5 and (stale_count / total_comparable) > 0.5:
            if CCR_DEBUG_ENABLED:
                prints(f"CCR: WIDESPREAD STALE DATA detected ({stale_count}/{total_comparable} plugins), resetting plugin tracking")
            last_versions = {}  # Reset all tracking to avoid false version change reports

        # Compare current vs last versions and log changes
        for name, (version, path) in current_versions.items():
            # Skip CCR itself - we handle it specially above
            if name == 'Calibre Config Reports':
                continue

            if CCR_DEBUG_ENABLED:
                prints(f"CCR: Processing plugin {name} v{version_tuple_to_string(version)}")
                prints(f"CCR: Plugin {name} in last_versions: {name in last_versions}")

            if name not in last_versions:
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: {name} not in last_versions - checking for recent removal")

                # Check if this was recently removed (indicating a downgrade/upgrade via file)
                recent_removal = _check_recent_removal(name)
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: Recent removal for {name}: {recent_removal is not None}")
                    if recent_removal:
                        prints(f"CCR: Recent removal details: {recent_removal}")

                if recent_removal:
                    removed_version_str = recent_removal.get('details', {}).get('version', '')
                    if removed_version_str:
                        try:
                            removed_version = normalize_version(removed_version_str)
                            comparison = compare_versions(version, removed_version)

                            if comparison > 0:
                                # Plugin updated
                                if CCR_DEBUG_ENABLED:
                                    prints(f"CCR: Plugin updated: {name} v{removed_version_str} -> v{version_tuple_to_string(version)}")

                                HistoryLogger.log_event('plugin_updated', {
                                    'event_type': 'Plugin updated',
                                    'name': name,
                                    'old_version': removed_version_str,
                                    'new_version': version_tuple_to_string(version)
                                })

                                # Remove the recent removal entry since we're replacing it with an update
                                _remove_recent_removal_entry(name)
                                continue
                            elif comparison < 0:
                                # Plugin downgraded
                                if CCR_DEBUG_ENABLED:
                                    prints(f"CCR: Plugin downgraded: {name} v{removed_version_str} -> v{version_tuple_to_string(version)}")

                                HistoryLogger.log_event('plugin_downgraded', {
                                    'event_type': 'Plugin downgraded',
                                    'name': name,
                                    'old_version': removed_version_str,
                                    'new_version': version_tuple_to_string(version)
                                })

                                # Remove the recent removal entry since we're replacing it with a downgrade
                                _remove_recent_removal_entry(name)
                                continue
                        except Exception:
                            pass  # Fall through to regular installation                # Regular new plugin installation
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: New plugin detected: {name} v{version_tuple_to_string(version)}")

                # Check for recent hook logging to prevent duplicates
                version_str = version_tuple_to_string(version)
                if _check_recent_hook_log(name, 'plugin_installed', version_str):
                    if CCR_DEBUG_ENABLED:
                        prints(f"CCR: Skipping duplicate plugin_installed for {name} - already logged by hooks")
                    continue

                HistoryLogger.log_event('plugin_installed', {
                    'event_type': 'Plugin installed',
                    'name': name,
                    'version': version_tuple_to_string(version),
                    'path': path
                })
            else:
                last_version, last_path = last_versions[name]
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: {name} found in last_versions with version {version_tuple_to_string(last_version)}")
                    prints(f"CCR: Comparing current {version_tuple_to_string(version)} vs last {version_tuple_to_string(last_version)}")

                if version != last_version:
                    comparison = compare_versions(version, last_version)
                    if CCR_DEBUG_ENABLED:
                        prints(f"CCR: Version comparison result: {comparison}")

                    if comparison > 0:
                        # Plugin updated
                        if CCR_DEBUG_ENABLED:
                            prints(f"CCR: Plugin updated: {name} v{version_tuple_to_string(last_version)} -> v{version_tuple_to_string(version)}")

                        # Check for recent hook logging to prevent duplicates
                        if _check_recent_hook_log(name, 'plugin_updated', version_tuple_to_string(version)):
                            if CCR_DEBUG_ENABLED:
                                prints(f"CCR: Skipping duplicate plugin_updated for {name} - already logged by hooks")
                        else:
                            HistoryLogger.log_event('plugin_updated', {
                                'event_type': 'Plugin updated',
                                'name': name,
                                'old_version': version_tuple_to_string(last_version),
                                'new_version': version_tuple_to_string(version)
                            })
                    elif comparison < 0:
                        # Plugin downgraded
                        if CCR_DEBUG_ENABLED:
                            prints(f"CCR: Plugin downgraded: {name} v{version_tuple_to_string(last_version)} -> v{version_tuple_to_string(version)}")

                        # Check for recent hook logging to prevent duplicates
                        if _check_recent_hook_log(name, 'plugin_downgraded', version_tuple_to_string(version)):
                            if CCR_DEBUG_ENABLED:
                                prints(f"CCR: Skipping duplicate plugin_downgraded for {name} - already logged by hooks")
                        else:
                            HistoryLogger.log_event('plugin_downgraded', {
                                'event_type': 'Plugin downgraded',
                                'name': name,
                                'old_version': version_tuple_to_string(last_version),
                                'new_version': version_tuple_to_string(version)
                            })
                    # else: same version, no change to log

        # Detect removals
        for name, (old_version, old_path) in last_versions.items():
            if name not in current_versions:
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: Plugin removed: {name} v{version_tuple_to_string(old_version)}")

                HistoryLogger.log_event('plugin_removed', {
                    'event_type': 'Plugin removed',
                    'name': name,
                    'version': version_tuple_to_string(old_version),
                    'path': old_path
                })        # Update tracked versions
        gprefs['calibre_config_reports_last_versions'] = current_versions
        _commit_gprefs_safely()

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Error monitoring plugin changes: {str(e)}")
            import traceback
            prints(f"CCR: Traceback: {traceback.format_exc()}")

    # Monitor GUI settings for changes (splash screen, animations, etc.)
    try:
        monitor_gui_preferences()
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Error monitoring GUI preferences: {str(e)}")
            import traceback
            prints(f"CCR: GUI Preferences Traceback: {traceback.format_exc()}")

    try:
        monitor_extra_settings()
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Error monitoring settings changes: {str(e)}")
            import traceback
            prints(f"CCR: Settings Traceback: {traceback.format_exc()}")

def _check_recent_removal(name):
    """Check if a plugin was recently removed (within the last few entries)"""
    try:
        history = HistoryLogger.get_history()
        # Check the last 20 entries for a recent removal of this plugin (increased from 10)
        for entry in reversed(history[-20:]):
            event_type = entry.get('event_type', '')
            details = entry.get('details', {})

            # Check if this is a plugin removal event
            if event_type == 'plugin_removed':
                # Try to get plugin name from details
                plugin_name = details.get('name', '')
                if plugin_name == name:
                    return entry
        return None
    except Exception:
        return None

def _remove_recent_removal_entry(name):
    """Remove the most recent removal entry for a plugin from history"""
    try:
        history = HistoryLogger.get_history()
        # Find and remove the most recent removal entry for this plugin
        for i in range(len(history) - 1, -1, -1):
            entry = history[i]
            if (entry.get('event_type') == 'plugin_removed' and
                entry.get('details', {}).get('name') == name):
                history.pop(i)
                break

        # Save the updated history
        import os
        import json
        if os.path.exists(HISTORY_FILE):
            with HistoryLogger._lock:
                with open(HISTORY_FILE, 'w', encoding='utf-8') as f:
                    json.dump(history, f, indent=2)
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"CCR: Error removing recent removal entry: {e}")


def monitor_gui_preferences():
    """Monitor changes to GUI preferences (directly from gui.py.json) and log them"""
    import os
    import json

    if CCR_DEBUG_ENABLED:
        prints("CCR: Starting monitor_gui_preferences")

    # --- Settings tracking by correct file/location ---
    # Each entry: (config_key, category, display_name, source, extra_info)
    # source: 'gui.py.json', 'gui.json', 'global.py.json', 'gprefs', 'db'
    settings_to_track = [
        # Searching tab (exact keys from calibre source)
        ('search_as_you_type', 'Search', 'Search as you type', 'gui.py.json', None),
        ('highlight_search_matches', 'Search', 'Highlight search matches instead of restricting the book list to the results', 'gui.py.json', None),
        ('show_highlight_toggle_button', 'Search', 'Show a quick toggle button to switch between highlighting and restricting results next to the Search bar', 'gui.json', None),
        # The following are actually stored in global.py.json:
        ('case_sensitive', 'Search', 'Case sensitive searching', 'global.py.json', None),
        ('use_primary_find_in_search', 'Search', 'Unaccented characters match accented characters and punctuation is ignored', 'global.py.json', None),
        ('limit_search_columns', 'Search', 'Limit the searched metadata', 'global.py.json', None),
        ('limit_search_columns_to', 'Search', 'Columns that non-prefixed searches are limited to', 'global.py.json', None),
        ('search_tool_bar_shows_text', 'Search', 'Show text next to buttons in the search bar', 'gui.py.json', None),
        ('allow_keyboard_search_in_library_views', 'Search', 'Use keyboard searching in the book list', 'gui.py.json', None),
        # Special: 'Show saved searches as a hierarchy' is stored as 'categories_using_hierarchy' in the db
        #('categories_using_hierarchy', 'Search', 'Show saved searches as a hierarchy', 'db', None), #disable temp
        ('separate_cover_flow', 'Interface', 'Separate Cover Flow', 'gui.py.json', None),
        ('grid_view_visible', 'Interface', 'Grid View Visible', 'gui.py.json', None),
        ('disable_animations', 'Performance', 'Disable Animations', 'gui.py.json', None),
        ('enforce_cpu_limit', 'Performance', 'Enforce CPU Limit', 'gui.py.json', None),
        ('auto_download_cover', 'Metadata', 'Auto Download Cover', 'gui.py.json', None),
        ('autolaunch_server', 'Content Server', 'Auto Launch Server', 'gui.py.json', None),
        # Added monitoring for read_file_metadata and swap_author_names (global.py.json)
        ('read_file_metadata', 'Adding books', 'Read file metadata', 'global.py.json', None),
        ('swap_author_names', 'Adding books', 'Swap author names', 'global.py.json', None),
        # Interface settings from gui.json (moved here to avoid duplicates)
        ('show_sb_all_actions_button', 'Interface', 'Show All Actions Button', 'gui.json', None),
        ('show_layout_buttons', 'Interface', 'Show Layout Buttons', 'gui.json', None),
        # Add more as needed, with correct file
    ]

    # --- Load all config files once ---
    config_files = {
        'gui.py.json': os.path.join(config_dir, 'gui.py.json'),
        'gui.json': os.path.join(config_dir, 'gui.json'),
        'global.py.json': os.path.join(config_dir, 'global.py.json'),
    }

    def load_json(path):
        try:
            with open(path, 'r', encoding='utf-8') as f:
                return json.load(f)
        except Exception:
            return {}

    loaded_json = {k: load_json(v) for k, v in config_files.items()}

    # --- DB special case: categories_using_hierarchy ---
    db_cache = {}
    def get_db_value(key):
        if key == 'categories_using_hierarchy':
            if 'categories_using_hierarchy' not in db_cache:
                try:
                    # Try to get current database safely
                    from calibre.gui2.ui import get_gui
                    gui = get_gui()
                    if gui is not None and hasattr(gui, 'current_db'):
                        db = gui.current_db
                        if db is not None:
                            db_cache['categories_using_hierarchy'] = db.prefs.get('categories_using_hierarchy', None)
                        else:
                            db_cache['categories_using_hierarchy'] = None
                    else:
                        db_cache['categories_using_hierarchy'] = None
                except Exception as e:
                    if CCR_DEBUG_ENABLED:
                        prints(f"CCR: Could not read categories_using_hierarchy from db: {e}")
                    db_cache['categories_using_hierarchy'] = None
            return db_cache['categories_using_hierarchy']
        return None

    # --- Main monitoring logic ---
    last_gui_states = gprefs.get('calibre_config_reports_last_gui_states', {})
    changed = False
    for setting, category, display_name, source, extra in settings_to_track:
        try:
            if source in loaded_json:
                current_value = loaded_json[source].get(setting, None)
            elif source == 'gprefs':
                current_value = gprefs.get(setting, None)
            elif source == 'db':
                current_value = get_db_value(setting)
            else:
                current_value = None
            old_value = last_gui_states.get(setting, None)
            # For list/set, store as list for comparison
            if isinstance(current_value, set):
                current_value = list(current_value)
            if isinstance(old_value, set):
                old_value = list(old_value)

            # Debug print for important settings
            if CCR_DEBUG_ENABLED and setting in (
                'case_sensitive', 'use_primary_find_in_search', 'limit_search_columns',
                'show_highlight_toggle_button', 'search_as_you_type', 'highlight_search_matches',
                'search_tool_bar_shows_text', 'allow_keyboard_search_in_library_views',
                'limit_search_columns_to', 'categories_using_hierarchy', 'disable_animations',
                'separate_cover_flow', 'grid_view_visible', 'enforce_cpu_limit', 'systray_icon'):
                prints(f"CCR DEBUG: {setting} current={current_value!r} old={old_value!r} source={source}")

            # Treat missing key as default for show_highlight_toggle_button (likely False)
            if setting == 'show_highlight_toggle_button':
                if current_value is None:
                    current_value = False
                if old_value is None:
                    old_value = False

            # Skip logging "changes" on first run (when old_value is None)
            # Only log actual changes when we have a previous value to compare against
            if old_value is not None and current_value != old_value:
                if CCR_DEBUG_ENABLED:
                    prints(f"CCR: Logging change for {setting}: {old_value!r} -> {current_value!r}")
                HistoryLogger.log_event('setting_changed', {
                    'category': category,
                    'setting': display_name,
                    'new_value': str(current_value),
                    'old_value': str(old_value),
                    'source': source,
                    'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                })
                last_gui_states[setting] = current_value
                changed = True
            elif old_value is None:
                # First run: store current value without logging as change
                last_gui_states[setting] = current_value
                changed = True
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"CCR: Error monitoring setting {setting}: {str(e)}")
            continue
    if changed:
        gprefs['calibre_config_reports_last_gui_states'] = last_gui_states
        _commit_gprefs_safely()
        if CCR_DEBUG_ENABLED:
            prints("CCR: GUI preferences monitoring completed with changes saved")
    elif CCR_DEBUG_ENABLED:
        prints("CCR: GUI preferences monitoring completed with no changes detected")

def monitor_extra_settings():
    """
    Monitor and log changes to extra settings from gui.json and tweaks.json.
    Logs a 'setting_changed' event if any tracked setting changes.
    """
    import json
    changed = False
    tracked_gui_settings = [
        # (setting_key, category, display_name)
        ('send_to_storage_card_by_default', 'Preferences', 'Send To Storage Card By Default'),
        ('confirm_delete', 'Preferences', 'Confirm Delete'),
        ('main_window_geometry', 'Window Geometry', 'Main Window Geometry'),
        ('new_version_notification', 'GUI Preferences', 'New Version Notification'),
        ('use_roman_numerals_for_series_number', 'GUI Preferences', 'Use Roman Numerals For Series Number'),
        ('sort_tags_by', 'GUI Preferences', 'Sort Tags By'),
        ('match_tags_type', 'GUI Preferences', 'Match Tags Type'),
        ('cover_flow_queue_length', 'GUI Preferences', 'Cover Flow Queue Length'),
        ('oldest_news', 'GUI Preferences', 'Oldest News'),
        ('upload_news_to_device', 'GUI Preferences', 'Upload News To Device'),
        ('delete_news_from_library_on_upload', 'GUI Preferences', 'Delete News From Library On Upload'),
        ('default_send_to_device_action', 'GUI Preferences', 'Default Send To Device Action'),
        # Removed search_as_you_type, worker_limit, and all other search settings (now tracked in monitor_gui_preferences)
        # Also removed auto_download_cover, enforce_cpu_limit, autolaunch_server (now tracked in monitor_gui_preferences from gui.py.json)
        # ('search_as_you_type', 'GUI Preferences', 'Search As You Type'),
        ('worker_limit', 'GUI Preferences', 'Worker Limit'),
        # ('get_social_metadata', 'GUI Preferences', 'Get Social Metadata'), # not needed
        ('overwrite_author_title_metadata', 'GUI Preferences', 'Overwrite Author Title Metadata'),        ('gui_layout', 'Interface', 'GUI Layout'),
        ('toolbar_icon_size', 'Interface', 'Icon Size'),
        #('show_avg_rating', 'GUI Preferences', 'Show Avg Rating'), #temporarily untracked
        # Correct Calibre setting and display name:
        ('show_splash_screen', 'Interface', 'Show the splash screen on startup'),
        ('show_tooltips_in_book_list', 'Interface', 'Show Tooltips In Book List'),
        ('icon_size', 'Interface', 'Icon Size'),
        # Theme/color settings (track both color_palette and dark/light theme toggles)
        ('color_palette', 'Interface', 'Color Theme'),
        ('dark_palette_name', 'Interface', 'Dark Theme'),
        ('light_palette_name', 'Interface', 'Light Theme'),
    ]
    # Track these settings from gui.json
    tracked_gui_json_settings = [
        # Remove duplicate/incorrect splash screen entry, only use the correct one above
        ('book_list_tooltips', 'Interface', 'Book List Tooltips'),
        # REMOVED: ('show_sb_all_actions_button', 'Interface', 'Show All Actions Button'), - now tracked in monitor_gui_preferences
        ('tag browser search box visible', 'Interface', 'Tag Browser Search Box Visible'),
        ('search bar visible', 'Interface', 'Search Bar Visible'),
        ('font', 'Interface', 'Font'),
        # REMOVED: ('show_layout_buttons', 'Interface', 'Show Layout Buttons'), - now tracked in monitor_gui_preferences
        # REMOVED: ('show_actions_button', 'Interface', 'Show Actions Button'), - duplicate of show_sb_all_actions_button
    ]

    # --- Track these settings from global.py.json ---
    tracked_global_json_settings = [
        ('output_format', 'File Formats', 'Default Output Format', 'Preferences ➔ Behavior ➔ Default actions'),
        ('input_format_order', 'File Formats', 'Input Format Order', 'Preferences ➔ Behavior'),
    ]
    global_json_path = os.path.join(config_dir, 'global.py.json')
    last_global_states = gprefs.get('calibre_config_reports_last_global_states', {})
    try:
        with open(global_json_path, 'r', encoding='utf-8') as f:
            global_json = json.load(f)
    except Exception:
        global_json = {}
    for key, category, display_name, ui_location in tracked_global_json_settings:
        current_value = global_json.get(key, None)
        old_value = last_global_states.get(key, None)
        # For input_format_order, display as comma-separated string
        if key == 'input_format_order':
            if isinstance(current_value, list):
                current_value_str = ', '.join(map(str, current_value))
            else:
                current_value_str = str(current_value)
            if isinstance(old_value, list):
                old_value_str = ', '.join(map(str, old_value))
            else:
                old_value_str = str(old_value)
        else:
            current_value_str = str(current_value)
            old_value_str = str(old_value)
        if current_value != old_value:
            HistoryLogger.log_event('setting_changed', {
                'category': category,
                'setting': display_name,
                'new_value': current_value_str,
                'old_value': old_value_str,
                'source': 'global.py.json',
                'ui_location': ui_location,
                'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            })
            last_global_states[key] = current_value
            changed = True
    if changed:
        gprefs['calibre_config_reports_last_global_states'] = last_global_states
    gui_json_path = os.path.join(config_dir, 'gui.json')
    tweaks_json_path = os.path.join(config_dir, 'tweaks.json')
    last_extra_states = gprefs.get('calibre_config_reports_last_extra_states', {})
    try:
        with open(gui_json_path, 'r', encoding='utf-8') as f:
            gui_json = json.load(f)
    except Exception:
        gui_json = {}
    for key, category, display_name in tracked_gui_json_settings:
        current_value = gui_json.get(key, None)
        old_value = last_extra_states.get(key, None)
        # Special handling for font (list to string)
        if key == 'font' and isinstance(current_value, list):
            current_value_str = ', '.join(map(str, current_value))
        else:
            current_value_str = str(current_value)
        if key == 'font' and isinstance(old_value, list):
            old_value_str = ', '.join(map(str, old_value))
        else:
            old_value_str = str(old_value)
        if current_value != old_value:
            HistoryLogger.log_event('setting_changed', {
                'category': category,
                'setting': display_name,
                'new_value': current_value_str,
                'old_value': old_value_str,
                'source': 'gui.json',
                'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            })
            last_extra_states[key] = current_value
            changed = True
    for key, category, display_name in tracked_gui_settings:
        if key == 'main_window_geometry':
            # Special handling: geometry-of-calibre_main_window_geometry
            geom = gui_json.get('geometry-of-calibre_main_window_geometry', {})
            current_value = geom.get('frame_geometry', None)
            old_value = last_extra_states.get('main_window_geometry', None)
            if current_value != old_value:
                HistoryLogger.log_event('setting_changed', {
                    'category': category,
                    'setting': display_name,
                    'new_value': str(current_value),
                    'old_value': str(old_value),
                    'source': 'gui.json',
                    'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                })
                last_extra_states['main_window_geometry'] = current_value
                changed = True
        else:
            # For gui_layout, check both gui_layout and main_window_central_widget_state['layout']
            if key == 'gui_layout':
                # Prefer main_window_central_widget_state['layout'] if present
                widget_state = gui_json.get('main_window_central_widget_state', {})
                current_value = widget_state.get('layout', gui_json.get('gui_layout', None))
            else:
                current_value = gui_json.get(key, None)
            old_value = last_extra_states.get(key, None)
            if current_value != old_value:
                display_val = str(current_value)
                HistoryLogger.log_event('setting_changed', {
                    'category': category,
                    'setting': display_name,
                    'new_value': display_val,
                    'old_value': str(old_value),
                    'source': 'gui.json',
                    'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                })
                last_extra_states[key] = current_value
                changed = True
    # --- Tweaks: author_sort_copy_method ---
    try:
        with open(tweaks_json_path, 'r', encoding='utf-8') as f:
            tweaks_json = json.load(f)
    except Exception:
        tweaks_json = {}
    key = 'author_sort_copy_method'
    current_value = tweaks_json.get(key, None)
    old_value = last_extra_states.get(key, None)
    # Special display: show 'comma' if value is None or []
    def author_sort_display(val):
        if val is None or val == []:
            return 'comma'
        return str(val)
    if current_value != old_value:
        HistoryLogger.log_event('setting_changed', {
            'category': 'Metadata Processing',
            'setting': 'Author Sort Copy Method',
            'new_value': author_sort_display(current_value),
            'old_value': author_sort_display(old_value),
            'source': 'tweaks.json',
            'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        })
        last_extra_states[key] = current_value
        changed = True
    # Track UI language from global.py.json
    global_json_path = os.path.join(config_dir, 'global.py.json')
    key = 'language'
    try:
        with open(global_json_path, 'r', encoding='utf-8') as f:
            global_json = json.load(f)
            current_value = global_json.get(key, None)
    except Exception:
        current_value = None
    old_value = gprefs.get('calibre_config_reports_last_language', None)
    if current_value != old_value:
        HistoryLogger.log_event('setting_changed', {
            'category': 'System',
            'setting': 'UI Language',
            'new_value': str(current_value),
            'old_value': str(old_value),
            'source': 'global.py.json',
            'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        })
        gprefs['calibre_config_reports_last_language'] = current_value
    # Track these settings from gui.py.json
    tracked_gui_py_json_settings = [
        ('disable_tray_notification', 'System Tray', 'Disable Tray Notification'),
    ]
    gui_py_json_path = os.path.join(config_dir, 'gui.py.json')
    try:
        with open(gui_py_json_path, 'r', encoding='utf-8') as f:
            gui_py_json = json.load(f)
    except Exception:
        gui_py_json = {}
    for key, category, display_name in tracked_gui_py_json_settings:
        current_value = gui_py_json.get(key, None)
        old_value = last_extra_states.get(key, None)
        if current_value != old_value:
            HistoryLogger.log_event('setting_changed', {
                'category': category,
                'setting': display_name,
                'new_value': str(current_value),
                'old_value': str(old_value),
                'source': 'gui.py.json',
                'timestamp': datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            })
            last_extra_states[key] = current_value
            changed = True
    if changed:
        gprefs['calibre_config_reports_last_extra_states'] = last_extra_states
