#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

# --- Plugin table population logic moved from main.py ---
from calibre.customize.ui import initialized_plugins, PluginInstallationType
from calibre.gui2 import gprefs
from calibre.utils.localization import _
from calibre_plugins.calibre_config_reports.size_utils import format_size
from calibre_plugins.calibre_config_reports.date_utils import format_datetime
from . import common_icons
import os, datetime, zipfile, sys
from PyQt5.QtGui import QIcon, QPixmap

# File stats cache for performance with many plugins
_file_stats_cache = None

def get_cached_file_stats(plugin_path):
    """Get cached file stats or read from disk if changed/missing"""
    global _file_stats_cache

    # Initialize cache if needed
    if _file_stats_cache is None:
        _file_stats_cache = gprefs.get('ccr_plugin_file_stats_cache', {})

    if not plugin_path or not os.path.exists(plugin_path):
        return None

    try:
        # Get current file mtime
        current_mtime = os.path.getmtime(plugin_path)
        cache_key = f"{plugin_path}|{current_mtime}"

        # Check if we have valid cached data
        if cache_key in _file_stats_cache:
            cached_data = _file_stats_cache[cache_key]
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][CACHE] Hit for {os.path.basename(plugin_path)}")
            return cached_data

        # Read fresh data from disk
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][CACHE] Miss for {os.path.basename(plugin_path)}, reading from disk")

        stats = os.stat(plugin_path)

        # Calculate install date
        if iswindows:
            install_time = datetime.datetime.fromtimestamp(stats.st_ctime)
        else:
            install_time = datetime.datetime.fromtimestamp(stats.st_mtime)
        install_date = install_time.strftime('%Y-%m-%d %H:%M')

        # Calculate local modified
        mtime = stats.st_mtime
        local_modified_dt = datetime.datetime.fromtimestamp(mtime)
        local_modified = format_datetime(local_modified_dt)

        # Calculate size
        size_bytes = stats.st_size
        formatted_size = format_size(size_bytes)

        # Cache the results
        cached_data = {
            'install_date': install_date,
            'local_modified': local_modified,
            'size_bytes': size_bytes,
            'formatted_size': formatted_size
        }

        # Store in cache and clean old entries for this file
        old_keys = [k for k in _file_stats_cache.keys() if k.startswith(f"{plugin_path}|")]
        for old_key in old_keys:
            _file_stats_cache.pop(old_key, None)

        _file_stats_cache[cache_key] = cached_data

        # Periodically save cache (every 10 entries to avoid excessive writes)
        if len(_file_stats_cache) % 10 == 0:
            try:
                gprefs['ccr_plugin_file_stats_cache'] = _file_stats_cache
            except Exception:
                pass

        return cached_data

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][CACHE] Error reading stats for {plugin_path}: {e}")
        return None

def save_file_stats_cache():
    """Save the file stats cache to gprefs"""
    global _file_stats_cache
    if _file_stats_cache is not None:
        try:
            gprefs['ccr_plugin_file_stats_cache'] = _file_stats_cache
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][CACHE] Saved {len(_file_stats_cache)} cached file stats")
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][CACHE] Failed to save cache: {e}")
from PyQt5.QtWidgets import QTableWidgetItem, QComboBox, QLabel, QHBoxLayout
from PyQt5.QtCore import Qt, QSize, QTimer
import dateutil.parser

# Import for category filtering functionality
from calibre.constants import numeric_version
from calibre.utils.https import get_https_resource_securely
from calibre.utils.config import JSONConfig
import json, bz2, traceback, re

# Debug controlled by environment variable so `CCR_DEBUG_ENABLED=1` enables debug
import os as _ccr_os
CCR_DEBUG_ENABLED = _ccr_os.environ.get('CCR_DEBUG_ENABLED', '0').lower() in ('1', 'true', 'yes')

# Try to use calibre's prints logger, fall back to builtins.print
try:
    from calibre.utils.logging import prints as _calibre_prints
except Exception:
    _calibre_prints = __import__('builtins').print

if CCR_DEBUG_ENABLED:
    # Use calibre's prints (or builtins.print if unavailable)
    prints = _calibre_prints
    prints("[CCR][DEBUG] calibre_config_reports.plugins_tab loaded")
else:
    # Silent no-op in production
    def prints(*args, **kwargs):
        return None

# Category filtering constants and functions
CATEGORY_SERVER = 'https://code.calibre-ebook.com/plugins/'
CATEGORY_INDEX_URL = f'{CATEGORY_SERVER}plugins.json.bz2'

# Session-based cache to avoid repeated network requests
_session_category_cache = None
_session_index_cache = None
_session_fetch_attempted = False

# Persistent cache for categories (lives under calibre config in CCR namespace)
category_cache = JSONConfig('plugins/calibre_config_reports/ccr_category_cache')

if CCR_DEBUG_ENABLED:
    prints(f"[CCR][DEBUG] Category cache initialized: {category_cache}")
    try:
        test_data = category_cache.get('test', 'default')
        prints(f"[CCR][DEBUG] Category cache test access successful: {test_data}")
    except Exception as e:
        prints(f"[CCR][DEBUG] Category cache test access failed: {e}")

# Human-readable category names mapping
CATEGORY_HUMAN_NAMES = {
    'Store': _('Sources of books'),
    'FileType': _('Customize handling of ebooks'),
    'LibraryClosed': _('Actions when closing libraries'),
    'Editor': _('Extend the calibre Editor'),
    'ConversionInput': _('Conversion from extra formats'),
    'ConversionOutput': _('Conversion to extra formats'),
    'Device': _('Devices to manage'),
    'MetadataWriter': _('Set metadata in files'),
    'MetadataReader': _('Get metadata from files'),
    'MetadataSource': _('Download metadata for books'),
    'UserInterface': _('Extend calibre generally'),
    # Some upstream code / indexes use the short form 'GUI' for the same category.
    # Ensure we map that to the same human-readable label so CCR shows the
    # friendly name instead of the raw 'GUI' token.
    'GUI': _('Extend calibre generally'),
}


def get_human_category_name(cat):
    """Return a human-readable name for a category key.

    Handles variant keys such as 'LibraryClosed' and 'Library Closed' by
    normalizing both sides (remove non-alphanumerics, lowercasing) and
    falling back to the original category string when no mapping exists.
    """
    if not cat:
        return ''
    # Try direct lookup first
    try:
        human = CATEGORY_HUMAN_NAMES.get(cat)
        if human:
            return human
    except Exception:
        pass

    # Normalized matching: strip non-alphanumeric and compare lowercased
    import re
    def _norm(s):
        return re.sub(r'[^0-9a-z]+', '', str(s).lower())

    target = _norm(cat)
    for k, v in CATEGORY_HUMAN_NAMES.items():
        try:
            if _norm(k) == target:
                return v
        except Exception:
            continue

    # Fallback to the original category string
    return cat

def _extract_categories_from_index(index_data):
    """Given the full plugin index mapping (name -> dict), return a flat
    mapping of plugin-key -> category. Adds both exact and lowercase keys.
    """
    categories = {}
    if isinstance(index_data, dict):
        for plugin_name, plugin_data in index_data.items():
            if isinstance(plugin_data, dict):
                category = plugin_data.get('category', '')
                if category:
                    key = plugin_data.get('index_name', plugin_name)
                    if key:
                        categories[key] = category
                        categories[str(key).lower()] = category
    return categories

def _coerce_categories_shape(obj):
    """Coerce any stored cache shape to the expected flat dict[str, str].
    Accepted inputs:
    - Flat dict of key->category string (preferred)
    - Dict-of-dicts shaped like the online index (will be converted)
    - Anything else returns {}
    """
    if not isinstance(obj, dict):
        return {}
    # If values are strings, assume already flat mapping
    if obj and all(isinstance(v, str) for v in obj.values()):
        # Only keep string values that are non-empty
        return {str(k): v for k, v in obj.items() if isinstance(v, str) and v}
    # If values look like index entries (dicts), convert using extractor
    if obj and all(isinstance(v, dict) for v in obj.values()):
        return _extract_categories_from_index(obj)
    # Mixed/unknown shapes -> best effort: try to pull category fields
    coerced = {}
    for k, v in obj.items():
        if isinstance(v, str) and v:
            coerced[str(k)] = v
        elif isinstance(v, dict):
            cat = v.get('category', '')
            key = v.get('index_name', k)
            if cat and key:
                coerced[str(key)] = cat
                coerced[str(key).lower()] = cat
    return coerced

def fetch_plugin_categories():
    """Fetch plugin categories and full index from the online plugin index (cached per session)."""
    global _session_category_cache, _session_index_cache, _session_fetch_attempted

    # Return cached result if available
    if _session_category_cache is not None:
        return _session_category_cache

    # If we've already tried and failed this session, don't try again
    if _session_fetch_attempted:
        return {}

    # Mark that we've attempted a fetch this session
    _session_fetch_attempted = True

    # First, try to read existing categories/index from persistent cache (flat map + raw index)
    try:
        raw_cached = category_cache.get('categories', {})
        existing_categories = _coerce_categories_shape(raw_cached)
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Read {len(existing_categories)} cached categories (post-coerce)")
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Failed to read category cache: {e}")
        existing_categories = {}
    try:
        existing_index = category_cache.get('index', {})
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Read {len(existing_index) if isinstance(existing_index, dict) else 0} plugins from cached index")
    except Exception:
        existing_index = {}

    categories = {}
    try:
        if CCR_DEBUG_ENABLED:
            prints("[CCR][DEBUG] Fetching plugin categories from server (once per session)...")
        raw = get_https_resource_securely(CATEGORY_INDEX_URL, timeout=10)
        if not raw:
            raise RuntimeError('Empty response')
        data = json.loads(bz2.decompress(raw))
        _session_index_cache = data if isinstance(data, dict) else {}
        prints(f"[CCR][DEBUG] Successfully fetched and decompressed online index, {len(_session_index_cache)} plugins found.")
        categories = _extract_categories_from_index(_session_index_cache)
    except Exception as e:
        prints(f"[CCR][DEBUG] Online fetch failed: {e}")
        # Fall back to existing cached flat map
        categories = existing_categories
        _session_index_cache = existing_index if isinstance(existing_index, dict) else {}
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Using {len(categories)} categories and {len(_session_index_cache)} plugins from existing cache")

    prints(f"[CCR][DEBUG] Extracted {len(categories)} categories.")
    # Cache the result for the entire session
    _session_category_cache = categories
    # Persist to disk for offline availability, but only if we have data
    if categories or _session_index_cache:
        try:
            # Do not clear; just update the expected keys
            category_cache.update({
                'categories': categories,
                'index': _session_index_cache or existing_index,
                'updated_at': datetime.datetime.utcnow().isoformat() + 'Z'
            })
            # Ensure the cache is saved
            try:
                category_cache.save()
            except:
                pass
            prints(f"[CCR][DEBUG] Saving {len(categories)} categories and {len(_session_index_cache or {})} index entries to persistent cache...")
            # Explicitly flush to disk and verify file creation
            cache_path = getattr(category_cache, 'path', None)
            if not cache_path:
                from calibre.utils.config import config_dir
                cache_path = os.path.join(config_dir, 'plugins', 'calibre_config_reports', 'ccr_category_cache.json')
            prints(f"[CCR][DEBUG] Category cache file path: {cache_path}")
            import json as _json
            # Always save to ensure the cache is updated
            try:
                parent_dir = os.path.dirname(cache_path)
                if not os.path.exists(parent_dir):
                    os.makedirs(parent_dir, exist_ok=True)
                with open(cache_path, 'w', encoding='utf-8') as f:
                    _json.dump({
                        'categories': categories,
                        'index': _session_index_cache or existing_index,
                        'updated_at': datetime.datetime.utcnow().isoformat() + 'Z'
                    }, f, indent=2)
                prints(f"[CCR][DEBUG] Category cache file forcibly saved, size: {os.path.getsize(cache_path)} bytes")
            except Exception as e2:
                prints(f"[CCR][DEBUG] Failed to forcibly save category cache file: {e2}")
            else:
                prints(f"[CCR][DEBUG] Category cache file exists, size: {os.path.getsize(cache_path)} bytes")
        except Exception as e:
            prints(f"[CCR][DEBUG] Failed to save category cache: {e}")
            pass
    else:
        if CCR_DEBUG_ENABLED:
            prints("[CCR][DEBUG] No new categories to save, keeping existing cache")
    prints(f"[CCR][DEBUG] Successfully cached {len(categories)} plugin categories")
    return categories

def get_available_categories(plugin_categories):
    """Get list of available categories from fetched data"""
    categories = set()
    for category in plugin_categories.values():
        if category:
            categories.add(category)
    return sorted(list(categories))

class PluginsTableHelperMixin:

    def populate_plugins_table(self):
        """Populate the plugins table with data"""
        plugins = list(initialized_plugins())
        # Suppress UI updates and sorting during batch population
        try:
            self.plugin_table.setSortingEnabled(False)
            self.plugin_table.viewport().setUpdatesEnabled(False)
        except Exception:
            pass
        plugins.sort(key=lambda x: x.name.lower())
        # ... (rest of the function body should be moved here from main.py)

    def setup_category_filter(self):
        """Setup category filtering dropdown if Calibre version >= 8.9"""
        if numeric_version < (8, 9, 0):
            # Hide category dropdown for older Calibre versions
            if hasattr(self, 'category_filter_dropdown'):
                self.category_filter_dropdown.setVisible(False)
                self.category_label.setVisible(False)
            return

        # Show category controls for Calibre >= 8.9
        if hasattr(self, 'category_filter_dropdown'):
            self.category_filter_dropdown.setVisible(True)
            self.category_label.setVisible(True)
            # Seed with safe default to avoid a blank, non-responsive dropdown
            if self.category_filter_dropdown.count() == 0:
                self.category_filter_dropdown.addItem(_('All Categories'))
                # Disabled until we load real categories
                self.category_filter_dropdown.setEnabled(False)
                self.category_filter_dropdown.setToolTip(_('Categories will load automatically (offline cache used if available)'))

            # Try to synchronously seed from persistent cache so Category column works offline
            try:
                if CCR_DEBUG_ENABLED:
                    prints("[CCR][DEBUG] Attempting to read category cache...")
                cached_raw = category_cache.get('categories', {})
                cached = _coerce_categories_shape(cached_raw)
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Read {len(cached)} categories from cache (post-coerce)")
            except Exception as e:
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Failed to read category cache: {e}")
                cached = {}
            if cached:
                # Store map for immediate use by populate_plugins_table
                self.plugin_categories = cached
                cats = get_available_categories(cached)
                if cats:
                    self.category_filter_dropdown.clear()
                    self.category_filter_dropdown.addItem(_('All Categories'), None)
                    # Use 'Not in Index' nomenclature
                    self.category_filter_dropdown.addItem(_('Not in Index'), '__not_in_index__')
                    # Move 'Extend calibre generally' (UserInterface/GUI) to second position
                    extend_cat = None
                    for cat in cats:
                        if cat in ('UserInterface', 'GUI'):
                            extend_cat = cat
                    if extend_cat:
                        human = get_human_category_name(extend_cat)
                        self.category_filter_dropdown.addItem(human, extend_cat)
                        # Remove it from the list so we don't add it twice below
                        cats = [c for c in cats if c != extend_cat]
                        # If the Editor category exists, place it immediately after the
                        # 'Extend calibre generally' entry so related UI plugins are grouped.
                        if 'Editor' in cats:
                            editor_human = get_human_category_name('Editor')
                            self.category_filter_dropdown.addItem(editor_human, 'Editor')
                            cats = [c for c in cats if c != 'Editor']
                    for cat in cats:
                        human = get_human_category_name(cat)
                        self.category_filter_dropdown.addItem(human, cat)
                    self.category_filter_dropdown.setEnabled(True)
                    self.category_filter_dropdown.setToolTip('')
                else:
                    self.category_filter_dropdown.setEnabled(False)
                    self.category_filter_dropdown.setToolTip(_('No categories available (offline and no cache).'))
                # Update visible rows now if table already populated
                try:
                    self._refresh_category_cells()
                    # Also refresh metadata cells using any cached full index
                    self._build_index_lookup()
                    self._refresh_metadata_cells()
                except Exception:
                    pass
            else:
                # Ensure the cache file exists even if empty (helps diagnostics)
                try:
                    if CCR_DEBUG_ENABLED:
                        prints("[CCR][DEBUG] Creating empty category cache file...")
                    category_cache.clear()
                    category_cache.update({'categories': {}, 'updated_at': datetime.datetime.utcnow().isoformat() + 'Z'})
                    cache_path = getattr(category_cache, 'path', None)
                    if not cache_path:
                        from calibre.utils.config import config_dir
                        cache_path = os.path.join(config_dir, 'plugins', 'calibre_config_reports', 'ccr_category_cache.json')
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Category cache file path: {cache_path}")
                    if os.path.exists(cache_path):
                        prints(f"[CCR][DEBUG] Category cache file exists, size: {os.path.getsize(cache_path)} bytes")
                    else:
                        prints(f"[CCR][DEBUG] Category cache file NOT found after save!")
                except Exception as e:
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Failed to create empty category cache: {e}")
                    pass
        # Defer any network work to avoid slowing down dialog opening
        QTimer.singleShot(0, self.load_categories_async)

    def load_categories_async(self):
        """Load categories after dialog is shown, preferring cache immediately, then fetch online in background"""
        try:
            # Use persistent/session cache immediately to populate UI fast
            plugin_categories = {}
            try:
                if _session_category_cache is not None:
                    plugin_categories = _session_category_cache
                else:
                    cached_raw = category_cache.get('categories', {})
                    plugin_categories = _coerce_categories_shape(cached_raw)

                # Also load cached index to ensure metadata is available immediately
                cached_index = _session_index_cache if isinstance(_session_index_cache, dict) else category_cache.get('index', {})
                if cached_index and isinstance(cached_index, dict):
                    # Update plugin cache with cached index data
                    if hasattr(self.parent_dialog, 'plugin_cache'):
                        self.parent_dialog.plugin_cache.update(cached_index)
                        self.plugin_cache = self.parent_dialog.plugin_cache
                        # Rebuild lookups with fresh data
                        self._build_plugin_cache_lookup()
                        self._build_index_lookup()
                        if CCR_DEBUG_ENABLED:
                            prints(f"[CCR][DEBUG] Loaded {len(cached_index)} index entries from cache in load_categories_async")
            except Exception as e:
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Error loading cached data in load_categories_async: {e}")
                plugin_categories = {}

            available_categories = get_available_categories(plugin_categories)

            if available_categories:
                # Update dropdown with real categories
                current_selection = self.category_filter_dropdown.currentText()
                self.category_filter_dropdown.clear()
                self.category_filter_dropdown.addItem(_('All Categories'), None)
                # Use unified nomenclature: 'Not in Index'
                self.category_filter_dropdown.addItem(_('Not in Index'), '__not_in_index__')
                # Move 'Extend calibre generally' (UserInterface/GUI) to second position if present
                extend_cat = None
                for cat in list(available_categories):
                    if cat in ('UserInterface', 'GUI'):
                        extend_cat = cat
                        break
                if extend_cat:
                    human = get_human_category_name(extend_cat)
                    self.category_filter_dropdown.addItem(human, extend_cat)
                    # Remove it from the list so we don't add it twice below
                    available_categories = [c for c in available_categories if c != extend_cat]
                    # If the Editor category exists, place it immediately after the
                    # 'Extend calibre generally' entry so related UI plugins are grouped.
                    if 'Editor' in available_categories:
                        editor_human = get_human_category_name('Editor')
                        self.category_filter_dropdown.addItem(editor_human, 'Editor')
                        available_categories = [c for c in available_categories if c != 'Editor']
                for cat in available_categories:
                    human = get_human_category_name(cat)
                    self.category_filter_dropdown.addItem(human, cat)
                self.category_filter_dropdown.setEnabled(True)
                self.category_filter_dropdown.setToolTip('')

                # Restore previous selection if it still exists
                index = self.category_filter_dropdown.findText(current_selection)
                if index >= 0:
                    self.category_filter_dropdown.setCurrentIndex(index)

                # Store mapping for filtering
                self.plugin_categories = plugin_categories
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Updated category filter with {len(available_categories)} categories")
                # Refresh Category column cells in-place
                try:
                    self._refresh_category_cells()
                    # Also refresh metadata cells using cached index
                    self._build_index_lookup()
                    self._refresh_metadata_cells()
                except Exception:
                    pass
            else:
                # If no categories in cache, show default item disabled for now
                self.plugin_categories = {}
                if self.category_filter_dropdown.count() == 0:
                    self.category_filter_dropdown.addItem(_('All Categories'))
                self.category_filter_dropdown.setEnabled(False)
                self.category_filter_dropdown.setToolTip(_('No categories available yet. Will update automatically.'))
                if CCR_DEBUG_ENABLED:
                    prints("[CCR][DEBUG] No cached categories, showing placeholder and scheduling background fetch")

            # Create and start background fetcher
            self._fetcher = BackgroundFetcher()
            self._fetcher.fetched.connect(self._on_background_fetch_complete)
            self._fetcher.start()
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Failed to load categories async: {e}")
            # If async loading fails, continue with empty categories
            self.plugin_categories = {}
    def _on_background_fetch_complete(self, cats, idx):
        """Handle completion of background fetch (runs in main thread)"""
        try:
            # Update categories dropdown if we received categories
            if cats:
                available = get_available_categories(cats)
                if available:
                    current_selection = self.category_filter_dropdown.currentText()
                    self.category_filter_dropdown.clear()
                    self.category_filter_dropdown.addItem(_('All Categories'), None)
                    # Use unified nomenclature: 'Not in Index'
                    self.category_filter_dropdown.addItem(_('Not in Index'), '__not_in_index__')
                    # Move 'Extend calibre generally' (UserInterface/GUI) to second position if present
                    extend_cat = None
                    for cat in list(available):
                        if cat in ('UserInterface', 'GUI'):
                            extend_cat = cat
                            break
                    if extend_cat:
                        human = get_human_category_name(extend_cat)
                        self.category_filter_dropdown.addItem(human, extend_cat)
                        # Remove it from the list so we don't add it twice below
                        available = [c for c in available if c != extend_cat]
                        # If the Editor category exists, place it immediately after the
                        # 'Extend calibre generally' entry so related UI plugins are grouped.
                        if 'Editor' in available:
                            editor_human = get_human_category_name('Editor')
                            self.category_filter_dropdown.addItem(editor_human, 'Editor')
                            available = [c for c in available if c != 'Editor']
                    for cat in available:
                        human = get_human_category_name(cat)
                        self.category_filter_dropdown.addItem(human, cat)
                    self.category_filter_dropdown.setEnabled(True)
                    self.category_filter_dropdown.setToolTip('')
                    sel_idx = self.category_filter_dropdown.findText(current_selection)
                    if sel_idx >= 0:
                        self.category_filter_dropdown.setCurrentIndex(sel_idx)
                    self.plugin_categories = cats
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Background updated category filter with {len(available)} categories")

            # Update plugin cache with fresh index data if present
            if idx and hasattr(self.parent_dialog, 'plugin_cache'):
                try:
                    self.parent_dialog.plugin_cache.update(idx)
                    self.plugin_cache = self.parent_dialog.plugin_cache
                    self._build_plugin_cache_lookup()
                    self._build_index_lookup()
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Updated plugin cache with {len(idx)} entries")
                except Exception as e:
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Failed to update plugin cache: {e}")

            # Refresh UI metadata cells now that cache/index are updated
            try:
                self._refresh_category_cells()
                self._refresh_metadata_cells()
                if CCR_DEBUG_ENABLED:
                    prints("[CCR][DEBUG] Completed background fetch refresh")
            except Exception as e:
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Error refreshing UI after background fetch: {e}")

        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Error handling background fetch completion: {e}")

# --- Plugin preference helpers moved from main.py ---
from qt.core import QApplication, QObject, QEvent, QCheckBox, QTimer

# Helper to handle Enter key behavior without triggering default buttons elsewhere
class _EnterKeyFilter(QObject):
    def __init__(self, handler, parent=None):
        super().__init__(parent)
        self.handler = handler

    def eventFilter(self, obj, event):
        try:
            if event.type() == QEvent.KeyPress and event.key() in (Qt.Key_Return, Qt.Key_Enter):
                if callable(self.handler):
                    self.handler()
                # Swallow the event so it doesn't trigger default buttons
                return True
        except Exception:
            pass
        return False

class PluginPreferencesHelperMixin:
    def handle_checkbox(self, checkbox):
        """Handle the user installed plugins checkbox by checking it"""
        checkbox.setChecked(True)
        if hasattr(checkbox.parent(), 'filter'):
            checkbox.parent().filter()
        return True

    def open_plugin_preferences(self):
        """Open the plugin preferences dialog with user installed plugins checked"""
        app = QApplication.instance()

        # Create an event filter to catch the checkbox show event
        class PrefsEventFilter(QObject):
            def __init__(self, callback, parent=None):
                QObject.__init__(self, parent)
                self.callback = callback

            def eventFilter(self, obj, event):
                if isinstance(obj, QCheckBox) and event.type() == QEvent.Show:
                    if 'user installed' in obj.text().lower():
                        return self.callback(obj)
                return False

        # Install event filter with callback
        event_filter = PrefsEventFilter(self.handle_checkbox)
        app.installEventFilter(event_filter)

        # Open dialog
        prefs = self.gui.iactions['Preferences']
        prefs.do_config(
            initial_plugin=('Advanced', 'Plugins'),
            close_after_initial=True
        )

        # Cleanup after delay
        QTimer.singleShot(1000, lambda: app.removeEventFilter(event_filter))
# --- Plugin management methods moved from main.py ---
from calibre.gui2 import error_dialog, dynamic
from calibre.customize.ui import add_plugin, config, remove_plugin
from calibre.gui2.dialogs.plugin_updater import notify_on_successful_install
from qt.core import QFileDialog
import os

def get_downloads_directory():
    import sys
    if sys.platform.startswith('win'):
        try:
            import winreg
            with winreg.OpenKey(winreg.HKEY_CURRENT_USER, 'SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders') as key:
                return winreg.QueryValueEx(key, '{374DE290-123F-4565-9164-39C4925E467B}')[0]
        except:
            pass
    candidates = [
        os.path.join(os.path.expanduser('~'), 'Downloads'),
        os.path.join(os.path.expanduser('~'), 'downloads'),
        os.path.join(os.path.expanduser('~'), 'Desktop', 'Downloads'),
    ]
    for candidate in candidates:
        if os.path.exists(candidate) and os.path.isdir(candidate):
            return candidate
    return os.path.expanduser('~')

class PluginManagerMixin:
    def load_plugin_from_file(self):
        """Load plugin ZIP file(s) and install them using Calibre's API.

        Supports multiple selection. Shows toolbar placement prompt for GUI plugins
        and calls Calibre's notify_on_successful_install. Presents a summary dialog
        on completion.
        """
        if CCR_DEBUG_ENABLED:
            prints("[CCR][DEBUG] Invoked load_plugin_from_file (plugins_tab.py)")

        # Remember last used directory
        last_dir = dynamic.get('load_plugin_last_directory', get_downloads_directory())
        paths = QFileDialog.getOpenFileNames(self, _('Select plugin ZIP file(s)'), os.path.expanduser(last_dir), filter=_('Plugin files (*.zip)'))[0]
        if not paths:
            if CCR_DEBUG_ENABLED:
                prints("[CCR][DEBUG] No file(s) selected, aborting plugin load.")
            return

        # Save directory for next time
        dynamic['load_plugin_last_directory'] = os.path.dirname(paths[0])

        success_count = 0
        error_count = 0
        errors = []

        for path in paths:
            try:
                installed_plugins = frozenset(config['plugins'])
                plugin = add_plugin(path)

                # Show toolbar placement prompt for GUI plugins
                try:
                    from calibre.customize import InterfaceActionBase, EditBookToolPlugin
                    from calibre.gui2.preferences.plugins import ConfigWidget
                    is_gui_plugin = isinstance(plugin, (InterfaceActionBase, EditBookToolPlugin))
                    if is_gui_plugin:
                        previously_installed = plugin.name in installed_plugins
                        ConfigWidget.check_for_add_to_toolbars(self.parent_dialog, plugin, previously_installed=previously_installed)
                except Exception:
                    # Non-fatal: ignore toolbar prompt failures
                    pass

                # Notify Calibre of successful install and handle restart if required
                restart_needed = notify_on_successful_install(self, plugin)
                if restart_needed:
                    success_count += 1
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Restart required after installing '{getattr(plugin, 'name', path)}'. Quitting calibre...")
                    self.gui.quit(restart=True)
                    return

                success_count += 1

            except Exception as e:
                error_count += 1
                errors.append(f"Failed to load {os.path.basename(path)}: {str(e)}")

        # Show a summary message if any installs succeeded (and no restart occurred)
        if success_count > 0:
            summary_msg = _('Successfully loaded %d plugin(s).') % success_count
            if error_count > 0:
                summary_msg += '\n' + (_('%d plugin(s) failed to load.') % error_count)
                if errors:
                    summary_msg += '\n\n' + _('Errors:') + '\n' + '\n'.join(errors)
            try:
                info_dialog(self, _('Load plugins'), summary_msg)
            except Exception:
                # If info_dialog isn't available in the calling context, ignore
                pass

    def _remove_plugin_by_name(self, plugin_name):
        """Remove a plugin by name using Calibre's API."""
        try:
            removed = remove_plugin(plugin_name)
            if not removed:
                raise Exception(f"Plugin '{plugin_name}' could not be removed (may bebuiltin or not found).")
        except Exception as e:
            raise


# Consolidate imports: reuse imports from top; only import unique modules needed here
from calibre.utils.config import config_dir
from calibre.utils.icu import sort_key
from calibre import iswindows

try:
    from qt.core import (
        QWidget, QVBoxLayout, QHBoxLayout, QTableWidget, QTableWidgetItem,
        QHeaderView, QSize, QAbstractItemView, QIcon, QLineEdit, QLabel,
        QComboBox, QPushButton, QMenu, QTimer, QApplication, Qt, QToolButton,
        QAction, QKeySequence, QShortcut,
        pyqtSignal, QThread
    )
except ImportError:
    from PyQt5.Qt import (
        QWidget, QVBoxLayout, QHBoxLayout, QTableWidget, QTableWidgetItem,
        QHeaderView, QSize, QAbstractItemView, QIcon, QLineEdit, QLabel,
        QComboBox, QPushButton, QMenu, QTimer, QApplication, Qt, QToolButton, QAction, QKeySequence, QShortcut
    )
    from PyQt5.QtCore import pyqtSignal, QThread

from calibre.gui2 import gprefs, info_dialog, question_dialog, error_dialog

from calibre_plugins.calibre_config_reports.size_utils import get_file_size_bytes, format_size


class BackgroundFetcher(QThread):
    """Worker thread for fetching plugin data in background"""
    fetched = pyqtSignal(dict, dict)  # Signal emitting (categories, index)

    def run(self):
        try:
            if CCR_DEBUG_ENABLED:
                prints("[CCR][DEBUG] Background fetcher thread starting")
            cats = fetch_plugin_categories()  # This calls the network fetch
            idx = _session_index_cache if isinstance(_session_index_cache, dict) else {}
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Background fetch complete: {len(cats)} categories, {len(idx)} index entries")
            self.fetched.emit(cats, idx)
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Background fetch error: {e}")
            self.fetched.emit({}, {})

class SizeTableWidgetItem(QTableWidgetItem):
    """Custom QTableWidgetItem that sorts by raw size"""
    def __init__(self, raw_size, formatted_size):
        super().__init__(formatted_size)
        self.raw_size = raw_size

    def __lt__(self, other):
        if isinstance(other, SizeTableWidgetItem):
            return self.raw_size < other.raw_size
        return super().__lt__(other)


class PluginsTab(QWidget, PluginsTableHelperMixin, PluginPreferencesHelperMixin, PluginManagerMixin):
    """A separate tab class for the plugins functionality"""

    def __init__(self, parent_dialog):
        super().__init__()
        self.parent_dialog = parent_dialog
        self.gui = parent_dialog.gui

        # Initialize cache and index lookup maps
        self._cache_lookup = {}
        self._index_lookup = {}

        # Try to load any existing cache data immediately
        try:
            # Load cached categories
            if _session_category_cache is not None:
                self.plugin_categories = _session_category_cache
            else:
                cached_raw = category_cache.get('categories', {})
                self.plugin_categories = _coerce_categories_shape(cached_raw)

            # Load cached index
            cached_index = _session_index_cache if isinstance(_session_index_cache, dict) else category_cache.get('index', {})
            if cached_index and isinstance(cached_index, dict):
                # Update parent's plugin cache with any cached index data
                self.parent_dialog.plugin_cache.update(cached_index)

            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Initialized with {len(self.plugin_categories)} categories and {len(cached_index)} index entries from cache")
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Error loading initial cache data: {e}")
            self.plugin_categories = {}

        # Now initialize plugin cache from parent
        self.plugin_cache = parent_dialog.plugin_cache

        # Build lookup tables with any cached data we have
        self._build_plugin_cache_lookup()
        self._build_index_lookup()

        # Icons needed for the plugins tab
        self.forum_icon = parent_dialog.forum_icon
        self.donate_icon = parent_dialog.donate_icon
        self.history_icon = parent_dialog.history_icon
        self.copy_icon = parent_dialog.copy_icon

        # Set up the UI
        layout = QVBoxLayout(self)

        # Plugin type filter dropdown and count label in its own row
        filter_type_layout = QHBoxLayout()
        filter_type_label = QLabel(_('Show:'))
        self.filter_type = QComboBox()
        self.filter_type.addItems([_('All Plugins'), _('User Plugins Only')])
        self.filter_type.setCurrentIndex(1)  # Set to User Plugins Only by default
        self.filter_type.currentIndexChanged.connect(self.apply_type_filter)
        self.count_label = QLabel('')
        filter_type_layout.addWidget(filter_type_label)
        filter_type_layout.addWidget(self.filter_type)
        filter_type_layout.addWidget(self.count_label)
        filter_type_layout.addStretch()
        layout.addLayout(filter_type_layout)

        # Search filter in its own row
        filter_layout = QHBoxLayout()
        filter_label = QLabel(_('Filter:'))
        self.filter_edit = QLineEdit()
        self.filter_edit.setPlaceholderText(_('Search by Name, Author, or Description...'))
        self.filter_edit.setClearButtonEnabled(True)
        self.filter_edit.textChanged.connect(self.apply_plugins_filter)
        filter_layout.addWidget(filter_label)
        # Prevent Enter from triggering default buttons; handle return explicitly
        try:
            self.filter_edit.returnPressed.connect(self.apply_plugins_filter)
        except Exception:
            pass
        # Additionally, install a key filter to swallow Enter so it cannot trigger other default buttons
        try:
            self.filter_edit.installEventFilter(_EnterKeyFilter(lambda: self.apply_plugins_filter()))
        except Exception:
            pass
        filter_layout.addWidget(self.filter_edit)
        # Column manager button in filter row (Alt+M)
        self.column_button = QPushButton(_('&Manage Columns'))
        # Use compact styling consistent with bottom action buttons
        self.column_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.column_button.setToolTip(_('Show/hide columns'))
        self.column_button.setAutoDefault(False)
        self.column_button.setDefault(False)
        self.column_button.clicked.connect(self.show_column_menu)
        filter_layout.addWidget(self.column_button)

        # Category filter (only for Calibre >= 8.9)
        self.category_label = QLabel(_('Category:'))
        self.category_filter_dropdown = QComboBox()
        self.category_filter_dropdown.setMinimumWidth(150)
        self.category_filter_dropdown.currentTextChanged.connect(self.apply_category_filter)
        filter_layout.addWidget(self.category_label)
        filter_layout.addWidget(self.category_filter_dropdown)

        # Add export buttons in filter row with icons (use QAction-backed QToolButtons so mnemonics/shortcuts
        # are registered reliably by Qt's action system and work on first show)
        csv_action = QAction(_('&Save CSV'), self)
        csv_action.setIcon(QIcon.ic('save.png'))
        try:
            csv_action.setShortcut(QKeySequence(_('Alt+S')))
        except Exception:
            pass
        csv_action.setShortcutContext(Qt.WidgetShortcut)
        csv_action.setToolTip(_('Export visible plugins to CSV'))
        csv_action.triggered.connect(lambda: self.parent_dialog.export_data('csv'))
        try:
            self.parent_dialog.addAction(csv_action)
        except Exception:
            pass
        # Expose as attribute so parent dialog can re-register on showEvent
        try:
            self.csv_action = csv_action
        except Exception:
            pass
        csv_export_button = QPushButton('\u00A0' + _('&Save CSV'))
        # Use compact styling consistent with bottom action buttons
        csv_export_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        csv_export_button.setIcon(QIcon.ic('save.png'))
        csv_export_button.setToolTip(_('Export visible plugins to CSV'))
        csv_export_button.setAutoDefault(False)
        csv_export_button.setDefault(False)
        try:
            csv_export_button.setFocusPolicy(Qt.FocusPolicy.StrongFocus)
        except Exception:
            try:
                csv_export_button.setFocusPolicy(Qt.StrongFocus)
            except Exception:
                pass
        csv_export_button.clicked.connect(lambda: self.parent_dialog.export_data('csv'))
        csv_action.triggered.connect(lambda: self.parent_dialog.export_data('csv'))

        # Create XLSX export QAction (was missing, caused NameError)
        xlsx_action = QAction(_('&Export XLSX'), self)
        xlsx_action.setIcon(QIcon.ic('save.png'))
        try:
            xlsx_action.setShortcut(QKeySequence(_('Alt+X')))
        except Exception:
            pass
        xlsx_action.setShortcutContext(Qt.WidgetShortcut)
        xlsx_action.setToolTip(_('Export visible plugins to XLSX'))
        xlsx_action.triggered.connect(lambda: self.parent_dialog.export_data('xlsx'))
        try:
            self.parent_dialog.addAction(xlsx_action)
        except Exception:
            pass
        # Expose as attribute so parent dialog can re-register on showEvent
        try:
            self.xlsx_action = xlsx_action
        except Exception:
            pass
        xlsx_export_button = QPushButton('\u00A0' + _('Export &XLSX'))
        # Use compact styling consistent with bottom action buttons
        xlsx_export_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        xlsx_export_button.setIcon(QIcon.ic('save.png'))
        xlsx_export_button.setToolTip(_('Export visible plugins to XLSX'))
        xlsx_export_button.setAutoDefault(False)
        xlsx_export_button.setDefault(False)
        try:
            xlsx_export_button.setFocusPolicy(Qt.FocusPolicy.StrongFocus)
        except Exception:
            try:
                xlsx_export_button.setFocusPolicy(Qt.StrongFocus)
            except Exception:
                pass
        xlsx_export_button.clicked.connect(lambda: self.parent_dialog.export_data('xlsx'))
        xlsx_action.triggered.connect(lambda: self.parent_dialog.export_data('xlsx'))

        # Create Restore Defaults QAction (ensure it's defined before use)
        restore_action = QAction(_('&Reset Columns'), self)
        try:
            restore_action.setShortcut(QKeySequence(_('Alt+R')))
        except Exception:
            pass
        restore_action.setShortcutContext(Qt.WidgetShortcut)
        restore_action.setToolTip(_('Restore default column visibility and ordering'))
        try:
            self.parent_dialog.addAction(restore_action)
        except Exception:
            pass
        # Expose as attribute so parent dialog can re-register on showEvent
        try:
            self.restore_action = restore_action
        except Exception:
            pass
        restore_defaults_button = QPushButton('\u00A0' + _('&Reset Columns'))
        # Use compact styling consistent with bottom action buttons
        restore_defaults_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        restore_defaults_button.setIcon(QIcon.ic('view-refresh.png'))
        restore_defaults_button.setToolTip(_('Restore default column visibility and ordering'))
        restore_defaults_button.setAutoDefault(False)
        restore_defaults_button.setDefault(False)
        try:
            restore_defaults_button.setFocusPolicy(Qt.FocusPolicy.StrongFocus)
        except Exception:
            try:
                restore_defaults_button.setFocusPolicy(Qt.StrongFocus)
            except Exception:
                pass
        restore_defaults_button.clicked.connect(self.restore_default_columns)
        restore_action.triggered.connect(self.restore_default_columns)

        filter_layout.addWidget(csv_export_button)
        filter_layout.addWidget(xlsx_export_button)
        filter_layout.addWidget(restore_defaults_button)

        layout.addLayout(filter_layout)        # Plugins table
        self.plugin_table = QTableWidget()
        self.setup_plugins_table()

        # Setup category filtering (only for Calibre >= 8.9)
        self.setup_category_filter()

        # Instead of adding the table directly, create a vertical splitter so
        # we can show a collapsible info panel below the table. The action
        # buttons (Remove, Load, Updater, etc.) remain below the splitter so
        # they're always visible.
        try:
            from qt.core import QSplitter, QTextBrowser
        except Exception:
            from PyQt5.QtWidgets import QSplitter, QTextBrowser

        self.splitter = QSplitter(Qt.Vertical)
        self.splitter.addWidget(self.plugin_table)

        # Info panel (initially hidden)
        self.info_panel = QWidget()
        self.info_panel_layout = QVBoxLayout(self.info_panel)
        self.info_panel_layout.setContentsMargins(6, 6, 6, 6)
        self.info_panel_layout.setSpacing(6)

        # Description browser (read-only)
        # QTextBrowser import already handled above; just construct the widget
        self.desc_browser = QTextBrowser()
        self.desc_browser.setOpenExternalLinks(False)
        self.desc_browser.setReadOnly(True)
        self.desc_browser.setFrameStyle(0)
        # Ensure the info pane isn't visually empty; set a minimum height so the area is noticeable
        try:
            self.desc_browser.setMinimumHeight(80)
        except Exception:
            pass
        # Helper to show a friendly hint when nothing is selected
        def _show_no_selection_hint():
            try:
                title = _('No plugin selected')
                hint = _('Please select a plugin to display its info')
                html = '<div style="padding:12px;"><b>{}</b><div style="margin-top:6px;">{}</div></div>'.format(title, hint)
                self.desc_browser.setHtml(html)
            except Exception:
                try:
                    self.desc_browser.setText(_('Please select a plugin to display its info'))
                except Exception:
                    self.desc_browser.setHtml('')

        # Show the hint by default so the area isn't blank when the info panel is opened with no selection
        _show_no_selection_hint()
        self.info_panel_layout.addWidget(self.desc_browser)

        # Forum label (clickable link) shown under the description
        self.forum_label = QLabel('')
        self.forum_label.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        self.forum_label.setOpenExternalLinks(False)
        self.forum_label.setWordWrap(True)
        self.info_panel_layout.addWidget(self.forum_label)

        # Hide by default
        self.info_panel.setVisible(False)

        self.splitter.addWidget(self.info_panel)
        # Ensure initial sizes favour the table
        try:
            self.splitter.setSizes([600, 200])
        except Exception:
            pass

        layout.addWidget(self.splitter)

        # Add buttons below the plugins table (splitter sits above)
        button_layout = QHBoxLayout()
        button_layout.setSpacing(16)  # Add spacing between buttons
        button_layout.setContentsMargins(16, 12, 16, 12)  # Add more padding around the button row

        # Add icons to buttons using custom plugin icons
        from . import common_icons

        # Info pane toggle - place first in the button row so it appears on the left
        # Create QAction for Alt+I accelerator
        info_action = QAction(QIcon.ic('dialog_information.png'), _('Show Plugin &Info'), self)
        info_action.setShortcut(QKeySequence(_('Alt+I')))
        info_action.setShortcutContext(Qt.WidgetShortcut)
        info_action.setCheckable(True)
        info_action.setChecked(False)

        try:
            # Use a QPushButton (icon passed into constructor) so spacing and
            # visual appearance match the other action buttons in this row.
            # Make it checkable so it behaves like a toggle.
            info_btn_icon = None
            try:
                info_btn_icon = QIcon.ic('dialog_information.png')
            except Exception:
                try:
                    info_btn_icon = common_icons.get_icon('images/info_icon')
                except Exception:
                    info_btn_icon = None

            # Prefix label with NBSP so icon and caption have consistent spacing
            label_text = '\u00A0' + _('Show Plugin &Info')
            if info_btn_icon:
                self.info_toggle_btn = QPushButton(info_btn_icon, label_text)
            else:
                self.info_toggle_btn = QPushButton(label_text)

            self.info_toggle_btn.setCheckable(True)
            self.info_toggle_btn.setChecked(False)
            self.info_toggle_btn.setAutoDefault(False)
            self.info_toggle_btn.setDefault(False)

            # Clicking toggles visibility of the info panel
            def _toggle_info(checked):
                try:
                    self.info_panel.setVisible(bool(checked))
                    # Update toggle text/icon and sync action state
                    new_text = '\u00A0' + (_('Hide Plugin &Info') if checked else _('Show Plugin &Info'))
                    try:
                        self.info_toggle_btn.setText(new_text)
                    except Exception:
                        pass
                    info_action.setChecked(checked)
                    info_action.setText(_('Hide Plugin &Info') if checked else _('Show Plugin &Info'))
                    # If panel is being shown and no valid selection exists, show the helpful hint
                    if checked:
                        try:
                            cur_row = self.plugin_table.currentRow() if hasattr(self, 'plugin_table') else -1
                            if cur_row is None or cur_row < 0:
                                # Reuse the selection handler to ensure consistent behavior
                                try:
                                    self._on_plugin_selection_changed(-1)
                                except Exception:
                                    _show_no_selection_hint()
                        except Exception:
                            _show_no_selection_hint()
                except Exception:
                    pass

            # Wire signals
            try:
                self.info_toggle_btn.toggled.connect(_toggle_info)
                info_action.toggled.connect(_toggle_info)
            except Exception:
                pass

            # Match styling used in Shortcuts tab for a compact, consistent look
            try:
                self.info_toggle_btn.setStyleSheet('padding: 6px 12px; font-size:9pt;')
            except Exception:
                pass

            button_layout.addWidget(self.info_toggle_btn)
        except Exception:
            # If QPushButton isn't available for some reason, skip gracefully
            self.info_toggle_btn = None

        # Register the action on the parent dialog so Alt+I shortcut is active
        try:
            self.parent_dialog.addAction(info_action)
        except Exception:
            pass
        # Expose as attribute for parent dialog re-registration
        try:
            self.info_action = info_action
        except Exception:
            pass

        # Create QAction-backed controls to ensure shortcuts/mnemonics are registered reliably
        # Open Plugin Preferences (Alt+P)
        prefs_action = QAction(common_icons.get_icon('images/plugins_prefs_icon'), _('&Open Plugin Preferences'), self)
        prefs_action.setShortcut(QKeySequence(_('Alt+P')))
        prefs_action.setShortcutContext(Qt.WidgetShortcut)
        prefs_action.triggered.connect(lambda: (prints("[CCR][DEBUG] QAction Alt+P triggered") if CCR_DEBUG_ENABLED else None) or self.gui.iactions['Preferences'].do_config(
            initial_plugin=('Advanced', 'Plugins'),
            close_after_initial=True
        ))
        # Register the action on the parent dialog so Qt knows about it early
        try:
            self.parent_dialog.addAction(prefs_action)
        except Exception:
            pass
        # Expose as attribute for parent dialog re-registration
        try:
            self.prefs_action = prefs_action
        except Exception:
            pass
        # Keep a visible QPushButton so mnemonics and focus behave as users expect,
        # but register the QAction on the dialog so the shortcut is active immediately.
        self.prefs_button = QPushButton(common_icons.get_icon('images/plugins_prefs_icon'), '\u00A0' + _('Open &Plugin Preferences'))
        # Use compact styling consistent with the Shortcuts tab
        self.prefs_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.prefs_button.setAutoDefault(False)
        self.prefs_button.setDefault(False)
        try:
            self.prefs_button.setFocusPolicy(Qt.FocusPolicy.StrongFocus)
        except Exception:
            try:
                self.prefs_button.setFocusPolicy(Qt.StrongFocus)
            except Exception:
                pass
        # Wire both GUI and action to the same handler so either mouse or accel works
        self.prefs_button.clicked.connect(lambda: self.gui.iactions['Preferences'].do_config(
            initial_plugin=('Advanced', 'Plugins'),
            close_after_initial=True
        ))
        button_layout.addWidget(self.prefs_button)

        # Open Plugin Updater (Alt+U)
        updater_action = QAction(common_icons.get_icon('images/plugin_updater_icon'), _('&Open Plugin Updater'), self)
        updater_action.setShortcut(QKeySequence(_('Alt+U')))
        updater_action.setShortcutContext(Qt.WidgetShortcut)
        def open_plugin_updater():
            if hasattr(self.gui, 'show_plugin_update_dialog'):
                self.gui.show_plugin_update_dialog()
            else:
                try:
                    from calibre.gui2.dialogs.plugin_updater import PluginUpdaterDialog, FILTER_UPDATE_AVAILABLE
                    d = PluginUpdaterDialog(self.gui, initial_filter=FILTER_UPDATE_AVAILABLE)
                    d.exec()
                    if getattr(d, 'do_restart', False):
                        self.gui.quit(restart=True)
                except Exception:
                    self.gui.iactions['Preferences'].do_config(
                        initial_plugin=('Advanced', 'Plugins'),
                        close_after_initial=True
                    )
        updater_action.triggered.connect(lambda: (prints("[CCR][DEBUG] QAction Alt+U triggered") if CCR_DEBUG_ENABLED else None) or open_plugin_updater())
        try:
            self.parent_dialog.addAction(updater_action)
        except Exception:
            pass
        # Expose as attribute for parent dialog re-registration
        try:
            self.updater_action = updater_action
        except Exception:
            pass
        self.updater_button = QPushButton(common_icons.get_icon('images/plugin_updater_icon'), '\u00A0' + _('Open Plugin &Updater'))
        # Use compact styling consistent with the Shortcuts tab
        self.updater_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.updater_button.setAutoDefault(False)
        self.updater_button.setDefault(False)
        try:
            self.updater_button.setFocusPolicy(Qt.FocusPolicy.StrongFocus)
        except Exception:
            try:
                self.updater_button.setFocusPolicy(Qt.StrongFocus)
            except Exception:
                pass
        self.updater_button.clicked.connect(open_plugin_updater)
        button_layout.addWidget(self.updater_button)

        # Load Plugin from File (Alt+L)
        load_action = QAction(common_icons.get_icon('images/plus_icon'), _('&Load Plugin from File'), self)
        load_action.setShortcut(QKeySequence(_('Alt+L')))
        load_action.setShortcutContext(Qt.WidgetShortcut)
        load_action.setToolTip(_('Load a plugin from a ZIP file'))
        load_action.triggered.connect(self.parent_dialog.load_plugin_from_file)
        try:
            self.parent_dialog.addAction(load_action)
        except Exception:
            pass
        # Expose as attribute for parent dialog re-registration
        try:
            self.load_action = load_action
        except Exception:
            pass
        self.load_plugin_btn = QPushButton(common_icons.get_icon('images/plus_icon'), '\u00A0' + _('&Load Plugin from File'))
        self.load_plugin_btn.setToolTip(_('Load a plugin from a ZIP file'))
        # Use compact styling consistent with the Shortcuts tab
        self.load_plugin_btn.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.load_plugin_btn.setAutoDefault(False)
        self.load_plugin_btn.setDefault(False)
        try:
            self.load_plugin_btn.setFocusPolicy(Qt.FocusPolicy.StrongFocus)
        except Exception:
            try:
                self.load_plugin_btn.setFocusPolicy(Qt.StrongFocus)
            except Exception:
                pass
        self.load_plugin_btn.clicked.connect(self.parent_dialog.load_plugin_from_file)
        button_layout.addWidget(self.load_plugin_btn)

        # Remove Selected Plugins (Alt+M) - may conflict with Manage Columns (Alt+M)
        self.remove_selected_btn = QPushButton(common_icons.get_icon('images/minus_icon'), '\u00A0' + _('Remove Selecte&d Plugins'))
        self.remove_selected_btn.setToolTip(_('Remove all checked plugins'))
        # Use compact styling consistent with the Shortcuts tab
        self.remove_selected_btn.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.remove_selected_btn.setAutoDefault(False)
        self.remove_selected_btn.setDefault(False)
        self.remove_selected_btn.clicked.connect(self.remove_selected_plugins)
        button_layout.addWidget(self.remove_selected_btn)


        button_layout.addStretch()
        layout.addLayout(button_layout)

        # Wire selection change to update info panel content
        try:
            sel_model = self.plugin_table.selectionModel()
            if sel_model is not None:
                sel_model.currentChanged.connect(lambda cur, prev: self._on_plugin_selection_changed(cur.row() if cur.isValid() else -1))
        except Exception:
            try:
                self.plugin_table.itemSelectionChanged.connect(lambda: self._on_plugin_selection_changed(self.plugin_table.currentRow()))
            except Exception:
                pass

        # Connect forum_label clicks to use calibre.open_url so tweaks are honoured
        def _forum_link_activated(url):
            if not url:
                return
            try:
                from calibre.gui2 import open_url
                open_url(url)
            except Exception:
                try:
                    from PyQt5.QtCore import QUrl
                    from PyQt5.QtGui import QDesktopServices
                    QDesktopServices.openUrl(QUrl(url))
                except Exception:
                    pass

        try:
            # QLabel provides linkActivated signal
            self.forum_label.linkActivated.connect(_forum_link_activated)
        except Exception:
            # Fallback: QTextBrowser can be used for the forum link too
            try:
                self.desc_browser.anchorClicked.connect(_forum_link_activated)
            except Exception:
                pass

        # Application-scoped QShortcuts to ensure Alt+key accelerators trigger even when
        # focus policies or platform-specific mnemonic handling interfere on first show.
        # Note: Removed QShortcuts to avoid conflicts with QAction shortcuts
        # try:
        #     sc_p = QShortcut(QKeySequence('Alt+P'), self.parent_dialog)
        #     sc_p.setContext(Qt.ApplicationShortcut)
        #     sc_p.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+P shortcut triggered") if CCR_DEBUG_ENABLED else None, (getattr(self, 'prefs_button', None) and getattr(self.prefs_button, 'animateClick', lambda *a: None)(50)) or self.gui.iactions['Preferences'].do_config(initial_plugin=('Advanced', 'Plugins'), close_after_initial=True)))
        # except Exception:
        #     pass
        # try:
        #     sc_u = QShortcut(QKeySequence('Alt+U'), self.parent_dialog)
        #     sc_u.setContext(Qt.ApplicationShortcut)
        #     sc_u.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+U shortcut triggered") if CCR_DEBUG_ENABLED else None, (getattr(self, 'updater_button', None) and getattr(self.updater_button, 'animateClick', lambda *a: None)(50)) or open_plugin_updater()))
        # except Exception:
        #     pass
        # try:
        #     sc_l = QShortcut(QKeySequence('Alt+L'), self.parent_dialog)
        #     sc_l.setContext(Qt.ApplicationShortcut)
        #     sc_l.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+L shortcut triggered") if CCR_DEBUG_ENABLED else None, (getattr(self, 'load_plugin_btn', None) and getattr(self.load_plugin_btn, 'animateClick', lambda *a: None)(50)) or self.parent_dialog.load_plugin_from_file()))
        # except Exception:
        #     pass
        # try:
        #     sc_s = QShortcut(QKeySequence('Alt+S'), self.parent_dialog)
        #     sc_s.setContext(Qt.ApplicationShortcut)
        #     sc_s.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+S shortcut triggered") if CCR_DEBUG_ENABLED else None, (getattr(self, 'csv_export_button', None) and getattr(self.csv_export_button, 'animateClick', lambda *a: None)(50)) or self.parent_dialog.export_data('csv')))
        # except Exception:
        #     pass
        # try:
        #     sc_x = QShortcut(QKeySequence('Alt+X'), self.parent_dialog)
        #     sc_x.setContext(Qt.ApplicationShortcut)
        #     sc_x.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+X shortcut triggered") if CCR_DEBUG_ENABLED else None, (getattr(self, 'xlsx_export_button', None) and getattr(self.xlsx_export_button, 'animateClick', lambda *a: None)(50)) or self.parent_dialog.export_data('xlsx')))
        # except Exception:
        #     pass
        # try:
        #     sc_r = QShortcut(QKeySequence('Alt+R'), self.parent_dialog)
        #     sc_r.setContext(Qt.ApplicationShortcut)
        #     sc_r.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+R shortcut triggered") if CCR_DEBUG_ENABLED else None, (getattr(self, 'restore_defaults_button', None) and getattr(self.restore_defaults_button, 'animateClick', lambda *a: None)(50)) or self.restore_default_columns()))
        # except Exception:
        #     pass

    def _build_plugin_cache_lookup(self):
        """Build a robust lookup map for plugin cache entries.
        Keys include:
        - exact dict key
        - lowercased dict key
        - entry['index_name'] and lowercased
        - entry['name'] (display) and lowercased
        """
        try:
            lookup = {}
            cache = getattr(self, 'plugin_cache', {}) or {}
            if isinstance(cache, dict):
                for k, info in cache.items():
                    try:
                        # Main keys
                        if k:
                            lookup[str(k)] = info
                            lookup[str(k).lower()] = info
                        # Index name
                        idx = None
                        if isinstance(info, dict):
                            idx = info.get('index_name') or info.get('id') or info.get('name')
                        if idx:
                            lookup[str(idx)] = info
                            lookup[str(idx).lower()] = info
                        # Display name
                        disp = None
                        if isinstance(info, dict):
                            disp = info.get('name') or info.get('title')
                        if disp:
                            lookup[str(disp)] = info
                            lookup[str(disp).lower()] = info
                    except Exception:
                        continue
            self._cache_lookup = lookup
        except Exception:
            self._cache_lookup = {}

    def _lookup_cache_info(self, plugin):
        """Return cache info dict for a given plugin using robust lookup."""
        def _normalize(n):
            if not n:
                return ''
            # remove bracketed suffixes like " {Reader}" and parenthetical notes
            n = re.sub(r"\{.*?\}", '', str(n))
            n = re.sub(r"\(.*?\)", '', n)
            # normalize whitespace and punctuation, lowercase
            n = re.sub(r"[^0-9a-z]+", ' ', n.lower()).strip()
            return n

        try:
            if not isinstance(self._cache_lookup, dict) or not self._cache_lookup:
                self._build_plugin_cache_lookup()
            name = getattr(plugin, 'name', '') or ''
            # direct exact matches
            if name in self._cache_lookup:
                return self._cache_lookup[name] or {}
            low = str(name).lower()
            if low in self._cache_lookup:
                return self._cache_lookup[low] or {}

            # try normalized forms (handles names like 'ePub Extended Metadata {Reader}')
            norm = _normalize(name)
            if norm:
                for k, info in self._cache_lookup.items():
                    try:
                        if not k:
                            continue
                        if _normalize(k) == norm:
                            return info or {}
                    except Exception:
                        continue

                # try substring/prefix matches as a best-effort fallback
                for k, info in self._cache_lookup.items():
                    try:
                        kn = _normalize(k)
                        if norm and (kn.startswith(norm) or norm.startswith(kn) or norm in kn or kn in norm):
                            return info or {}
                    except Exception:
                        continue
        except Exception:
            pass

        # Final fallback: direct plugin_cache dict access (best-effort)
        try:
            pc = (self.plugin_cache or {})
            name = getattr(plugin, 'name', '') or ''
            if name in pc:
                return pc.get(name, {}) or {}
            low = str(name).lower()
            if low in pc:
                return pc.get(low, {}) or {}
        except Exception:
            pass
        return {}


    def _build_index_lookup(self):
        """Build a lookup from the cached full plugin index for offline metadata.
        Keys include dict key, index_name/id/name, and display name (exact + lowercase)."""
        lookup = {}
        try:
            raw_index = _session_index_cache if isinstance(_session_index_cache, dict) else category_cache.get('index', {})
            if isinstance(raw_index, dict):
                for k, info in raw_index.items():
                    try:
                        if not isinstance(info, dict):
                            continue
                        if k:
                            lookup[str(k)] = info
                            lookup[str(k).lower()] = info
                        idx = info.get('index_name') or info.get('id') or info.get('name')
                        if idx:
                            lookup[str(idx)] = info
                            lookup[str(idx).lower()] = info
                        disp = info.get('name') or info.get('title')
                        if disp:
                            lookup[str(disp)] = info
                            lookup[str(disp).lower()] = info
                    except Exception:
                        continue
        except Exception:
            lookup = {}
        self._index_lookup = lookup

    def _lookup_index_info(self, plugin):
        """Lookup an entry in the full index for the given plugin by name."""
        def _normalize(n):
            if not n:
                return ''
            n = re.sub(r"\{.*?\}", '', str(n))
            n = re.sub(r"\(.*?\)", '', n)
            n = re.sub(r"[^0-9a-z]+", ' ', n.lower()).strip()
            return n

        try:
            if not isinstance(self._index_lookup, dict) or not self._index_lookup:
                self._build_index_lookup()
            name = getattr(plugin, 'name', '') or ''
            # exact
            if name in self._index_lookup:
                return self._index_lookup[name] or {}
            low = str(name).lower()
            if low in self._index_lookup:
                return self._index_lookup[low] or {}

            norm = _normalize(name)
            if norm:
                for k, info in self._index_lookup.items():
                    try:
                        if not k:
                            continue
                        if _normalize(k) == norm:
                            return info or {}
                    except Exception:
                        continue

                # best-effort substring/prefix match
                for k, info in self._index_lookup.items():
                    try:
                        kn = _normalize(k)
                        if norm and (kn.startswith(norm) or norm.startswith(kn) or norm in kn or kn in norm):
                            return info or {}
                    except Exception:
                        continue
        except Exception:
            pass
        return {}

    def setup_plugins_table(self):
        """Setup the plugins table"""
        self.plugin_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.plugin_table.setAlternatingRowColors(True)
        self.plugin_table.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)
        self.plugin_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

        # Set up 22 columns (add debug column for category lookup)
        self.plugin_table.setColumnCount(22)

        headers = [
            '',  # 0 Remove (checkbox)
            _('Icon'),  # 1
            _('Name'),  # 2
            _('Version'),  # 3
            _('Author'),  # 4
            _('Installed Date'),  # 19
            _('Local Modified'),  # 20
            _('Size'),  # 16
            _('Released'),  # 17
            _('Forum Thread'),  # 12
            _('Min Calibre'),  # 5
            _('Type'),  # 6
            _('Donate'),  # 14
            _('Category'),  # 15 (Not in Index)
            _('Description'),  # 11
            _('Install Type'),  # 7
            _('Can Disable'),  # 8
            _('Customizable'),  # 9
            _('Platforms'),  # 10
            _('Copy Info'),  # 13
            _('Deprecated'),  # 18
            _('Is Disabled')  # 21 (shows if plugin is disabled)
        ]
        self.plugin_table.setHorizontalHeaderLabels(headers)

        # Set custom minus_icon.png icon in Remove column header (column 0)
        minus_icon = common_icons.get_icon('images/minus_icon')
        remove_header_item = self.plugin_table.horizontalHeaderItem(0)
        if remove_header_item is not None:
            remove_header_item.setIcon(minus_icon)
            remove_header_item.setToolTip(_('Remove selected plugins'))
            remove_header_item.setText('')  # Remove any text, icon only

        # Always define header right after header labels
        header = self.plugin_table.horizontalHeader()

        # Set column widths with min/max constraints - only if no saved preferences exist
        from calibre.gui2 import gprefs
        saved_widths = gprefs.get('calibre_config_reports_plugins_column_widths', None)

        if not saved_widths:
            # Only set initial widths if no saved preferences exist
            # Set Remove (checkbox) column to minimum width and prevent stretching
            self.plugin_table.setColumnWidth(0, 30)   # Remove (checkbox, now cross mark, 30px)
            from PyQt5.QtWidgets import QHeaderView
            header.setSectionResizeMode(0, QHeaderView.ResizeMode.Fixed)
            self.plugin_table.setMinimumWidth(0)
            self.plugin_table.setColumnWidth(1, 48)   # Icon
            self.plugin_table.setColumnWidth(2, 180)  # Name
            self.plugin_table.setColumnWidth(4, 100)  # Author
            self.plugin_table.setColumnWidth(11, 300) # Description

        # Enable sorting
        self.plugin_table.setSortingEnabled(True)

        # Enable context menu and other features
        self.plugin_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.plugin_table.customContextMenuRequested.connect(self.show_context_menu)
        self.plugin_table.cellClicked.connect(self.handle_cell_click)
        # Prefer a single activation path to prevent double dialogs
        # Use itemActivated which triggers for double-click and Enter
        try:
            self.plugin_table.itemActivated.connect(lambda item: self.show_plugin_info_dialog(item.row(), item.column()) if item else None)
        except Exception:
            pass

        # Enable column reordering
        header = self.plugin_table.horizontalHeader()

    # Keep a low absolute minimum so fixed small columns (Remove/Icon) can be narrower
        header.setMinimumSectionSize(15)
        header.setSectionsMovable(True)
        header.sectionMoved.connect(lambda: self.parent_dialog.save_column_state('plugins'))
        # Use a safer callback that doesn't rely on self.plugins_tab access
        def on_section_resized():
            try:
                self.parent_dialog.save_column_state('plugins')
            except AttributeError:
                # If plugins_tab is not accessible, try to save directly using the table
                if hasattr(self.parent_dialog, 'plugins_tab') and hasattr(self.parent_dialog.plugins_tab, 'plugin_table'):
                    self.parent_dialog.save_column_state('plugins')

        def enforce_version_column_width(logical_index, old_size, new_size):
            # Enforce maximum width of 75px for Version column (index 3)
            if logical_index == 3 and new_size > 75:
                self.plugin_table.setColumnWidth(3, 75)
            # Call the save function after enforcing constraints
            on_section_resized()

        header.sectionResized.connect(enforce_version_column_width)
        header.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        header.customContextMenuRequested.connect(self.show_plugin_header_context_menu)
        # Enforce Remove column (0) and Icon column (1) as fixed width
        from PyQt5.QtWidgets import QHeaderView

        header.setSectionResizeMode(0, QHeaderView.ResizeMode.Fixed)
        header.setSectionResizeMode(1, QHeaderView.ResizeMode.Fixed)  # Icon column

        # Force the icon column to stay at exactly 48px
        header.resizeSection(1, 48)

        self.plugin_table.setIconSize(QSize(28, 28))
        header.setSortIndicatorShown(True)

        # If the user has no saved hidden-columns gprefs for Plugins, ensure
        # the Local Modified column (logical index 6) is hidden by default on
        # fresh installs or fresh profiles. Persist this immediately so the
        # main restore flow won't re-show it later.
        try:
            from calibre.gui2 import gprefs
            if gprefs.get('calibre_config_reports_plugins_hidden_columns', None) is None:
                try:
                    # Hide only Local Modified (index 6) by default
                    if 6 < self.plugin_table.columnCount():
                        self.plugin_table.setColumnHidden(6, True)
                        gprefs['calibre_config_reports_plugins_hidden_columns'] = [6]
                        try:
                            gprefs.save()
                        except Exception:
                            pass
                except Exception:
                    pass
        except Exception:
            pass

        # Set up event handler to block sorting on checkbox and icon columns
        def block_icon_sort(logicalIndex):
            if logicalIndex in (0, 1):
                QApplication.instance().beep()
                return
        header.sectionClicked.connect(block_icon_sort)

        # Note: The actual column visibility and ordering is set in populate_plugins_table
        # to ensure it's applied after table population

    def _resolve_cache_info(self, plugin):
        """Return plugin cache info preferring strict plugin.name (as in the
        known-good backup) with a fallback to robust index/display-name lookup.
        Adds debug breadcrumbs when enabled.
        """
        ci = {}
        try:
            # 1) Strict plugin.name first (backup logic)
            if isinstance(self.plugin_cache, dict):
                ci = self.plugin_cache.get(getattr(plugin, 'name', ''), {}) or {}
            # 2) Fallback to robust lookup
            if not ci:
                ci = self._lookup_cache_info(plugin) or {}
        except Exception:
            ci = {}
        if CCR_DEBUG_ENABLED:
            try:
                name = getattr(plugin, 'name', '')
                has_rel = bool(ci.get('last_modified'))
                has_forum = bool(ci.get('thread_url'))
                has_donate = bool(ci.get('donate') or ci.get('donate_url'))
                prints(f"[CCR][DEBUG] CacheInfo for '{name}': last_modified={has_rel}, forum={has_forum}, donate={has_donate}")
            except Exception:
                pass
        return ci

    def _category_for_plugin(self, plugin, cache_info=None):
        """Resolve a single plugin's category from offline-first cache.
        Try plugin.name first (backup parity), then index_name/display name.
        """
        if numeric_version < (8, 9, 0):
            return ''
        cat_map = getattr(self, 'plugin_categories', {}) if hasattr(self, 'plugin_categories') else {}
        if not cat_map:
            return ''
        candidates = []
        try:
            pname = getattr(plugin, 'name', '')
            if pname:
                candidates.extend([pname, str(pname).lower()])
        except Exception:
            pass
        try:
            info = cache_info if isinstance(cache_info, dict) else self._lookup_cache_info(plugin)
            idx = (info or {}).get('index_name') or (info or {}).get('name')
            if idx:
                candidates.extend([idx, str(idx).lower()])
        except Exception:
            pass
        # Try direct candidate matches first
        for k in candidates:
            if k in cat_map:
                return cat_map.get(k, '')

        # If not found, try normalized matching against category map keys
        def _normalize(n):
            if not n:
                return ''
            n = re.sub(r"\{.*?\}", '', str(n))
            n = re.sub(r"\(.*?\)", '', n)
            return re.sub(r"[^0-9a-z]+", ' ', n.lower()).strip()

        try:
            # build normalized map if needed
            norm_candidates = [_normalize(c) for c in candidates if c]
            for key, val in cat_map.items():
                try:
                    kn = _normalize(key)
                    for nc in norm_candidates:
                        if not nc:
                            continue
                        if kn == nc or kn.startswith(nc) or nc.startswith(kn) or nc in kn or kn in nc:
                            return val
                except Exception:
                    continue
        except Exception:
            pass

        # As a final step, consult the full index lookup (if present) for a category
        try:
            idx_info = self._lookup_index_info(plugin)
            if idx_info and isinstance(idx_info, dict):
                c = idx_info.get('category') or ''
                if c:
                    return c
        except Exception:
            pass

        return ''

    def _refresh_category_cells(self):
        """Refresh Category column in-place from current plugin_categories map.
        Called after background category fetch; avoids full table rebuild.
        """
        try:
            col = 13  # Category
            for row in range(self.plugin_table.rowCount()):
                remove_item = self.plugin_table.item(row, 0)
                if not remove_item:
                    continue
                plugin = remove_item.data(Qt.ItemDataRole.UserRole)
                if not plugin:
                    continue
                cache_info = self._resolve_cache_info(plugin)
                cat = self._category_for_plugin(plugin, cache_info)
                item = self.plugin_table.item(row, col)
                if not item:
                    item = QTableWidgetItem()
                    self.plugin_table.setItem(row, col, item)
                if cat:
                    human_cat = CATEGORY_HUMAN_NAMES.get(cat, cat)
                    item.setText(human_cat)
                    item.setToolTip(human_cat)
                else:
                    item.setText(_('Not in Index'))
                    item.setToolTip(_('Not listed in official plugin index'))
                if CCR_DEBUG_ENABLED:
                    try:
                        prints(f"[CCR][DEBUG] Category refreshed for '{getattr(plugin,'name','')}': '{item.text()}'")
                    except Exception:
                        pass
        except Exception:
            # Non-fatal
            pass

    def populate_plugins_table(self):
        """Populate the plugins table with data"""
        # Get installed plugins
        plugins = list(initialized_plugins())
        plugins.sort(key=lambda x: x.name.lower())        # Fetch plugin categories if Calibre version >= 8.9
        # Ensure cache lookup is up to date
        self._build_plugin_cache_lookup()
        plugin_categories = {}
        if numeric_version >= (8, 9, 0):
            # Use cached categories only here to keep UI snappy; background fetch will update later
            try:
                if _session_category_cache is not None:
                    plugin_categories = _session_category_cache
                else:
                    cached_raw = category_cache.get('categories', {})
                    plugin_categories = _coerce_categories_shape(cached_raw)
            except Exception:
                plugin_categories = {}
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] populate_plugins_table using cached categories: {len(plugin_categories)} entries")

        # Store for use by category filter
        self.plugin_categories = plugin_categories

        # Filter based on type if needed
        if self.filter_type.currentIndex() == 1:  # User Plugins Only
            plugins = [p for p in plugins if getattr(p, 'installation_type', None) == PluginInstallationType.EXTERNAL]

        # Apply text filter if present
        filter_text = self.filter_edit.text().lower().strip()
        if filter_text:
            plugins = [p for p in plugins if filter_text in p.name.lower() or
                      filter_text in p.author.lower() or
                      filter_text in p.description.lower()]

        self.plugin_table.setRowCount(len(plugins))
        visible_count = 0

        from qt.core import QApplication
        for idx, plugin in enumerate(plugins):
            # Allow GUI events intermittently for responsiveness
            if idx and idx % 20 == 0:
                QApplication.processEvents()
            row = idx
            # 0: Remove (checkbox only for user-installed plugins)
            remove_item = QTableWidgetItem()
            installation_type = getattr(plugin, 'installation_type', None)
            if installation_type == PluginInstallationType.EXTERNAL:
                remove_item.setFlags(remove_item.flags() | Qt.ItemFlag.ItemIsUserCheckable)
                remove_item.setCheckState(Qt.CheckState.Unchecked)
            else:
                # Not user-installed: no checkbox, not selectable
                remove_item.setFlags(remove_item.flags() & ~Qt.ItemFlag.ItemIsUserCheckable)
            remove_item.setData(Qt.ItemDataRole.UserRole, plugin)
            remove_item.setText("")
            remove_item.setTextAlignment(Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignVCenter)
            self.plugin_table.setItem(row, 0, remove_item)

            # 1: Icon column (legacy-compatible, robust fallback)
            icon = None
            found = False
            # 1. Try to get icon from iactions (matches legacy logic)
            gui = getattr(self.parent_dialog, 'gui', None)
            if gui and hasattr(gui, 'iactions'):
                for key in gui.iactions.keys():
                    if plugin.name in key or key in plugin.name:
                        ia = gui.iactions[key]
                        if hasattr(ia, 'qaction'):
                            icon = ia.qaction.icon()
                            if icon and not icon.isNull():
                                found = True
                                break
            # 2. Try plugin.icon() method (legacy logic)
            if not found and hasattr(plugin, 'icon') and callable(plugin.icon):
                try:
                    icon = plugin.icon()
                    if icon and icon.isNull():
                        found = True
                except Exception:
                    icon = None
            # 3. Try plugin.icon attribute (QIcon or QPixmap)
            if not found and hasattr(plugin, 'icon') and not callable(plugin.icon):
                try:
                    if isinstance(plugin.icon, QIcon) and not plugin.icon.isNull():
                        icon = plugin.icon
                        found = True
                    elif isinstance(plugin.icon, QPixmap) and not plugin.icon.isNull():
                        icon = QIcon(plugin.icon)
                        found = True
                except Exception:
                    icon = None
            # 4. Try to load icon from plugin ZIP file (enhanced to handle SVG and other formats)
            if not found or not icon or icon.isNull():
                if hasattr(plugin, 'plugin_path') and plugin.plugin_path:
                    import zipfile
                    try:
                        from qt.core import QPixmap, QSvgRenderer
                        from qt.core import QIcon as QtIcon
                    except ImportError:
                        from PyQt5.QtGui import QPixmap
                        from PyQt5.QtSvg import QSvgRenderer
                        from PyQt5.QtGui import QIcon as QtIcon

                    # First, check for special icons for known plugins
                    plugin_display_name = getattr(plugin, 'name', '')
                    special_icons = {
                        'Author Book Count': 'images/abcicon.png',
                        'Editor Chains': 'images/editor_chains.png',
                        'Diaps Editing Toolbag': 'images/smarten_icon.png',
                        'Access Aide': 'icon/icon.png',
                        'KFX Input': 'kfx.png',
                        'KFX Output': 'kfx.png',
                    }
                    if plugin_display_name in special_icons:
                        icon_path_in_zip = special_icons[plugin_display_name]
                        try:
                            with zipfile.ZipFile(plugin.plugin_path, 'r') as zf:
                                if icon_path_in_zip in zf.namelist():
                                    with zf.open(icon_path_in_zip) as icon_file:
                                        data = icon_file.read()
                                        pixmap = QPixmap()
                                        if pixmap.loadFromData(data):
                                            icon = QIcon(pixmap)
                                            found = True
                        except Exception:
                            pass

                    # If no special icon found, try common icon file patterns
                    if not found:
                        # Common icon file patterns to look for
                        icon_patterns = [
                            'images/plugin.svg',
                            'images/plugin.png',
                            'images/icon.svg',
                            'images/icon.png',
                            'plugin.svg',
                            'plugin.png',
                            'icon.svg',
                            'icon.png'
                        ]

                        try:
                            with zipfile.ZipFile(plugin.plugin_path, 'r') as zf:
                                # Get list of files in the ZIP
                                zip_files = zf.namelist()

                                # Try to find an icon file
                                icon_file_path = None
                                for pattern in icon_patterns:
                                    if pattern in zip_files:
                                        icon_file_path = pattern
                                        break

                                # If no exact match, look for any image file in images/ folder
                                if not icon_file_path:
                                    for zip_file in zip_files:
                                        if (zip_file.startswith('images/') and
                                            any(zip_file.lower().endswith(ext) for ext in ['.png', '.svg', '.jpg', '.jpeg', '.gif', '.bmp'])):
                                            icon_file_path = zip_file
                                            break

                                # Load the icon file
                                if icon_file_path:
                                    with zf.open(icon_file_path) as icon_file:
                                        data = icon_file.read()

                                        # Handle SVG files
                                        if icon_file_path.lower().endswith('.svg'):
                                            try:
                                                renderer = QSvgRenderer()
                                                if renderer.load(data):
                                                    pixmap = QPixmap(32, 32)  # Standard icon size
                                                    pixmap.fill(Qt.GlobalColor.transparent)
                                                    try:
                                                        from qt.core import QPainter
                                                    except ImportError:
                                                        from PyQt5.QtGui import QPainter
                                                    painter = QPainter(pixmap)
                                                    renderer.render(painter)
                                                    painter.end()
                                                    if not pixmap.isNull():
                                                        icon = QIcon(pixmap)
                                                        found = True
                                            except Exception:
                                                pass
                                        else:
                                            # Handle other image formats (PNG, JPG, etc.)
                                            pixmap = QPixmap()
                                            if pixmap.loadFromData(data):
                                                icon = QIcon(pixmap)
                                                found = True
                        except Exception:
                            pass

            # 5. Fallback to plugins.png (renumbered from 6)
            if not found or not icon or icon.isNull():
                icon_path = os.path.join(os.path.dirname(__file__), 'images', 'plugins.png')
                if os.path.exists(icon_path):
                    pixmap = QPixmap(icon_path)
                    if not pixmap.isNull():
                        icon = QIcon(pixmap)
                    else:
                        icon = QIcon.ic('plugins.png')
                else:
                    icon = QIcon.ic('plugins.png')
            icon_item = QTableWidgetItem()
            if icon:
                icon_item.setIcon(icon)
            icon_item.setSizeHint(QSize(28, 28))
            self.plugin_table.setItem(row, 1, icon_item)
            icon_item.setData(Qt.ItemDataRole.UserRole, plugin)            # 2: Name
            name_item = QTableWidgetItem(plugin.name)
            name_item.setData(Qt.ItemDataRole.UserRole, plugin)
            self.plugin_table.setItem(row, 2, name_item)

            # 3: Version
            version = '.'.join(map(str, plugin.version))
            self.plugin_table.setItem(row, 3, QTableWidgetItem(version))

            # 4: Author
            self.plugin_table.setItem(row, 4, QTableWidgetItem(plugin.author))

            # 5: Installed Date (moved from column 19)
            plugin_path = getattr(plugin, 'plugin_path', None)
            installation_type = getattr(plugin, 'installation_type', None)
            install_date = ''

            # Fetch cached stats once per plugin and reuse for multiple columns
            cached_stats = None
            if plugin_path:
                try:
                    cached_stats = get_cached_file_stats(plugin_path)
                except Exception:
                    cached_stats = None

            if installation_type == PluginInstallationType.EXTERNAL and plugin_path:
                if cached_stats:
                    install_date = cached_stats.get('install_date', _('N/A'))
                else:
                    install_date = _('N/A')
            elif installation_type == PluginInstallationType.BUILTIN:
                install_date = _('Built-in')
            elif installation_type == PluginInstallationType.SYSTEM:
                install_date = _('System')
            else:
                install_date = _('N/A')

            self.plugin_table.setItem(row, 5, QTableWidgetItem(install_date))

            # 6: Local Modified (moved from column 20)
            local_modified_str = ""
            if cached_stats:
                local_modified_str = cached_stats.get('local_modified', '')

            local_modified_item = QTableWidgetItem(local_modified_str)
            self.plugin_table.setItem(row, 6, local_modified_item)

            # 7: Size (moved from column 16)
            raw_size = 0
            formatted_size = ""
            if cached_stats:
                raw_size = cached_stats.get('size_bytes', 0)
                formatted_size = cached_stats.get('formatted_size', '')

            size_item = SizeTableWidgetItem(raw_size, formatted_size)
            self.plugin_table.setItem(row, 7, size_item)

            # 8: Released (moved from column 17)
            cache_info = self._resolve_cache_info(plugin)
            index_info = self._lookup_index_info(plugin)
            info_pref = index_info or cache_info
            last_modified = info_pref.get('last_modified', '')
            if last_modified:
                try:
                    dt = dateutil.parser.parse(last_modified)
                    last_modified_str = format_datetime(dt)
                except Exception:
                    last_modified_str = last_modified
            else:
                last_modified_str = ''
            released_item = QTableWidgetItem(last_modified_str)
            self.plugin_table.setItem(row, 8, released_item)

            # 9: Forum Thread (moved from column 12)
            forum_item = QTableWidgetItem()
            forum_url = info_pref.get('thread_url', '')
            if forum_url and forum_url.strip():
                forum_item.setData(Qt.ItemDataRole.UserRole, forum_url)
                forum_item.setIcon(self.forum_icon)
                forum_item.setToolTip(forum_url)
                forum_item.setText(forum_url)
            else:
                forum_item.setText(_('N/A'))
            self.plugin_table.setItem(row, 9, forum_item)

            # 10: Min Calibre (moved from column 5)
            min_ver = '.'.join(map(str, plugin.minimum_calibre_version))
            self.plugin_table.setItem(row, 10, QTableWidgetItem(min_ver))

            # 11: Type (moved from column 6)
            plugin_type = getattr(plugin, 'type', 'Unknown')
            type_text = plugin_type if isinstance(plugin_type, str) else str(plugin_type)
            self.plugin_table.setItem(row, 11, QTableWidgetItem(type_text))

            # 12: Donate (moved from column 14)
            donate_item = QTableWidgetItem()
            # For parity with the backup, prefer 'donate' then fall back to 'donate_url'; prefer index info
            donate_url = info_pref.get('donate', '') or info_pref.get('donate_url', '')
            if donate_url and donate_url.strip():
                donate_item.setData(Qt.ItemDataRole.UserRole, donate_url)
                donate_item.setIcon(self.donate_icon)
                donate_item.setToolTip(_('Support plugin author'))
                donate_item.setText(_('Support'))
            else:
                donate_item.setText(_('N/A'))
            self.plugin_table.setItem(row, 12, donate_item)

            # 13: Category (moved from column 15)
            category_item = QTableWidgetItem()
            # Debug: track lookup process
            debug_steps = []
            cat_map = getattr(self, 'plugin_categories', {}) if hasattr(self, 'plugin_categories') else {}
            candidates = []
            pname = getattr(plugin, 'name', '')
            if pname:
                candidates.extend([pname, str(pname).lower()])
            info = info_pref
            idx = (info or {}).get('index_name') or (info or {}).get('name')
            if idx:
                candidates.extend([idx, str(idx).lower()])
            found_cat = ''
            found_key = ''
            for k in candidates:
                if k in cat_map:
                    found_cat = cat_map.get(k, '')
                    found_key = k
                    debug_steps.append(f'exact:{k}')
                    break
            if not found_cat:
                # Try normalized matching
                def _normalize(n):
                    if not n:
                        return ''
                    n = re.sub(r"\{.*?\}", '', str(n))
                    n = re.sub(r"\(.*?\)", '', n)
                    return re.sub(r"[^0-9a-z]+", ' ', n.lower()).strip()
                norm_candidates = [_normalize(c) for c in candidates if c]
                for key, val in cat_map.items():
                    kn = _normalize(key)
                    for nc in norm_candidates:
                        if kn == nc or kn.startswith(nc) or nc.startswith(kn) or nc in kn or kn in nc:
                            found_cat = val
                            found_key = key
                            debug_steps.append(f'norm:{key}')
                            break
                    if found_cat:
                        break
            if not found_cat:
                # Fallback: consult index
                idx_info = self._lookup_index_info(plugin)
                if idx_info and isinstance(idx_info, dict):
                    c = idx_info.get('category') or ''
                    if c:
                        found_cat = c
                        found_key = idx_info.get('index_name') or idx_info.get('name') or ''
                        debug_steps.append(f'index:{found_key}')
            if found_cat:
                human_cat = CATEGORY_HUMAN_NAMES.get(found_cat, found_cat)
                category_item.setText(human_cat)
                category_item.setToolTip(human_cat)
            else:
                category_item.setText(_('Not in Index'))
                category_item.setToolTip(_('Not listed in official plugin index'))
                debug_steps.append('fail')
            self.plugin_table.setItem(row, 13, category_item)

            # 21: Cat Debug (new debug column)
            is_disabled = getattr(plugin, 'is_disabled', None)
            # Use official is_disabled() from calibre.customize.ui
            try:
                from calibre.customize.ui import is_disabled as official_is_disabled
                is_disabled = official_is_disabled(plugin)
            except Exception:
                # Fallback: use attribute/method if present
                is_disabled = getattr(plugin, 'is_disabled', None)
                if callable(is_disabled):
                    is_disabled = is_disabled()
            disabled_item = QTableWidgetItem(_('Yes') if is_disabled else _('No'))
            disabled_item.setToolTip(_('Plugin is currently disabled') if is_disabled else _('Plugin is enabled'))
            self.plugin_table.setItem(row, 21, disabled_item)

            # 14: Description (moved from column 11)
            desc_item = QTableWidgetItem(plugin.description)
            desc_item.setToolTip(plugin.description)
            self.plugin_table.setItem(row, 14, desc_item)

            # 15: Install Type (moved from column 7)
            install_type = str(getattr(plugin, 'installation_type', _('Unknown')))
            self.plugin_table.setItem(row, 15, QTableWidgetItem(install_type))

            # 16: Can Disable (moved from column 8)
            can_disable = str(getattr(plugin, 'can_be_disabled', True))
            self.plugin_table.setItem(row, 16, QTableWidgetItem(can_disable))

            # 17: Customizable (moved from column 9)
            customizable = str(plugin.is_customizable())
            self.plugin_table.setItem(row, 17, QTableWidgetItem(customizable))

            # 18: Platforms (moved from column 10)
            platforms = ', '.join(getattr(plugin, 'supported_platforms', ['all']))
            self.plugin_table.setItem(row, 18, QTableWidgetItem(platforms))

            # 19: Copy Info (moved from column 13)
            copy_item = QTableWidgetItem()
            copy_item.setIcon(self.copy_icon)
            copy_item.setText(_('Copy'))
            copy_item.setToolTip(_('Click to copy plugin info to clipboard'))
            self.plugin_table.setItem(row, 19, copy_item)

            # 20: Deprecated (moved from column 18)
            deprecated = cache_info.get('deprecated', False)
            deprecated_item = QTableWidgetItem(_('Yes') if deprecated else _('No'))
            self.plugin_table.setItem(row, 20, deprecated_item)

            visible_count += 1
            self.plugin_table.setRowHidden(row, False)

        # Hide any extra rows
        for row in range(len(plugins), self.plugin_table.rowCount()):
            self.plugin_table.setRowHidden(row, True)

        # Update count label
        count_text = _("%d plugin%s") % (visible_count, _('s') if visible_count != 1 else '')
        if filter_text:
            count_text += _(" matching '%s'") % filter_text
        if self.filter_type.currentIndex() == 1:
            count_text += _(" (User plugins only)")
        self.count_label.setText(count_text)

        # Adjust column widths - let main dialog's restore system handle this
        # self.plugin_table.resizeColumnsToContents()  # Removed - this overwrites restored widths

        # When Description column is hidden, set Author column width to 130 only if not already set by user
        if self.plugin_table.isColumnHidden(11):
            # Only set width if it hasn't been restored from preferences
            if not hasattr(self, '_widths_restored') or not self._widths_restored:
                self.plugin_table.setColumnWidth(4, 130)
            # Don't force Fixed mode - let user resize freely
            # header = self.plugin_table.horizontalHeader()
            # from PyQt5.QtWidgets import QHeaderView
            # header.setSectionResizeMode(4, QHeaderView.ResizeMode.Fixed)

        # Update statistics
        if hasattr(self.parent_dialog, 'update_statistics') and hasattr(self.parent_dialog, 'plugins_tab'):
            self.parent_dialog.update_statistics()

        # After populating, re-apply icon size and row height in case of table reset
        self.plugin_table.setIconSize(QSize(28, 28))
        self.plugin_table.verticalHeader().setDefaultSectionSize(36)

        # Immediate refresh with any cached data we have
        try:
            self._build_index_lookup()
            self._refresh_metadata_cells()
            if CCR_DEBUG_ENABLED:
                prints("[CCR][DEBUG] Performed immediate metadata refresh after population")
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Error in immediate metadata refresh: {e}")

        # Schedule quick refresh to catch any pending cache updates
        try:
            QTimer.singleShot(100, self._delayed_metadata_refresh)

        except Exception:
            pass

        # Schedule fallback refresh in case background fetch completes later
        try:
            QTimer.singleShot(2000, self._delayed_metadata_refresh)
        except Exception:
            pass

        # Save file stats cache after populating table
        save_file_stats_cache()

    def _get_plugin_size(self, plugin):
        """Get plugin size in bytes and return both raw and formatted size"""
        if not plugin:
            return 0, "0B"
        try:
            # Get plugin path and size
            plugin_path = getattr(plugin, 'plugin_path', None)
            if not plugin_path:
                return 0, "N/A"

            size_bytes = get_file_size_bytes(plugin_path)
            return size_bytes, format_size(size_bytes)
        except Exception:
            return 0, "N/A"

    def _refresh_metadata_cells(self):
        """Refresh Released (col 8), Forum (col 9), Donate (col 12) using full index (offline-first) with cache fallback."""
        try:
            for row in range(self.plugin_table.rowCount()):
                remove_item = self.plugin_table.item(row, 0)
                if not remove_item:
                    continue
                plugin = remove_item.data(Qt.ItemDataRole.UserRole)
                if not plugin:
                    continue
                idx_info = self._lookup_index_info(plugin)
                cache_info = self._resolve_cache_info(plugin)
                info = idx_info or cache_info or {}

                # Released
                last_modified = info.get('last_modified', '')
                if last_modified:
                    try:
                        dt = dateutil.parser.parse(last_modified)
                        last_modified_str = format_datetime(dt)
                    except Exception:
                        last_modified_str = last_modified
                else:
                    last_modified_str = ''
                item_rel = self.plugin_table.item(row, 8)
                if not item_rel:
                    item_rel = QTableWidgetItem()
                    self.plugin_table.setItem(row, 8, item_rel)
                item_rel.setText(last_modified_str)

                # Forum
                forum_url = info.get('thread_url', '')
                item_forum = self.plugin_table.item(row, 9)
                if not item_forum:
                    item_forum = QTableWidgetItem()
                    self.plugin_table.setItem(row, 9, item_forum)
                if forum_url and forum_url.strip():
                    item_forum.setData(Qt.ItemDataRole.UserRole, forum_url)
                    item_forum.setIcon(self.forum_icon)
                    # Show the actual forum URL in the tooltip so users can hover to see/copy it
                    item_forum.setToolTip(forum_url)
                    item_forum.setText(forum_url)
                else:
                    item_forum.setText(_('N/A'))

                # Donate
                donate_url = info.get('donate', '') or info.get('donate_url', '')
                item_donate = self.plugin_table.item(row, 12)
                if not item_donate:
                    item_donate = QTableWidgetItem()
                    self.plugin_table.setItem(row, 12, item_donate)
                if donate_url and donate_url.strip():
                    item_donate.setData(Qt.ItemDataRole.UserRole, donate_url)
                    item_donate.setIcon(self.donate_icon)
                    item_donate.setToolTip(_('Support plugin author'))
                    item_donate.setText(_('Support'))
                else:
                    item_donate.setText(_('N/A'))

                if CCR_DEBUG_ENABLED:
                    try:
                        prints(f"[CCR][DEBUG] Metadata refreshed for '{getattr(plugin,'name','')}': released='{last_modified_str}', forum={'Y' if forum_url else 'N'}, donate={'Y' if donate_url else 'N'}")
                    except Exception:
                        pass
        except Exception:
            pass

    def apply_type_filter(self):
        """Apply filter based on plugin type selection"""
        self.populate_plugins_table()

    def apply_plugins_filter(self):
        """Apply text filter to plugins table using cached data without repopulating"""
        filter_text = self.filter_edit.text().lower().strip()
        category_filter_raw = self.category_filter_dropdown.currentData() if hasattr(self, 'category_filter_dropdown') and self.category_filter_dropdown.isVisible() else None

        import re
        def _normalize(n):
            if not n:
                return ''
            n = re.sub(r"\{.*?\}", '', str(n))
            n = re.sub(r"\(.*?\)", '', n)
            return re.sub(r"[^0-9a-z]+", ' ', n.lower()).strip()

        visible_count = 0
        for row in range(self.plugin_table.rowCount()):
            remove_item = self.plugin_table.item(row, 0)
            if not remove_item:
                continue
            plugin = remove_item.data(Qt.ItemDataRole.UserRole)
            if not plugin:
                continue

            text_match = True
            if filter_text:
                text_match = (filter_text in plugin.name.lower() or
                             filter_text in plugin.author.lower() or
                             filter_text in plugin.description.lower())

            category_match = True
            if hasattr(self, 'plugin_categories') and category_filter_raw is not None:
                # Resolve category using the same robust logic used to populate the Category column
                try:
                    cache_info = self._resolve_cache_info(plugin)
                    plugin_category = self._category_for_plugin(plugin, cache_info) or ''
                except Exception:
                    plugin_category = ''

                if category_filter_raw == '__not_in_index__':
                    category_item = self.plugin_table.item(row, 13)
                    category_text = category_item.text() if category_item else ''
                    category_match = (category_text == _('Not in Index'))
                else:
                    category_match = (_normalize(plugin_category) == _normalize(category_filter_raw))

            show_row = text_match and category_match
            self.plugin_table.setRowHidden(row, not show_row)
            if show_row:
                visible_count += 1

        # Update count label with filter information
        count_text = _("%d plugin%s") % (visible_count, _('s') if visible_count != 1 else '')
        if filter_text:
            count_text += _(" matching '%s'") % filter_text
        if self.filter_type.currentIndex() == 1:
            count_text += _(" (User plugins only)")
        self.count_label.setText(count_text)

        # Update statistics
        if hasattr(self.parent_dialog, 'update_statistics'):
            self.parent_dialog.update_statistics()

    def apply_category_filter(self):
        """Apply category filter using already cached plugin categories"""
        # Don't fetch again - use cached data
        if not hasattr(self, 'plugin_categories'):
            return

        # Just call the text filter which now handles both filters
        self.apply_plugins_filter()

    def _on_plugin_selection_changed(self, row):
        """Update the info panel (description and forum link) for the selected row."""
        try:
            if row is None or row < 0 or row >= self.plugin_table.rowCount():
                # Show a helpful message when nothing is selected instead of leaving the pane blank
                try:
                    title = _('No plugin selected')
                    hint = _('Please select a plugin to display its info')
                    html = '<div style="padding:12px;"><b>{}</b><div style="margin-top:6px;">{}</div></div>'.format(title, hint)
                    self.desc_browser.setHtml(html)
                    # Clear forum label since there is no plugin to link
                    self.forum_label.setText('')
                except Exception:
                    pass
                return

            # Get plugin object stored in column 0 UserRole
            remove_item = self.plugin_table.item(row, 0)
            plugin = remove_item.data(Qt.ItemDataRole.UserRole) if remove_item else None
            if plugin:
                # Show a single header line: name – version – author, then description below
                try:
                    name = getattr(plugin, 'name', '') or _('Unknown Plugin')
                    version = getattr(plugin, 'version', '') or ''
                    author = getattr(plugin, 'author', '') or _('Unknown Author')
                    header = f"{name} – {version} – {author}".strip(' –')
                    desc = (getattr(plugin, 'description', None) or '').strip()
                    # Compose the info box: header line, then description (both plain text)
                    if desc:
                        self.desc_browser.setText(f"{header}\n{desc}")
                    else:
                        self.desc_browser.setText(header or _('No description available'))
                except Exception:
                    # On any error, show the helpful no-selection hint instead of leaving blank
                    try:
                        title = _('No plugin selected')
                        hint = _('Please select a plugin to display its info')
                        html = '<div style="padding:12px;"><b>{}</b><div style="margin-top:6px;">{}</div></div>'.format(title, hint)
                        self.desc_browser.setHtml(html)
                    except Exception:
                        self.desc_browser.setHtml('')

                # Forum link stored in UserRole of forum column (9)
                try:
                    forum_item = self.plugin_table.item(row, 9)
                    forum_url = forum_item.data(Qt.ItemDataRole.UserRole) if forum_item else ''
                except Exception:
                    forum_url = ''
                if forum_url:
                    # Make a clickable link that will be handled by our handler
                    # Use format() to avoid f-string parsing issues with nested quotes
                    link_html = '<a href="{0}">{1}</a>'.format(forum_url, _('Open Forum Thread'))
                    self.forum_label.setText(link_html)
                else:
                    self.forum_label.setText(_('No forum thread available'))
            else:
                # No plugin object - clear
                try:
                    desc_item = self.plugin_table.item(row, 14)
                    desc = desc_item.text() if desc_item else ''
                    self.desc_browser.setText(desc)
                except Exception:
                    # Show helpful hint when we can't retrieve a description
                    try:
                        title = _('No plugin selected')
                        hint = _('Please select a plugin to display its info')
                        html = '<div style="padding:12px;"><b>{}</b><div style="margin-top:6px;">{}</div></div>'.format(title, hint)
                        self.desc_browser.setHtml(html)
                    except Exception:
                        self.desc_browser.setHtml('')
                try:
                    self.forum_label.setText('')
                except Exception:
                    pass
        except Exception:
            pass

    def show_column_menu(self):
        """Show menu to toggle plugin columns visibility"""
        headers = [self.plugin_table.horizontalHeaderItem(i).text()
                  for i in range(self.plugin_table.columnCount())]

        menu = QMenu(self)
        for i, header in enumerate(headers):
            action = menu.addAction(header)
            action.setCheckable(True)
            action.setChecked(not self.plugin_table.isColumnHidden(i))
            action.triggered.connect(lambda checked, col=i:
                self.parent_dialog.toggle_column_to_state(self.plugin_table, col, checked))

        menu.popup(self.column_button.mapToGlobal(
            self.column_button.rect().bottomLeft()))

    def restore_default_columns(self):
        """Restore default column visibility and ordering for plugins table"""
        from calibre.gui2 import question_dialog

        if not question_dialog(self, _('Restore Default Columns'),
                             _('This will restore the default column visibility and ordering for the Plugins tab. Continue?')):
            return

        # Define the default visible columns in the desired order (sequential indices based on header order)
        desired_columns = [
            0,  # Remove (checkbox)
            1,  # Icon
            2,  # Name
            3,  # Version
            4,  # Author
            5,  # Installed Date
            7,  # Size
            8,  # Released
            9,  # Forum Thread
            10, # Min Calibre
            11, # Type
            12, # Donate
            13, # Category
            14, # Description
            19, # Copy Info
        ]

        # Reset column visibility
        for col in range(self.plugin_table.columnCount()):
            self.plugin_table.setColumnHidden(col, col not in desired_columns)

        # Reset column ordering to default
        header = self.plugin_table.horizontalHeader()

        # Create the visual index mapping based on desired column order
        for visual_index, logical_index in enumerate(desired_columns):
            current_visual = header.visualIndex(logical_index)
            if current_visual != visual_index:
                header.moveSection(current_visual, visual_index)

        # Restore default column widths for visible columns
        default_widths = {
            0: 30,    # Remove (checkbox)
            1: 48,    # Icon
            2: 180,   # Name
            3: 75,    # Version (enforced max 75px)
            4: 100,   # Author
            5: 110,   # Installed Date
            6: 110,   # Local Modified
            7: 80,    # Size
            8: 110,   # Released
            9: 90,    # Forum Thread
            10: 100,  # Min Calibre
            11: 90,   # Type
            12: 90,   # Donate
            13: 120,  # Category
            14: 300,  # Description
            19: 80,   # Copy Info
        }
        for col, width in default_widths.items():
            self.plugin_table.setColumnWidth(col, width)

        # Clear saved preferences so defaults apply on next restart
        from calibre.gui2 import gprefs
        gprefs.pop('calibre_config_reports_plugins_hidden_columns', None)
        gprefs.pop('calibre_config_reports_plugins_column_order', None)

        # Save the default state immediately so it persists and doesn't get overridden
        # by the restore_column_state method in main.py
        self.parent_dialog.save_column_state('plugins')

        # DON'T call save_column_state here - it would overwrite what we just cleared!
        # The preferences are already cleared, and the table is already in default state

        from calibre.gui2 import info_dialog
        info_dialog(self, _('Columns Restored'),
                   _('Default column visibility, ordering, and widths have been restored.'), show=True)

    def show_context_menu(self, pos):
        """Show context menu for plugins table (right-click)"""
        plugin_table = self.plugin_table
        index = plugin_table.indexAt(pos)
        menu = QMenu(self)

        if index.isValid():
            row = index.row()
            name_item = plugin_table.item(row, 2)  # Name is in column 2
            if not name_item:
                return

            plugin_name = name_item.text()
            from calibre.customize.ui import initialized_plugins
            plugin = None
            for p in initialized_plugins():
                if p.name == plugin_name:
                    plugin = p
                    break

            if plugin:
                cache_info = self._resolve_cache_info(plugin)

                # Copy info action
                copy_action = menu.addAction(self.copy_icon, _('Copy plugin info'))
                copy_action.triggered.connect(lambda: self.copy_plugin_info_to_clipboard(plugin, cache_info, row))

                # Add separator
                menu.addSeparator()

                # Forum thread action if available
                forum_url = cache_info.get('thread_url', '')
                if forum_url:
                    visit_action = menu.addAction(self.forum_icon, _('Visit forum thread'))
                    # Show URL in the menu item's tooltip as well so hovering the menu reveals it
                    try:
                        visit_action.setToolTip(forum_url)
                    except Exception:
                        pass
                    from PyQt5.QtCore import QUrl
                    from PyQt5.QtGui import QDesktopServices
                    visit_action.triggered.connect(
                        lambda: (    # Use calibre.open_url so user tweaks (openers_by_scheme) are respected
                            (lambda u: __import__('calibre').gui2.open_url(u))(forum_url)
                        )
                    )

                # Donation link if available (prefer 'donate' then fallback to 'donate_url' to match backup)
                donate_url = cache_info.get('donate', '') or cache_info.get('donate_url', '')
                if donate_url:
                    donate_action = menu.addAction(self.donate_icon, _('Make a donation'))
                    from PyQt5.QtCore import QUrl
                    from PyQt5.QtGui import QDesktopServices
                    donate_action.triggered.connect(
                        lambda: (    # Use calibre.open_url so user tweaks (openers_by_scheme) are respected
                            (lambda u: __import__('calibre').gui2.open_url(u))(donate_url)
                        )
                    )

                # History if available
                history = cache_info.get('history', '')
                if history:
                    history_action = menu.addAction(_('View plugin history'))
                    history_action.triggered.connect(
                        lambda: self.parent_dialog.show_history_dialog(plugin.name)
                    )

        menu.popup(plugin_table.viewport().mapToGlobal(pos))

    def copy_plugin_info_to_clipboard(self, plugin, cache_info, row):
        """Copy plugin information to clipboard (context menu action)."""
        info_text = self.format_plugin_info(plugin)
        QApplication.clipboard().setText(info_text)
        # Show visual feedback (use Description column for feedback icon)
        self.parent_dialog.show_copy_feedback(row, 11, self.plugin_table)

    def format_plugin_info(self, plugin):
        """Return formatted plugin info string for dialog and clipboard, matching legacy output."""
        # Determine disabled state using official helper when available
        is_disabled = getattr(plugin, 'is_disabled', None)
        try:
            if callable(is_disabled):
                is_disabled = is_disabled()
        except Exception:
            is_disabled = None
        if is_disabled is None:
            try:
                from calibre.customize.ui import is_disabled as _is_disabled
                is_disabled = _is_disabled(plugin)
            except Exception:
                # Fallback: default to False if we cannot determine
                is_disabled = False

        # Determine release date from cache/index info (if available)
        release_date = ''
        try:
            cache_info = self._resolve_cache_info(plugin)
            last_mod = cache_info.get('last_modified') if isinstance(cache_info, dict) else None
            if last_mod:
                try:
                    # Use dateutil to parse if available
                    import dateutil.parser
                    from .date_utils import format_datetime
                    dt = dateutil.parser.parse(last_mod)
                    release_date = format_datetime(dt)
                except Exception:
                    release_date = str(last_mod)
        except Exception:
            release_date = ''

        fields = [
            ("Plugin", plugin.name),
            ("Version", '.'.join(map(str, plugin.version))),
            ("Author", plugin.author),
            ("Minimum Calibre Version", '.'.join(map(str, plugin.minimum_calibre_version))),
            ("Type", getattr(plugin, 'type', 'Unknown')),
            ("Description", plugin.description),
            ("Platforms", ', '.join(getattr(plugin, 'supported_platforms', ['all']))),
            ("Release Date", release_date),
            ("Can be disabled", str(getattr(plugin, 'can_be_disabled', True))),
            ("Is Disabled", _('Yes') if is_disabled else _('No')),
            ("Is customizable", str(plugin.is_customizable())),
        ]
        return '\n'.join(f"{k}: {v}" for k, v in fields)

    def show_plugin_header_context_menu(self, pos):
        """Show context menu for plugin table header (column visibility and width), using full-featured AdjustColumnSize dialog."""
        from calibre.gui2.library.views import AdjustColumnSize
        header = self.plugin_table.horizontalHeader()
        col = header.logicalIndexAt(pos)
        if col < 0:
            return
        col_name = self.plugin_table.horizontalHeaderItem(col).text()
        menu = QMenu(self)
        adjust_action = menu.addAction(_('Adjust column width...'))
        adjust_action.triggered.connect(lambda: AdjustColumnSize(self.plugin_table, col, col_name).exec_())
        menu.addSeparator()
        # Add column visibility toggles, always reflecting current state
        for i in range(self.plugin_table.columnCount()):
            header_name = self.plugin_table.horizontalHeaderItem(i).text()
            action = menu.addAction(header_name)
            action.setCheckable(True)
            action.setChecked(not self.plugin_table.isColumnHidden(i))
            action.triggered.connect(lambda checked, col=i:
                self.parent_dialog.toggle_column_to_state(self.plugin_table, col, checked))
        menu.popup(header.mapToGlobal(pos))


    def handle_cell_click(self, row, col):
        """Handle clicks on special cells (forum, copy, donate icons)"""
        self.parent_dialog.handle_cell_click(row, col)

    def show_plugin_info_dialog(self, row, col):
        """Show a dialog with copyable plugin information (double-click)"""
        plugin_table = self.plugin_table
        name_item = plugin_table.item(row, 2)  # Name column
        if not name_item:
            return
        plugin_name = name_item.text()
        from calibre.customize.ui import initialized_plugins
        plugin = None
        for p in initialized_plugins():
            if p.name == plugin_name:
                plugin = p
                break
        if not plugin:
            return
        info_text = self.format_plugin_info(plugin)
        # Show dialog
        from PyQt5.QtWidgets import QDialog, QVBoxLayout, QLabel, QDialogButtonBox, QPushButton, QApplication
        from PyQt5.QtCore import Qt
        dialog = QDialog(self)
        dialog.setWindowTitle(_("Plugin Info"))
        dialog.setMinimumWidth(450)
        layout = QVBoxLayout(dialog)
        label = QLabel(info_text)
        label.setWordWrap(True)
        label.setTextInteractionFlags(label.textInteractionFlags() | Qt.TextInteractionFlag.TextSelectableByMouse)
        layout.addWidget(label)
        # Add copy button
        copy_btn = QPushButton(_("Copy to Clipboard"))
        copy_btn.clicked.connect(lambda: QApplication.clipboard().setText(info_text))
        layout.addWidget(copy_btn)
        # Add close button
        buttons = QDialogButtonBox(QDialogButtonBox.StandardButton.Close)
        buttons.rejected.connect(dialog.reject)
        layout.addWidget(buttons)
        dialog.resize(450, dialog.sizeHint().height())
        dialog.exec()

    # Note: Enter handling is covered by itemActivated; no separate handler needed

    def remove_selected_plugins(self):
        """Remove all checked plugins from the table and Calibre, and prompt for restart if needed (legacy behavior)."""
        from calibre.gui2 import error_dialog, question_dialog, info_dialog, dynamic
        from calibre.customize.ui import remove_plugin, initialized_plugins
        from calibre.gui2.dialogs.plugin_updater import notify_on_successful_install
        checked_plugins = []
        for row in range(self.plugin_table.rowCount()):
            item = self.plugin_table.item(row, 0)
            if item and item.checkState() == Qt.CheckState.Checked:
                plugin = item.data(Qt.ItemDataRole.UserRole)
                if plugin:
                    checked_plugins.append(plugin)
        if not checked_plugins:
            error_dialog(self, _('No plugins selected'), _('Please check one or more plugins to remove.'), show=True)
            return
        # Confirm removal
        names = ', '.join(p.name for p in checked_plugins)
        if not question_dialog(self, _('Remove Plugins'), _('Are you sure you want to remove the following plugins?\n\n') + names, show_copy_button=False):
            return
        errors = []
        removed_names = []
        built_in_plugins = []
        restart_needed = False
        for plugin in checked_plugins:
            # Only allow removal of user-installed (external) plugins
            installation_type = getattr(plugin, 'installation_type', None)
            if installation_type != PluginInstallationType.EXTERNAL:
                built_in_plugins.append(plugin.name)
                continue
            try:
                removed = remove_plugin(plugin.name)
                if removed:
                    removed_names.append(plugin.name)
                    restart_needed = True
                else:
                    errors.append(plugin.name)
            except Exception as e:
                errors.append(f"{plugin.name}: {e}")
        if removed_names:
            info_dialog(self, _('Plugins removed'), _('The following plugins were removed:') + '\n' + '\n'.join(removed_names), show=True)
        if built_in_plugins:
            error_dialog(self, _('Cannot remove built-in plugins'), _('The following plugins are built-in and cannot be uninstalled:') + '\n' + '\n'.join(built_in_plugins), show=True)
        if errors:
            error_dialog(self, _('Error removing plugins'), _('Some plugins could not be removed:') + '\n' + '\n'.join(errors), show=True)
        self.populate_plugins_table()
        if restart_needed:
            if question_dialog(self, _('Restart calibre'), _('You must restart calibre for plugin changes to take effect. Restart now?'), show_copy_button=False):
                self.gui.quit(restart=True)
    def _delayed_metadata_refresh(self):
        """Simple fallback refresh in case background fetch didn't complete"""
        try:
            # Only refresh if we have index data now
            if hasattr(self, '_index_lookup') and self._index_lookup:
                self._build_index_lookup()
                self._refresh_metadata_cells()
                if CCR_DEBUG_ENABLED:
                    prints('[CCR][DEBUG] Performed fallback delayed metadata refresh')
        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f'[CCR][DEBUG] Error in fallback delayed metadata refresh: {e}')
