#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

"""
Plugin Installation Hook - Real-time monitoring of plugin installations
Hooks directly into Calibre's plugin installation functions to catch installs immediately.

The challenge: Plugin updates/downgrades call remove_plugin() then add_plugin().
Solution: Track recent removals and detect when an add follows a remove of the same plugin.
"""

import datetime
import threading
from calibre.utils.logging import prints

# Debug flag
import os as _ccr_os
CCR_DEBUG_ENABLED = _ccr_os.environ.get('CCR_DEBUG_ENABLED', '0').lower() in ('1', 'true', 'yes')

# When debug logging is disabled, replace prints with a no-op so existing
# debug calls do not emit output in release builds.
if not CCR_DEBUG_ENABLED:
    def prints(*args, **kwargs):
        return None
else:
    try:
        from calibre.utils.logging import prints as prints
    except Exception:
        def prints(*args, **kwargs):
            __import__('builtins').print(*args, **kwargs)
HOOK_ENABLED = True  # Enable hooks for real-time monitoring

# Track recent removals to detect updates/downgrades
_recent_removals = {}  # {plugin_name: (version, timestamp)}
_removal_timeout = 300  # seconds - 5 minutes to handle slow manual operations

# Track recently logged events to prevent duplicates
_recent_logs = {}  # {(plugin_name, event_type, version): timestamp}
_log_dedup_timeout = 30  # seconds - prevent duplicate logs within this time

if CCR_DEBUG_ENABLED:
    prints("[CCR][DEBUG] plugin_install_hook module loaded")

# Store original functions to avoid infinite recursion
_original_add_plugin = None
_original_remove_plugin = None
_hook_installed = False

def install_plugin_hooks():
    """Install hooks into Calibre's plugin installation system"""
    global _original_add_plugin, _original_remove_plugin, _hook_installed

    # Allow disabling hooks via preference
    if not HOOK_ENABLED:
        if CCR_DEBUG_ENABLED:
            prints("[CCR][DEBUG] Plugin hooks disabled by preference")
        return

    if _hook_installed:
        if CCR_DEBUG_ENABLED:
            prints("[CCR][DEBUG] Plugin hooks already installed, skipping")
        return

    try:
        from calibre.customize.ui import add_plugin, remove_plugin
        from calibre.constants import numeric_version

        # Version compatibility check - disable hooks for untested Calibre versions
        if numeric_version >= (9, 0, 0):  # Future major version
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Skipping plugin hooks - untested Calibre version {numeric_version}")
            return

        # Store original functions
        _original_add_plugin = add_plugin
        _original_remove_plugin = remove_plugin

        # Install our hooked versions
        import calibre.customize.ui
        calibre.customize.ui.add_plugin = _hooked_add_plugin
        calibre.customize.ui.remove_plugin = _hooked_remove_plugin

        _hook_installed = True

        if CCR_DEBUG_ENABLED:
            prints("[CCR][DEBUG] Successfully installed plugin installation hooks")

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Failed to install plugin hooks: {e}")

def _hooked_add_plugin(path):
    """Hooked version of add_plugin that logs installations to CCR history"""
    if CCR_DEBUG_ENABLED:
        prints(f"[CCR][DEBUG] Plugin installation detected: {path}")

    try:
        # Call original function
        result = _original_add_plugin(path)

        # Log the installation immediately
        if result:
            _log_plugin_installation(result, path)

        return result

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Error in hooked add_plugin: {e}")
        # Still call original even if logging fails
        return _original_add_plugin(path)

def _hooked_remove_plugin(plugin_name):
    """Hooked version of remove_plugin that logs removals to CCR history"""
    if CCR_DEBUG_ENABLED:
        prints(f"[CCR][DEBUG] Plugin removal detected: {plugin_name}")

    try:
        # Get plugin info before removal
        plugin_info = _get_plugin_info_before_removal(plugin_name)

        # Call original function
        result = _original_remove_plugin(plugin_name)

        # Track the removal for potential update/downgrade detection
        if result and plugin_info:
            _track_removal(plugin_name, plugin_info['version'])

        return result

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Error in hooked remove_plugin: {e}")
        # Still call original even if logging fails
        return _original_remove_plugin(plugin_name)

def _get_plugin_info_before_removal(plugin_name):
    """Get plugin version before it's removed"""
    try:
        from calibre.customize.ui import initialized_plugins
        for plugin in initialized_plugins():
            if plugin.name == plugin_name:
                return {
                    'version': '.'.join(map(str, plugin.version)),
                    'author': plugin.author
                }
    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Error getting plugin info: {e}")
    return None

def _track_removal(plugin_name, version):
    """Track a plugin removal for update/downgrade detection"""
    global _recent_removals
    _recent_removals[plugin_name] = (version, datetime.datetime.now())

    # Clean up old removals (but don't log them - they might still be part of slow updates)
    cutoff = datetime.datetime.now() - datetime.timedelta(seconds=_removal_timeout)
    _recent_removals = {k: v for k, v in _recent_removals.items() if v[1] > cutoff}

def _log_plugin_installation(plugin, path):
    """Log plugin installation to CCR history immediately"""
    try:
        from calibre_plugins.calibre_config_reports.settings_monitor import HistoryLogger
        from calibre_plugins.calibre_config_reports.version_utils import compare_versions, normalize_version, version_tuple_to_string

        plugin_name = plugin.name
        # Normalize version formatting to ensure consistency
        raw_version = '.'.join(map(str, plugin.version))
        normalized_version_tuple = normalize_version(raw_version)
        new_version = version_tuple_to_string(normalized_version_tuple) if normalized_version_tuple else raw_version

        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Logging installation of plugin: {plugin_name} v{new_version} (raw: {raw_version})")

        # Check if this follows a recent removal (indicating update/downgrade)
        global _recent_removals, _recent_logs
        event_type = 'plugin_installed'
        event_label = 'Plugin installed'
        old_version = None

        if plugin_name in _recent_removals:
            old_version, removal_time = _recent_removals[plugin_name]

            # Check if removal was recent enough to be considered part of an update
            time_since_removal = (datetime.datetime.now() - removal_time).total_seconds()
            if time_since_removal <= _removal_timeout:
                # This is an update/downgrade, not a fresh install
                try:
                    comparison = compare_versions(
                        normalize_version(new_version),
                        normalize_version(old_version)
                    )

                    if comparison > 0:
                        # Plugin updated
                        event_type = 'plugin_updated'
                        event_label = 'Plugin updated'
                    elif comparison < 0:
                        # Plugin downgraded
                        event_type = 'plugin_downgraded'
                        event_label = 'Plugin downgraded'
                    else:
                        # Same version - reinstall
                        event_type = 'plugin_reinstalled'
                        event_label = 'Plugin reinstalled'

                    # Remove from recent removals since we've handled it
                    del _recent_removals[plugin_name]

                except Exception as e:
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Error comparing versions for {plugin_name}: {e}")
                    # Fall through to regular install logging

        # If we still think this is a fresh install, double-check against tracked gprefs state
        if event_type == 'plugin_installed':
            try:
                # last_versions stored as name -> (version_tuple, path)
                from calibre.gui2 import gprefs as _gprefs
                tracked = _gprefs.get('calibre_config_reports_last_versions', {}) or {}
                prev = tracked.get(plugin_name)
                if isinstance(prev, (list, tuple)) and len(prev) >= 1:
                    prev_version_tuple = prev[0]
                    # Compare new version (normalize to tuple) vs prev tuple
                    cmp2 = compare_versions(normalized_version_tuple, prev_version_tuple)
                    if cmp2 > 0:
                        event_type = 'plugin_updated'
                        event_label = 'Plugin updated'
                        old_version = version_tuple_to_string(prev_version_tuple)
                    elif cmp2 < 0:
                        event_type = 'plugin_downgraded'
                        event_label = 'Plugin downgraded'
                        old_version = version_tuple_to_string(prev_version_tuple)
                    else:
                        event_type = 'plugin_reinstalled'
                        event_label = 'Plugin reinstalled'
            except Exception:
                pass

        # Check for duplicate logging (prevent same event within timeout)
        log_key = (plugin_name, event_type, new_version)
        now = datetime.datetime.now()

        if log_key in _recent_logs:
            last_log_time = _recent_logs[log_key]
            time_since_last_log = (now - last_log_time).total_seconds()
            if time_since_last_log < _log_dedup_timeout:
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Skipping duplicate log for {plugin_name} {event_type} (logged {time_since_last_log:.1f}s ago)")
                return

        # Log the event
        if event_type in ['plugin_updated', 'plugin_downgraded'] and old_version:
            HistoryLogger.log_event(event_type, {
                'event_type': event_label,
                'name': plugin_name,
                'old_version': old_version,
                'new_version': new_version,
                'path': path
            })
        else:
            # Regular plugin installation or reinstall
            HistoryLogger.log_event(event_type, {
                'event_type': event_label,
                'name': plugin_name,
                'version': new_version,
                'author': plugin.author,
                'path': path
            })

        # Track this log to prevent duplicates
        _recent_logs[log_key] = now

        # Clean up old log entries
        cutoff = now - datetime.timedelta(seconds=_log_dedup_timeout)
        _recent_logs = {k: v for k, v in _recent_logs.items() if v > cutoff}

        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Successfully logged {event_type} for {plugin_name}")

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Failed to log plugin installation: {e}")

def _log_plugin_removal(plugin_name, version=None):
    """Log plugin removal to CCR history immediately (only if not followed by reinstall)"""
    try:
        from calibre_plugins.calibre_config_reports.settings_monitor import HistoryLogger

        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Logging removal of plugin: {plugin_name}")

        # Note: We don't immediately log removals anymore since they might be part of updates
        # The removal tracking in _track_removal() handles this
        # Only log standalone removals (detected by cleanup or timeout)

        HistoryLogger.log_event('plugin_removed', {
            'event_type': 'Plugin removed',
            'name': plugin_name,
            'version': version or 'Unknown'
        })

        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Successfully logged removal of {plugin_name}")

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Failed to log plugin removal: {e}")

def uninstall_plugin_hooks():
    """Restore original plugin functions (cleanup)"""
    global _original_add_plugin, _original_remove_plugin, _hook_installed

    if not _hook_installed:
        return

    try:
        import calibre.customize.ui
        if _original_add_plugin:
            calibre.customize.ui.add_plugin = _original_add_plugin
        if _original_remove_plugin:
            calibre.customize.ui.remove_plugin = _original_remove_plugin

        _hook_installed = False

        if CCR_DEBUG_ENABLED:
            prints("[CCR][DEBUG] Successfully uninstalled plugin hooks")

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Failed to uninstall plugin hooks: {e}")

def check_recent_log(plugin_name, event_type, version):
    """Check if this event was recently logged to prevent duplicates from different monitoring paths"""
    global _recent_logs
    log_key = (plugin_name, event_type, version)

    if log_key in _recent_logs:
        last_log_time = _recent_logs[log_key]
        time_since_last_log = (datetime.datetime.now() - last_log_time).total_seconds()
        return time_since_last_log < _log_dedup_timeout

    return False

def cleanup_standalone_removals():
    """No-op: Standalone removals for plugin version events are now handled by robust version comparison logic in settings_monitor.py. Hooks only supplement real-time detection."""
    pass