#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import json
import os
import bz2
import datetime
import threading
from pathlib import Path
import os

# Early suppression: if the environment variable CCR_DEBUG_ENABLED is not set to a
# truthy value, monkeypatch calibre's global prints function to a no-op before
# any other modules import it. This prevents debug output emitted during module
# import time in release builds. You can enable debug at runtime by setting
# CCR_DEBUG_ENABLED=1 in the environment before launching Calibre.
if os.environ.get('CCR_DEBUG_ENABLED', '0').lower() not in ('1', 'true', 'yes'):
    try:
        import calibre.utils.logging as _ccr_clog
        _ccr_clog.prints = lambda *a, **k: None
    except Exception:
        # If we can't access calibre logging (e.g., running in tests), ignore.
        pass

# Import Qt components without multimedia first

from qt.core import (
    QAbstractItemView, QApplication, QCheckBox, QColor, QComboBox,
    QDesktopServices, QDialog, QDialogButtonBox, QEvent, QFont, QFrame,
    QHBoxLayout, QHeaderView, QIcon, QKeySequence, QLabel, QLineEdit,
    QMenu, QObject, QPushButton, QSize, QSizePolicy, QAction,
    QTabWidget, QTabBar, QTableWidget, QTableWidgetItem, QTimer, QUrl,
    QVBoxLayout, QWidget, Qt, QPixmap, QToolButton
)



from calibre.constants import config_dir, isportable, get_portable_base, DEBUG, iswindows
from calibre.gui2 import gprefs
from calibre.utils.config import JSONConfig, config_dir
from calibre.customize.ui import initialized_plugins, PluginInstallationType
from calibre.gui2.keyboard import Manager as KeyboardManager
from calibre.utils.localization import _
from calibre.gui2.preferences.main import Preferences
from calibre.gui2.preferences.keyboard import ConfigWidget
from calibre.gui2.library.views import AdjustColumnSize
# from calibre import prints
from calibre.utils.logging import prints
from . import common_icons
from . import size_utils
from .date_utils import get_file_times, format_datetime
from .settings_monitor import (
    monitor_and_log_config_events, HistoryLogger,
    get_setting_config_info, load_all_config_files, get_setting_value
)
from .plugin_install_hook import install_plugin_hooks
from .settings_tab import SettingsTab
# Initialize translations
load_translations()

# Determine debug mode from environment so setting CCR_DEBUG_ENABLED=1 outside
# the process enables debug output for testing. Use a private name to avoid
# colliding with any module-level `os` usage elsewhere.
import os as _ccr_os
CCR_DEBUG_ENABLED = _ccr_os.environ.get('CCR_DEBUG_ENABLED', '0').lower() in ('1', 'true', 'yes')

# If debug is disabled, replace local prints with a no-op so existing
# debug calls throughout the module don't emit output in production.
if not CCR_DEBUG_ENABLED:
    def prints(*args, **kwargs):
        # swallow debug output in release builds
        return None
else:
    # Bind to calibre's logging prints when debug is enabled so messages appear
    try:
        from calibre.utils.logging import prints as prints
    except Exception:
        # Fallback to a simple stdout print if calibre's helper isn't available
        def prints(*args, **kwargs):
            __builtins__.print(*args, **kwargs)

# Create plugin cache in CCR config folder with proper namespace - uses subfolder to avoid cluttering config root
plugin_updates_cache = JSONConfig('plugins/calibre_config_reports/ccr_plugin_cache')

def migrate_cache_file():
    """Migrate old plugin cache files from config root to CCR subfolder with ccr_ prefix"""
    from calibre.utils.config import config_dir
    import os
    import json

    # Old cache files that were in config root - be very specific to avoid conflicts
    old_cache_files = [
        os.path.join(config_dir, 'calibre_config_reports_plugin_cache.json'),  # Recent version
        os.path.join(config_dir, 'plugin_updates.json')  # Version 1.03 and earlier (only if it contains CCR data)
    ]

    # Only migrate if new cache is empty and we find an old cache file
    if not plugin_updates_cache:
        for old_cache_file in old_cache_files:
            if os.path.exists(old_cache_file):
                try:
                    with open(old_cache_file, 'r', encoding='utf-8') as f:
                        old_cache_data = json.load(f)

                    # For plugin_updates.json, verify it's actually CCR data before migrating
                    # Check if it has the structure we expect from CCR cache
                    if old_cache_file.endswith('plugin_updates.json'):
                        # Simple validation: check if it has expected plugin data structure
                        if not old_cache_data or not isinstance(old_cache_data, dict):
                            continue

                        # Check if it looks like CCR plugin cache data with VERY specific markers
                        # that are unique to the Calibre plugin index from code.calibre-ebook.com
                        sample_entries = list(old_cache_data.values())[:3]
                        is_ccr_cache = any(
                            isinstance(entry, dict) and
                            'thread_url' in entry and  # MobileRead forum URL (very CCR-specific)
                            'original_url' in entry and  # Download URL from MobileRead
                            'index_name' in entry and   # Plugin index name
                            isinstance(entry.get('version'), list) and  # Version as array [x,y,z]
                            'mobileread.com' in str(entry.get('thread_url', ''))  # Confirm MobileRead
                            for entry in sample_entries
                        )

                        if not is_ccr_cache:
                            if CCR_DEBUG_ENABLED:
                                prints(f"[CCR][DEBUG] Skipping {old_cache_file} - doesn't match CCR plugin cache signature")
                            continue                    # Copy data to new cache location
                    plugin_updates_cache.clear()
                    plugin_updates_cache.update(old_cache_data)
                    plugin_updates_cache.save()

                    # Remove old cache file to clean up config root
                    os.remove(old_cache_file)

                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Migrated plugin cache from {old_cache_file} to CCR subfolder")
                    break  # Stop after migrating the first file found

                except Exception as e:
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Failed to migrate plugin cache from {old_cache_file}: {e}")
                    continue  # Try the next file if this one fails

# Perform cache migration on module load
migrate_cache_file()

# Ensure the plugin cache file exists on disk even if empty. JSONConfig lazily
# creates files; in fresh setups that can leave no file. Write a minimal empty
# object once so later reads and external tools can rely on its presence.
try:
    from calibre.utils.config import config_dir as _ccr_cfgdir
    _ccr_cache_path = os.path.join(_ccr_cfgdir, 'plugins', 'calibre_config_reports', 'ccr_plugin_cache.json')
    _ccr_cache_dir = os.path.dirname(_ccr_cache_path)
    if not os.path.exists(_ccr_cache_dir):
        os.makedirs(_ccr_cache_dir, exist_ok=True)
    # If file is missing, save an empty dict once
    if not os.path.exists(_ccr_cache_path):
        plugin_updates_cache.clear()
        # no entries yet, but force creation
        plugin_updates_cache.save()
except Exception:
    pass

# Session-based cache for plugin data to avoid repeated network requests
_session_plugin_cache = None
_session_plugin_fetch_attempted = False

HISTORY_FILE = os.path.join(config_dir, 'plugins', 'calibre_config_reports', 'ccr_history.json')
 # prints(f"CCR: History file path is {HISTORY_FILE}")

# Ensure the directory exists
history_dir = os.path.dirname(HISTORY_FILE)
if not os.path.exists(history_dir):
    try:
        os.makedirs(history_dir)
    except Exception as e:
        pass

def load_plugin_cache():
    """Load the plugin cache quickly, skip network fetch on initial load (cached per session)"""
    global _session_plugin_cache

    # Return cached result if available
    if _session_plugin_cache is not None:
        return _session_plugin_cache

    try:
        # Only try JSONConfig on initial load - don't fetch from network
        cache = dict(plugin_updates_cache)
        _session_plugin_cache = cache
        return cache
    except Exception:
        _session_plugin_cache = {}
        return {}

def load_plugin_cache_with_network():
    """Load the plugin cache, falling back to direct fetch if needed (slower, cached per session)"""
    global _session_plugin_cache, _session_plugin_fetch_attempted

    # Return cached result if available
    if _session_plugin_cache is not None:
        return _session_plugin_cache

    try:
        # Try JSONConfig first
        cache = dict(plugin_updates_cache)
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Loaded cache with {len(cache)} entries")

        # Check if we need to fetch from network
        needs_fetch = False
        if not cache:
            if CCR_DEBUG_ENABLED:
                prints("[CCR][DEBUG] Cache is empty, need to fetch from network")
            needs_fetch = True
        else:
            # Check if cache has release date info (last_modified field)
            sample_entries = list(cache.values())[:3]  # Check first 3 entries
            has_last_modified = any('last_modified' in entry for entry in sample_entries)
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Sample entries checked: {len(sample_entries)}, has_last_modified: {has_last_modified}")
                for i, entry in enumerate(sample_entries):
                    entry_name = list(cache.keys())[i] if i < len(cache) else f"entry_{i}"
                    prints(f"[CCR][DEBUG] Entry {entry_name}: has last_modified = {'last_modified' in entry}")

            if not has_last_modified:
                if CCR_DEBUG_ENABLED:
                    prints("[CCR][DEBUG] Cache lacks last_modified fields, need to refresh from network")
                needs_fetch = True

        # Try to use existing plugin_updates.json as fallback if our cache is empty/invalid
        if not cache or needs_fetch:
            # Look for fallback file in the dev directory (temporary solution)
            # Look for fallback file in sensible locations. Remove any hardcoded
            # developer-only paths (e.g. c:\dev31) so end users won't reference
            # files that don't exist on their machines.
            possible_locations = [
                os.path.join(os.path.dirname(__file__), '..', '..', 'plugin_updates.json'),
                os.path.join(os.getcwd(), 'plugin_updates.json')
            ]

            for fallback_file in possible_locations:
                if os.path.exists(fallback_file):
                    try:
                        with open(fallback_file, 'r', encoding='utf-8') as f:
                            fallback_data = json.load(f)
                        if fallback_data and any('last_modified' in entry for entry in list(fallback_data.values())[:3]):
                            if CCR_DEBUG_ENABLED:
                                prints(f"[CCR][DEBUG] Using fallback cache file {fallback_file} with {len(fallback_data)} entries")
                            cache = fallback_data
                            needs_fetch = False

                            # Copy to our proper cache for future use
                            plugin_updates_cache.clear()
                            for name, info in fallback_data.items():
                                plugin_updates_cache[name] = info

                            # Force save to ensure the cache file gets created
                            try:
                                plugin_updates_cache.save()
                                if CCR_DEBUG_ENABLED:
                                    prints("[CCR][DEBUG] Cache file saved successfully")
                            except Exception as save_e:
                                if CCR_DEBUG_ENABLED:
                                    prints(f"[CCR][DEBUG] Failed to save cache file: {save_e}")
                            break
                    except Exception as e:
                        if CCR_DEBUG_ENABLED:
                            prints(f"[CCR][DEBUG] Failed to load fallback cache from {fallback_file}: {e}")
                        continue        # If cache needs updating and we haven't tried fetching this session, try fetching directly
        if needs_fetch and not _session_plugin_fetch_attempted:
            _session_plugin_fetch_attempted = True

            from calibre.utils.https import get_https_resource_securely
            import bz2

            SERVER = 'https://code.calibre-ebook.com/plugins/'
            INDEX_URL = f'{SERVER}plugins.json.bz2'

            try:
                if CCR_DEBUG_ENABLED:
                    prints("[CCR][DEBUG] Fetching plugin updates from server (once per session)...")
                raw = get_https_resource_securely(INDEX_URL)
                if raw:
                    data = json.loads(bz2.decompress(raw).decode('utf-8'))

                    # Clear old cache and store new data
                    plugin_updates_cache.clear()
                    for name, info in data.items():
                        plugin_updates_cache[name] = info

                    # Force save to ensure the cache file gets created
                    try:
                        plugin_updates_cache.save()
                        if CCR_DEBUG_ENABLED:
                            prints("[CCR][DEBUG] Network cache file saved successfully")
                    except Exception as save_e:
                        if CCR_DEBUG_ENABLED:
                            prints(f"[CCR][DEBUG] Failed to save network cache file: {save_e}")

                    # Get fresh cache
                    cache = dict(plugin_updates_cache)
                    if CCR_DEBUG_ENABLED:
                        prints(f"[CCR][DEBUG] Successfully cached {len(cache)} plugin updates with release dates")
                else:
                    if CCR_DEBUG_ENABLED:
                        prints("[CCR][DEBUG] No data received from network fetch")
            except Exception as e:
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Failed to fetch plugin updates: {e}")

        # Cache the result for the entire session
        _session_plugin_cache = cache
        return cache

    except Exception as e:
        if CCR_DEBUG_ENABLED:
            prints(f"[CCR][DEBUG] Exception in load_plugin_cache_with_network: {e}")
        _session_plugin_cache = {}
        return {}

def get_install_type():
    """Get a string indicating the Calibre installation type and location"""
    if isportable:
        portable_base = get_portable_base()
        if (portable_base):
            return f'portable-{os.path.basename(portable_base)}'
        return 'portable'
    return os.path.basename(os.path.dirname(config_dir))

def _is_portable_install(self):
    """Check if this is a portable installation"""
    from calibre.utils.config import portable_base
    return bool(portable_base())

def _get_library_paths(self):
    """Get all configured library paths"""
    from calibre.gui2 import gprefs
    library_usage = gprefs.get('library_usage_stats', {})
    return sorted(library_usage.keys())





class PluginCatalogDialog(QDialog):
    # Hardcoded global shortcuts removed. Use Calibre-registered shortcuts instead.
    def export_to_csv(self, filepath, headers, data):
        """Export data to CSV format"""
        import csv
        with open(filepath, 'w', newline='', encoding='utf-8') as f:
            writer = csv.writer(f)
            writer.writerow(headers)
            writer.writerows(data)

    def export_to_xlsx(self, filepath, headers, data):
        """Export data to XLSX format using local openpyxl"""
        import sys, os
        plugin_dir = os.path.dirname(os.path.realpath(__file__))
        try:
            sys.path.insert(0, plugin_dir)
            from openpyxl import Workbook
            from openpyxl.styles import Font, PatternFill
            sys.path.pop(0)
        except ImportError as e:
            from calibre.gui2 import error_dialog
            error_dialog(self, _('Missing dependency'),
                        _('Could not import openpyxl: %s\nPlugin directory: %s') % (str(e), plugin_dir),
                        show=True)
            raise
        wb = Workbook()
        ws = wb.active
        ws.title = self._get_current_tab_name()
        header_font = Font(bold=True)
        header_fill = PatternFill(start_color='CCCCCC', end_color='CCCCCC', fill_type='solid')
        for col, header in enumerate(headers, 1):
            cell = ws.cell(row=1, column=col)
            cell.value = header
            cell.font = header_font
            cell.fill = header_fill
        for row_idx, row_data in enumerate(data, 2):
            for col_idx, value in enumerate(row_data, 1):
                cell = ws.cell(row_idx, column=col_idx)
                cell.value = value
        for col in ws.columns:
            max_length = 0
            for cell in col:
                if cell.value is not None:
                    try:
                        cell_str = str(cell.value)
                        str_length = len(cell_str)
                        max_length = max(max_length, str_length)
                    except Exception:
                        continue
            adjusted_width = (max_length + 2) if max_length > 0 else 10
            try:
                ws.column_dimensions[col[0].column_letter].width = min(adjusted_width, 50)
            except Exception:
                continue
        wb.save(filepath)

    def format_plugin_info(self, plugin):
        """Return formatted plugin info string for dialog and clipboard, matching legacy output."""
        # This matches your requested field order and formatting
        fields = [
            ("Plugin", plugin.name),
            ("Version", '.'.join(map(str, plugin.version))),
            ("Author", plugin.author),
            ("Minimum Calibre Version", '.'.join(map(str, plugin.minimum_calibre_version))),
            ("Type", getattr(plugin, 'type', 'Unknown')),
            ("Description", plugin.description),
            ("Platforms", ', '.join(getattr(plugin, 'supported_platforms', ['all']))),
            ("Installation Type", str(getattr(plugin, 'installation_type', 'Unknown'))),
            ("Can be disabled", str(getattr(plugin, 'can_be_disabled', True))),
            ("Is customizable", str(plugin.is_customizable())),
        ]
        return '\n'.join(f"{k}: {v}" for k, v in fields)
    def handle_cell_click(self, row, col):
        """Handle clicks on special cells in the plugins table"""
        # Get the plugin table from the plugins tab
        plugin_table = self.plugins_tab.plugin_table

        # Get the item that was clicked
        item = plugin_table.item(row, col)
        if not item:
            return

        # Handle different column types
        if col == 9:  # Forum Thread column
            forum_url = item.data(Qt.ItemDataRole.UserRole)
            if forum_url and forum_url.strip():
                # Use calibre's open_url so it respects user tweaks (openers_by_scheme)
                try:
                    from calibre.gui2 import open_url
                    open_url(forum_url)
                except Exception:
                    # Fallback to Qt if calibre helper is unavailable
                    from PyQt5.QtCore import QUrl
                    from PyQt5.QtGui import QDesktopServices
                    QDesktopServices.openUrl(QUrl(forum_url))

        elif col == 12:  # Donate column
            donate_url = item.data(Qt.ItemDataRole.UserRole)
            if donate_url and donate_url.strip():
                try:
                    from calibre.gui2 import open_url
                    open_url(donate_url)
                except Exception:
                    from PyQt5.QtCore import QUrl
                    from PyQt5.QtGui import QDesktopServices
                    QDesktopServices.openUrl(QUrl(donate_url))

        elif col == 19:  # Copy Info column
            # Get plugin data from the remove item (column 0) which stores the plugin object
            remove_item = plugin_table.item(row, 0)
            if remove_item:
                plugin = remove_item.data(Qt.ItemDataRole.UserRole)
                if plugin:
                    # Copy plugin info to clipboard
                    info_text = self.plugins_tab.format_plugin_info(plugin)
                    from PyQt5.QtWidgets import QApplication
                    QApplication.clipboard().setText(info_text)
                    # Show simple, temporary visual feedback in the Copy Info cell
                    # by swapping text/icon to "Copied" with a checkmark, then revert.
                    item = plugin_table.item(row, col)
                    if item:
                        from PyQt5.QtCore import QTimer
                        original_text = item.text()
                        try:
                            # Prefer our ok icon if available
                            item.setIcon(self.copy_done_icon)
                        except Exception:
                            pass
                        item.setText(_('Copied'))
                        # Revert after a short delay
                        def _revert():
                            it = plugin_table.item(row, col)
                            if it:
                                try:
                                    it.setIcon(self.copy_icon)
                                except Exception:
                                    pass
                                it.setText(original_text or _('Copy'))
                        QTimer.singleShot(1200, _revert)
    def export_data(self, format_type):
        """
        Export data for the current tab (Plugins or Custom Columns) to CSV or XLSX,
        respecting current column visibility and visual order.
        """
        from calibre.gui2 import choose_save_file, info_dialog, error_dialog
        tab_idx = self.tabs.currentIndex()
        tab_name = self.tabs.tabText(tab_idx)
        # Determine which table to export
        if tab_name == _('Custom Columns'):
            table = self.columns_table
            empty_msg = _('No custom columns match the current filter criteria.')
            success_msg = _('Successfully exported custom columns to %s')
        else:
            # Default to Plugins tab
            table = self.plugins_tab.plugin_table
            empty_msg = _('No plugins match the current filter criteria.')
            success_msg = _('Successfully exported plugins to %s')

        # Only export visible (filtered) rows
        visible_rows = [row for row in range(table.rowCount()) if not table.isRowHidden(row)]
        if not visible_rows:
            error_dialog(self, _('Export Failed'), empty_msg, show=True)
            return

        # Get filename with timestamp
        default_name = self._get_export_filename(format_type)
        # Choose save location
        if format_type == 'csv':
            filters = _('CSV files (*.csv)')
            title = _('Export to CSV')
        else:
            filters = _('Excel files (*.xlsx)')
            title = _('Export to Excel')
        save_path = choose_save_file(self, title, filters, initial_filename=default_name)
        if not save_path:
            return
        try:
            # --- Export columns in visual order and only visible ---
            header = table.horizontalHeader()
            visual_to_logical = {}
            for logical_index in range(header.count()):
                visual_index = header.visualIndex(logical_index)
                visual_to_logical[visual_index] = logical_index
            # Sorted by visual position
            headers = []
            visible_columns = []
            for visual_index in sorted(visual_to_logical.keys()):
                logical_index = visual_to_logical[visual_index]
                # For Plugins tab, always skip columns 0 and 1 (Remove, Icon)
                if tab_name != _('Custom Columns') and logical_index in (0, 1):
                    continue
                if not table.isColumnHidden(logical_index):
                    headers.append(table.horizontalHeaderItem(logical_index).text())
                    visible_columns.append(logical_index)
            # Data
            data = []
            for row in visible_rows:
                row_data = []
                for col in visible_columns:
                    item = table.item(row, col)
                    row_data.append(item.text() if item else '')
                data.append(row_data)
            if format_type == 'csv':
                self.export_to_csv(save_path, headers, data)
            else:
                self.export_to_xlsx(save_path, headers, data)
            info_dialog(self, _('Export Complete'),
                        success_msg % save_path,
                        show=True, show_copy_button=False)
        except Exception as e:
            error_dialog(self, _('Export Failed'),
                         _('Failed to export data: %s') % str(e),
                         show=True)

    def save_column_widths(self, table_name):
        """Save the column widths for a table"""
        table = None
        if table_name == 'plugins':
            table = self.plugins_tab.plugin_table
        elif table_name == 'columns':
            table = self.columns_table
        elif table_name == 'shortcuts':
            table = self.shortcuts_table
        elif table_name == 'settings':
            table = self.tabs.widget(3).settings_table

        if table:
            widths = [table.columnWidth(col) for col in range(table.columnCount())]
            gprefs[f'calibre_config_reports_{table_name}_column_widths'] = widths

    def restore_column_widths(self, table_name):
        """Restore the column widths for a table from preferences"""
        table = None
        if table_name == 'plugins':
            table = getattr(self.plugins_tab, 'plugin_table', None)
        elif table_name == 'columns':
            table = getattr(self, 'columns_table', None)
        elif table_name == 'shortcuts':
            table = getattr(self, 'shortcuts_table', None)
        elif table_name == 'settings':
            # Get the settings table from the SettingsTab instance at index 3
            settings_tab = self.tabs.widget(3)
            if settings_tab and hasattr(settings_tab, 'settings_table'):
                table = settings_tab.settings_table
            else:
                return

        if table:
            widths = gprefs.get(f'calibre_config_reports_{table_name}_column_widths', None)
            # Normalize widths: support list, dict (possibly with string keys), or string repr
            try:
                if widths is None:
                    norm_widths = None
                elif isinstance(widths, dict):
                    # Convert dict of {index: width} to list ordered by index
                    items = []
                    for k, v in widths.items():
                        try:
                            idx = int(k)
                        except Exception:
                            continue
                        items.append((idx, int(v)))
                    items.sort()
                    norm_widths = [w for i, w in items]
                elif isinstance(widths, (list, tuple)):
                    norm_widths = [int(w) if w is not None else 0 for w in widths]
                elif isinstance(widths, str):
                    # Try to eval a dict/list safely
                    try:
                        import ast
                        parsed = ast.literal_eval(widths)
                        if isinstance(parsed, dict):
                            items = []
                            for k, v in parsed.items():
                                try:
                                    idx = int(k)
                                except Exception:
                                    continue
                                items.append((idx, int(v)))
                            items.sort()
                            norm_widths = [w for i, w in items]
                        elif isinstance(parsed, (list, tuple)):
                            norm_widths = [int(w) for w in parsed]
                        else:
                            norm_widths = None
                    except Exception:
                        norm_widths = None
                else:
                    norm_widths = None
            except Exception:
                norm_widths = None
            widths = norm_widths
            if widths:
                # Set flag to indicate widths are being restored
                if table_name == 'plugins' and hasattr(self, 'plugins_tab'):
                    self.plugins_tab._widths_restored = True

                for col, width in enumerate(widths):
                    if col < table.columnCount():
                            # Ensure width is an integer and not too small (gprefs might store as string)
                            try:
                                width_int = int(width)
                                # Special handling for plugins table icon column (column 1) and remove column (column 0)
                                if table_name == 'plugins' and col == 0:
                                    # Remove column should always be 30px
                                    table.setColumnWidth(col, 30)
                                elif table_name == 'plugins' and col == 1:
                                    # Icon column should always be 48px
                                    table.setColumnWidth(col, 48)
                                else:
                                    min_width = 110  # Minimum reasonable width for any column - increased to 110px
                                    if width_int < min_width:
                                        # Use minimum width if saved value is too small
                                        table.setColumnWidth(col, min_width)
                                    else:
                                        table.setColumnWidth(col, width_int)
                            except (ValueError, TypeError):
                                # Skip invalid width values
                                pass

    def load_plugin_from_file(self):
        """Load a plugin from a ZIP file, using Calibre's API, with debug prints for install flow tracking."""
        # from calibre.utils.logging import prints
        from calibre.gui2 import error_dialog, dynamic
        import calibre.customize.ui as ccu
        config = ccu.config
        from calibre.gui2.dialogs.plugin_updater import notify_on_successful_install
        from qt.core import QFileDialog
        from calibre.utils.localization import _
        import os
        import sys

        def get_downloads_directory():
            if sys.platform.startswith('win'):
                try:
                    import winreg
                    with winreg.OpenKey(winreg.HKEY_CURRENT_USER, 'SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders') as key:
                        return winreg.QueryValueEx(key, '{374DE290-123F-4565-9164-39C4925E467B}')[0]
                except:
                    pass
            candidates = [
                os.path.join(os.path.expanduser('~'), 'Downloads'),
                os.path.join(os.path.expanduser('~'), 'downloads'),
                os.path.join(os.path.expanduser('~'), 'Desktop', 'Downloads'),
            ]
            for candidate in candidates:
                if os.path.exists(candidate) and os.path.isdir(candidate):
                    return candidate
            return os.path.expanduser('~')

        last_dir = dynamic.get('load_plugin_last_directory', get_downloads_directory())
        paths = QFileDialog.getOpenFileNames(self, _('Select plugin ZIP file(s)'), os.path.expanduser(last_dir), filter=_('Plugin files (*.zip)'))[0]
        if not paths:
            return
        if paths:
            dynamic['load_plugin_last_directory'] = os.path.dirname(paths[0])

        # Load each selected plugin
        success_count = 0
        error_count = 0
        errors = []

        try:
            for path in paths:
                try:
                    installed_plugins = frozenset(config['plugins'])
                    plugin = ccu.add_plugin(path)
                    # --- BEGIN: Show toolbar placement prompt for GUI plugins, like Calibre built-in ---
                    try:
                        from calibre.customize import InterfaceActionBase, EditBookToolPlugin
                        from calibre.gui2.preferences.plugins import ConfigWidget
                        is_gui_plugin = isinstance(plugin, (InterfaceActionBase, EditBookToolPlugin))
                        if is_gui_plugin:
                            previously_installed = plugin.name in installed_plugins
                            ConfigWidget.check_for_add_to_toolbars(self, plugin, previously_installed=previously_installed)
                    except Exception as e:
                        pass
                    # --- END: Show toolbar placement prompt ---
                    restart_needed = notify_on_successful_install(self, plugin)
                    if restart_needed:
                        success_count += 1
                        self.gui.quit(restart=True)
                        return  # Exit early if restart is needed

                    success_count += 1

                except Exception as e:
                    error_count += 1
                    error_msg = f"Failed to load {os.path.basename(path)}: {str(e)}"
                    errors.append(error_msg)

            # Show summary message if no restart was needed
            if success_count > 0:
                summary_msg = f"Successfully loaded {success_count} plugin(s)."
                if error_count > 0:
                    summary_msg += f"\n{error_count} plugin(s) failed to load."
                    if errors:
                        summary_msg += "\n\nErrors:\n" + "\n".join(errors)

                from calibre.gui2 import info_dialog, error_dialog
                if error_count > 0:
                    error_dialog(self, _('Plugin Loading Results'), summary_msg, show=True)
                else:
                    info_dialog(self, _('Plugin Loading Complete'), summary_msg, show=True)

                # Refresh the plugins table to show newly loaded plugins
                self.plugins_tab.populate_plugins_table()

            elif error_count > 0:
                error_dialog(self, _('Plugin Loading Failed'),
                           f"Failed to load all {len(paths)} plugin(s).\n\nErrors:\n" + "\n".join(errors),
                           show=True)

        except Exception as e:
            return error_dialog(self, _('Error loading plugin'), _('Failed to load plugin: %s') % str(e), show=True)

    def __init__(self, gui):
        # from calibre.utils.logging import prints
        QDialog.__init__(self, gui)

        # Prevent flicker: disable updates until dialog fully set up
        try:
            self.setUpdatesEnabled(False)
        except Exception:
            pass

        # Ensure our plugin install/remove hooks are installed so history logging works
        try:
            install_plugin_hooks()
        except Exception:
            pass

        # Initialize plugin cache quickly from disk only (no network) to avoid UI delay
        self.plugin_cache = load_plugin_cache()
        self.gui = gui

        # Set window title and restore previous geometry/size
        self.setWindowTitle(_('Calibre Config Reports'))
        geometry = gprefs.get('calibre_config_reports_dialog_geometry', None)
        if (geometry):
            self.restoreGeometry(geometry)
        else:
            self.resize(1200, 700)  # Default size if no saved geometry

        # Set minimum size constraints
        self.setMinimumWidth(1200)  # Ensure dialog is never smaller than 1170px
        self.setMinimumHeight(650)  # Ensure dialog is never smaller than 650px

        # Apply more discreet scrollbar styling and global button padding
        self.setStyleSheet("""
            QScrollBar::handle {
               border: 1px solid #5B6985; /* subtle border */
            }
            QScrollBar:vertical {
                background: transparent;
                width: 12px;
                margin: 0px 0px 0px 0px;
            }
            QScrollBar::handle:vertical {
                background: rgba(140, 172, 204, 0.25); /* subtle blue, low opacity */
                min-height: 24px;
                border-radius: 4px;
            }
            QScrollBar::handle:vertical:hover {
                background: rgba(140, 172, 204, 0.45);
            }
            QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {
                height: 0px;
                subcontrol-origin: margin;
            }
            QScrollBar:horizontal {
                background: transparent;
                height: 12px;
                margin: 0px 0px 0px 0px;
            }
            QScrollBar::handle:horizontal {
                background: rgba(140, 172, 204, 0.25);
                min-width: 24px;
                border-radius: 4px;
            }
            QScrollBar::handle:horizontal:hover {
                background: rgba(140, 172, 204, 0.45);
            }
            QScrollBar::add-line:horizontal, QScrollBar::sub-line:horizontal {
                width: 0px;
                subcontrol-origin: margin;
            }
            QScrollBar::up-arrow:vertical, QScrollBar::down-arrow:vertical,
            QScrollBar::left-arrow:horizontal, QScrollBar::right-arrow:horizontal {
                width: 0px;
                height: 0px;
                background: none;
            }
            QScrollBar::add-page:vertical, QScrollBar::sub-page:vertical,
            QScrollBar::add-page:horizontal, QScrollBar::sub-page:horizontal {
                background: none;
            }
        """)
        # --- Add clear button to Settings tab filter (if not already present) ---
        # This is handled in settings_tab.py, but ensure consistency if needed
        # --- End clear button for Settings tab ---



        # Create main layout
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        # Create info container with proper styling
        info_container = QWidget()
        info_container.setObjectName("infoContainer")
        info_container.setStyleSheet("""
            #infoContainer {
                background: palette(window);
                margin-bottom: 8px;
                padding: 4px;
            }
        """)

        # Enable context menu for copying system info
        info_container.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        info_container.customContextMenuRequested.connect(self.show_system_info_context_menu)

        # Keep a reference to the info layout so other handlers (eg. cell click)
        # can add temporary visual feedback into the info container.
        self.info_layout = QVBoxLayout(info_container)
        self.info_layout.setContentsMargins(4, 4, 4, 4)
        self.info_layout.setSpacing(0)


        # Add a non-clickable info label showing install type
        parent_folder = os.path.dirname(config_dir)
        if isportable:
            info_text = f"Portable Mode @ {parent_folder}"
        else:
            # For non-portable installs show a short, clear label
            info_text = "Default Install"
        info_label = QLabel(info_text)
        info_label.setStyleSheet("font-size: 11pt; font-weight: bold; color: palette(text);")
        info_label.setTextInteractionFlags(Qt.TextInteractionFlag.NoTextInteraction)
        info_label.setAlignment(Qt.AlignmentFlag.AlignHCenter)
        self.info_layout.addWidget(info_label)

        # Add subtle gradient separator
        self.info_layout.addSpacing(8)
        separator = QFrame()
        separator.setStyleSheet("""
            QFrame {
                border: none;
                height: 1px;
                margin: 0px 40px;
                background: qlineargradient(
                    x1:0, y1:0, x2:1, y2:0,
                    stop:0 transparent,
                    stop:0.25 rgba(140, 172, 204, 0.5),
                    stop:0.75 rgba(140, 172, 204, 0.5),
                    stop:1 transparent
                );
            }
        """)
        separator.setFixedHeight(1)
        self.info_layout.addWidget(separator)

        # Add statistics bar
        stats_bar = self.initialize_stats_bar()
        # Set minimal margins on stats_bar if it's a layout or widget
        if hasattr(stats_bar, 'setContentsMargins'):
            stats_bar.setContentsMargins(0, 0, 0, 0)
        self.info_layout.addLayout(stats_bar)

        # Add the info container to main layout
        layout.addWidget(info_container)

        # Initialize icons
        icon = common_icons.get_icon('images/iconplugin')
        if (icon and not icon.isNull()):
            self.setWindowIcon(icon)
        self.forum_icon = common_icons.get_icon('images/mobileread')
        self.copy_icon = QIcon.ic('edit-copy.png')
        self.copy_done_icon = QIcon.ic('ok.png')
        self.donate_icon = common_icons.get_icon('images/donate')
        self.history_icon = common_icons.get_icon('images/history')

        # Plugin cache is now handled in plugins_tab.py

        # Create tab widget with styling
        self.tabs = QTabWidget(self)
        #self.tabs.setMinimumWidth(1100)
        self.tabs.setStyleSheet("""
            QTabWidget::pane {
                border-top: none;
                   margin-left: 8px;
                   margin-right: 8px;
            }
            QTabBar::tab {
                min-width: 164px;
                width: 3px;
                padding-top: 6px;
                padding-bottom: 6px;
                font-size: 10pt;
                background: transparent;
                border-top: 1px solid palette(mid);
                border-left: 1px solid palette(mid);
                border-right: 1px solid palette(mid);
                border-top-left-radius: 8px;
                border-top-right-radius: 8px;
                /*margin-right: 2px;*/
            }
            QTabBar::tab:selected {
                font-weight: bold;
                font-style: normal;
                border-top: 2px solid palette(link);
                border-left: 2px solid palette(link);
                border-right: 2px solid palette(link);
                color: palette(link);
            }
            QTabBar::tab:!selected {
                color: palette(text);
                margin-top: 2px;
            }
        """)
        # Make all tabs expand equally to fill the width
        # self.tabs.tabBar().setExpanding(True)
        layout.addWidget(self.tabs)

        # Bottom-right action row: Open configuration folder & Quit
        bottom_row = QHBoxLayout()
        bottom_row.addStretch(1)

        self.open_cfg_btn = QPushButton(_('&Open configuration folder'))
        self.open_cfg_btn.setToolTip(_('Open calibre configuration/profile folder'))
        self.open_cfg_btn.setIcon(QIcon.ic('folder.png'))
        self.open_cfg_btn.setAutoDefault(False)
        self.open_cfg_btn.setDefault(False)
        self.open_cfg_btn.clicked.connect(lambda: self.open_path(config_dir))
        bottom_row.addWidget(self.open_cfg_btn)

        self.quit_btn = QPushButton(_('&Quit'))
        self.quit_btn.setToolTip(_('Close this window'))
        # Note: We intentionally allow Enter to activate focused buttons for consistency with calibre dialogs
        self.quit_btn.setIcon(QIcon.ic('window-close.png'))
        self.quit_btn.setAutoDefault(False)
        self.quit_btn.setDefault(False)
        self.quit_btn.clicked.connect(self.reject)
        bottom_row.addWidget(self.quit_btn)

        layout.addLayout(bottom_row)

    # Begin tab creation block
    # --- Add tabs in fixed order ---
        # 1. Plugins - using the PluginsTab class from plugins_tab.py
        from calibre_plugins.calibre_config_reports.plugins_tab import PluginsTab
        self.plugins_tab = PluginsTab(self)
        self.tabs.addTab(self.plugins_tab, _('Plugins'))
        # 2. Custom Columns
        columns_tab = QWidget()
        columns_layout = QVBoxLayout()
        columns_tab.setLayout(columns_layout)
        # Search bar for custom columns
        columns_filter_layout = QHBoxLayout()
        columns_filter_label = QLabel(_('Filter:'))
        self.columns_filter_edit = QLineEdit()
        self.columns_filter_edit.setPlaceholderText(_('Search custom columns...'))
        self.columns_filter_edit.textChanged.connect(self.apply_columns_filter)
        # Prevent Enter from activating default buttons; make Return a no-op for this filter
        try:
            # Ensure pressing Enter inside the filter does nothing (avoids triggering nearby buttons)
            self.columns_filter_edit.returnPressed.connect(lambda: None)
        except Exception:
            pass
        columns_filter_layout.addWidget(columns_filter_label)
        # Add a clear action inside the QLineEdit (more reliable than manual QToolButton)
        clear_action = self.columns_filter_edit.addAction(QIcon.ic('clear_left.png'), QLineEdit.TrailingPosition)
        try:
            clear_action.setToolTip(_('Clear filter'))
            clear_action.setVisible(False)
            # QAction.triggered doesn't always exist on some Qt bindings; wrap safely
            try:
                clear_action.triggered.connect(self.columns_filter_edit.clear)
            except Exception:
                # Fallback: use a small lambda that clears the field when activated
                clear_action.triggered.connect(lambda: self.columns_filter_edit.clear())
        except Exception:
            pass

        # Show/hide the clear action based on text presence
        self.columns_filter_edit.textChanged.connect(lambda text: clear_action.setVisible(bool(text)))
        columns_filter_layout.addWidget(self.columns_filter_edit)

        # Add Columns button
        self.columns_column_button = QPushButton(_('&Manage Columns'))
        self.columns_column_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        self.columns_column_button.setToolTip(_('Show/hide columns'))
        self.columns_column_button.setAutoDefault(False)
        self.columns_column_button.setDefault(False)
        self.columns_column_button.clicked.connect(self.show_columns_column_menu)
        columns_filter_layout.addWidget(self.columns_column_button)

        # Add export buttons for columns tab with icons
        columns_csv_export_button = QPushButton(_('Export &CSV'))
        columns_csv_export_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        columns_csv_export_button.setIcon(QIcon.ic('save.png'))
        columns_csv_export_button.setToolTip(_('Export visible custom columns to CSV'))
        columns_csv_export_button.setAutoDefault(False)
        columns_csv_export_button.setDefault(False)
        columns_csv_export_button.clicked.connect(lambda: self.export_data('csv'))
        columns_xlsx_export_button = QPushButton(_('Export &XLSX'))
        columns_xlsx_export_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        columns_xlsx_export_button.setIcon(QIcon.ic('save.png'))
        columns_xlsx_export_button.setToolTip(_('Export visible custom columns to Excel (XLSX)'))
        columns_xlsx_export_button.setAutoDefault(False)
        columns_xlsx_export_button.setDefault(False)
        columns_xlsx_export_button.clicked.connect(lambda: self.export_data('xlsx'))
        columns_filter_layout.addWidget(columns_csv_export_button)
        columns_filter_layout.addWidget(columns_xlsx_export_button)

        # Add Reset Columns button for Custom Columns tab
        from calibre.gui2 import question_dialog, info_dialog
        reset_columns_button = QPushButton('\u00A0' + _('&Reset Columns'))
        reset_columns_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        reset_columns_button.setIcon(QIcon.ic('view-refresh.png'))
        reset_columns_button.setToolTip(_('Restore default column visibility and order'))
        # Prevent Enter in nearby inputs from activating this button
        reset_columns_button.setAutoDefault(False)
        reset_columns_button.setDefault(False)
        reset_columns_button.clicked.connect(self.restore_default_columns)
        columns_filter_layout.addWidget(reset_columns_button)
        columns_layout.addLayout(columns_filter_layout)

        # Custom columns table
        self.columns_table = QTableWidget()
        self.setup_columns_table()
        columns_layout.addWidget(self.columns_table)

        # Add Custom Columns tab
        self.tabs.addTab(columns_tab, _('Custom Columns'))
        # 3. Shortcuts
        shortcuts_tab = QWidget()
        shortcuts_layout = QVBoxLayout(shortcuts_tab)

        # Search bar for shortcuts at the top
        shortcuts_filter_layout = QHBoxLayout()
        shortcuts_filter_label = QLabel(_('Filter:'))
        self.shortcuts_filter_edit = QLineEdit()
        self.shortcuts_filter_edit.setPlaceholderText(_('Search shortcuts by name, group or key combination...'))
        self.shortcuts_filter_edit.setToolTip(_('Search shortcuts by name, group, or key combination.\n\nTips:\n- Type "alt" to find all Alt shortcuts\n- Enable "Regex" for patterns like "alt\\+f[0-9]+" (Alt+F1, F2, etc.)\n- Use the dropdown to filter by shortcut type (User Plugins, Built-in Features, etc.)'))
        self.shortcuts_filter_edit.textChanged.connect(lambda: self.apply_shortcuts_filter(self.shortcuts_filter_edit.text()))
        try:
            self.shortcuts_filter_edit.returnPressed.connect(lambda: self.apply_shortcuts_filter(self.shortcuts_filter_edit.text()))
        except Exception:
            pass
        shortcuts_filter_layout.addWidget(shortcuts_filter_label)
        # Add clear button visually inside the QLineEdit (like plugins tab)
        self.shortcuts_filter_clear_btn = QToolButton(self.shortcuts_filter_edit)
        self.shortcuts_filter_clear_btn.setIcon(QIcon.ic('clear_left.png'))
        self.shortcuts_filter_clear_btn.setToolTip(_('Clear filter'))
        self.shortcuts_filter_clear_btn.setCursor(Qt.CursorShape.ArrowCursor)
        self.shortcuts_filter_clear_btn.setVisible(False)
        self.shortcuts_filter_clear_btn.setStyleSheet('QToolButton { border: none; padding: 0px; }')
        self.shortcuts_filter_clear_btn.setFixedSize(18, 18)
        self.shortcuts_filter_clear_btn.clicked.connect(self.shortcuts_filter_edit.clear)
        self.shortcuts_filter_edit.textChanged.connect(
            lambda text: self.shortcuts_filter_clear_btn.setVisible(bool(text)))
        from PyQt5.QtWidgets import QStyle
        frame = self.shortcuts_filter_edit.style().pixelMetric(QStyle.PixelMetric.PM_DefaultFrameWidth)
        self.shortcuts_filter_edit.setTextMargins(0, 0, self.shortcuts_filter_clear_btn.width() + frame + 2, 0)
        def move_clear_btn():
            self.shortcuts_filter_clear_btn.move(
                self.shortcuts_filter_edit.rect().right() - self.shortcuts_filter_clear_btn.width() - frame,
                (self.shortcuts_filter_edit.rect().height() - self.shortcuts_filter_clear_btn.height()) // 2)
        self.shortcuts_filter_edit.resizeEvent = lambda event: move_clear_btn()
        move_clear_btn()
        shortcuts_filter_layout.addWidget(self.shortcuts_filter_edit)

        # Add shortcuts filter dropdown (user-friendly presets)
        shortcuts_type_label = QLabel(_('Show:'))
        self.shortcuts_type_dropdown = QComboBox()
        self.shortcuts_type_dropdown.addItems([
            _('All Shortcuts'),
            _('Key Combos (Ctrl+X, etc.)'),
            _('Single Keys (F1, M, etc.)'),
            _('Unused Shortcuts'),
            _('Assigned Shortcuts'),
            _('User defined shortcuts'),
            _('Plugin shortcuts'),
            _('Built-in Features')
        ])
        self.shortcuts_type_dropdown.setMinimumWidth(140)
        self.shortcuts_type_dropdown.setToolTip(_('Choose what type of shortcuts to display'))
        self.shortcuts_type_dropdown.currentIndexChanged.connect(lambda: self.apply_shortcuts_filter(self.shortcuts_filter_edit.text()))
        shortcuts_filter_layout.addWidget(shortcuts_type_label)
        shortcuts_filter_layout.addWidget(self.shortcuts_type_dropdown)

        # Keep regex as an advanced option (smaller, less prominent)
        self.shortcuts_regex_checkbox = QCheckBox(_('Regex'))
        self.shortcuts_regex_checkbox.setToolTip(_('Use regular expressions for filtering (advanced users)'))
        self.shortcuts_regex_checkbox.stateChanged.connect(lambda: self.apply_shortcuts_filter(self.shortcuts_filter_edit.text()))
        shortcuts_filter_layout.addWidget(self.shortcuts_regex_checkbox)

        # Add column manager button (shorter label to reduce clutter). Use a
        # mnemonic on 'm' so Alt+M still maps to this control when appropriate.
        self.shortcuts_column_button = QPushButton(_('Colu&mns'))
        # Make buttons slightly smaller here so the filter edit has more room
        self.shortcuts_column_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.shortcuts_column_button.setToolTip(_('Show/hide columns'))
        self.shortcuts_column_button.setAutoDefault(False)
        self.shortcuts_column_button.setDefault(False)
        self.shortcuts_column_button.clicked.connect(self.show_shortcuts_column_menu)
        shortcuts_filter_layout.addWidget(self.shortcuts_column_button)

        # Add Reset Columns button for Shortcuts tab (mirrors Plugins/Settings behavior)
        self.shortcuts_restore_defaults_button = QPushButton('\u00A0' + _('&Reset Columns'))
        # Match icon and slightly smaller sizing used on other tabs
        self.shortcuts_restore_defaults_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.shortcuts_restore_defaults_button.setIcon(QIcon.ic('view-refresh.png'))
        self.shortcuts_restore_defaults_button.setToolTip(_('Restore default column visibility and order'))
        self.shortcuts_restore_defaults_button.setAutoDefault(False)
        self.shortcuts_restore_defaults_button.setDefault(False)
        self.shortcuts_restore_defaults_button.clicked.connect(self.restore_default_shortcuts_columns)
        shortcuts_filter_layout.addWidget(self.shortcuts_restore_defaults_button)

        # Add refresh button for shortcuts
        self.shortcuts_refresh_button = QPushButton(_('Re&fresh'))
        self.shortcuts_refresh_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        self.shortcuts_refresh_button.setIcon(QIcon.ic('view-refresh.png'))
        self.shortcuts_refresh_button.setToolTip(_('Reload shortcuts from Calibre (click this after changing shortcuts in Preferences while CCR is open)'))
        self.shortcuts_refresh_button.setAutoDefault(False)
        self.shortcuts_refresh_button.setDefault(False)
        self.shortcuts_refresh_button.clicked.connect(self.refresh_shortcuts)
        shortcuts_filter_layout.addWidget(self.shortcuts_refresh_button)

        # Add export buttons for shortcuts tab with icons
        shortcuts_csv_export_button = QPushButton(_('Export &CSV'))
        shortcuts_csv_export_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        shortcuts_csv_export_button.setIcon(QIcon.ic('save.png'))
        shortcuts_csv_export_button.setToolTip(_('Export visible shortcuts to CSV'))
        shortcuts_csv_export_button.setAutoDefault(False)
        shortcuts_csv_export_button.setDefault(False)
        shortcuts_csv_export_button.clicked.connect(lambda: self.export_shortcuts('csv'))
        shortcuts_xlsx_export_button = QPushButton(_('Export &XLSX'))
        shortcuts_xlsx_export_button.setStyleSheet('padding: 6px 12px; font-size:9pt;')
        shortcuts_xlsx_export_button.setIcon(QIcon.ic('save.png'))
        shortcuts_xlsx_export_button.setToolTip(_('Export visible shortcuts to Excel (XLSX)'))
        shortcuts_xlsx_export_button.setAutoDefault(False)
        shortcuts_xlsx_export_button.setDefault(False)
        shortcuts_xlsx_export_button.clicked.connect(lambda: self.export_shortcuts('xlsx'))
        shortcuts_filter_layout.addWidget(shortcuts_csv_export_button)
        shortcuts_filter_layout.addWidget(shortcuts_xlsx_export_button)

        shortcuts_layout.addLayout(shortcuts_filter_layout)

        # Shortcuts table
        self.shortcuts_table = QTableWidget()
        self.setup_shortcuts_table()
        # On Enter/Return, open Keyboard Shortcuts preferences and swallow event
        try:
            # Double-click already opens; add itemActivated for Enter
            self.shortcuts_table.itemActivated.connect(lambda item: self.gui.iactions['Preferences'].do_config(
                initial_plugin=('Advanced', 'Keyboard'), close_after_initial=True) if item else None)
        except Exception:
            pass
        shortcuts_layout.addWidget(self.shortcuts_table)

        # Add Shortcuts tab
        self.tabs.addTab(shortcuts_tab, _('Shortcuts'))
        # 4. Calibre Settings
        settings_tab = self.setup_settings_tab()
        self.tabs.addTab(settings_tab, _('Settings'))
        # 5. History
        history_tab = self.setup_history_tab()
        self.tabs.addTab(history_tab, _('History'))
        # 6. Help (new tab)
        help_tab = self.setup_help_tab()
        self.tabs.addTab(help_tab, _('Help'))

        # Connect tab change handler and lazy loader
        self.tabs.currentChanged.connect(self.on_tab_changed)
        self.tabs.currentChanged.connect(self._lazy_load_tab)

        # Tables will be populated after dialog is shown to keep UI responsive

        # Restore last active tab (if out of range, default to 0)
        last_tab = gprefs.get('calibre_config_reports_last_tab', 0)  # Default to first tab
        if last_tab >= self.tabs.count():
            last_tab = 0
        self.tabs.setCurrentIndex(last_tab)

        # Defer heavy operations to after dialog is shown
        QTimer.singleShot(50, self.perform_deferred_initialization)

        # Register CCR actions with Calibre's keyboard manager (unassigned by default)
        self._setup_ccr_actions()
        # Snapshot used to avoid expensive repopulation of Shortcuts tab when nothing changed
        try:
            self._shortcuts_snapshot = None
        except Exception:
            pass
        # Re-enable UI updates now that initialization is complete
        try:
            self.setUpdatesEnabled(True)
        except Exception:
            pass
        # Initialize set of populated tabs for lazy loading
        try:
            self._populated_tabs = set()
        except Exception:
            pass

    def _populate_all_tables(self):
        """Populate plugins and other tables after dialog is shown to avoid blocking UI during init."""
        # Populate plugins tab first for column state restoration
        try:
            if hasattr(self.plugins_tab, 'populate_plugins_table'):
                # Temporarily disable sorting to speed up insertion
                try:
                    self.plugins_tab.plugin_table.setSortingEnabled(False)
                except Exception:
                    pass
                self.plugins_tab.populate_plugins_table()
                try:
                    self.plugins_tab.plugin_table.setSortingEnabled(True)
                except Exception:
                    pass
        except Exception:
            pass
        # Populate other tables (columns, shortcuts, settings)
        try:
            self.populate_tables()
        except Exception:
            pass

    def _setup_ccr_actions(self):
        """Create QActions and register them with Calibre's keyboard manager.
        Users can assign their own keys in Preferences > Keyboard.
        """
        try:
            kb = getattr(self.gui, 'keyboard', None)
            if kb is None:
                return

            # Export CSV
            self.ccr_export_csv = QAction(_('Export current tab to CSV'), self)
            self.ccr_export_csv.triggered.connect(lambda: self.export_current_tab('csv'))
            self.addAction(self.ccr_export_csv)
            kb.register_shortcut(
                'Calibre Config Reports::Export CSV',
                _('CCR: Export current tab to CSV'),
                default_keys=(), action=self.ccr_export_csv,
                group=_('Calibre Config Reports')
            )

            # Export XLSX
            self.ccr_export_xlsx = QAction(_('Export current tab to Excel (XLSX)'), self)
            self.ccr_export_xlsx.triggered.connect(lambda: self.export_current_tab('xlsx'))
            self.addAction(self.ccr_export_xlsx)
            kb.register_shortcut(
                'Calibre Config Reports::Export XLSX',
                _('CCR: Export current tab to Excel (XLSX)'),
                default_keys=(), action=self.ccr_export_xlsx,
                group=_('Calibre Config Reports')
            )

            # Next/Previous tab (optional; unassigned by default)
            self.ccr_next_tab = QAction(_('Next tab'), self)
            self.ccr_next_tab.triggered.connect(lambda: self.change_tab(1))
            self.addAction(self.ccr_next_tab)
            kb.register_shortcut(
                'Calibre Config Reports::Next Tab',
                _('CCR: Next tab'), default_keys=(), action=self.ccr_next_tab,
                group=_('Calibre Config Reports')
            )

            self.ccr_prev_tab = QAction(_('Previous tab'), self)
            self.ccr_prev_tab.triggered.connect(lambda: self.change_tab(-1))
            self.addAction(self.ccr_prev_tab)
            kb.register_shortcut(
                'Calibre Config Reports::Previous Tab',
                _('CCR: Previous tab'), default_keys=(), action=self.ccr_prev_tab,
                group=_('Calibre Config Reports')
            )

            # Load plugin(s) from file (unassigned by default)
            self.ccr_load_plugin = QAction(_('Load plugin(s) from file...'), self)
            self.ccr_load_plugin.triggered.connect(self.load_plugin_from_file)
            self.addAction(self.ccr_load_plugin)
            kb.register_shortcut(
                'Calibre Config Reports::Load plugin from file',
                _('CCR: Load plugin(s) from file'),
                default_keys=(), action=self.ccr_load_plugin,
                group=_('Calibre Config Reports')
            )

            # Copy system information (unassigned by default)
            self.ccr_copy_system_info = QAction(_('Copy system information'), self)
            self.ccr_copy_system_info.triggered.connect(self.copy_system_info_to_clipboard)
            self.addAction(self.ccr_copy_system_info)
            kb.register_shortcut(
                'Calibre Config Reports::Copy system information',
                _('CCR: Copy system information'),
                default_keys=(), action=self.ccr_copy_system_info,
                group=_('Calibre Config Reports')
            )

            # Apply any existing user-defined shortcuts to these actions
            kb.finalize()
            # If Shortcuts tab exists, refresh it so new actions appear immediately
            try:
                if hasattr(self, 'shortcuts_table'):
                    self.refresh_shortcuts()
            except Exception:
                pass
            # --- Add application-level QShortcuts so Alt accelerators work when focus is in filter boxes or tables ---
            try:
                from PyQt5.QtWidgets import QShortcut
                from PyQt5.QtGui import QKeySequence
                # Alt+M -> Manage Columns / Remove Selected
                sc_m = QShortcut(QKeySequence('Alt+M'), self)
                sc_m.setContext(Qt.ApplicationShortcut)
                sc_m.activated.connect(lambda: self._handle_alt_m())

                # Alt+S -> Export CSV (current tab)
                sc_s = QShortcut(QKeySequence('Alt+S'), self)
                sc_s.setContext(Qt.ApplicationShortcut)
                sc_s.activated.connect(lambda: self.export_current_tab('csv'))

                # Alt+U -> Open Plugin Updater
                sc_u = QShortcut(QKeySequence('Alt+U'), self)
                sc_u.setContext(Qt.ApplicationShortcut)
                sc_u.activated.connect(lambda: self._handle_alt_u())

                # Alt+P -> Open Plugin Preferences / Load Plugin
                sc_p = QShortcut(QKeySequence('Alt+P'), self)
                sc_p.setContext(Qt.ApplicationShortcut)
                sc_p.activated.connect(lambda: self._handle_alt_p())
                # Alt+L -> Load Plugin from File
                sc_l = QShortcut(QKeySequence('Alt+L'), self)
                sc_l.setContext(Qt.ApplicationShortcut)
                sc_l.activated.connect(lambda: self._handle_alt_l())

                # Alt+D -> Remove Selected Plugins
                sc_d = QShortcut(QKeySequence('Alt+D'), self)
                sc_d.setContext(Qt.ApplicationShortcut)
                sc_d.activated.connect(lambda: self._handle_alt_d())

                # Alt+X -> Export XLSX
                sc_x = QShortcut(QKeySequence('Alt+X'), self)
                sc_x.setContext(Qt.ApplicationShortcut)
                sc_x.activated.connect(lambda: self.export_current_tab('xlsx'))

                # Alt+R -> Reset / Restore Columns
                sc_r = QShortcut(QKeySequence('Alt+R'), self)
                sc_r.setContext(Qt.ApplicationShortcut)
                sc_r.activated.connect(lambda: self._handle_alt_r())
            except Exception:
                pass
        except Exception:
            # Non-fatal if keyboard manager not ready
            pass

    def _handle_alt_m(self):
        """Handle Alt+M: prefer plugin tab Manage Columns, then Remove Selected if focus in plugins tab buttons"""
        try:
            # Prefer the current tab's column button if present (helps keyboard users
            # open the appropriate Columns menu when focus is on other tabs).
            try:
                cur_idx = self.tabs.currentIndex() if hasattr(self, 'tabs') else None
                # Shortcuts tab index is 2 in our layout
                if cur_idx == 2 and hasattr(self, 'shortcuts_restore_defaults_button') and hasattr(self, 'shortcuts_column_button'):
                    w = self.shortcuts_column_button
                    if w and w.isVisible():
                        w.click()
                        return
            except Exception:
                pass

            # If current tab didn't handle it, fall back to Plugins tab column button
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'column_button'):
                w = self.plugins_tab.column_button
                if w and w.isVisible():
                    w.click()
                    return
            # Fallback: if plugins tab has remove_selected_btn, trigger it (dangerous)
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'remove_selected_btn'):
                btn = self.plugins_tab.remove_selected_btn
                if btn and btn.isVisible():
                    btn.click()
                    return
        except Exception:
            pass

    def _handle_alt_u(self):
        """Handle Alt+U: open plugin updater if available"""
        try:
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'updater_button'):
                btn = self.plugins_tab.updater_button
                if btn and btn.isVisible() and btn.isEnabled():
                    try:
                        btn.click()
                        return
                    except Exception:
                        pass
                    # Try animateClick as a more forceful UI simulation
                    try:
                        btn.animateClick(50)
                        return
                    except Exception:
                        pass
            # Fallback: attempt to open updater via GUI API directly
            try:
                if hasattr(self.gui, 'show_plugin_update_dialog'):
                    self.gui.show_plugin_update_dialog()
                    return
                else:
                    from calibre.gui2.dialogs.plugin_updater import PluginUpdaterDialog, FILTER_UPDATE_AVAILABLE
                    d = PluginUpdaterDialog(self.gui, initial_filter=FILTER_UPDATE_AVAILABLE)
                    d.exec()
                    if getattr(d, 'do_restart', False):
                        self.gui.quit(restart=True)
                    return
            except Exception:
                pass
        except Exception:
            pass

    def _handle_alt_l(self):
        """Handle Alt+L: Load plugin from file if plugins tab present"""
        try:
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'load_plugin_btn'):
                btn = self.plugins_tab.load_plugin_btn
                if btn and btn.isVisible() and btn.isEnabled():
                    try:
                        btn.click()
                        return
                    except Exception:
                        pass
                    try:
                        btn.animateClick(50)
                        return
                    except Exception:
                        pass
            # Fallback: call the plugins_tab's load routine directly if available
            try:
                if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'load_plugin_from_file'):
                    # call as method on the plugins_tab instance
                    self.plugins_tab.load_plugin_from_file()
                    return
            except Exception:
                pass
        except Exception:
            pass

    def _handle_alt_d(self):
        """Handle Alt+D: trigger Remove Selected Plugins"""
        try:
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'remove_selected_btn'):
                btn = self.plugins_tab.remove_selected_btn
                if btn and btn.isVisible():
                    btn.click()
                    return
        except Exception:
            pass

    def _handle_alt_r(self):
        """Handle Alt+R: Reset/Restore defaults for current tab (plugins -> restore columns)"""
        try:
            # Prefer plugins restore defaults button
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'restore_defaults_button'):
                btn = self.plugins_tab.restore_defaults_button
                if btn and btn.isVisible():
                    btn.click()
                    return
            # Fallback to other tabs' reset functionality if implemented
            # For history/settings/columns, call saved handlers if any
            try:
                current = self.tabs.tabText(self.tabs.currentIndex())
                if current == _('Custom Columns') and hasattr(self, 'columns_column_button'):
                    # No direct reset button in columns tab; rely on existing handlers if any
                    return
            except Exception:
                pass
        except Exception:
            pass

    def _handle_alt_p(self):
        """Handle Alt+P: prefer Plugin Preferences, then Load Plugin"""
        try:
            # Try Plugin Preferences first
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'prefs_button'):
                btn = self.plugins_tab.prefs_button
                if btn and btn.isVisible() and btn.isEnabled():
                    try:
                        btn.click()
                        return
                    except Exception:
                        pass
                    try:
                        btn.animateClick(50)
                        return
                    except Exception:
                        pass
            # Fallback: open Preferences dialog directly
            try:
                if hasattr(self.gui, 'iactions') and 'Preferences' in self.gui.iactions:
                    prefs = self.gui.iactions['Preferences']
                    prefs.do_config(initial_plugin=('Advanced', 'Plugins'), close_after_initial=True)
                    return
            except Exception:
                pass

            # If Preferences didn't run, try Load Plugin
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'load_plugin_btn'):
                btn = self.plugins_tab.load_plugin_btn
                if btn and btn.isVisible() and btn.isEnabled():
                    try:
                        btn.click()
                        return
                    except Exception:
                        pass
                    try:
                        btn.animateClick(50)
                        return
                    except Exception:
                        pass
            # Final fallback: call load routine directly
            try:
                if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'load_plugin_from_file'):
                    self.plugins_tab.load_plugin_from_file()
                    return
            except Exception:
                pass
        except Exception:
            pass

    def change_tab(self, delta):
        try:
            idx = self.tabs.currentIndex()
            count = self.tabs.count()
            self.tabs.setCurrentIndex((idx + delta) % count)
        except Exception:
            pass

    def restore_default_shortcuts_columns(self):
        """Restore default column visibility/order/widths for the Shortcuts tab"""
        from calibre.gui2 import question_dialog, info_dialog
        if not question_dialog(self, _('Restore Default Columns'),
                               _('Reset columns to defaults for the Shortcuts tab?')):
            return

        # Default visible columns (logical indices)
        # Desired visual order: Action, Group, Assigned, Default
        # Logical indices map: 0=Action,1=Group,2=Default,3=Assigned
        desired_columns = [0, 1, 3, 2]

        # Apply visibility
        for col in range(self.shortcuts_table.columnCount()):
            try:
                self.shortcuts_table.setColumnHidden(col, col not in desired_columns)
            except Exception:
                pass

        # Apply visual order
        header = self.shortcuts_table.horizontalHeader()
        for target_visual_pos, logical_idx in enumerate(desired_columns):
            try:
                current_visual = header.visualIndex(logical_idx)
                if current_visual != target_visual_pos:
                    header.moveSection(current_visual, target_visual_pos)
            except Exception:
                pass

        # Reset widths to defaults (match the visual order above)
        try:
            # Action - make wider so action text is readable on reset
            self.shortcuts_table.setColumnWidth(0, 400)
            # Group
            self.shortcuts_table.setColumnWidth(1, 180)
            # Assigned (narrower)
            self.shortcuts_table.setColumnWidth(3, 120)
            # Default (slightly wider)
            self.shortcuts_table.setColumnWidth(2, 140)
        except Exception:
            pass

        # Clear saved gprefs for shortcuts column state so restore is persistent
        gprefs.pop('calibre_config_reports_shortcuts_hidden_columns', None)
        gprefs.pop('calibre_config_reports_shortcuts_column_order', None)
        gprefs.pop('calibre_config_reports_shortcuts_column_widths', None)
        try:
            gprefs.save()
        except Exception:
            pass

        # Persist the new default state
        try:
            self.save_column_state('shortcuts')
        except Exception:
            pass

        info_dialog(self, _('Columns Restored'), _('Shortcuts columns restored to defaults.'), show=True)

    def restore_default_columns(self):
        """Restore default column visibility/order/widths for the Custom Columns tab"""
        from calibre.gui2 import question_dialog, info_dialog
        if not question_dialog(self, _('Restore Default Columns'),
                               _('Reset columns to defaults for the Custom Columns tab?')):
            return

        table = getattr(self, 'columns_table', None)
        if table is None:
            return

        header = table.horizontalHeader()

        # Default: show all columns in logical order (0..n-1)
        desired_columns = list(range(table.columnCount()))

        # Apply visibility
        for col in range(table.columnCount()):
            try:
                table.setColumnHidden(col, col not in desired_columns)
            except Exception:
                pass

        # Apply visual order
        for target_visual_pos, logical_idx in enumerate(desired_columns):
            try:
                current_visual = header.visualIndex(logical_idx)
                if current_visual != target_visual_pos:
                    header.moveSection(current_visual, target_visual_pos)
            except Exception:
                pass

        # Reset widths to sensible defaults
        try:
            # Provide defaults for up to 8 columns; if fewer columns exist only apply what fits
            default_widths = [120, 100, 80, 200, 80, 180, 150, 200]
            for i, w in enumerate(default_widths):
                if i < table.columnCount():
                    table.setColumnWidth(i, w)
        except Exception:
            pass

        # Clear saved gprefs for columns state so restore is persistent
        gprefs.pop('calibre_config_reports_columns_hidden_columns', None)
        gprefs.pop('calibre_config_reports_columns_column_order', None)
        gprefs.pop('calibre_config_reports_columns_column_widths', None)
        try:
            gprefs.save()
        except Exception:
            pass

        # Persist the new default state
        try:
            self.save_column_state('columns')
        except Exception:
            pass

        info_dialog(self, _('Columns Restored'), _('Custom Columns restored to defaults.'), show=True)

    def export_current_tab(self, format_type):
        """Export the current tab to CSV/XLSX by dispatching to the appropriate exporter."""
        try:
            tab_idx = self.tabs.currentIndex()
            tab_name = self.tabs.tabText(tab_idx)
        except Exception:
            tab_name = ''

        try:
            if tab_name == _('Plugins') or tab_name == _('Custom Columns'):
                # Uses generic export_data() that switches table internally
                self.export_data(format_type)
            elif tab_name == _('Shortcuts'):
                self.export_shortcuts(format_type)
            elif tab_name == _('Settings'):
                self.export_settings(format_type)
            elif tab_name == _('History'):
                self.export_history(format_type)
            else:
                # Fallback: try generic export
                self.export_data(format_type)
        except Exception:
            pass

    def refresh_shortcuts(self):
        """Refresh the shortcuts list from Calibre's current keyboard map"""
        try:
            if hasattr(self.gui, 'keyboard') and self.gui.keyboard is not None:
                self.gui.keyboard.finalize()
        except Exception:
            pass
        # Suppress UI updates during manual refresh to avoid flashing
        try:
            self.setUpdatesEnabled(False)
        except Exception:
            pass
        # Force rebuild on manual refresh by clearing cached snapshot
        try:
            self._shortcuts_snapshot = None
        except Exception:
            pass
        try:
            self.populate_shortcuts_tab()
            self.apply_shortcuts_filter(self.shortcuts_filter_edit.text())
            # Restore saved widths/order/visibility so a refresh doesn't reset user preferences
            self.restore_column_state('shortcuts')
        except Exception:
            pass
        finally:
            # Re-enable UI updates after refresh
            try:
                self.setUpdatesEnabled(True)
            except Exception:
                pass

    def perform_deferred_initialization(self):
        """Perform heavy operations after dialog is visible without blocking UI"""
        try:
            # Run settings monitoring (local I/O) - keep on main thread
            from .settings_monitor import (
                monitor_and_log_config_events,
                get_setting_config_info, load_all_config_files, get_setting_value
            )
            monitor_and_log_config_events()

            # Run network-backed cache refresh in a background thread to avoid UI stalls
            import threading
            def _refresh_cache_bg():
                try:
                    fresh = load_plugin_cache_with_network()
                except Exception:
                    fresh = None
                # Apply on UI thread
                QTimer.singleShot(0, lambda: self._apply_fresh_plugin_cache(fresh))

            threading.Thread(target=_refresh_cache_bg, daemon=True).start()

        except Exception:
            # If deferred operations fail, continue anyway
            pass

    def _apply_fresh_plugin_cache(self, fresh):
        """Apply refreshed plugin cache on UI thread"""
        if isinstance(fresh, dict) and fresh:
            self.plugin_cache = fresh
            if hasattr(self.plugins_tab, 'plugin_cache'):
                self.plugins_tab.plugin_cache = self.plugin_cache
                # Rebuild lookup map in plugins tab to reflect new cache keys
                try:
                    if hasattr(self.plugins_tab, '_build_plugin_cache_lookup'):
                        self.plugins_tab._build_plugin_cache_lookup()
                except Exception:
                    pass
            # Refresh the plugins table so new cache fields (e.g., last_modified) appear
            try:
                if hasattr(self.plugins_tab, 'populate_plugins_table'):
                    self.plugins_tab.populate_plugins_table()
            except Exception:
                pass

    def setup_settings_tab(self):
        """Setup the Calibre Settings tab"""
        # Use the new separated SettingsTab class
        settings_tab = SettingsTab(self)
        return settings_tab

    def setup_settings_table(self):
        """Setup the settings table structure"""
        self.settings_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.settings_table.setAlternatingRowColors(True)
        self.settings_table.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)
        self.settings_table.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)
        self.settings_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

        # Setup columns
        self.settings_table.setColumnCount(4)
        headers = [_('Category'), _('Setting'), _('Value'), _('Source')]
        self.settings_table.setHorizontalHeaderLabels(headers)

        # Enable sorting
        self.settings_table.setSortingEnabled(True)

        # Enable context menu
        self.settings_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.settings_table.customContextMenuRequested.connect(self.show_settings_context_menu)

        # Enable double-click to open main Preferences dialog
        self.settings_table.cellDoubleClicked.connect(self.open_main_preferences_dialog)

        # Enable column reordering
        header = self.settings_table.horizontalHeader()
        header.setSectionsMovable(True)
        header.sectionMoved.connect(lambda: self.save_column_state('settings'))

        # Use a safer callback that doesn't rely on self.settings_table access
        def on_settings_section_resized():
            try:
                self.save_column_state('settings')
            except AttributeError:
                # If settings_table is not accessible, try to save directly using the table
                if hasattr(self, 'settings_table'):
                    self.save_column_state('settings')
        header.sectionResized.connect(on_settings_section_resized)

        # Restore column state
        self.restore_column_state('settings')

    def populate_settings_table(self):
        """Populate the settings table with configuration data"""
        settings_data = []

        # Add installation type and libraries info
        is_portable = self._is_portable_install()
        settings_data.append((_('Installation'), _('Type'), _('Portable') if is_portable else _('Standard'), 'global.py.json'))

        # Get all library paths for non-portable installations
        if not is_portable:
            library_paths = self._get_library_paths()
            for i, path in enumerate(library_paths, 1):
                settings_data.append((_('Libraries'), _('Library %d') % i, path, 'gui.json'))

        # Add settings from global.py.json
        try:
            with open(os.path.join(config_dir, 'global.py.json'), 'r') as f:
                global_config = json.load(f)
                settings_data.extend([
                    (_('System'), _('Language'), global_config.get('language', ''), 'global.py.json'),
                    (_('Library Management'), _('Primary Library Path'), global_config.get('library_path', ''), 'global.py.json'),
                    (_('Library Management'), _('Read File Metadata'), str(global_config.get('read_file_metadata', True)), 'global.py.json'),
                    (_('Library Management'), _('Check Duplicates on Copy/Move'), str(global_config.get('check_for_dupes_on_ctl', False)), 'global.py.json'),
                    (_('Library Management'), _('Mark New Books'), str(global_config.get('mark_new_books', False)), 'global.py.json'),
                    (_('Library Management'), _('New Book Tags'), ', '.join(global_config.get('new_book_tags', [])), 'global.py.json'),
                    (_('Library Management'), _('Worker Process Priority'), global_config.get('worker_process_priority', 'normal'), 'global.py.json'),
                    (_('File Formats'), _('Default Output Format'), global_config.get('output_format', ''), 'global.py.json'),
                    # Truncate Input Format Order for export (first 19, then +more if needed)
                    (_('File Formats'), _('Input Format Order'),
                        (lambda v: (
                            (', '.join(v[:19]) + ',+more') if isinstance(v, list) and len(v) > 19
                            else (', '.join(v)) if isinstance(v, list)
                            else v
                        ))(global_config.get('input_format_order', [])),
                        'global.py.json'),
                    (_('File Patterns'), _('Filename Pattern'), global_config.get('filename_pattern', ''), 'global.py.json'),
                    (_('Network'), _('Network Timeout'), str(global_config.get('network_timeout', 5)) + ' ' + _('seconds'), 'global.py.json'),
                    (_('Search'), _('Case Sensitive'), str(global_config.get('case_sensitive', False)), 'global.py.json'),
                    (_('Search'), _('Use Primary Find in Search'), str(global_config.get('use_primary_find_in_search', True)), 'global.py.json'),
                    (_('Search'), _('Limit Search Columns'), str(global_config.get('limit_search_columns', False)), 'global.py.json'),
                    (_('Search'), _('Limited Search Columns'), ', '.join(global_config.get('limit_search_columns_to', [])), 'global.py.json'),
                    (_('Search'), _('Numeric Collation'), str(global_config.get('numeric_collation', False)), 'global.py.json'),
                    (_('Author Handling'), _('Swap Author Names'), str(global_config.get('swap_author_names', False)), 'global.py.json'),
                    (_('Categories'), _('User Categories'), str(len(global_config.get('user_categories', {}))), 'global.py.json'),
                    (_('Device'), _('Manage Device Metadata'), global_config.get('manage_device_metadata', 'manual'), 'global.py.json')
                ])
        except Exception as e:
            pass

        # Add TTS settings
        try:
            with open(os.path.join(config_dir, 'tts.json'), 'r') as f:
                tts_config = json.load(f)
                if 'engines' in tts_config:
                    if 'piper' in tts_config['engines']:
                        voice = tts_config['engines']['piper'].get('voice', '')
                        settings_data.append((_('TTS'), _('Default Voice (Piper)'), voice, 'tts.json'))
                    if 'defaults' in tts_config:
                        settings_data.append((_('TTS'), _('Key Sequences'), str(tts_config['defaults']), 'tts.json'))
        except Exception as e:
            pass

        # Add interface settings from gui.json and gui.py.json
        try:
            with open(os.path.join(config_dir, 'gui.json'), 'r') as f:
                gui_config = json.load(f)
        except Exception:
            gui_config = {}

        try:
            with open(os.path.join(config_dir, 'gui.py.json'), 'r') as f:
                gui_py_config = json.load(f)
        except Exception:
            gui_py_config = {}

            # Add the auto-convert setting
            auto_convert = gui_config.get('manual_add_auto_convert', False)
            settings_data.append(
                (_('Adding Books'), _('Auto Convert Added Books'),
                str(auto_convert), 'gui.json')
            )

            # Get the actual GUI layout from the main window state
            widget_state = gui_config.get('main_window_central_widget_state', {})
            actual_layout = widget_state.get('layout', gui_config.get('gui_layout', 'wide'))

            # Add window geometry information
            geom = gui_config.get('geometry-of-calibre_main_window_geometry', {})
            if (geom):
                settings_data.extend([
                    (_('Window Geometry'), _('calibre_main_window_geometry - Position'), f"X: {geom.get('geometry', {}).get('x')}, Y: {geom.get('geometry', {}).get('y')}", 'gui.json'),
                    (_('Window Geometry'), _('calibre_main_window_geometry - Size'), f"Width: {geom.get('geometry', {}).get('width')}, Height: {geom.get('geometry', {}).get('height')}", 'gui.json'),
                    (_('Window Geometry'), _('calibre_main_window_geometry - Screen'), f"Resolution: {geom.get('screen', {}).get('geometry_in_logical_pixels', {}).get('width')}x{geom.get('screen', {}).get('geometry_in_logical_pixels', {}).get('height')}", 'gui.json'),
                    (_('Window Geometry'), _('calibre_main_window_geometry - Monitor'), f"{geom.get('screen', {}).get('manufacturer', '')}", 'gui.json'),
                ])

            settings_data.extend([
                (_('Interface'), _('GUI Layout'), actual_layout, 'gui.json'),
                (_('Interface'), _('Color Theme'), gui_config.get('color_palette', 'system'), 'gui.json'),
                (_('Interface'), _('Dark Theme'), gui_config.get('dark_palette_name', ''), 'gui.json'),
                (_('Interface'), _('Light Theme'), gui_config.get('light_palette_name', ''), 'gui.json'),
                (_('Interface'), _('Grid View'), str(gui_py_config.get('grid_view_visible', False)), 'gui.py.json'),
                (_('Interface'), _('Cover Flow Queue Length'), str(gui_config.get('cover_flow_queue_length', 6)), 'gui.json'),
                (_('Interface'), _('Separate Cover Flow'), str(gui_py_config.get('separate_cover_flow', False)), 'gui.py.json'),
                (_('Interface'), _('Sort Tags By'), gui_config.get('sort_tags_by', 'name'), 'gui.json'),
                (_('Interface'), _('Match Tags Type'), gui_config.get('match_tags_type', 'any'), 'gui.json'), #don't add extra parentheses after 'any'
                (_('Interface'), _('Main Search History'), ', '.join(gui_config.get('main_search_history', [])), 'gui.json'),
                (_('Interface'), _('Search Bar Visible'), str(gui_config.get('search bar visible', True)), 'gui.json'),
                (_('Interface'), _('Tag Browser Search Box'), str(gui_config.get('tag browser search box visible', False)), 'gui.json'),
                (_('Performance'), _('Search As You Type'), str(gui_py_config.get('search_as_you_type', False)), 'gui.py.json'),
                (_('Performance'), _('Disable Animations'), str(gui_py_config.get('disable_animations', False)), 'gui.py.json'),
                (_('Performance'), _('Enforce CPU Limit'), str(gui_py_config.get('enforce_cpu_limit', True)), 'gui.py.json'),
                (_('Performance'), _('Worker Limit'), str(gui_config.get('worker_limit', 6)), 'gui.json'),
                (_('System Tray'), _('Show System Tray Icon'), str(gui_py_config.get('systray_icon', False)), 'gui.py.json'),
                (_('System Tray'), _('Disable Tray Notification'), str(gui_py_config.get('disable_tray_notification', False)), 'gui.py.json'),
                (_('System Tray'), _('Autolaunch Server'), str(gui_py_config.get('autolaunch_server', False)), 'gui.py.json'),
                (_('Tables'), _('Columns'), ', '.join(gui_config.get('column_map', [])), 'gui.json'),
                (_('Updates'), _('New Version Notification'), str(gui_config.get('new_version_notification', False)), 'gui.json'),
                (_('Updates'), _('Auto Download Cover'), str(gui_py_config.get('auto_download_cover', False)), 'gui.py.json'),
                (_('Network'), _('Oldest News'), str(gui_config.get('oldest_news', 60)), 'gui.json')
            ])
        except Exception as e:
            pass

        # Add tool buttons configuration
        try:
            toolbar_actions = gui_config.get('action-layout-toolbar', [])
            if toolbar_actions:
                settings_data.append((_('Toolbar'), _('Main Toolbar Actions'), ', '.join(filter(None, toolbar_actions)), 'gui.json'))

            child_actions = gui_config.get('action-layout-toolbar-child', [])
            if child_actions:
                settings_data.append((_('Toolbar'), _('Child Toolbar Actions'), ', '.join(filter(None, child_actions)), 'gui.json'))

            device_actions = gui_config.get('action-layout-toolbar-device', [])
            if device_actions:
                settings_data.append((_('Toolbar'), _('Device Toolbar Actions'), ', '.join(filter(None, device_actions)), 'gui.json'))
        except Exception as e:
            pass

        # Add device settings if available
        device_file = os.path.join(config_dir, 'device_drivers_SMART_DEVICE_APP.py.json')
        if os.path.exists(device_file):
            try:
                with open(device_file, 'r') as f:
                    device_config = json.load(f)
                    settings_data.extend([
                        (_('Device'), _('Save Template'), device_config.get('save_template', ''), 'device_drivers_SMART_DEVICE_APP.py.json'),
                        (_('Device'), _('Read Metadata'), str(device_config.get('read_metadata', True)), 'device_drivers_SMART_DEVICE_APP.py.json'),
                        (_('Device'), _('Use Author Sort'), str(device_config.get('use_author_sort', False)), 'device_drivers_SMART_DEVICE_APP.py.json'),
                        (_('Device'), _('Use Subdirectories'), str(device_config.get('use_subdirs', True)), 'device_drivers_SMART_DEVICE_APP.py.json')
                    ])

                    # Add format priorities
                    formats = device_config.get('format_map', [])
                    if formats:
                        settings_data.append((_('Device'), _('Format Priorities'), ', '.join(formats), 'device_drivers_SMART_DEVICE_APP.py.json'))
            except Exception as e:
                pass

        # Add tweaks
        try:
            with open(os.path.join(config_dir, 'tweaks.json'), 'r') as f:
                tweaks_config = json.load(f)
                # Organize tweaks by category
                for key, value in tweaks_config.items():
                    if 'author' in key.lower():
                        category = _('Author Tweaks')
                    elif 'cover' in key.lower():
                        category = _('Cover Tweaks')
                    elif 'metadata' in key.lower():
                        category = _('Metadata Tweaks')

                        # Port settings if available
                        if len(extra_customization) > 5:
                            settings_data.append((_('Device Connection'), _('Port Number'), str(extra_customization[5]), 'device_drivers_SMART_DEVICE_APP.py.json'))
        except Exception as e:
            if DEBUG:
                pass

            self.settings_table.setItem(row, 2, QTableWidgetItem(value))
            self.settings_table.setItem(row, 3, QTableWidgetItem(source))

            # Add tooltip for settings
            description = self._get_setting_description(setting)
            if description:
                self.settings_table.item(row, 1).setToolTip(description)

        # Adjust column widths
        self.settings_table.resizeColumnsToContents()

    def show_system_info_context_menu(self, pos):
        """Show context menu for system info area"""
        menu = QMenu(self)

        copy_action = menu.addAction(_('Copy System Information'))
        copy_action.setIcon(self.copy_icon)
        copy_action.triggered.connect(self.copy_system_info_to_clipboard)

        menu.popup(self.mapToGlobal(pos))

    def copy_system_info_to_clipboard(self):
        """Copy system information to clipboard as text"""
        try:
            import platform
            from calibre.constants import __version__, config_dir, isportable
            from . import PluginCatalogPlugin

            # Build system info text
            info_lines = []

            # Calibre version
            info_lines.append(f"Calibre: {__version__}")

            # Operating system info
            info_lines.append(f"Operating System: {platform.platform()}")

            # Installation type
            parent_folder = os.path.dirname(config_dir)
            if isportable:
                info_lines.append(f"Installation Type: Portable Mode @ {parent_folder}")
            else:
                info_lines.append("Installation Type: Standard Install")

            # CCR version
            ccr_version = ".".join(str(x) for x in PluginCatalogPlugin.version)
            info_lines.append(f"CCR Plugin Version: {ccr_version}")

            # Statistics (if available)
            plugin_text = self.plugin_stats.text() if hasattr(self, 'plugin_stats') else ""
            if plugin_text:
                info_lines.append(f"Statistics: {plugin_text}")

            column_text = self.column_stats.text() if hasattr(self, 'column_stats') else ""
            if column_text:
                info_lines.append(f"Custom Columns: {column_text}")

            shortcut_text = self.shortcut_stats.text() if hasattr(self, 'shortcut_stats') else ""
            if shortcut_text:
                info_lines.append(f"Shortcuts: {shortcut_text}")

            # Join all lines
            system_info = "\n".join(info_lines)

            # Copy to clipboard
            from PyQt5.QtWidgets import QApplication
            clipboard = QApplication.clipboard()
            clipboard.setText(system_info)

            # Show temporary feedback
            from calibre.gui2 import info_dialog
            info_dialog(self, _('System Information Copied'),
                       _('System information has been copied to clipboard.'),
                       show=True, show_copy_button=False)

        except Exception as e:
            from calibre.gui2 import error_dialog
            error_dialog(self, _('Copy Failed'),
                        _('Failed to copy system information: %s') % str(e),
                        show=True)

    def copy_settings_row(self, row):
        """Copy all data for a settings row to clipboard"""
        data = []
        for col in range(self.settings_table.columnCount()):
            header = self.settings_table.horizontalHeaderItem(col).text()
            cell_text = self.settings_table.item(row, col).text()
            data.append(f"{header}: {cell_text}")

        QApplication.clipboard().setText('\n'.join(data))
        self.show_copy_feedback(row, 0, self.settings_table)

    def show_settings_context_menu(self, pos):
        """Show context menu for settings table"""
        index = self.settings_table.indexAt(pos)
        menu = QMenu(self)

        if index.isValid():
            # Copy row data action
            copy_action = menu.addAction(self.copy_icon, _('Copy row data'))
            copy_action.triggered.connect(lambda: self.copy_settings_row(index.row()))

            # Add separator
            menu.addSeparator()

            # Add column visibility submenu
            columns_menu = menu.addMenu(_('Show/Hide Columns'))
            for i in range(self.settings_table.columnCount()):
                col_name = self.settings_table.horizontalHeaderItem(i).text()
                action = columns_menu.addAction(col_name)
                action.setCheckable(True)
                action.setChecked(not self.settings_table.isColumnHidden(i))
                action.triggered.connect(lambda checked, col=i:
                    self.toggle_column_to_state(self.settings_table, col, checked))

        menu.popup(self.settings_table.viewport().mapToGlobal(pos))

    def copy_history_row(self, row):
        """Copy all selected history rows to clipboard, one per line."""
        table = self.history_table if hasattr(self, 'history_table') else None
        if not table:
            return
        selected_rows = set()
        # Support both selection and single row
        for item in table.selectedItems():
            selected_rows.add(item.row())
        if not selected_rows and row >= 0 and row < table.rowCount():
            selected_rows.add(row)
        if not selected_rows:
            return  # No valid rows to copy
        lines = []
        for r in sorted(selected_rows):
            line = []
            for c in range(table.columnCount()):
                item = table.item(r, c)
                line.append(item.text() if item else '')
            lines.append('\t'.join(line))
        QApplication.clipboard().setText('\n'.join(lines))
        # Optionally show feedback (e.g., highlight copied rows)

    def show_history_context_menu(self, pos):
        """Show context menu for history table"""
        index = self.history_table.indexAt(pos)
        menu = QMenu(self)

        # Check if there are selected rows
        selected_rows = set()
        for item in self.history_table.selectedItems():
            selected_rows.add(item.row())

        if selected_rows or index.isValid():
            # Copy row data action
            copy_action = menu.addAction(self.copy_icon, _('Copy row data'))
            copy_action.triggered.connect(lambda: self.copy_history_row(index.row()))

        # Add separator and utility actions
        menu.addSeparator()

        # Add action to clear initialization entries
        clear_init_action = menu.addAction(_('Clear initialization entries'))

        clear_init_action.setToolTip(_('Remove history entries from plugin startup/initialization'))
        clear_init_action.triggered.connect(self.clear_initialization_history)

        menu.popup(self.history_table.viewport().mapToGlobal(pos))

    def clear_initialization_history(self):
        """Clear history entries that appear to be from plugin initialization"""
        from calibre.gui2 import question_dialog, info_dialog
        # Ensure we don't shadow module-level imports; use them consistently
        import os, json

        if not question_dialog(self, _('Clear initialization entries'),
                             _('This will remove history entries that appear to be from plugin startup/initialization. Continue?')):
            return

        try:
            # Try to fetch the official plugin index to help identify plugins if online.
            # If the fetch fails (offline/DNS error), fall back to a local-only heuristic and continue.
            from calibre.utils.https import get_https_resource_securely
            import bz2, socket

            SERVER = 'https://code.calibre-ebook.com/plugins/'
            INDEX_URL = f'{SERVER}plugins.json.bz2'

            raw = None
            try:
                raw = get_https_resource_securely(INDEX_URL)
            except Exception:
                # Network failure - proceed with local-only heuristic below
                raw = None

            installed_plugin_ids = set()
            if raw:
                try:
                    data = json.loads(bz2.decompress(raw).decode('utf-8'))
                    installed_plugin_ids = set(data.keys())
                except Exception:
                    installed_plugin_ids = set()

            from calibre.utils.config import config_dir

            # Load existing history
            history_file = os.path.join(config_dir, 'plugins', 'calibre_config_reports', 'ccr_history.json')
            if os.path.exists(history_file):
                with open(history_file, 'r', encoding='utf-8') as f:
                    history = json.load(f)

                # Filter out entries that look like initialization
                filtered_history = []
                removed_count = 0

                for entry in history:
                    try:
                        # Primary heuristic: setting_changed where old_value is None/"None"/"null"
                        if (entry.get('event_type') == 'setting_changed' and
                            entry.get('details', {}).get('old_value') in ['None', None, 'null']):
                            removed_count += 1
                            continue
                        # Add more heuristics here if desired (e.g., timestamp ranges, known plugin IDs)
                        filtered_history.append(entry)
                    except Exception:
                        # If any entry is malformed, skip deleting it to be safe
                        filtered_history.append(entry)

                # Save filtered history back
                try:
                    with open(history_file, 'w', encoding='utf-8') as f:
                        json.dump(filtered_history, f, indent=2)
                except Exception:
                    # If writing fails, surface an error to the user
                    from calibre.gui2 import error_dialog
                    error_dialog(self, _('Error clearing history'),
                                 _('Failed to write cleaned history file.'), show=True)
                    return

                # Refresh the history table
                self.populate_history_table()

                # Inform user about result; if we couldn't contact the server, mention fallback
                if raw is None:
                    info_dialog(self, _('Cleanup complete (offline)'),
                               _('Could not reach code.calibre-ebook.com; performed local-only cleanup and removed %d initialization entries from history.') % removed_count,
                               show=True)
                else:
                    info_dialog(self, _('Cleanup complete'),
                               _('Removed %d initialization entries from history.') % removed_count,
                               show=True)

        except Exception as e:
            from calibre.gui2 import error_dialog
            error_dialog(self, _('Error clearing history'),
                        _('Failed to clear initialization entries: %s') % str(e),
                        show=True)

    def _get_current_tab_name(self):
        """Get the name of the currently selected tab"""
        current_index = self.tabs.currentIndex()
        return self.tabs.tabText(current_index).replace('&', '')

    def _get_export_timestamp(self):
        """Get a standardized timestamp for export filenames"""
        import datetime
        return datetime.datetime.now().strftime('_%Y-%m-%d_%H-%M-%S')

    def _get_export_filename(self, format_type):
        """Generate export filename with tab name and timestamp"""
        tab_name = self._get_current_tab_name().lower()
        timestamp = self._get_export_timestamp()
        return f'calibre_{tab_name}{timestamp}.{format_type}'

    def closeEvent(self, e):
        # Stop the audio player when closing the dialog
        if hasattr(self, 'media_player'):
            self.media_player.stop()

        # Save dialog geometry
        gprefs['calibre_config_reports_dialog_geometry'] = bytearray(self.saveGeometry())
        # Save column widths for all tables
        self.save_column_widths('plugins')
        self.save_column_widths('columns')
        self.save_column_widths('shortcuts')
        self.save_column_widths('settings')
        # Save column order and visibility for shortcuts tab
        self.save_column_state('shortcuts')
        # Save current tab
        gprefs['calibre_config_reports_last_tab'] = self.tabs.currentIndex()
        super().closeEvent(e)




    def setup_columns_table(self):
        """Setup the custom columns table"""
        self.columns_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.columns_table.setAlternatingRowColors(True)
        self.columns_table.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)
        self.columns_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

        # Set up columns
        self.columns_table.setColumnCount(8)
        # Add 'Web Search Template' column (Calibre >= 7.26)
        headers = [_('Column header'), _('Lookup name'), _('Type'), _('Description'),
                   _('Is Composite'), _('Template'), _('Web Search Template'), _('Display Settings')]
        self.columns_table.setHorizontalHeaderLabels(headers)

        # Enable sorting
        self.columns_table.setSortingEnabled(True)

        # Enable context menu
        self.columns_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.columns_table.customContextMenuRequested.connect(self.show_columns_context_menu)

        # Enable double-click for info dialog
        self.columns_table.cellDoubleClicked.connect(self.show_column_info_dialog)

        # Set column widths and enable reordering
        header = self.columns_table.horizontalHeader()
        header.setSectionsMovable(True)
        header.sectionMoved.connect(lambda: self.save_column_state('columns'))

        # Use a safer callback that doesn't rely on self.columns_table access
        def on_columns_section_resized():
            try:
                self.save_column_state('columns')
            except AttributeError:
                # If columns_table is not accessible, try to save directly using the table
                if hasattr(self, 'columns_table'):
                    self.save_column_state('columns')
        header.sectionResized.connect(on_columns_section_resized)

    def setup_shortcuts_table(self):
        """Setup the shortcuts table"""
        self.shortcuts_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.shortcuts_table.setAlternatingRowColors(True)
        self.shortcuts_table.setSelectionMode(QAbstractItemView.SelectionMode.SingleSelection)
        self.shortcuts_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

        # Set up columns
        # Columns: Action | Group | Default | Assigned
        self.shortcuts_table.setColumnCount(4)
        headers = [_('Action'), _('Group'), _('Default'), _('Assigned')]
        self.shortcuts_table.setHorizontalHeaderLabels(headers)

        # Enable sorting and context menu
        self.shortcuts_table.setSortingEnabled(True)
        self.shortcuts_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.shortcuts_table.customContextMenuRequested.connect(self.show_shortcuts_context_menu)

        # Prefer a single activation path (handled via itemActivated in __init__) to avoid double dialogs

        # Enable column reordering
        header = self.shortcuts_table.horizontalHeader()
        header.setSectionsMovable(True)
        # Accept any signal args to avoid signature mismatch errors
        header.sectionMoved.connect(lambda *a: self.save_column_state('shortcuts'))

        # Initialize column width constraints
        # Do NOT cap all columns globally; only enforce a max for the Action column via a resize handler
        header.setMinimumSectionSize(20)   # Minimum width for all columns

        # Connect to section resized signal to enforce Action column width constraint
        def enforce_shortcuts_action_width(logical_index, old_size, new_size):
            if logical_index == 0 and new_size > 450:  # Action column (index 0)
                self.shortcuts_table.setColumnWidth(0, 450)

        header.sectionResized.connect(enforce_shortcuts_action_width)
        # Accept any signal args to avoid signature mismatch errors
        header.sectionResized.connect(lambda *a: self.save_column_state('shortcuts'))

    def populate_tables(self):
        """Populate both tables"""
        self.populate_columns_table()
        self.populate_shortcuts_tab()
        # Don't call populate_settings_table() here - it's now handled by SettingsTab

        # Restore saved column states (includes visibility, order, and widths)
        self.restore_column_state('plugins')
        self.restore_column_state('columns')
        self.restore_column_state('shortcuts')
        self.restore_column_state('settings')

        # Update statistics
        self.update_statistics()

    def _is_portable_install(self):
        """Check if this is a portable installation"""
        from calibre.constants import isportable
        return isportable

    def _get_library_paths(self):
        """Get all configured library paths"""
        from calibre.gui2 import gprefs
        library_usage = gprefs.get('library_usage_stats', {})
        return sorted(library_usage.keys())

    def _get_human_readable_size(self, size_in_bytes):
        """Convert size in bytes to human readable format"""
        if not size_in_bytes:
            return '0 B'
        for unit in ['B', 'KB', 'MB', 'GB']:
            if size_in_bytes < 1024.0:
                if unit == 'B':
                    return f"{int(size_in_bytes)} {unit}"
                return f"{size_in_bytes:.1f} {unit}"
            size_in_bytes /= 1024.0
        return f"{size_in_bytes:.1f} GB"

    def _get_setting_description(self, setting_name):
        """Get the description/tooltip for a setting"""
        descriptions = {
            'Default Output Format': _('The default format to convert e-books to'),
            'Input Format Order': _('Order of formats to prefer when getting formats from input books'),
            'Filename Pattern': _('Pattern to match filenames when adding books from the filesystem'),
            'Case Sensitive': _('Make searching case sensitive when searching your library'),
            'Limit Search Columns': _('Restrict searching to specific columns to improve search performance'),
            'Limited Search Columns': _('List of columns to restrict searching to when Limit Search Columns is enabled'),
            'Read File Metadata': _('Read metadata from e-book files when adding books'),
            'Mark New Books': _('Mark newly added books until calibre is restarted'),
            'New Book Tags': _('Tags to automatically apply to newly added books'),
            'Numeric Collation': _('Sort numbers based on numerical value instead of as text'),
            'Network Timeout': _('Timeout in seconds for network operations'),
            'Worker Process Priority': _('Priority level for background processes'),
            'Manage Device Metadata': _('How to handle metadata when sending books to devices'),
            'Swap Author Names': _('Swap author first and last names when reading metadata'),
            'User Categories': _('User-created categories for organizing books'),
            'GUI Layout': _('Layout of the main calibre window (wide/narrow)'),
            'Show System Tray Icon': _('Show calibre icon in system tray/notification area'),
            'Disable Tray Notifications': _('Turn off popup notifications in system tray'),
            'Highlight Search Matches': _('Highlight all search matches in book list'),
            'Separate Cover Flow': _('Show cover flow in separate window'),
            'Worker Limit': _('Maximum number of simultaneous conversion/download jobs'),
            'Database Path': _('Location of the calibre library database file'),
            'Auto Download Cover': _('Automatically download covers from internet'),
            'Get Social Metadata': _('Download social metadata like tags and ratings'),
            'Search As You Type': _('Update search results while typing'),
            'Enforce CPU Limit': _('Limit CPU cores used for background jobs'),
            'Grid View Visible': _('Show books in a grid of covers'),
            'Sort Tags By': _('How to sort tags in Tag browser'),
            'Format Priorities': _('Order of format preference for device'),
            'Save Template': _('Template for filenames when saving to device'),
            'Use Author Sort': _('Use author sort value on device'),
            'Use Subdirectories': _('Create subdirectories on device'),
            'Quick View Open at Shutdown': _('Keep Quick View window open on restart'),
            'Color Theme': _('Light/Dark/System theme setting'),
            'Cover Flow Queue Length': _('Number of covers to show in flow'),
            'Tag Browser Hidden Categories': _('Categories hidden from Tag browser'),
            'Main Search History': _('Recently used search terms'),
            'Plugin Search History': _('Recent plugin searches'),
            'Oldest News': _('Days to keep downloaded news'),
            'Toolbar Actions': _('Buttons shown on toolbar'),
            'Auto Convert Added Books': _('When enabled, automatically converts added books to the preferred output format configured in Calibre Settings')
        }
        return descriptions.get(setting_name, '')

    def populate_columns_table(self):
        """Populate the custom columns table with data"""
        db = self.gui.current_db.new_api
        self.columns_table.setRowCount(0)

        # Get custom columns using the correct API
        custom_columns = {}
        for field, meta in db.field_metadata.custom_iteritems():
            display = meta.get('display', {})
            custom_columns[meta['label']] = {

                'name': meta.get('name', meta['label']),
                'label': meta['label'],
                'datatype': meta.get('datatype', 'text'),
                'display': display,
                'is_multiple': meta.get('is_multiple', False),
                'description': meta.get('description', '')  # Description may be at the root level
            }

            # If description is not at root level, try to get it from display dict
            if not custom_columns[meta['label']]['description'] and 'description' in display:
                custom_columns[meta['label']]['description'] = display['description']

        # Sort custom columns by label (human-readable name)
        sorted_columns = sorted(custom_columns.values(), key=lambda col: col['label'].lower())

        # Populate the table with sorted custom columns
        self.columns_table.setRowCount(len(sorted_columns))
        from calibre.constants import numeric_version
        for row, column in enumerate(sorted_columns):
            name = column['name']
            label = column['label']
            datatype = column['datatype']
            display = column['display']
            is_composite = column.get('is_multiple', False)
            description = column.get('description', '')

            # Name
            name_item = QTableWidgetItem(name)
            name_item.setData(Qt.ItemDataRole.UserRole, column)
            self.columns_table.setItem(row, 0, name_item)

            # Label
            label_item = QTableWidgetItem(label)
            self.columns_table.setItem(row, 1, label_item)

            # Type
            type_item = QTableWidgetItem(datatype)
            self.columns_table.setItem(row, 2, type_item)

            # Description
            desc_item = QTableWidgetItem(description)
            if description:
                desc_item.setToolTip(description)
            self.columns_table.setItem(row, 3, desc_item)

            # Is Composite
            is_composite_str = _('Yes') if datatype == 'composite' else _('No')
            composite_item = QTableWidgetItem(is_composite_str)
            self.columns_table.setItem(row, 4, composite_item)

            # Template
            template = display.get('composite_template', '')
            template_item = QTableWidgetItem(template)
            if template:
                template_item.setToolTip(template)
            self.columns_table.setItem(row, 5, template_item)

            # Web Search Template (Calibre >= 7.26)
            web_search_template = ''
            if numeric_version >= (7, 26, 0):
                web_search_template = display.get('web_search_template', '')
            web_search_item = QTableWidgetItem(web_search_template)
            if web_search_template:
                web_search_item.setToolTip(web_search_template)
            self.columns_table.setItem(row, 6, web_search_item)

            # Display Settings
            display_dict = display.copy()
            if 'composite_template' in display_dict:
                del display_dict['composite_template']
            if 'web_search_template' in display_dict:
                del display_dict['web_search_template']
            display_settings = self.get_display_settings_text(display_dict)

            settings_item = QTableWidgetItem(display_settings)
            settings_item.setData(Qt.ItemDataRole.UserRole, display_settings)
            settings_item.setToolTip(display_settings)
            self.columns_table.setItem(row, 7, settings_item)

        # Adjust column widths
        try:
            # Only auto-size on first run; keep user widths if already saved
            widths = gprefs.get('calibre_config_reports_columns_column_widths', None)
            if widths:
                # Support either dict (string keys) or list/tuple of widths
                if isinstance(widths, dict):
                    for k, v in widths.items():
                        try:
                            idx = int(k)
                            w = int(v) if v is not None else 0
                            if 0 <= idx < self.columns_table.columnCount() and w > 0:
                                self.columns_table.setColumnWidth(idx, min(w, 1000))
                        except Exception:
                            continue
                elif isinstance(widths, (list, tuple)):
                    for idx, val in enumerate(widths):
                        try:
                            w = int(val) if val is not None else 0
                            if idx < self.columns_table.columnCount() and w > 0:
                                self.columns_table.setColumnWidth(idx, min(w, 1000))
                        except Exception:
                            continue
            else:
                # Apply sensible, readable defaults for first-run (do not persist here)
                default_widths = [120, 100, 80, 200, 80, 180, 150, 200]  # Added 150px for Web Search Template
                for idx, w in enumerate(default_widths):
                    if idx < self.columns_table.columnCount():
                        try:
                            self.columns_table.setColumnWidth(idx, w)
                        except Exception:
                            pass
        except Exception:
            # Best-effort only
            pass

    def get_display_settings_text(self, display_settings):
        """Format display settings as a complete JSON string"""
        if not display_settings:
            return ''
        try:
            return str(display_settings)
        except Exception:
            return ''

    def get_column_data(self):
        """Get column data in exportable format"""
        data = []
        headers = ['Name', 'Label', 'Type', 'Description', 'Is Composite', 'Template', 'Display Settings']

        for col in self.db.custom_column_label_map:
            col_data = self.db.custom_column_label_map[col]
            is_composite = 'Yes' if col_data['datatype'] == 'composite' else 'No'
            template = col_data.get('display', {}).get('composite_template', '')
            display_settings = self.get_display_settings_text(col_data.get('display', {}))

            row = [
                col,
                col_data['name'],
                col_data['datatype'],
                col_data['display'].get('description', ''),
                is_composite,
                template,
                display_settings
            ]
            data.append(row)

        return headers, data

    def populate_shortcuts_tab(self):
        """Populate the shortcuts table with current keyboard mappings"""
        shortcuts_list = []
        keyboard = self.gui.keyboard

        if not keyboard:
            return

        # Get all plugins and their installation types for proper shortcut classification
        from calibre.customize.ui import initialized_plugins, PluginInstallationType
        plugin_info = {}
        for plugin in initialized_plugins():
            plugin_info[plugin.name] = plugin.installation_type

        # Get interface actions to determine plugin types (keep for backward compatibility)
        interface_actions = {}
        try:
            from calibre.customize.ui import interface_actions as get_interface_actions
            for plugin in get_interface_actions():
                # Store plugin installation type for later lookup
                interface_actions[plugin.name] = plugin.installation_type
        except Exception:
            pass

        # Prefer the in-memory keyboard config map (keyboard.config['map']) when
        # available. This is authoritative for Calibre's current 'main'
        # shortcuts and avoids scanning many on-disk files. If the in-memory
        # map is not present or empty, fall back to reading
        # <config_dir>/shortcuts/main.json (conservative on-disk fallback).
        user_defined_keys = set()
        try:
            # In-memory map (preferred)
            try:
                inmem_map = keyboard.config.get('map', {})
                if isinstance(inmem_map, dict) and inmem_map:
                    user_defined_keys.update(inmem_map.keys())
            except Exception:
                inmem_map = {}

            # Only fall back to disk if in-memory map is empty
            if not user_defined_keys:
                cfg_dir = config_dir
                main_path = os.path.join(cfg_dir, 'shortcuts', 'main.json')
                # Also accept main.py.json as some installs use that
                if not os.path.exists(main_path):
                    alt = os.path.join(cfg_dir, 'shortcuts', 'main.py.json')
                    if os.path.exists(alt):
                        main_path = alt

                if os.path.exists(main_path):
                    try:
                        with open(main_path, 'r', encoding='utf-8') as f:
                            data = json.load(f)
                        if isinstance(data, dict) and 'map' in data and isinstance(data['map'], dict):
                            user_defined_keys.update(data['map'].keys())
                    except Exception:
                        pass
        except Exception:
            user_defined_keys = set()
        else:
            # Helpful debug info when CCR_DEBUG_ENABLED is on so we can see what was read
            try:
                if CCR_DEBUG_ENABLED:
                    prints(f"[CCR][DEBUG] Found {len(user_defined_keys)} user-defined shortcut keys (scanned {cfg_dir})")
            except Exception:
                pass

        # Get all shortcuts and their proper groups
        disk_flag_count = 0
        runtime_flag_count = 0
        # Build a lightweight snapshot of current runtime mapping so we can avoid
        # rebuilding the entire table if nothing changed since last population.
        try:
            curr_snapshot = {}
        except Exception:
            curr_snapshot = None
        for unique_name, shortcut in keyboard.shortcuts.items():
            if shortcut.get('action') is None:
                continue
            try:
                # Get the name directly from the shortcut data
                name = shortcut.get('name', '')

                # Determine the proper group. Prefer the authoritative mapping
                # in keyboard.groups (what Calibre's preferences UI uses).
                group = None
                try:
                    for g, names in keyboard.groups.items():
                        if unique_name in names:
                            group = g
                            break
                except Exception:
                    group = None

                # Fallbacks when the mapping doesn't contain the shortcut
                if not group or (isinstance(group, str) and not group.strip()):
                    # Try to extract group from unique_name for Interface Actions
                    if _('Interface Action:') in unique_name:
                        try:
                            # Format is typically "Interface Action: Group Name (Plugin Name) - action"
                            group = unique_name.split(_('Interface Action:'))[1].split('(')[0].strip()
                        except Exception:
                            group = _('Miscellaneous')
                    else:
                        # For plugin actions, try to extract from the first part of unique name
                        parts = unique_name.split(':')
                        if len(parts) > 1:
                            group = parts[0].strip()
                        else:
                            # Use the same default Calibre uses for missing groups
                            from calibre.utils.localization import pgettext
                            group = pgettext('keyboard shortcuts', _('Miscellaneous'))

                # Get the default key sequence(s)
                default_keys = shortcut.get('default_keys', ()) or ()
                if default_keys:
                    try:
                        default_text = ', '.join(str(QKeySequence(k, QKeySequence.SequenceFormat.PortableText).toString(QKeySequence.SequenceFormat.NativeText)) for k in default_keys)
                    except Exception:
                        # Some plugins supply non-QKeySequence-compatible values; fall back
                        # to joining the raw values. Guard against empty strings there too.
                        default_text = ', '.join(filter(None, (str(x) for x in default_keys)))
                else:
                    default_text = _('None')

                # Defensive: ensure default_text is never empty or just whitespace.
                try:
                    if not default_text or str(default_text).strip() == '':
                        default_text = _('None')
                except Exception:
                    default_text = _('None')

                # Get the current assigned key sequence(s)
                keys = keyboard.keys_map.get(unique_name, ())
                if keys:
                    assigned_text = ', '.join(str(k.toString(QKeySequence.SequenceFormat.NativeText)) for k in keys)
                else:
                    assigned_text = _('None')

                # Ensure we never have empty shortcut text
                if not assigned_text or assigned_text.strip() == '':
                    assigned_text = _('None')

                # Determine if this is from a user plugin or built-in
                is_user_plugin = False

                # Check if this shortcut belongs to any plugin by examining the unique_name
                # Find the most specific (longest) plugin name that matches
                matching_plugins = []
                for plugin_name, installation_type in plugin_info.items():
                    if plugin_name in unique_name:
                        matching_plugins.append((plugin_name, installation_type))

                # Sort by plugin name length (descending) to prefer more specific matches
                matching_plugins.sort(key=lambda x: len(x[0]), reverse=True)

                if matching_plugins:
                    # Use the most specific (longest) plugin name match
                    plugin_name, installation_type = matching_plugins[0]
                    if installation_type == PluginInstallationType.EXTERNAL:
                        is_user_plugin = True
                # Fallback: Check Interface Actions for backward compatibility
                if not is_user_plugin and _('Interface Action:') in unique_name:
                    try:
                        # Extract plugin name from unique_name
                        plugin_part = unique_name.split('(')[1].split(')')[0]
                        # Check if this plugin is user-installed (EXTERNAL)
                        plugin_installation_type = interface_actions.get(plugin_part)
                        if plugin_installation_type == 1:  # PluginInstallationType.EXTERNAL
                            is_user_plugin = True
                    except Exception:
                        pass

                # Determine if this unique_name was explicitly defined by the user
                # First check on-disk user-defined keys discovered earlier
                disk_flag = unique_name in user_defined_keys
                if disk_flag:
                    disk_flag_count += 1

                # Runtime detection: some shortcuts are changed at runtime but not
                # persisted to disk (viewer actions, ephemeral UI actions). The
                # Calibre keyboard Manager.finalize() sets shortcut['set_to_default']
                # to False when the current assignment differs from the default.
                # Use that as a heuristic to mark runtime user modifications.
                runtime_flag = False
                try:
                    # If the shortcut dict contains set_to_default and it's False,
                    # treat it as a user-modified assignment at runtime.
                    if shortcut.get('set_to_default', True) is False:
                        runtime_flag = True
                        runtime_flag_count += 1
                except Exception:
                    runtime_flag = False

                is_user_defined = disk_flag or runtime_flag
                shortcuts_list.append((name, group, default_text, assigned_text, unique_name, is_user_plugin, is_user_defined))

                # Add to snapshot: assigned_text and set_to_default flag (if present)
                try:
                    st_default = bool(shortcut.get('set_to_default', True))
                except Exception:
                    st_default = True
                try:
                    curr_snapshot[unique_name] = (assigned_text, st_default)
                except Exception:
                    pass
            except Exception:
                continue

        # If we have a previous snapshot and current snapshot is identical, skip
        # rebuilding the table to avoid UI lag when switching tabs rapidly.
        try:
            prev = getattr(self, '_shortcuts_snapshot', None)
            if prev is not None and curr_snapshot == prev:
                # Still reapply the active filter and restore column state to keep UI consistent
                try:
                    self.apply_shortcuts_filter(self.shortcuts_filter_edit.text())
                except Exception:
                    pass
                return
            # Save current snapshot for future comparisons
            self._shortcuts_snapshot = curr_snapshot
        except Exception:
            pass

        # Helpful debug summary of how many shortcuts were discovered by each method
        try:
            if CCR_DEBUG_ENABLED:
                try:
                    prints(f"[CCR][DEBUG] user-defined flags: {disk_flag_count} found on-disk, {runtime_flag_count} found via runtime non-defaults (total scanned={len(keyboard.shortcuts)})")
                except Exception:
                    pass
        except Exception:
            pass

        # Sort all shortcuts by group and name
        shortcuts_list.sort(key=lambda x: (x[1].lower(), x[0].lower()))

        # Clear and populate table
        self.shortcuts_table.setRowCount(len(shortcuts_list))
        self.shortcuts_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

        for row, (name, group, default_text, assigned_text, unique_name, is_user_plugin, is_user_defined) in enumerate(shortcuts_list):
            name_item = QTableWidgetItem(name)
            name_item.setData(Qt.ItemDataRole.UserRole, unique_name)
            name_item.setData(Qt.ItemDataRole.UserRole + 1, is_user_plugin)  # Store plugin type info
            # Store whether this assignment originates from a user shortcut file
            name_item.setData(Qt.ItemDataRole.UserRole + 2, is_user_defined)
            self.shortcuts_table.setItem(row, 0, name_item)

            group_item = QTableWidgetItem(group)
            self.shortcuts_table.setItem(row, 1, group_item)

            default_item = QTableWidgetItem(default_text)
            self.shortcuts_table.setItem(row, 2, default_item)

            assigned_item = QTableWidgetItem(assigned_text)
            # Bold only if assigned != default; italic if user-defined
            try:
                f = assigned_item.font()
                changed = False
                if assigned_text != default_text:
                    f.setBold(True)
                    changed = True
                if is_user_defined:
                    f.setItalic(True)
                    changed = True
                if changed:
                    assigned_item.setFont(f)
            except Exception:
                pass
            self.shortcuts_table.setItem(row, 3, assigned_item)

        # Adjust column widths only if no saved widths exist
        try:
            widths = gprefs.get('calibre_config_reports_shortcuts_column_widths', None)
            # Be more thorough in checking if widths exist and are valid
            has_valid_widths = (widths and
                              (isinstance(widths, (list, tuple)) and len(widths) > 0) or
                              (isinstance(widths, dict) and len(widths) > 0) or
                              (isinstance(widths, str) and widths.strip()))

            if not has_valid_widths:
                self.shortcuts_table.resizeColumnsToContents()
                # Make Assigned column a bit wider for better visibility
                assigned_col_width = self.shortcuts_table.columnWidth(3)
                self.shortcuts_table.setColumnWidth(3, int(assigned_col_width * 1.2))
        except Exception:
            pass

    def apply_shortcuts_filter(self, text):
        """Filter shortcuts table based on text input and dropdown selection"""
        use_regex = self.shortcuts_regex_checkbox.isChecked()
        filter_type = self.shortcuts_type_dropdown.currentIndex()
        table = self.shortcuts_table
        import re
        filter_text = text.strip()

        for row in range(table.rowCount()):
            show_row = True

            # Get shortcut data
            assigned_col_index = 3  # Assigned column index after adding Default
            shortcut_item = table.item(row, assigned_col_index)
            group_item = table.item(row, 1)     # Group column
            name_item = table.item(row, 0)      # Name column

            shortcut_text = shortcut_item.text() if shortcut_item else ''
            group_text = group_item.text() if group_item else ''
            name_text = name_item.text() if name_item else ''

            # Get plugin type information
            is_user_plugin = False
            if name_item:
                is_user_plugin = name_item.data(Qt.ItemDataRole.UserRole + 1) or False
            # Get user-defined flag stored during population
            is_user_defined = False
            if name_item:
                is_user_defined = name_item.data(Qt.ItemDataRole.UserRole + 2) or False

            # Apply dropdown filter first
            if filter_type == 1:  # Key Combinations (Ctrl+X, Alt+F, etc.)
                show_row = '+' in shortcut_text and shortcut_text != _('None')
            elif filter_type == 2:  # Single Keys (F1, M, Delete, etc.)
                show_row = '+' not in shortcut_text and shortcut_text != _('None') and shortcut_text.strip() != ''
            elif filter_type == 3:  # Unused Shortcuts
                is_unused = (not shortcut_text or
                            shortcut_text.strip() == '' or
                            shortcut_text.strip() == _('None'))
                show_row = is_unused
            elif filter_type == 4:  # Assigned Shortcuts
                is_unused = (not shortcut_text or
                            shortcut_text.strip() == '' or
                            shortcut_text.strip() == _('None'))
                show_row = not is_unused
            elif filter_type == 5:  # User defined (assignments present in user's shortcut files)
                show_row = is_user_defined
            elif filter_type == 6:  # User Plugins (External plugins installed by the user)
                show_row = is_user_plugin
            elif filter_type == 7:  # Built-in Features (Core Calibre functionality)
                show_row = not is_user_plugin
            # filter_type == 0 (All Shortcuts) shows everything, so no filter needed

            # Apply text filter if present and row still visible
            if show_row and filter_text:
                text_match = False
                search_text = filter_text.lower()  # Always case-insensitive for simplicity

                # Check all columns for match
                for col in range(table.columnCount()):
                    item = table.item(row, col)
                    if item:
                        cell_text = item.text().lower()
                        try:
                            if use_regex:
                                if re.search(search_text, cell_text, re.IGNORECASE):
                                    text_match = True
                                    break
                            else:
                                if search_text in cell_text:
                                    text_match = True
                                    break
                        except Exception:
                            # Invalid regex, fallback to plain search
                            if search_text in cell_text:
                                text_match = True
                                break
                show_row = text_match

            table.setRowHidden(row, not show_row)

        self.update_statistics()    # Plugin functionality moved to plugins_tab.py


    def apply_columns_filter(self, text):
        """Filter the custom columns table"""
        filter_text = self.columns_filter_edit.text().lower()
        for row in range(self.columns_table.rowCount()):
            show = False
            for col in range(self.columns_table.columnCount()):
                text = self.columns_table.item(row, col).text().lower()
                if filter_text in text:
                    show = True
                    break
            self.columns_table.setRowHidden(row, not show)

        # Update statistics
        self.update_statistics()

    def apply_settings_filter(self, text):
        """Filter the settings table based on search text"""
        filter_text = text.lower()
        for row in range(self.settings_table.rowCount()):
            show = False
            for col in range(self.settings_table.columnCount()):
                item = self.settings_table.item(row, col)
                if item and filter_text in item.text().lower():
                    show = True
                    break
            self.settings_table.setRowHidden(row, not show)

        # Update statistics
        self.update_statistics()




    def export_shortcuts(self, format_type):
        """Export keyboard shortcuts data to CSV or XLSX format"""
        # Only export visible (filtered) shortcuts
        visible_rows = []
        for row in range(self.shortcuts_table.rowCount()):
            if not self.shortcuts_table.isRowHidden(row):
                visible_rows.append(row)

        if not visible_rows:
            from calibre.gui2 import error_dialog
            error_dialog(self, _('Export Failed'),
                      _('No shortcuts match the current filter criteria.'),
                      show=True)
            return

        # Get filename with timestamp
        default_name = self._get_export_filename(format_type)

        # Choose save location
        if format_type == 'csv':
            filters = _('CSV files (*.csv)')
            title = _('Export to CSV')
        else:
            filters = _('Excel files (*.xlsx)')
            title = _('Export to Excel')

        from calibre.gui2 import choose_save_file, info_dialog, error_dialog
        save_path = choose_save_file(self, title, filters,
                                  initial_filename=default_name)

        if not save_path:
            return

        try:
            # Collect data from table
            data = []
            headers = []
            visible_columns = []

            # Get headers from visible columns only
            for col in range(self.shortcuts_table.columnCount()):
                if not self.shortcuts_table.isColumnHidden(col):
                    headers.append(self.shortcuts_table.horizontalHeaderItem(col).text())
                    visible_columns.append(col)

            # Get visible rows and columns only
            for row in visible_rows:
                row_data = []
                for col in visible_columns:
                    item = self.shortcuts_table.item(row, col)
                    row_data.append(item.text() if item else '')
                data.append(row_data)

            if format_type == 'csv':
                self.export_to_csv(save_path, headers, data)
            else:
                self.export_to_xlsx(save_path, headers, data)

            info_dialog(self, _('Export Complete'),
                      _('Successfully exported keyboard shortcuts to %s') % save_path,
                      show=True, show_copy_button=False)

        except Exception as e:
            error_dialog(self, _('Export Failed'),
                      _('Failed to export keyboard shortcuts: %s') % str(e),
                      show=True)

    def export_settings(self, format_type):
        """Export settings data to CSV or XLSX format"""
        # Only export visible (filtered) settings
        visible_rows = []

        # Get the settings tab instance and its table
        settings_tab = self.tabs.widget(3)
        settings_table = settings_tab.settings_table  # Access the settings_table from the SettingsTab instance

        for row in range(settings_table.rowCount()):
            if not settings_table.isRowHidden(row):
                visible_rows.append(row)

        if not visible_rows:
            from calibre.gui2 import error_dialog
            error_dialog(self, _('Export Failed'),
                      _('No settings match the current filter criteria.'),
                      show=True)
            return

        # Get filename with timestamp
        default_name = self._get_export_filename(format_type)

        # Choose save location
        if format_type == 'csv':
            filters = _('CSV files (*.csv)')
            title = _('Export to CSV')
        else:
            filters = _('Excel files (*.xlsx)')
            title = _('Export to Excel')

        from calibre.gui2 import choose_save_file, info_dialog, error_dialog
        save_path = choose_save_file(self, title, filters,
                                  initial_filename=default_name)

        if not save_path:
            return

        try:
            # Collect data from table
            data = []
            headers = []
            visible_columns = []

            # Get headers from visible columns only
            for col in range(settings_table.columnCount()):
                if not settings_table.isColumnHidden(col):
                    headers.append(settings_table.horizontalHeaderItem(col).text())
                    visible_columns.append(col)

            # Get visible rows and columns only
            for row in visible_rows:
                row_data = []
                # Get the setting name for this row (assume column 1 is the setting name)
                setting_name = settings_table.item(row, 1).text() if settings_table.item(row, 1) else ''
                for col in visible_columns:
                    item = settings_table.item(row, col)
                    cell_text = item.text() if item else ''
                    # Robust: check column header for 'Value' instead of col==2
                    header_text = settings_table.horizontalHeaderItem(col).text()
                    if setting_name == _('Input Format Order') and header_text == _('Value'):
                        import ast
                        try:
                            v = cell_text
                            if v.startswith('[') and v.endswith(']'):
                                vlist = ast.literal_eval(v)
                                if isinstance(vlist, list):
                                    if len(vlist) > 19:
                                        cell_text = ', '.join(vlist[:19]) + ',+more'
                                    else:
                                        cell_text = ', '.join(vlist)
                        except Exception:
                            pass
                        if ',' in cell_text:
                            parts = [p.strip() for p in cell_text.split(',')]
                            if len(parts) > 19:
                                cell_text = ', '.join(parts[:19]) + ',+more'
                    row_data.append(cell_text)
                data.append(row_data)

            if format_type == 'csv':
                self.export_to_csv(save_path, headers, data)
            else:
                self.export_to_xlsx(save_path, headers, data)

            info_dialog(self, _('Export Complete'),
                      _('Successfully exported settings to %s') % save_path,
                      show=True, show_copy_button=False)

        except Exception as e:
            error_dialog(self, _('Export Failed'),
                      _('Failed to export settings: %s') % str(e),
                      show=True)

    def show_columns_context_menu(self, pos):
        """Show context menu for custom columns table"""
        index = self.columns_table.indexAt(pos)
        menu = QMenu(self)

        if index.isValid():
            # Copy data action
            copy_action = menu.addAction(self.copy_icon, _('Copy column data'))
            copy_action.triggered.connect(lambda: self.copy_column_data(index.row()))

            # If composite column, add action to copy template
            if self.columns_table.item(index.row(), 4).text() == _('Yes'):
                template_action = menu.addAction(self.copy_icon, _('Copy template'))
                template_action.triggered.connect(lambda: self.copy_template(index.row()))

            # Add action to copy display settings
            settings_action = menu.addAction(self.copy_icon, _('Copy display settings'))
            settings_action.triggered.connect(lambda: self.copy_display_settings(index.row()))

        menu.popup(self.columns_table.viewport().mapToGlobal(pos))

    def copy_column_data(self, row):
        """Copy all data for a custom column to clipboard"""
        data = []
        for col in range(self.columns_table.columnCount()):
            header = self.columns_table.horizontalHeaderItem(col).text()
            cell_text = self.columns_table.item(row, col).text() if self.columns_table.item(row, col) else ''
            data.append(f"{header}: {cell_text}")

        QApplication.clipboard().setText('\n'.join(data))
        self.show_copy_feedback(row, 0, self.columns_table)

    def copy_template(self, row):
        """Copy composite column template to clipboard"""
        template = self.columns_table.item(row, 5).text()
        QApplication.clipboard().setText(template)
        self.show_copy_feedback(row, 5, self.columns_table)

    def copy_display_settings(self, row):
        """Copy display settings to clipboard"""
        settings = self.columns_table.item(row, 6).data(Qt.ItemDataRole.UserRole)
        QApplication.clipboard().setText(settings)
        self.show_copy_feedback(row, 6, self.columns_table)

    def show_copy_feedback(self, row, col, table=None):
        """Show visual feedback when copying info"""
        if table is None:
            table = self.plugins_tab.plugin_table
        item = table.item(row, col)
        if not item:
            return

        # Save original icon if it exists
        orig_icon = item.icon()

        # Show success icon
        item.setIcon(self.copy_done_icon)

        # Reset icon after a short delay
        QTimer.singleShot(800, lambda: item.setIcon(orig_icon))




    # Plugin table context menu and info dialog logic is now handled in plugins_tab.py
    # (see PluginsTab.show_context_menu, show_plugin_info_dialog, etc.)

    def save_dialog_size(self):
        # Save dialog size
        size = self.size()
        gprefs['calibre_config_reports_dialog_size'] = (size.width(), size.height())

    def accept(self):
        self.save_dialog_size()
        QDialog.accept(self)

    def reject(self):
        # Stop the audio player when closing the dialog
        if hasattr(self, 'media_player'):
            self.media_player.stop()

        self.save_dialog_size()
        QDialog.reject(self)

    def show_shortcuts_column_menu(self):
        """Show menu to toggle shortcut columns visibility"""
        # Safely obtain header labels; when header items are missing (e.g., after
        # some table manipulations) fall back to the model's headerData to avoid
        # crashes and to ensure hidden columns still appear in the menu.
        headers = []
        model = self.shortcuts_table.model()
        for i in range(self.shortcuts_table.columnCount()):
            hitem = self.shortcuts_table.horizontalHeaderItem(i)
            if hitem and hitem.text():
                headers.append(hitem.text())
            else:
                try:
                    # Qt header data fallback
                    label = model.headerData(i, Qt.Orientation.Horizontal)
                    headers.append(str(label) if label is not None else f'Column {i}')
                except Exception:
                    headers.append(f'Column {i}')

        menu = QMenu(self)
        for i, header in enumerate(headers):
            action = menu.addAction(header)
            action.setCheckable(True)
            action.setChecked(not self.shortcuts_table.isColumnHidden(i))
            # Use default arg to bind current i correctly
            action.triggered.connect(lambda checked, col=i: self.toggle_column_to_state(self.shortcuts_table, col, checked))

        menu.popup(self.shortcuts_column_button.mapToGlobal(
            self.shortcuts_column_button.rect().bottomLeft()))

    def save_column_state(self, table_name):
        """Save the column order state and visibility for a table"""
        table = None
        if table_name == 'plugins':
            table = self.plugins_tab.plugin_table
        elif table_name == 'columns':
            table = self.columns_table
        elif table_name == 'shortcuts':
            table = self.shortcuts_table
        elif table_name == 'settings':
            # Get the settings table from the SettingsTab instance at index 3
            settings_tab = self.tabs.widget(3)
            if settings_tab and hasattr(settings_tab, 'settings_table'):
                table = settings_tab.settings_table
            else:
                return  # Exit early if the settings tab or its table doesn't exist yet

        if table:
            # Save column visibility state
            hidden_columns = []
            for col in range(table.columnCount()):
                if table.isColumnHidden(col):
                    hidden_columns.append(col)
            gprefs[f'calibre_config_reports_{table_name}_hidden_columns'] = hidden_columns

            # Save column order
            header = table.horizontalHeader()
            visual_indices = []
            for logical_index in range(header.count()):
                visual_indices.append(header.visualIndex(logical_index))
            gprefs[f'calibre_config_reports_{table_name}_column_order'] = visual_indices

            # Also save header labels by visual order for schema-robust restore
            labels_by_visual = []
            try:
                for visual_index in range(header.count()):
                    logical = header.logicalIndex(visual_index)
                    item = table.horizontalHeaderItem(logical)
                    labels_by_visual.append(item.text() if item else '')
                gprefs[f'calibre_config_reports_{table_name}_column_order_labels'] = labels_by_visual
            except Exception:
                # Best-effort only
                pass

            # Save column widths
            widths = []
            for col in range(table.columnCount()):
                widths.append(table.columnWidth(col))
            gprefs[f'calibre_config_reports_{table_name}_column_widths'] = widths
            # Try to flush gprefs to disk so the state persists across restarts
            try:
                gprefs.save()
            except Exception:
                # Not critical if save isn't available in this environment
                pass

    def restore_column_state(self, table_name):
        """Restore the column order state and visibility for a table"""
        table = None
        if table_name == 'plugins':
            table = self.plugins_tab.plugin_table
        elif table_name == 'columns':
            table = self.columns_table
        elif table_name == 'shortcuts':
            table = self.shortcuts_table
        elif table_name == 'settings':
            # Get the settings table from the SettingsTab instance at index 3
            settings_tab = self.tabs.widget(3)
            if settings_tab and hasattr(settings_tab, 'settings_table'):
                table = settings_tab.settings_table
            else:
                return  # Exit early if the settings tab or its table doesn't exist yet


        if table:
            # Temporarily disable sorting and block signals during restoration to avoid
            # saving intermediate states while we move sections/resize columns
            header = table.horizontalHeader()
            try:
                orig_sorting = table.isSortingEnabled()
            except Exception:
                orig_sorting = False
            try:
                table.setSortingEnabled(False)
            except Exception:
                pass
            try:
                table.blockSignals(True)
                header.blockSignals(True)
            except Exception:
                pass

            widths = gprefs.get(f'calibre_config_reports_{table_name}_column_widths', None)
            # Normalize widths to a simple list of ints if stored as dict/string from older versions
            try:
                if widths is None:
                    norm_widths = None
                elif isinstance(widths, dict):
                    items = []
                    for k, v in widths.items():
                        try:
                            idx = int(k)
                            items.append((idx, int(v)))
                        except Exception:
                            continue
                    items.sort()
                    norm_widths = [w for i, w in items]
                elif isinstance(widths, (list, tuple)):
                    norm_widths = [int(w) if w is not None else 0 for w in widths]
                elif isinstance(widths, str):
                    import ast
                    try:
                        parsed = ast.literal_eval(widths)
                    except Exception:
                        parsed = None
                    if isinstance(parsed, dict):
                        items = []
                        for k, v in parsed.items():
                            try:
                                idx = int(k)
                                items.append((idx, int(v)))
                            except Exception:
                                continue
                        items.sort()
                        norm_widths = [w for i, w in items]
                    elif isinstance(parsed, (list, tuple)):
                        norm_widths = [int(w) if w is not None else 0 for w in parsed]
                    else:
                        norm_widths = None
            except Exception:
                norm_widths = None
            widths = norm_widths
            hidden_columns = gprefs.get(f'calibre_config_reports_{table_name}_hidden_columns', None)

            # For plugins tab, handle column restoration properly
            if table_name == 'plugins':
                # If we have saved settings, apply them
                if hidden_columns is not None:
                    # Apply the column visibility
                    for col in range(table.columnCount()):
                        table.setColumnHidden(col, col in hidden_columns)

                    # Restore column order
                    visual_indices = gprefs.get(f'calibre_config_reports_{table_name}_column_order', None)
                    if visual_indices:
                        # Build desired logical order by visual position to avoid index churn
                        count = min(len(visual_indices), header.count())
                        desired_by_visual = [None] * count
                        for logical_index, visual_index in enumerate(visual_indices[:count]):
                            if 0 <= visual_index < count:
                                desired_by_visual[visual_index] = logical_index
                        # Fill any gaps with current sequence to stay safe
                        cur_seq = [header.logicalIndex(i) for i in range(count)]
                        ci = 0
                        for i in range(count):
                            if desired_by_visual[i] is None:
                                desired_by_visual[i] = cur_seq[ci]
                                ci += 1
                        # Move each desired logical to its target visual position
                        for target_pos, logical in enumerate(desired_by_visual):
                            current_pos = header.visualIndex(logical)
                            if current_pos != target_pos:
                                header.moveSection(current_pos, target_pos)
                else:
                    # No saved settings exist, apply defaults
                    from calibre_plugins.calibre_config_reports.plugins_tab import PluginsTab
                    if hasattr(table.parent(), 'populate_plugins_table'):
                        # Re-run the default column setup from populate_plugins_table
                        plugins_tab = table.parent()
                        if hasattr(plugins_tab, 'plugin_table') and plugins_tab.plugin_table == table:
                            # Apply default column visibility
                            desired_columns = [
                                0,  # Remove (checkbox)
                                1,  # Icon
                                2,  # Name
                                3,  # Version
                                4,  # Author
                                5,  # Installed Date
                                7,  # Size
                                8,  # Released
                #          9,  # Forum Thread
                #          10, # Min Calibre
                #          11, # Type
                #          12, # Donate
                #          13, # Category
                #          14  # Description
                            ]

                            # Hide columns not in the desired list
                            for col in range(table.columnCount()):
                                table.setColumnHidden(col, col not in desired_columns)

                            # Store the hidden columns in preferences
                            gprefs['calibre_config_reports_plugins_hidden_columns'] = [
                                col for col in range(table.columnCount())
                                if col not in desired_columns
                            ]

                            # Set the visual column order
                            for target_visual_pos, logical_idx in enumerate(desired_columns):
                                current_visual = header.visualIndex(logical_idx)
                                if current_visual != target_visual_pos:
                                    header.moveSection(current_visual, target_visual_pos)

                            # Save the column order after applying defaults
                            self.save_column_state(table_name)
                            # Ensure a wider Action column for fresh installs
                            try:
                                self.shortcuts_table.setColumnWidth(0, 400)
                            except Exception:
                                pass
            # For shortcuts tab, if no saved settings exist, enforce the correct
            # default visual order: Action, Group, Default, Assigned. This fixes
            # fresh-install cases where the Assigned column can appear before Default.
            if table_name == 'shortcuts' and (hidden_columns is None or not hidden_columns):
                try:
                    # Default logical indices in desired visual order.
                    # Ensure Assigned (logical 3) appears before Default (logical 2)
                    # so fresh installs show: Action, Group, Assigned, Default.
                    desired_shortcuts_cols = [0, 1, 3, 2]
                    # Ensure all columns are visible by default
                    for col in range(table.columnCount()):
                        table.setColumnHidden(col, False)
                    # Move sections to desired order
                    for target_visual_pos, logical_idx in enumerate(desired_shortcuts_cols):
                        current_visual = header.visualIndex(logical_idx)
                        if current_visual != target_visual_pos:
                            header.moveSection(current_visual, target_visual_pos)
                    # Save the default state so subsequent launches restore correctly
                    self.save_column_state(table_name)
                except Exception:
                    pass
            # Continue with common column width restoration logic below
            # (Removed the early return that was preventing width restoration)

            # If we still don't have hidden_columns (e.g., for other tabs without defaults), use empty list
            if hidden_columns is None:
                hidden_columns = []

            # Apply the column visibility
            for col in range(table.columnCount()):
                table.setColumnHidden(col, col in hidden_columns)

            # Restore column order
            visual_indices = gprefs.get(f'calibre_config_reports_{table_name}_column_order', None)
            labels_by_visual = gprefs.get(f'calibre_config_reports_{table_name}_column_order_labels', None)
            applied_order = False
            if visual_indices and len(visual_indices) == header.count():
                # Robust reordering: compute desired logical per visual index
                count = header.count()
                desired_by_visual = [None] * count
                for logical_index, visual_index in enumerate(visual_indices[:count]):
                    if 0 <= visual_index < count:
                        desired_by_visual[visual_index] = logical_index
                # Fill gaps with current order to keep array valid
                cur_seq = [header.logicalIndex(i) for i in range(count)]
                ci = 0
                for i in range(count):
                    if desired_by_visual[i] is None:
                        desired_by_visual[i] = cur_seq[ci]
                        ci += 1
                # Apply moves left-to-right
                for target_pos, logical in enumerate(desired_by_visual):
                    current_pos = header.visualIndex(logical)
                    if current_pos != target_pos:
                        header.moveSection(current_pos, target_pos)
                applied_order = True
            # Fallback: label-based reordering if indices are missing/mismatched
            if not applied_order and labels_by_visual and isinstance(labels_by_visual, list):
                # Map current header labels to logical indices
                label_to_logical = {}
                for i in range(header.count()):
                    li = header.logicalIndex(i)
                    item = table.horizontalHeaderItem(li)
                    text = item.text() if item else ''
                    # Only first occurrence is stored; labels are expected unique
                    if text not in label_to_logical:
                        label_to_logical[text] = li
                desired_by_visual = []
                for text in labels_by_visual:
                    li = label_to_logical.get(text)
                    if li is not None:
                        desired_by_visual.append(li)
                # Ensure list has correct length
                cur_seq = [header.logicalIndex(i) for i in range(header.count())]
                for li in cur_seq:
                    if li not in desired_by_visual:
                        desired_by_visual.append(li)
                # Apply moves
                for target_pos, logical in enumerate(desired_by_visual[:header.count()]):
                    current_pos = header.visualIndex(logical)
                    if current_pos != target_pos:
                        header.moveSection(current_pos, target_pos)

            # Restore column widths
            if widths:
                for col, width in enumerate(widths):
                    if col < table.columnCount():
                        try:
                            width_int = int(width)
                            # Special handling for plugins table columns with constraints
                            if table_name == 'plugins' and col == 0:
                                # Remove column should always be 30px
                                table.setColumnWidth(col, 30)
                            elif table_name == 'plugins' and col == 1:
                                # Icon column should always be 48px
                                table.setColumnWidth(col, 48)
                            elif table_name == 'plugins' and col == 3:
                                # Version column max 75px
                                table.setColumnWidth(col, min(width_int, 75))
                            elif table_name == 'shortcuts' and col == 0:
                                # Action column max 450px
                                table.setColumnWidth(col, min(width_int, 450))
                            else:
                                # Apply saved width for all other columns
                                min_width = 80  # Increased from 15 to 80px for better readability
                                table.setColumnWidth(col, max(width_int, min_width))
                        except (ValueError, TypeError):
                            # Skip invalid width values
                            pass

            # Manual column width mapping for Custom Columns tab (only if no saved widths)
            elif table_name == 'columns':
                # Example: adjust as needed for your preferred widths
                table.setColumnWidth(0, 120)  # Name
                table.setColumnWidth(1, 100)  # Label
                table.setColumnWidth(2, 80)   # Type
                table.setColumnWidth(3, 200)  # Description
                table.setColumnWidth(4, 80)   # Is Composite
                table.setColumnWidth(5, 180)  # Template
                table.setColumnWidth(6, 220)  # Display Settings
            # Manual column width mapping for Calibre Settings tab
            elif table_name == 'settings':
                pass  # No hardcoded column widths; settings_tab.py controls widths

            # Persist the finalized state once to ensure preferences match the restored order
            try:
                self.save_column_state(table_name)
            except Exception:
                pass

            # Re-enable sorting and signals after restoration
            try:
                header.blockSignals(False)
                table.blockSignals(False)
            except Exception:
                pass
            try:
                table.setSortingEnabled(orig_sorting)
            except Exception:
                pass

    def show_shortcuts_context_menu(self, pos):
        """Show context menu for shortcuts table"""
        index = self.shortcuts_table.indexAt(pos)
        menu = QMenu(self)

        if index.isValid():
            # Open shortcuts action
            edit_action = menu.addAction(_('Open Shortcuts'))
            edit_action.triggered.connect(lambda: self.edit_shortcut(index))

            menu.addSeparator()

        # Add column visibility toggle
        self.add_column_visibility_menu(menu)

        menu.popup(self.shortcuts_table.viewport().mapToGlobal(pos))

    def edit_shortcut(self, index):
        """Open the shortcut editor for the selected shortcut"""
        row = index.row()
        name_item = self.shortcuts_table.item(row, 0)
        if not name_item:
            return

        # Just use Calibre's built-in preferences dialog
        self.gui.iactions['Preferences'].do_config(
            initial_plugin=('Advanced', _('Keyboard')),
            close_after_initial=True
        )

        # Refresh display after preferences closes
        self.gui.keyboard.finalize()
        self.populate_shortcuts_tab()
        # Ensure user column state persists after repopulation
        try:
            self.restore_column_state('shortcuts')
        except Exception:
            pass

    def add_column_visibility_menu(self, menu):
        """Add column visibility submenu to context menu"""
        columns_menu = menu.addMenu(_('Show/Hide Columns'))
        model = self.shortcuts_table.model()
        for i in range(self.shortcuts_table.columnCount()):
            hitem = self.shortcuts_table.horizontalHeaderItem(i)
            if hitem and hitem.text():
                col_name = hitem.text()
            else:
                try:
                    col_name = model.headerData(i, Qt.Orientation.Horizontal) or f'Column {i}'
                except Exception:
                    col_name = f'Column {i}'

            action = columns_menu.addAction(str(col_name))
            action.setCheckable(True)
            action.setChecked(not self.shortcuts_table.isColumnHidden(i))
            action.triggered.connect(lambda checked, col=i: self.toggle_column_to_state(self.shortcuts_table, col, checked))

    def on_tab_changed(self, index):
        """Handle tab change event"""
        gprefs['calibre_config_reports_last_tab'] = index

        # If switching to the shortcuts tab, enforce the action column width.
        # NOTE: We intentionally do NOT automatically refresh/repopulate the
        # shortcuts table here because that can cause noticeable UI lag when
        # users switch tabs frequently. CCR provides a Refresh button which the
        # user can press after making changes to Calibre shortcuts. This keeps
        # tab switching snappy and avoids surprising pauses.
        if index == 2:  # Shortcuts tab index
            # Enforce action column width only (do not trigger expensive rebuild)
            try:
                header = self.shortcuts_table.horizontalHeader()
                # Ensure Action column (0) does not exceed configured max
                if self.shortcuts_table.columnCount() > 0:
                    self.shortcuts_table.setColumnWidth(0, min(self.shortcuts_table.columnWidth(0), 450))
            except Exception:
                pass

    def edit_custom_column(self, row, col):
        """Show message about editing custom columns via Preferences"""
        from calibre.gui2 import info_dialog
        info_dialog(self, _('Edit Custom Columns'),
                   _('To edit custom columns, please go to:\n\n'
                   'Preferences > Interface > Add your own columns\n\n'
                   'You can access this via the Preferences button on the main toolbar '
                   'or by pressing Ctrl+P'),
                   show=True)

    def show_history_dialog(self, plugin_name):
        """Show plugin history in a dialog"""
        dialog = QDialog(self)
        dialog.setWindowTitle(_('Plugin History'))
        dialog.setMinimumWidth(600)
        dialog.setMinimumHeight(400)

        layout = QVBoxLayout(dialog)

        # Get the forum thread URL from plugin cache
        cache_info = self.plugin_cache.get(plugin_name, {})
        forum_url = cache_info.get('thread_url', '')

        # Initialize history text
        history_text = ''
        if forum_url:
            history_text += _('Click the link above to open the forum thread in your browser.')
        else:
            history_text = _('No version history or forum thread available for this plugin.')

        text = QLabel(history_text)
        text.setStyleSheet("""
            QLabel {
                color: palette(text);
                padding: 10px;
                background: palette(base);
                border-radius: 5px;
            }
        """)

        text = QLabel(history_text, dialog)
        text.setWordWrap(True)
        text.setTextFormat(Qt.TextFormat.RichText)
        text.setTextInteractionFlags(Qt.TextInteractionFlag.TextSelectableByMouse)
        text.setStyleSheet("""
            QLabel {
                background-color: palette(base);
                padding: 10px;
                border-radius: 5px;
                font-size: 10pt;
                line-height: 1.4;
            }
        """)
        layout.addWidget(text)

        buttons = QDialogButtonBox(QDialogButtonBox.StandardButton.Close)
        buttons.rejected.connect(dialog.reject)
        layout.addWidget(buttons)

        dialog.exec()

    # Plugin functionality moved to plugins_tab.py

    # Plugin functionality moved to plugins_tab.py

    def show_column_info_dialog(self, row, col):
        """Show a dialog with copyable column display settings when user double-clicks a column"""
        # Get display settings and template from the column
        settings = self.columns_table.item(row, 6).data(Qt.ItemDataRole.UserRole)
        name = self.columns_table.item(row, 0).text()
        is_composite = self.columns_table.item(row, 4).text() == _('Yes')
        template = self.columns_table.item(row, 5).text() if is_composite else None

        # Show the column info dialog
        from .action import ColumnInfoDialog
        dialog = ColumnInfoDialog(self, name, settings, template)
        dialog.exec()

    def update_statistics(self):
        """Update the statistics labels for all tables"""
        # Plugin stats
        if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'plugin_table'):
            plugin_table = self.plugins_tab.plugin_table
            visible_plugins = sum(1 for row in range(plugin_table.rowCount())
                                if not plugin_table.isRowHidden(row))
            total_plugins = plugin_table.rowCount()
            self.plugin_stats.setText(_("🧩 %s: %d/%d") % (_("Plugins"), visible_plugins, total_plugins))

        # Custom columns stats
        if hasattr(self, 'columns_table') and self.columns_table is not None:
            try:
                visible_columns = sum(1 for row in range(self.columns_table.rowCount())
                                    if not self.columns_table.isRowHidden(row))
                total_columns = self.columns_table.rowCount()
            except Exception:
                visible_columns, total_columns = 0, 0
        else:
            visible_columns, total_columns = 0, 0
        self.column_stats.setText(_("🏛 %s: %d/%d") % (_("Custom Columns"), visible_columns, total_columns))

        # Shortcuts stats
        if hasattr(self, 'shortcuts_table') and self.shortcuts_table is not None:
            try:
                visible_shortcuts = sum(1 for row in range(self.shortcuts_table.rowCount())
                                      if not self.shortcuts_table.isRowHidden(row))
                total_shortcuts = self.shortcuts_table.rowCount()
            except Exception:
                visible_shortcuts, total_shortcuts = 0, 0
        else:
            visible_shortcuts, total_shortcuts = 0, 0
        self.shortcut_stats.setText(_("⚡ %s: %d/%d") % (_("Shortcuts"), visible_shortcuts, total_shortcuts))

    def open_path(self, path):
        """Open path in system file explorer, works cross-platform"""
        if iswindows:
            os.startfile(path)
        elif os.path.exists('/usr/bin/xdg-open'):  # Linux
            os.system('xdg-open "%s"' % path)
        else:  # macOS and others
            os.system('open "%s"' % path)

    def show_settings_column_menu(self):
        """Show menu to toggle settings columns visibility"""
        headers = [self.settings_table.horizontalHeaderItem(i).text()
                  for i in range(self.settings_table.columnCount())]

        menu = QMenu(self)
        for i, header in enumerate(headers):
            action = menu.addAction(header)
            action.setCheckable(True)
            action.setChecked(not self.settings_table.isColumnHidden(i))
            action.triggered.connect(lambda checked, col=i:
                self.toggle_column_to_state(self.settings_table, col, checked))

        menu.popup(self.settings_column_button.mapToGlobal(
            self.settings_column_button.rect().bottomLeft()))

    def initialize_stats_bar(self):
        """Initialize the statistics bar without audio controls"""
        import platform
        from calibre.constants import __version__
        stats_layout = QHBoxLayout()

        # Left-side compact block: Calibre version (first line) and OS info (second line,
        # elided with tooltip). Keeping this as a small fixed-width widget helps the
        # central stats feel visually centered regardless of OS string length.
        from PyQt5.QtGui import QFontMetrics
        from PyQt5.QtWidgets import QSizePolicy

        left_widget = QWidget()
        left_layout = QVBoxLayout(left_widget)
        left_layout.setContentsMargins(6, 2, 6, 2)
        left_layout.setSpacing(0)

        # Calibre version (stronger visual weight)
        calibre_label = QLabel(_('Calibre {}').format(__version__))
        calibre_label.setStyleSheet('QLabel { color: #AABBCC; font-size: 8pt; font-weight: 600; padding: 0px; }')
        left_layout.addWidget(calibre_label)

        # OS info (muted, smaller and elided if long)
        os_full = platform.platform()
        os_label = QLabel()
        os_label.setStyleSheet('QLabel { color: #AABBCC; font-size: 8pt; padding: 0px; }')
        fm = QFontMetrics(os_label.font())
        # Reserve ~300px for the left block when eliding
        elided_os = fm.elidedText(os_full, Qt.TextElideMode.ElideRight, 300)
        os_label.setText(elided_os)
        os_label.setToolTip(os_full)
        left_layout.addWidget(os_label)

        # Keep the left block width predictable so the central stats remain centered
        left_widget.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Preferred)
        left_widget.setMinimumWidth(300)
        stats_layout.addWidget(left_widget)

        # Create central widget to hold stats centered
        stats_widget = QWidget()
        stats_widget.setMinimumWidth(400)  # Ensure enough width for centering

        # Inner layout for stats that will be centered
        inner_layout = QHBoxLayout(stats_widget)
        inner_layout.setSpacing(20)  # Add spacing between stats

        # Stats labels
        self.plugin_stats = QLabel("")
        self.column_stats = QLabel("")
        self.shortcut_stats = QLabel("")
        for label in [self.plugin_stats, self.column_stats, self.shortcut_stats]:
            label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            label.setStyleSheet('QLabel { padding: 2px 8px; font-size: 11pt; }')
            inner_layout.addWidget(label)

        # Add stretches to center the stats widget (we'll balance left/right with a
        # small filler so the visual center aligns with the dialog center)
        stats_layout.addStretch(1)
        stats_layout.addWidget(stats_widget)
        stats_layout.addStretch(1)

        # Right-side filler (adjusted to balance left block) - starts at 0
        right_filler = QWidget()
        right_filler.setFixedWidth(0)
        stats_layout.addWidget(right_filler)

        # Add CCR version label to the right side
        from . import PluginCatalogPlugin
        version = ".".join(str(x) for x in PluginCatalogPlugin.version)
        version_label = QLabel(_('CCR v.{}').format(version))
        version_label.setStyleSheet("""
            QLabel {
            color: #AABBCC;
            font-size: 9pt;
            padding: 4px;
                }
            """)
        stats_layout.addWidget(version_label)

        # Helper to adjust the right filler so left and (filler+version) balance
        def _adjust_stats_balance():
            try:
                left_w = left_widget.width()
                version_w = version_label.sizeHint().width()
                # filler so that left_w == filler + version_w
                filler_w = max(0, left_w - version_w)
                right_filler.setFixedWidth(filler_w)
            except Exception:
                pass

        # Run once after layout is settled
        QTimer.singleShot(0, _adjust_stats_balance)

        # Wrap/override the dialog resizeEvent to keep the filler updated on resize
        orig_resize = getattr(self, 'resizeEvent', None)
        def _new_resize(event):
            try:
                _adjust_stats_balance()
            except Exception:
                pass
            if orig_resize:
                try:
                    orig_resize(event)
                except Exception:
                    pass
            else:
                super(PluginCatalogDialog, self).resizeEvent(event)

        # Bind to the instance so we don't change the class method globally
        self.resizeEvent = _new_resize

        return stats_layout

    def _get_plugin_size(self, plugin):
        """Get plugin size in bytes and return both raw and formatted size"""
        if not plugin:
            return 0, "0B"
        try:
            # Get plugin path and size
            plugin_path = getattr(plugin, 'plugin_path', None)
            if not plugin_path:
                return 0, "N/A"

            from .size_utils import get_file_size_bytes, format_size
            size_bytes = get_file_size_bytes(plugin_path)
            return size_bytes, format_size(size_bytes)
        except Exception:
            return 0, "N/A"

    def show_columns_column_menu(self):
        """Show menu to toggle custom columns table column visibility"""
        headers = [self.columns_table.horizontalHeaderItem(i).text()
                  for i in range(self.columns_table.columnCount())]

        menu = QMenu(self)
        for i, header in enumerate(headers):
            action = menu.addAction(header)
            action.setCheckable(True)
            action.setChecked(not self.columns_table.isColumnHidden(i))
            action.triggered.connect(lambda checked, col=i:
                self.toggle_column_to_state(self.columns_table, col, checked))

        menu.popup(self.columns_column_button.mapToGlobal(
            self.columns_column_button.rect().bottomLeft()))

    def toggle_column_to_state(self, table, col, checked):
        """Toggle column visibility to match the checked state"""
        if col >= 0:
            # Set hidden state opposite to checked - column is visible when checked is True
            table.setColumnHidden(col, not checked)

            # Special handling for the Author column when Description column is toggled
            if table == self.plugins_tab.plugin_table:
                description_col =  10  # Description column index
                author_col = 3        # Author column index

                # When Description is shown but Author is too wide, constrain it
                if col == description_col and checked:  # Description is being shown
                    if table.columnWidth(author_col) > 200:
                        table.setColumnWidth(author_col, 130)

                # When Description is hidden, constrain Author width
                elif col == description_col and not checked:  # Description is being hidden
                    table.setColumnWidth(author_col, 130)

            # Special handling for the shortcuts table - maintain Action column width
            elif table == self.shortcuts_table:
                # DON'T resize columns to contents - this destroys saved widths!
                # Just enforce the max width constraint for Action column
                # If the Action column was just shown, ensure it has a sensible width
                if checked and col == 0:
                    # Try to restore saved width from preferences
                    try:
                        widths = gprefs.get('calibre_config_reports_shortcuts_column_widths', None)
                        if widths and isinstance(widths, (list, tuple)) and len(widths) > 0:
                            saved = widths[0]
                            try:
                                saved_w = int(saved) if saved is not None else 0
                            except Exception:
                                saved_w = 0
                            if saved_w > 15:
                                table.setColumnWidth(0, min(saved_w, 450))
                            else:
                                table.setColumnWidth(0, 200)
                    except Exception:
                        # Best-effort default
                        table.setColumnWidth(0, 200)
                else:
                    if table.columnWidth(0) > 450:
                        table.setColumnWidth(0, 450)

            # For other tables, ensure newly visible columns have reasonable widths
            if checked:  # Column is being made visible
                # Set a minimum width for any column being shown
                current_width = table.columnWidth(col)
                min_visible_width = 80  # Minimum reasonable width for any visible column
                if current_width < min_visible_width:
                    table.setColumnWidth(col, min_visible_width)

            # Only resize to contents for non-plugins tables, and only if not setting specific widths above
            if table != getattr(self, 'shortcuts_table', None):
                # Don't resize to contents as it can make columns too narrow - just use current widths
                pass

            self.update_statistics()

            # Save the new state. The settings table lives inside the SettingsTab
            # instance (tabs.widget(3)). Don't assume self.settings_table exists.
            try:
                if hasattr(self, 'plugins_tab') and table == getattr(self.plugins_tab, 'plugin_table', None):
                    self.save_column_state('plugins')
                elif table == getattr(self, 'columns_table', None):
                    self.save_column_state('columns')
                elif table == getattr(self, 'shortcuts_table', None):
                    self.save_column_state('shortcuts')
                else:
                    # Check for settings table inside the SettingsTab instance (tab index 3)
                    try:
                        settings_tab = self.tabs.widget(3) if hasattr(self, 'tabs') else None
                        if settings_tab and hasattr(settings_tab, 'settings_table') and table == settings_tab.settings_table:
                            self.save_column_state('settings')
                    except Exception:
                        # last-resort: if table has a known objectName or attribute indicating settings, try saving settings
                        pass
            except Exception:
                # Non-fatal if saving fails
                pass

    def open_main_preferences_dialog(self, row, col):
        """Open the main Calibre Preferences dialog on double-click in settings table"""
        self.gui.iactions['Preferences'].do_config()

    def setup_history_tab(self):
        tab = QWidget()
        layout = QVBoxLayout(tab)

        # --- Filter bar for history tab (replicates other tabs) ---
        filter_layout = QHBoxLayout()
        filter_label = QLabel(_('Filter:'))
        self.history_filter_edit = QLineEdit()
        self.history_filter_edit.setPlaceholderText(_('Search history...'))
        self.history_filter_edit.textChanged.connect(self.apply_history_filter)
        try:
            self.history_filter_edit.returnPressed.connect(lambda: self.apply_history_filter(self.history_filter_edit.text()))
        except Exception:
            pass
        filter_layout.addWidget(filter_label)
        # Add clear button visually inside the QLineEdit (like plugins tab)
        self.history_filter_clear_btn = QToolButton(self.history_filter_edit)
        self.history_filter_clear_btn.setIcon(QIcon.ic('clear_left.png'))
        self.history_filter_clear_btn.setToolTip(_('Clear filter'))
        self.history_filter_clear_btn.setCursor(Qt.CursorShape.ArrowCursor)
        self.history_filter_clear_btn.setVisible(False)
        self.history_filter_clear_btn.setStyleSheet('QToolButton { border: none; padding: 0px; }')
        self.history_filter_clear_btn.setFixedSize(18, 18)
        self.history_filter_clear_btn.clicked.connect(self.history_filter_edit.clear)
        self.history_filter_edit.textChanged.connect(
            lambda text: self.history_filter_clear_btn.setVisible(bool(text)))
        from PyQt5.QtWidgets import QStyle
        frame = self.history_filter_edit.style().pixelMetric(QStyle.PixelMetric.PM_DefaultFrameWidth)
        self.history_filter_edit.setTextMargins(0, 0, self.history_filter_clear_btn.width() + frame + 2, 0)
        def move_clear_btn():
            self.history_filter_clear_btn.move(
                self.history_filter_edit.rect().right() - self.history_filter_clear_btn.width() - frame,
                (self.history_filter_edit.rect().height() - self.history_filter_clear_btn.height()) // 2)
        self.history_filter_edit.resizeEvent = lambda event: move_clear_btn()
        move_clear_btn()
        filter_layout.addWidget(self.history_filter_edit)

        # Add export buttons for history tab with icons
        history_csv_export_button = QPushButton(_('Export &CSV'))
        history_csv_export_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        history_csv_export_button.setIcon(QIcon.ic('save.png'))
        history_csv_export_button.setToolTip(_('Export visible history to CSV'))
        history_csv_export_button.setAutoDefault(False)
        history_csv_export_button.setDefault(False)
        history_csv_export_button.clicked.connect(lambda: self.export_history('csv'))
        history_xlsx_export_button = QPushButton(_('Export &XLSX'))
        history_xlsx_export_button.setStyleSheet('padding: 7px 18px; font-size:10pt;')
        history_xlsx_export_button.setIcon(QIcon.ic('save.png'))
        history_xlsx_export_button.setToolTip(_('Export visible history to Excel (XLSX)'))
        history_xlsx_export_button.setAutoDefault(False)
        history_xlsx_export_button.setDefault(False)
        history_xlsx_export_button.clicked.connect(lambda: self.export_history('xlsx'))
        filter_layout.addWidget(history_csv_export_button)
        filter_layout.addWidget(history_xlsx_export_button)

        layout.addLayout(filter_layout)

        # Create history table
        self.history_table = QTableWidget()
        self.history_table.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.history_table.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.history_table.setAlternatingRowColors(True)
        self.history_table.setSortingEnabled(True)
        self.history_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
        self.history_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

        # Enable context menu
        self.history_table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.history_table.customContextMenuRequested.connect(self.show_history_context_menu)

        # Initial population
        self.populate_history_table()

        # Add table to layout
        layout.addWidget(self.history_table)

        return tab
    def apply_history_filter(self, text):
        """Filter the history table based on search text"""
        filter_text = text.lower()
        for row in range(self.history_table.rowCount()):
            show = False
            for col in range(self.history_table.columnCount()):
                item = self.history_table.item(row, col)
                if item and filter_text in item.text().lower():
                    show = True
                    break
            self.history_table.setRowHidden(row, not show)

    def export_history(self, format_type):
        """Export history data to CSV or XLSX format"""
        from calibre.gui2 import choose_save_file, info_dialog, error_dialog
        visible_rows = [row for row in range(self.history_table.rowCount()) if not self.history_table.isRowHidden(row)]
        if not visible_rows:
            error_dialog(self, _('Export Failed'),
                         _('No history entries to export.'),
                         show=True)
            return
        # Get filename with timestamp
        default_name = self._get_export_filename(format_type)
        # Choose save location
        if format_type == 'csv':
            filters = _('CSV files (*.csv)')
            title = _('Export to CSV')
        else:
            filters = _('Excel files (*.xlsx)')
            title = _('Export to Excel')
        save_path = choose_save_file(self, title, filters, initial_filename=default_name)
        if not save_path:
            return
        try:
            # Collect data from table
            headers = [self.history_table.horizontalHeaderItem(i).text() for i in range(self.history_table.columnCount()) if not self.history_table.isColumnHidden(i)]
            data = []
            for row in visible_rows:
                row_data = []
                for col in range(self.history_table.columnCount()):
                    if not self.history_table.isColumnHidden(col):
                        item = self.history_table.item(row, col)
                        cell_text = item.text() if item else ''
                        row_data.append(cell_text)

                        # (debug prints removed)

                data.append(row_data)
            if format_type == 'csv':
                self.export_to_csv(save_path, headers, data)
            else:
                self.export_to_xlsx(save_path, headers, data)
            info_dialog(self, _('Export Complete'),
                        _('Successfully exported history to %s') % save_path,
                        show=True, show_copy_button=False)
        except Exception as e:
            error_dialog(self, _('Export Failed'),
                         _('Failed to export history: %s') % str(e),
                         show=True)

    def populate_history_table(self):
        try:
            monitor_and_log_config_events()
        except Exception as e:
            pass

        try:
            # Automatically fix malformed entries before displaying
            fixed_count = self.fix_malformed_history_entries()
            if fixed_count > 0 and CCR_DEBUG_ENABLED:
                prints(f"CCR: Fixed {fixed_count} malformed history entries")

            history = HistoryLogger.get_history()
            # Ensure the table does not re-sort or repaint while we populate it
            try:
                self.history_table.setSortingEnabled(False)
                self.history_table.blockSignals(True)
                self.history_table.setUpdatesDisabled(True)
            except Exception:
                pass

            self.history_table.setColumnCount(3)
            self.history_table.setHorizontalHeaderLabels([_('Timestamp'), _('Event'), _('Details')])
            # Clear any existing contents and set the new row count
            try:
                self.history_table.clearContents()
            except Exception:
                pass
            self.history_table.setRowCount(len(history))

            for row, entry in enumerate(reversed(history)):
                # Debug logging for all entries
                if CCR_DEBUG_ENABLED:
                    entry_time = entry.get('timestamp', '')
                    entry_type = entry.get('event_type', '')
                    entry_details = entry.get('details', {})
                    if '08:21:01' in entry_time:
                        prints(f"CCR DEBUG: Processing entry {row}: time={entry_time}, type={entry_type}")
                        if isinstance(entry_details, dict) and 'setting' in entry_details:
                            prints(f"CCR DEBUG: Setting name: '{entry_details.get('setting', '')}'")

                # Handle timestamp
                timestamp = entry.get('timestamp', '')
                self.history_table.setItem(row, 0, QTableWidgetItem(timestamp))

                # Handle event type and details
                event_type = entry.get('event_type', '')
                details = entry.get('details', {})  # Ensure details is a dict

                # Get display event type from details if available
                if isinstance(details, dict):
                    display_event_type = details.get('event_type', '')
                    if not display_event_type:
                        # Fallback if event_type not in details
                        if event_type == 'setting_changed':
                            display_event_type = 'Setting changed'
                        elif event_type == 'calibre_updated':
                            display_event_type = 'Calibre updated'
                        elif event_type == 'calibre_reverted':
                            display_event_type = 'Calibre reverted'
                        elif event_type == 'plugin_installed':
                            display_event_type = 'Plugin installed'
                        elif event_type == 'plugin_updated':
                            display_event_type = 'Plugin updated'
                        elif event_type == 'plugin_removed':
                            display_event_type = 'Plugin removed'
                        elif event_type == 'plugin_downgraded':
                            display_event_type = 'Plugin downgraded'
                        elif event_type == 'plugin_reinstalled':
                            display_event_type = 'Plugin reinstalled'
                        else:
                            display_event_type = event_type.replace('_', ' ').capitalize()
                else:
                    display_event_type = event_type.replace('_', ' ').capitalize()

                self.history_table.setItem(row, 1, QTableWidgetItem(display_event_type))

                # Format details string based on event type - ensure we're working with current entry
                current_details = entry.get('details', {})  # Get fresh details for this specific entry
                if isinstance(current_details, dict):
                    name = current_details.get('name', '')
                    if event_type in ['plugin_updated', 'plugin_downgraded']:
                        old_version = current_details.get('old_version', 'Unknown')
                        new_version = current_details.get('new_version', 'Unknown')
                        details_str = f"{name} (v{old_version} ⇒ v{new_version})"
                    elif event_type == 'plugin_installed':
                        version = current_details.get('version', 'Unknown')
                        details_str = f"{name} (v{version})"
                    elif event_type == 'plugin_removed':
                        version = current_details.get('version', 'Unknown')
                        details_str = f"{name} (v{version})"
                    elif event_type == 'plugin_reinstalled':
                        version = current_details.get('version', 'Unknown')
                        details_str = f"{name} (v{version})"
                    elif event_type == 'calibre_updated' or event_type == 'calibre_reverted':
                        old_version = current_details.get('old_version', 'Unknown')
                        new_version = current_details.get('new_version', 'Unknown')
                        details_str = f"Calibre (v{old_version} ⇒ v{new_version})"
                    elif event_type == 'setting_changed':
                        # Use direct access to avoid any potential variable contamination
                        # Extract values directly from current_details without intermediate variables
                        setting_category = str(current_details.get('category', ''))
                        setting_name = str(current_details.get('setting', ''))
                        setting_old_value = str(current_details.get('old_value', ''))
                        setting_new_value = str(current_details.get('new_value', ''))
                        setting_source = str(current_details.get('source', ''))

                        # Debug logging for duplicate detection
                        if timestamp and ('08:57:46' in timestamp or '08:29:19' in timestamp or '08:21:01' in timestamp):
                            print(f"CCR DEBUG: Row {row}, Timestamp: {timestamp}")
                            print(f"CCR DEBUG: Setting: '{setting_name}', Source: '{setting_source}'")
                            print(f"CCR DEBUG: Old: '{setting_old_value}', New: '{setting_new_value}'")
                            print(f"CCR DEBUG: Category: '{setting_category}'")
                            print(f"CCR DEBUG: Final details_str: '{setting_category}: {setting_name} changed from {setting_old_value} to {setting_new_value}'")
                            print("---")
                        if setting_name == 'Input Format Order':
                            # Trim long format lists for display
                            if setting_old_value and len(setting_old_value) > 40:
                                setting_old_trimmed = setting_old_value[:37] + "..."
                            else:
                                setting_old_trimmed = setting_old_value

                            if setting_new_value and len(setting_new_value) > 40:
                                setting_new_trimmed = setting_new_value[:37] + "..."
                            else:
                                setting_new_trimmed = setting_new_value

                            if setting_old_value and setting_old_value != 'None':
                                details_str = f"{setting_category}: {setting_name} changed from {setting_old_trimmed} to {setting_new_trimmed}"
                            else:
                                details_str = f"{setting_category}: {setting_name} set to {setting_new_trimmed}"
                        else:
                            if setting_old_value and setting_old_value != 'None':
                                details_str = f"{setting_category}: {setting_name} changed from {setting_old_value} to {setting_new_value}"
                            else:
                                details_str = f"{setting_category}: {setting_name} set to {setting_new_value}"
                    else:
                        # For other event types
                        details_str = str(name) if name else str(current_details)
                else:
                    # If details is already a string, use as is
                    details_str = str(current_details)

                # Create details cell with tooltip
                details_item = QTableWidgetItem(details_str)
                if isinstance(current_details, dict):
                    tooltip_lines = []
                    for k, v in current_details.items():
                        tooltip_lines.append(f"{k}: {v}")
                    tooltip = "\n".join(tooltip_lines)
                    details_item.setToolTip(tooltip)

                self.history_table.setItem(row, 2, details_item)

                # Debug: verify what was actually set in the table
                if timestamp and ('08:57:46' in timestamp or '08:29:19' in timestamp or '08:21:01' in timestamp):
                    stored_item = self.history_table.item(row, 2)
                    stored_text = stored_item.text() if stored_item else 'None'
                    print(f"CCR DEBUG: STORED in table row {row}: '{stored_text}'")
                    print(f"CCR DEBUG: Expected: '{details_str}'")
                    print(f"CCR DEBUG: Match: {stored_text == details_str}")
                    print("---")

            # Apply alternating row colors and other formatting
            self.history_table.setAlternatingRowColors(True)
            self.history_table.setSelectionBehavior(QAbstractItemView.SelectionBehavior.SelectRows)
            self.history_table.setEditTriggers(QAbstractItemView.EditTrigger.NoEditTriggers)

            # Adjust column widths
            self.history_table.resizeColumnsToContents()
            # Set Details column (index 2) to min 550px
            min_details_width = 550
            current_width = self.history_table.columnWidth(2)
            if current_width < min_details_width:
                self.history_table.setColumnWidth(2, min_details_width)

            # Add timestamp to default sort
            try:
                # Re-enable updates/signals and sorting now that population is complete
                self.history_table.setUpdatesDisabled(False)
                self.history_table.blockSignals(False)
                self.history_table.setSortingEnabled(True)
                if self.history_table.rowCount() > 0:
                    self.history_table.sortByColumn(0, Qt.SortOrder.DescendingOrder)
            except Exception:
                pass

        except Exception as e:
            pass
            # Add an error item to the table
            self.history_table.setRowCount(1)
            self.history_table.setItem(0, 0, QTableWidgetItem(datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')))
            self.history_table.setItem(0, 1, QTableWidgetItem("Error"))
            self.history_table.setItem(0, 2, QTableWidgetItem(f"Failed to load history: {str(e)}"))
            self.history_table.resizeColumnsToContents()

    def fix_malformed_history_entries(self):
        """Fix history entries that have missing plugin name/version information and correct event types"""
        try:
            history = HistoryLogger.get_history()
            if not history:
                return 0

            fixed_count = 0
            # Track indices of CCR install events so we can convert 2nd+ to 'reinstalled'
            ccr_install_indices = []

            for idx, entry in enumerate(history):
                event_type = entry.get('event_type', '')
                details = entry.get('details', {})

                if isinstance(details, dict):
                    name = details.get('name', '')
                    version = details.get('version', '')

                    # Detect whether this entry refers to CCR
                    details_str = str(details).lower()
                    path = details.get('path', '').lower()
                    is_ccr_entry = (
                        name == 'Calibre Config Reports' or
                        'calibre_config_reports' in details_str or 'calibre_config_reports' in path or
                        ' ccr' in details_str or 'config reports' in details_str
                    )

                    # Record CCR install entries regardless of being malformed or not
                    if is_ccr_entry and event_type == 'plugin_installed':
                        ccr_install_indices.append(idx)

                    # Fix entries that are missing name or have Unknown/missing version
                    # BUT: Don't overwrite version info for update/downgrade events as that destroys history
                    if (event_type in ['plugin_installed', 'plugin_reinstalled', 'plugin_updated', 'plugin_removed'] and
                        (not name or name.strip() == '' or version == 'Unknown' or not version)):

                        # If this is a CCR entry (based on clues), fill in name/version
                        if (is_ccr_entry or not name or name.strip() == ''):

                            details['name'] = 'Calibre Config Reports'
                            # Only fix version if it's truly missing/unknown AND this isn't an update/downgrade event
                            if (not version or version == 'Unknown') and event_type not in ['plugin_updated', 'plugin_downgraded']:
                                # Get current version dynamically (no hardcoded fallback)
                                try:
                                    from .settings_monitor import get_ccr_version
                                    current_version = get_ccr_version()
                                    if current_version:
                                        details['version'] = current_version
                                except Exception:
                                    pass

                            fixed_count += 1

            # Convert 2nd+ CCR 'installed' entries to 'reinstalled'
            if len(ccr_install_indices) > 1:
                for idx in ccr_install_indices[1:]:
                    entry = history[idx]
                    details = entry.get('details', {}) if isinstance(entry.get('details', {}), dict) else {}
                    entry['event_type'] = 'plugin_reinstalled'
                    # Ensure details carries a friendly label
                    details['event_type'] = 'Plugin reinstalled'
                    entry['details'] = details
                    fixed_count += 1

            if fixed_count > 0:
                # Save the fixed history
                with HistoryLogger._lock:
                    with open(HISTORY_FILE, 'w', encoding='utf-8') as f:
                        json.dump(history, f, indent=2)

            return fixed_count

        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"CCR: Error fixing malformed history entries: {str(e)}")
            return 0

    def setup_help_tab(self):
        tab = QWidget()
        layout = QHBoxLayout(tab)  # Change to horizontal layout for columns

        # ===== COLUMN 1: Download Links =====
        downloads_column = QWidget()
        downloads_layout = QVBoxLayout(downloads_column)

        # Header
        download_title = QLabel(_('<b>Calibre download links</b>'))
        download_title.setAlignment(Qt.AlignmentFlag.AlignHCenter)
        download_title.setStyleSheet('font-size: 13pt;')
        downloads_layout.addWidget(download_title)

        # Helper to wire QLabel links to calibre.open_url so tweaks are respected
        def _wire_label(label):
            try:
                # Ensure Qt does not open links directly
                label.setOpenExternalLinks(False)
            except Exception:
                pass
            try:
                label.linkActivated.connect(lambda u: __import__('calibre').gui2.open_url(u))
            except Exception:
                try:
                    # Fallback to calling open_url when activated
                    label.linkActivated.connect(lambda u: from_calibre_open(u))
                except Exception:
                    pass

        def from_calibre_open(u):
            try:
                from calibre.gui2 import open_url
                open_url(u)
            except Exception:
                pass

        # Windows download
        win_container = QWidget()
        win_layout = QHBoxLayout(win_container)
        win_layout.setContentsMargins(0, 4, 0, 4)  # Reduce vertical margins
        win_img = QLabel()
        win_icon = common_icons.get_icon('images/windows_icon')
        if win_icon and not win_icon.isNull():
            win_img.setPixmap(win_icon.pixmap(50, 50))
        win_layout.addWidget(win_img)
        win_link = QLabel('<a href="https://calibre-ebook.com/download_windows">Download for Windows</a>')
        _wire_label(win_link)
        win_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        win_link.setStyleSheet('font-size: 11pt;')
        win_layout.addWidget(win_link)
        win_layout.addStretch()
        downloads_layout.addWidget(win_container)

        # macOS download
        mac_container = QWidget()
        mac_layout = QHBoxLayout(mac_container)
        mac_layout.setContentsMargins(0, 4, 0, 4)  # Reduce vertical margins
        mac_img = QLabel()
        mac_icon = common_icons.get_icon('images/macos_icon')
        if mac_icon and not mac_icon.isNull():
            mac_img.setPixmap(mac_icon.pixmap(50, 50))
        mac_layout.addWidget(mac_img)
        mac_link = QLabel('<a href="https://calibre-ebook.com/download_osx">Download for macOS</a>')
        _wire_label(mac_link)
        mac_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        mac_link.setStyleSheet('font-size: 11pt;')
        mac_layout.addWidget(mac_link)
        mac_layout.addStretch()
        downloads_layout.addWidget(mac_container)

        # Linux download
        linux_container = QWidget()
        linux_layout = QHBoxLayout(linux_container)
        linux_layout.setContentsMargins(0, 4, 0, 4)  # Reduce vertical margins
        linux_img = QLabel()
        linux_icon = common_icons.get_icon('images/linux_icon')
        if linux_icon and not linux_icon.isNull():
            linux_img.setPixmap(linux_icon.pixmap(50, 50))
        linux_layout.addWidget(linux_img)
        linux_link = QLabel('<a href="https://calibre-ebook.com/download_linux">Download for Linux</a>')
        _wire_label(linux_link)
        linux_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        linux_link.setStyleSheet('font-size: 11pt;')
        linux_layout.addWidget(linux_link)
        linux_layout.addStretch()
        downloads_layout.addWidget(linux_container)

        # Portable version
        portable_container = QWidget()
        portable_layout = QHBoxLayout(portable_container)
        portable_layout.setContentsMargins(0, 4, 0, 4)  # Reduce vertical margins
        portable_img = QLabel()
        portable_icon = common_icons.get_icon('images/portable_icon')
        if portable_icon and not portable_icon.isNull():
            portable_img.setPixmap(portable_icon.pixmap(50, 50))
        portable_layout.addWidget(portable_img)
        portable_link = QLabel('<a href="https://calibre-ebook.com/download_portable">Portable Version</a>')
        _wire_label(portable_link)
        portable_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        portable_link.setStyleSheet('font-size: 11pt;')
        portable_layout.addWidget(portable_link)
        portable_layout.addStretch()
        downloads_layout.addWidget(portable_container)

        # News link
        news_container = QWidget()
        news_layout = QHBoxLayout(news_container)
        news_layout.setContentsMargins(0, 4, 0, 4)
        news_img = QLabel()
        news_icon = common_icons.get_icon('news.png')
        if news_icon and not news_icon.isNull():
            news_img.setPixmap(news_icon.pixmap(50, 50))
        news_layout.addWidget(news_img)
        news_link = QLabel("<a href='https://calibre-ebook.com/whats-new'>What's new</a>")
        _wire_label(news_link)
        news_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        news_link.setStyleSheet('font-size: 11pt;')
        news_layout.addWidget(news_link)
        news_layout.addStretch()
        downloads_layout.addWidget(news_container)

        # Preview download
        preview_container = QWidget()
        preview_layout = QHBoxLayout(preview_container)
        preview_layout.setContentsMargins(0, 4, 0, 4)  # Reduce vertical margins
        preview_img = QLabel()
        preview_icon = common_icons.get_icon('images/preview_icon')
        if preview_icon and not preview_icon.isNull():
            preview_img.setPixmap(preview_icon.pixmap(50, 50))
        preview_layout.addWidget(preview_img)
        preview_link = QLabel('<a href="https://download.calibre-ebook.com/preview/">Download preview release</a>')
        _wire_label(preview_link)
        preview_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        preview_link.setStyleSheet('font-size: 11pt;')
        preview_layout.addWidget(preview_link)
        preview_layout.addStretch()
        downloads_layout.addWidget(preview_container)

        downloads_layout.addStretch()
        layout.addWidget(downloads_column)

        # ===== COLUMN 2: Manual Links =====
        manuals_column = QWidget()
        manuals_layout = QVBoxLayout(manuals_column)

        # Header
        manual_title = QLabel(_('<b>Calibre User Manual & Help</b>'))
        manual_title.setAlignment(Qt.AlignmentFlag.AlignHCenter)
        manual_title.setStyleSheet('font-size: 13pt;')
        manuals_layout.addWidget(manual_title)

        # Links
        help_links = [
            (_('Getting Started'), 'https://manual.calibre-ebook.com/intro.html'),
            (_('Search & Sort'), 'https://manual.calibre-ebook.com/gui.html#search-sort'),
            (_('Basic templates'), 'https://manual.calibre-ebook.com/template_lang.html#basic-templates'),
            (_('Adding your favorite news website'), 'https://manual.calibre-ebook.com/news.html'),
            (_('E-book viewer'), 'https://manual.calibre-ebook.com/viewer.html'),
            (_('E-book conversion'), 'https://manual.calibre-ebook.com/conversion.html'),
            (_('Editing e-books'), 'https://manual.calibre-ebook.com/edit.html'),
            (_('Comparing e-books'), 'https://manual.calibre-ebook.com/diff.html'),
            (_('Content server'), 'https://manual.calibre-ebook.com/server.html'),
            (_('Frequently Asked Questions'), 'https://manual.calibre-ebook.com/faq.html'),
            (_('Tutorials'), 'https://manual.calibre-ebook.com/tutorials.html'),
            (_('Regular expressions'), 'https://manual.calibre-ebook.com/regexp.html'),
            (_('Customizing calibre'), 'https://manual.calibre-ebook.com/customize.html'),
            (_('Command Line Interface'), 'https://manual.calibre-ebook.com/generated/en/cli-index.html'),
        ]
        for title, url in help_links:
            link = QLabel(f'<a href="{url}">{title}</a>')
            _wire_label(link)
            link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
            link.setAlignment(Qt.AlignmentFlag.AlignLeft)
            link.setStyleSheet('font-size: 11pt;')
            manuals_layout.addWidget(link)

        manuals_layout.addStretch()
        layout.addWidget(manuals_column)

        # ===== COLUMN 3: About Plugin =====
        about_column = QWidget()
        about_layout = QVBoxLayout(about_column)

        # Header
        about_title = QLabel(_('<b>About this plugin</b>'))
        about_title.setAlignment(Qt.AlignmentFlag.AlignHCenter)
        about_title.setStyleSheet('font-size: 13pt;')
        about_layout.addWidget(about_title)

        # Plugin description
        desc_text = _("Calibre Config Reports provides an easy way to view and export detailed information about your Calibre setup, including plugins, custom columns, shortcuts and key settings.")
        desc_label = QLabel(desc_text)
        desc_label.setWordWrap(True)
        desc_label.setStyleSheet('font-size: 11pt;')
        about_layout.addWidget(desc_label)

        # Handy shortcuts section removed; CCR actions are user-assignable and unassigned by default

        # Support section
        support_title = QLabel(_("<b>Support & Resources</b>"))
        support_title.setStyleSheet('font-size: 12pt; margin-top: 15px;')
        about_layout.addWidget(support_title)

        # MobileRead forum link
        mr_container = QWidget()
        mr_layout = QHBoxLayout(mr_container)
        mr_img = QLabel()
        mr_icon = common_icons.get_icon('images/mobileread')
        if mr_icon and not mr_icon.isNull():
            mr_img.setPixmap(mr_icon.pixmap(24, 24))
        else:
            mr_img.setText("🔗")
        mr_layout.addWidget(mr_img)
        mr_link = QLabel('<a href="https://www.mobileread.com/forums/showthread.php?t=369555">MobileRead Forum Thread</a>')
        _wire_label(mr_link)
        mr_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        mr_link.setStyleSheet('font-size: 11pt;')
        mr_layout.addWidget(mr_link)
        mr_layout.addStretch()
        about_layout.addWidget(mr_container)

        # Donate link
        donate_container = QWidget()
        donate_layout = QHBoxLayout(donate_container)
        donate_img = QLabel()
        donate_icon = common_icons.get_icon('images/donate')
        if donate_icon and not donate_icon.isNull():
            donate_img.setPixmap(donate_icon.pixmap(24, 24))
        else:
            donate_img.setText("❤️")
        donate_layout.addWidget(donate_img)
        donate_link = QLabel('<a href="https://ko-fi.com/comfy_n">Support Development</a>')
        _wire_label(donate_link)
        donate_link.setTextInteractionFlags(Qt.TextInteractionFlag.TextBrowserInteraction)
        donate_link.setStyleSheet('font-size: 11pt;')
        donate_layout.addWidget(donate_link)
        donate_layout.addStretch()
        about_layout.addWidget(donate_container)

        about_layout.addStretch()
        layout.addWidget(about_column)

        return tab

    def showEvent(self, event):
        """Overridden show event to ensure proper shortcut activation"""
        super().showEvent(event)
        # Use a timer to ensure shortcuts are activated after dialog is fully shown
        QTimer.singleShot(50, self._activate_shortcuts)

        # Add reliable application-scoped shortcuts as fallback that directly click the buttons
        self._install_direct_application_shortcuts()
    # (Focus workaround removed) - avoid focus/debug helpers that caused side-effects
        # Delay heavy table population slightly to allow the window to paint
        try:
            QTimer.singleShot(150, self._delayed_populate)
        except Exception:
            pass
        # Deferred initial tab loading to avoid UI freeze - give a short delay so initial paint occurs
        try:
            QTimer.singleShot(150, lambda: self._lazy_load_tab(self.tabs.currentIndex()))
        except Exception:
            pass

    # Focus debug helpers removed to simplify behavior and avoid side-effects when showing dialog.

    def _lazy_load_tab(self, index):
        """Lazy-populate the given tab index the first time it is shown.

        Accepts either an integer index (from QTabWidget.currentChanged) or
        no-op values. After populating a tab once, subsequent calls are ignored.
        """
        try:
            # Handle being called with no/None index
            if index is None:
                try:
                    index = self.tabs.currentIndex()
                except Exception:
                    return
            # Ensure we have a set to track populated tabs
            if not hasattr(self, '_populated_tabs'):
                try:
                    self._populated_tabs = set()
                except Exception:
                    return
            # If already populated, skip
            if index in self._populated_tabs:
                return

            # Map tab index to populate action (Plugins=0, Custom Columns=1, Shortcuts=2, Settings=3, History=4)
            if index == 0:
                # Plugins tab
                try:
                    if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'populate_plugins_table'):
                        self.plugins_tab.populate_plugins_table()
                except Exception:
                    pass
            elif index == 1:
                try:
                    self.populate_columns_table()
                except Exception:
                    pass
            elif index == 2:
                try:
                    self.populate_shortcuts_tab()
                except Exception:
                    pass
            elif index == 3:
                try:
                    # Settings tab may be a SettingsTab instance stored as tabs.widget(3)
                    settings_widget = self.tabs.widget(3)
                    if settings_widget and hasattr(settings_widget, 'populate_settings_table'):
                        settings_widget.populate_settings_table()
                    else:
                        self.populate_settings_table()
                except Exception:
                    pass
            elif index == 4:
                try:
                    self.populate_history_table()
                except Exception:
                    pass

            # Mark as populated regardless of errors to avoid repeated attempts
            try:
                self._populated_tabs.add(index)
            except Exception:
                pass
        except Exception:
            # Swallow to avoid breaking tab changes
            pass
    def _delayed_populate(self):
        """Populate heavy tables after initial UI render."""
        # Populate heavy tables after initial paint. Do NOT disable dialog-wide
        # updates here — that can prevent the window from painting and cause a
        # blank/white frame. The plugins table itself yields to the event loop
        # intermittently during population for responsiveness.
        # Ensure any pending paint events are processed so the dialog appears
        # before heavy insertion begins.
        try:
            from qt.core import QApplication
            QApplication.processEvents()
        except Exception:
            pass
        try:
            if hasattr(self, 'plugins_tab') and hasattr(self.plugins_tab, 'populate_plugins_table'):
                self.plugins_tab.populate_plugins_table()
        except Exception:
            pass
        try:
            # Populate other tables (columns, shortcuts, settings)
            self.populate_tables()
        except Exception:
            pass

    def _activate_shortcuts(self):
        """Activate QAction shortcuts after dialog is fully shown"""
        try:
            if hasattr(self, 'plugins_tab'):
                plugins_tab = self.plugins_tab

                # Re-register all QAction shortcuts to ensure they work on first run
                actions_to_register = [
                    ('prefs_action', 'Alt+P'),
                    ('updater_action', 'Alt+U'),
                    ('load_action', 'Alt+L'),
                    ('csv_action', 'Alt+S'),
                    ('xlsx_action', 'Alt+X'),
                    ('restore_action', 'Alt+R')
                ]

                for action_name, shortcut_key in actions_to_register:
                    if hasattr(plugins_tab, action_name):
                        action = getattr(plugins_tab, action_name)
                        # Remove and re-add to ensure proper registration
                        try:
                            self.removeAction(action)
                        except:
                            pass
                        self.addAction(action)

                        if CCR_DEBUG_ENABLED:
                            prints(f"[CCR][DEBUG] Re-registered {action_name} for {shortcut_key}")

                if CCR_DEBUG_ENABLED:
                    prints("[CCR][DEBUG] All QAction shortcuts re-registered in _activate_shortcuts")

        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Error in _activate_shortcuts: {e}")

    def _install_direct_application_shortcuts(self):
        """Create direct application-scoped shortcuts that bypass QAction issues"""
        try:
            if not hasattr(self, 'plugins_tab'):
                return

            # Store shortcut references to prevent garbage collection
            if not hasattr(self, '_app_shortcuts'):
                self._app_shortcuts = []

            # Create ApplicationShortcut for Alt+P (Plugin Preferences)
            sc_p = QShortcut(QKeySequence('Alt+P'), self)
            sc_p.setContext(Qt.ApplicationShortcut)
            sc_p.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+P shortcut triggered") if CCR_DEBUG_ENABLED else None) or
                self.plugins_tab.prefs_button.animateClick(100))
            self._app_shortcuts.append(sc_p)

            # Create ApplicationShortcut for Alt+U (Plugin Updater)
            sc_u = QShortcut(QKeySequence('Alt+U'), self)
            sc_u.setContext(Qt.ApplicationShortcut)
            sc_u.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+U shortcut triggered") if CCR_DEBUG_ENABLED else None) or
                self.plugins_tab.updater_button.animateClick(100))
            self._app_shortcuts.append(sc_u)

            # Create ApplicationShortcut for Alt+L (Load Plugin)
            sc_l = QShortcut(QKeySequence('Alt+L'), self)
            sc_l.setContext(Qt.ApplicationShortcut)
            sc_l.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+L shortcut triggered") if CCR_DEBUG_ENABLED else None) or
                self.plugins_tab.load_plugin_btn.animateClick(100))
            self._app_shortcuts.append(sc_l)

            # Create ApplicationShortcut for Alt+S (Save CSV)
            sc_s = QShortcut(QKeySequence('Alt+S'), self)
            sc_s.setContext(Qt.ApplicationShortcut)
            sc_s.activated.connect(lambda: self.export_data('csv'))
            self._app_shortcuts.append(sc_s)

            # Create ApplicationShortcut for Alt+X (Export XLSX)
            sc_x = QShortcut(QKeySequence('Alt+X'), self)
            sc_x.setContext(Qt.ApplicationShortcut)
            sc_x.activated.connect(lambda: self.export_data('xlsx'))
            self._app_shortcuts.append(sc_x)

            # Create ApplicationShortcut for Alt+R (Reset Columns)
            sc_r = QShortcut(QKeySequence('Alt+R'), self)
            sc_r.setContext(Qt.ApplicationShortcut)
            sc_r.activated.connect(lambda: (prints("[CCR][DEBUG] Alt+R shortcut triggered") if CCR_DEBUG_ENABLED else None) or
                self.plugins_tab.restore_default_columns())
            self._app_shortcuts.append(sc_r)

            if CCR_DEBUG_ENABLED:
                prints("[CCR][DEBUG] Added direct ApplicationShortcuts for all Alt keys")

        except Exception as e:
            if CCR_DEBUG_ENABLED:
                prints(f"[CCR][DEBUG] Error adding direct ApplicationShortcuts: {e}")

