"""Date utilities for the Calibre Config Reports plugin."""

import os
import platform
from datetime import datetime
from pathlib import Path

def get_file_times(file_path):
    """Get creation and modification times for a file.

    Args:
        file_path: Path to the file

    Returns:
        tuple: (created_time, modified_time) as datetime objects
    """
    path = Path(file_path)
    stats = path.stat()

    # Get creation time (platform specific)
    if platform.system() == 'Windows':
        created = datetime.fromtimestamp(stats.st_ctime)
    else:
        # On Unix systems, try to get birthtime, fallback to ctime
        try:
            created = datetime.fromtimestamp(stats.st_birthtime)
        except AttributeError:
            created = datetime.fromtimestamp(stats.st_ctime)

    # Get modification time
    modified = datetime.fromtimestamp(stats.st_mtime)

    return created, modified

def format_datetime(dt):
    """Format a datetime object for display.

    Args:
        dt: datetime object

    Returns:
        str: Formatted date string
    """
    if dt:
        return dt.strftime('%Y-%m-%d %H:%M')
    return ''
