#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

try:
    from qt.core import (
        QMenu, QApplication, QDialog, QVBoxLayout, QLabel,
        QDialogButtonBox, QPushButton, QClipboard, Qt, QTextEdit,
        QUrl
    )
    # No audio/multimedia imports needed
except ImportError:
    # Full PyQt5 fallback
    from PyQt5.Qt import (
        QMenu, QApplication, QDialog, QVBoxLayout, QLabel,
        QDialogButtonBox, QPushButton, QClipboard, Qt, QTextEdit,
        QUrl
    )
    # No audio/multimedia imports needed

from calibre.gui2 import Application
from calibre.gui2.actions import InterfaceAction
from calibre.utils.localization import _
from . import common_icons

class PluginCatalogAction(InterfaceAction):

    name = 'Calibre Config Reports'
    # better not translate plugin name
    action_spec = ('Calibre Config Reports', None,
                  _('View detailed information about plugins and settings'), 'None')

    def genesis(self):
        # Plugin hooks disabled - causes conflicts with main monitoring
        # try:
        #     from .plugin_install_hook import install_plugin_hooks
        #     install_plugin_hooks()
        # except Exception as e:
        #     from calibre.utils.logging import prints
        #     prints(f"[CCR][DEBUG] Failed to install plugin hooks: {e}")

        common_icons.set_plugin_icon_resources(self.name, {})
        self.rebuild_icon()
        self.qaction.triggered.connect(self.show_dialog)

        app = Application.instance()
        if app:
            app.palette_changed.connect(self.rebuild_icon)

    def rebuild_icon(self):
        """Update the icon based on current theme"""
        icon = common_icons.get_icon('images/iconplugin.png')
        if icon and not icon.isNull():
            self.qaction.setIcon(icon)

    def show_dialog(self):
        from calibre_plugins.calibre_config_reports.main import PluginCatalogDialog
        d = PluginCatalogDialog(self.gui)
        d.exec()

class PluginInfoDialog(QDialog):
    """Dialog to show copyable plugin information"""
    def __init__(self, parent, plugin_name, plugin_info):
        QDialog.__init__(self, parent)
        self.setWindowTitle(_('Plugin Info: {}').format(plugin_name))
        self.setMinimumWidth(450)

        layout = QVBoxLayout(self)

        # Create a label with the plugin info
        self.info_label = QLabel(plugin_info)
        self.info_label.setTextInteractionFlags(
            self.info_label.textInteractionFlags() |
            Qt.TextInteractionFlag.TextSelectableByMouse
        )
        self.info_label.setWordWrap(True)
        layout.addWidget(self.info_label)

        # Add buttons
        button_box = QDialogButtonBox()

        # Copy button
        copy_button = QPushButton(_('Copy to Clipboard'))
        copy_button.clicked.connect(self.copy_to_clipboard)
        button_box.addButton(copy_button, QDialogButtonBox.ButtonRole.ActionRole)

        # Close button
        button_box.addButton(QDialogButtonBox.StandardButton.Close)
        button_box.rejected.connect(self.reject)

        layout.addWidget(button_box)

    def copy_to_clipboard(self):
        """Copy the plugin info to clipboard"""
        clipboard = QApplication.clipboard()
        clipboard.setText(self.info_label.text())

class ColumnInfoDialog(QDialog):
    def __init__(self, parent, column_name, info_text, template=None):
        super().__init__(parent)
        self.setWindowTitle(f'Column Information - {column_name}')
        self.resize(600, 400)

        layout = QVBoxLayout(self)

        # Add information text
        heading = QLabel("Column settings")
        heading.setStyleSheet("font-weight: bold; font-size: 12pt; margin-bottom: 10px;")
        layout.addWidget(heading)

        # Help text
        help_text = QLabel("Edit custom columns in:\nPreferences > Interface > Add your own columns")
        help_text.setStyleSheet("margin-bottom: 10px;")
        layout.addWidget(help_text)

        # Show template if available
        if template:
            template_label = QLabel("Template:")
            template_label.setStyleSheet("font-weight: bold; margin-top: 10px;")
            layout.addWidget(template_label)

            template_text = QTextEdit()
            template_text.setPlainText(template)
            template_text.setReadOnly(True)
            template_text.setMaximumHeight(100)
            layout.addWidget(template_text)

        # Add text area with the info
        settings_label = QLabel("Display Settings:")
        settings_label.setStyleSheet("font-weight: bold; margin-top: 10px;")
        layout.addWidget(settings_label)

        self.text_area = QTextEdit()
        self.text_area.setPlainText(info_text)
        self.text_area.setReadOnly(True)
        layout.addWidget(self.text_area)

        # Add Copy button and Close button
        button_box = QDialogButtonBox()
        copy_button = QPushButton("Copy")
        copy_button.clicked.connect(self.copy_to_clipboard)
        button_box.addButton(copy_button, QDialogButtonBox.ButtonRole.ActionRole)
        button_box.addButton(QDialogButtonBox.StandardButton.Close)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

    def copy_to_clipboard(self):
        """Copy the text to clipboard"""
        text = self.text_area.toPlainText()
        QApplication.clipboard().setText(text)
