#!/usr/bin/env python
# -*- coding: utf-8 -*-

__license__ = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

import os
from calibre.customize import InterfaceActionBase
from calibre.utils.localization import _
from calibre.gui2 import gprefs

class PluginCatalogPlugin(InterfaceActionBase):
    name = 'Calibre Config Reports' # plugin name better not translate
    description = _(
        'A comprehensive configuration reporting tool for Calibre that provides detailed information about your setup. '
        'It displays installed plugins, custom columns, keyboard shortcuts, and Calibre settings in an organized, searchable interface. '
        'The plugin offers powerful filtering capabilities, export options, and the ability to copy configuration details '
        'for troubleshooting or sharing.'
        )
    supported_platforms = ['windows', 'osx', 'linux']
    author = 'Comfy.n'
    version = (2, 0, 9)
    minimum_calibre_version = (6, 25, 0)

    actual_plugin = 'calibre_plugins.calibre_config_reports.action:PluginCatalogAction'

    def is_customizable(self):
        return False

    def initialize(self):
        """Setup plugin icon resources"""
        # First call the parent's initialize method to ensure proper toolbar integration
        result = super().initialize()

        from calibre.utils.zipfile import ZipFile
        from calibre.utils.config import config_dir
        from calibre_plugins.calibre_config_reports.common_icons import set_plugin_icon_resources
        import os

        try:
            # Setup plugin config directory
            plugin_config_dir = os.path.join(config_dir, 'plugins', 'calibre_config_reports')
            os.makedirs(plugin_config_dir, exist_ok=True)

            with ZipFile(self.plugin_path, 'r') as zf:
                # Handle icons
                resources = {}
                icon_paths = [
                    'images/iconplugin_dark.png',
                    'images/iconplugin_light.png',
                    'images/mobileread_dark.png',
                    'images/mobileread_light.png',
                    'images/donate.png',
                    'images/windows_icon.png',
                    'images/linux_icon.png',
                    'images/preview_icon.png',
                    'images/news_icon.png',
                    'images/portable_icon.png',
                    'images/macos_icon.png',
                    'images/plus_icon.png',
                    'images/minus_icon.png',
                    'images/plugins_prefs_icon.png',
                    'images/plugin_updater_icon.png'
                ]
                for icon_path in icon_paths:
                    try:
                        with zf.open(icon_path, 'r') as f:
                            resources[icon_path] = f.read()
                    except:
                        pass
                set_plugin_icon_resources(self.name, resources)

        except Exception:
            pass

        # Return the result from the parent's initialize method
        return result
