#!./luajit
-- kindleVNCviewer (smoothed e-ink updates)

local ffi         = require("ffi")
local blitbuffer  = require("ffi/blitbuffer")
local fb          = require("ffi/framebuffer").open("/dev/fb0")
local evloop      = require("ffi/eventloop")
local input       = require("ffi/input")
local posix       = require("ffi/posix_h")
local rfb         = require("ffi/rfbclient")

-- globals used by config.lua
local password        = nil
local client          = nil
local rfbFramebuffer  = nil
local configfile      = "config.lua"

-- base behavior
local waitRefresh     = 220     -- debounce before driving EPDC (ms)
local rotateFB        = 0
local reconnecting    = false
local debug           = false
local blitfunc        = nil

-- bbox accumulation for refresh
local update_x1       = nil
local update_x2       = 0
local update_y1       = 0
local update_y2       = 0

-- optional “pixel budget” full refresh (disabled by default here)
local refresh_full_every_256th_pxup = 0
local refresh_full_ctr = 0

-- IDs / constants
local TIMER_REFRESH = 10

-- Kindle EPDC waveform constants
local WAVEFORM_MODE_INIT      = 0x0
local WAVEFORM_MODE_DU        = 0x1
local WAVEFORM_MODE_GC16      = 0x2
local WAVEFORM_MODE_GC4       = WAVEFORM_MODE_GC16
local WAVEFORM_MODE_GC16_FAST = 0x3
local WAVEFORM_MODE_A2        = 0x4
local WAVEFORM_MODE_GL16      = 0x5
local WAVEFORM_MODE_GL16_FAST = 0x6
local WAVEFORM_MODE_AUTO      = 0x101

-- default waveforms if we do a “full” vs “partial” refresh
local waveform_default_fast = WAVEFORM_MODE_GC16
local waveform_default_slow = WAVEFORM_MODE_GC16

----------------------------------------------------------------
-- Tuning knobs to smooth flicker on e-ink
----------------------------------------------------------------
local MIN_BOX_EDGE        = 24       -- skip super tiny rects (cursor spam)
local QUANT               = 16       -- quantize bbox to 16-px grid
local FAST_AREA_THRESH    = 220*220  -- below → use A2; above → GL16
local FULL_REFRESH_EVERY_SEC = 30    -- periodic full refresh to kill ghosting
local last_full_refresh_ts = 0

----------------------------------------------------------------
-- tiny helper to (re)request updates from server
----------------------------------------------------------------
local function request_update(c, incremental, x, y, w, h)
  if not c then return end
  local ok = pcall(function()
    if rfb.SendFramebufferUpdateRequest then
      -- rfb.SendFramebufferUpdateRequest(client, inc, x, y, w, h)
      rfb.SendFramebufferUpdateRequest(c, incremental and 1 or 0,
        x or 0, y or 0, w or c.width, h or c.height)
    end
  end)
  if debug and not ok then
    io.stderr:write("WARN: SendFramebufferUpdateRequest not available\n")
  end
end

----------------------------------------------------------------
-- Public API (used by config.lua)
----------------------------------------------------------------
function SendKeyEvent(key, pressed)
  rfb.SendKeyEvent(client, key, pressed)
end
function SendPointerEvent(x, y, buttonMask)
  rfb.SendPointerEvent(client, x, y, buttonMask)
end
function Quit()
  os.exit(0)
end

----------------------------------------------------------------
-- Optional pixel-budget full refresh (disabled unless you enable the var)
----------------------------------------------------------------
local function do_refresh_full(w, h)
  if refresh_full_every_256th_pxup == 0 then
    return false
  end
  refresh_full_ctr = refresh_full_ctr + w*h
  local budget = bit.rshift(fb.bb:getWidth() * fb.bb:getHeight() * refresh_full_every_256th_pxup, 8)
  if refresh_full_ctr >= budget then
    refresh_full_ctr = 0
    return true
  end
  return false
end

----------------------------------------------------------------
-- Coalesced refresh timer → draws bbox & drives EPDC
----------------------------------------------------------------
local function refreshTimerFunc()
  if not update_x1 then return end

  local x = update_x1
  local y = update_y1
  local w = update_x2 - update_x1
  local h = update_y2 - update_y1
  update_x1 = nil

  if w <= 0 or h <= 0 then return end

  if debug then
    io.stdout:write("eink update ", x, ",", y, " ", w, "x", h, "\n")
  end

  -- copy pixels into local blitbuffer
  fb.bb:blitFrom(rfbFramebuffer, x, y, x, y, w, h, blitfunc)

  -- time-based full refresh to reduce ghosting
  local now = os.time()
  local do_full_time = (FULL_REFRESH_EVERY_SEC > 0) and (now - last_full_refresh_ts >= FULL_REFRESH_EVERY_SEC)

  if do_full_time then
    last_full_refresh_ts = now
    if debug then io.stdout:write("time-based full eink refresh\n") end
    fb:refresh(1, waveform_default_slow)
  elseif do_refresh_full(w, h) then
    if debug then io.stdout:write("pixel-budget full eink refresh\n") end
    fb:refresh(1, waveform_default_slow)
  else
    local area = w * h
    if area <= FAST_AREA_THRESH then
      -- small areas: A2 is quickest, minimal flashing
      if debug then io.stdout:write("fast eink refresh (A2)\n") end
      fb:refresh(0, WAVEFORM_MODE_A2, x, y, w, h)
    else
      -- medium/large areas: GL16 is cleaner than GC16 on partials
      if debug then io.stdout:write("medium eink refresh (GL16)\n") end
      fb:refresh(0, WAVEFORM_MODE_GL16, x, y, w, h)
    end
  end

  -- keep the pipeline flowing
  request_update(client, true, 0, 0, client and client.width, client and client.height)

  -- harmless EPDC nudge (sometimes helps surface pending refreshes)
  os.execute("eips -s >/dev/null 2>&1")
end

----------------------------------------------------------------
-- Called by libvncclient when new pixels arrive
----------------------------------------------------------------
local function updateFromRFB(_, x, y, w, h)
  -- ignore super-tiny rectangles (cursor spam)
  if w < MIN_BOX_EDGE and h < MIN_BOX_EDGE then
    if debug then io.stdout:write("skip tiny ", x, ",", y, " ", w, "x", h, "\n") end
    return
  end

  -- expand bbox
  if not update_x1 then
    update_x1 = x;           update_y1 = y
    update_x2 = x + w;       update_y2 = y + h
  else
    if update_x1 > x       then update_x1 = x       end
    if update_y1 > y       then update_y1 = y       end
    if update_x2 < x + w   then update_x2 = x + w   end
    if update_y2 < y + h   then update_y2 = y + h   end
  end

  -- quantize bbox to a grid to stabilize refresh area
  local function qdown(v, q) return math.floor(v / q) * q end
  local function qup(v, q)   return math.floor((v + q - 1) / q) * q end
  local maxW, maxH = fb.bb:getWidth(), fb.bb:getHeight()
  update_x1 = math.max(0, qdown(update_x1, QUANT))
  update_y1 = math.max(0, qdown(update_y1, QUANT))
  update_x2 = math.min(maxW, qup(update_x2, QUANT))
  update_y2 = math.min(maxH, qup(update_y2, QUANT))

  if debug then
    io.stdout:write("RFB update ", x, ",", y, " ", w, "x", h,
      "  -> bbox ", update_x1, ",", update_y1, " ",
      (update_x2 - update_x1), "x", (update_y2 - update_y1), "\n")
  end

  if not evloop.timer_running(TIMER_REFRESH) then
    evloop.register_timer_in_ms(waitRefresh, refreshTimerFunc, TIMER_REFRESH)
  end
end

----------------------------------------------------------------
-- Password callback for libvncclient
----------------------------------------------------------------
local function passwordCallback(_)
  if password then
    return ffi.C.strndup(ffi.cast("char*", password), 8192)
  end
  io.stderr:write("Password requested but none configured.\n")
  return nil
end

----------------------------------------------------------------
-- Connect to VNC server and prepare local framebuffer
----------------------------------------------------------------
local function connect()
  -- Ask libvncclient for 8 bits/sample, 3 samples, 4 bytes/pixel (32bpp)
  local c = rfb.rfbGetClient(8, 3, 4)

  c.GetPassword           = passwordCallback
  c.canHandleNewFBSize    = 0
  c.GotFrameBufferUpdate  = updateFromRFB

  -- pass CLI args to libvncclient (encodings etc.)
  local argc = ffi.new("int[1]")
  argc[0] = #arg + 1
  local argv = ffi.new("char*[?]", #arg + 1)
  argv[0] = ffi.cast("char*", "kindlevncviewer")
  for k, v in ipairs(arg) do
    argv[k] = ffi.cast("char*", v)
  end

  assert(rfb.rfbInitClient(c, argc, argv) ~= 0, "cannot initialize client")

  if debug then
    io.stdout:write(string.format(
      "Client pixfmt: bpp=%d depth=%d bigendian=%d truecolour(nonzero)=%d rshift=%d gshift=%d bshift=%d\n",
      c.format.bitsPerPixel, c.format.depth, c.format.bigEndian,
      c.format.trueColour, c.format.redShift, c.format.greenShift, c.format.blueShift))
  end

  -- expose width/height as globals used by config.lua
  client_width  = c.width
  client_height = c.height

  -- wrap the libvncclient framebuffer as a BlitBuffer
  rfbFramebuffer = blitbuffer.new(
    c.width, c.height,
    blitbuffer.TYPE_BBBGR32,
    c.frameBuffer
  )
  rfbFramebuffer:invert()

  -- initial full update
  request_update(c, false, 0, 0, c.width, c.height)

  return c
end

----------------------------------------------------------------
-- CLI / usage
----------------------------------------------------------------
local function usage()
  io.stderr:write([[
kVNCViewer (smoothed)
A VNC viewer for e-ink devices  —  GPLv2
Usage:
  ./luajit vncviewer_smooth.lua [options...] <server>:<display>

Helpful options:
  -password <pwd>
  -shared -encodings raw
  -debug
]])
  os.exit(1)
end

local function try_open_input(device)
  local ok = pcall(input.open, device)
  if not ok then
    io.stderr:write("could not open input device ", device, " (possibly harmless)\n")
  end
end

-- parse CLI like original
if #arg == 0 then usage() end
for i, value in ipairs(arg) do
  if value == "-h" or value == "-?" or value == "--help" then usage() end
  if value == "-password" and arg[i+1] then
    password = arg[i+1]; arg[i+1] = ""
  elseif value == "-config" and arg[i+1] then
    configfile = arg[i+1]; arg[i+1] = ""
  elseif value == "-rotateFB" and arg[i+1] then
    rotateFB = tonumber(arg[i+1]); arg[i+1] = ""
  elseif value == "-waitRefresh" and arg[i+1] then
    waitRefresh = tonumber(arg[i+1]); arg[i+1] = ""
  elseif value == "-refreshFullAfterPixels" and arg[i+1] then
    refresh_full_every_256th_pxup = 256 * tonumber(arg[i+1]); arg[i+1] = ""
  elseif value == "-fast" then
    waveform_default_fast = WAVEFORM_MODE_A2
    waveform_default_slow = WAVEFORM_MODE_GL16
  elseif value == "-medium" then
    waveform_default_fast = WAVEFORM_MODE_GL16
    waveform_default_slow = WAVEFORM_MODE_GL16
  elseif value == "-dither_bw" then
    blitfunc = fb.bb.setPixelDithered
  elseif value == "-debug" then
    debug = true
  elseif value == "-reconnecting" then
    reconnecting = true
  elseif value == "-version" then
    io.stdout:write("KindleVNCviewer (smoothed) — see https://github.com/hwhw/kindlevncviewer\n")
    os.exit(0)
  end
end

-- config & setup
dofile(configfile)
fb.bb:rotate(rotateFB)

try_open_input("/dev/input/event0")
try_open_input("/dev/input/event1")
try_open_input("/dev/input/event2")

repeat
  client = connect()

  local running = true

  -- drive libvncclient socket
  evloop.register_fd(client.sock, {
    read = function()
      assert(rfb.HandleRFBServerMessage(client) ~= 0, "Error handling RFB server message.")
      -- keep asking for updates even if server is quiet
      request_update(client, true, 0, 0, client.width, client.height)
    end,
    err = function()
      if not reconnecting then
        io.stderr:write("connection error, quitting.\n"); os.exit(1)
      else
        io.stderr:write("connection error, retrying in 1s...\n"); running = false
      end
    end,
    hup = function()
      if not reconnecting then
        io.stderr:write("remote party hung up, quitting.\n"); os.exit(1)
      else
        io.stderr:write("remote party hung up, retrying in 1s...\n"); running = false
      end
    end
  })

  while running do
    local event = input.waitForEvent()
    handleInput(0, event.type, event.code, event.value)
  end

  ffi.C.sleep(1)
until not reconnecting

