#!./luajit
local ffi = require("ffi")
local blitbuffer = require("ffi/blitbuffer")
local fb = require("ffi/framebuffer").open("/dev/fb0")
local evloop = require("ffi/eventloop")
local input = require("ffi/input")
local posix = require("ffi/posix_h")
local rfb = require("ffi/rfbclient")

local password, client, rfbFramebuffer = nil, nil, nil
local configfile = "config.lua"

local waitRefresh = 150
local rotateFB = 0
local reconnecting = false
local debug = false
local blitfunc = nil

local update_x1 = nil
local update_x2, update_y1, update_y2 = 0,0,0
local refresh_full_every_256th_pxup, refresh_full_ctr = 512, 0
local TIMER_REFRESH = 10

-- Eink waveforms
local WAVEFORM_MODE_GC16 = 0x2
local WAVEFORM_MODE_A2   = 0x4
local WAVEFORM_MODE_GL16 = 0x5
local waveform_default_fast = WAVEFORM_MODE_GC16
local waveform_default_slow = WAVEFORM_MODE_GC16

-- Request update helper (handles symbol name variants)
local function request_update(c, incremental, x, y, w, h)
  if not c then return end
  x = x or 0; y = y or 0
  w = w or c.width; h = h or c.height
  local inc = incremental and 1 or 0
  local function try(tbl, name)
    local f = tbl and tbl[name]
    if type(f) == "cdata" or type(f) == "function" then
      f(c, x, y, w, h, inc); return true
    end
    return false
  end
  local ok = try(rfb,"SendFramebufferUpdateRequest")
         or try(rfb,"rfbSendFramebufferUpdateRequest")
         or try(ffi.C,"SendFramebufferUpdateRequest")
         or try(ffi.C,"rfbSendFramebufferUpdateRequest")
  if debug and not ok then
    io.stdout:write("No *FramebufferUpdateRequest* symbol; relying on libvncclient auto-requests.\n")
  end
end

-- public API (for config.lua)
function SendKeyEvent(key, pressed) rfb.SendKeyEvent(client, key, pressed) end
function SendPointerEvent(x, y, buttonMask) rfb.SendPointerEvent(client, x, y, buttonMask) end
function Quit() os.exit(0) end

local function do_refresh_full(w, h)
  if refresh_full_every_256th_pxup == 0 then return false end
  refresh_full_ctr = refresh_full_ctr + w*h
  if refresh_full_ctr >= bit.rshift(fb.bb:getWidth() * fb.bb:getHeight() * refresh_full_every_256th_pxup, 8) then
    refresh_full_ctr = 0; return true
  end
  return false
end

local function refreshTimerFunc()
  if not update_x1 then return end
  local x, y = update_x1, update_y1
  local w, h = update_x2 - update_x1, update_y2 - update_y1
  if w < 1 or h < 1 then update_x1 = nil; return end

  if debug then io.stdout:write("eink update ", x, ",", y, " ", w, "x", h, "\n") end

  fb.bb:blitFrom(rfbFramebuffer, x, y, x, y, w, h, blitfunc)

  if do_refresh_full(w, h) then
    if debug then io.stdout:write("slow eink refresh\n") end
    fb:refresh(1, waveform_default_slow)
  else
    if debug then io.stdout:write("fast eink refresh\n") end
    fb:refresh(0, waveform_default_fast, x, y, w, h)
  end

  -- Kindle EPDC sometimes needs a nudge; harmless if redundant
  os.execute("eips -s >/dev/null 2>&1")

  update_x1 = nil
  -- keep updates flowing
  request_update(client, true, 0, 0, client and client.width, client and client.height)
end

local function updateFromRFB(c, x, y, w, h)
  if debug then io.stdout:write("RFB update ", x, ",", y, " ", w, "x", h, "\n") end
  if not update_x1 then
    update_x1, update_y1 = x, y
    update_x2, update_y2 = x+w, y+h
  else
    if update_x1 > x     then update_x1 = x     end
    if update_x2 < x + w then update_x2 = x + w end
    if update_y1 > y     then update_y1 = y     end
    if update_y2 < y + h then update_y2 = y + h end
  end
  if not evloop.timer_running(TIMER_REFRESH) then
    evloop.register_timer_in_ms(waitRefresh, refreshTimerFunc, TIMER_REFRESH)
  end
end

local function passwordCallback(_c)
  if password then return ffi.C.strndup(ffi.cast("char*", password), 8192) end
  io.stderr:write("got request for password, but no password was configured.\n")
  return nil
end

local function connect()
  local c = rfb.rfbGetClient(8,3,4) -- want 32bpp (24 depth) from libvncclient
  c.GetPassword = passwordCallback
  c.canHandleNewFBSize = 0
  c.GotFrameBufferUpdate = updateFromRFB

  local argc = ffi.new("int[1]"); argc[0] = #arg + 1
  local argv = ffi.new("char*[?]", #arg+1)
  argv[0] = ffi.cast("char*", "kindlevncviewer")
  for k, v in ipairs(arg) do argv[k] = ffi.cast("char*", v) end

  assert(rfb.rfbInitClient(c, argc, argv) ~= 0, "cannot initialize client")

  if debug and c.format then
    local f = c.format
    io.stdout:write(string.format(
      "Client pixfmt: bpp=%d depth=%d bigendian=%d rshift=%d gshift=%d bshift=%d\n",
      f.bitsPerPixel, f.depth, f.bigEndian, f.redShift, f.greenShift, f.blueShift))
  end

  rfbFramebuffer = blitbuffer.new(c.width, c.height, blitbuffer.TYPE_BBBGR32, c.frameBuffer)
  rfbFramebuffer:invert()

  -- full update once to kickstart things
  request_update(c, false, 0, 0, c.width, c.height)

  -- and a first full EPDC refresh so first frame shows
  fb:refresh(1, waveform_default_slow)
  os.execute("eips -s >/dev/null 2>&1")

  return c
end

local function usage()
  io.stderr:write([[kVNCViewer (GPLv2)

Usage:
  luajit kindlevncviewer.lua [options...] <server>:<display>

Options: -password <pw>  -config <file>  -rotateFB <deg>  -waitRefresh <ms>
         -refreshFullAfterPixels <n>  -dither_bw  -medium  -fast
         -debug  -reconnecting
]])
  os.exit(1)
end

local function try_open_input(dev)
  local ok = pcall(input.open, dev)
  if not ok and debug then io.stderr:write("could not open input ", dev, " (possibly harmless)\n") end
end

-- Touch → pointer bridge
local EV_SYN, EV_KEY, EV_ABS = 0, 1, 3
local SYN_REPORT = 0
local BTN_TOUCH, BTN_LEFT = 330, 272
local ABS_X, ABS_Y = 0, 1
local ABS_MT_SLOT, ABS_MT_POSITION_X, ABS_MT_POSITION_Y, ABS_MT_TRACKING_ID = 0x2f, 0x35, 0x36, 0x39
local mt_slot, mt_down, mt_x, mt_y = 0, false, 0, 0
local st_x, st_y, st_down = 0, 0, false
local FB_W, FB_H = fb.bb:getWidth(), fb.bb:getHeight()

local function send_pointer(x, y, down)
  if not client then return end
  local w = client.width or FB_W
  local h = client.height or FB_H
  local sx = (w > 0 and FB_W > 0) and (w / FB_W) or 1.0
  local sy = (h > 0 and FB_H > 0) and (h / FB_H) or 1.0
  local rx = math.floor(x * sx + 0.5)
  local ry = math.floor(y * sy + 0.5)
  rx = math.max(0, math.min(w-1, rx))
  ry = math.max(0, math.min(h-1, ry))
  if debug then io.stdout:write(string.format("Pointer %d @ %d,%d (src %d,%d)\n", down and 1 or 0, rx, ry, x, y)) end
  rfb.SendPointerEvent(client, rx, ry, down and 1 or 0)
  request_update(client, true, 0, 0, w, h)
end

local function handle_touch(type_, code, value)
  if type_ == EV_ABS then
    if     code == ABS_X                then st_x = value
    elseif code == ABS_Y                then st_y = value
    elseif code == ABS_MT_SLOT          then mt_slot = value
    elseif code == ABS_MT_TRACKING_ID   then mt_down = (value ~= -1)
    elseif code == ABS_MT_POSITION_X    then mt_x = value
    elseif code == ABS_MT_POSITION_Y    then mt_y = value
    end
  elseif type_ == EV_KEY then
    if code == BTN_TOUCH or code == BTN_LEFT then st_down = (value ~= 0) end
  elseif type_ == EV_SYN and code == SYN_REPORT then
    if mt_x ~= 0 or mt_y ~= 0 then send_pointer(mt_x, mt_y, mt_down)
    else                         send_pointer(st_x, st_y, st_down) end
  end
end

-- args
if #arg == 0 then usage() end
for i,v in ipairs(arg) do
  if v == "-h" or v == "-?" or v == "--help" then usage() end
  if v == "-password" and arg[i+1] then password = arg[i+1]; arg[i+1] = ""
  elseif v == "-config" and arg[i+1] then configfile = arg[i+1]; arg[i+1] = ""
  elseif v == "-rotateFB" and arg[i+1] then rotateFB = tonumber(arg[i+1]); arg[i+1] = ""
  elseif v == "-waitRefresh" and arg[i+1] then waitRefresh = tonumber(arg[i+1]); arg[i+1] = ""
  elseif v == "-refreshFullAfterPixels" and arg[i+1] then refresh_full_every_256th_pxup = 256 * tonumber(arg[i+1]); arg[i+1] = ""
  elseif v == "-fast" then waveform_default_fast = WAVEFORM_MODE_A2; waveform_default_slow = WAVEFORM_MODE_GL16
  elseif v == "-medium" then waveform_default_fast = WAVEFORM_MODE_GL16; waveform_default_slow = WAVEFORM_MODE_GL16
  elseif v == "-dither_bw" then blitfunc = fb.bb.setPixelDithered
  elseif v == "-debug" then debug = true
  elseif v == "-reconnecting" then reconnecting = true
  elseif v == "-version" then
    io.stdout:write("KindleVNCviewer version ", require("version"), "\n",
      "see http://github.com/hwhw/kindlevncviewer for source code\n")
    os.exit(0)
  end
end

-- config brings handleInput()
dofile(configfile)

fb.bb:rotate(rotateFB)
try_open_input("/dev/input/event0"); try_open_input("/dev/input/event1"); try_open_input("/dev/input/event2")

-- Wrap config handleInput to add touch bridge + debug print
local cfg_handleInput = _G.handleInput
_G.handleInput = function(dev, type_, code, value)
  if debug then io.stdout:write(string.format("EV dev=%d type=%d code=%d val=%d\n", dev or -1, type_ or -1, code or -1, value or -1)) end
  handle_touch(type_, code, value)
  if cfg_handleInput then return cfg_handleInput(dev, type_, code, value) end
end

repeat
  client = connect()

  local running = true
  evloop.register_fd(client.sock, {
    read = function()
      assert(rfb.HandleRFBServerMessage(client) ~= 0, "Error handling RFB server message.")
    end,
    err = function()
      if not reconnecting then io.stderr:write("connection error, quitting.\n"); os.exit(1)
      else io.stderr:write("connection error, retrying in 1s...\n"); running = false end
    end,
    hup = function()
      if not reconnecting then io.stderr:write("remote party hung up, quitting.\n"); os.exit(1)
      else io.stderr:write("remote party hung up, retrying in 1s...\n"); running = false end
    end
  })

  while running do
    local event = input.waitForEvent()
    handleInput(0, event.type, event.code, event.value)
  end

  ffi.C.sleep(1)
until not reconnecting

