-- fb_probe.lua — robust solid fills + checkerboard for Kindle fb0
local ffi = require("ffi")

local function sys_read(path)
  local f = io.open(path, "r")
  if not f then return nil end
  local s = f:read("*a")
  f:close()
  if s then s = s:gsub("%s+$","") end
  return s
end

local function fbset_geom()
  local f = io.popen("fbset -s 2>/dev/null"); if not f then return end
  local s = f:read("*a") or ""; f:close()
  local W,H,VW,VH,BPP = s:match("geometry%s+(%d+)%s+(%d+)%s+(%d+)%s+(%d+)%s+(%d+)")
  if W then return tonumber(W), tonumber(H), tonumber(VW), tonumber(VH), tonumber(BPP) end
end

local function sysfs_geom()
  local vs = sys_read("/sys/class/graphics/fb0/virtual_size")    -- e.g., "1088,6144"
  local rs = sys_read("/sys/class/graphics/fb0/bits_per_pixel")   -- e.g., "8"
  local xs = sys_read("/sys/class/graphics/fb0/stride")           -- some kernels expose this
  local VW, VH = nil, nil
  if vs then VW, VH = vs:match("(%d+),(%d+)"); VW = tonumber(VW); VH = tonumber(VH) end
  local BPP = rs and tonumber(rs) or nil
  local STRIDE = xs and tonumber(xs) or nil -- bytes per row if present
  return VW, VH, BPP, STRIDE
end

local fb  = require("ffi/framebuffer").open("/dev/fb0")
local bb  = fb.bb

-- safe method getter
local function get_m(bb, name)
  local m = bb[name]
  if type(m) == "function" then
    return m(bb)
  end
end

-- safe field getter (won't throw on missing FFI field)
local function get_f(bb, name)
  local ok, v = pcall(function() return bb[name] end)
  return ok and v or nil
end

-- 1) Width/Height from methods or fbset
local W = get_m(bb, "getWidth")
local H = get_m(bb, "getHeight")
if not (W and H) then
  local fw, fh = fbset_geom()
  W, H = fw or W, fh or H
end
assert(W and H, "Unable to determine framebuffer width/height")

-- 2) Stride (bytes per row)
-- Prefer a real byte stride if exposed, else derive: (virtual_width * bpp/8)
local stride_bytes = get_f(bb, "stride") or get_f(bb, "pitch")
if not stride_bytes then
  local VW, _, BPP, STRIDE = sysfs_geom()
  if STRIDE then
    stride_bytes = STRIDE
  else
    local _,_,VW2,_,BPP2 = fbset_geom() or {}
    VW = VW or VW2 or W
    BPP = BPP or BPP2 or 8
    stride_bytes = math.floor(VW * (BPP/8))
  end
end
assert(stride_bytes and stride_bytes > 0, "Unable to determine stride (bytes/row)")

-- 3) Raw data pointer
local data_ptr = get_f(bb, "data")
assert(data_ptr, "bb.data pointer not exposed")
data_ptr = ffi.cast("uint8_t*", data_ptr)

print(string.format("fb: W=%d H=%d stride=%d bytes/bpp=%s",
  W, H, stride_bytes, sys_read("/sys/class/graphics/fb0/bits_per_pixel") or "?"))

local function fill_byte(val)
  for y = 0, H - 1 do
    local row = data_ptr + y * stride_bytes
    ffi.fill(row, W, val) -- W bytes because bpp=8
  end
end

-- Waveforms (same constants kVNC uses)
local WAVEFORM_MODE_GC16 = 0x2

-- 1) Black
fill_byte(0x00)
fb:refresh(1, WAVEFORM_MODE_GC16)
os.execute("sleep 1")

-- 2) White
fill_byte(0xFF)
fb:refresh(1, WAVEFORM_MODE_GC16)
os.execute("sleep 1")

-- 3) Checkerboard (32px tiles)
local TILE_SHIFT = 5
for y = 0, H - 1 do
  local row = data_ptr + y * stride_bytes
  local yb  = bit.rshift(y, TILE_SHIFT)
  for x = 0, W - 1 do
    local xb = bit.rshift(x, TILE_SHIFT)
    row[x] = (((xb + yb) % 2) == 0) and 0x00 or 0xFF
  end
end
fb:refresh(1, WAVEFORM_MODE_GC16)
print("DONE")

