__license__ = "GPL v3"
__copyright__ = "2025, Hugo Batista <intellireading at hugobatista.com>"
__docformat__ = "markdown en"

# this file is a common file to all IntelliReading metaguiding plugins.
# !!!! Do not edit this file out of common folder !!!!
# pylint: disable=import-error
import sys
import time
from functools import partial
import traceback
import os


from calibre import prints
from calibre.constants import (
    preferred_encoding,
)
from calibre.utils.logging import (
    ANSIStream,
)
from polyglot.builtins import is_py3
from polyglot.io import PolyglotStringIO

if is_py3:
    from typing import List


class Logger:
    LEVELS = {"DEBUG": 0, "INFO": 1, "WARN": 2, "ERROR": 3}

    def __init__(self) -> None:
        self.log_level = "INFO"
        if "CALIBRE_DEVELOP_FROM" in os.environ or "CALIBRE_DEBUG" in os.environ or "calibre-debug" in sys.argv[0]:
            self.log_level = "DEBUG"

        # According to Kovid, calibre always uses UTF-8 for the Python 3 version
        self.preferred_encoding = "UTF-8" if is_py3 else preferred_encoding
        self.outputs = [ANSIStream()]

        self.debug = partial(self.print_formatted_log, "DEBUG")
        self.info = partial(self.print_formatted_log, "INFO")
        self.warn = self.warning = partial(self.print_formatted_log, "WARN")
        self.error = partial(self.print_formatted_log, "ERROR")

    def __call__(self, logmsg) -> None:
        self.info(logmsg)

    def _tag_args(self, level, *args) -> List[str]:
        _now = time.localtime()
        _buf = PolyglotStringIO()
        _tagged_args = []
        for _arg in args:
            prints(time.strftime("%Y-%m-%d %H:%M:%S", _now), file=_buf, end=" ")
            _buf.write("[")
            prints(level, file=_buf, end="")
            _buf.write("] ")
            prints(_arg, file=_buf, end="")

            _tagged_args.append(_buf.getvalue())
            _buf.truncate(0)

        return _tagged_args

    def _prints(self, level: str, *args, **kwargs) -> None:
        for _o in self.outputs:
            _o.prints(self.LEVELS[level], *args, **kwargs)
            if hasattr(_o, "flush"):
                _o.flush()

    def print_formatted_log(self, level: str, *args, **kwargs) -> None:
        _tagged_args = self._tag_args(level, *args)
        self._prints(level, *_tagged_args, **kwargs)

    def exception(self, *args, **kwargs) -> None:
        _limit = kwargs.pop("limit", None)
        _tagged_args = self._tag_args("ERROR", *args)
        self._prints("ERROR", *_tagged_args, **kwargs)
        self._prints("ERROR", traceback.format_exc(_limit))


log = Logger()


def open_donation_link():
    """Open the PayPal donation link in the default browser."""
    from calibre.gui2 import open_url

    try:
        open_url("https://go.hugobatista.com/donate-intellireading")
        log.info("Opened donation link in browser")
    except Exception as e:
        log.error(f"Error opening donation link: {e}")


def show_donate_message(title, message, skip_dialog_name):
    """Show donate message for first-time users."""
    try:
        from calibre.gui2 import question_dialog
        from calibre.gui2.ui import get_gui

        gui = get_gui()
        # Only show GUI dialog if GUI is available
        if gui is not None:
            log.info("Showing donate message")

            # Show question dialog with donation option
            show_donation = question_dialog(
                gui,
                title,
                message,
                skip_dialog_name=skip_dialog_name,
                default_yes=False,  # False makes "Maybe Later" the default
                skip_dialog_skipped_value=False,
                yes_icon=None,
                no_icon=None,
                yes_text="☕ Donate",
                no_text="Maybe Later",
            )

            if show_donation:
                open_donation_link()

    except Exception as e:
        log.error(f"Error showing donate message: {e}")
