import csv
from PyQt6.QtWidgets import QFileDialog, QMessageBox
from calibre.gui2.actions import InterfaceAction

class AuthorsTaggerPluginUI(InterfaceAction):
    name = 'Authors Tagger Plugin'
    action_spec = ('Authors Tagger', 'images/icon.png', 'Add tags from CSV to authors', None)

    def genesis(self):
        icon = get_icons('images/icon.png', 'Authors Tagger Plugin')
        self.qaction.setIcon(icon)
        self.qaction.triggered.connect(self.run_author_tagging)

    def run_author_tagging(self):
        file_path, _ = QFileDialog.getOpenFileName(self.gui, 'Select CSV file', '', 'CSV Files (*.csv)')
        if not file_path:
            return

        author_tags = {}
        try:
            with open(file_path, newline='', encoding='utf-8') as csvfile:
                reader = csv.DictReader(csvfile)
                for row in reader:
                    author = row.get('author', '').strip()
                    tags_cell = row.get('tags', '').strip()
                    if author:
                        # Split tags on commas and remove extra whitespace
                        tags = {tag.strip() for tag in tags_cell.split(',')} if tags_cell else set()
                        if author in author_tags:
                            author_tags[author].update(tags)
                        else:
                            author_tags[author] = tags
        except Exception as e:
            QMessageBox.critical(self.gui, 'Error', f'Failed to read CSV file:\n{e}')
            return

        if not author_tags:
            QMessageBox.information(self.gui, 'No Data', 'No authors/tags found in CSV file.')
            return

        db = self.gui.current_db.new_api
        updated_books = {}

        # For each author, search books and prepare updates
        for author, tags_to_add in author_tags.items():
            search_query = f'authors:"{author}"'
            try:
                matching_book_ids = list(db.search(search_query))
            except Exception as e:
                QMessageBox.warning(self.gui, 'Search Error', f'Error searching for author "{author}": {e}')
                continue

            for book_id in matching_book_ids:
                mi = db.get_metadata(book_id, get_cover=False)
                current_tags = set(mi.tags) if mi.tags else set()
                new_tags = current_tags | tags_to_add
                if new_tags != current_tags:
                    mi.tags = list(new_tags)
                    updated_books[book_id] = mi

        # Apply metadata updates in batch
        for book_id, mi in updated_books.items():
            self.gui.current_db.set_metadata(book_id, mi)

        QMessageBox.information(
            self.gui,
            'Authors Tagger',
            f'Updated tags in {len(updated_books)} book(s).'
        )
