#!/usr/bin/env python
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)
import six

__license__   = 'GPL v3'
__copyright__ = '2011, Rodrigo Coin Curvo. 2019-2022, Thiago Oliveira'
__docformat__ = 'restructuredtext en'

# PyQt libraries
try:
    from qt.core import QLabel, QGridLayout, Qt, QGroupBox, QCheckBox, QSpinBox
except ImportError:
    from PyQt5.Qt import QLabel, QGridLayout, Qt, QGroupBox, QCheckBox, QSpinBox

# Calibre libraries
from calibre.gui2.metadata.config import ConfigWidget as DefaultConfigWidget
from calibre.utils.config import JSONConfig

# Load translation files (.mo) on the folder 'translations'
load_translations()

STORE_NAME = 'Options'
KEY_MAX_PAGES = 'maxPages'
KEY_MAX_DOWNLOADS = 'maxDownloads'
KEY_THRESHOLD = 'threshold'
KEY_TRY_EXCHANGING = 'tryExchanging'

DEFAULT_STORE_VALUES = {
    KEY_MAX_PAGES: 1,
    KEY_MAX_DOWNLOADS: 1,
    KEY_THRESHOLD: 0,
    KEY_TRY_EXCHANGING: True
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Skoob')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES


def get_option(option_key):
    default_value = DEFAULT_STORE_VALUES[option_key]
    return plugin_prefs[STORE_NAME].get(option_key, default_value)


class ConfigWidget(DefaultConfigWidget):

    def __init__(self, plugin):
        DefaultConfigWidget.__init__(self, plugin)
        self.fields_view.setMinimumHeight(170)
        self.l.setRowStretch(0, 1)
        c = plugin_prefs[STORE_NAME]

        other_group_box = QGroupBox(_('Other options'), self)
        self.l.addWidget(other_group_box, self.l.rowCount(), 0, 1, 2)
        other_group_box_layout = QGridLayout()
        other_group_box.setLayout(other_group_box_layout)

        max_pages_label = QLabel(_('Maximum search result pages to analyse (1 = fastest):'), self)
        max_pages_label.setToolTip(_('Usually one page is enough, but\n'
                                     'if the search is too broad, the best\n'
                                     'result may not be in the first page.\n'))
        other_group_box_layout.addWidget(max_pages_label, 0, 0, 1, 1)
        self.max_pages_spin = QSpinBox(self)
        self.max_pages_spin.setMinimum(1)
        self.max_pages_spin.setMaximum(5)
        self.max_pages_spin.setProperty('value', c.get(KEY_MAX_PAGES, DEFAULT_STORE_VALUES[KEY_MAX_PAGES]))
        other_group_box_layout.addWidget(self.max_pages_spin, 0, 1, 1, 1)
        other_group_box_layout.setColumnStretch(2, 1)

        max_downloads_label = QLabel(_('Maximum title/author search matches to download/evaluate (1 = fastest):'), self)
        max_downloads_label.setToolTip(_('More matches means higher chance of better\n'
                                         'metadata (but not necessarily).\n'))
        other_group_box_layout.addWidget(max_downloads_label, 1, 0, 1, 1)
        self.max_downloads_spin = QSpinBox(self)
        self.max_downloads_spin.setMinimum(1)
        self.max_downloads_spin.setMaximum(5)
        self.max_downloads_spin.setProperty('value', c.get(KEY_MAX_DOWNLOADS, DEFAULT_STORE_VALUES[KEY_MAX_DOWNLOADS]))
        other_group_box_layout.addWidget(self.max_downloads_spin, 1, 1, 1, 1)
        other_group_box_layout.setColumnStretch(2, 1)

        threshold_label = QLabel(_('[Experimental] Minimum similarity threshold (%):'), self)
        threshold_label.setToolTip(_('Try to eliminate results that are not\n'
                                     'very similar to the search based on a\n'
                                     'composition of Title and Author similarities.\n'
                                     'This is not quite precise, but 20-30 seems\n'
                                     'to be a good guess. Default is no discarding (0).\n'))
        other_group_box_layout.addWidget(threshold_label, 2, 0, 1, 1)
        self.threshold_spin = QSpinBox(self)
        self.threshold_spin.setMinimum(0)
        self.threshold_spin.setMaximum(100)
        self.threshold_spin.setProperty('value', c.get(KEY_THRESHOLD, DEFAULT_STORE_VALUES[KEY_THRESHOLD]))
        other_group_box_layout.addWidget(self.threshold_spin, 2, 1, 1, 1)
        other_group_box_layout.setColumnStretch(2, 1)

        self.try_exchanging_checkbox = QCheckBox(_('Try exchanging title and authors if nothing found'), self)
        self.try_exchanging_checkbox.setToolTip(
                _('If no result is found, try searching again with\n'
                  'title and authors exchanged.\n'))
        self.try_exchanging_checkbox.setChecked(c.get(KEY_TRY_EXCHANGING, DEFAULT_STORE_VALUES[KEY_TRY_EXCHANGING]))
        other_group_box_layout.addWidget(self.try_exchanging_checkbox, 3, 0, 1, 3)

    def commit(self):
        DefaultConfigWidget.commit(self)
        new_prefs = {}
        new_prefs[KEY_MAX_PAGES] = int(six.text_type(self.max_pages_spin.value()))
        new_prefs[KEY_MAX_DOWNLOADS] = int(six.text_type(self.max_downloads_spin.value()))
        new_prefs[KEY_THRESHOLD] = int(six.text_type(self.threshold_spin.value()))
        new_prefs[KEY_TRY_EXCHANGING] = self.try_exchanging_checkbox.checkState() == Qt.Checked
        plugin_prefs[STORE_NAME] = new_prefs

