# -*- coding: utf-8 -*-
"""AutoMultiConvert — configurable converter with silent/sync toggle (v1.0.0)

Options
-------
✓ Target format list (default: EPUB,KFX)
✓ Delete original if not target (default: ON)
✓ Process silently (run conversions after import so Add‑Books dialog closes) — default: OFF

Tested on calibre 5.0 → 8.5
"""

import threading, logging, traceback
from calibre.customize import FileTypePlugin
from calibre.ebooks.conversion.plumber import Plumber
from calibre.ptempfile import PersistentTemporaryFile
from calibre.utils.logging import Log
from calibre.utils.config import JSONConfig

prefs = JSONConfig('plugins/auto_multi_convert')
prefs.setdefault('targets', 'EPUB,KFX')
prefs.setdefault('delete_original', True)
prefs.setdefault('silent', False)          # False = synchronous

def targets():
    return [t.strip().upper() for t in prefs['targets'].split(',') if t.strip()]

class AutoMultiConvert(FileTypePlugin):
    name = 'Auto Multi-Convert'
    description = 'Convert imports to chosen formats (sync or silent)'
    version = (1, 0, 0)
    author = 'Jack'
    minimum_calibre_version = (5, 0, 0)
    supported_platforms = ['windows', 'osx', 'linux']
    on_postimport = True
    file_types = {'*'}

    # ---------- UI ----------
    def is_customizable(self):
        return True

    def config_widget(self):
        from PyQt5.Qt import QWidget, QVBoxLayout, QLabel, QLineEdit, QCheckBox
        w = QWidget()
        lay = QVBoxLayout(w)
        lay.addWidget(QLabel('Comma-separated target formats (e.g. EPUB,KFX,AZW3):'))
        self.le = QLineEdit(prefs['targets'], w)
        lay.addWidget(self.le)
        self.cb_del = QCheckBox('Delete original when it is NOT one of the targets', w)
        self.cb_del.setChecked(prefs['delete_original'])
        lay.addWidget(self.cb_del)
        self.cb_silent = QCheckBox('Process silently (background thread)', w)
        self.cb_silent.setChecked(prefs['silent'])
        lay.addWidget(self.cb_silent)
        return w

    def save_settings(self, widget):
        prefs['targets'] = self.le.text().strip().upper() or 'EPUB,KFX'
        prefs['delete_original'] = bool(self.cb_del.isChecked())
        prefs['silent'] = bool(self.cb_silent.isChecked())

    # ---------- import hook ----------
    def postimport(self, book_id, book_format, db):
        if prefs['silent']:
            threading.Thread(target=self._convert_worker,
                             args=(book_id, book_format, db),
                             daemon=True).start()
        else:
            self._convert_worker(book_id, book_format, db)

    # ---------- core logic ----------
    def _convert_worker(self, book_id, book_format, db):
        try:
            src_fmt = book_format.upper()
            fmt_str = db.formats(book_id, index_is_id=True) or ''
            existing = {f.upper() for f in fmt_str.split(',') if f}
            missing = [t for t in targets() if t not in existing]

            if not missing and src_fmt in targets():
                return

            src_path = db.format_abspath(book_id, book_format, index_is_id=True)

            for tgt in missing:
                try:
                    with PersistentTemporaryFile(suffix='.' + tgt.lower()) as tmp:
                        Plumber(src_path, tmp.name, Log()).run()
                        db.add_format(book_id, tgt, tmp.name, index_is_id=True)
                        logging.debug('AMConvert: %s → %s OK (ID %s)', src_fmt, tgt, book_id)
                except Exception as e:
                    logging.error('AMConvert: conversion %s→%s failed (ID %s): %s',
                                  src_fmt, tgt, book_id, e)
                    return

            if prefs['delete_original'] and src_fmt not in targets():
                db.remove_format(book_id, src_fmt, index_is_id=True, notify=True)
                logging.debug('AMConvert: removed original %s (ID %s)', src_fmt, book_id)

        except Exception:
            logging.error('AMConvert fatal:\n%s', traceback.format_exc())

Plugin = AutoMultiConvert
