#!/usr/bin/env python
# -*- coding: utf-8 -*-
import subprocess, os, os.path, sys, tempfile, shutil, imghdr, platform
from subprocess import Popen, PIPE
from os.path import expanduser, splitext, basename
from urllib.parse import quote

from epub_utils import unzip_epub_to_dir, epub_zip_up_book_contents
from sigil_bs4 import BeautifulSoup

# return safe id
def safe_id(href):
    ''' returns a safe id '''
    string_list = list(href)
    for letter in range(1, len(href)): 
        if not href[letter].isalnum() and href[letter] not in ['-', '_', '.']:
             string_list[letter] = '_'
    href = "".join(string_list)
    if not href[0].isalpha():
        href = 'x' + href 
    return href

# simple Java wrapper
def w2lWrapper(*args):
    ''' runs writer2latex '''
    process = Popen(list(args), stdout=PIPE, stderr=PIPE)
    ret = process.communicate()
    return ret

# main routine
def run(bk):

    # check for supported versions
    if bk.launcher_version() < 20160707:
        print("This plugin requires Sigil 0.9.8 or later.\n\nPlease click OK to close the Plugin Runner window.")
        return -1

    # Try to import PySide6, then PyQt5
    try:
        from PySide6.QtWidgets import QApplication, QFileDialog, QWidget
        from PySide6.QtGui import QClipboard, QIcon
    except ImportError:
        try:
            from PyQt5.QtWidgets import QApplication, QFileDialog, QWidget
            from PyQt5.QtGui import QClipboard, QIcon
        except ImportError:
            raise RuntimeError("No supported GUI framework found (PySide6 or PyQt5).")

    def get_file_path(initialdir):
        ''' displays a Qt file selection dialog '''
        app = QApplication.instance() or QApplication(sys.argv)
        w = QWidget() # dummy widget to prevent python task bar icon
        dialog = QFileDialog(w)
        dialog.setDirectory(initialdir)
        dialog.setWindowTitle("Select .odt file")
        #dialog.setFileMode(QFileDialog.ExistingFile) # one file only
        icon_path = os.path.join(os.path.dirname(__file__), "plugin.png")
        dialog.setWindowIcon(QIcon(icon_path))
        dialog.setNameFilter("Open Document Files (*.odt)")
        # prevent Python task bar icon
        # https://www.mobileread.com/forums/showpost.php?p=4516423&postcount=47
        dialog.setOptions(QFileDialog.DontUseNativeDialog) 

        if dialog.exec():
            fpath = dialog.selectedFiles()[0]
        else:
            # Cancel button clicked
            fpath = None
        app.quit()

        return fpath
        
    #------------------------
    # initialize variables
    #------------------------
    odt_file = None
    font_dir = None
    ibooks_xml = None
    add_resources = None
    w2l14_size = 558470

    # get the writer2latex Java binary and config file paths
    w2l_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'writer2xhtml.jar')
    w2l_config_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'config.xml')
    w2l_css_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'epub.css')
    ibooks_xml_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'com.apple.ibooks.display-options.xml')
    resource_dir = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'resources')
    
    # get document folder preference
    prefs = bk.getPrefs()
    if 'initialdir' in prefs:
        initialdir = prefs['initialdir']
    else:
        initialdir = expanduser('~')

    # get custom resource dir
    if 'resource_dir' in prefs and os.path.isdir(prefs['resource_dir']):
        resource_dir = prefs['resource_dir']
        
    # get preferred epub version
    if 'output_format' in prefs and prefs['output_format'] in ['epub', 'epub3']:
        
        # make sure that epub3 isn't used with w2l 1.14
        if prefs['output_format'] == 'epub3' and os.path.getsize(w2l_path) == w2l14_size:
            output_format = 'epub'
            prefs['output_format'] = output_format
            bk.savePrefs(prefs)
        else:
            output_format = prefs['output_format']
    else:
        # input container has no bk.epub_version() :(
        epubversion = bk._w.epub_version
        if epubversion.startswith("3"):
            output_format = 'epub3'
        else:
            output_format = 'epub'
        prefs['output_format'] = output_format

    # save prefs
    bk.savePrefs(prefs)
            
    if 'ibooks_xml' in prefs:
        ibooks_xml = prefs['ibooks_xml']

    if 'add_resources' in prefs:
        add_resources = prefs['add_resources']

    # get ODT file name
    odt_file = get_file_path(initialdir)
    if not odt_file:
        print('No .odt file selected.\n\nClick OK to close the Plugin Runner dialog.')
        return 0

    # save document folder as preference
    if not 'initialdir' in prefs and initialdir:
        prefs['initialdir'] = os.path.dirname(odt_file)
        bk.savePrefs(prefs)
        
    # process ODT file
    if odt_file:
        
        # assemble w2l command line
        args = ['java', '-jar', w2l_path, '-' + output_format]
        if os.path.isfile(w2l_config_path):
            args.append('-config=' + w2l_config_path)
        if os.path.isfile(w2l_css_path):
            args.append('-stylesheet=' + w2l_css_path)
        args.append(odt_file)
        
        # run w2l
        result = w2lWrapper(*args)
        epub_path = os.path.splitext(odt_file)[0] + '.epub'

        # display w2l messages
        print('writer2latex.jar command line:\n' + ' '.join(args))
        w2l_warnings = result[0].decode('utf-8', 'ignore').splitlines()
        print('\nwriter2latex.jar messages:')
        for line in w2l_warnings:
            print(line)

        if result[1]:
            java_warnings = result[1].decode('utf-8', 'ignore').splitlines()
            print('Writer2xhtml failed!\n\nJava error messages:')
            for line in java_warnings:
                print(line)
            return -1

        if os.path.isfile(epub_path):
        
            # check for resource files
            if os.path.isdir(resource_dir):
                resource_files = os.listdir(resource_dir)

            # optionally add ibooks xml file and/or fonts
            if ibooks_xml or resource_files:
                # unzip the epub to a temporary folder for further processing
                tempdir = tempfile.mkdtemp()
                unzip_epub_to_dir(epub_path, tempdir)
                os.remove(epub_path)

                # add ibooks xml file
                print('\nODTImport plugin messages:')
                if ibooks_xml and os.path.isfile(ibooks_xml_path):
                    shutil.copyfile(ibooks_xml_path, os.path.join(tempdir, 'META-INF', 'com.apple.ibooks.display-options.xml'))
                    print('\ncom.apple.ibooks.display-options.xml added.')
                else:
                    print('\nibooks_xml=false or com.apple.ibooks.display-options.xml missing!')

                # add resource files and update .opf file
                if resource_files and add_resources:

                    # read original .opf file
                    opf_path = os.path.join(tempdir, 'OEBPS','book.opf')
                    with open(opf_path, 'r', encoding='utf-8') as fp:
                        data = fp.read()
         
                    # parse book.opf with bs4 in xml mode
                    soup = BeautifulSoup(data, 'xml')
                    orig_soup = str(soup)

                    # get the names of all existing manifest items
                    item_list = []
                    manifest_items = soup.find_all('item')
                    for manifest_item in manifest_items:
                        item_list.append(basename(manifest_item['href']))

                    # process all files in the plugin resources folder
                    for resource_file in resource_files:
                        file_name, extension = splitext(resource_file)
                        
                        # limit resource files to known mime types and files that don't already exist in the epub
                        if extension in ['.ttf', '.otf', '.css', '.jpg', '.jpeg', '.png', '.gif', '.bmp', '.webp', '.tif', '.svg', '.xhtml'] and resource_file not in item_list:
                            
                            # define mimetype
                            if resource_file.endswith('tf'):
                                media_type = 'application/font-sfnt'
                            elif resource_file.endswith('.css'):
                                media_type = 'text/css'
                            elif resource_file.endswith('.xhtml'):
                                media_type = 'application/xhtml+xml'
                            elif resource_file.endswith('.svg'):
                                media_type = 'image/svg+xml'
                            else:
                                extension = imghdr.what(os.path.join(resource_dir, resource_file))
                                if extension:
                                    mime_type = 'image/{}.format(extension)'
                                else:
                                    # default to JPEG
                                    mime_type = 'image/jpeg'
                                    extension = '.jpg'
                            new_tag = soup.new_tag('item', href = quote(resource_file))
                            new_tag['id'] = safe_id(resource_file)
                            new_tag['media-type'] = media_type
                            soup.manifest.append(new_tag)
                            
                            # add .xhtml files at the end of the spine
                            if resource_file.endswith('.xhtml'):
                                new_tag = soup.new_tag('itemref', idref = safe_id(resource_file))
                                soup.spine.append(new_tag)

                            # copy resource file
                            shutil.copyfile(os.path.join(resource_dir, resource_file), os.path.join(tempdir, 'OEBPS', resource_file))
                            print(resource_file + ' added.')
                            
                        else:
                            print(resource_file + ' skipped.')

                    # update book.opf
                    if str(soup) != orig_soup:
                        with open(opf_path,'w', encoding='utf-8') as fp:
                            fp.write(str(soup))
                        print('book.opf updated.')

                else:
                    print('add_resources=false or no resource files found.' )

                # zip everything up
                epub_zip_up_book_contents(tempdir, epub_path)
                shutil.rmtree(tempdir)

            # import epub into Sigil
            if os.path.isfile(epub_path):
                with open(epub_path,'rb') as fp:
                    data = fp.read()
                bk.addotherfile('dummy.epub', data)
                os.remove(epub_path)
                return 0
        else:
            return -1

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
