#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys, re, os
from sigil_bs4 import BeautifulSoup, Comment
from plugin_utils import Qt, QtCore, QtGui, QtWidgets
from plugin_utils import loadUi, PluginApplication, iswindows

plugin_dir = os.path.dirname(os.path.abspath(__file__))

# main GUI
class GUI(QtWidgets.QDialog):
    ''' main GUI class '''

    def __init__(self, bk):
        ''' initiate the class and populate the text boxes with prefs '''
        self.bk = bk
        self.prefs = self.bk.getPrefs()

        super(GUI, self).__init__()
        loadUi(os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'FootnoteLinkerQT.ui'), self)

        # define connects
        self.buttonBox.accepted.connect(self.accept)
        self.accepted.connect(self.get_values)
        self.buttonBox.rejected.connect(self.reject)
        self.rejected.connect(self.cancel_button)

        # populate text boxes with preference values
        if 'anchor' in self.prefs:
            self.anchorEntry.setText(self.prefs['anchor'])
        if 'anchorid' in self.prefs:
            self.anchorIDEntry.setText(self.prefs['anchorid'])
        if 'fndef' in self.prefs:
            self.fndefEntry.setText(self.prefs['fndef'])
        if 'fndefid' in self.prefs:
            self.idEntry.setText(self.prefs['fndefid'])
        if 'backlink' in self.prefs:
            self.backlinkEntry.setText(self.prefs['backlink'])
        self.show()

    def get_values(self):
        ''' saves all values as preferences '''
        global Cancel
        Cancel = False
        self.prefs['anchor'] = self.anchorEntry.text().strip()
        self.prefs['anchorid'] = self.anchorIDEntry.text().strip()
        self.prefs['fndef'] = self.fndefEntry.text().strip()
        self.prefs['fndefid'] = self.idEntry.text().strip()
        self.prefs['backlink'] = self.backlinkEntry.text().strip()
        self.bk.savePrefs(self.prefs)

    def cancel_button(self):
        ''' set the global Cancel variable '''
        global Cancel
        Cancel = True
        self.close

def run(bk):
    ''' main plugin routine '''
    if not bk.launcher_version() >= 20170115:
        print('This plugin requires at least Sigil 0.9.8.\n\nClick OK to close.')
        return -1

    # load preferences
    global prefs
    prefs = bk.getPrefs()

    # setup GUI
    mdp = True if iswindows else False
    app_icon = os.path.join(plugin_dir, 'app.ico')
    app = PluginApplication(sys.argv, bk, app_icon=app_icon, match_dark_palette=mdp,
                            load_qtbase_translations=False, load_qtplugin_translations=False)

    window = GUI(bk)
    app.exec()

    if Cancel:
        print('Terminated by user.\nClick OK to close.')
        return -1

    # update preferences
    prefs = bk.getPrefs()
    fn_class = prefs.get('anchor', False)
    fn_id = prefs.get('anchorid', False)
    fndef_class = prefs.get('fndef', False)
    fndef_id = prefs.get('fndefid', False)
    backlink_class = prefs.get('backlink', False)
    aside_class = prefs.get('aside_class', False)

    # find out if it's an epub2 or epub3 book
    epubversion = bk.epub_version()

    # Find footnote definitions file
    fn_def_id = None
    if epubversion.startswith("3"):
        opf_soup = BeautifulSoup(bk.get_opf(), 'lxml')
        nav_item = opf_soup.find('item', {'properties': 'nav'})
        if nav_item:
            nav_href = nav_item['href']
            nav_id = bk.href_to_id(nav_href)
            landmarks = {}
            nav_soup = BeautifulSoup(bk.readfile(nav_id), 'html.parser')
            for landmark in nav_soup.find_all('a', {'epub:type': re.compile('.*?')}):
                landmarks[landmark['epub:type']] = landmark['href']
            if 'footnotes' in landmarks:
                base_name = os.path.basename(landmarks['footnotes'])
                fn_def_id = bk.basename_to_id(base_name)
            elif 'endnotes' in landmarks:
                base_name = os.path.basename(landmarks['endnotes'])
                fn_def_id = bk.basename_to_id(base_name)
            elif 'rearnotes' in landmarks:
                base_name = os.path.basename(landmarks['rearnotes'])
                fn_def_id = bk.basename_to_id(base_name)
                print('Note: the "rearnotes" semantic has been deprecated. You might want to use the "endnotes" semantic instead.')
            else:
                print('\nError: Missing footnotes/endnotes epub3 landmark.\nUse Add Semantics > Footnotes/Endnotes.\nPlugin terminated!')
                return -1
        else:
            print('\nError: nav document not found!\nPlugin terminated!')
            return -1
    else:
        opf_guide_items = {ref_type: href for ref_type, _, href in bk.getguide()}
        if 'notes' not in opf_guide_items:
            print('Error: Missing Notes guide item.\nUse Add Semantics > Notes.\nPlugin terminated!')
            return -1
        fn_def_id = bk.href_to_id(opf_guide_items['notes'])

    if not fn_def_id:
        print('Footnote definition file id not found!\nClick OK to close.')
        return -1

    # Process all files in a single pass
    sup_counter = 0
    fn_to_href = {}
    context = {}
    file_updates = {}
    file_list = list(bk.text_iter())

    print('\nProcessing footnote anchor files...\n')
    for html_id, href in file_list:
        if html_id == fn_def_id:
            continue  # Skip the footnote definitions file
        html = bk.readfile(html_id)
        soup = BeautifulSoup(html, 'html.parser')
        orig_soup = str(soup)
        all_tags = soup.find_all(attrs={"class": fn_class})

        for each_tag in all_tags:
            sup_counter += 1
            if bk.launcher_version() >= 20190927 and not bk.epub_is_standard():
                fn_def_id_bookpath = bk.id_to_bookpath(fn_def_id)
                html_id_bookpath = bk.id_to_bookpath(html_id)
                from_bookpath = fn_def_id_bookpath
                to_bookpath = html_id_bookpath
                relativepath = bk.get_relativepath(from_bookpath, to_bookpath)
                fn_to_href[sup_counter] = relativepath + '#' + fn_id + str(sup_counter)
                from_bookpath = html_id_bookpath
                to_bookpath = fn_def_id_bookpath
                relativepath = bk.get_relativepath(from_bookpath, to_bookpath)
                href_to_def = relativepath + '#' + fndef_id + str(sup_counter)
            else:
                fn_to_href[sup_counter] = '../' + href + '#' + fn_id + str(sup_counter)
                href_to_def = '../' + bk.id_to_href(fn_def_id) + '#' + fndef_id + str(sup_counter)

            if each_tag.a:
                each_tag.a['id'] = fn_id + str(sup_counter)
                each_tag.a['href'] = href_to_def
                if epubversion.startswith("3"):
                    each_tag.a['epub:type'] = 'noteref'
                each_tag.a.string = '[' + str(sup_counter) + ']'
            else:
                each_tag.name = 'a'
                each_tag['id'] = fn_id + str(sup_counter)
                each_tag['href'] = href_to_def
                if epubversion.startswith("3"):
                    each_tag['epub:type'] = 'noteref'
                each_tag.string = '[' + str(sup_counter) + ']'

        if str(soup) != orig_soup:
            if epubversion.startswith("3") and 'xmlns:epub' not in soup.html.attrs:
                soup.html['xmlns:epub'] = 'http://www.idpf.org/2007/ops'
            try:
                file_updates[html_id] = str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  "))
            except:
                file_updates[html_id] = str(soup)
            print('Footnote anchor file:', href, 'queued for update')

    # Process footnote definitions
    def_counter = 0
    html = bk.readfile(fn_def_id)
    html = re.sub(r'(class="{}"[^>]*>)\d+\.*\s*'.format(fndef_class), r'\1 ', html)
    soup = BeautifulSoup(html, 'html.parser')
    orig_soup = str(soup)
    all_tags = soup.find_all(attrs={"class": fndef_class})
    fndef_number = len(all_tags)

    print('\nChecking number of anchors and definitions...')
    print('Footnote anchors:', sup_counter)
    print('Footnote definitions:', fndef_number)

    # display an error message if the number of footnotes and definition don't match
    if sup_counter != fndef_number:
        print('\nError: Number of footnote anchors and definitions do not match!')
        print('No files will be updated. Please ensure the counts match and try again.')
        print('OK to close the Plugin Runner window.')
        return -1

    # process footnote definitions
    print('\nProcessing footnote definitions...')
    for index, each_tag in enumerate(all_tags, start=1):
        def_counter += 1
        if epubversion.startswith("3"):
            if each_tag.name == 'aside':
                each_tag['id'] = fndef_id + str(def_counter)
                if aside_class:
                    each_tag['class'] = aside_class
            elif each_tag.parent.name in ('div', 'aside'):
                if each_tag.parent.name == 'div':
                    each_tag.parent.name = 'aside'
                each_tag.parent['id'] = fndef_id + str(def_counter)
                each_tag.parent['class'] = aside_class or fndef_class
                if not aside_class:
                    del each_tag['class']
            elif each_tag.parent.parent.name in ('div', 'aside'):
                if each_tag.parent.parent.name == 'div':
                    each_tag.parent.parent.name = 'aside'
                each_tag.parent.parent['id'] = fndef_id + str(def_counter)
                each_tag.parent.parent['class'] = aside_class or fndef_class
                if not aside_class:
                    del each_tag['class']
            else:
                aside = soup.new_tag('aside', id=fndef_id + str(def_counter), **{'epub:type': 'footnote'})
                aside['class'] = aside_class or fndef_class
                if not aside_class:
                    del each_tag['class']
                each_tag.wrap(aside)
        else:
            each_tag['id'] = fndef_id + str(def_counter)

        backlink = soup.new_tag('a', href=fn_to_href[def_counter], **{'class': backlink_class})
        backlink.string = str(def_counter)
        if not each_tag.a and each_tag.name != 'a':
            each_tag.insert(0, backlink)
        elif each_tag.name == 'a':
            each_tag.replace_with(backlink)
        else:
            each_tag.a.replace_with(backlink)

        if each_tag.span and each_tag.span.string and each_tag.span.string.isdigit():
            each_tag.span.decompose()

    if str(soup) != orig_soup:
        if epubversion.startswith("3") and 'xmlns:epub' not in soup.html.attrs:
            soup.html['xmlns:epub'] = 'http://www.idpf.org/2007/ops'
        try:
            file_updates[fn_def_id] = str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  "))
        except:
            file_updates[fn_def_id] = str(soup)
        print('Footnote definitions file:', bk.id_to_href(fn_def_id), 'queued for update')

    # Write all updates
    for html_id, content in file_updates.items():
        bk.writefile(html_id, content)
        print('File updated:', bk.id_to_href(html_id))

    if sup_counter == 0:
        print('No <sup class="' + fn_class + '"> tags found!\nPlugin terminated.')
        return -1

    print('\nProcessing complete. Please click OK to close the Plugin Runner window.')
    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())