#!/usr/bin/env python
# -*- coding: utf-8 -*-
from io import BytesIO
from os.path import splitext, basename
import uuid, sys, os, tempfile, imghdr
from urllib.parse import urlparse
import urllib.request


# main routine
def run(bk):

    # check for supported versions
    if bk.launcher_version() < 20160707:
        print("This plugin requires Sigil 0.9.8 or later.\n\nPlease click OK to close the Plugin Runner window.")
        return -1

    # Try to import PySide6, then PyQt5
    try:
        from PySide6.QtWidgets import QApplication
        from PySide6.QtGui import QClipboard
    except ImportError:
        try:
            from PyQt5.QtWidgets import QApplication
            from PyQt5.QtGui import QClipboard
        except ImportError:
            raise RuntimeError("No supported GUI framework found (PySide6 or PyQt5)")

    # define random .jpg file name
    image_name = 'img_{}.jpg'.format(uuid.uuid4().hex)
    image_data = None
    mime_type = 'image/jpeg'

    if sys.platform.startswith('win'):
        # get image from Windows clipboard
        try:
            from PIL import ImageGrab
            img = ImageGrab.grabclipboard()
            if img:
                clipboard_data = BytesIO()
                img.save(clipboard_data, 'jpeg', quality=90)
                image_data = clipboard_data.getvalue()
        except:
            print('Pillow import error or unsupported clipboard image format.')

    if sys.platform.startswith('linux'):
        # get image from GTK clipboard
        try:
            from gi.repository import Gtk, Gdk
            clipboard = Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD)
            img = clipboard.wait_for_image()
            if img:
                temp_file = tempfile.NamedTemporaryFile().name
                img.savev(temp_file, "jpeg", ["quality"], ["90"])
                img_file = open(temp_file, "rb")
                image_data = img_file.read()
                img_file.close()
                os.remove(temp_file)
        except:
            print('GTK import error or unsupported clipboard image format.')

    if not image_data:
        # try to download an image from clipboard text using Qt

        try:
            app = QApplication.instance()
            if not app:
                app = QApplication(sys.argv)
                created_app = True
            else:
                created_app = False
            clipboard = app.clipboard()
            clipboard_text = clipboard.text(QClipboard.Clipboard)
            if created_app:
                app.quit()
            if clipboard_text and clipboard_text.lower().endswith(('.jpg', '.jpeg', '.png', '.gif', '.bmp', '.webp')):
                file_name, extension = splitext(basename(urlparse(clipboard_text).path))
                image_name = file_name + extension
                temp_file = os.path.join(tempfile.gettempdir(), image_name)
                urllib.request.urlretrieve(clipboard_text, temp_file)
                if extension == '':
                    extension = imghdr.what(temp_file)
                    if extension:
                        mime_type = 'image/{}.format(extension)'
                    else:
                        # default to JPEG
                        mime_type = 'image/jpeg'
                        extension = '.jpg'
                    image_name = image_name + '.' + extension
                img_file = open(temp_file, "rb")
                image_data = img_file.read()
                img_file.close()
                os.remove(temp_file)
            else:
                print("Clipboard text does not appear to be an image URL:\n{}".format(clipboard_text))
        except Exception as e:
            print('URL: {}\nUnsupported clipboard format or image retrieval error:\n{}'.format(clipboard_text, e))

    # add image to epub
    if image_data:
        try:
            bk.addfile(image_name, image_name, image_data, mime_type)
            print(image_name + ' added')
        except:
            print(image_name + ' couldn\'t be added')
    else:
        print('No clipboard image found.')

    print('\nDone.\n\nPlease click OK to close the Plugin Runner window.')
    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
