#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

from __future__ import unicode_literals, division, absolute_import, print_function

import sys
import os
import tempfile, shutil
import re

try:
    from urllib.parse import unquote
except ImportError:
    from urllib import unquote

from epub_utils import epub_zip_up_book_contents, build_idpf_encryption_xml, Idpf_encryption_key, Idpf_mangle_fonts

from PySide6 import QtWidgets

_USER_HOME = os.path.expanduser("~")


# the plugin entry point
def run(bk):

    if bk.launcher_version() < 20230315:
        print("This plugin requires Sigil-2.0.0 or later")
        return -1
    
    # create a temp directory in which to build the epub
    temp_dir = tempfile.mkdtemp()

    # copy all files to a temporary destination folder
    # to get all fonts, css, images, and etc
    bk.copy_book_contents_to(temp_dir)

    # get the current opf
    opfdata = bk.get_opf()

    # collect info on any fonts needing to be obfuscated
    fontlist = []
    for (id, href, mimetype) in bk.font_iter():
        filename = href.split("/")[-1]
        fontlist.append((id, href, filename))
    
    # parse the opf extracting the unique identifier and removing the Sigil meta branding
    res = []
    identifierid = ""
    identifier = None
    get_contents = False
    ps = bk.qp
    ps.setContent(opfdata)
    for text, tagprefix, tagname, tagtype, tagattr in ps.parse_iter():
        if text is not None:
            if get_contents:
                identifier = text
                get_contents = False
            res.append(text)
            continue
        if tagname == "package" and tagtype == "begin":
            identifierid = tagattr.get("unique-identifier", "")
        elif tagname == "dc:identifier" and tagtype == "begin":
            if tagattr.get("id","") == identifierid:
                get_contents = True
        elif tagname == "meta" and tagtype == "single" and tagattr.get("name","") == "Sigil version":
            continue
        res.append(ps.tag_info_to_xml(tagname, tagtype, tagattr))
    opfdata = "".join(res)

    # write out the revised content.opf
    opf_book_path = bk.get_opfbookpath();
    
    with open(os.path.join(temp_dir, opf_book_path),'wb') as f:
        f.write(opfdata.encode('utf-8'))

    if identifier is None and len(fontlist > 0):
        print("Error: The required unique identifier can not be found so fonts can not be obfuscated")
        return 1

    # handle any required font mangling and generation of encryption.xml file
    if len(fontlist) > 0:
        key = Idpf_encryption_key(identifier)
        mangled_fonts = []
        for (id, href, filename) in fontlist:
            fontdata = bk.readfile(id)
            font_book_path = bk.id_to_bookpath(id)
            mangled_data = Idpf_mangle_fonts(key, fontdata)
            mangled_fonts.append(font_book_path)
            with open(os.path.join(temp_dir,font_book_path),'wb') as f:
                f.write(mangled_data)
        encryptionxml = build_idpf_encryption_xml(mangled_fonts)
        with open(os.path.join(temp_dir,"META-INF", "encryption.xml"),'wb') as f:
            f.write(encryptionxml.encode('utf-8'))

    # Make sure a mimetype exists
    data = "application/epub+zip"
    with open(os.path.join(temp_dir,"mimetype"),'wb') as f:
        f.write(data.encode('utf-8'))

    # ask the user where he/she wants to store the new epub
    # TODO use dc:title from the OPF file instead
    doctitle = "filename"
    fname = cleanup_file_name(doctitle) + ".epub"

    # ask for name and location to save
    suggest = _USER_HOME + '/' + fname
    app = QtWidgets.QApplication(sys.argv)
    fpath, seleted_filter = QtWidgets.QFileDialog.getSaveFileName(None, 'Save as ...', suggest, "ePubs (*.epub)")
    app.quit()

    if not fpath:
        shutil.rmtree(temp_dir)
        print("output plugin cancelled by user")
        return 0

    epub_zip_up_book_contents(temp_dir, fpath)
    shutil.rmtree(temp_dir)

    print("Output Plugin Complete")
    # Setting the proper Return value is important.
    # 0 - means success
    # anything else means failure
    return 0
 
# borrowed from calibre from calibre/src/calibre/__init__.py
# added in removal of non-printing chars
# and removal of . at start

def cleanup_file_name(name):
    import string
    _filename_sanitize = re.compile(r'[\xae\0\\|\?\*<":>\+/]')
    substitute='_'
    one = ''.join(char for char in name if char in string.printable)
    one = _filename_sanitize.sub(substitute, one)
    one = re.sub(r'\s', '_', one).strip()
    one = re.sub(r'^\.+$', '_', one)
    one = one.replace('..', substitute)
    # Windows doesn't like path components that end with a period
    if one.endswith('.'):
        one = one[:-1]+substitute
    # Mac and Unix don't like file names that begin with a full stop
    if len(one) > 0 and one[0:1] == '.':
        one = substitute+one[1:]
    return one

def main():
    print("I reached main when I should not have\n")
    return -1
    
if __name__ == "__main__":
    sys.exit(main())

