#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

# Copyright 2021 Kevin B. Hendricks, Stratford Ontario Canada
# Copyright 2021 Doug Massay

# This plugin's source code is available under the GNU LGPL Version 2.1 or GNU LGPL Version 3 License.
# See https://www.gnu.org/licenses/old-licenses/lgpl-2.1.en.html or
# https://www.gnu.org/licenses/lgpl.html for the complete text of the license.

import sys
import os
import argparse
import tempfile, shutil
import inspect
import urllib.parse # URL 파싱을 위해 추가
import threading
import socketserver
import http.server
import socket

from plugin_utils import QtCore, QtWidgets
from plugin_utils import QtWebEngineWidgets
from plugin_utils import QWebEnginePage, QWebEngineProfile, QWebEngineScript, QWebEngineSettings
from plugin_utils import PluginApplication, iswindows, ismacos

# 스크립트가 실행되는 디렉토리의 절대 경로 (플러그인 루트 디렉토리)
SCRIPT_DIR = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))


# HTTP 서버 관련 전역 변수
_server_thread = None
_httpd = None
_server_port = 0

class CustomHTTPRequestHandler(http.server.SimpleHTTPRequestHandler):
    def translate_path(self, path):
        """
        URL의 경로 구성요소를 로컬 파일 이름 구문으로 변환합니다.
        요청 URL에서 쿼리 문자열을 제거하고, Readium Reader의
        EPUB 콘텐츠 요청 경로를 특별히 처리합니다.
        """
        # 1. URL에서 쿼리 문자열 부분을 제거합니다.
        # 예: '/viewer/cloud-reader-lite/index.html?epub=...' -> '/viewer/cloud-reader-lite/index.html'
        parsed_url = urllib.parse.urlparse(path)
        path_without_query = parsed_url.path

        # 2. 선행 슬래시를 제거합니다.
        relative_path_from_request = path_without_query.lstrip('/')
        actual_file_path = None # 초기화

        # 3. Readium이 내부적으로 EPUB 콘텐츠를 요청하는 경로 패턴을 처리합니다.
        #    Readium은 종종 `index.html`이 로드된 후, 그 상대 경로(`./epub_content/...`)로
        #    요청을 보내는데, 웹 브라우저가 이를 `index.html`의 현재 URL (`/viewer/cloud-reader-lite/`)을 기준으로
        #    해석하여 `/viewer/cloud-reader-lite/epub_content/...`와 같이 서버에 보낼 수 있습니다.
        #    또는 쿼리 파라미터로 받은 정보로 `index.html` 내부에서 `/epub_content/...`와 같은
        #    루트 기준 경로를 직접 요청할 수도 있습니다.
        
        # Readium이 `/viewer/cloud-reader-lite/epub_content/...` 형태로 요청할 경우
        if relative_path_from_request.startswith('viewer/cloud-reader-lite/epub_content/'):
            actual_file_path = os.path.join(SCRIPT_DIR, relative_path_from_request)
        # Readium이 `/epub_content/...` 형태로 요청할 경우 (이것이 가장 흔한 문제 원인)
        elif relative_path_from_request.startswith('epub_content/'):
            # 실제 파일 시스템에서 EPUB 콘텐츠가 있는 경로의 베이스는 SCRIPT_DIR/viewer/cloud-reader-lite/ 입니다.
            epub_content_base_dir = os.path.join(SCRIPT_DIR, 'viewer', 'cloud-reader-lite')
            actual_file_path = os.path.join(epub_content_base_dir, relative_path_from_request)
        # 그 외의 일반적인 요청 (예: index.html 자체 요청, css/js 파일 요청 등)
        else:
            actual_file_path = os.path.join(SCRIPT_DIR, relative_path_from_request)

        # 4. 보안: 요청된 경로가 서버의 루트 디렉토리(SCRIPT_DIR) 밖으로 나가지 않도록 합니다.
        abs_actual_file_path = os.path.abspath(actual_file_path)
        common_prefix = os.path.commonprefix([abs_actual_file_path, SCRIPT_DIR])
        
        if not common_prefix == SCRIPT_DIR:
            print(f"SECURITY WARNING: Attempt to access path outside server root: {abs_actual_file_path}", file=sys.stderr)
            return None 
        
        # 디버깅을 위해 변환된 경로를 출력합니다.
        print(f"DEBUG: HTTP server translating original path '{path}' (stripped to '{path_without_query}') to file '{abs_actual_file_path}'", file=sys.stdout)
        
        return abs_actual_file_path


class WebPage(QWebEnginePage):

    def __init__(self, profile, parent=None):
       QWebEnginePage.__init__(self, profile, parent)

    def javaScriptConsoleMessage(self, level, msg, linenumber, source_id):
        # 웹뷰에서 발생하는 JavaScript 콘솔 메시지를 출력하여 디버깅에 활용
        prefix = {
            QWebEnginePage.JavaScriptConsoleMessageLevel.InfoMessageLevel: 'INFO',
            QWebEnginePage.JavaScriptConsoleMessageLevel.WarningMessageLevel: 'WARNING'
        }.get(level, 'ERROR')
        if prefix in ('ERROR') and not 'ResizeObserver loop limit exceeded' in msg:
            try:
                print('%s: %s:%s: %s' % (prefix, source_id, linenumber, msg), file=sys.stderr)
                sys.stderr.flush()
            except EnvironmentError:
                pass

    def acceptNavigationRequest(self, url, req_type, is_main_frame):
        # 로컬 파일 시스템 내에서의 탐색은 허용해야 합니다.
        # 'data' 및 'blob' 스키마는 허용
        if req_type == QWebEnginePage.NavigationType.NavigationTypeReload:
            return True
        if req_type == QWebEnginePage.NavigationType.NavigationTypeBackForward:
            return True
        if url.scheme() in ('data', 'blob'):
            return True
        # HTTP 스키마는 로컬 서버일 경우에만 허용, 외부 링크는 차단
        if url.scheme() in ('http', 'https'):
            # 로컬 서버의 IP(127.0.0.1)와 포트가 일치하는지 확인
            if url.host() == '127.0.0.1' and url.port() == _server_port:
                return True
            # 외부 HTTP/HTTPS 링크 클릭 시 차단
            if req_type == QWebEnginePage.NavigationType.NavigationTypeLinkClicked:
                print('Blocking external HTTP/HTTPS navigation request to: ', url.toString())
                return False;
        print('Blocking navigation request to:', url.toString())
        return False


class WebView(QtWebEngineWidgets.QWebEngineView):

    def __init__(self, parent=None):
        QtWebEngineWidgets.QWebEngineView.__init__(self, parent)
        app = PluginApplication.instance()
        # 플러그인 설정 폴더 경로
        pfolder = os.path.dirname(app.bk._w.plugin_dir) + '/plugins_prefs/' + app.bk._w.plugin_name
        localstorepath = os.path.join(pfolder, 'local-storage')
        os.makedirs(localstorepath, exist_ok=True)
        print(f"Local Storage Path: {localstorepath}")
        
        # 뷰어 사이즈 힌트 설정 (플러그인 창 크기 조절용)
        w = app.primaryScreen().availableGeometry().width()
        self._size_hint = QtCore.QSize(int(w/3), int(w/2))
        
        # QWebEngineProfile 설정 (캐시, 영구 저장소 등)
        profile_name = 'ReadiumReaderSigilPluginSettings'
        self._profile = QWebEngineProfile(profile_name)
        # HTTP 캐시를 메모리에만 저장하도록 설정
        self._profile.setHttpCacheType(QWebEngineProfile.MemoryHttpCache)
        
        # QWebEnginePage 인스턴스 생성 및 WebView에 설정
        self._page = WebPage(self._profile, self)
        self.setPage(self._page)
        
        # WebView의 설정 (JavaScript, 윈도우 팝업, 클립보드 접근 등)
        s = self.settings()
        s.setAttribute(QWebEngineSettings.WebAttribute.JavascriptEnabled, True)
        s.setAttribute(QWebEngineSettings.WebAttribute.JavascriptCanOpenWindows, True)
        s.setAttribute(QWebEngineSettings.WebAttribute.JavascriptCanAccessClipboard, True)
        s.setAttribute(QWebEngineSettings.WebAttribute.AllowWindowActivationFromJavaScript, True)
        
        # Readium 설정(로컬 저장소)을 플러그인 prefs 폴더에 저장
        self._page.profile().setPersistentStoragePath(localstorepath)
        print(f"Is Off The Record: {self._page.profile().isOffTheRecord()}")
        print(f"Cache Path: {self._page.profile().cachePath()}")
        print(f"HTTP Cache Type: {self._page.profile().httpCacheType()}")
        print(f"Persistent Storage Path: {self._page.profile().persistentStoragePath()}")

    def sizeHint(self):
        return self._size_hint


class MainWindow(QtWidgets.QMainWindow):

    def __init__(self, query, prefs, *args, **kwargs):
        super(MainWindow, self).__init__(*args, **kwargs)

        self.query = query
        self.prefs = prefs

        # QWebEngineView 생성
        self.browser = WebView()

        # 디버그 라인: update_title 메서드가 callable한지 확인 (이전 AttributeError 해결 확인용)
        print(f"DEBUG: update_title method callable: {callable(getattr(self, 'update_title', None))}")

        # 로딩 완료 시 창 타이틀 업데이트 액션 연결
        self.browser.loadFinished.connect(self.update_title)

        readerpath = os.path.join(SCRIPT_DIR, 'viewer', 'cloud-reader-lite', 'index.html')
        
        if not os.path.exists(readerpath):
            print(f"CRITICAL ERROR: index.html not found at: {readerpath}", file=sys.stderr)
        else:
            print(f"DEBUG: index.html found at: {readerpath}")

        # HTTP 서버를 통해 서비스되는 index.html의 기본 URL
        base_url = f"http://127.0.0.1:{_server_port}/viewer/cloud-reader-lite/index.html"
        # EPUB 경로를 쿼리 파라미터로 추가
        bookurl_str = f"{base_url}?{self.query}"

        # 구성된 URL을 웹뷰에 설정
        bookurl = QtCore.QUrl(bookurl_str)
        print(f"DEBUG: Final bookurl = {bookurl.toString()}")

        self.browser.setUrl(bookurl)

        self.setCentralWidget(self.browser)

        # 저장된 창 설정 로드 및 창 표시
        self.readsettings()
        self.show()

    def readsettings(self):
        b64val = self.prefs.get('geometry', None)
        if b64val:
            self.restoreGeometry(QtCore.QByteArray.fromBase64(QtCore.QByteArray(b64val.encode('ascii'))))

    # 창 타이틀을 업데이트하는 메서드
    def update_title(self):
        if not self.browser or not self.browser.isVisible():
            return
        height = self.browser.height()
        width =    self.browser.width()
        self.setWindowTitle('Screen Size:'  +  ' (%dx%d)' % (width, height))

    def done(self):
        self.close()

    def resizeEvent(self, ev):
        QtWidgets.QMainWindow.resizeEvent(self, ev)
        self.update_title()

    def closeEvent(self, ev):
        # 창 크기/위치 저장
        b64val = str(self.saveGeometry().toBase64(), 'ascii')
        self.prefs['geometry'] = b64val
        QtWidgets.QMainWindow.closeEvent(self, ev)
        # 플러그인 종료 시 HTTP 서버 중지
        stop_http_server()


# 사용 가능한 포트를 찾는 헬퍼 함수
def find_free_port():
    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
        s.bind(('127.0.0.1', 0)) # 0을 사용하여 운영체제가 사용 가능한 포트를 할당하도록 함
        return s.getsockname()[1]

# HTTP 서버 시작 함수
def start_http_server(port, directory):
    global _httpd, _server_thread, _server_port
    _server_port = port
    
    # SimpleHTTPRequestHandler는 os.chdir을 사용한 디렉토리 변경에 의존하므로,
    # 여기서는 directory를 SCRIPT_DIR로 넘겨주고, CustomHTTPRequestHandler에서
    # translate_path를 오버라이드하여 SCRIPT_DIR을 직접 사용하도록 합니다.
    # 따라서 이 os.chdir 부분은 여전히 필요 없지만 문제가 없으니 유지합니다.
    original_cwd = os.getcwd()
    os.chdir(directory) 

    Handler = CustomHTTPRequestHandler
    try:
        _httpd = socketserver.TCPServer(("127.0.0.1", port), Handler, bind_and_activate=False)
        _httpd.allow_reuse_address = True
        _httpd.server_bind()
        _httpd.server_activate()

        print(f"Serving HTTP on 127.0.0.1 port {port} from directory {directory}")
        # 서버를 별도 스레드에서 실행
        _server_thread = threading.Thread(target=_httpd.serve_forever)
        _server_thread.daemon = True  # 메인 프로그램 종료 시 스레드도 종료되도록 설정
        _server_thread.start()
    except Exception as e:
        print(f"Error starting HTTP server: {e}", file=sys.stderr)
        _httpd = None
        _server_thread = None
    finally:
        os.chdir(original_cwd)


# HTTP 서버 중지 함수
def stop_http_server():
    global _httpd, _server_thread
    if _httpd:
        print("Stopping HTTP server...")
        _httpd.shutdown() # 서버 스레드 종료 신호
        _httpd.server_close() # 소켓 닫기
        if _server_thread and _server_thread.is_alive():
            _server_thread.join(timeout=1) # 스레드가 종료될 때까지 최대 1초 대기
        _httpd = None
        _server_thread = None
        print("HTTP server stopped.")


# 플러그인 진입점
def run(bk):
    # 중요: QApplication이 생성되기 전에 환경 변수를 설정하여 GPU 가속 비활성화 시도
    # Sigil 플러그인 환경에서는 이 시점이 이미 늦을 수 있으므로 효과가 없을 수도 있습니다.
    os.environ["QTWEBENGINE_CHROMIUM_FLAGS"] = "--disable-gpu --disable-gpu-compositing"
    print(f"DEBUG: QTWEBENGINE_CHROMIUM_FLAGS set to: {os.environ.get('QTWEBENGINE_CHROMIUM_FLAGS')}")

    if bk.launcher_version() < 20210430:
        print("\nThis plugin requires Sigil-1.6.0 or later to function.")
        return -1

    # 사용자 환경설정 로드
    prefs = bk.getPrefs()

    # EPUB 뷰어의 홈 디렉토리 및 EPUB 콘텐츠 저장 디렉토리 설정
    viewer_home = os.path.join(SCRIPT_DIR, 'viewer', 'cloud-reader-lite')
    epub_home = os.path.join(viewer_home, 'epub_content')
    os.makedirs(epub_home, exist_ok = True) # 디렉토리가 없으면 생성

    # EPUB 콘텐츠를 임시 디렉토리로 복사
    bookdir = tempfile.mkdtemp(suffix=None, prefix=None, dir=epub_home)
    bookdir_name = os.path.split(bookdir)[-1] # 임시 디렉토리 이름
    bk.copy_book_contents_to(bookdir) # Sigil API를 사용하여 EPUB 콘텐츠 복사
    
    # mimetype 파일 생성 (EPUB 유효성 검사용)
    data = 'application/epub+zip'
    mpath = os.path.join(bookdir, 'mimetype')
    with open(mpath, 'wb') as f:
        f.write(data.encode('utf-8'))
        f.close()

    # Readium에 전달할 쿼리 파라미터 (EPUB 콘텐츠의 상대 경로)
    # Readium의 index.html을 기준으로 'epub_content/임시디렉토리명/'으로 경로를 전달합니다.
    query = 'epub=epub_content/' +  bookdir_name + '/'

    # HTTP 서버 시작
    port = find_free_port()
    start_http_server(port, SCRIPT_DIR) # SCRIPT_DIR을 서버의 루트 디렉토리로 설정


    icon = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'plugin.svg')
    # PyQt 애플리케이션 생성
    app = PluginApplication(sys.argv, bk, app_icon=icon)
    app.setApplicationName("Readium Cloud Reader Lite Demo")

    # 메인 윈도우 객체 생성 및 표시
    window = MainWindow(query, prefs)
    app.exec_() # PyQt 이벤트 루프 실행

    # 임시 폴더 정리
    shutil.rmtree(bookdir)

    print("Readium Reader Session Complete")
    bk.savePrefs(prefs) # 변경된 설정 저장

    stop_http_server() # 명시적으로 서버 중지 호출

    # PyQt5 < 5.14 버전에서 발생할 수 있는 종료 시 충돌 방지
    del window, app
    return 0 # 성공적으로 종료

def main():
    print("I reached main when I should not have\n")
    return -1

if __name__ == "__main__":
    sys.exit(main())