#!/bin/bash

# Return "true" if it's ok to keep the kindle alive, preventing
# it from entering a suspended state.
#
# Depends on the following variables:
#
# ${CMD} = the command that invoked this method.
# ${INSTALLROOT} = the installation root.
#
# Debugging output is sent to stderr, which the caller can then
# redirect as necessary.

source "${INSTALLROOT}/config/config.sh"
source "${INSTALLROOT}/lib/sh/fn_logger.sh"

# usage: fn_kindle_keepalive [ pluggedin ]
#
# where:
#
# pluggin = an optional "true/false" parameter that states whether the
#   kindle is plugged into a power source. If "true", we presume that
#   it is. Otherwise, we rely soley on powerd's battStateInfo property,
#   which does not always reflect whether the kindle is plugged in.
fn_kindle_keepalive() {
    local DEBUG=1
    local pluggedin=""
    if [[ $# -gt 0 ]]
    then
        pluggedin="$1"
    else
        pluggedin=false
    fi

    # See if the kindle is connected to a power source and if it's
    # ok to keep it alive, not allowing it to suspend.
    if [[ "${pluggedin}" == true ]]
    then
        # Our processEventDeviceEvents.sh script has already determined
        # that the kindle is plugged into a power source.
        echo true
        return
    fi

    # We use the information contained within powerd's battStateInfo
    # property to determine if the kindle is either charging or its
    # battery level is such that it's ok to prevent it from suspending.
    #
    # The battStateInfo property value looks something like this:
    #
    # {"battInfo":[{"type":"PRIMARY","cap":99,"charging":true}],"state":"STANDALONE_USB","aggr_batt":99,"usb":true}
    #
    # where:
    #
    # cap = the battery's capacity.
    # charging/usb = true if the kindle is charging, false if not.
    #
    # In practice, when "charging" goes to false, so does "usb", even if
    # still connected to a power source. Therefore, we cannot rely on
    # "usb" to know if the kindle is connected to a power source.
    #
    # To avoid stressing the battery, the kindle will allow the
    # battery to discharge even when connected to a power source.
    # Therefore,  we must rely on the battery level in addition to
    # the "charging" member. I've seen my Kindle Signature 12th Gen
    # discharge to 97%, and sometimes lower, before it begins
    # charging again.
    #
    # Note that the kindle's busybox version of sed(1) does not support
    # '[0-9]+' (for one or more), so we had to use '[0-9][0-9]*'
    # instead in the sed(1) commands below.
    local cap=""
    local charging=""
    local usb=""
    
    local battStateInfo=$(lipc-get-prop com.lab126.powerd battStateInfo 2>/dev/null)
    if [[ -z "${battStateInfo}" ]]
    then
        # The kindle doesn't support the battStateInfo property. Try the older
        # "status" property.
        battStateInfo=$(lipc-get-prop com.lab126.powerd status 2>/dev/null | tr '\n' ' ')
        cap=$(echo \"${battStateInfo}\" | sed -e 's/.*Battery Level: \([0-9][0-9]*\).*/\1/')
        chargingpattern='.*Charging: Yes.*'
        usbpattern=DOESNOTEXIST
    else
        cap=$(echo "${battStateInfo}" | sed -e 's/.*"cap":\([0-9][0-9]*\).*/\1/')
        chargingpattern='.*"charging":true.*'
        usbpattern='.*"usb":true.*'
    fi
    
    # Do not include quotes around the pattern. Doing so causes the matching
    # operator to take the pattern as literal string instead of a pattern.
    #
    # The kindle's busybox version of bash(1) does not support the
    # +~ operator. So we use expr as an alternative.
    #if [[ ! "${cap}" =~ [0-9][0-9]* ]]; then cap=0; fi
    #if [[ "${battStateInfo}" =~ ${chargingpattern} ]]; then charging=true; else charging=false; fi
    #if [[ "${battStateInfo}" =~ ${usbpattern} ]]; then usb=true; else usb=false; fi

    if ! expr "${cap}" : "[0-9][0-9]*" >/dev/null; then cap=0; fi
    if expr "${battStateInfo}" : "${chargingpattern}" >/dev/null; then charging=true; else charging=false; fi
    if expr "${battStateInfo}" : "${usbpattern}" >/dev/null; then usb=true; else usb=false; fi

    if [[ "${usb}" == true || "${charging}" == true || "${cap}" -ge "${BATTLEVELTHRESHOLD}" ]]
    then
        retval=true
    else
        retval=false
        if [[ "${DEBUG}" == 1 ]]; then (fn_logger_loginfo "${CMD} - fn_kindle_keepalive" "not pluggedin, returning ${retval}, cap = ${cap}, charging = ${charging}, usb = ${usb}"); fi
    fi

    echo "$retval"
}
