#!/bin/bash
IFS=" 	
"
PATH="/bin:/usr/bin:/sbin:/usr/sbin"
CMD=$(basename "$0")
CMDPATH=$(dirname "$0")
if [[ "${CMDPATH}" == "" ]]; then CMDPATH="./"; fi
CMDPATH=$(realpath "${CMDPATH}")
INSTALLROOT=$(realpath "${CMDPATH}/..")

source "${INSTALLROOT}/config/config.sh"
source "${INSTALLROOT}/lib/sh/fn_logger.sh"
config="${INSTALLROOT}/config/homeauto.json"

# busybox bash does not support arrays.
# Therefore, we mimic them using regular variables.
#declare -A config_dict    # NOT SUPPORTED

if [[ ! -f "$config" ]]; then
    fn_logger_logerr "$CMD" "Could read configuration, file = $config." "File not found"
    exit 1
fi

# Read the JSON configuration file
#
# We don't use a pipe to send the output from ${JQ}
# to the while loop because don't want the while loop
# running as a child process. We need the variables
# set within the while loop avaiable outside.
#
# The kindle's busybox version of bash(1) does not support
# associative arrays. Nor does it support Process Substitution
# notation such as:
#
# while read i; do echo $i; done < <("${JQ}" ...)
#
# The above would generate the following syntax error:
#
# syntax error: unexpected redirection
#
# So we have to use a temporary file to hold the output
# from ${JQ} instead.
tmpfile="/tmp/homeauto$$"
trap "trap '' EXIT SIGINT SIGTERM; rm -f \"${tmpfile}\"" EXIT SIGINT SIGTERM
"${JQ}" -r 'to_entries | .[] | "\(.key):\(.value)"' "${config}" > "${tmpfile}"
while IFS=":," read -r key value
do
    key=$(echo "${key}" | tr -d ' "')
    value=$(echo "${value}" | tr -d ' "')

    #config_dict["${key}"]="${value}"       # NOT SUPPORTED
    eval "${key}=\"${value}\""
done < "${tmpfile}"

#dirigeraIP="${configdict[dirigeraIP]}"     # NOT SUPPORTED
if [[ -z "${dirigeraIP}" ]]; then
    fn_logger_logerr "$CMD" "Configuration missing 'dirigeraIP' definition, file = $config." ""
    exit 1
fi

CODE_ALPHABET="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-._~"
CODE_LENGTH=128
authUrl="https://${dirigeraIP}:8443/v1/oauth/authorize"
tokenUrl="https://${dirigeraIP}:8443/v1/oauth/token"

getChar() {
    echo "${CODE_ALPHABET:RANDOM%${#CODE_ALPHABET}:1}"
}

getCodeVerifier() {
    local s=""
    for (( a=0; a<CODE_LENGTH; a++ )); do
        s+=$(getChar)
    done
    echo "$s"
}

createCodeChallenge() {
    local codeVerifier="$1"
    local sha256Hash
    sha256Hash=$(echo -n "$codeVerifier" | openssl dgst -sha256 -binary | openssl base64 | tr -d '=' | tr '/+' '_-' | tr -d '\n')
    echo "$sha256Hash"
}

codeVerifier=$(getCodeVerifier)
params="audience=homesmart.local&response_type=code&code_challenge=$(createCodeChallenge "$codeVerifier")&code_challenge_method=S256"
response=$(curl -k -G "$authUrl" --data "$params")
code=$(echo "$response" | $JQ -r '.code')

read -n 1 -s -r -p 'Wait for button pressed (actionbutton on Dirigera)...' # Press button on device and press a key in cmd-prompt to "release" code below afterwards

data="code=$code&name=$(hostname)&grant_type=authorization_code&code_verifier=$codeVerifier"
headers="Content-Type: application/x-www-form-urlencoded"
response=$(curl -k -X POST "$tokenUrl" -H "$headers" -d "$data")
echo "$(echo "$response" | $JQ -r '.access_token')"
