#!/bin/bash
IFS=" 	
"
PATH="/bin:/usr/bin:/sbin:/usr/sbin"
CMD=$(basename "$0")
CMDPATH=$(dirname "$0")
if [[ "${CMDPATH}" == "" ]]; then CMDPATH="./"; fi
CMDPATH=$(realpath "${CMDPATH}")
INSTALLROOT=$(realpath "${CMDPATH}/..")

source "${INSTALLROOT}/config/config.sh"
source "${INSTALLROOT}/lib/sh/fn_logger.sh"

config="${INSTALLROOT}/config/homeauto.json"

if [ "$#" -ne 2 ]; then
    echo "usage: ${command_name} deviceID state
where:
    deviceID = device identifier
    state = \"on\" or \"off\""
    exit 1
fi

device_id="$1"
state=$(echo "$2" | tr '[:upper:]' '[:lower:]')

case "${state}" in
    "on")
        state="true"
        ;;
    "off")
        state="false"
        ;;
    *)
        echo "Invalid state (${state}). Must be \"on\" or \"off\"."
        exit 1
        ;;
esac

# busybox bash does not support arrays.
# Therefore, we mimic them using regular variables.
#declare -A config_dict    # NOT SUPPORTED

if ! [[ -f "$config" ]]; then
    fn_logger_logerr "$CMD" "Could read configuration, file = $config." "File not found"
    exit 1
fi

# Read the JSON configuration file
#
# We don't use a pipe to send the output from ${JQ}
# to the while loop because don't want the while loop
# running as a child process. We need the variables
# set within the while loop avaiable outside.
#
# The kindle's busybox version of bash(1) does not support
# associative arrays. Nor does it support Process Substitution
# notation such as:
#
# while read i; do echo $i; done < <("${JQ}" ...)
#
# The above would generate the following syntax error:
#
# syntax error: unexpected redirection
#
# So we have to use a temporary file to hold the output
# from ${JQ} instead.
tmpfile="/tmp/homeauto$$"
trap "trap '' EXIT SIGINT SIGTERM; rm -f \"${tmpfile}\"" EXIT SIGINT SIGTERM
"${JQ}" -r 'to_entries | .[] | "\(.key):\(.value)"' "${config}" > "${tmpfile}"
while IFS=":," read -r key value
do
    key=$(echo "${key}" | tr -d ' "')
    value=$(echo "${value}" | tr -d ' "')

    #config_dict["${key}"]="${value}"       # NOT SUPPORTED
    eval "${key}=\"${value}\""
done < "${tmpfile}"

#dirigeraIP="${config_dict[dirigeraIP]}"    # NOT SUPPORTED
if [ -z "${dirigeraIP}" ]; then
    fn_logger_logerr "$CMD" "Configuration missing 'dirigeraIP' definition, file = $config." ""
    exit 1
fi

#accessToken="${config_dict[accessToken]}"  # NOT SUPPORTED
if [ -z "${accessToken}" ]; then
    fn_logger_logerr "$CMD" "Configuration missing 'accessToken' definition, file = $config." ""
    exit 1
fi

export CURL_INSECURE=1
devicesURL="https://${dirigeraIP}:8443/v1/devices/"
deviceURL="${devicesURL}${device_id}"

# Create JSON payload
json_payload=$($JQ -n '[{"attributes": {"isOn": '$state'}}]')

# Send PATCH request using curl
curl -k -X PATCH "${deviceURL}" -H "Authorization: Bearer ${accessToken}" -H "Content-Type: application/json" -d "$json_payload" > /dev/null 2>&1
