#!/bin/bash
#
# The main event processing loop.
#
# The reason for this script is so variables assigned
# by logic within the while loop that services lipc
# events are available outside while loop.
#
# Invoking:
#
# printf 'com.lab126.powerd\ncom.lab126.btfd\ncom.kindleautox.processevents\ncom.kindleautox.media' |\
# lipc-wait-event -l -m ${eventlist} 2>/dev/null |\
# while read event
# do
#     ...
# done
#
# causes any variables assigned from logic within the while loop to be
# only available inside the while loop (this includes "globals"
# assigned by functions that might be called within the while loop).
# This is because in the example above the while loop is treated as
# a subshell, and what might appear to be global variables are actually
# local to the while loop subshell.
#
# By placing most of the logic in a separate script and invoking:
#
# printf 'com.lab126.powerd\ncom.lab126.btfd\ncom.kindleautox.processevents\ncom.kindleautox.media' |\
# lipc-wait-event -l -m "eventnamelist" 2>/dev/null |\
# processeventsloop.sh
#
# Makes the variables assigned within the while loop contained within
# the processeventsloop.sh script available to the entire script and
# are treated as globals.
#
IFS=" 	
"
PATH="/bin:/usr/bin:/sbin:/usr/sbin"
CMD="${0##*/}"
CMDPATH="${0%/*}"
if [[ "${CMDPATH}" == "$0" ]]; then CMDPATH="."; fi
CMDPATH=$(realpath "${CMDPATH}")
INSTALLROOT=$(realpath "${CMDPATH}/..")

fn_onreadcfg() {
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "reading configuration details" 2>>"${LOGFILE}"; fi
    source "${INSTALLROOT}/config/config.sh"

    # We rely on btconfig.sh's definitions of ${BTDEVICE_NAME}
    # and ${BTDEVICE_ADDR} as the Bluetooth device to connect to
    # until we receive the first "com.lab126.btfd Disconnect_Result"
    # event, at which time we will use the kindle's configured
    # device.
    #
    # Unfortunely, while the kindle's btfd source has a
    # "BTconnectedDevName" which identifies the connected
    # Bluetooth device name, it doesn't have a similar option
    # that identifies the device's MAC address. Futhermore,
    # it's possible  that we are invoked at time when the
    # Bluetooth device is powered on but not connected.
    source "${INSTALLROOT}/config/btconfig.sh"
}

fn_onreadcfg

source "${INSTALLROOT}/lib/sh/fn_logger.sh"
source "${INSTALLROOT}/lib/sh/fn_kindle.sh"
source "${INSTALLROOT}/lib/sh/fn_str.sh"

# Note that the following stops the kindle from ever
# displaying the screensaver and, therefore, suspending,
# given that the kindle only suspends after a few seconds
# of having gone into its screensaver mode. Another side-effect
# of setting this option is that the power button is ignored.
#
# lipc-set-prop com.lab126.powerd preventScreenSaver 1
#
# To enable the screensaver again,invoke:
#
# lipc-set-prop com.lab126.powerd preventScreenSaver 0
#
# If we set preventScreenSaver to 1, the power button seems to be
# ignored totally and there isn't an event that we could grab
# while allowing our screensaver to appear to be able to
# know that we should stop the screensaver and set
# preventScreenSaver back to 0.
#
# This is not the behaviour that we want. We want to show
# a screensaver when the kindle would normally show a screen
# saver and preventing it from suspending while if the
# kindle is plugged in. This was the default behavior of
# my Kindle Touch, but not the Kindle Signature 12th Gen.
# The Signature suspends while showing the screensaver even
# if plugged in.
#
# We also want to allow the user to press the power button
# to exit the screensaver. Using the preventScreenSaver property
# prevents this.

LOGFILE="${LOGDIR}/processevents"
inscreensavermode=false

# We use the following to know if it's safe to play media files.
safetoplaymedia=true

# We use the following to know if we are actively playing
# media and, if so, details about it.
playingmedia=false
mediaoutersubshellpid=""
mediainnersubshellpid=""
mediaplayerpid=""
mediaclientpid=""
mediafile=""

# We use the following to know if we are actively showing
# in image on the screen or drawing the current date/time.
#
# drawingpid could be either the screensaver or
# showtime process.
drawing=false
drawingpid=""
drawingclientpid=false

# The following will either be the path to an image file or
# the text "date/time".
drawingcontent=""

fn_exitactions() {
    if [[ "${inscreensavermode}" == true ]]
    then
        fn_leavescreensavermode
        # The wakeUp propery only exists when the kindle is displaying
        # the screensaver.
        lipc-set-prop com.lab126.powerd wakeUp 1
    fi
    
    if [[ ! -z "${mediaouterplayersubshellpid}" ]]
    then
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "terminating outer player subshell, pid = ${mediaouterplayersubshellpid}" 2>>"${LOGFILE}"; fi
        kill "${mediaouterplayersubshellpid}"
    fi
    
    if [[ ! -z "${drawingpid}" ]]
    then
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "terminating drawing subshell, pid = ${drawingpid}" 2>>"${LOGFILE}"; fi
        kill "${drawingpid}"
    fi
}

fn_onexit() {
    # Ignore these now so we don't get invoked again on multiple interrupts.
    trap "" EXIT SIGINT SIGTERM
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "exiting" 2>>"${LOGFILE}"; fi
    fn_exitactions
    exit 0
}

# Trap requires a semicoln after the the last command.
trap "{ fn_onexit; }" EXIT

fn_onabort() {
    # Ignore these now so we don't get invoked again on multiple interrupts.
    trap "" EXIT SIGINT SIGTERM
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "aborting" 2>>"${LOGFILE}"; fi
    fn_exitactions
    exit 1
}

# Trap requires a semicoln after the the last command.
trap "{ fn_onabort; }" SIGINT SIGTERM

# usage: fn_subshellplayingcompeteactions
fn_outerplayersubshellcompleteactions() {
    # Tell the parent process that we are done.
    lipc-send-event com.kindleautox.media playingcomplete >>"${LOGFILE}" 2>&1
    
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "sent playingcomplete event" 2>>"${LOGFILE}"; fi
}

# usage: fn_onouterplayersubshellexit
fn_onouterplayersubshellexit() {
    trap '' EXIT SIGINT SIGTERM;
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "outer player subshell exiting" 2>>"${LOGFILE}"; fi
    fn_outerplayersubshellcompleteactions
    exit 0
}

# usage: fn_onouterplayersubshellabort
fn_onouterplayersubshellabort() {
    trap '' EXIT SIGINT SIGTERM;
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "outer player subshell aborting" 2>>"${LOGFILE}"; fi
    if [[ ! -z "${mediainnerplayersubshellpid}" ]]
    then
        ps -p "${mediaplayerpid}" > /dev/null 2>&1
        if [[ $? -eq 0 ]]
        then
            if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "terminating inner player subshell, pid = ${mediainnerplayersubshellpid}" 2>>"${LOGFILE}"; fi
            kill ${mediainnerplayersubshellpid}
            wait
        fi
    fi

    fn_outerplayersubshellcompleteactions
    exit 1
}

# usage: fn_oninnerplayersubshellabort
fn_oninnerplayersubshellabort() {
    trap '' SIGINT SIGTERM;
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "inner player subshell aborting" 2>>"${LOGFILE}"; fi
    if [[ ! -z "${mediaplayerpid}" ]]
    then
        ps -p "${mediaplayerpid}" > /dev/null 2>&1
        if [[ $? -eq 0 ]]
        then
            player=$(basename "${PLAYER}")
            if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "terminating ${mediaplayer}, pid = ${playerpid}" 2>>"${LOGFILE}"; fi
            kill ${mediaplayerpid}
            wait
        fi
    fi
    exit 1
}

# Invoked by the child process responsible for showing a screen
# saver image when it receives SIGINT or SIGTERM. We need to
# tell the parent that we are done showng the image.
#
# usage: fn_onshowimageabort
fn_onshowimageabort() {
    trap '' SIGINT SIGTERM
    
    # Tell the parent process that we are done.
    lipc-send-event com.kindleautox.screensaver showimagecomplete >>"${LOGFILE}" 2>&1
}

# Invoked by the child process responsible for showing the
# current date/time when it receives SIGINT or SIGTERM. We need to
# tell the parent that we are done shoiwng the image..
#
# usage: fn_onshowtimeabort
fn_onshowtimeabort() {
    trap '' SIGINT SIGTERM
    
    # Tell the parent process that we are done.
    lipc-send-event com.kindleautox.screensaver showtimecomplete >>"${LOGFILE}" 2>&1
}

fn_enterscreensavermode() {
    inscreensavermode=true
    (
        # Don't inherit the parent's signal handlers.
        trap - EXIT SIGINT SIGTERM

        # Because we've most likely been invoked as a result of
        # seeing the kindle's "com.lab126.powerd goingToScreenSaver" event,
        # we must wait a bit before we display ours to ensure that
        # we don't accidentally display ours first, only to have
        # it  overwritten by the kindle. 1 second was not long enough.
        sleep 2
        ${INSTALLROOT}/screensaver/showimage.sh >>"${LOGFILE}" 2>&1
    )&
}

fn_leavescreensavermode() {
    inscreensavermode=false
}

fn_onusbdisconnected() {
    # The following events are received when unplugging the
    # kindle from a power source. Blank lines indicate a
    # pause before we receive the next set of events.
    #
    # Unplug...
    # com.lab126.powerd notCharging
    # com.lab126.powerd usbdisconnected
    # com.lab126.powerd battStateInfoChanged "{"battInfo":[{"type":"PRIMARY","cap":100,"charging":false}],"state":"STANDALONE","aggr_batt":100,"usb":false}"
    # Plug back in...
    # com.lab126.powerd charging
    #
    # com.lab126.powerd battStateInfoChanged "{"battInfo":[{"type":"PRIMARY","cap":100,"charging":false}],"state":"STANDALONE","aggr_batt":100,"usb":false}"
    # com.lab126.powerd notCharging
    #
    # Unplug...
    # com.lab126.powerd notCharging
    # com.lab126.powerd usbdisconnected
    # com.lab126.powerd battStateInfoChanged "{"battInfo":[{"type":"PRIMARY","cap":100,"charging":false}],"state":"STANDALONE","aggr_batt":100,"usb":false}"
    # Plug back in...
    # com.lab126.powerd charging
    #
    # com.lab126.powerd notReadyToSuspend
    # com.lab126.powerd battStateInfoChanged "{"battInfo":[{"type":"PRIMARY","cap":100,"charging":false}],"state":"STANDALONE","aggr_batt":100,"usb":false}"
    # com.lab126.powerd notCharging
    #
    # Unplug...
    # com.lab126.powerd notCharging
    # com.lab126.powerd usbdisconnected
    # com.lab126.powerd battStateInfoChanged "{"battInfo":[{"type":"PRIMARY","cap":100,"charging":false}],"state":"STANDALONE","aggr_batt":100,"usb":false}"
    #
    # Plug back in...
    # com.lab126.powerd charging
    #
    # com.lab126.powerd notReadyToSuspend
    # com.lab126.powerd battStateInfoChanged "{"battInfo":[{"type":"PRIMARY","cap":100,"charging":false}],"state":"STANDALONE","aggr_batt":100,"usb":false}"
    # com.lab126.powerd notCharging
    #
    # Therefore, to know if the kindle is plugged in we can look for a
    # "charging" event and only change our internal state to
    # unplugged when we see a "usbdisconnected" event. We don't
    # want to set our internal state to unplugged when we see
    # a "notCharging" event because the kindl allows itself
    # discharge when plugged in. In other words, we will see
    # a "notCharging" event when the kindle is unplugged from
    # its power source as well as when it allows itself to discharge.
    #
    # Unlike the "usbdisconnected" event, there isn't a
    # "usbconnected" event. The only indication that the
    # kindle has been plugged into a power source is seeing
    # a "charging"  event after seeing a "usbdisconnected" event.
    pluggedin=false
}

fn_oncharging() {
    pluggedin=true
}

# The following is just a placeholder. We already print the event's
# details if debugging at the top of our event processing loop.
fn_onbattStateInfoChanged() {
    # ${data} for this event is of the format:
    #
    # {"battInfo":[{"type":"PRIMARY","cap":99,"charging":true}],"state":"STANDALONE_USB","aggr_batt":99,"usb":true}
    true
}

fn_ongoingtoscreensaver() {
    fn_enterscreensavermode
}

fn_onoutofscreensaver() {
    fn_leavescreensavermode
}

fn_onreadytosuspend() {
    local keepalive=false
    if [[ "${pluggedin}" == true ]]
    then
        keepalive=true
    else
        # The following gives you a chance to use the kindle
        # without plugging it in right away. We also use this
        # just in case we've been started in a situation where we
        # can't determine if the kindle is plugged in, such as
        # being started when the kindle is plugged in but discharging.
        keepalive=$(fn_kindle_keepalive "${pluggedin}" 2>>"${LOGFILE}")
    fi

    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "keepalive = ${keepalive}" 2>>"${LOGFILE}"; fi

    if [[ "${inscreensavermode}" == true && "${keepalive}" == true ]]
    then
        # It's not safe to play media while we are trying to keep
        # the kindle from suspending.
        safetoplaymedia=false

        # Don't let the kindle suspend.
        #
        # Older kindles do this already. If charging they never
        # suspend even while the screensaver is visible. My
        # Kindle Touch behaved this way; however, my Kindle
        # Signature 12th Gen does not ... it will eventually
        # suspend, even when charging.
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "deferring suspend" 2>>"${LOGFILE}"; fi

        # When the screensaver is visible, the kindle's framework dispatches
        # a readyToSuspend event in 60 seconds. Aborting the suspend
        # causes the kindle to dispatch another readyToSuspend in
        # 60 seconds.
        #
        # Not doing anything (i.e., not aborting the suspension)
        # results in the kindle suspending in 5 seconds.
        #
        # Setting deferSuspend (in seconds) delays when the kindle
        # goes into a suspend state by delaying when the kindle
        # dispatches another readyToSuspend event.
        #
        # By default, the deferSuspend property value
        # is 5 seconds. This property is only available
        # during the readyToSuspend event. Trying to set
        # this property outside of the readyToSuspend event
        # window results in an error.
        #
        # The kindle resets deferSuspend to its default of 5 seconds
        # each time it dispatches a readyTosuspend event.
        #
        # Increasing deferSuspend and then aborting the suspend has
        # no effect. The kindle dispatches another readyToSuspend
        # event in 60 seconds instead of in deferSuspend seconds.
        #
        # Setting deferSuspend causes the kindle to dispatch
        # another readyToSuspend event at the end of the given
        # interval. Therefore, we use the deferSuspend property
        # to reduce the number of readyToSuspend events that
        # we see.
        #
        # We don't want to use a large value for this though, because
        # we cannot change it outside of the readyToSuspend event
        # processing. We can't, for example, decrease the value to
        # allow for a more immediate suspension in the case the user
        # removes the kindle from its charger and the battery level is
        # low, or if the kindle is already not charging and the battery
        # level decreases to a low level.
        #
        # If we were in the middle of playing media, terminate it
        # because it most likely has been affected by the loss of
        # connection with the Bluetooth device. We will restart it
        # automatically below.
        if [[ "${playingmedia}" == true ]]
        then
            # Save the media file reference because it will get
            # clobbered as we invoke fn_onmediastop.
            replaymediafile="${mediafile}"
            fn_onmediastop
        fi

        lipc-set-prop com.lab126.powerd deferSuspend "${DEFERSUSPENDINSECS}"

        # We invoke the following logic in the background (see & after the
        # closing parenthesis) because we don't want to hold up the processing
        # of events while we sleep or play media below.
        (
            # Don't inherit the parent's signal handlers.
            trap - EXIT SIGINT SIGTERM

            # Give the kindle a chance to respond to the deferSuspend above
            # before enabling media and possibly playing a media file
            # that may have been interrupted. Not doing so can result
            # in a choppy playback.
            sleep 5

            # It should now be safe to play media again.
            lipc-send-event com.kindleautox.media safetoplay -s true >>"${LOGFILE}" 2>&1

            # If we were playing media at the time we received this event
            # media playback may have been interrupted. Repeat it if so.
            if [[ ! -z "${replaymediafile}" ]]
            then
                if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "replaying media, mediafile = \"$mediafile\"" 2>>"${LOGFILE}"; fi
                lipc-send-event com.kindleautox.media play -i "$$" -s "${replaymediafile}" >>"${LOGFILE}" 2>&1
                lipc-wait-event com.kindleautox.media "done$$" 2>>"${LOGFILE}"
            fi
        )&
    fi
}

# usage: fn_ondisconnectresult data
fn_ondisconnectresult() {
    data="$1"

    # It's not safe to play media while we are servicing
    # this event.
    safetoplaymedia=false

    # We invoke the following logic in the background (see & after the
    # closing parenthesis) because we don't want to hold up the processing
    # of events while btconnect.sh sleeps, giving the kindle time to
    # respond.
    (
        # Don't inherit the parent's signal handlers.
        trap - EXIT SIGINT SIGTERM

        # Force the kindle to reconnect to the Bluetooth device so
        # it is ready for the next time we want to use it and so
        # the device itself does not go into a deep sleep. Battery powered
        # speakers tend to go into a deep sleep if nothing is connected to
        # them, even when plugged in.
        #
        # If we were in the middle of playing media, terminate it
        # because it most likely has been affected by the loss of
        # connection with the Bluetooth device. We will restart it
        # automatically below.
        if [[ "${playingmedia}" == true ]]
        then
            replaymediafile="${mediafile}"
            fn_onmediastop
        fi

        # ${data} for this event looks like:
        #
        # 0 "EWA Audio A106Pro" "49:41:7E:DB:AD:F3"
        status=$(echo "${data}" | cut '-d ' -f1)
        BTDEVICE_NAME=$(echo "${data}" | cut '-d"' -f2)
        BTDEVICE_ADDR=$(echo "${data}" | cut '-d"' -f4)
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "bluetooth device disconnected, status = ${status}, btdevname = ${BTDEVICE_NAME}, btdevaddr = ${BTDEVICE_ADDR}" 2>>"${LOGFILE}"; fi

        # The following handles reconnecting only if in screensaver
        # mode and connected to a power source.
        "${INSTALLROOT}/bin/btconnect.sh" "${pluggedin}" "${BTDEVICE_NAME}" "${BTDEVICE_ADDR}" 2>>"${LOGFILE}"
        if [[ $? -ne 0 ]]
        then
            # We could not establish a connection. The device might
            # be asleep/powered off. An error would have already been
            # reported to stderr.
            replaymediafile=""
            exit 1
        fi

        lipc-send-event com.kindleautox.media safetoplay -s true >>"${LOGFILE}" 2>&1

        # If we were playing media at the time we received this event
        # media playback was probably interrupted. Repeat it if so.
        if [[ ! -z "${replaymediafile}" ]]
        then
            if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "replaying media, mediafile = \"$mediafile\"" 2>>"${LOGFILE}"; fi
            lipc-send-event com.kindleautox.media play -i $$ -s "${replaymediafile}" >>"${LOGFILE}" 2>&1
            lipc-wait-event com.kindleautox.media "done$$" 2>>"${LOGFILE}"
            replaymediafile=""
        fi
    )&
}

# usage: fn_onsafetoplay data
fn_onsafetoplay() {
    data="$1"

    # ${data} for this event is of the format:
    #
    # "true" | "false"
    safetoplaymedia=$(echo "${data}" | cut '-d"' -f2)
}

# usage: fn_onmediaplaywhenpluggedinandinscreensavermode data
fn_onmediaplaywhenpluggedinandinscreensavermode() {
    local data="$1"

    # ${data} for this event is of the format:
    #
    # clientpid "mediafile"
    #
    # where
    #
    # clientpid = the client's pid
    # mediafile = the media file to play
    local clientpid=$(echo "${data}" | cut '-d ' -f1)
    local mediafile2=$(echo "${data}" | cut '-d"' -f2)

    if [[ "${pluggedin}" == true ]]
    then
        keepalive=true
    else
        # The following gives you a chance to use the kindle
        # without plugging it in right away. We also use this
        # just in case we've been started in a situation where we
        # can't determine if the kindle is plugged in, such as
        # being started when the kindle is plugged in but discharging.
        keepalive=$(fn_kindle_keepalive "${pluggedin}" 2>>"${LOGFILE}")
    fi

    if [[ "${keepalive}" == true && "${inscreensavermode}" == true ]]
    then
        fn_onmediaplay "${data}"
    else
        info="either not in plugged in or not in screen saver mode; ignoring request to play ${mediafile2}"
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "${info}" 2>>"${LOGFILE}"; fi
        lipc-send-event com.kindleautox.media "done${clientpid}" -i 1 -s "${info}" >>"${LOGFILE}" 2>&1
    fi
}

# Play the given media file. Ignore if we are actively playing
# another.
#
# usage: fn_onmediaplay data
fn_onmediaplay() {
    local data="$1"

    # ${data} for this event is of the format:
    #
    # clientpid "mediafile"
    #
    # where
    #
    # clientpid = the client's pid.
    # mediafile = the media file to play.
    #
    local clientpid=$(echo "${data}" | cut '-d ' -f1)
    local mediafile2=$(echo "${data}" | cut '-d"' -f2)

    if [[ "${playingmedia}" == true ]]
    then
        info="actively playing ${mediafile}; ignoring request to play ${mediafile2}"
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "${info}" 2>>"${LOGFILE}"; fi
        lipc-send-event com.kindleautox.media "done${clientpid}" -i 1 -s "${info}" >>"${LOGFILE}" 2>&1
    elif [[ -f "${mediafile2}" ]]
    then
        playingmedia=true
        mediaclientpid="${clientpid}"
        mediafile="${mediafile2}"
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "playing ${mediafile}" 2>>"${LOGFILE}"; fi

        # Spawn a subshell so we can terminate the inner player
        # subshell (what invokes ${PLAYER}) when we need to
        # stop playing prematurely, such as seeing another
        # play event while we are still processing another,
        # or if we are terminated while playing media.
        #
        # We use \${varname} variable references in key places
        # below so the parent shell doesn't substitute them
        # for their value as it spawns the subshell.
        # ($${varname} is not supported by the kinde's busybox
        # version of bash). We want these variables evaluated
        # as the subshell executes. There are some variables,
        # such as , where its ok if the parent
        # level substitution is ok.
        #
        # If the variable is within normal double quotes (not
        # escaped ones), the \$ escape is not needed and will
        # actually cause problems because the variable won't
        # be expanded at runtime. The escape is only needed
        # if the variable does not appear between normal
        # double quotes or it appears between escaped double
        # quotes.
        (
            # OUTERPLAYERSUBSHELL
            #
            # Don't inherit the parent's signal handlers.
            trap "fn_onouterplayersubshellexit;" EXIT

            # Terminate our children if we are killed.
            #
            # Kill our player subshell if we are killed.
            # SIGINT and SIGTERM are not sent automatically
            # to our children.
            trap "fn_onouterplayersubshellabort;" SIGTERM SIGINT

            if [[ "${safetoplaymedia}" == true ]]
            then
                # Ensure that the configured Bluetooth audio device is connected.
                "${INSTALLROOT}/bin/btconnect.sh" "${pluggedin}" "${BTDEVICE_NAME}" "${BTDEVICE_ADDR}" 2>>"${LOGFILE}"
                if [[ $? -ne 0 ]]
                then
                    # We could not establish a connection. The device might
                    # be asleep/powered off. An error would have already been
                    # reported to stderr.
                    exit 1
                fi
            else
                # We must be in the middle of restoring our
                # Bluetooth connection or trying to prevent
                # the kindle from suspending.
                #
                # Simulate a do-while statement to check for things
                # to change.
                if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "waiting for it to become safe to play" 2>>"${LOGFILE}"; fi
                count=0
                while
                    sleep 5
                    count=$((count + 1))
                    [[ "${safetoplaymedia}" != true && "${count}" -le 5 ]]
                do true; done

                if [[ "${safetoplaymedia}" != true ]]
                then
                   fn_logger_logerr "${CMD}" "it's not safe to play media after 25 seconds, exiting" 2>>"${LOGFILE}"
                   exit 1
                fi
            fi

            if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "spawning subshell to invoke ${PLAYER}, mediafile = ${mediafile}" 2>>"${LOGFILE}"; fi

            # Spawn another subshell to invoke ${PLAYER} so we can
            # terminate ${PLAYER} if we are killed.
            (
                # INNERPLAYERSUBSHELL
                #
                # Don't inherit the parent's signal handlers.
                trap - EXIT

                # Kill ${PLAYER} if we are killed.
                trap "fn_oninnerplayersubshellabort;" SIGTERM SIGINT

                # We send the mediapayerfinished event as part of
                # this subshell in case we need to stop the player
                # prematurely, such as when we see another play event.
                # If we kill the player, we don't want to send the
                # mediaplayerfinished message.
                #
                # Invoke the player in the background so we can
                # get its pid. We need this so we can kill it if
                # we ourself are terminated.
                (
                   # PLAYER
                   # Don't inherit the parent's signal handlers.
                   trap - EXIT SIGINT SIGTERM
                   exec "${PLAYER}" "${mediafile}" >/dev/null 2>&1
                ) &
                # Quotes needed below so $! isn't interpreted until this
                # subshell runs.
                mediaplayerpid="$!"
                wait
            ) &
            # Quotes needed below so $! isn't interpreted until this
            # subshell runs.
            mediainnerplayersubshellpid="$!"
            wait
        )&
        mediaouterplayersubshellpid=$!
    else
       error="media file does not exist, file = \"${mediafile2}\""
       fn_logger_logerr "${CMD}" "${error}" 2>>"${LOGFILE}"
       lipc-send-event com.kindleautox.media "done${clientpid}" -i 2 -s "${error}" >>"${LOGFILE}" 2>&1
    fi
}

fn_onmediastop() {
    if [[ "${playingmedia}" == true ]]
    then
        # Don't hold up other events while we terminate playing.
        # More specifically, we could be called to stop playing
        # while servicing a "ready to suspend" event and cannot
        # afford to wait for playing to stop. We only have a few
        # seconds to respond to the Kindle's framework if we
        # want to delay the suspension.
        #
        # Our outer player subshell (see OUTERPLAYERSUBSHELL
        # comment) will dispatch a "playingcomplete" event
        # as it exits. Clients, including a new outer player
        # subshell that is terminating an existing player before
        # playing new media, should wait for the "playingcomplete"
        # event before proceeding.
        (
            # Don't inherit the parent's signal handlers.
            trap - EXIT SIGINT SIGTERM

            if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "terminating media player, pid = ${mediaplayerpid}" 2>>"${LOGFILE}"; fi
            if [[ ! -z "${mediaouterplayersubshellpid}" ]]
            then
                kill "${mediaouterplayersubshellpid}"
            fi
        )&
    fi
}

fn_onmediaplayingcomplete() {
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "media player finished playing \"${mediafile}\"" 2>>"${LOGFILE}"; fi

    clientpid=${mediaclientpid}
    
    playingmedia=false
    mediaouterplayersubshellpid=""
    mediainnerplayersubshellpid=""
    mediaplayerpid=""
    mediaclientpid=""
    mediafile=""

    # Tell the client that we are done.
    lipc-send-event com.kindleautox.media "done${clientpid}" -i 0 >>"${LOGFILE}" 2>&1
}

# usage: fn_onshowimage data
fn_onshowimage() {
    local data="$1"

    # ${data} for this event is of the format:
    #
    # clientpid "png_file"
    #
    # where:
    #
    # clientpid = the client's pid
    # png_file = is the image to show on the screen.
    local clientpid=$(echo "${data}" | cut '-d ' -f1)
    image=$(echo "${data}" | cut '-d"' -f2)
    if [[ "${inscreensavermode}" == true ]]
    then
        # Don't draw if we are already drawing.
        #
        if [[ "${drawing}" == true ]]
        then
            info="actively showing ${image}; ignoring request to show ${image}"
            if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "${info}" 2>>"${LOGFILE}"; fi
            lipc-send-event com.kindleautox.screensaver "showimagedone${clientpid}" -i 1 -s "${info}" >>"${LOGFILE}" 2>&1
        elif [[ -f "${image}" ]]
        then
            drawing=true
            drawingclientpid="${clientpid}"
            drawingcontent="${image}"

            # Display the image in the background so we don't hold up
            # processing of other events.
            #
            # The following can be found in /etc/upstart/functions (f_display).
            (
                # Don't inherit the parent's signal handlers.
                trap - EXIT

                trap "fn_onshowimageabort;" SIGINT SIGTERM
                
                # Clear the screen.
                # Note that eips(1) is being deprecrated/replaced by eips_v2.
                #
                # We redirect stdout to /dev/null to avoid seeing the
                # following messages:
                #
                # clear:
                # update_to_display: update_mode=PARTIAL, wave_mode=2 inverted=0
                #
                # render image:
                # update_to_display: update_mode=FULL, wave_mode=2 inverted=0
                if [[ "${HASEIPS_V2}" == true ]]
                then
                    "${EIPS}" clear >/dev/null 2>>"${LOGFILE}"
                else
                    "${EIPS}" -c >/dev/null 2>>"${LOGFILE}"
                fi

                # We need to give the kindle framework time to finish
                # servicing the request above. This sleep is taken
                # directly from /etc/upstart/functions:f_display.
                sleep 1

                # Show the image.
                if [[ "${HASEIPS_V2}" == true ]]
                then
                    "${EIPS}" png "${image}" -f -W >/dev/null 2>>"${LOGFILE}"
                else
                    "${EIPS}" -f -g "${image}" >/dev/null 2>>"${LOGFILE}"
                fi

                # Tell the parent that we are done.
                #
                # Without the sleep below we end up seeing a
                # "Failed to open LIPC" message on older/slower devices.
                sleep 1
                
                # Tell the parent process that we are done.
                lipc-send-event com.kindleautox.screensaver showimagecomplete >>"${LOGFILE}" 2>&1
            )&
            drawingpid=$!
        else
            error="image file does not exist, file = \"${image}\""
            fn_logger_logerr "${CMD}" "${error}" 2>>"${LOGFILE}"
            lipc-send-event com.kindleautox.screensaver "showimagedone${clientpid}" -i 2 -s "${error}" >>"${LOGFILE}" 2>&1
        fi
    else
        info="not in screensaver mode; ignoring request to show ${image}"
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "${info}" 2>>"${LOGFILE}"; fi
        lipc-send-event com.kindleautox.screensaver "showimagedone${clientpid}" -i 1 -s "${info}" >>"${LOGFILE}" 2>&1
    fi
}

# usage: fn_onshowimagecomplete
fn_onshowimagecomplete() {
    local clientpid="${drawingclientpid}"
    drawing=false
    drawingpid=""
    drawingclientpid=""
    drawingcontent=""

    # Tell the client that we are done.
    lipc-send-event com.kindleautox.screensaver "showimagedone${clientpid}" -i 0 >>"${LOGFILE}" 2>&1
}

# usage: fn_onshowtime data
fn_onshowtime() {
    local data="$1"

    # ${data} for this event is of the format:
    #
    # clientpid
    #
    # where:
    #
    # clientpid = the client's pid
    local clientpid=$(echo "${data}" | cut '-d ' -f1)

    if [[ "${inscreensavermode}" == true ]]
    then
        # Don't draw if we are already drawing.
        if [[ "${drawing}" == true ]]
        then
            info="actively drawing on screen; ignoring request to show the current date/time"
            if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "${info}" 2>>"${LOGFILE}"; fi
            lipc-send-event com.kindleautox.screensaver "showtimedone${clientpid}" -i 1 -s "${info}" >>"${LOGFILE}" 2>&1
        else
            drawing=true
            drawingclientpid="${clientpid}"
            drawingcontent="date/time"

            # Display the time in the background so we don't hold up
            # processing of other events.
            (
                # Don't inherit the parent's signal handlers.
                trap - EXIT

                trap "fn_onshowtimeabort;" SIGINT SIGTERM

                # We place the $(date) invocation within double quotes
                # so the subshell evaluates the $(date) expression
                # instead of the parent. Without this the parent will
                # evaluate the expression prior to invoking this subshell.
                time="$(date +'%m.%d.%Y %H:%M')"
                RFONT="/usr/java/lib/fonts/Futura-Bold.ttf"

                # We redirect ${FBINK} stdout and stderr to /dev/null to
                # avoid seeing any output. This command sends a lot of
                # normal messages to stderr.
                if [[ ! -z "${FBINK}" ]]
                then
                    "${FBINK}" -t "regular=$RFONT,size=17,top=0,padding=HORIZONTAL,format" --center "${time}" >/dev/null 2>&1
                # else, do nothing - not on hardware that we know about.
                fi
                
                # Tell the parent process that we are done.
                lipc-send-event com.kindleautox.screensaver showtimecomplete >>"${LOGFILE}" 2>&1
            )&
            drawingpid=$!
        fi
    else
        info="not in screensaver mode; ignoring request to show the current date/time"
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "${info}" 2>>"${LOGFILE}"; fi
        lipc-send-event com.kindleautox.screensaver "showtimedone${clientpid}" -i 1 -s "${info}" >>"${LOGFILE}" 2>&1
    fi
}

# usage: fn_onshowtimecomplete
fn_onshowtimecomplete() {
    local clientpid="${drawingclientpid}"
    drawing=false
    drawingpid=""
    drawingclientpid=""
    drawingcontent=""

    # Tell the client that we are done.
    lipc-send-event com.kindleautox.screensaver "showtimedone${clientpid}" -i 0 >>"${LOGFILE}" 2>&1
}

# We use the information contained within powerd's battStateInfo
# property to determine if the kindle is charging, implying that
# it's plugged in.
#
# The battStateInfo property value looks something like this:
#
# {"battInfo":[{"type":"PRIMARY","cap":99,"charging":true}],"state":"STANDALONE_USB","aggr_batt":99,"usb":true}
#
# where:
#
# cap = the battery's capacity.
# charging/usb = true if the kindle is charging, false if not.
#
# In practice, when "charging" goes to false, so does "usb", even if
# still connected to a power source. Therefore, we cannot rely on
# "usb" to know if the kindle is connected to a power source.
#
# To avoid stressing the battery, the kindle will allow the
# battery to discharge even when connected to a power source.
# Therefore,  we must rely on the battery level in addition to
# the "charging" member. I've seen my Kindle Signature 12th Gen
# discharge to 97%, and sometimes lower, before it begins
# charging again.
#
# Note that the kindle's busybox version of sed(1) does not support
# '[0-9]+' (for one or more), so we had to use '[0-9][0-9]*'
# instead in the sed(1) commands below.

pluggedin=false
battStateInfo=$(lipc-get-prop com.lab126.powerd battStateInfo 2>/dev/null)
if [[ -z "${battStateInfo}" ]]
then
    # The kindle doesn't support the battStateInfo property. Try the older
    # "status" property.
    battStateInfo=$(lipc-get-prop com.lab126.powerd status 2>/dev/null | tr '\n' ' ')
    pluggedinpattern='.*Charging: Yes.*'
else
    pluggedinpattern='.*"charging":true.*'
fi

# Do not include quotes around the pattern. Doing so causes the matching
# operator to take the pattern as literal string instead of a pattern.
#
# The kindle's busybox version of bash(1) does not support the
# +~ operator. So we use expr as an alternative.
#if [[ "${battStateInfo}" =~ ${pluggedinpattern} ]]; then pluggedin=true; else pluggedin=false; fi
if expr "${battStateInfo}" : "${pluggedinpattern}" >/dev/null; then pluggedin=true; else pluggedin=false; fi

if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "initialized, pluggedin = ${pluggedin}" 2>>"${LOGFILE}"; fi

while read event
do
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "event = \"${event}\"" 2>>"${LOGFILE}"; fi
    case "${event}" in
    "Event Source"*|"Waiting for events"*)
        # We can ignore these lines. These are reported by lipc-wait-event
        # as confirmation that it is waiting on multiple sources.
        #
        # Example:
        #
        # Event Source: com.lab126.powerd
        # Event Source: com.kindleautox.media
        # Waiting for events...
        #
        # Note that we can see the "Event Source/Waiting for events"
        # messages for different sources at different times.
        # The kindle framework seems to report these messages
        # the first time an event for the given source is
        # dispatched and received.
        continue
        ;;
    esac

    # $event contains all of the event details in the form:
    #
    # source name data
    #
    # where:
    #
    # data can contain multiple values, separated by spaces with string
    # values enclosed in double # quotes.
    #
    # We cannot use IFS substitution because the data
    # may include spaces. Therefore, we just use cut.
    #
    # Note that the data contains an extra space at the end for
    # some reason, so we remove it.
    #
    # Remove leading/trailing whitespace characters.
    event=$(fn_str_trim "${event}")

    source=$(echo "${event}" | cut '-d ' -f1)
    name=$(echo "${event}" | cut '-d ' -f2)
    data=$(echo "${event}" | cut '-d ' -f3-)

    if [[ -z "${data}" ]]; then data='""'; fi	# So our info msgs look ok.

    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "${source} ${name} recvd, data = ${data}, inscreensavermode = ${inscreensavermode}" 2>>"${LOGFILE}"; fi

    case "${event}" in

    # The readcfg event has no parameters, so don't include a space
    # after the event name.
    "com.kindleautox.gen readcfg"*)
        fn_onreadcfg
        ;;
        
    # The stop event has no parameters, so don't include a space
    # after the event name.
    "com.kindleautox.gen stop"*)
        fn_onexit
        ;;

    # The usbdisconnected event has no parameters, so don't include a space
    # after the event name.
    "com.lab126.powerd usbdisconnected"*)
        fn_onusbdisconnected
        ;;

    # The charging event has no parameters, so don't include a space
    # after the event name.
    "com.lab126.powerd charging"*)
        fn_oncharging
        ;;

    "com.lab126.powerd battStateInfoChanged "*)
        fn_onbattStateInfoChanged
        ;;

    "com.lab126.powerd goingToScreenSaver "*)
        fn_ongoingtoscreensaver
        ;;

    "com.lab126.powerd outOfScreenSaver "*)
        fn_onoutofscreensaver
        ;;

    "com.lab126.powerd readyToSuspend "*)
        fn_onreadytosuspend
        ;;

    "com.lab126.powerd suspending "*)
        # We should never get here.
        lipc-set-prop com.lab126.powerd wakeUp 1
        sleep 5
        ;;
 
    "com.lab126.btfd Disconnect_Result "*)
        fn_ondisconnectresult "${data}"
        ;;

    # Used by subshells to tell the parent that it's safe to play.
    "com.kindleautox.media safetoplay "*)
        fn_onsafetoplay "${data}"
        ;;

    "com.kindleautox.media playwhenpluggedinandinscreensavermode "*)
        fn_onmediaplaywhenpluggedinandinscreensavermode "${data}"
        ;;

    # Normal media player event state machine:
    #
    # play->playingcomplete->done${clientpid}
    #
    # State machine when already playing media:
    #
    # play->done${clientpid}
    #
    # State machine when we need to stop playing prematurely
    # for some reason, like attempting to play another media
    # file while one is already playing.
    # 
    # play->stop->playingcomplete->done${clientpid}
    #
    # The "done" event is meant for clients that
    # invoke "lipc-send-event com.kindleautox.media play".
    #
    # ${clientpid} = the pid given to us by the client in
    # its play event.
    #
    # The reason for appending the client's pid is so the
    # client can listen for its specific "done" event.
    "com.kindleautox.media play "*)
        fn_onmediaplay "${data}"
        ;;

    # The stop event has no parameters, so don't include a space
    # after the event name. We support the stop event in case we ever
    # want to stop playing media from other software modules.
    "com.kindleautox.media stop"*)
        fn_onstop
        ;;

    # The playingcomplete has no parameters, so don't include a space
    # after the event name.
    #
    # Used by subshell to tell the parent that playing is complete.
    "com.kindleautox.media playingcomplete"*)
        fn_onmediaplayingcomplete
        ;;

    # Screensaver showimage state machine:
    #
    # This state machine applies even if an interrupt is received
    # by the subshell that displays the image.
    #
    # showimage->showimagecomplete->showimagedone${clientpid}
    #
    # State machine when already actively showing an image:
    #
    # showimage->showimagedone${clientpid}
    #
    # where:
    #
    # The "showimagedone" event is meant for clients that
    # invoke "lipc-send-event com.kindleautox.screensaver showimage".
    #
    # ${clientpid} = the pid given to us by the client in
    # its showimage request.
    #
    # The reason for appending the client's pid is so the
    # client can listen for its specific "done" event.
    "com.kindleautox.screensaver showimage "*)
        fn_onshowimage "${data}"
        ;;

    # The showimagecomplete event has no parameters, so don't include a space
    # after the event name.
    #
    # Used by subshell to tell the parent that drawing is complete.
    "com.kindleautox.screensaver showimagecomplete"*)
        fn_onshowimagecomplete
        ;;

    # Screensaver showtime state machine:
    #
    # This state machine applies even if an interrupt is received
    # by the subshell that displays the time.
    #
    # showtime->showtimecomplete->showtimedone${clientpid}
    #
    # State machine when already actively showing the time:
    #
    # showimage->showimagedone${clientpid}
    #
    # where:
    #
    # The "showtimedone" event is meant for clients that
    # invoke "lipc-send-event com.kindleautox.screensaver showtime".
    #
    # ${clientpid} = the pid given to us by the client in
    # its showtime request.
    #
    # The reason for appending the client's pid is so the
    # client can listen for its specific "done" event.
    "com.kindleautox.screensaver showtime "*)
        fn_onshowtime "${data}"
        ;;

    # The showtimecomplete event has no parameters, so don't include a space
    # after the event name.
    #
    # Used by subshell to tell the parent that drawing is complete.
    "com.kindleautox.screensaver showtimecomplete"*)
        fn_onshowtimecomplete
        ;;

    *)
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "unsupported event (${event}) recvd" 2>>"${LOGFILE}"; fi
        ;;

    esac
done


