#!/bin/bash
#
# Ensure that the kindle is connected to the given Bluetooth device, but
# only if it's ok to keep the kindle alive (see fn_kindle_keepalive()).
# All logging and error output is sent to stderr, which the caller can
# redirect as neccessary.
#
# usage: btconnect.sh pluggedin [name macadddr]
#
# where:
#
# pluggedin = a "true/false" parameter that if "true" states
#     that the kindle is plugged into a power source.
#     Otherwise, we presume "false" and rely on the
#     kindle's battery statistics.
#
# name = the name of the bluetooth device to connect to, as taken from
#     /var/local/zbluetooth/bt_config.conf
#
# macaddr = the MAC address of the blootooth device to connect to,
#     as taken from the same file.
#
# If "name" and/or "macaddr" are missing, we use ${BTDEVICE_NAME}
# and ${BTDEVICE_ADDR} from ../config/config.sh instead.
#
# Note that this script will sleep 5 seconds before returning
# to give the kindle time to respond.
IFS=" 	
"
PATH="/bin:/usr/bin:/sbin:/usr/sbin"
CMD="${0##*/}"
CMDPATH="${0%/*}"
if [[ "${CMDPATH}" == "$0" ]]; then CMDPATH="."; fi
CMDPATH=$(realpath "${CMDPATH}")
INSTALLROOT=$(realpath "${CMDPATH}/..")

source "${INSTALLROOT}/config/config.sh"
source "${INSTALLROOT}/config/btconfig.sh"
source "${INSTALLROOT}/lib/sh/fn_logger.sh"
source "${INSTALLROOT}/lib/sh/fn_kindle.sh"

if [[ ! -d "/var/local/zbluetooth" ]]
then
    # We are on a device that doesn't support Bluetooth connections.
    exit 0
fi

usage() {
   cat <<!
usage: btconnect.sh pluggedin [name macadddr]

where:

pluggedin = a "true/false" parameter that if "true" states
    that the kindle is plugged into a power source.
    Otherwise, we presume "false" and rely on the
    kindle's battery statistics.

name = the name of the bluetooth device to connect to, as taken from
    /var/local/zbluetooth/bt_devices.xml (pre 5.17.1 versions) or
    /var/local/zbluetooth/bt_config.conf (5.17.1 and above)

macaddr = the MAC address of the blootooth device to connect to,
    as taken from the same file.

If "name" and/or "macaddr" are missing, we use \${BTDEVICE_NAME}
and \${BTDEVICE_ADDR} from ../config/config.sh instead.
!
}

if [[ $# == 0 ]]
then
    usage
    exit 1
fi

pluggedin="$1"
if [[ "${pluggedin}" != true ]]
then
    pluggedin=false
fi

if [[ $# -gt 2 ]]
then
    connectToDevName="$2"
    connectToDevAddr="$3"
else
    connectToDevName="${BTDEVICE_NAME}"
    connectToDevAddr="${BTDEVICE_ADDR}"
fi

# Force the MAC address to all uppercase letters.
# "lipc-set-prop com.lab126.btfd Connect" requires this.
#
# The kindle's buxybox version of bash does not support the
# use of "^^" parameter expansion. So we use tr(1) instead.
#connectToDevAddr="${connectToDevAddr^^}"
connectToDevAddr=$(echo ${connectToDevAddr} | tr '[:lower:]' '[:upper:]')

LOGFILE="${LOGDIR}/btconnect"

# Note that the BTconnectedDevName property exists only when a device
# is connected.
connectedDevName=$(lipc-get-prop com.lab126.btfd BTconnectedDevName 2>/dev/null)

if [[ "${connectToDevName}" != "${connectedDevName}" ]]
then
    # Only establish a bluetooth connection if we are connected to a power
    # source.
    if [[ "$(fn_kindle_keepalive \"${pluggedin}\")" == true ]]
    then
        # Ensure that the desired Bluetooth audio device is connected.
        if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "connecting to \"${connectToDevName}\""; fi
        lipc-set-prop com.lab126.btfd Connect "${connectToDevAddr}"

	# The following does not seem to have any effect. I thought maybe
        # it might help keep the Bluetooth connection alive all the time.
        #
        # lipc-set-prop com.lab126.btfd ensureBTconnection 1

        # Give the kindle time to respond.
        sleep 5

        connectedDevName=$(lipc-get-prop com.lab126.btfd BTconnectedDevName 2>/dev/null)
        if [[ "${connectToDevName}" != "${connectedDevName}" ]]
        then
            fn_logger_logerr "${CMD}" "could not establish connection to \"${connectToDevName}; is the device on?\""
            exit 1
        fi
    fi
else
    if [[ "${DEBUG}" == 1 ]]; then fn_logger_loginfo "${CMD}" "already connected to \"${connectToDevName}\""; fi
fi
