#!/usr/bin/env python
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__ = 'GPL v3'
__copyright__ = '2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025 Doitsu'

# standard libraries
import zipfile
import re, os, locale, sys, tempfile, socket, urllib, json, shutil
from os.path import expanduser, basename
from datetime import datetime, timedelta

# Qt
from qt.core import (
    QTextEdit, QDockWidget, QApplication, QAction, 
    QFileDialog, QMessageBox, QDialog, QListWidget, QVBoxLayout, 
    QListWidgetItem, QDialogButtonBox, Qt, QEventLoop, QBrush, QColor
)

# Calibre libraries
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.plugin import Tool
from calibre.gui2.tweak_book.ui import Main
from calibre.utils.config import config_dir, JSONConfig
from calibre.constants import iswindows, islinux, isosx

# make sure the plugin will work with the Python 3 version of Calibre
if sys.version_info[0] == 2:
    from urllib import urlopen, urlretrieve, unquote
else:
    from urllib.request import urlopen, urlretrieve
    from urllib.parse import unquote

# get epubcheck.jar version number
def get_epc_version(epc_path):
    version = ''

    # make sure that epubcheck.jar actually exists
    if os.path.exists(epc_path):

        # read .jar file as zip file
        archive = zipfile.ZipFile(epc_path)

        # make sure that pom.xml exists
        if 'META-INF/maven/org.w3c/epubcheck/pom.xml' in archive.namelist():
            pom_data = archive.read('META-INF/maven/org.w3c/epubcheck/pom.xml')
            archive.close()

            # parse pom.xml as ElementTree
            from xml.etree import ElementTree as ET
            root = ET.fromstring(pom_data)
            tag = root.find("*//{http://maven.apache.org/POM/4.0.0}tag")
            # look for <tag>
            if tag is not None:
                version = tag.text
            else:
                # look for <version>
                project_version = root.find("{http://maven.apache.org/POM/4.0.0}version")
                if project_version is not None:
                    version = 'v' + project_version.text
        else:
            print('pom.xml not found!') 
    else:
        print('epubcheck.jar not found!')

    return version

# get latest EPUBCheck version
def latest_epc_version(github_url):
    latest_version = ''
    browser_download_url = ''
    if is_connected():
        # check for github updates 
        response = urlopen(github_url).read().decode('utf-8')
        parsed_json = json.loads(response)
        latest_version = parsed_json[0]['tag_name']
        browser_download_url = parsed_json[0]['assets'][0]['browser_download_url']
    return latest_version, browser_download_url

# code provided by DiapDealer
def is_connected():
    try:
        sock = socket.create_connection(('8.8.8.8', 53), 1)
        sock.close()
        return True
    except:
        pass

# code provided by DiapDealer
def string_to_date(datestring):
    return datetime.strptime(datestring, "%Y-%m-%d %H:%M:%S.%f")

# DiapDealer's temp folder code
from contextlib import contextmanager

@contextmanager
def make_temp_directory():
    import tempfile
    import shutil
    temp_dir = tempfile.mkdtemp()
    yield temp_dir
    shutil.rmtree(temp_dir)

# jar wrapper for epubcheck
def jarWrapper(*args):
    import subprocess
    startupinfo = None

    # stop the windows console popping up every time the prog is run
    if iswindows:
        startupinfo = subprocess.STARTUPINFO()
        startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        startupinfo.wShowWindow = subprocess.SW_HIDE

    process = subprocess.Popen(list(args), stdout=subprocess.PIPE, stderr=subprocess.PIPE, startupinfo=startupinfo)
    ret = process.communicate()
    returncode = process.returncode
    return ret, returncode


# get JVM bitness (https://stackoverflow.com/questions/2062020)
def get_arch(java_path):
    arch = '64'
    args = [java_path, '-XshowSettings:properties', '-version']
    ret, retcode = jarWrapper(*args)
    arch_pattern = re.compile(r'sun.arch.data.model = (\d+)')
    arch_info = arch_pattern.search(ret[1].decode('utf-8'))
    if arch_info:
        if len(arch_info.groups()) == 1:
            arch = arch_info.group(1)
            print('Java bitness detected:', arch)
    else:
        print('Java bitness not detected!' )
    return arch


# parses the json file
def parse_epubcheck_json(json_output):
    try:
        data = json.loads(json_output)
    except Exception as e:
        print("Failed to parse JSON output: {0}".format(e))
        print("Raw output was:\n", json_output)
        sys.exit(1)

    # EPUBCheck won't return the number of INFO messages in JSON mode :(
    nInfo = 0
    checker = data.get("checker", {})
    publication = data.get("publication", {})
    messages = data.get("messages", [])

    checkerVersion = checker.get("checkerVersion", "")
    nFatal = checker.get("nFatal", 0)
    nError = checker.get("nError", 0)
    nWarning = checker.get("nWarning", 0)
    nUsage = checker.get("nUsage", 0)
    ePubVersion = publication.get("ePubVersion", "")

    error_messages = []
    for msg in messages:
        # Extract locations
        locations = msg.get("locations", [])
        severity = msg.get("severity", "")
        # increase INFO message counter
        if severity == 'INFO': nInfo += 1
        ID = msg.get("ID", "")
        message = msg.get("message", "")
        for loc in locations:
            line = loc.get("line", "")
            column = loc.get("column", "")
            path = unquote(loc.get("path", ""))
            file_name = os.path.basename(path)
            if line == "-1" or column == "-1" or path.endswith('.epub'): path = 'NA'
            loc_message = "{0} Line:{1} Col:{2} {3}({4}): {5}".format(file_name, line, column, severity, ID, message)
            error_messages.append((path, line, column, severity, loc_message))

    return {
        "checkerVersion": checkerVersion,
        "nFatal": nFatal,
        "nError": nError,
        "nWarning": nWarning,
        "nUsage": nUsage,
        "nInfo" : nInfo,
        "ePubVersion": ePubVersion,
        "error_messages": error_messages,
        "publication" : publication
    }


# format extended publication information
def format_publication_info(publication):
    lines = []
    for key, value in publication.items():
        if value is None or value == []:
            continue  # Skip keys with None or empty list values
        if isinstance(value, list):
            # Format list items, one per line, indented
            pretty_value = "\n    " + "\n    ".join(str(item) for item in value)
        else:
            pretty_value = str(value)
        lines.append(f"{key}: {pretty_value}")
    return "\n\n" + "\n".join(lines)


class DemoTool(Tool):

    #: Set this to a unique name it will be used as a key
    name = 'epub-check'

    #: If True the user can choose to place this tool in the plugins toolbar
    allowed_in_toolbar = True

    #: If True the user can choose to place this tool in the plugins menu
    allowed_in_menu = True

    def create_action(self, for_toolbar=True):
        # Create an action, this will be added to the plugins toolbar and
        # the plugins menu
        ac = QAction(get_icons('images/icon.png'), 'Run EpubCheck', self.gui)  # noqa
        if not for_toolbar:
            # Register a keyboard shortcut for this toolbar action. We only
            # register it for the action created for the menu, not the toolbar,
            # to avoid a double trigger
            self.register_shortcut(ac, 'epub-check-tool', default_keys=('Ctrl+Shift+Alt+G',))
        ac.triggered.connect(self.ask_user)
        return ac

    def ask_user(self):

        #-----------------------------------
        # define EPUBCheck paths and URL
        #-----------------------------------
        epubcheck_dir = os.path.join(config_dir, 'plugins', 'EPUBCheck')
        if not os.path.isdir(epubcheck_dir):
            os.makedirs(epubcheck_dir)
        epc_path = os.path.join(epubcheck_dir, 'epubcheck.jar')
        epc_lib_dir = os.path.join(epubcheck_dir, 'lib')
        github_url = 'https://api.github.com/repos/w3c/epubcheck/releases'

        #----------------------------------------
        # get user preference file
        #----------------------------------------
        prefs = JSONConfig('plugins/EpubCheck')

        #----------------------------------------
        # set default preferences
        #----------------------------------------
        if prefs == {}:
            prefs.set('extra', False)
            prefs.set('close_cb', False)
            prefs.set('clipboard_copy', False)
            prefs.set('usage', False)
            prefs.set('github', True)
            prefs.set('last_time_checked', str(datetime.now() - timedelta(days=7)))
            prefs.set('check_interval', 7)
            prefs.set('java_path', 'java')
            prefs.set('is32bit', get_arch('java') == '32')
            prefs.commit()

        #---------------------------
        # get preferences
        #---------------------------
        debug = prefs.get('debug', False)
        extra = prefs.get('extra', False)
        locale = prefs.get('locale', None)
        close_cb = prefs.get('close_cb', False)
        clipboard_copy = prefs.get('clipboard_copy', False)
        usage = prefs.get('usage', False)
        github = prefs.get('github', True)
        last_time_checked = prefs.get('last_time_checked', str(datetime.now() - timedelta(days=7)))
        check_interval = prefs.get('check_interval', 7)
        java_path = prefs.get('java_path', 'java').replace('\\\\', '/').replace('\\', '/')
        is32bit = prefs.get('is32bit', get_arch(java_path) == '32')

        #-----------------------------------------------------
        # create a savepoint
        #----------------------------------------------------
        self.boss.add_savepoint('Before: EPUBCheck')

        #--------------------------------------------------------------------
        # create temp directory and unpack epubcheck files
        #--------------------------------------------------------------------
        with make_temp_directory() as td:
            # write current container to temporary epub
            epub_name = os.path.basename(self.current_container.path_to_ebook)
            if debug: print(epub_name)
            epub_path = os.path.join(td, epub_name)
            self.boss.commit_all_editors_to_container()
            self.current_container.commit(epub_path)

            # check if the EPUBCheck Java files were downloaded
            if not os.path.isdir(epubcheck_dir) or not os.path.isfile(epc_path) or not os.path.isdir(epc_lib_dir):
                epc_missing = True
            else:
                epc_missing = False

            #----------------------------
            # check for EPUBCheck updates
            #----------------------------
            if github or epc_missing:

                # make sure we have an Internet connection
                if is_connected():

                    # compare current date against last update check date
                    time_delta = (datetime.now() - string_to_date(last_time_checked)).days
                    if (time_delta >= check_interval) or epc_missing:

                        # display searching for updates... message
                        if epc_missing:
                            self.gui.show_status_message("No EPUBCheck files found.", 7)
                        else:
                            self.gui.show_status_message("Running update check...", 7)
                        QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)

                        # update time stamp in EpubCheck.json
                        prefs.set('last_time_checked', str(datetime.now()))
                        prefs.commit()

                        # get current epubcheck version from epubcheck.jar
                        epc_version = get_epc_version(epc_path)
                        print('epc_version', epc_version)

                        # get latest version and browser download url 
                        latest_version, browser_download_url = latest_epc_version(github_url)
                        print('latest_version:', latest_version, 'browser_download_url:', browser_download_url)

                        if 'alpha' in browser_download_url or 'beta' in browser_download_url and not epc_missing: browser_download_url = '' # exclude alpha/beta versions

                        # only run the update if a new version is available
                        if latest_version != epc_version and latest_version !='' and browser_download_url != '':
                            answer = QMessageBox.question(self.gui, "EPUBCheck update available", "EPUBCheck {} is available.\nDo you want to download the latest version?".format(latest_version))

                            # update EPUBCheck
                            if answer == QMessageBox.StandardButton.Yes:

                                # create a temp folder
                                with make_temp_directory() as td:
                                    base_name = os.path.basename(browser_download_url)
                                    root_path = os.path.splitext(base_name)[0]
                                    zip_file_name = os.path.join(td, base_name)

                                    # display status message
                                    self.gui.show_status_message("Downloading {}...".format(browser_download_url), 3)
                                    QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)

                                    # download the zip file
                                    urlretrieve(browser_download_url, zip_file_name)

                                    # make sure the file was actually downloaded
                                    if os.path.exists(zip_file_name):

                                        # display status message
                                        self.gui.show_status_message("{} downloaded.".format(browser_download_url), 3)
                                        QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)

                                        # read zip file
                                        # https://stackoverflow.com/questions/19618268/extract-and-rename-zip-file-folder
                                        archive = zipfile.ZipFile(zip_file_name)
                                        files = archive.namelist()
                                        files_to_extract = [m for m in files if (m.startswith(root_path + '/lib') or m == root_path + '/epubcheck.jar')]
                                        archive.extractall(td, files_to_extract)
                                        archive.close()

                                        # temp paths to epubcheck.jar and the /lib folder
                                        temp_epc_path = os.path.join(td, root_path + '/epubcheck.jar')
                                        temp_epc_lib_dir = os.path.join(td, root_path + '/lib')

                                        # make sure the files were actually extracted
                                        if os.path.isdir(temp_epc_lib_dir) and os.path.isfile(temp_epc_path):

                                            epc_missing = False

                                            # delete /lib folder
                                            if os.path.isdir(epc_lib_dir):
                                                shutil.rmtree(epc_lib_dir)

                                            # delete epubcheck.jar
                                            if os.path.exists(epc_path):
                                                os.remove(epc_path)

                                            # move new files to the plugin folder
                                            shutil.move(temp_epc_lib_dir, epubcheck_dir)
                                            shutil.move(temp_epc_path, epubcheck_dir)

                                            # ensure you have execute rights for unix based platforms
                                            if isosx or islinux:
                                                os.chmod(epc_path, 0o744)

                                            # display update successful message
                                            self.gui.show_status_message("EPUBCheck updated to EPUBCheck {}".format(latest_version), 5)
                                            QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)

                                        else:

                                            # display unzip error message
                                            self.gui.show_status_message("EPUBCheck update failed. The EPUBCheck .zip file couldn\'t be unpacked.", 10)
                                            QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
                                    else:

                                        # display download error message
                                        self.gui.show_status_message("EPUBCheck update failed. The latest EPUBCheck .zip file couldn\'t be downloaded", 10)
                                        QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)

                        else:

                            # display miscellanenous internal error messages
                            if latest_version != '':
                                self.gui.show_status_message("No new EPUBCheck version found.", 5)
                                QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
                            elif epc_version == '':
                                self.gui.show_status_message("Current EPUBCheck version not found.", 5)
                                QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
                            else:
                                self.gui.show_status_message("Internal error: update check failed.", 5)
                                QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
                else:

                    # display no Internet error message
                    self.gui.show_status_message("Update check skipped: no Internet.", 5)
                    QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)

            # double-check that the Java files were downloaded
            if epc_missing:
                QMessageBox.critical(self.gui, "EPUBCheck Java files missing!", 'Please re-run the plugin while connected to the Internet.')
                return 

            #-------------------------------------
            # assemble epubcheck parameters
            #-------------------------------------

            # display busy cursor
            QApplication.setOverrideCursor(Qt.WaitCursor)

            # define epubcheck command line parameters
            if is32bit:
                args = [java_path, '-Dfile.encoding=UTF8', '-Xss1024k', '-jar', epc_path, epub_path, '-q']
            else:
                args = [java_path, '-Dfile.encoding=UTF8', '-jar', epc_path, epub_path, '-q']

            # display messages in a different language
            if locale is not None:
                args.extend(['--locale', locale])

            # display usage messages
            if usage:
                args.append('--usage')

            args.extend(['--json', '-'])
            
            if debug:
                print(args)

            # run epubcheck
            self.gui.show_status_message("Running EPUBCheck...", 3)
            QApplication.processEvents(QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
            result, returncode = jarWrapper(*args)
            stdout = result[0].decode('utf-8', errors = 'ignore')
            stderr = result[1].decode('utf-8', errors = 'ignore')

            # check for Java errors
            if returncode == 1 and 'java.lang.' in stderr:
                QApplication.restoreOverrideCursor()
                #QMessageBox.critical(self.gui, "Fatal Java error", stderr)
                max_len = 1000
                msg = stderr if len(stderr) < max_len else stderr[:max_len] + "..." 
                error_dialog(self.gui, "Fatal Java Error", msg, show=True)
                # stack overflow messages are usually caused by 32bit Java binaries
                prefs.set('is32bit', True)
                return

        #--------------------------------------------
        # process output
        #--------------------------------------------
        info = parse_epubcheck_json(stdout)
        checkerVersion = info["checkerVersion"]
        ePubVersion = info["ePubVersion"]
        nFatal = info["nFatal"]
        nError = info["nError"]
        nWarning = info["nWarning"]
        nUsage = info["nUsage"]
        nInfo = info["nInfo"]
        error_messages = info["error_messages"]

        # copy results to the clipboard
        if clipboard_copy:
            if debug:
                QApplication.clipboard().setText(stdout)
            else:
                QApplication.clipboard().setText('\n'.join([t[4] for t in error_messages]))


        if debug:
            print("checkerVersion: {}".format(checkerVersion))
            print("ePubVersion: {}".format(ePubVersion))
            print("nFatal: {}".format(nFatal))
            print("nError: {}".format(nError))
            print("nWarning: {}".format(nWarning))
            print("nUsage: {}\n".format(nUsage))
            print("nInfo: {}\n".format(nInfo))
            print(error_messages)

        #-----------------
        # process errors
        #-----------------
        if any([nFatal, nError, nWarning, nUsage, nInfo]):
            if error_messages != []:
                #---------------------------------------------------------------
                # auxiliary routine for loading the file into the editor
                #---------------------------------------------------------------
                def GotoLine(item):
                    # get list item number
                    current_row = listWidget.currentRow()

                    # get error information
                    filepath, line, col, err_code, message = error_messages[current_row]

                    # go to the file
                    if not os.path.basename(filepath).endswith('NA'):
                        if line != -1:
                            self.boss.edit_file(filepath)
                            editor = self.boss.gui.central.current_editor
                            if editor is not None and editor.has_line_numbers:
                                if col is not None:
                                    editor.editor.go_to_line(line, col=col - 1)
                                else:
                                    editor.current_line = line
                        else:
                            QMessageBox.information(self.gui, "Unknown line number", "EPUBCheck didn't report a line number for this error.")
                    else:
                        QMessageBox.information(self.gui, "Unknown file name", "EPUBCheck didn't report the name of the file that caused this error.")

                #------------------------------------------------------------------------------------------------
                # remove existing EPUBCheck dock and close Check Ebook dock
                #------------------------------------------------------------------------------------------------
                for widget in self.gui.children():
                    if isinstance(widget, QDockWidget) and widget.objectName() == 'epubcheck-dock':
                        #self.gui.removeDockWidget(widget)
                        #widget.close()
                        widget.setParent(None)
                    if isinstance(widget, QDockWidget) and widget.objectName() == 'check-book-dock' and close_cb == True:
                        widget.close()

                #----------------------------------
                # define dock widget layout
                #----------------------------------
                try:
                    is_dark_theme = QApplication.instance().is_dark_theme
                except:
                    is_dark_theme = False
                listWidget = QListWidget()
                l = QVBoxLayout()
                l.addWidget(listWidget)
                dock_widget = QDockWidget(self.gui)
                dock_widget.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea | Qt.BottomDockWidgetArea | Qt.TopDockWidgetArea)
                dock_widget.setObjectName('epubcheck-dock')
                dock_widget.setWindowTitle('EPUBCheck')
                dock_widget.setWidget(listWidget)

                #--------------------------------------------
                # add error messages to list widget
                #--------------------------------------------
                for error_msg in error_messages:
                    filename, line, col, err_code, message = error_msg
                    item = QListWidgetItem(message)

                    # select background color based on severity
                    if err_code.startswith(('ERROR', 'FATAL')):
                        bg_color = QBrush(QColor(255, 230, 230))
                    elif err_code.startswith('WARNING'):
                        bg_color = QBrush(QColor(255, 255, 230))
                    else:
                        bg_color = QBrush(QColor(224, 255, 255))
                    item.setBackground(QColor(bg_color))
                    if is_dark_theme:
                        item.setForeground(QBrush(QColor("black")))
                    listWidget.addItem(item)
                    #print(filename, line, col, err_code, message)
                listWidget.itemClicked.connect(GotoLine)

                # add dock widget to the dock
                self.gui.addDockWidget(Qt.TopDockWidgetArea, dock_widget)

            # hide busy cursor
            QApplication.restoreOverrideCursor()
        else:
            #------------------------------------------------------------------------------------------------
            # remove existing EpubCheck/FlightCrew docks and close Check Ebook dock
            #------------------------------------------------------------------------------------------------
            for widget in self.gui.children():
                if isinstance(widget, QDockWidget) and widget.objectName() == 'epubcheck-dock':
                    #self.gui.removeDockWidget(widget)
                    #widget.close()
                    widget.setParent(None)
                if isinstance(widget, QDockWidget) and widget.objectName() == 'check-book-dock' and close_cb == True:
                    widget.close()

            #----------------------------------
            # define dock widget layout
            #----------------------------------
            textbox = QTextEdit()
            l = QVBoxLayout()
            l.addWidget(textbox)
            dock_widget = QDockWidget(self.gui)
            dock_widget.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea | Qt.BottomDockWidgetArea | Qt.TopDockWidgetArea)
            dock_widget.setObjectName('epubcheck-dock')
            dock_widget.setWindowTitle('EPUBCheck')
            
            #------------------------------------
            # recreate "classic" status message
            #------------------------------------
            status_message = (
                "EPUBCheck {0}\n"
                "Validating using EPUB version {1} rules.\n"
                "No errors or warnings detected.\n"
                "Messages: {2} fatals / {3} errors / {4} warnings / {5} infos / {6} usages\n\n"
                "EPUBCheck completed"
            ).format(checkerVersion, ePubVersion, nFatal, nError, nWarning, nInfo, nUsage)

            if extra:
                status_message += format_publication_info(info["publication"])

            textbox.setText(status_message)
            dock_widget.setWidget(textbox)
            self.gui.addDockWidget(Qt.TopDockWidgetArea, dock_widget)

            # hide busy cursor
            QApplication.restoreOverrideCursor()
