#!/usr/bin/env python

# from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2023-2024, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Standard libraries
import os
import re
import datetime
from timeit import default_timer as timer
from calendar import month_abbr as m
import calendar
from functools import partial
import locale
from copy import deepcopy
try:
    from collections.abc import Iterable
except ImportError:
    from typing import Iterable

# PyQt libraries
from qt.webengine import QWebEngineView
from qt.core import (Qt, QApplication, QTreeWidget, QTreeWidgetItem, QTreeWidgetItemIterator, QVBoxLayout, QHBoxLayout,
                     QGridLayout, QProgressBar, QLabel, QtGui, QtCore, QAbstractItemView, QDialogButtonBox, QComboBox,
                     QGroupBox, QPushButton, QMenu, QHeaderView, QLineEdit, QSize, QIcon, QSpinBox, QtWidgets, QUrl,
                     QEvent)

# Calibre libraries
from calibre.ebooks.metadata import MetaInformation
from calibre.utils.localization import ngettext as n
from calibre.gui2 import choose_files, choose_save_file, error_dialog, info_dialog, warning_dialog, question_dialog
from calibre.gui2.widgets2 import Dialog
from calibre.gui2.metadata.basic_widgets import DateEdit
from calibre.db.listeners import EventType
from calibre.utils.config import JSONConfig, config_dir
from calibre_plugins.Reading_Goal.config import prefs, get_icon
from calibre_plugins.Reading_Goal.__init__ import PLUGIN_VERSION, PLUGIN_NAME
from calibre_plugins.Reading_Goal.graphics import (h_html, h_bar, v_html, v_bar, daily_html, buttons_html,
                                                   jan_buttons_html, dec_buttons_html)


# Load translation files (.mo) on the folder 'translations'
load_translations()


# Check for dark theme
def is_dark_theme() -> bool:
    return QApplication.instance().is_dark_theme


def do_config(self, current_tab: int = 0) -> bool:
    from calibre.gui2.widgets2 import Dialog
    from calibre_plugins.Reading_Goal.config import ConfigWidget

    class ConfigDialog(Dialog):

        def __init__(self, gui, tab):
            self.gui = gui
            self.tab = tab
            Dialog.__init__(self, _('Options'), 'plugin-reading-goal-config-dialog')
            self.setWindowIcon(get_icon('_config.png'))

        def setup_ui(self) -> None:
            Dialog.setWindowFlag(self, Qt.WindowContextHelpButtonHint, True)
            box = QVBoxLayout(self)
            widget = ConfigWidget(self.gui, self.tab)
            box.addWidget(widget)
            button = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
            box.addWidget(button)
            button.accepted.connect(self.accept)
            button.rejected.connect(self.reject)

        def accept(self) -> None:
            self.widget.save_settings()
            Dialog.accept(self)

        def reject(self) -> None:
            Dialog.reject(self)

    d = ConfigDialog(self, current_tab)
    return d.exec_()


def show_configuration(self, tab: int = 0) -> bool:
    restart_message = _('Calibre must be restarted before the plugin can be configured')
    # Check if a restart is needed. If the restart is needed, but the user does not
    # trigger it, the result is true, and we do not do the configuration.
    if check_if_restart_needed(self, restart_message=restart_message):
        return False

    result = do_config(self, tab)
    restart_message = _('New custom columns have been created. You will need'
                        '\nto restart calibre for this change to be applied.'
                        )
    check_if_restart_needed(self, restart_message=restart_message)
    return result


def check_if_restart_needed(self, restart_message: str = None, restart_needed: bool = False) -> bool:
    if self.gui.must_restart_before_config or restart_needed:
        if restart_message is None:
            restart_message = _('Calibre must be restarted before the plugin can be configured')
        from calibre.gui2 import show_restart_warning
        do_restart = show_restart_warning(restart_message)
        if do_restart:
            self.gui.quit(restart=True)
        else:
            return True
    return False


class ReadingGoalTools:

    def __init__(self, gui):
        self.tool = self
        self.gui = gui

        self.goal_data = {}

        # Current database schema version
        self.current_schema_version = 1.5

        # Get preferences
        self.prefs = prefs

        # Set up a listener to get changes to reading progress and auto add column
        self.gui.add_db_listener(self.metadata_changed_event)

    def check_custom_columns(self, custom_column_name: str) -> bool:
        custom_columns = self.gui.library_view.model().custom_columns
        for key, column in custom_columns.items():
            lookup_name = '#' + column['label']
            if lookup_name == custom_column_name:
                return True
        return False

    def library_changed_event(self) -> None:
        self.library_id = self.gui.current_db.library_id

    def metadata_changed_event(self, db, event_type: EventType, event_data: tuple[str, set[int]]) -> Dialog | None:
        auto_add_challenges_columns = set()
        # {'challenge name': {'#custom column': 'value'}}
        if self.prefs['auto_add_challenges']:
            try:
                challenge_dict = eval(self.prefs['auto_add_challenges'])
            except:
                return error_dialog(self.gui, _('Reading Goal: Invalid python dictionary'),
                                    _('Verify your challenges dictionary. It must be a valid python dictionary.'),
                                    show_copy_button=False, show=True)
            for challenge in challenge_dict.items():
                auto_add_challenges_columns.add(list(challenge[1].items())[0][0])

        if event_type is EventType.metadata_changed:
            if event_data[0] == self.prefs['reading_progress_column']:
                self.manage_reading_goal(id_list=map(str, event_data[1]), option='metadata_changed')
            elif event_data[0] == self.prefs['auto_add_column']:
                self.manage_reading_goal(id_list=map(str, event_data[1]), option='auto_add_goal')
            elif event_data[0] in auto_add_challenges_columns:
                self.manage_reading_goal(id_list=map(str, event_data[1]), option='auto_add_challenges')
        return None

    def manage_reading_goal(self, id_list: Iterable[str] = None, option: str = None,
                            challenge_name: str = None) -> Dialog | None:
        # --- Entry point for the plugin --- #
        id_list = id_list if id_list else []

        # Library data
        from calibre.library import current_library_name
        self.library_id = self.gui.current_db.library_id
        self.library_name = current_library_name()
        db = self.gui.current_db.new_api

        # Check if the saved custom columns still exist
        prefs_custom_columns = {
            'reading_progress_column': self.prefs['reading_progress_column'],
            'status_date_column': self.prefs['status_date_column'],
            'page_count_column': self.prefs['page_count_column'],
            'genre_column': self.prefs['genre_column']
        }
        for key, column in prefs_custom_columns.items():
            if column != 'tags' and not self.check_custom_columns(column):
                self.prefs[key] = ''

        # Create auto backup
        self.create_backup(self.library_id, self.library_name)

        self.start = timer()

        # The menu option clicked by the user
        self.option = option

        # Get currently selected books
        if not id_list:
            ids = map(str, self.gui.library_view.get_selected_ids())
        else:
            ids = id_list

        # Check plugin config and book selection
        if not self.config_check(ids, option):
            return None

        if not self.check_legacy_database():
            return None

        # Reading goal data
        self.goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.library_id)

        # Get current date
        self.year = str(datetime.datetime.now().astimezone().year)

        if self.year not in self.goal_data:
            self.goal_data[self.year] = {}

        # Create a copy of the current year database, so we can manipulate it
        data = deepcopy(self.goal_data[self.year])

        # Update database automatically, so we always have the latest records
        if option in ('edit_goal', 'goal_statistics'):
            # if len(self.goal_data[self.year]) > 0:
            result = self.update_database(db, data)
            if not result:
                return None
            self.goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.library_id)
            # Create an updated copy of the current year database
            data = deepcopy(self.goal_data[self.year])

        # Scan the whole library if the auto add settings have changed
        ans = self.check_auto_add(db, data)
        if ans:
            self.goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.library_id)
            # Create an updated copy of the current year database
            data = deepcopy(self.goal_data[self.year])

        # Update on metadata change
        if self.option == 'metadata_changed':
            self.add_to_goal(ids, db, data, mode='metadata_update')

        # Auto add to goal
        if self.option == 'auto_add_goal':
            self.auto_add(ids, db, data, option='goal')

        # Auto add to challenges
        if self.option == 'auto_add_challenges':
            self.auto_add(ids, db, data, option='challenges')

        # Add to goal
        if self.option == 'add_to_goal':
            s_date = None
            if self.prefs['allow_other_years']:
                data, s_date = self.add_other_year(ids, db)
                if data is None and s_date is None:
                    return None
            self.add_to_goal(ids, db, data, s_date=s_date)

        # Remove from goal
        if self.option == 'remove_from_goal':
            ans = question_dialog(self.gui, _('Are you sure?'),
                                  _('This can\'t be undone. Are you sure you want to remove the selected '
                                    'books from your goal?'), skip_dialog_name='plugin-reading-goal-remove-goal-again',
                                  skip_dialog_msg=_('Show this confirmation again'), show_copy_button=False)
            if ans:
                self.remove_from_goal(ids, data)

        # Edit the goal
        if self.option == 'edit_goal':
            self.edit_goal(db, data)

        # Reading goal statistics
        if self.option == 'goal_statistics':
            # Check if there is data for selected year
            if len(data) == 0 or data['summary']['goal_books_count'] == 0:
                return error_dialog(self.gui, _('No books on your reading goal'),
                                    _('You must first add some books to your reading goal'),
                                    show_copy_button=False, show=True)
            else:
                self.statistics_main(data)

        # Add to challenge
        if self.option == 'add_to_challenge':
            self.add_to_challenge(ids, db, data, challenge_name)

        # Remove from challenge
        if self.option == 'remove_from_challenge':
            self.remove_from_challenge(ids, data, challenge_name)

        # Custom challenges
        if self.option == 'custom_challenges':
            self.custom_challenges(db, data)

        return None

    def check_auto_add(self, db, data: dict[str, dict]) -> bool:
        if prefs['auto_add_check']:
            all_ids = map(str, db.all_book_ids())
            self.auto_add(all_ids, db, data, option='all')
            prefs['auto_add_check'] = False
            return True
        else:
            return False

    def create_backup(self, library_id: str, library_name: str) -> Dialog | None:
        if self.prefs['backup_interval'] != '':
            if int(self.prefs['backup_interval']) > 0:
                # Check if the library entry exists
                try:
                    self.prefs['backup_date'][self.library_name]
                except TypeError:
                    self.prefs['backup_date'] = {}
                    self.prefs['backup_date'][self.library_name] = (datetime.datetime.today().astimezone()
                                                                    - datetime.timedelta(days=15))
                except KeyError:
                    self.prefs['backup_date'][self.library_name] = (datetime.datetime.today().astimezone()
                                                                    - datetime.timedelta(days=15))
                # Compare current date against last backup date
                time_delta = (datetime.datetime.today().astimezone().date()
                              - self.prefs['backup_date'][self.library_name].astimezone().date()).days
                if time_delta >= int(self.prefs['backup_interval']):
                    try:
                        timestamp = str(datetime.date.today())
                        database_path = os.path.join(config_dir, 'plugins')
                        database_file = os.path.join(database_path, 'Reading_Goal_Data_' + library_id + '.json')
                        backup_file_path = os.path.join(self.prefs['backup_path'], 'Reading_Goal_Data_'
                                                        + timestamp + ' (' + library_name + ')' + '.json')
                        if backup_file_path:
                            if not os.path.isdir(self.prefs['backup_path']):
                                os.mkdir(self.prefs['backup_path'])
                            with open(database_file, 'r') as df:
                                with open(backup_file_path, 'w') as bf:
                                    bf.write(df.read())
                            temp_date = deepcopy(self.prefs['backup_date'])
                            temp_date[self.library_name] = datetime.datetime.today().astimezone()
                            self.prefs['backup_date'] = temp_date
                    except:
                        import traceback
                        return error_dialog(self.gui, _('Auto backup'),
                                            _('Auto backup failed. Click on \'Show details\' for more info.'),
                                            det_msg=traceback.format_exc(), show_copy_button=True, show=True)

                # Remove old backup files
                if self.prefs['backup_files'] != '':
                    backup_path = self.prefs['backup_path']
                    files = os.listdir(backup_path)
                    files = [os.path.join(backup_path, f) for f in files if
                             not os.path.isdir(os.path.join(backup_path, f)) and f.startswith('Reading_Goal_Data')]
                    files = sorted(files, key=os.path.getmtime)
                    for file in files[:-int(self.prefs['backup_files'])]:
                        os.remove(file)

        return None

    def reread_count(self, book_id: str) -> None:
        reread_count = -1  # Stars with -1, so the first occurrence gets 0
        years_list = []
        # Create the years list
        for year in self.goal_data:  # Get the years already in the database
            if year != 'general_info':
                years_list.append(year)
        if self.year not in years_list:  # Add the current year
            years_list.append(self.year)
        years_list = sorted(years_list)
        current_year_idx = years_list.index(self.year)
        # Iterate over all the years, updating the count for each year
        for year in years_list:
            year_idx = years_list.index(year)
            if book_id in self.goal_data[year]:  # When we are updating the database
                if year_idx == current_year_idx:
                    if self.goal_data[year][book_id]['status'] > 0:
                        reread_count += 1
                else:
                    if self.goal_data[year][book_id]['status'] >= 100:
                        reread_count += 1

                # Check the book records for rereads in the same year
                _records = self.goal_data[year][book_id]['records']
                _records = dict(sorted(_records.items(), key=lambda x: int(x[0])))
                record_idx_list = list(_records.keys())
                i = 0
                for record_idx, record in _records.items():
                    if record['status'] == 100:
                        if len(record_idx_list) > i + 1:
                            reread_count += 1
                    i += 1

                self.goal_data[year][book_id]['reread_count'] = reread_count
                if reread_count > 0:
                    if self.goal_data[year][book_id]['status'] < 100:
                        self.goal_data[year][book_id]['shelf'] = 'Rereading'
                    else:
                        self.goal_data[year][book_id]['shelf'] = 'Reread'

        return None

    def add_to_goal(self, ids: Iterable[str], db, data: dict[str, dict], mode: str = 'adding',
                    challenge_name: str = None, s_date: datetime.datetime = None,
                    auto_add: bool = False, s_year: str = None) -> bool:
        # A temp dict to help us manipulate the records
        temp_data = {}

        # List of added books
        changed_list = []

        # List of updated books
        updated_list = []

        # Books with invalid records
        marked_books = {}

        # When deleting past years' records, the year must be provided
        if s_year:
            self.year = s_year

        # Check database conformance
        if mode == 'updating':
            self.goal_data, data = self.check_database(db, self.year, mode)

            # Update reread count
            if self.prefs['use_rereading_color'] or self.prefs['rereading_shelf']:
                _ids = []
                for y in self.goal_data:
                    if y != 'general_info':
                        for _id in self.goal_data[y]:
                            if _id != 'summary':
                                _ids.append(_id)
                for _book_id in _ids:
                    self.reread_count(_book_id)
        for book_id in ids:
            # Metadata update event check
            if mode == 'metadata_update':
                if book_id not in data:
                    continue

            # Don't update abandoned books
            if ((book_id in data and data[book_id]['shelf'] != 'Abandoned' and not s_year)
                    or book_id not in data or mode in ('adding', 'linking')):
                title, title_sort, authors, series, reading_progress, page_count, date, genre = (
                    self.get_book_info(db, int(book_id), s_date))
            else:
                title = data[book_id]['title']
                title_sort = data[book_id]['title_sort']
                authors = data[book_id]['authors']
                series = data[book_id]['series']
                reading_progress = data[book_id]['status']
                page_count = data[book_id]['page_count']
                date = data[book_id]['date']
                if date:
                    date = date.astimezone()
                genre = data[book_id]['genre']

            # Set challenges
            challenges = []
            if book_id in data:
                if 'challenges' in data[book_id]:
                    if data[book_id]['challenges']:
                        challenges = data[book_id]['challenges']
                    else:
                        challenges.append('Annual')
                else:
                    challenges.append('Annual')
            else:
                challenges.append('Annual')
                if challenge_name:
                    challenges.append(challenge_name)

            # Check if the book belongs to the chosen reading goal year
            if date and str(date.year) != self.year and not s_year:
                marked_books.update({int(book_id): 'invalid_year'})
                continue

            # Check if a finished book has no date/page count set
            if reading_progress == 100 and (page_count <= 0 or date is None):
                marked_books.update({int(book_id): 'missing_data'})
                continue

            # Check if the book is already on the database
            if book_id not in data or (data[book_id]['shelf'] == 'Abandoned' and mode in ('adding', 'linking')):
                changed_list.append(book_id)
            elif book_id in data:
                updated_list.append(book_id)

            # Organize in shelves
            if book_id in data and (data[book_id]['shelf'] == 'Abandoned' and mode not in ('adding', 'linking')):
                shelf = 'Abandoned'
            else:
                if reading_progress == 0:
                    shelf = 'Want to read'
                elif 0 < reading_progress < 100:
                    shelf = 'Reading'
                else:
                    shelf = 'Read'

            # Check reread books
            current_year_count = 0
            if self.prefs['use_rereading_color'] or self.prefs['rereading_shelf']:
                reread_count = -1  # Stars with -1, so the first occurrence gets 0
                years_list = []
                # Create the years list
                for year in self.goal_data:  # Get the years already in the database
                    if year != 'general_info':
                        years_list.append(year)
                if self.year not in years_list:  # Add the current year
                    years_list.append(self.year)
                years_list = sorted(years_list)
                current_year_idx = years_list.index(self.year)
                # Iterate over all the years, updating the count for each year
                for year in years_list:
                    year_idx = years_list.index(year)
                    if book_id in self.goal_data[year]:  # When we are updating the database
                        if year_idx == current_year_idx:
                            if self.goal_data[str(year)][book_id]['status'] > 0:
                                reread_count += 1
                        else:
                            if self.goal_data[str(year)][book_id]['status'] >= 100:
                                reread_count += 1

                        # Check the book records for rereads in the same year
                        _records = self.goal_data[str(year)][book_id]['records']
                        _records = dict(sorted(_records.items(), key=lambda x: int(x[0])))
                        record_idx_list = list(_records.keys())
                        i = 0
                        for record_idx, record in _records.items():
                            if record['status'] == 100:
                                if len(record_idx_list) > i+1:
                                    reread_count += 1
                            i += 1

                        self.goal_data[str(year)][book_id]['reread_count'] = reread_count
                        if reread_count > 0:
                            if self.goal_data[str(year)][book_id]['status'] < 100:
                                self.goal_data[str(year)][book_id]['shelf'] = 'Rereading'
                            else:
                                self.goal_data[str(year)][book_id]['shelf'] = 'Reread'
                        if year_idx == current_year_idx:
                            current_year_count = reread_count
                    elif year_idx == current_year_idx:  # New book being added
                        reread_count += 1
                        current_year_count = reread_count

                if current_year_count > 0:
                    if reading_progress == 0:
                        shelf = 'Want to read'
                    elif 0 < reading_progress < 100:
                        if self.prefs['rereading_shelf'] or self.prefs['use_rereading_color']:
                            shelf = 'Rereading'
                        else:
                            shelf = 'Reading'
                    else:
                        shelf = 'Reread'

            # Primary record (when book was first add to goal)
            # Records are meant to keep track of the reading progress
            record = {
                'records': {
                    '0': {
                        'date': date,
                        'status': reading_progress,
                        'read_pages': round(reading_progress * page_count / 100)
                    }
                }
            }

            # General book info and latest reading record
            book = {
                'title': title,
                'title_sort': title_sort,
                'authors': authors,
                'series': series,
                'page_count': page_count,
                'date': date,
                'status': reading_progress,
                'read_pages': round(reading_progress * page_count / 100),
                'shelf': shelf,
                'genre': genre,
                'reread_count': current_year_count,
                'challenges': challenges
            }
            # Last year pages are used to migrate unfinished books to the next year's goal
            if book_id not in data:
                last_year_pages = 0
            else:
                last_year_pages = data[book_id]['last_year_pages']
            book.update({'last_year_pages': last_year_pages})

            # Don't overwrite primary record
            if book_id not in data:
                book.update(record)
            else:
                # Manage records
                existing_records = data[book_id]['records']
                if len(existing_records) > 0:
                    last_record_key = max(map(int, existing_records))
                    last_record = existing_records[str(last_record_key)]
                    new_record_key = str(int(last_record_key) + 1)
                    new_record = {
                            new_record_key: {
                                'date': date,
                                'status': reading_progress,
                                'read_pages': round(reading_progress * page_count / 100)
                            }
                    }
                    # Check if this is really a new record
                    if new_record[new_record_key]['status'] != last_record['status']:
                        existing_records.update(new_record)
                elif mode == 'updating':
                    new_record = {
                        '0': {
                            'date': date,
                            'status': reading_progress,
                            'read_pages': round(reading_progress * page_count / 100)
                        }
                    }
                    existing_records.update(new_record)

                # Update the book's records
                book.update({'records': existing_records})
            # Now update the book entry as a whole
            temp_data[str(book_id)] = book

        # Update the temp database
        data.update(temp_data)

        # Update the goal's summary
        if mode in ('updating', 'metadata_update'):
            self.update_summary(data, mode='updating')
        else:
            self.update_summary(data)

        # Update our original database with the temp data
        self.goal_data[self.year] = data

        # Remove empty years
        remove_list = []
        current_year = datetime.datetime.now().astimezone().year
        for year in self.goal_data:
            if self.goal_data[str(year)] == {} and str(year) != str(current_year):
                remove_list.append(year)
        for year in remove_list:
            del self.goal_data[str(year)]

        end = timer()
        time_diff = end - self.start
        if mode == 'updating':
            print('Database updating time: ', datetime.timedelta(seconds=time_diff))
        elif mode == 'adding':
            print('Add to goal time: ', datetime.timedelta(seconds=time_diff))

        # Inform the user how many books were added
        if mode == 'adding' and not auto_add:
            self.final_message(changed_list, updated_list=updated_list, option='add_to_goal')

        # Mark books with an invalid year
        title = msg = ''
        if 'invalid_year' in marked_books.values() and 'missing_data' not in marked_books.values():
            title = _('Invalid year')
            msg = _('These books could not be updated/added to your reading goal '
                    'because the reading date does not match the reading goal year')

        # Mark books missing necessary data
        if 'missing_data' in marked_books.values() and 'invalid_year' not in marked_books.values():
            title = _('Missing necessary data')
            msg = _('These books are missing necessary data. '
                    'You must set a reading date and page count for finished books.')

        # Mark all invalid books
        if 'invalid_year' in marked_books.values() and 'missing_data' in marked_books.values():
            title = _('Invalid records')
            msg = _('These books are either missing necessary data or have a invalid year.\n\n'
                    'Search terms:\n'
                    'Books with a invalid year: marked:"=invalid_year"\n'
                    'Books missing necessary data: marked:"=missing_data"')

        if len(marked_books) > 0:
            self.mark_rows(marked_books)
            warning_dialog(self.gui, title, msg, show_copy_button=False, show=True)
            return False
        return True

    def remove_from_goal(self, ids: Iterable[str], data: dict[str, dict], year: str = None, mode: str = None) -> None:
        # List of removed books
        changed_list = []

        # Remove the entries from the database
        for book_id in ids:
            try:
                del data[book_id]
                changed_list.append(book_id)

            except KeyError:
                pass  # Ignore books that are not on the reading goal

        # Update the goal's summary
        self.update_summary(data)

        # Update our original database with the temp data
        if year:
            self.goal_data[year] = data
        else:
            self.goal_data[self.year] = data

        # Update the timestamp
        if 'general_info' in self.goal_data:
            self.goal_data['general_info']['timestamp'] = datetime.datetime.now().astimezone()
        else:
            self.goal_data['general_info'] = {}
            self.goal_data['general_info']['timestamp'] = datetime.datetime.now().astimezone()

        if mode != 'editing':
            end = timer()
            time_diff = end - self.start
            print('Remove from goal time: ', datetime.timedelta(seconds=time_diff))
            # Inform the user how many books were removed
            self.final_message(changed_list, option='remove_from_goal')

        return None

    def add_to_challenge(self, ids: Iterable[str], db, data: dict[str, dict],
                         challenge_name: str, mode: str = None) -> None:
        added_ids = []
        out_ids = []

        # Add challenge to book info
        if 'summary' in data:
            if 'challenges_dict' in data['summary']:
                # Check if the challenge exists
                if challenge_name not in data['summary']['challenges_dict']:
                    return
                for book_id in ids:
                    # Check if the book belongs to the current year's goal
                    if book_id in data:
                        challenges = data[book_id]['challenges']
                        if challenge_name not in challenges:
                            added_ids.append(book_id)
                            challenges.append(challenge_name)
                    else:
                        out_ids.append(book_id)

            self.update_summary(data, mode='updating')
            self.update_json(data)

        msg = n(_('{0} book added to the {1} challenge').format(len(added_ids), challenge_name),
                _('{0} books added to the {1} challenge').format(len(added_ids), challenge_name), len(added_ids))

        if not mode:
            info_dialog(self.gui, _('Added'), msg, show_copy_button=False, show=True)

            if len(out_ids) > 0:
                self.mark_rows(map(int, out_ids), marked_text='out_of_scope')
                ans = question_dialog(self.gui, _('Not in your current goal'),
                                      _('These books are not in your current reading goal. Do you want to them?'),
                                      skip_dialog_name='plugin-reading-goal-add-challenge-again',
                                      skip_dialog_msg=_('Show this confirmation again'),
                                      show_copy_button=False)
                if ans:
                    self.add_to_goal(out_ids, db, data, mode='adding', challenge_name=challenge_name)

    def remove_from_challenge(self, ids: Iterable[str], data: dict[str, dict], challenge_name: str) -> None:
        removed_ids = []
        out_ids = []

        if 'summary' in data:
            for book_id in ids:
                # Check it the book exists belongs to the current year's goal
                if book_id in data:
                    if book_id != 'summary' and 'challenges' in data[book_id]:
                        if challenge_name in data[book_id]['challenges']:
                            removed_ids.append(book_id)
                            data[book_id]['challenges'].remove(challenge_name)
                else:
                    out_ids.append(book_id)

            self.update_summary(data, mode='updating')
            self.update_json(data)

        msg = n(_('{0} book removed to the {1} challenge').format(len(removed_ids), challenge_name),
                _('{0} books removed from the {1} challenge').format(len(removed_ids), challenge_name),
                len(removed_ids))
        info_dialog(self.gui, _('Added'), msg, show_copy_button=False, show=True)

    def config_check(self, ids: Iterable[str], option: str) -> bool:
        # Check if selection is empty
        if not ids and option in ('add_to_goal', 'remove_from_goal'):
            error_dialog(self.gui, _('Empty selection'), _('No books selected'), show_copy_button=False, show=True)
            return False

        # Check if the necessary custom columns are set
        custom_cols = self.gui.library_view.model().custom_columns
        col = None
        if len(self.prefs['page_count_column']) == 0 or self.prefs['page_count_column'] not in custom_cols:
            col = _('page count custom column')
        if len(self.prefs['status_date_column']) == 0 or self.prefs['status_date_column'] not in custom_cols:
            col = _('status date custom column')
        if len(self.prefs['reading_progress_column']) == 0 or self.prefs['reading_progress_column'] not in custom_cols:
            col = _('reading progress custom column')

        if col:
            if question_dialog(self.gui, _('Configuration needed'), _('You must choose the {}.').format(col) +
                               _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                show_configuration(self, tab=1)
                return False
            else:
                return False
        else:
            return True

    def check_legacy_database(self) -> bool:
        # Check for legacy database name
        old_database_file = os.path.join(config_dir, 'plugins/Reading_Goal_Data.json')
        new_database_file = os.path.join(config_dir, 'plugins/Reading_Goal_Data_' + self.library_id + '.json')
        if os.path.isfile(old_database_file):
            ans = question_dialog(self.gui, _('Library association'),
                                  _('Reading Goal now uses a different database for each library. Your database is '
                                    'about to be associated with your current library. If this is not the correct '
                                    'library, click NO and change to the correct one.'), show_copy_button=False)
            if ans:
                os.rename(old_database_file, new_database_file)
            else:
                return False
        return True

    def add_other_year(self, ids: Iterable[str], db) -> tuple[dict | None, datetime.datetime | None]:
        # Check if all the selected books belongs to the same year
        year_list = []
        s_date = None
        _ids = deepcopy(ids)
        for book_id in _ids:
            mi = db.get_proxy_metadata(int(book_id))
            date = mi.get(self.prefs['status_date_column'])
            if date:
                book_year = date.astimezone().year
            else:
                book_year = datetime.datetime.now().astimezone().year
            year_list.append(book_year)
        if all(i == year_list[0] for i in year_list):
            detected_year = year_list[0]
            while True:
                input_dialog = InputDialog(self.gui, int(detected_year))
                input_dialog.resize(input_dialog.sizeHint())
                result = input_dialog.exec_()
                if result:
                    if input_dialog.button.text() == _('Date'):
                        value = input_dialog.year_spinbox.value()
                        self.year = str(value)
                        if len(self.year) == 4:
                            break
                        else:
                            error_dialog(self.gui, _('Invalid year information'),
                                         _('You must inform the year in a four digit format'),
                                         show_copy_button=False, show=True)
                    else:
                        value = input_dialog.date_widget.current_val.astimezone()
                        self.year = str(value.year)
                        s_date = value
                        break

                if not result:
                    return None, None
        else:
            error_dialog(self.gui, _('Invalid year information'),
                         _('When adding multiple books, they must have\n'
                           'the same year in the reading date column'),
                         show_copy_button=False, show=True)
            return None, None

        if self.year not in self.goal_data:
            self.goal_data[self.year] = {}

        # Create a copy of the selected year database
        data = deepcopy(self.goal_data[self.year])

        return data, s_date

    def update_json(self, data: dict[str, dict], year: str = None) -> None:
        # Update the original database
        if not year:
            year = datetime.datetime.now().astimezone().year
        goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.library_id)
        goal_data[str(year)] = data

    def check_database(self, db, year: str, mode: str) -> tuple[dict, dict]:
        # sv_1.1 - Breaking changes: pv_1.0.4 > Genre view / Reread shelf
        # sv_1.2 - Breaking changes: pv_1.0.8 > Group general info
        # sv_1.3 - Breaking changes: pv_1.7.2 > Add series info to database
        # sv_1.4 - Breaking changes: pv_1.7.2 > Add series index info to database
        # sv_1.5 - Breaking changes: pv_1.7.2 > Bug fix for series_index = 0.0
        goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.library_id)
        # Check if the general info is grouped. If not, group them.
        if 'general_info' not in goal_data:
            goal_data['general_info'] = {}
            if 'timestamp' in goal_data:
                goal_data['general_info']['timestamp'] = goal_data['timestamp']
            if 'autofill_done' in goal_data:
                goal_data['general_info']['autofill_done'] = goal_data['autofill_done']
            if 'schema_version' in goal_data:
                goal_data['general_info']['schema_version'] = goal_data['schema_version']
            del goal_data['timestamp'], goal_data['autofill_done'], goal_data['schema_version']
        # Check if the schema version changed
        database_schema_version = goal_data['general_info']['schema_version'] \
            if 'schema_version' in goal_data['general_info'] else 0
        if ('schema_version' not in goal_data['general_info']
                or database_schema_version < self.current_schema_version):
            # Since it changed, we must populate the new fields created
            if mode == 'updating':
                for _year, _books in goal_data.items():
                    if _year != 'general_info':
                        for _book_id, _book_info in _books.items():
                            _genre = 'Undefined'
                            if _book_id != 'summary':
                                mi = db.get_proxy_metadata(int(_book_id))
                                # Genre info
                                if 'genre' not in _book_info:
                                    if self.prefs['genre_column']:
                                        if self.prefs['genre_column'] == 'tags':
                                            _genre = mi.tags
                                        else:
                                            _genre = mi.get(self.prefs['genre_column'])
                                        if not _genre:
                                            _genre = 'Undefined'
                                else:
                                    _genre = _book_info['genre']
                                goal_data[_year][_book_id]['genre'] = _genre
                                # Reread counter
                                if 'reread_count' not in _book_info:
                                    goal_data[_year][_book_id]['reread_count'] = 0
                                if 'series' not in _book_info or database_schema_version < 1.5:
                                    series_index = mi.series_index
                                    if series_index is not None:
                                        if series_index.is_integer():
                                            series_index = int(series_index)
                                    series = mi.series
                                    if series:
                                        series = series + ' [' + str(series_index) + ']'
                                    goal_data[_year][_book_id]['series'] = series

        goal_data['general_info']['schema_version'] = self.current_schema_version

        # Remove rereading shelf based on the user preference
        if not self.prefs['rereading_shelf'] and not self.prefs['use_rereading_color'] and mode == 'updating':
            for _year, _books in goal_data.items():
                if _year != 'general_info':
                    for _book_id, _book_info in _books.items():
                        if _book_id != 'summary':
                            reading_progress = goal_data[_year][_book_id]['status']
                            if goal_data[_year][_book_id]['shelf'] == 'Abandoned':
                                shelf = 'Abandoned'
                            else:
                                if reading_progress == 0:
                                    shelf = 'Want to read'
                                elif 0 < reading_progress < 100:
                                    shelf = 'Reading'
                                else:
                                    shelf = 'Read'
                            goal_data[_year][_book_id]['shelf'] = shelf

        # Check if it's a new year
        current_year = datetime.datetime.now().astimezone().year
        if 'timestamp' in goal_data['general_info']:
            last_timestamp = goal_data['general_info']['timestamp']
            if current_year > last_timestamp.year:
                goal_data['general_info']['autofill_done'] = False
            else:
                goal_data['general_info']['autofill_done'] = True

        # Update the timestamp
        goal_data['general_info']['timestamp'] = datetime.datetime.now().astimezone()

        # Autofill current year's reading goal
        if 'autofill_done' not in goal_data['general_info']:
            goal_data['general_info']['autofill_done'] = False
        else:
            if self.prefs['autofill'] and not goal_data['general_info']['autofill_done']:
                self.autofill(goal_data)

        # Create an updated copy of the current year database
        data = deepcopy(goal_data[year])

        return goal_data, data

    def edit_database(self, queue_dict: dict, db, data: dict[str, dict], year: str) -> list[str]:
        changed_list = []
        for job, ids in queue_dict.items():
            for book_id in ids:
                changed_list.append(book_id)
            if job == 'remove_from_goal' and ids:
                self.remove_from_goal(ids, data, mode='editing')
            else:
                if job == 'add_link' and ids:
                    add_ids = []
                    remove_book_ids = []
                    for old_book_id, new_book_id in ids.items():
                        remove_book_ids.append(old_book_id)
                        add_ids.append(new_book_id)
                    self.remove_from_goal(remove_book_ids, data, mode='editing')
                    self.add_to_goal(add_ids, db, data, mode='linking')
                if job == 'abandon' and ids:
                    for book_id in ids:
                        data[book_id]['shelf'] = 'Abandoned'
                elif job == 'delete_record' and ids:
                    for book_id, records_ids in ids.items():
                        for idx in records_ids:
                            del data[book_id]['records'][idx]
                        # Update our original database with the temp data
                        self.goal_data[year] = data
                        # Update the book entry
                        self.add_to_goal([book_id], db, data, mode='updating', s_year=year)
                        self.goal_data['general_info']['timestamp'] = datetime.datetime.now().astimezone()
                if job != 'delete_record':
                    # Update our original database with the temp data
                    self.goal_data[year] = data
                    # Update the timestamp
                    self.goal_data['general_info']['timestamp'] = datetime.datetime.now().astimezone()
        return changed_list

    def update_database(self, db, data: dict[str, dict]) -> bool:
        # Detect books removed from calibre
        missing_books = self.deleted_books(db, data)
        if missing_books:
            msg1 = n(_('One of your reading goal books was removed from calibre. '),
                     _('{} of your reading goal books were removed from calibre. ').format(len(missing_books)),
                     len(missing_books))
            msg2 = _('Choose what to do in the next dialog.')
            warning_dialog(self.gui, _('Broken links'), msg1 + msg2, show_copy_button=False, show=True)
            self.manage_deleted_books(db, data, missing_books)
            return False

        ids = [key for key in data.keys() if key != 'summary']
        result = self.add_to_goal(ids, db, data, mode='updating')
        if not result:
            return False

        return True

    def auto_add(self, ids: Iterable[str], db, data: dict[str, dict], option: str = 'goal') -> Dialog | None:
        if option in ('goal', 'all'):
            # List of auto added to goal
            add_to_goal_ids = []
            for book_id in ids:
                if book_id in data:
                    continue
                mi = db.get_proxy_metadata(int(book_id))
                auto_add_col = mi.metadata_for_field(self.prefs['auto_add_column'])
                if self.prefs['auto_add_value']:
                    if auto_add_col['datatype'] == 'bool':
                        # Try to get some consistency
                        if self.prefs['auto_add_value'] in ('True', 'true', _('True'),
                                                            'Yes', 'yes', _('Yes'), _('yes')):
                            value = 'True'
                        elif self.prefs['auto_add_value'] in ('False', 'false', _('False'),
                                                              'No', 'no', _('No'), _('no')):
                            value = 'False'
                        else:
                            value = self.prefs['auto_add_value']
                        if value == str(mi.get(self.prefs['auto_add_column'])):
                            add_to_goal_ids.append(book_id)
                    elif auto_add_col['datatype'] == 'text':  # Especial case for a tags-like column
                        if self.prefs['auto_add_value'] in str(
                                mi.get(self.prefs['auto_add_column'])):
                            add_to_goal_ids.append(book_id)
                    else:
                        if self.prefs['auto_add_value'] == str(
                                mi.get(self.prefs['auto_add_column'])):
                            add_to_goal_ids.append(book_id)
                elif auto_add_col:
                    if mi.get(self.prefs['auto_add_column']) is not None and len(
                            str(mi.get(self.prefs['auto_add_column']))) > 0:
                        add_to_goal_ids.append(book_id)
            if add_to_goal_ids:
                self.add_to_goal(add_to_goal_ids, db, data, mode='adding', auto_add=True)

        if option in ('challenges', 'all') and self.prefs['auto_add_challenges']:
            try:
                challenge_dict = eval(self.prefs['auto_add_challenges'])
            except:
                return error_dialog(self.gui, _('Reading Goal: Invalid python dictionary'),
                                    _('Verify your challenges dictionary. It must be a valid python dictionary.'),
                                    show_copy_button=False, show=True)
            if challenge_dict:
                for challenge in challenge_dict.items():
                    # List of auto added to challenge
                    add_to_challenges_ids = []
                    for book_id in ids:
                        mi = db.get_proxy_metadata(int(book_id))
                        challenge_col = list(challenge[1].items())[0][0]
                        challenge_value = list(challenge[1].items())[0][1]
                        if challenge_col == 'tags':
                            calibre_field = mi.tags
                        else:
                            calibre_field = mi.metadata_for_field(challenge_col)
                        if not calibre_field:
                            return None
                        if challenge_col != 'tags':
                            if calibre_field['datatype'] != 'text':
                                if challenge_value == str(mi.get(challenge_col)):
                                    add_to_challenges_ids.append(book_id)
                            else:
                                if challenge_value in mi.get(challenge_col):
                                    add_to_challenges_ids.append(book_id)
                        else:
                            if challenge_value in calibre_field:
                                add_to_challenges_ids.append(book_id)
                    # Add new books to goal first
                    add_to_goal_ids = []
                    for _id in add_to_challenges_ids:
                        if _id not in data:
                            add_to_goal_ids.append(_id)
                    if add_to_goal_ids:
                        self.add_to_goal(add_to_goal_ids, db, data, mode='adding', auto_add=True)
                    # Then add to challenges
                    if add_to_challenges_ids:
                        self.add_to_challenge(add_to_challenges_ids, db, data, challenge_name=challenge[0],
                                              mode='adding')

        return None

    @staticmethod
    def update_summary(data: dict[str, dict], mode: str = None) -> None:
        # Reset counters
        goal_books_count = 0
        read_books_count = 0
        goal_page_count = 0
        read_page_count = 0
        last_year_pages_count = 0
        read_books_month_count = {
            m[1]: 0, m[2]: 0, m[3]: 0, m[4]: 0, m[5]: 0, m[6]: 0,
            m[7]: 0, m[8]: 0, m[9]: 0, m[10]: 0, m[11]: 0, m[12]: 0
        }

        # Get all the challenges
        if mode == 'updating':
            c_set = set()
            if len(data) > 0:
                for book_id in data:
                    if book_id != 'summary':
                        if 'challenges' in data[book_id]:
                            for c in data[book_id]['challenges']:
                                c_set.add(c)
                        else:
                            data[book_id]['challenges'] = ['Annual']

        # Iterate over the database to compile the summary
        c_dict = {}
        if len(data) > 0:
            for book_id in data:
                if book_id != 'summary':
                    goal_books_count += 1
                    goal_page_count += data[book_id]['page_count']
                    read_page_count += data[book_id]['read_pages']
                    last_year_pages_count += data[book_id]['last_year_pages']
                    if data[book_id]['status'] == 100:
                        read_books_count += 1
                        month = data[book_id]['date'].astimezone().strftime('%b')
                        read_books_month_count[month] += 1
                    if mode == 'updating':
                        for c in c_set:
                            if c in data[book_id]['challenges']:
                                if c not in c_dict:
                                    c_dict[c] = {'ids': [], 'read_books': []}
                                    c_dict[c]['ids'].append(book_id)
                                else:
                                    c_dict[c]['ids'].append(book_id)
                                if data[book_id]['status'] == 100:
                                    if 'summary' in data:
                                        if 'challenges_dict' in data['summary']:
                                            try:
                                                c_start = data['summary']['challenges_dict'][c]['start'].astimezone()
                                                c_end = data['summary']['challenges_dict'][c]['end'].astimezone()
                                                if c_start <= data[book_id]['date'].astimezone() <= c_end:
                                                    c_dict[c]['read_books'].append(book_id)
                                            except:
                                                pass
            if mode == 'updating':
                if 'summary' in data:
                    challenges_dict = data['summary']['challenges_dict'] if (
                                'challenges_dict' in data['summary'] and goal_books_count != 0) else {}
                    chal_dict = deepcopy(challenges_dict) if challenges_dict != {} else {i: {} for i in
                                                                                              c_dict.keys()}
                    for challenge in chal_dict:
                        try:
                            chal_dict[challenge]['count'] = len(c_dict[challenge]['ids'])
                        except KeyError:
                            chal_dict[challenge]['count'] = 0
                        if challenge == 'Annual':
                            y = datetime.datetime.now().astimezone().year
                            chal_dict[challenge]['start'] = datetime.datetime(y, 1, 1)
                            chal_dict[challenge]['end'] = datetime.datetime(y, 12, 31)
                            if len(c_dict['Annual']['ids']) > 0:
                                chal_dict[challenge]['progress'] = round(len(c_dict['Annual']['read_books']) /
                                                                         len(c_dict['Annual']['ids']) * 100)
                            else:
                                chal_dict[challenge]['progress'] = 0
                        else:
                            if 'start' in challenges_dict[challenge]:
                                chal_dict[challenge]['start'] = challenges_dict[challenge]['start']
                                chal_dict[challenge]['end'] = challenges_dict[challenge]['end']
                            if chal_dict[challenge]['count'] > 0:
                                chal_dict[challenge]['progress'] = round(len(c_dict[challenge]['read_books']) /
                                                                         len(c_dict[challenge]['ids']) * 100)
                            else:
                                chal_dict[challenge]['progress'] = 0
                    challenges_dict.update(chal_dict)
                else:
                    challenges_dict = {}
            else:
                if 'summary' in data:
                    if 'challenges_dict' in data['summary']:
                        challenges_dict = data['summary']['challenges_dict']
                    else:
                        challenges_dict = {}
                else:
                    challenges_dict = {}

            data['summary'] = {
                'goal_books_count': goal_books_count,
                'goal_page_count': goal_page_count - last_year_pages_count,
                'read_page_count': read_page_count - last_year_pages_count,
                'read_books_count': read_books_count,
                'read_books_month_count': read_books_month_count,
                'goal_percentage': round(read_books_count / goal_books_count * 100, 1) if goal_books_count else 0,
                'challenges_dict': challenges_dict
            }

    @staticmethod
    def autofill(goal_data: dict[str, dict]) -> None:
        current_year = datetime.datetime.now().astimezone().year
        last_year = current_year - 1
        if str(current_year) in goal_data:
            existing_data = deepcopy(goal_data[str(current_year)])
        else:
            existing_data = {}
        transfer_data = {}
        for year, data in goal_data.items():
            if str(year) != 'general_info':
                if int(year) == last_year:
                    for book_id, values in data.items():
                        if book_id != 'summary':
                            new_values = deepcopy(values)
                            if values['status'] < 100:
                                # Remove existing Challenges
                                new_values['challenges'] = []
                                # Get unfinished books
                                if values['read_pages'] > 0:
                                    new_values['last_year_pages'] = values['read_pages']
                                if values['shelf'] != 'Abandoned':
                                    transfer_data[str(book_id)] = new_values
                    transfer_data.update(existing_data)
        if transfer_data != {}:
            goal_data[str(current_year)] = transfer_data
        goal_data['general_info']['autofill_done'] = True

    def final_message(self, changed_list: list[str], updated_list: list[str] = None, option: str = None) -> None:
        updated_list = updated_list if updated_list else []
        # Choose a message for the user based on the current task
        message = {}
        msg_title = ''
        p = len(changed_list) if len(changed_list) > 0 else len(changed_list) + 1
        if option in ('add_to_goal', 'remove_from_goal', 'edit_goal'):
            message_dict = {
                'add_to_goal': n(_('was added to your reading goal'), _('were added to your reading goal'), p),
                'remove_from_goal': n(_('was removed from your reading goal'),
                                      _('were removed from your reading goal'), p),
                'edit_goal': n(_('was updated in your database'), _('were updated in your database'), p),
            }
            message = message_dict[option]
            if option == 'edit_goal':
                msg_title = _('Database updated')
            else:
                msg_title = _('Reading goal updated')

        # Final message to the user, show how many books were updated
        if len(changed_list) == 1:
            info_dialog(self.gui, msg_title, _('1 book {}').format(message), show_copy_button=False, show=True)
        elif len(changed_list) > 1:
            info_dialog(self.gui, msg_title, _('{0} books {1}').format(p, message),
                        show_copy_button=False, show=True)
        else:
            warning_dialog(self.gui, msg_title, _('No book {}').format(message), show_copy_button=False, show=True)

        # Books not added, only updated
        msg = n(_('was updated in your database'), _('were updated in your database'), len(updated_list))
        if len(updated_list) == 1:
            info_dialog(self.gui, msg_title, _('1 book {}').format(msg),
                        show_copy_button=False, show=True)
        elif len(updated_list) > 1:
            info_dialog(self.gui, msg_title, _('{0} books {1}').format(len(updated_list), msg),
                        show_copy_button=False, show=True)

    def get_book_info(self, db, book_id: int, s_date: datetime.datetime)\
            -> tuple[str, str, list[str], str, int, int, datetime.datetime, str]:
        # Get book info from calibre
        mi = db.get_proxy_metadata(book_id)
        title = mi.title
        title_sort = mi.title_sort
        authors = mi.authors
        series_index = mi.series_index
        if series_index is not None:
            if series_index.is_integer():
                series_index = int(series_index)
        series = mi.series
        if series:
            series = series + ' [' + str(series_index) + ']'
        reading_progress = mi.get(self.prefs['reading_progress_column'])
        if reading_progress:
            # try:
            #     reading_progress = int(round(reading_progress))
            # except TypeError:
            #     reading_progress = int(reading_progress)
            if type(reading_progress) not in (int, float):
                reading_progress = int(reading_progress)
            if reading_progress < 0:
                reading_progress = 0
            elif reading_progress > 100:
                reading_progress = 100
        else:
            reading_progress = 0
        page_count = mi.get(self.prefs['page_count_column'])
        page_count = page_count if page_count is not None else 0
        if s_date is None:
            date = mi.get(self.prefs['status_date_column'])
        else:
            date = s_date
        if date:
            date = date.astimezone()
        genre = 'Undefined'
        if self.prefs['genre_column']:
            if self.prefs['genre_column'] == 'tags':
                genre = mi.tags
            else:
                genre = mi.get(self.prefs['genre_column'])
            if not genre:
                genre = 'Undefined'
        return title, title_sort, authors, series, reading_progress, page_count, date, genre

    @staticmethod
    def deleted_books(db, data: dict[str, dict]) -> list[str]:
        # Return a list of the reading goal's books removed from calibre
        deleted_books = []
        for book_id in data:
            if book_id != 'summary':
                if int(book_id) not in db.all_book_ids() and data[book_id]['shelf'] != 'Abandoned':
                    deleted_books.append(book_id)
        return deleted_books

    def manage_deleted_books(self, db, data: dict[str, dict], missing_books: list[str]) -> None:
        tree = self.edit_tree(data, ids=missing_books)
        self.book_list(_('Missing books'), 'plugin-reading-goal-missing-books-dialog',
                       get_icon('missing.png'), tree, db, data)

    def link_book(self, db, similar_books: Iterable[str]) -> str | None:
        tool = self
        _gui = self.gui

        class LinkBookDialog(Dialog):

            def __init__(self):
                Dialog.__init__(self, _('Similar books'), 'plugin-reading-goal-similar-books-dialog', parent=tool.gui)
                self.setWindowIcon(get_icon('add_link.png'))
                self.selected_book_id = None

            def setup_ui(self) -> None:
                layout = QVBoxLayout(self)
                layout.addSpacing(5)

                # Add instruction label
                self.label = QLabel(_('Select the calibre book you want linked to this entry:'))
                layout.addWidget(self.label)
                layout.addSpacing(10)

                # QTreeWidget to display similar books
                self.tree = QTreeWidget()
                layout.addWidget(self.tree)

                # Selection mode
                self.tree.setSelectionMode(QAbstractItemView.SingleSelection)

                # Add similar books to the list
                self.tree.setHeaderLabels(['book_ids', _('Title'), _('Authors'), _('Series'), _('Publisher')])
                self.tree.header().setDefaultAlignment(Qt.AlignCenter)

                for book_id in similar_books:
                    mi = db.get_proxy_metadata(int(book_id))
                    title, title_sort, authors, series, series_idx, publisher = (
                        mi.title, mi.title_sort, ' & '.join(mi.authors), mi.series, mi.series_index, mi.publisher)
                    if series_idx:
                        if series_idx.is_integer():
                            series_idx = int(series_idx)
                    if series:
                        series = series + ' [' + str(series_idx) + ']'
                    title_opt = title if prefs['sorting'] == 'Title' else title_sort
                    item = QTreeWidgetItem([book_id, title_opt, authors, series, publisher])
                    # Set text alignment (PyQt 6.4+ only)
                    if float(QtCore.PYQT_VERSION_STR[0:3]) >= 6.4:
                        item.setTextAlignment(2, QtCore.Qt.AlignCenter)
                        item.setTextAlignment(3, QtCore.Qt.AlignCenter)
                        item.setTextAlignment(4, QtCore.Qt.AlignCenter)
                    self.tree.addTopLevelItem(item)

                # Resize mode
                self.tree.header().setStretchLastSection(False)
                self.tree.header().setSectionResizeMode(1, QHeaderView.ResizeMode.Stretch)

                # Enable sorting
                self.tree.setSortingEnabled(True)
                self.tree.sortItems(1, Qt.AscendingOrder)
                self.tree.setColumnHidden(0, True)

                # Add Ok/Cancel buttons
                self.ac_button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok |
                                                      QDialogButtonBox.StandardButton.Cancel)
                layout.addWidget(self.ac_button_box)
                self.ac_button_box.button(QDialogButtonBox.StandardButton.Ok).clicked.connect(self.accept)
                self.ac_button_box.button(QDialogButtonBox.StandardButton.Cancel).clicked.connect(self.reject)

                # Set up a listener to catch a selection change
                self.tree.itemSelectionChanged.connect(self.manage_buttons)

                # Auto select if there is only one book listed
                self.ac_button_box.button(QDialogButtonBox.StandardButton.Ok).setEnabled(False)
                if self.tree.topLevelItemCount() > 1:
                    self.tree.setCurrentItem(None)
                else:
                    self.tree.setCurrentItem(self.tree.topLevelItem(0))

                # Restore state
                if self.prefs['state']['LinkBookDialog']:
                    self.tree.header().restoreState(prefs['state']['LinkBookDialog'])
                else:
                    self.tree.setColumnWidth(1, 250)
                    self.tree.setColumnWidth(2, 200)
                    self.tree.setColumnWidth(3, 200)
                    self.tree.setColumnWidth(4, 70)

            def manage_buttons(self) -> None:
                # Disable Ok button if no book is selected
                if len(self.tree.selectedItems()) == 0:
                    self.ac_button_box.button(QDialogButtonBox.StandardButton.Ok).setEnabled(False)
                else:
                    self.ac_button_box.button(QDialogButtonBox.StandardButton.Ok).setEnabled(True)

            def accept(self) -> None:
                selected_item = self.tree.currentItem()
                if selected_item:
                    self.selected_book_id = selected_item.text(0)
                Dialog.accept(self)

            def reject(self) -> None:
                Dialog.reject(self)

            def sizeHint(self) -> QSize:
                return QSize(800, 250)

        d = LinkBookDialog()
        d.exec_()

        return d.selected_book_id

    def custom_challenges(self, db, data: dict[str, dict]) -> Dialog | None:
        tool = self
        _gui = self.gui
        library_id = self.gui.current_db.library_id
        goal_data = JSONConfig('plugins/Reading_Goal_Data_' + library_id)

        if not data:
            return error_dialog(self.gui, _('No books on your reading goal'),
                                _('You must first add some books to your reading goal'),
                                show_copy_button=False, show=True)

        # Update the database
        if not tool.update_database(db, data):
            return None

        # Update summary
        tool.update_summary(data, mode='updating')

        class CustomChallengesDialog(Dialog):
            def __init__(self):
                Dialog.__init__(self, _('Custom challenges'), 'plugin-reading-goal-challenges-dialog', parent=tool.gui)
                self.setWindowIcon(get_icon('challenges.png'))

                prefs = JSONConfig('plugins/Reading_Goal')
                self.challenges_dict = {}

            def setup_ui(self) -> None:
                self.layout = QGridLayout(self)

                # Create tree
                self.current_year = datetime.datetime.now().astimezone().year
                self.tree = self.create_tree(self.current_year)

                # Doubleclick to edit the challenge
                self.tree.itemDoubleClicked.connect(partial(self.edit_challenge, 'editing'))

                self.layout.addWidget(self.tree, 0, 0, 7, 1)

                # Add challenge button
                self.add_challenge_button = QPushButton(get_icon('challenge_add.png'), '', self)
                self.add_challenge_button.setToolTip(_('Create a new challenge'))
                self.add_challenge_button.setAutoDefault(False)
                self.layout.addWidget(self.add_challenge_button, 1, 1, 1, 1)
                self.add_challenge_button.clicked.connect(partial(self.edit_challenge, 'adding'))

                # Remove challenge button
                self.remove_challenge_button = QPushButton(get_icon('challenge_remove.png'), '', self)
                self.remove_challenge_button.setToolTip(_('Delete selected challenge'))
                self.layout.addWidget(self.remove_challenge_button, 3, 1, 1, 1)
                self.remove_challenge_button.setAutoDefault(False)
                self.remove_challenge_button.clicked.connect(self.remove_challenge)

                # Edit challenge button
                self.edit_challenge_button = QPushButton(get_icon('challenge_edit.png'), '', self)
                self.edit_challenge_button.setToolTip(_('Edit selected challenge'))
                self.edit_challenge_button.setAutoDefault(False)
                self.layout.addWidget(self.edit_challenge_button, 5, 1, 1, 1)
                self.edit_challenge_button.clicked.connect(partial(self.edit_challenge, 'editing'))

                # Year combobox
                self.year_menu = QComboBox()
                available_years = [key for key in goal_data.keys() if key not in ('general_info', 'All')]
                _available_years = deepcopy(available_years)
                for year in _available_years:
                    if 'challenges_dict' in goal_data[year]['summary']:
                        if goal_data[year]['summary']['challenges_dict'] == {} and str(year) != str(self.current_year):
                            available_years.remove(year)
                # Don't show future years, since they have no statistic
                available_years = [year for year in available_years if int(year)
                                   <= datetime.datetime.now().astimezone().year]
                available_years.reverse()
                years_list = map(str, available_years)
                self.year_menu.addItems(years_list)
                self.year_menu.setCurrentText(str(self.current_year))
                self.year_menu.setFixedWidth(70)
                self.h_layout = QHBoxLayout()
                self.layout.addLayout(self.h_layout, 8, 0, 1, 1)
                self.h_layout.addWidget(self.year_menu)
                self.year_menu.setFocus()
                self.year_menu.currentTextChanged.connect(self.on_year_changed)
                self.year_menu.currentIndexChanged.connect(self.manage_buttons2)

                # Set up a listener to catch a selection change
                self.tree.itemSelectionChanged.connect(self.manage_buttons)

            def manage_buttons(self) -> None:
                if self.year_menu.currentText() == str(self.current_year):
                    if self.tree.selectedItems() and self.tree.selectedItems()[0].text(0) == _('Annual'):
                        self.remove_challenge_button.setEnabled(False)
                        self.edit_challenge_button.setEnabled(False)
                        if is_dark_theme():
                            self.remove_challenge_button.setIcon(get_icon('challenge_remove_disabled.png'))
                            self.edit_challenge_button.setIcon(get_icon('challenge_edit_disabled.png'))
                    else:
                        self.remove_challenge_button.setEnabled(True)
                        self.remove_challenge_button.setIcon(get_icon('challenge_remove.png'))
                        self.edit_challenge_button.setEnabled(True)
                        self.edit_challenge_button.setIcon(get_icon('challenge_edit.png'))

            def manage_buttons2(self) -> None:
                if self.year_menu.currentText() != str(self.current_year):
                    self.add_challenge_button.setEnabled(False)
                    self.remove_challenge_button.setEnabled(False)
                    self.edit_challenge_button.setEnabled(False)
                    if is_dark_theme():
                        self.add_challenge_button.setIcon(get_icon('challenge_add_disabled.png'))
                        self.remove_challenge_button.setIcon(get_icon('challenge_remove_disabled.png'))
                        self.edit_challenge_button.setIcon(get_icon('challenge_edit_disabled.png'))
                else:
                    self.add_challenge_button.setEnabled(True)
                    self.add_challenge_button.setIcon(get_icon('challenge_add.png'))
                    self.remove_challenge_button.setEnabled(True)
                    self.remove_challenge_button.setIcon(get_icon('challenge_remove.png'))
                    self.edit_challenge_button.setEnabled(True)
                    self.edit_challenge_button.setIcon(get_icon('challenge_edit.png'))

            def is_selection_empty(self) -> bool:
                if len(self.tree.selectedItems()) == 0:
                    error_dialog(_gui, _('Empty selection'), _('No challenge selected'),
                                 show_copy_button=False, show=True)
                    return True
                else:
                    return False

            def create_tree(self, year: str | int) -> QTreeWidget:
                # Get updated data
                g_data = JSONConfig('plugins/Reading_Goal_Data_' + library_id)

                tree = QTreeWidget()

                # Set item's spacing
                tree.setStyleSheet(
                    """
                    QTreeView::item {
                        margin-top: 10px;
                        margin-bottom: 10px;
                        height: 20px;
                    }
                    """
                )

                tree.setHeaderLabels([_('Challenge'), _('Start'), _('End'), _('Progress'), _('Books')])
                tree.header().setDefaultAlignment(Qt.AlignCenter)

                # Challenges
                challenges_dict = g_data[str(year)]['summary']['challenges_dict']

                # Set data
                for challenge, c_data in challenges_dict.items():
                    if challenge == 'Annual':
                        challenge = _('Annual')
                    start = c_data['start'].astimezone()
                    start = QtCore.QDate(start.year, start.month, start.day)
                    end = c_data['end'].astimezone()
                    end = QtCore.QDate(end.year, end.month, end.day)

                    child = QTreeWidgetItem(tree)
                    child.setData(0, Qt.DisplayRole, challenge)
                    child.setData(1, Qt.DisplayRole, start)
                    child.setData(2, Qt.DisplayRole, end)
                    # child.setData(3, Qt.DisplayRole, c_data['progress'])
                    child.setData(4, Qt.DisplayRole, c_data['count'] if 'count' in c_data else 0)

                    # Progress bar
                    bar = QProgressBar()
                    if is_dark_theme():
                        chunk_color = '#3684dd'
                    else:
                        chunk_color = '#369cdd'
                    style = (
                        """
                        QProgressBar{{
                            background-color: transparent;
                            text-align: center;
                            border: 0px
                        }}
                        QProgressBar::chunk {{
                            background: {0};
                            border-radius: 3px;
                        }}
                        """
                    ).format(chunk_color)
                    bar.setStyleSheet(style)
                    bar.setValue(c_data['progress'] if 'progress' in c_data else 0)
                    tree.setItemWidget(child, 3, bar)

                    if float(QtCore.PYQT_VERSION_STR[0:3]) >= 6.4:  # PyQt 6.4+ only
                        for i in range(1, 5):
                            child.setTextAlignment(i, QtCore.Qt.AlignCenter)

                    # Resize mode
                    tree.header().setStretchLastSection(False)
                    tree.header().setSectionResizeMode(0, QHeaderView.ResizeMode.Stretch)

                    # Set tree properties
                    tree.setRootIsDecorated(False)
                    tree.setAlternatingRowColors(True)
                    tree.sortItems(0, Qt.AscendingOrder)
                    tree.setSelectionMode(QAbstractItemView.SingleSelection)
                    tree.setCurrentItem(None)
                    tree.clearSelection()

                    # Restore state
                    if self.prefs['state']['CustomChallengesDialog']:
                        tree.header().restoreState(prefs['state']['CustomChallengesDialog'])
                    else:
                        tree.setColumnWidth(0, 300)
                        tree.setColumnWidth(1, 120)
                        tree.setColumnWidth(2, 120)
                        tree.setColumnWidth(3, 200)
                        tree.setColumnWidth(4, 60)

                return tree

            def on_year_changed(self, text: str | int) -> None:
                self.update_tree(text)

            def update_tree(self, year: str | int) -> None:
                # Save header state
                Dialog.save_state(self)
                # Update dialog with the new tree
                self.tree.hide()
                new_tree = self.create_tree(year)
                self.tree = new_tree
                self.tree.itemDoubleClicked.connect(partial(self.edit_challenge, 'editing'))
                self.layout.addWidget(self.tree, 0, 0, 7, 1)
                # Set up a listener to catch a selection change
                self.tree.itemSelectionChanged.connect(self.manage_buttons)

            def edit_dialog(self, title: str, icon: str, name: str | None, start: datetime.datetime | None,
                            end: datetime.datetime | None) -> dict | None:
                parent = self

                class EditChallengeDialog(Dialog):

                    def __init__(self):
                        Dialog.__init__(self, title, 'plugin-reading-goal-edit-challenge-dialog', parent=parent)
                        self.setWindowIcon(get_icon(icon))

                        self.values = {}

                    def setup_ui(self) -> None:
                        v_layout = QVBoxLayout(self)
                        v_layout.addSpacing(10)
                        layout = QGridLayout()
                        v_layout.addLayout(layout)
                        layout.setVerticalSpacing(25)

                        # Challenge name
                        self.challenge_label = QLabel(_('Name:'))
                        layout.addWidget(self.challenge_label, 0, 0, 1, 1)
                        self.challenge_name = QLineEdit()
                        self.challenge_name.setText(name if name else '')
                        layout.addWidget(self.challenge_name, 0, 1, 1, 5)

                        # Start date
                        y = datetime.datetime.now().astimezone().year
                        m = datetime.datetime.now().astimezone().month
                        d = datetime.datetime.now().astimezone().day
                        self.date_start_label = QLabel(_('Start') + ':')
                        layout.addWidget(self.date_start_label, 1, 0, 1, 1)
                        self.date_start_widget = DateEdit(self, create_clear_button=False)
                        # Challenges must be defined within the current year
                        self.date_start_widget.setMinimumDate(QtCore.QDate(y-1, 12, 31))  # Avoid getting 'undefined'
                        self.date_start_widget.setMaximumDate(QtCore.QDate(y, 12, 31))
                        self.date_start_widget.current_val = start if start else datetime.datetime(y, m, d).astimezone()
                        layout.addWidget(self.date_start_widget, 1, 1, 1, 2)

                        # End date
                        self.date_end_label = QLabel(_('End') + ':')
                        layout.addWidget(self.date_end_label, 1, 3, 1, 1)
                        self.date_end_widget = DateEdit(self, create_clear_button=False)
                        # Challenges must be defined within the current year
                        self.date_end_widget.setMinimumDate(QtCore.QDate(y, 1, 1))
                        self.date_end_widget.setMaximumDate(QtCore.QDate(y, 12, 31))
                        self.date_end_widget.current_val = end if end else datetime.datetime(y, 12, 31).astimezone()
                        layout.addWidget(self.date_end_widget, 1, 4, 1, 2)

                        # Ok/Cancel buttons
                        self.button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok |
                                                           QDialogButtonBox.StandardButton.Cancel)
                        layout.addWidget(self.button_box, 2, 1, 1, 5)
                        self.button_box.accepted.connect(self.accept)
                        self.button_box.rejected.connect(self.reject)

                    def reject(self) -> None:
                        Dialog.reject(self)
                        return None

                    def accept(self) -> Dialog | None:
                        challenges_dict = data['summary']['challenges_dict']
                        if self.challenge_name.text() in challenges_dict:
                            return error_dialog(self, _('Invalid challenge name'), _('This name is already in use'),
                                                show_copy_button=False, show=True)
                        # Check if all the fields are filled
                        self.values['name'] = self.challenge_name.text()
                        self.values['start'] = self.date_start_widget.current_val
                        self.values['end'] = self.date_end_widget.current_val
                        if all(i not in ('', None) for i in self.values.values()):
                            Dialog.accept(self)
                        return None

                e = EditChallengeDialog()
                if e.exec_() == e.Accepted:
                    return e.values
                else:
                    return None

            @staticmethod
            def update_challenge(result: dict, old_name: str = None) -> None:
                challenges_dict = data['summary']['challenges_dict']
                if not old_name:
                    # Add new challenge
                    challenges_dict[result['name']] = {}
                    challenges_dict[result['name']]['start'] = result['start'].astimezone()
                    challenges_dict[result['name']]['end'] = result['end'].astimezone()
                else:
                    # Edit the challenges dict
                    challenges_dict[result['name']] = challenges_dict.pop(old_name)
                    challenges_dict[result['name']]['start'] = result['start'].astimezone()
                    challenges_dict[result['name']]['end'] = result['end'].astimezone()
                    # Edit the info across the book list
                    for book_id in data:
                        if book_id != 'summary':
                            if old_name in data[book_id]['challenges']:
                                c_list = data[book_id]['challenges']
                                c_list = map(lambda x: x.replace(old_name, result['name']), c_list)
                                data[book_id]['challenges'] = c_list
                tool.update_json(data)

            def edit_challenge(self, mode: str) -> None:
                result = None
                old_name = None
                if self.year_menu.currentText() != str(self.current_year):
                    return
                if self.tree.selectedItems():
                    if self.tree.selectedItems()[0].data(0, Qt.DisplayRole) == _('Annual'):
                        return
                if mode == 'editing':
                    if not self.is_selection_empty():
                        title, icon = _('Edit challenge'), 'challenge_edit.png'
                        old_name = self.tree.selectedItems()[0].data(0, Qt.DisplayRole)
                        old_start = self.tree.selectedItems()[0].data(1, Qt.DisplayRole)
                        old_start = datetime.datetime(old_start.year(), old_start.month(), old_start.day()).astimezone()
                        old_end = self.tree.selectedItems()[0].data(2, Qt.DisplayRole)
                        old_end = datetime.datetime(old_end.year(), old_end.month(), old_end.day()).astimezone()
                        if old_name != 'Annual':
                            result = self.edit_dialog(title, icon, old_name, old_start, old_end)
                elif mode == 'adding':
                    title, icon = _('Add challenge'), 'challenge_add.png'
                    name = start = end = None
                    result = self.edit_dialog(title, icon, name, start, end)

                if result:
                    if mode == 'editing':
                        self.update_challenge(result, old_name=old_name)
                    elif mode == 'adding':
                        self.update_challenge(result)
                    self.update_tree(self.current_year)

                    if mode == 'adding':
                        info_dialog(self, _('Challenge added'), _('The new challenge was created'),
                                    show_copy_button=False, show=True)

            def remove_challenge(self) -> None:
                if not self.is_selection_empty():
                    ans = question_dialog(self, _('Are you sure?'),
                                          _('This can\'t be undone. Are you sure you want to remove the selected '
                                            'challenge?'),
                                          skip_dialog_name='plugin-reading-goal-remove-challenge-again',
                                          skip_dialog_msg=_('Show this confirmation again'), show_copy_button=False)
                    if ans:
                        name = self.tree.selectedItems()[0].text(0)
                        # Remove it from the book info
                        for book_id in data:
                            if book_id != 'summary':
                                if name in data[book_id]['challenges']:
                                    data[book_id]['challenges'].remove(name)
                        # Remove the challenge from the summary
                        del data['summary']['challenges_dict'][name]

                        tool.update_json(data)
                        tool.update_summary(data, mode='updating')
                        self.update_tree(self.current_year)

            def sizeHint(self) -> QSize:
                return QSize(800, 350)

        d = CustomChallengesDialog()
        d.exec_()

        return None

    def mark_rows(self, ids: dict[int, str] | Iterable[int], marked_text: str = '') -> None:
        if marked_text:
            marked_ids = dict.fromkeys(ids, marked_text)
            self.gui.current_db.set_marked_ids(marked_ids)
            self.gui.search.set_search_string('marked:%s' % marked_text)
        else:
            self.gui.current_db.set_marked_ids(ids)
            if 'invalid_year' not in ids.values():
                self.gui.search.set_search_string('marked:missing_data')
            elif 'missing_data' not in ids.values():
                self.gui.search.set_search_string('marked:invalid_year')
            else:
                self.gui.search.set_search_string('marked:true')

    def customize_child(self, child: QTreeWidgetItem, book_shelf: str, reread_count: int = None) -> None:
        color_dict = {}
        # Select foreground/background color based on book shelf
        if self.prefs['shelf_colors']:
            try:
                color_dict = eval(self.prefs['shelf_colors'])
            except:
                error_dialog(self.gui, _('Reading Goal: Invalid python dictionary'),
                             _('Verify your shelf colors dictionary. It must be a valid python dictionary.'),
                             show_copy_button=False, show=True)
        if book_shelf == 'Abandoned':
            fg_color = 'white'
        else:
            fg_color = 'black'
        if book_shelf == 'Want to read':
            if self.prefs['shelf_colors'] and 1 in color_dict:
                bg_color = QtGui.QBrush(QtGui.QColor(color_dict[1][0], color_dict[1][1], color_dict[1][2]))
            else:
                bg_color = QtGui.QBrush(QtGui.QColor(180, 230, 255))
        elif book_shelf == 'Reading':
            if self.prefs['shelf_colors'] and 2 in color_dict:
                bg_color = QtGui.QBrush(QtGui.QColor(color_dict[2][0], color_dict[2][1], color_dict[2][2]))
            else:
                bg_color = QtGui.QBrush(QtGui.QColor(255, 255, 145))
        elif book_shelf == 'Rereading':
            if self.prefs['shelf_colors'] and 3 in color_dict:
                bg_color = QtGui.QBrush(QtGui.QColor(color_dict[3][0], color_dict[3][1], color_dict[3][2]))
            else:
                bg_color = QtGui.QBrush(QtGui.QColor(250, 190, 80))
        elif book_shelf == 'Reread' or reread_count > 0:
            if reread_count >= 1:
                if self.prefs['keep_rereading_color']:
                    if self.prefs['shelf_colors'] and 3 in color_dict:
                        bg_color = QtGui.QBrush(QtGui.QColor(color_dict[3][0], color_dict[3][1], color_dict[3][2]))
                    else:
                        bg_color = QtGui.QBrush(QtGui.QColor(250, 190, 80))
                else:
                    if self.prefs['shelf_colors'] and 4 in color_dict:
                        bg_color = QtGui.QBrush(QtGui.QColor(color_dict[4][0], color_dict[4][1], color_dict[4][2]))
                    else:
                        bg_color = QtGui.QBrush(QtGui.QColor(180, 255, 180))
            else:
                bg_color = QtGui.QBrush(QtGui.QColor(180, 255, 180))
        elif book_shelf == 'Read':
            if self.prefs['shelf_colors'] and 4 in color_dict:
                bg_color = QtGui.QBrush(QtGui.QColor(color_dict[4][0], color_dict[4][1], color_dict[4][2]))
            else:
                bg_color = QtGui.QBrush(QtGui.QColor(180, 255, 180))
        else:
            if self.prefs['shelf_colors'] and 5 in color_dict:
                bg_color = QtGui.QBrush(QtGui.QColor(color_dict[5][0], color_dict[5][1], color_dict[5][2]))
            else:
                bg_color = QtGui.QBrush(QtGui.QColor(120, 120, 120))

        for i in range(0, 8):
            child.setForeground(i, QtGui.QBrush(QtGui.QColor(fg_color)))
            child.setBackground(i, QtGui.QColor(bg_color))

        # Set text alignment (PyQt 6.4+ only)
        if float(QtCore.PYQT_VERSION_STR[0:3]) >= 6.4:
            child.setTextAlignment(5, QtCore.Qt.AlignCenter)
            child.setTextAlignment(6, QtCore.Qt.AlignCenter)
            child.setTextAlignment(7, QtCore.Qt.AlignCenter)

    @staticmethod
    def update_items_count(tree: QTreeWidget) -> None:
        # Set book count on parent items
        iterator = QTreeWidgetItemIterator(tree)
        while iterator.value():
            parent_unique_book_ids = set()
            item = iterator.value()  # Parent
            item_child_count = item.childCount()
            if item_child_count > 0:
                for i in range(0, item_child_count):
                    child_unique_book_ids = set()
                    while item.childCount() > 0:  # Look for a book, which has no child
                        item = item.child(0)  # First child becomes the parent
                    for b in range(0, item.parent().childCount()):  # Iterate over all the children
                        if item.parent().child(b).text(10):  # Has a title - Book entry
                            child_unique_book_ids.add(item.parent().child(b).text(1))  # Add to category list
                            parent_unique_book_ids.add(item.parent().child(b).text(1))  # Add to outer list
                    if len(child_unique_book_ids) > 0:  # Category has children
                        has_count = re.search('\\(\\d+\\)', item.parent().text(0))
                        if has_count:
                            parent_text = re.sub(' \\(\\d+\\)', '', item.parent().text(0))
                        else:
                            parent_text = item.parent().text(0)
                        if not item.text(10):
                            item.parent().setText(0, parent_text + ' (' + str(len(child_unique_book_ids)) + ')')
                    if i < (item_child_count - 1):  # Check if the outer list is over
                        item = iterator.value().child(i+1)

            has_count = re.search('\\(\\d+\\)', iterator.value().text(0))
            if has_count:
                parent_text = re.sub(' \\(\\d+\\)', '', iterator.value().text(0))
            else:
                parent_text = iterator.value().text(0)
            if not iterator.value().text(1):  # Not a book/record - must be a category
                iterator.value().setText(0, parent_text + ' (' + str(len(parent_unique_book_ids)) + ')')
            iterator += 1

    def customize_tree(self, tree: QTreeWidget) -> None:
        self.update_items_count(tree)

        # Restore state
        if self.prefs['state']['EditGoalDialog']:
            self.tree.header().restoreState(prefs['state']['EditGoalDialog'])
        else:
            tree.setColumnWidth(0, 350)
            tree.setColumnWidth(3, 200)
            tree.setColumnWidth(4, 200)
            tree.setColumnWidth(5, 80)
            tree.setColumnWidth(6, 100)
            tree.setColumnWidth(7, 60)

        # Resize mode
        tree.header().setStretchLastSection(False)
        tree.header().setSectionResizeMode(0, QHeaderView.ResizeMode.Stretch)

        # Selection mode
        tree.setSelectionMode(QAbstractItemView.ExtendedSelection)

        # Enable sorting
        tree.setSortingEnabled(True)
        tree.sortItems(0, Qt.AscendingOrder)
        if self.prefs['sorting'] == 'Date (asc)':
            tree.header().setSortIndicator(6, Qt.AscendingOrder)
        elif self.prefs['sorting'] == 'Date (desc)':
            tree.header().setSortIndicator(6, Qt.DescendingOrder)

        # Enable animation
        tree.setAnimated(True)

        # Other properties
        self.tree.setExpandsOnDoubleClick(True)
        self.tree.setRootIsDecorated(True)

        # Hide columns
        for col in (1, 2, 8, 9, 10):
            tree.setColumnHidden(col, True)

    @staticmethod
    def get_tooltip(records: dict, page_count: int, book_shelf: str) -> tuple[str, int]:
        speed = 0
        tooltip = ''
        if len(records) > 1:
            book_records = dict(sorted(records.items(), key=lambda x: int(x[0])))
            f_idx, first_record = list(book_records.items())[0]
            l_idx, last_record = list(book_records.items())[-1]
            first_valid_record = first_record
            if not first_record['date']:
                for idx, record in list(book_records.items()):
                    if record['date']:
                        first_valid_record = record
                        break
            try:
                days = (last_record['date'].astimezone().date() -
                        first_valid_record['date'].astimezone().date()).days + 1
                days = days if days > 0 else 1
            except:
                days = 1
            pages = last_record['read_pages'] - first_valid_record['read_pages']
            speed = round(pages / days, 1)
            r_days = round((page_count - last_record['read_pages']) / speed) if speed > 0 else '∞'
            if book_shelf in ('Reading', 'Rereading'):
                if r_days == 1:
                    tooltip = _(str(speed) + _(' pages/day  |  ') + str(r_days) + _(' day to finish'))
                else:
                    tooltip = _(str(speed) + _(' pages/day  |  ') + str(r_days) + _(' days to finish'))
            elif book_shelf in ('Read', 'Reread'):
                # tooltip = _(str(speed) + _(' pages/day'))
                r_days = round((page_count - first_valid_record['read_pages']) / speed) if speed > 0 else '∞'
                if r_days == 1:
                    tooltip = _(str(speed) + _(' pages/day  |  ') + _('finished in ') + str(r_days) + _(' day'))
                else:
                    tooltip = _(str(speed) + _(' pages/day  |  ') + _('finished in ') + str(r_days) + _(' days'))
        return tooltip, speed

    def edit_tree(self, _data: dict[str, dict], ids: list[str] = None) -> QTreeWidget:
        ids = ids if ids else []

        # Get preferences
        self.prefs = prefs

        # QTreeWidget displaying books by shelf
        self.tree = QTreeWidget()

        layout = QVBoxLayout()
        layout.addWidget(self.tree)
        self.tree.setHeaderLabels([_('Title'), 'book_ids', 'shelf', _('Authors'), _('Series'),
                                   _('Progress'), _('Date'), _('Pages'), 'records', 'genre', 'title'])
        self.tree.header().setDefaultAlignment(Qt.AlignCenter)

        # Shelves
        self.want_read = QTreeWidgetItem(self.tree)
        self.want_read.setText(0, '1. ' + _('Want to read'))
        self.want_read.setExpanded(True)

        self.reading = QTreeWidgetItem(self.tree)
        self.reading.setText(0, '2. ' + _('Reading'))
        self.reading.setExpanded(True)

        # Check the use of the rereading shelf
        if self.prefs['rereading_shelf']:
            self.rereading = QTreeWidgetItem(self.tree)
            self.rereading.setText(0, '3. ' + _('Rereading'))
            self.rereading.setExpanded(True)

            self.read = QTreeWidgetItem(self.tree)
            self.read.setText(0, '4. ' + _('Read'))
            self.read.setExpanded(True)

            self.abandoned = QTreeWidgetItem(self.tree)
            self.abandoned.setText(0, '5. ' + _('Abandoned'))
            self.abandoned.setExpanded(True)
        else:
            self.read = QTreeWidgetItem(self.tree)
            self.read.setText(0, '3. ' + _('Read'))
            self.read.setExpanded(True)

            self.abandoned = QTreeWidgetItem(self.tree)
            self.abandoned.setText(0, '4. ' + _('Abandoned'))
            self.abandoned.setExpanded(True)

        if self.prefs['auto_collapse']:
            auto_collapse_list = self.prefs['auto_collapse'].replace(' ', '').split(',')
            for shelf in auto_collapse_list:
                if shelf == '1':
                    self.want_read.setExpanded(False)
                elif shelf == '2':
                    self.reading.setExpanded(False)
                elif shelf == '3':
                    if self.prefs['rereading_shelf']:
                        self.rereading.setExpanded(False)
                    else:
                        self.read.setExpanded(False)
                elif shelf == '4':
                    if self.prefs['rereading_shelf']:
                        self.read.setExpanded(False)
                    else:
                        self.abandoned.setExpanded(False)
                elif shelf == '5':
                    if self.prefs['rereading_shelf']:
                        self.abandoned.setExpanded(False)

        if ids:
            new_data = {}
            for book_id in _data:
                if book_id in ids:
                    new_data[book_id] = _data[book_id]
            _data = new_data

        # Add books to the tree
        for book_id, values in _data.items():
            if book_id != 'summary':
                try:
                    series = values['series']
                except KeyError:
                    series = ''
                (title, title_sort, authors, reading_progress, date, page_count, book_shelf, records, genre,
                 reread_count, challenges) = (values['title'], values['title_sort'], ' & '.join(values['authors']),
                                              values['status'], values['date'], values['page_count'], values['shelf'],
                                              values['records'], values['genre'], values['reread_count'],
                                              values['challenges'])
                reading_progress = '{0:.0f}%'.format(reading_progress)
                if date:
                    date = date.astimezone()
                    date_time = QtCore.QDateTime(QtCore.QDate(date.year, date.month, date.day),
                                                 QtCore.QTime(date.hour, date.minute, date.second))
                    date = QtCore.QDate(date.year, date.month, date.day)
                else:
                    date_time = QtCore.QDateTime(1970, 1, 1, 1, 1, 1)
                title_opt = 'title_sort' if (self.prefs['sorting']
                                             in ('Title sort', 'Date (asc)', 'Date (desc)')) else 'title'
                if genre and type(genre) is list:
                    if len(genre) > 1:
                        genre = ', '.join(str(i) for i in genre)
                    else:
                        genre = genre[0]
            else:
                break

            if book_shelf == 'Want to read':
                parent = self.want_read
            elif book_shelf == 'Reading':
                parent = self.reading
            elif book_shelf in ('Read', 'Reread'):
                parent = self.read
            elif book_shelf == 'Abandoned':
                parent = self.abandoned
            else:
                if self.prefs['rereading_shelf']:
                    parent = self.rereading
                else:
                    parent = self.reading

            _challenges = _('Challenges: ')
            title_tooltip = None
            if len(challenges) > 1:
                for chal in challenges:
                    if chal != 'Annual':
                        _challenges += chal + ', '
                title_tooltip = values[title_opt] + '\n\n' + _challenges

            speed_tooltip, speed = self.get_tooltip(records, page_count, book_shelf)

            if (reread_count > 0 and (self.prefs['rereading_shelf'] or self.prefs['use_rereading_color'])
                    and self.prefs['use_rereading_counter']):
                reread_suffix = ' (' + str(reread_count) + 'x)'
            else:
                reread_suffix = ''

            child = QTreeWidgetItem(parent, [values[title_opt] + reread_suffix, book_id, book_shelf, str(authors)])
            if title_tooltip:
                child.setToolTip(0, title_tooltip.strip(', '))
            else:
                child.setToolTip(0, values[title_opt])
            child.setToolTip(3, str(authors))
            child.setData(4, Qt.DisplayRole, series)
            child.setToolTip(4, series)
            child.setData(5, Qt.DisplayRole, reading_progress)
            child.setData(6, Qt.DisplayRole, date)
            child.setToolTip(6, QtCore.QLocale.system().toString(date_time))
            child.setData(7, Qt.DisplayRole, page_count)
            if book_shelf != 'Want to read':
                child.setToolTip(7, speed_tooltip if speed > 0 else _('N/A'))
            child.setData(9, Qt.DisplayRole, genre)
            child.setData(10, Qt.DisplayRole, title)
            child.setData(11, Qt.DisplayRole, date_time)

            self.customize_child(child, book_shelf, reread_count=reread_count)

            parent.addChild(child)

        self.customize_tree(self.tree)

        return self.tree

    def edit_goal(self, db, data: dict[str, dict]) -> None:
        tree = self.edit_tree(data)
        self.book_list(_('Edit reading goal'), 'plugin-reading-goal-edit-goal-dialog',
                       get_icon('goal_edit.png'), tree, db, data)

    def book_list(self, title: str, unique_name: str, icon: str, tree: QTreeWidget, db, data: dict[str, dict]) -> None:
        tool = self
        _gui = self.gui

        class EditGoalDialog(Dialog):

            # Last current item
            last_book_current_item = None
            last_record_current_item = None

            def __init__(self):
                Dialog.__init__(self, title, unique_name, parent=tool.gui)
                self.setWindowIcon(icon)
                self.data = data
                self.view = None

                # Get preferences
                self.prefs = prefs

                # Queued jobs
                self.queue_dict = {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}
                self.remove_ids = []
                self.abandon_ids = []

            def setup_ui(self) -> None:
                dialog_whatsthis = self.get_whatsthis()
                Dialog.setWindowFlag(self, Qt.WindowContextHelpButtonHint, True)

                self.layout = QGridLayout(self)
                self.h_layout1 = QHBoxLayout()
                self.layout.addLayout(self.h_layout1, 0, 0, 1, 1)

                # --- Add top controls --- #
                if 'edit' in unique_name:
                    # Change view combobox
                    self.chang_view_combobox = QComboBox()
                    # Add standard views
                    self.chang_view_combobox.addItem(get_icon('shelf_view.png'), _('Shelf view'))
                    self.chang_view_combobox.addItem(get_icon('genre_view.png'), _('Genre view'))
                    # self.chang_view_combobox.setFixedWidth(70)
                    self.h_layout1.addWidget(self.chang_view_combobox)
                    self.chang_view_combobox.currentTextChanged.connect(partial(self.change_view, _filter=None))

                    # Year combobox
                    goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                    self.year_menu = QComboBox()
                    current_year = datetime.datetime.now().astimezone().year
                    available_years = [key for key in goal_data.keys() if key != 'general_info']
                    available_years.reverse()
                    years_list = map(str, available_years)
                    self.year_menu.addItems(years_list)
                    self.year_menu.addItem(_('All'))
                    self.year_menu.setCurrentText(str(current_year))
                    self.year_menu.setFixedWidth(70)
                    self.h_layout1.addWidget(self.year_menu)
                    self.year_menu.currentTextChanged.connect(self.update_tree)
                    self.year_menu.currentIndexChanged.connect(self.manage_buttons)

                    # Add custom views
                    self.add_views(current_year)

                    # Expand / collapse all buttons
                    self.tree = tree
                    self.ec_button_box = QDialogButtonBox()
                    self.expand_button = self.ec_button_box.addButton(_('Expand all'), QDialogButtonBox.ActionRole)
                    self.collapse_button = self.ec_button_box.addButton(_('Collapse all'), QDialogButtonBox.ActionRole)
                    self.expand_button.setAutoDefault(False)
                    self.collapse_button.setAutoDefault(False)
                    self.expand_button.clicked.connect(self.expand_all)
                    self.collapse_button.clicked.connect(self.collapse_all)
                    self.h_layout1.addWidget(self.ec_button_box)

                # --- Add tree --- #
                self.tree = tree
                self.tree.setWhatsThis(dialog_whatsthis)
                self.layout.addWidget(self.tree, 1, 0, 6, 1)

                # --- Add lateral controls --- #
                # Relink button
                self.relink_button = QPushButton(get_icon('add_link.png'), '', self)
                self.relink_button.setToolTip(_('Link to another book'))
                self.relink_button.setAutoDefault(False)
                self.layout.addWidget(self.relink_button, 2, 1, 2, 1)
                self.relink_button.clicked.connect(partial(self.handler, option='add_link'))

                # Remove button
                self.remove_button = QPushButton(get_icon('goal_remove.png'), '', self)
                self.remove_button.setToolTip(_('Remove from reading goal'))
                self.layout.addWidget(self.remove_button, 3, 1, 2, 1)
                self.remove_button.setAutoDefault(False)
                self.remove_button.clicked.connect(self.remove)

                # Abandon button
                self.abandon_button = QPushButton(get_icon('abandon.png'), '', self)
                self.abandon_button.setToolTip(_('Mark the book as abandoned'))
                self.abandon_button.setAutoDefault(False)
                self.layout.addWidget(self.abandon_button, 4, 1, 2, 1)
                self.abandon_button.clicked.connect(self.abandon)

                h_layout = QHBoxLayout()
                self.layout.addLayout(h_layout, 7, 0, 1, 1)

                # --- Add bottom controls --- #
                if 'edit' in unique_name:
                    # --- Database menu --- #
                    b = self.database_button = QPushButton(get_icon('database.png'), _('Database'), self)
                    b.setAutoDefault(False)
                    menu = self.database_menu = QMenu(b)
                    self._backup = menu.addAction(get_icon('backup.png'), _('Backup'),
                                                  partial(self.backup_database, option='backup'))
                    self._import = menu.addAction(get_icon('import.png'), _('Import'),
                                                  partial(self.backup_database, option='import'))
                    self._clear = menu.addAction(get_icon('delete.png'), _('Clear'),
                                                 partial(self.backup_database, option='clear'))
                    b.setMenu(menu)

                    h_layout.addWidget(b)

                # Apply / Cancel buttons
                self.ac_button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Apply |
                                                      QDialogButtonBox.StandardButton.Cancel)
                self.ac_button_box.button(QDialogButtonBox.StandardButton.Apply).setEnabled(False)
                self.ac_button_box.button(QDialogButtonBox.StandardButton.Apply).clicked.connect(
                    partial(self.handler, option='apply_changes'))
                self.ac_button_box.rejected.connect(self.reject)

                # Mark selected button
                if 'edit' in unique_name:
                    self.show_marked_button = self.ac_button_box.addButton(_('Mark selected books'),
                                                                           QDialogButtonBox.ActionRole)
                    self.show_marked_button.setIcon(QIcon.ic('marked.png'))
                    self.show_marked_button.setAutoDefault(False)
                    self.show_marked_button.clicked.connect(partial(self.handler, option='mark_books'))

                h_layout.addWidget(self.ac_button_box)

                # Set up a listener to catch a selection change
                self.tree.itemSelectionChanged.connect(self.manage_buttons)

                if prefs['jump_to_selected']:
                    ids = _gui.library_view.get_selected_ids()
                    if len(ids) == 1:
                        item = self.tree.findItems(str(ids[0]), Qt.MatchFixedString | Qt.MatchRecursive, 1)
                        if len(item) == 1:
                            self.tree.setCurrentItem(item[0])
                else:
                    self.tree.setCurrentItem(None)

            def expand_all(self) -> None:
                self.tree.expandAll()

            def collapse_all(self) -> None:
                self.tree.collapseAll()

            def remove(self) -> None:
                if 'edit' in unique_name:
                    if self.year_menu.currentText() == str(datetime.datetime.now().astimezone().year):
                        self.handler(option='remove_from_goal')
                    else:
                        self.handler(option='remove_other_years')
                else:
                    self.handler(option='remove_from_goal')

            def abandon(self) -> None:
                if 'edit' in unique_name:
                    if self.year_menu.currentText() == str(datetime.datetime.now().astimezone().year):
                        self.handler(option='abandon')
                    else:
                        self.handler(option='abandon_other_years')
                else:
                    self.handler(option='abandon')

            @staticmethod
            def get_whatsthis() -> str:
                # Help information
                theme_suffix = '_dark' if is_dark_theme() else '_light'
                icons_path = str(os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images'))
                help_img_src = '<img src="' + os.path.join(icons_path, 'help.png') + '" width="25" height="25"> '
                link_img_src = os.path.join(icons_path, 'add_link' + theme_suffix + '.png')
                remove_img_src = os.path.join(icons_path, 'goal_remove' + theme_suffix + '.png')
                abandon_img_src = os.path.join(icons_path, 'abandon' + theme_suffix + '.png')
                records_img_src = os.path.join(icons_path, 'records' + theme_suffix + '.png')
                li_style = ' style="vertical-align:top; margin: 0.5em 0"'
                if 'edit' in unique_name:
                    show_records_whatsthis = \
                        '<li' + li_style + '>' + '<b><img src="' + records_img_src + '" width="25" height="25"> ' + \
                        _('Show book\'s records</b>: manage your reading records for this book. A new record '
                          'is created every time you change your reading reading progress. You can delete '
                          'past records, but not the the current one.') + '</li>'
                    introduction_whatsthis = \
                        '<h4 style="vertical-align:top">' + help_img_src + title + '</h4>' + \
                        '<p>' + _('Here you can easily edit your reading goal. Just select the books you want to edit '
                                  'and then click on the desired option on the right.') + '</p>' + \
                        '<p>' + _('The options are:') + '</p>'
                else:
                    show_records_whatsthis = ''
                    introduction_whatsthis = \
                        '<h4 style="vertical-align:top">' + help_img_src + title + '</h4>' + \
                        '<p>' + _('Here you must deal with the books that were removed from calibre. Just select the '
                                  'books you want to edit and then click on the desired option on the right.') + \
                        '</p>' + \
                        '<p>' + _('The options are:') + '</p>'
                dialog_whatsthis = \
                    introduction_whatsthis + \
                    '<ul style="list-style-type:none">' + \
                    '<li' + li_style + '>' + '<b><img src="' + link_img_src + '" width="25" height="25"> ' + \
                    _('Link to another book</b>: you can change the calibre book linked to this entry. Useful '
                      'in case you want to change the edition you are reading, or when the original book in '
                      'your library was removed, leaving the entry orphaned.') + '</li>' + \
                    '<li' + li_style + '>' + '<b><img src="' + remove_img_src + '" width="25" height="25"> ' + \
                    _('Remove from reading goal</b>: remove the book from you reading goal, just like using '
                      'the main menu option. No record is kept.') + '</li>' + \
                    '<li' + li_style + '>' + '<b><img src="' + abandon_img_src + '" width="25" height="25"> ' + \
                    _('Mark the book as abandoned</b>: differently from removing, the existing records are '
                      'kept, and the last reading progress still counts for your goal statistics. '
                      'Abandoned books will not be updated or moved to your next year\'s goal.') + \
                    '</li>' + \
                    show_records_whatsthis + \
                    '</ul>' + \
                    '<p>' + _('After you are done editing, you must apply the changes. They are committed only when '
                              'you press the apply button.') + '</p>'

                return dialog_whatsthis

            def get_all_books(self) -> dict[str, dict]:
                goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                updated_data = {}
                for year, values in goal_data.items():
                    if year != 'general_info':
                        for book_id, book_info in values.items():
                            if book_id != 'summary':
                                if book_id in updated_data:
                                    existing_records = updated_data[book_id]['records']
                                    updated_records = {}
                                    count = 0
                                    for k, v in existing_records.items():
                                        updated_records[str(count)] = v
                                        count += 1
                                    new_records = book_info['records']
                                    for k, v in new_records.items():
                                        updated_records[str(count)] = v
                                        count += 1
                                    book_info['records'] = updated_records
                                updated_data[book_id] = book_info
                self.relink_button.setEnabled(False)
                self.remove_button.setEnabled(False)
                self.abandon_button.setEnabled(False)
                if is_dark_theme():
                    self.relink_button.setIcon(get_icon('add_link_disabled.png'))
                    self.remove_button.setIcon(get_icon('goal_remove_disabled.png'))
                    self.abandon_button.setIcon(get_icon('abandon_disabled.png'))
                return updated_data

            def update_tree(self) -> None:
                # Save header state
                Dialog.save_state(self)

                # Check for unapplied changes when changing view
                if self.queue_dict != {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}:
                    ans = question_dialog(self, _('Unapplied changes'),
                                          _('You have unapplied changes. Are you sure you want to discard them?'),
                                          show_copy_button=False)
                    if ans:
                        self.queue_dict = {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}
                        self.ac_button_box.button(QDialogButtonBox.StandardButton.Apply).setEnabled(False)
                    else:
                        return

                year = self.year_menu.currentText()

                if not year:
                    year = str(datetime.datetime.now().astimezone().year)
                goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                if year not in goal_data and year != _('All'):
                    goal_data[year] = {}

                # All years view
                if year == _('All'):
                    updated_data = self.get_all_books()
                    count = self.chang_view_combobox.count()
                    for i in reversed(range(2, count)):
                        self.chang_view_combobox.removeItem(i)
                else:
                    updated_data = deepcopy(goal_data[year])
                    # Add custom views for the current year
                    self.add_views(year)

                # Create an updated tree
                new_tree = tool.edit_tree(updated_data)

                # Hide the old tree
                self.tree.hide()
                # Add the updated tree
                self.tree = new_tree
                self.layout.addWidget(self.tree, 1, 0, 6, 1)
                # Use the new updated data
                self.data = updated_data

                self.change_view()

                # Redefine What's This
                dialog_whatsthis = self.get_whatsthis()
                self.tree.setWhatsThis(dialog_whatsthis)

            @staticmethod
            def generate_tag_dict(book_tags: set) -> dict:
                tag_dict = {}
                for tag in sorted(book_tags):
                    node = tag_dict
                    for level in tag.split('.', 2):
                        if level:
                            node = node.setdefault(level, dict())
                return tag_dict

            def build_tag_tree(self, tag_dict: dict, parent: QTreeWidgetItem = None) -> None:
                for key, value in tag_dict.items():
                    item = QTreeWidgetItem(parent)
                    item.setText(0, (key if key != 'Undefined' else '[' + _('No genre') + ']'))
                    if isinstance(value, dict):
                        self.build_tag_tree(tag_dict=value, parent=item)

            def get_tree(self, book_tags: set) -> QTreeWidget:
                _tree = QTreeWidget()
                _tree.setHeaderLabels([_('Title'), 'book_ids', 'shelf', _('Authors'),
                                      _('Series'), _('Progress'), _('Date'), _('Pages')])
                _tree.header().setDefaultAlignment(Qt.AlignHCenter)
                _tree.setSelectionMode(QAbstractItemView.ExtendedSelection)
                tag_dict = self.generate_tag_dict(book_tags)
                self.build_tag_tree(tag_dict, _tree)
                return _tree

            def add_views(self, year: str | int) -> None:
                # Add custom views
                goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                updated_data = deepcopy(goal_data[str(year)])
                if 'summary' in updated_data and 'challenges_dict' in updated_data['summary']:
                    challenges_dict = updated_data['summary']['challenges_dict']
                    try:
                        count = self.chang_view_combobox.count()
                        for i in reversed(range(2, count)):
                            self.chang_view_combobox.removeItem(i)
                    except:
                        pass
                    if len(challenges_dict) > 1:
                        self.chang_view_combobox.insertSeparator(2)
                        for challenge in challenges_dict:
                            if challenge != 'Annual':
                                self.chang_view_combobox.addItem(challenge)

            def shelf_view(self) -> None:
                # Update buttons
                try:
                    self.ec_button_box.removeButton(self.filter_button)
                except:
                    pass

                year = self.year_menu.currentText()
                if not year:
                    year = str(datetime.datetime.now().astimezone().year)
                if year == _('All'):
                    _updated_data = self.get_all_books()
                else:
                    goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                    _updated_data = deepcopy(goal_data[year])
                # Hide the old tree
                self.tree.hide()
                # Add the updated tree
                new_tree = ReadingGoalTools.edit_tree(tool, _updated_data)
                self.tree = new_tree
                # self.tree.expandAll()
                self.layout.addWidget(self.tree, 1, 0, 6, 1)

            def genre_view(self, _filter: str = None) -> None:
                # Update buttons
                try:
                    self.ec_button_box.removeButton(self.filter_button)
                except:
                    pass

                # Get the complete data, in case it has been filtered
                year = self.year_menu.currentText()
                if not year:
                    year = str(datetime.datetime.now().astimezone().year)
                if year == _('All'):
                    _updated_data = self.get_all_books()
                else:
                    goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                    _updated_data = deepcopy(goal_data[year])
                # Hide the old tree
                self.tree.hide()
                # Add the updated tree
                new_tree = ReadingGoalTools.edit_tree(tool, _updated_data)
                self.tree = new_tree

                # Get a list of all tags
                all_tags = set()
                iterator = QTreeWidgetItemIterator(self.tree)
                while iterator.value():
                    item = iterator.value()
                    if item.parent() and item.text(0):
                        tags = item.text(9).split(', ')
                        for tag in tags:
                            all_tags.add(tag)
                    iterator += 1
                # Create the tag tree hierarchy
                tag_tree = self.get_tree(all_tags)

                # Create an updated tree
                new_tree = tool.edit_tree(_updated_data)

                # Add books to the tree
                iterator = QTreeWidgetItemIterator(new_tree)
                while iterator.value():
                    item = iterator.value()
                    tags = item.text(9).split(', ')
                    for tag in tags:
                        if tag:
                            gp = None
                            parent = None
                            # Remove trailing dots
                            tag = tag.strip('.')
                            # Split hierarchical tags up to three levels
                            tag = tag.split('.', 2)
                            parent_name = tag[-1] if tag[0] != 'Undefined' else '[' + _('No genre') + ']'
                            grandparent_name = tag[-2] if tag[0] != 'Undefined' and len(tag) > 1 else None
                            if grandparent_name:
                                # List of all possible grandparents
                                gp_list = tag_tree.findItems(grandparent_name, Qt.MatchFixedString |
                                                             Qt.MatchCaseSensitive | Qt.MatchRecursive, 0)
                                for gp_candidate in gp_list:
                                    for i in range(0, gp_candidate.childCount()):
                                        # Find the right grandparent by checking the parent name
                                        if gp_candidate.child(i).text(0) == parent_name:
                                            gp = gp_candidate
                            parent_list = tag_tree.findItems(parent_name, Qt.MatchFixedString | Qt.MatchCaseSensitive |
                                                             Qt.MatchRecursive, 0)
                            for par in parent_list:
                                if gp:
                                    if par.parent() == gp:
                                        parent = par
                                else:
                                    if not par.parent():
                                        parent = par

                            child = QTreeWidgetItem(parent)
                            # Check if a filter is active
                            reread_count = 0
                            if _filter:
                                reread_count = _updated_data[str(item.text(1))]['reread_count']
                                for i in range(0, 12):
                                    if item.text(2) == _filter:
                                        child.setData(i, Qt.DisplayRole, item.data(i, Qt.DisplayRole))
                                    if _filter == 'Read':
                                        if item.text(2) == 'Reread' or reread_count >= 1:
                                            child.setData(i, Qt.DisplayRole, item.data(i, Qt.DisplayRole))
                            else:
                                reread_count = _updated_data[str(item.text(1))]['reread_count']
                                for i in range(0, 12):
                                    child.setData(i, Qt.DisplayRole, item.data(i, Qt.DisplayRole))

                            # Mark the genre view tree
                            child.setData(12, Qt.DisplayRole, 'Genre view')

                            # Set tooltips
                            child.setToolTip(0, item.data(0, Qt.DisplayRole))
                            child.setToolTip(3, item.data(3, Qt.DisplayRole))
                            child.setToolTip(4, item.data(4, Qt.DisplayRole))

                            # Set date tooltip
                            book_id = item.text(1)
                            date = _updated_data[book_id]['date']
                            if date:
                                date = date.astimezone()
                                date_time = QtCore.QDateTime(QtCore.QDate(date.year, date.month, date.day),
                                                             QtCore.QTime(date.hour, date.minute, date.second))
                            else:
                                date_time = QtCore.QDateTime(1970, 1, 1, 1, 1, 1)
                            child.setToolTip(6, QtCore.QLocale.system().toString(date_time))

                            # Set speed tooltip
                            records = _updated_data[book_id]['records']
                            book_shelf = item.text(2)
                            page_count = int(item.text(7))
                            tooltip, speed = tool.get_tooltip(records, page_count, book_shelf)
                            if book_shelf != 'Want to read':
                                child.setToolTip(7, tooltip if speed > 0 else _('N/A'))

                            ReadingGoalTools.customize_child(tool, child, item.text(2), reread_count=reread_count)
                    iterator += 1

                ReadingGoalTools.customize_tree(tool, tag_tree)

                # Remove empty genres
                if _filter:
                    purge_list = tag_tree.findItems('(0)', Qt.MatchEndsWith | Qt.MatchRecursive, 0)
                    for item in purge_list:
                        item_parent = item.parent()
                        if item_parent:
                            item.parent().takeChild(item.parent().indexOfChild(item))
                        else:
                            tag_tree.takeTopLevelItem(tag_tree.indexOfTopLevelItem(item))

                # Hide the old tree
                self.tree.hide()
                # Add the updated tree
                self.tree = tag_tree

                # Filter button
                fb = self.filter_button = self.ec_button_box.addButton(_('Filter'), QDialogButtonBox.ActionRole)
                fb.setAutoDefault(False)
                menu = self.database_menu = QMenu(fb)
                self.no_filter = menu.addAction(_('All'), partial(self.change_view, _filter=None))
                self.want_to_read = menu.addAction(_('Want to read'), partial(self.change_view, _filter='Want to read'))
                self.reading = menu.addAction(_('Reading'), partial(self.change_view, _filter='Reading'))
                if prefs['rereading_shelf']:
                    self.rereding = menu.addAction(_('Rereading'), partial(self.change_view, _filter='Rereading'))
                self.read = menu.addAction(_('Read'), partial(self.change_view, _filter='Read'))
                self.abandoned = menu.addAction(_('Abandoned'), partial(self.change_view, _filter='Abandoned'))
                fb.setMenu(menu)
                self.ec_button_box.addButton(self.expand_button, QDialogButtonBox.ActionRole)
                self.ec_button_box.addButton(self.collapse_button, QDialogButtonBox.ActionRole)

                self.tree.expandAll()
                self.layout.addWidget(self.tree, 1, 0, 6, 1)

                # Restore state
                if self.prefs['state']['EditGoalDialog']:
                    self.tree.header().restoreState(prefs['state']['EditGoalDialog'])

            def change_view(self, _filter: str = None) -> None:
                # Save header state
                Dialog.save_state(self)

                self.view = self.chang_view_combobox.currentText()
                if self.view == _('Genre view'):
                    # Check if the genre custom columns is set
                    custom_cols = _gui.library_view.model().custom_columns
                    if not self.prefs['genre_column'] or (self.prefs['genre_column'] not in custom_cols and
                                                          self.prefs['genre_column'] != 'tags'):
                        if question_dialog(self, _('Configuration needed'),
                                           _('You must choose the genre custom column.') +
                                           _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                            result = show_configuration(tool, tab=1)
                            if result and self.prefs['genre_column']:
                                goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                                _data = goal_data[str(datetime.datetime.now().astimezone().year)]
                                ReadingGoalTools.update_database(tool, db, _data)
                            else:
                                self.chang_view_combobox.setCurrentText(_('Shelf view'))
                                return None
                        else:
                            self.chang_view_combobox.setCurrentText(_('Shelf view'))
                            return None

                # Check for unapplied changes when changing view
                if self.queue_dict != {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}:
                    ans = question_dialog(self, _('Unapplied changes'),
                                          _('You have unapplied changes. Are you sure you want to discard them?'),
                                          show_copy_button=False)
                    if ans:
                        self.queue_dict = {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}
                        self.ac_button_box.button(QDialogButtonBox.StandardButton.Apply).setEnabled(False)
                    else:
                        return None

                self.view = self.chang_view_combobox.currentText()
                year = self.year_menu.currentText()

                if self.view in (_('Genre view'), _('Shelf view')) and year != _('All'):
                    # Enable the right buttons
                    self.relink_button.setEnabled(True)
                    self.remove_button.setEnabled(True)
                    self.abandon_button.setEnabled(True)
                    self.relink_button.setIcon(get_icon('add_link.png'))
                    self.remove_button.setIcon(get_icon('goal_remove.png'))
                    self.abandon_button.setIcon(get_icon('add_link.png'))

                if self.view in (_('Genre view'), _('Shelf view')):
                    count = self.year_menu.count()
                    if self.year_menu.itemText(count - 1) != _('All'):
                        self.year_menu.addItem(_('All'))

                if self.view == _('Genre view'):
                    self.genre_view(_filter=_filter)

                elif self.view == _('Shelf view'):
                    self.shelf_view()

                # Challenges view
                else:
                    # Update buttons
                    try:
                        self.ec_button_box.removeButton(self.filter_button)
                    except:
                        pass
                    # Disable the right buttons
                    self.relink_button.setEnabled(False)
                    self.remove_button.setEnabled(False)
                    self.abandon_button.setEnabled(False)
                    if is_dark_theme():
                        self.relink_button.setIcon(get_icon('add_link_disabled.png'))
                        self.remove_button.setIcon(get_icon('goal_remove_disabled.png'))
                        self.abandon_button.setIcon(get_icon('add_link_disabled.png'))

                    # Remove 'All' years option
                    count = self.year_menu.count()
                    if self.year_menu.itemText(count-1) == _('All'):
                        self.year_menu.removeItem(count-1)

                    year = self.year_menu.currentText()
                    goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                    self.data = goal_data[year]
                    new_data = deepcopy(self.data)

                    # Filter the database to show only books in this challenge
                    challenge = self.view

                    for book_id in self.data:
                        if book_id != 'summary':
                            if 'challenges' in self.data[book_id]:
                                if challenge not in self.data[book_id]['challenges']:
                                    del new_data[book_id]
                            else:
                                del new_data[book_id]

                    # Hide the old tree
                    self.tree.hide()
                    # Add the updated tree
                    new_tree = ReadingGoalTools.edit_tree(tool, new_data)
                    self.tree = new_tree
                    # self.tree.expandAll()
                    # self.data = new_data
                    self.layout.addWidget(self.tree, 1, 0, 6, 1)

                # Redefine our listener, since this is a new tree
                self.tree.itemSelectionChanged.connect(self.manage_buttons)

                # Redefine What's This
                dialog_whatsthis = self.get_whatsthis()
                self.tree.setWhatsThis(dialog_whatsthis)

                return None

            def backup_database(self, option: str = None) -> Dialog | None:
                try:
                    timestamp = str(datetime.date.today())
                    database_path = os.path.join(config_dir, 'plugins')
                    database_file = os.path.join(database_path, 'Reading_Goal_Data_' + tool.library_id + '.json')

                    # Backup database
                    if option == 'backup':
                        backup_file_path = choose_save_file(self, 'plugin-reading-goal-save-file',
                                                            _('Select a folder to save a copy of the database file'),
                                                            filters=[(_('JSON file'), ['json'])],
                                                            initial_filename='Reading_Goal_Data_' + timestamp + '.json')
                        if backup_file_path:
                            with open(database_file, 'r') as df:
                                with open(backup_file_path, 'w') as bf:
                                    bf.write(df.read())
                            return info_dialog(self, _('Backup'), _('Backup completed'),
                                               show_copy_button=False, show=True)

                    # Import a database file
                    elif option == 'import':
                        ans = question_dialog(self, _('Are you sure?'),
                                              _('Notice that the existing database will be completely replaced '
                                                'with the new one. Do you want to proceed?'),
                                              show_copy_button=False, override_icon=QIcon.ic('dialog_warning.png'))
                        if ans:
                            import_file_path = choose_files(self, 'plugin-reading-goal-import-file',
                                                            _('Select the database file you want to import'),
                                                            filters=[(_('JSON file'), ['json'])],
                                                            select_only_single_file=True)
                            if import_file_path:
                                with open(import_file_path[0], 'r') as i_f:
                                    candidate = i_f.read()
                                    if 'general_info' not in candidate and 'timestamp' not in candidate:
                                        return error_dialog(_gui, _('Invalid database'),
                                                            _('The selected file is not a valid database file'),
                                                            show_copy_button=True, show=True)
                                    with open(database_file, 'w') as b_f:
                                        b_f.write(candidate)

                                # -- Update our dialog with a new tree, since a new database was imported
                                # Create an updated copy of the current year database
                                goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                                if self.year_menu.currentText() in goal_data:
                                    year = self.year_menu.currentText()
                                else:
                                    year = str(datetime.datetime.now().astimezone().year)
                                new_goal_data = ReadingGoalTools.check_database(tool, db, year, mode='updating')[0]

                                # This is ugly, but JSONConfig object does not accept the dictionary .update() method
                                temp_data = deepcopy(new_goal_data)
                                year_list = []
                                for y in new_goal_data:
                                    if y != 'general_info':
                                        year_list.append(y)
                                for y in year_list:
                                    del new_goal_data[str(y)]  # We have to first delete every year
                                for _y, _b_info in temp_data.items():
                                    new_goal_data[str(_y)] = _b_info  # Then, add it again with the updated info

                                self.update_tree()

                                # Update the year menu
                                self.year_menu.clear()
                                available_years = [key for key in new_goal_data.keys() if key != 'general_info']
                                available_years.reverse()
                                years_list = map(str, available_years)
                                self.year_menu.addItems(years_list)

                                return info_dialog(self, _('Backup'), _('Database updated'),
                                                   show_copy_button=False, show=True)

                    # Clear the database
                    elif option == 'clear':
                        ans = question_dialog(self, _('Are you sure?'),
                                              _('All existing data will be permanently deleted. '
                                                'Do you want to proceed?'),
                                              show_copy_button=False, override_icon=QIcon.ic('dialog_warning.png'))
                        if ans:
                            os.remove(database_file)
                            self.update_tree()
                            self.year_menu.clear()
                            return None
                    return None

                except:
                    import traceback
                    return error_dialog(_gui, _('Backup'),
                                        _('Backup/import failed. Click on \'Show details\' for more info.'),
                                        det_msg=traceback.format_exc(), show_copy_button=True, show=True)

            def manage_buttons(self) -> None:
                # Disable relink button when viewing other years
                if 'edit' in unique_name:
                    if self.year_menu.currentText() != str(datetime.datetime.now().astimezone().year):
                        self.relink_button.setEnabled(False)
                        if is_dark_theme():
                            self.relink_button.setIcon(get_icon('add_link_disabled.png'))
                    else:
                        self.relink_button.setEnabled(True)
                        self.relink_button.setIcon(get_icon('add_link.png'))

                # Disable 'Abandon' button for finished books, or already abandoned
                disable = []
                for item in self.tree.selectedItems():
                    if item.parent() and item.text(1):
                        if item.text(2) == 'Abandoned':
                            disable.append(True)
                        if item.text(10):
                            if float(item.text(5)[0:-1]) == 100:
                                disable.append(True)
                        else:
                            if float(item.parent().text(5)[0:-1]) == 100:
                                disable.append(True)

                if True in disable:
                    self.abandon_button.setEnabled(False)
                    if is_dark_theme():
                        self.abandon_button.setIcon(get_icon('abandon_disabled.png'))
                else:
                    self.abandon_button.setEnabled(True)
                    self.abandon_button.setIcon(get_icon('abandon.png'))

                # Display the 'Show records' button when a single book is selected
                if 'edit' in unique_name:
                    item = self.tree.currentItem()
                    if self.last_book_current_item and item and item.text(10):
                        self.tree.removeItemWidget(self.last_book_current_item, 7)
                        self.last_book_current_item.takeChildren()
                    elif self.last_record_current_item and item and not item.text(0):
                        self.tree.removeItemWidget(self.last_record_current_item, 7)

                    if len(self.tree.selectedItems()) == 1:
                        if item and item.parent() and item.text(10):
                            records_button = QPushButton(get_icon('records.png'), '')
                            records_button.setToolTip(_('Show book\'s records'))
                            records_button.setMaximumHeight(20)
                            records_button.clicked.connect(partial(self.handler, option='show_records'))
                            self.tree.setItemWidget(item, 7, records_button)
                            self.last_book_current_item = item
                        else:
                            # Display the delete record button
                            if item and item.parent() and item.text(8) and self.year_menu.currentText() != _('All'):
                                most_recent_record = [item.parent().text(5), item.parent().text(6)]
                                selected_record = [item.text(5), item.text(6)]
                                if selected_record != most_recent_record:
                                    delete_record_button = QPushButton(get_icon('delete.png'), '')
                                    delete_record_button.setToolTip(_('Delete this record'))
                                    delete_record_button.setMaximumHeight(20)
                                    delete_record_button.clicked.connect(partial(self.handler, option='delete_record'))
                                    self.tree.setItemWidget(item, 7, delete_record_button)
                                    self.last_record_current_item = item

            def handler(self, option: str = None) -> Dialog | None:
                # Get selected items and queue the jobs
                for item in self.tree.selectedItems():
                    if item and item.text(1) and option not in ('apply_changes', 'add_link', 'show_records'):
                        # Queue jobs for selected books
                        if option not in ('delete_record', 'mark_books'):
                            if option not in ('remove_other_years', 'abandon_other_years'):
                                self.queue_dict[option].append(item.text(1))
                            elif option == 'abandon_other_years':
                                self.abandon_ids.append(item.text(1))
                            else:
                                self.remove_ids.append(item.text(1))
                        else:
                            if option == 'mark_books':
                                pass
                            else:  # 'delete_record'
                                if item.text(1) in self.queue_dict[option]:
                                    self.queue_dict[option][item.text(1)].append(str(item.text(8)))
                                else:
                                    self.queue_dict[option][item.text(1)] = [str(item.text(8))]

                # Check if selection is empty
                if option not in ('apply_changes', 'show_records'):
                    if len(self.tree.selectedItems()) == 0:
                        return error_dialog(_gui, _('Empty selection'), _('No books selected'),
                                            show_copy_button=False, show=True)

                # Enable apply button
                if self.queue_dict != {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}\
                        or self.remove_ids != [] or self.abandon_ids != []:
                    self.ac_button_box.button(QDialogButtonBox.StandardButton.Apply).setEnabled(True)

                if option == 'apply_changes':
                    self.apply_changes()

                elif option == 'add_link':
                    self.add_link()

                elif option in ('remove_from_goal', 'remove_other_years'):
                    self.remove_book()

                elif option in ('abandon', 'abandon_other_years'):
                    self.abandon_book()

                elif option == 'show_records':
                    self.show_records()

                elif option == 'delete_record':
                    self.delete_record()

                elif option == 'mark_books':
                    self.mark_books()

                ReadingGoalTools.update_items_count(self.tree)

                return None

            def apply_changes(self) -> None:
                if 'edit' in unique_name:
                    year = self.year_menu.currentText()
                else:
                    year = str(datetime.datetime.now().astimezone().year)
                if len(self.remove_ids) > 0:
                    ReadingGoalTools.remove_from_goal(tool, self.remove_ids, self.data, year=year, mode='editing')
                    changed_list = self.remove_ids
                elif len(self.abandon_ids) > 0:
                    for book_id in self.abandon_ids:
                        self.data[book_id]['shelf'] = 'Abandoned'
                        goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                        goal_data[year] = self.data
                    changed_list = self.abandon_ids
                else:
                    changed_list = ReadingGoalTools.edit_database(tool, self.queue_dict, db, self.data, year)

                if 'edit' in unique_name:
                    self.queue_dict = {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}
                    self.remove_ids = []
                    self.abandon_ids = []
                    self.update_tree()

                elif 'missing' in unique_name:
                    # Remove resolved items from the tree
                    remove_items_ids = []
                    for option, ids in self.queue_dict.items():
                        if option != 'remove_from_goal':  # Those are already removed from tree
                            for book_id in ids:
                                remove_items_ids.append(book_id)
                    for book_id in remove_items_ids:
                        item = self.tree.findItems(book_id, Qt.MatchFixedString | Qt.MatchRecursive, 1)[0]
                        item.parent().removeChild(item)
                        ReadingGoalTools.update_items_count(self.tree)

                    self.queue_dict = {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}

                self.ac_button_box.button(QDialogButtonBox.StandardButton.Apply).setEnabled(False)
                # Reset new tree attributes
                self.last_book_current_item = None
                self.last_record_current_item = None
                # Redefine our listener, since this is a new tree
                self.tree.itemSelectionChanged.connect(self.manage_buttons)

                if len(changed_list) > 0:
                    ReadingGoalTools.final_message(tool, changed_list, option='edit_goal')

            def add_link(self) -> Dialog | None:
                item = self.tree.currentItem()
                if len(self.tree.selectedItems()) == 1 and item and item.parent():
                    if not item.text(0):
                        item = item.parent()
                    book_title = item.text(10)
                    book_authors = item.text(3).split(' & ')
                    mi = MetaInformation(book_title, book_authors)
                    similar_books = map(str, db.find_identical_books(mi))

                    # Open dialog to select the new linked book
                    selected_book_id = ReadingGoalTools.link_book(tool, db, similar_books)

                    # Queue job for selected book
                    if selected_book_id:
                        self.queue_dict['add_link'].update({item.text(1): selected_book_id})
                        item.setText(0, item.text(0) + ' ' + _('[Linked]'))
                        for i in range(0, 8):
                            item.setForeground(i, QtGui.QBrush(QtGui.QColor(255, 0, 0)))
                        self.ac_button_box.button(QDialogButtonBox.StandardButton.Apply).setEnabled(True)
                        self.tree.clearSelection()

                    return None

                elif len(self.tree.selectedItems()) > 1:
                    return error_dialog(_gui, _('Multiple books'), _('You can only link one book at a time'),
                                        show_copy_button=False, show=True)
                return None

            def remove_book(self) -> None:
                for item in self.tree.selectedItems():
                    if item and item.parent():
                        if item.text(10):  # Has a title - Book entry
                            l_remove = self.tree.findItems(item.text(1), Qt.MatchFixedString | Qt.MatchRecursive, 1)
                            for i in l_remove:
                                i.parent().removeChild(i)
                self.tree.clearSelection()

            def abandon_book(self) -> None:
                for item in self.tree.selectedItems():
                    if item and item.parent():
                        if self.prefs['rereading_shelf']:
                            abandon_shelf = self.tree.findItems('5. ' + _('Abandoned'),
                                                                Qt.MatchStartsWith | Qt.MatchRecursive, 0)
                        else:
                            abandon_shelf = self.tree.findItems('4. ' + _('Abandoned'),
                                                                Qt.MatchStartsWith | Qt.MatchRecursive, 0)
                        if abandon_shelf:
                            if item.text(10):  # Have a title - Book entry
                                old_parent = item.parent()
                            elif item.text(1):  # No title, but has a book id - Book record
                                item = item.parent()
                                old_parent = item.parent()
                            idx = old_parent.indexOfChild(item)
                            item_without_parent = old_parent.takeChild(idx)
                            item_without_parent.takeChildren()
                            if self.prefs['rereading_shelf']:
                                self.tree.topLevelItem(4).addChild(item_without_parent)
                            else:
                                self.tree.topLevelItem(3).addChild(item_without_parent)
                        else:
                            item_without_parent = item
                        l_remove = self.tree.findItems(item_without_parent.text(1),
                                                       Qt.MatchFixedString | Qt.MatchRecursive, 1)
                        for i in l_remove:
                            i.setText(2, 'Abandoned')
                            if self.prefs['shelf_colors']:
                                try:
                                    color_dict = eval(self.prefs['shelf_colors'])
                                except:
                                    error_dialog(self.gui, _('Reading Goal: Invalid python dictionary'),
                                                 _('Verify your shelf colors dictionary. '
                                                   'It must be a valid python dictionary.'),
                                                 show_copy_button=False, show=True)
                            if self.prefs['shelf_colors'] and 5 in color_dict:
                                bg_color = QtGui.QBrush(
                                    QtGui.QColor(color_dict[5][0], color_dict[5][1], color_dict[5][2]))
                            else:
                                bg_color = QtGui.QBrush(QtGui.QColor(120, 120, 120))
                            for idx in range(0, 8):
                                i.setBackground(idx, QtGui.QColor(bg_color))
                                i.setForeground(idx, QtGui.QBrush(QtGui.QColor('white')))
                self.tree.clearSelection()

            def show_records(self) -> None:
                item = self.tree.currentItem()
                if item and item.childCount() > 0:
                    item.takeChildren()
                elif item:
                    book_id = item.text(1)
                    book_records = self.data[book_id]['records']
                    if self.prefs['sorting'] == 'Date (desc)':
                        book_records = dict(sorted(book_records.items(), key=lambda x: int(x[0]), reverse=True))
                    else:
                        book_records = dict(sorted(book_records.items(), key=lambda x: int(x[0])))
                    for record_idx, record in book_records.items():
                        if book_id in self.queue_dict['delete_record']:
                            if str(record_idx) in self.queue_dict['delete_record'][book_id]:
                                continue
                        book_shelf, reading_progress, date, read_pages = (
                            item.text(2), record['status'], record['date'], record['read_pages'])
                        reading_progress = '{0:.0f}%'.format(reading_progress)
                        if date:
                            date = date.astimezone()
                            date_time = QtCore.QDateTime(QtCore.QDate(date.year, date.month, date.day),
                                                         QtCore.QTime(date.hour, date.minute, date.second))
                            date = QtCore.QDate(date.year, date.month, date.day)
                        child = QTreeWidgetItem(item, ['', book_id, book_shelf, '', '', '', '', '', str(record_idx)])
                        child.setData(5, Qt.DisplayRole, reading_progress)
                        child.setData(6, Qt.DisplayRole, date)
                        if date:
                            child.setToolTip(6, QtCore.QLocale.system().toString(date_time))
                        child.setData(7, Qt.DisplayRole, read_pages)
                        if float(QtCore.PYQT_VERSION_STR[0:3]) >= 6.4:
                            for col in range(4, 8):
                                child.setTextAlignment(col, item.textAlignment(col))  # PyQt 6.4+ only

                        # Put % and pages increments in a tooltip
                        day_read_pages = ''
                        day_status = ''
                        if self.data[book_id]['records'][record_idx]['date']:
                            if len(self.data[book_id]['records']) > 1:
                                records_keys = self.data[book_id]['records'].keys()
                                records_keys = sorted(map(int, records_keys))
                                current_record_index = records_keys.index(int(record_idx))
                                if current_record_index > 0:
                                    try:
                                        past_record_index = current_record_index - 1
                                        past_record_year = self.data[book_id]['records'][
                                            str(records_keys[past_record_index])]['date'].astimezone().year
                                        past_record_pages = self.data[book_id]['records'][
                                            str(records_keys[past_record_index])]['read_pages']
                                        past_record_status = self.data[book_id]['records'][
                                            str(records_keys[past_record_index])]['status']
                                        current_record_year = self.data[book_id]['records'][
                                            str(records_keys[current_record_index])]['date'].astimezone().year
                                        current_record_pages = self.data[book_id]['records'][
                                            str(records_keys[current_record_index])]['read_pages']
                                        current_record_status = self.data[book_id]['records'][
                                            str(records_keys[current_record_index])]['status']
                                        if past_record_year == current_record_year:
                                            day_read_pages = current_record_pages - past_record_pages
                                            day_status = round(current_record_status - past_record_status)
                                        elif day_status:
                                            day_read_pages = self.data[book_id]['records'][record_idx]['read_pages']
                                            day_status = round(self.data[book_id]['records'][record_idx]['status'])
                                        else:
                                            day_read_pages = ''
                                            day_status = ''
                                    except AttributeError:
                                        day_read_pages = ''
                                        day_status = ''
                                else:
                                    day_read_pages = self.data[book_id]['records'][record_idx]['read_pages']
                                    day_status = round(self.data[book_id]['records'][record_idx]['status'])
                            else:
                                day_read_pages = self.data[book_id]['records'][record_idx]['read_pages']
                                day_status = round(self.data[book_id]['records'][record_idx]['status'])
                        if day_read_pages:
                            child.setToolTip(7, '+ ' + str(day_read_pages) + _(' pages'))
                        if day_status:
                            child.setToolTip(5, '+ ' + str(day_status) + '%')

                        item.addChild(child)
                    item.setExpanded(True)

            def delete_record(self) -> None:
                item = self.tree.currentItem()
                if item:
                    idx = item.parent().indexOfChild(item)
                    item.parent().takeChild(idx)

            def mark_books(self) -> None:
                marked_ids = []
                # First level
                for item in self.tree.selectedItems():
                    # Level 1 book
                    if item.text(1):
                        marked_ids.append(int(item.text(1)))
                    # Second level
                    else:
                        for idx in range(0, item.childCount()):
                            item.child(idx).setSelected(True)
                            # Level 2 book
                            if item.child(idx).text(1):
                                marked_ids.append(int(item.child(idx).text(1)))
                            # Third level
                            else:
                                for idx2 in range(0, item.child(idx).childCount()):
                                    item.child(idx).child(idx2).setSelected(True)
                                    # Level 3 book
                                    if item.child(idx).child(idx2).text(1):
                                        marked_ids.append(int(item.child(idx).child(idx2).text(1)))
                                    # Level 4 book
                                    else:
                                        for idx3 in range(0, item.child(idx).child(idx2).childCount()):
                                            item.child(idx).child(idx2).child(idx3).setSelected(True)
                                            marked_ids.append(int(item.child(idx).child(idx2).child(idx3).text(1)))

                ReadingGoalTools.mark_rows(tool, marked_ids, marked_text=self.year_menu.currentText())

            def reject(self) -> None:
                if (self.queue_dict != {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}
                        or self.remove_ids != [] or self.abandon_ids != []):
                    ans = question_dialog(self, _('Unapplied changes'),
                                          _('You have unapplied changes. Are you sure you want to discard them?'),
                                          show_copy_button=False)
                    if ans:
                        Dialog.reject(self)
                else:
                    Dialog.reject(self)

            def closeEvent(self, event: QEvent) -> None:
                if (self.queue_dict != {'add_link': {}, 'remove_from_goal': [], 'abandon': [], 'delete_record': {}}
                        or self.remove_ids != [] or self.abandon_ids != []):
                    ans = question_dialog(self, _('Unapplied changes'),
                                          _('You have unapplied changes. Are you sure you want to discard them?'),
                                          show_copy_button=False)
                    if ans:
                        event.accept()
                    else:
                        event.ignore()
                else:
                    event.accept()

            def keyPressEvent(self, event: QEvent) -> None:
                if event.key() == QtCore.Qt.Key_Delete:
                    if self.remove_button.isVisible():
                        self.handler('remove_from_goal')
                else:
                    super().keyPressEvent(event)

            def sizeHint(self) -> QSize:
                return QSize(880, 500)

        end = timer()
        time_diff = end - self.start
        print('Edit goal time: ', datetime.timedelta(seconds=time_diff))

        d = EditGoalDialog()
        d.exec_()

    def statistics_tree(self, new_data: dict[str, dict] = None, year: str | int = None) -> tuple[QTreeWidget, list]:
        locale.setlocale(locale.LC_ALL, '')

        _year = year if year is not None else self.year

        # List of args used to feed the statistics graphic
        args = []

        if _year != _('All'):
            # Number of read books / total books
            read_books_count = new_data['summary']['read_books_count']
            goal_books_count = new_data['summary']['goal_books_count']
            args.append([read_books_count, goal_books_count])  # args[0]
            args[0][0] = locale.format_string('%d', args[0][0], grouping=True)
            args[0][1] = locale.format_string('%d', args[0][1], grouping=True)

            # Percentage of Reading Goal
            goal_percentage = new_data['summary']['goal_percentage']
            args.append(goal_percentage)  # args[1]

            # Read pages
            read_page_count = new_data['summary']['read_page_count']
            args.append(read_page_count)  # args[2]
            args[2] = locale.format_string('%d', args[2], grouping=True)

            # Pages yet to read
            to_read_pages = (int(new_data['summary']['goal_page_count']) -
                             int(new_data['summary']['read_page_count']))
            args.append(to_read_pages)  # args[3]
            args[3] = locale.format_string('%d', args[3], grouping=True)

            # Current speed
            days = (datetime.date.today() - datetime.date(int(_year), 1, 1)).days + 1
            args.append(round(read_page_count / days) if days < 365 else round(read_page_count / 365))  # args[4]
            args[4] = locale.format_string('%d', args[4], grouping=True)

            # Calculate the ideal speed
            last_day = datetime.date(int(_year), 12, 31)
            remaining_days = (last_day - datetime.date.today()).days + 1
            ideal_speed = round(int(to_read_pages)/remaining_days) if remaining_days != 0 else -1
            args.append(ideal_speed if int(ideal_speed) > 0 else '—')  # args[5]
            if args[5] != '—':
                args[5] = locale.format_string('%d', args[5], grouping=True)

            # Month info (average)
            if _year != self.year:
                avg_books_by_month = round(read_books_count / 12, 1)
                avg_page_count = round(read_page_count / 12)
            else:
                y = datetime.datetime.now().astimezone()
                avg_books_by_month = round(new_data['summary']['read_books_count'] / y.month, 1)
                avg_page_count = round(new_data['summary']['read_page_count'] / y.month)
            args.append(locale.format_string('%.1f', avg_books_by_month, grouping=True))  # args[6]
            args.append(locale.format_string('%d', avg_page_count, grouping=True))  # args[7]

        else:
            goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.library_id)
            book_count = 0
            page_count = 0
            for year, values in goal_data.items():
                if year != 'general_info':
                    book_count += values['summary']['read_books_count']
                    page_count += values['summary']['read_page_count']
            book_size = round(page_count / book_count) if book_count > 0 else 0
            book_count_avg = book_count / (len(goal_data) - 1)
            page_count_avg = page_count / (len(goal_data) - 1)
            args.append(locale.format_string('%.1f', book_count_avg, grouping=True))  # args[0]
            args.append(locale.format_string('%.d', page_count_avg, grouping=True))  # args[1]
            args.append(locale.format_string('%.d', book_size, grouping=True))  # args[2]

        # Create a widget to display the statistics
        tree = QTreeWidget()
        tree.setFixedHeight(145)
        tree.setStyleSheet(
            """
            QTreeWidget::item {
                padding: 0 0 2px 0;
            }
            """
        )
        layout = QVBoxLayout()
        layout.addWidget(tree)
        tree.setHeaderHidden(True)
        tree.setHeaderLabels(['Col1', 'Col2'])
        tree.header().setDefaultAlignment(Qt.AlignHCenter)

        if _year != _('All'):
            # Read pages statistics
            pages = QTreeWidgetItem(tree)
            pages.setText(0, _('Pages'))
            pages.setExpanded(True)
            pages_stats = QTreeWidgetItem(pages, [_('Read:') + '  ' + str(args[2]),
                                                  '         ' + _('To read:') + '  ' + str(args[3])])

            # Reading speed statistics
            speed = QTreeWidgetItem(tree)
            speed.setText(0, _('Speed'))
            speed.setExpanded(True)
            speed_stats = QTreeWidgetItem(speed, [_('Current:') + '  ' + str(args[4]),
                                                  '         ' + _('Ideal:') + '   ' + str(args[5])])

            # Month average
            month = QTreeWidgetItem(tree)
            month.setText(0, _('Month (avg.)'))
            month.setExpanded(True)
            # _('Read:') / _('Read:  '), so the translator can choose. pt-BR: 'Lidas' / 'Lidos'
            month_stats = QTreeWidgetItem(month, [_('Read:  ') + str(args[6]),
                                                  '         ' + _('Pages:') + '   ' + str(args[7])])

        else:
            # Average year statistics
            average = QTreeWidgetItem(tree)
            average.setText(0, _('Average annual values'))
            average.setExpanded(True)
            book_count_item = QTreeWidgetItem(average, [_('Read books:') + '  ' + str(args[0])])
            page_count_item = QTreeWidgetItem(average, [_('Read pages:') + '  ' + str(args[1])])
            book_size_item = QTreeWidgetItem(average, [_('Pages per book:') + '  ' + str(args[2])])

        # Auto adjust column sizes
        tree.resizeColumnToContents(0)
        tree.resizeColumnToContents(1)

        # Avoid selection and focus
        tree.setSelectionMode(QAbstractItemView.NoSelection)
        tree.setFocusPolicy(Qt.NoFocus)

        return tree, args

    def statistics_main(self, data: dict[str, dict]) -> None:
        tree, args = ReadingGoalTools.statistics_tree(self, new_data=data)
        if tree is not None:
            ReadingGoalTools.show_statistics(self, tree, args)
        else:
            return

    def show_statistics(self, tree_widget: QTreeWidget, args: list) -> None:
        tool = self
        goal_data = JSONConfig('plugins/Reading_Goal_Data_' + self.library_id)
        year = datetime.datetime.now().astimezone().year
        data = deepcopy(goal_data[str(year)])

        class ReadingGoalStatisticsDialog(Dialog):

            def __init__(self):
                Dialog.__init__(self, _('Reading goal statistics'), 'plugin-reading-goal-stats-dialog', parent=tool.gui)
                self.setWindowIcon(get_icon('stats.png'))

            def setup_ui(self) -> None:
                layout = QVBoxLayout(self)
                self.group_box1 = QGroupBox()
                self.group_box1_layout = QVBoxLayout()
                self.group_box1.setLayout(self.group_box1_layout)
                layout.addWidget(self.group_box1)

                # Percentage bar
                self.text = QLabel(_('You\'ve already read {0} out of {1} books').format(args[0][0], args[0][1]), self)
                self.text.setAlignment(QtCore.Qt.AlignCenter)
                self.group_box1_layout.addWidget(self.text)
                self.group_box1_layout.addSpacing(10)
                self.bar = QProgressBar()
                self.bar.setStyleSheet(
                    """
                    QProgressBar{
                        height: 2px;
                        border: 1.2px solid grey;
                        border-radius: 3px;
                        text-align: center;
                    }
                    """
                )
                if int(args[1]) > 100:
                    self.bar.setRange(0, int(args[1]))
                    self.bar.setFormat('%v%')
                self.bar.setValue(int(args[1]))
                self.group_box1_layout.addWidget(self.bar)

                # Graphic
                month_html = self.get_month_html(data)

                self.group_box2 = QGroupBox()
                self.group_box2_layout = QVBoxLayout()
                self.group_box2.setLayout(self.group_box2_layout)
                layout.addWidget(self.group_box2)
                self.browser = QWebEngineView()
                self.browser.page().setHtml(month_html)
                self.group_box2_layout.addWidget(self.browser)
                self.group_box2_layout.addSpacing(10)
                self.browser.page().navigationRequested.connect(self.daily_pages)

                # Reading statistics
                self.group_box3 = QGroupBox()
                self.group_box3_layout = QVBoxLayout()
                self.group_box3.setLayout(self.group_box3_layout)
                layout.addWidget(self.group_box3)
                self.group_box3_layout.addWidget(tree_widget)
                layout.addSpacing(10)

                # Views
                self.h_layout = QHBoxLayout()
                layout.addLayout(self.h_layout)
                self.change_view_menu = QComboBox()
                self.change_view_menu.addItem(get_icon('month_view.png'), _('By month'))
                self.change_view_menu.addItem(get_icon('genre_view.png'), _('By genre'))
                self.h_layout.addWidget(self.change_view_menu, 0, Qt.AlignRight)
                self.change_view_menu.currentTextChanged.connect(self.change_view)

                # Year combobox
                self.year_menu = QComboBox()
                self.current_year = datetime.datetime.now().astimezone().year
                available_years = [key for key in goal_data.keys() if key not in ('general_info', _('All'))]
                # Don't show future years, since they have no statistic
                available_years = [year for year in available_years
                                   if int(year) <= datetime.datetime.now().astimezone().year]
                available_years.reverse()
                years_list = map(str, available_years)
                self.year_menu.addItems(years_list)
                self.year_menu.addItem(_('All'))
                self.year_menu.setCurrentText(str(self.current_year))
                self.h_layout.addWidget(self.year_menu, 0, Qt.AlignLeft)
                self.year_menu.currentIndexChanged.connect(self.update_year)
                self.year_menu.setFocus()

            def daily_pages(self, request) -> None:
                # Three letter month
                url = request.url().path().strip('/').lower()
                if len(url) == 3:
                    request.reject()
                    selected_year = self.year_menu.currentText()

                    # Create an empty dict for the whole calendar
                    daily_pages_count = {
                        m[1]: {}, m[2]: {}, m[3]: {}, m[4]: {}, m[5]: {}, m[6]: {},
                        m[7]: {}, m[8]: {}, m[9]: {}, m[10]: {}, m[11]: {}, m[12]: {}
                    }
                    daily_pages_count = {k.lower(): v for k, v in daily_pages_count.items()}
                    for _month in range(1, 13):
                        month_length = calendar.monthrange(int(selected_year), _month)[1]
                        for _day in range(1, month_length + 1):
                            daily_pages_count[m[_month].lower()][str(_day)] = 0

                    # Separate every record according to the day of the year
                    for book_id in goal_data[selected_year]:
                        if book_id != 'summary':
                            for record in goal_data[selected_year][book_id]['records']:
                                if goal_data[selected_year][book_id]['records'][record]['date']:
                                    record_year = goal_data[selected_year][book_id]['records'][record][
                                                  'date'].astimezone().year
                                    month = goal_data[selected_year][book_id]['records'][
                                            record]['date'].astimezone().strftime('%b').lower()
                                    day = goal_data[selected_year][book_id]['records'][record]['date'].astimezone().day
                                    if str(record_year) == str(selected_year):
                                        if len(goal_data[selected_year][book_id]['records']) > 1:
                                            records_keys = goal_data[selected_year][book_id]['records'].keys()
                                            records_keys = sorted(map(int, records_keys))
                                            current_record_index = records_keys.index(int(record))
                                            if current_record_index > 0:
                                                past_record_index = current_record_index - 1
                                                past_record_pages = goal_data[selected_year][book_id]['records'][
                                                    str(records_keys[past_record_index])]['read_pages']
                                                current_record_pages = goal_data[selected_year][book_id]['records'][
                                                    str(records_keys[current_record_index])]['read_pages']
                                                day_read_pages = current_record_pages - past_record_pages
                                            else:
                                                day_read_pages = goal_data[selected_year][book_id][
                                                    'records'][record]['read_pages']
                                        else:
                                            day_read_pages = goal_data[selected_year][book_id]['records'][record][
                                                'read_pages']
                                        daily_pages_count[month][str(day)] += day_read_pages

                    # Create a data array to feed Google Charts
                    daily_data = [[_('Days'), _('Pages')]]
                    temp_daily_data = []
                    for day in daily_pages_count[url]:
                        temp_daily_data.append([str(day), daily_pages_count[url][day]])
                    for item in temp_daily_data:
                        daily_data.append(item)
                    if self.change_view_menu.count() == 2:
                        self.change_view_menu.addItem(get_icon('month_view.png'), _('By day'))
                    self.change_view_menu.setCurrentText(_('By day'))

                    # Format the graph
                    h_axis_title = _('Days')
                    v_axis_title = _('Pages')
                    previous_button = _('Previous')
                    next_button = _('Next')
                    main_title = url.upper()
                    background_color = '#121212' if is_dark_theme() else '#ffffff'
                    text_color = '#ddd' if is_dark_theme() else '#000000'
                    if url == m[12].lower():
                        _daily_html = re.sub(buttons_html, dec_buttons_html, daily_html)
                    elif url == m[1].lower():
                        _daily_html = re.sub(buttons_html, jan_buttons_html, daily_html)
                    else:
                        _daily_html = daily_html
                    formated_daily_html = _daily_html.format(daily_data, background_color, text_color, h_axis_title,
                                                             v_axis_title, main_title, previous_button, next_button)
                    self.browser.page().setHtml(formated_daily_html)

                    self.month_name = deepcopy(url).lower()
                    self.months = [x.lower() for x in daily_pages_count.keys()]

                elif url == 'previous':
                    index = self.months.index(self.month_name)
                    if index > 0:
                        new_url = QUrl('http://reading_goal/' + self.months[index - 1])
                        self.browser.load(new_url)
                    else:
                        request.reject()
                elif url == 'next':
                    request.reject()
                    try:
                        index = self.months.index(self.month_name)
                        new_url = QUrl('http://reading_goal/' + self.months[index + 1])
                        self.browser.load(new_url)
                    except IndexError:
                        request.reject()

                self.year_menu.setFocus()

            def update_year(self) -> Dialog | None:
                _year = str(self.year_menu.currentText())
                if self.change_view_menu.currentText() == _('By day'):
                    self.change_view_menu.setCurrentText(_('By month'))
                    self.change_view_menu.removeItem(2)
                if _year != _('All'):
                    # Check if there is data for selected year
                    if _year not in goal_data:
                        self.close()
                        return error_dialog(tool.gui, _('No data'),
                                            _('There is no data for the selected year'), show_copy_button=False,
                                            show=True)

                    # Create a copy of the current year database
                    new_data = deepcopy(goal_data[_year])

                    # Get updated values
                    new_tree_widget, new_args = tool.statistics_tree(new_data=new_data, year=_year)

                    # Update widget with the new values
                    self.text.setText(_('You\'ve already read {0} out of {1} books').
                                      format(new_args[0][0], new_args[0][1]))
                    if int(new_args[1]) > 100:
                        self.bar.setRange(0, int(new_args[1]))
                        self.bar.setFormat('%v%')
                    self.bar.setValue(int(new_args[1]))
                    self.bar.show()

                else:
                    # Get all years tree
                    new_tree_widget, new_args = tool.statistics_tree(year=_year)

                if _year != _('All'):
                    month_html = self.get_month_html(new_data)
                    self.browser.page().setHtml(month_html)
                    self.group_box1.show()
                    self.group_box2.show()
                    self.group_box3.show()
                    self.change_view_menu.setItemText(0, _('By month'))
                else:
                    year_html = self.get_year_html()
                    self.browser.page().setHtml(year_html)
                    self.group_box1.hide()
                    self.group_box2.show()
                    self.group_box3.show()
                    self.change_view_menu.setItemText(0, _('By year'))

                for i in reversed(range(self.group_box3_layout.count())):
                    self.group_box3_layout.itemAt(i).widget().setParent(None)

                self.group_box3_layout.addWidget(new_tree_widget)

                # if _year != _('All'):
                self.change_view()

                return None

            def change_view(self) -> None:
                view = self.change_view_menu.currentText()
                db = tool.gui.current_db.new_api
                y = self.year_menu.currentText()
                _goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)

                if view == _('By genre'):
                    # Check if the genre custom columns is set
                    custom_cols = tool.gui.library_view.model().custom_columns
                    if not prefs['genre_column'] or (prefs['genre_column'] not in custom_cols and
                                                     prefs['genre_column'] != 'tags'):
                        if question_dialog(self, _('Configuration needed'),
                                           _('You must choose the genre custom column.') +
                                           _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                            result = show_configuration(tool, tab=1)
                            if result and prefs['genre_column']:
                                ReadingGoalTools.update_database(tool, db, data)
                                _goal_data = JSONConfig('plugins/Reading_Goal_Data_' + tool.library_id)
                            else:
                                if y != _('All'):
                                    self.change_view_menu.setCurrentText(_('By month'))
                                else:
                                    self.change_view_menu.setCurrentText(_('By year'))
                                return
                        else:
                            if y != _('All'):
                                self.change_view_menu.setCurrentText(_('By month'))
                            else:
                                self.change_view_menu.setCurrentText(_('By year'))
                            return
                if y != _('All'):
                    new_data = deepcopy(_goal_data[y])
                    if view == _('By month'):
                        month_html = self.get_month_html(new_data)
                        self.browser.page().setHtml(month_html)
                        self.group_box1.show()
                        self.group_box2.show()
                        self.group_box3.show()
                        self.change_view_menu.removeItem(2)
                    elif view == _('By day'):
                        pass
                    else:
                        genre_html = self.get_genre_html(new_data)
                        self.browser.page().setHtml(genre_html)
                        self.group_box1.hide()
                        self.group_box2.show()
                        self.group_box3.hide()
                        self.change_view_menu.removeItem(2)
                else:
                    all_data = {}
                    for _year, values in _goal_data.items():
                        if _year != 'general_info':
                            for book_id, book_info in values.items():
                                if book_id != 'summary':
                                    all_data[book_id] = book_info
                    if view == _('By year'):
                        year_html = self.get_year_html()
                        self.browser.page().setHtml(year_html)
                        self.group_box1.hide()
                        self.group_box2.show()
                        self.group_box3.show()
                    else:
                        genre_html = self.get_genre_html(g_data=all_data)
                        self.browser.page().setHtml(genre_html)
                        self.group_box1.hide()
                        self.group_box2.show()
                        self.group_box3.hide()

            @staticmethod
            def get_colors() -> tuple[str, str, str]:
                bar_color = '#3061de' if is_dark_theme() else '#4d77e3'
                text_color = '#ddd' if is_dark_theme() else '#000000'
                background_color = '#121212' if is_dark_theme() else '#ffffff'
                return bar_color, text_color, background_color

            @staticmethod
            def get_month_stats(_data: dict[str, dict]) -> dict[str, list]:
                # Get the biggest number of read books in a month
                max_read_val = max(_data['summary']['read_books_month_count'].values())

                # Create a dict with all the months statistics
                month_stats = {}
                for month in _data['summary']['read_books_month_count']:
                    read_books = _data['summary']['read_books_month_count'][month]
                    percent = read_books / max_read_val * 100 if max_read_val else 0
                    month_stats[month] = [read_books, percent]

                return month_stats

            def get_month_html(self, _data: dict[str, dict]) -> str:
                # Bar model
                month_bar = v_bar

                # Get a dict of month stats (unordered): 'apr': [read_books, percent], 'aug': [...]
                month_stats = self.get_month_stats(_data)

                # Get a bar for each month
                bars = []
                for i in range(1, 13):
                    bar = month_bar.format(month_stats[m[i]][0], month_stats[m[i]][1],
                                           '<a href="http://reading_goal/' + m[i].lower() + '">' + m[i] + '</a>')
                    bars.append(bar)
                bars_string = ''.join(str(b) for b in bars)

                # Insert the bars in the html file
                month_html = v_html.replace('<!-- Placeholder -->', bars_string)

                # Format html
                bar_color, text_color, background_color = self.get_colors()
                month_html = month_html.format(text_color, bar_color, background_color, _('MONTHLY READINGS'))
                return month_html

            @staticmethod
            def get_year_stats() -> dict[str, dict]:
                year_stats = {}
                for _year, values in goal_data.items():
                    if _year != 'general_info':
                        if len(values) > 1:
                            year_stats[_year] = {
                                'book_count': values['summary']['read_books_count'],
                                'pages_count': values['summary']['read_page_count'],
                                'book_size': round(values['summary']['read_page_count'] /
                                                   values['summary']['read_books_count']) if values['summary'][
                                    'read_books_count'] else 0
                            }
                        else:
                            year_stats[_year] = {
                                'book_count': 0,
                                'pages_count': 0,
                                'book_size': 0
                            }

                return year_stats

            def get_year_html(self) -> str:
                # Bar model
                year_bar = v_bar

                # Dict with all the years
                year_stats = self.get_year_stats()

                # Get a bar for each year
                bars = []
                book_count_list = []
                for year in year_stats:
                    book_count_list.append(int(year_stats[year]['book_count']))
                max_book_count = max(book_count_list)
                for year in year_stats:
                    bar = year_bar.format(year_stats[year]['book_count'],
                                          (year_stats[year]['book_count'] /
                                           max_book_count * 100) if max_book_count > 0 else 0, year)
                    bars.append(bar)
                bars_string = ''.join(str(b) for b in bars)

                # Insert the bars in the html file
                year_html = v_html.replace('<!-- Placeholder -->', bars_string)

                # Format html
                bar_color, text_color, background_color = self.get_colors()
                year_html = year_html.format(text_color, bar_color, background_color, _('ANNUAL READINGS'), '300px')

                return year_html

            def get_genre_stats(self, g_data: dict[str, dict] = None) -> tuple[dict, int]:
                _year = str(self.year_menu.currentText())
                genre_set = set()
                genre_dict = {}
                book_count = 0
                if g_data:
                    _data = g_data
                else:
                    _data = deepcopy(goal_data[_year])
                # Get a set of all genres
                for book_id, book_info in _data.items():
                    if book_id != 'summary':
                        book_count += 1
                        genre = book_info['genre']
                        if isinstance(genre, list):
                            for g in genre:
                                genre_set.add(g)
                        else:
                            genre_set.add(genre)
                # Get a dict with the book count for every genre
                for genre in genre_set:
                    if genre != 'Undefined':
                        genre_dict[genre] = []
                        for book_id, book_info in _data.items():
                            if book_id != 'summary':
                                if book_info['status'] == 100:
                                    book_genre = book_info['genre']
                                    if genre in book_genre:
                                        genre_dict[genre].append(book_id)
                g_dict = {genre: len(ids) for (genre, ids) in genre_dict.items()}
                g_dict_sorted = dict(sorted(g_dict.items(), key=lambda x: x[1], reverse=True))

                return g_dict_sorted, book_count

            def get_genre_html(self, g_data: dict[str, dict] = None) -> str:
                # Bar model
                genre_bar = h_bar

                # Get a list of tuples: (genre, genre_count)
                genre_dict, book_count = self.get_genre_stats(g_data=g_data)

                # Get top genres
                bars = []
                top_genres_dict = {}
                top_genres_count = 0
                count = 0
                for genre, genre_count in genre_dict.items():
                    top_genres_dict[genre] = genre_count
                    top_genres_count = top_genres_count + genre_count
                    count += 1
                    if count == 12:
                        break
                others_count = book_count - top_genres_count
                # top_genres_dict['others'] = others_count

                max_book_count = max(top_genres_dict.values()) if len(top_genres_dict) > 0 else 0

                # Get a bar for each genre
                for genre, genre_count in top_genres_dict.items():
                    if max_book_count > 0:
                        bar = genre_bar.format(genre, genre_count / max_book_count * 100, genre_count)
                        bars.append(bar)
                bars_string = ''.join(str(b) for b in bars)

                # Insert the bars in the html file
                genre_html = h_html.replace('<!-- Placeholder -->', bars_string)

                # Format html
                bar_color, text_color, background_color = self.get_colors()
                genre_html = genre_html.format(text_color, bar_color, background_color, _('BY GENRE'), '400px')

                return genre_html

            def sizeHint(self) -> QSize:
                return QSize(410, 690)

            def resizeEvent(self, event: QEvent) -> None:
                self.browser.reload()

        end = timer()
        time_diff = end - self.start
        print('Reading goal statistics time: ', datetime.timedelta(seconds=time_diff))

        d = ReadingGoalStatisticsDialog()
        d.exec_()


class InputDialog(Dialog):
    def __init__(self, parent, detected_year):
        self.detected_year = detected_year
        Dialog.__init__(self, _('Reading goal\'s year'), 'plugin-reading-goal-input-dialog', parent=parent)
        self.setWindowIcon(get_icon('goal_add.png'))

    def setup_ui(self) -> None:
        self.layout = QGridLayout(self)
        self.year_spinbox = QSpinBox()
        self.year_spinbox.setRange(1900, 2100)
        self.year_spinbox.setValue(self.detected_year)
        self.layout.addWidget(self.year_spinbox, 0, 0, 1, 2)
        self.button = QPushButton(_('Date'))
        self.layout.addWidget(self.button, 0, 2, 1, 1)
        self.button.clicked.connect(self.change_widget)
        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.layout.addWidget(self.button_box, 1, 0, 1, 3)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)

    def change_widget(self) -> None:
        if self.button.text() == _('Date'):
            self.date_widget = DateEdit(self, create_clear_button=False)
            self.date_widget.setMinimumDate(QtCore.QDate(1970, 1, 1))
            self.date_widget.setMaximumDate(QtCore.QDate(2100, 12, 31))
            self.date_widget.current_val = datetime.datetime.now().astimezone()
            self.year_spinbox.hide()
            self.layout.addWidget(self.date_widget, 0, 0, 1, 2)
            self.button.setText(_('Year'))
        else:
            self.date_widget.hide()
            self.year_spinbox.show()
            self.button.setText(_('Date'))

    def save_state(self) -> None:
        pass

    def accept(self) -> None:
        Dialog.accept(self)

    def reject(self) -> None:
        Dialog.reject(self)

    def sizeHint(self) -> QSize:
        return QSize(250, 100)


class Dialog(Dialog):
    prefs = JSONConfig('plugins/Reading_Goal')

    def save_state(self) -> None:
        try:
            # Get header state
            state_dict = prefs['state']
            state = self.tree.header().saveState()
            state_dict[str(self.metaObject().className())] = state
            prefs['state'] = state_dict
        except (NameError, AttributeError):
            pass

    def reject(self) -> None:
        self.save_geometry(self.prefs_for_persistence, self.name + '-geometry')
        self.save_state()
        QtWidgets.QDialog.reject(self)

    def accept(self) -> None:
        self.save_geometry(self.prefs_for_persistence, self.name + '-geometry')
        self.save_state()
        QtWidgets.QDialog.accept(self)

    def closeEvent(self, event: QEvent) -> None:
        self.save_geometry(self.prefs_for_persistence, self.name + '-geometry')
        self.save_state()
        event.accept()


class QTreeWidgetItem(QtWidgets.QTreeWidgetItem):
    def __lt__(self, other: QTreeWidgetItem) -> bool | super:
        column = self.treeWidget().sortColumn()
        if column == 4:  # Fix for Series sorting (account for series_index)
            first = self.text(4)
            first_series_name = re.sub(' \\[\\d+\\.*\\d*]', '', first)
            first_series_index = re.search('\\[\\d+\\.*\\d*]', first)
            second = other.text(4)
            second_series_name = re.sub(' \\[\\d+\\.*\\d*]', '', second)
            second_series_index = re.search('\\[\\d+\\.*\\d*]', second)
            if first_series_name == second_series_name and first_series_index and second_series_index:
                return float(first_series_index[0].strip('[]')) < float(second_series_index[0].strip('[]'))
        if column == 5:  # Fix for Progress sorting (excludes the % symbol before sorting)
            first = self.text(5)[:-1] if self.text(5)[:-1] != '' else 0
            second = other.text(5)[:-1] if other.text(5)[:-1] != '' else 0
            return float(first) < float(second)
        elif column == 6:  # Fix for Date sorting (uses the datetime)
            first = self.data(11, Qt.DisplayRole) if self.data(11, Qt.DisplayRole) is not None else QtCore.QDateTime(
                1970, 1, 1, 1, 1, 1)
            second = other.data(11, Qt.DisplayRole) if other.data(11, Qt.DisplayRole) is not None else QtCore.QDateTime(
                1970, 1, 1, 1, 1, 1)
            return first < second
        elif self.data(12, Qt.DisplayRole) == 'Genre view' and column == 0:
            # List books inside the genre first, then list subgenres
            sort_order = self.treeWidget().header().sortIndicatorOrder()
            if not self.data(1, Qt.DisplayRole) and other.data(1, Qt.DisplayRole):
                result = sort_order != Qt.AscendingOrder
            elif self.data(1, Qt.DisplayRole) and not other.data(1, Qt.DisplayRole):
                result = sort_order == Qt.AscendingOrder
            else:
                return super().__lt__(other)
            return result
        else:
            return super().__lt__(other)
