#!/usr/bin/env python
__license__   = 'GPL v3'
__copyright__ = '2025, Comfy.n'
__docformat__ = 'restructuredtext en'

from qt.core import QDialog, QTreeView, QPushButton, QModelIndex, Qt
from calibre.gui2.actions import InterfaceAction
from calibre.gui2 import Application
from . import common_icons

class KoboTouchConfigButtonAction(InterfaceAction):
    name = 'KoboTouch Config'
    # The action_spec creates the keyboard shortcut automatically:
    # (text, icon, tooltip, keyboard shortcut)
    action_spec = ('KoboTouch Config', None, 'Quick access to KoboTouch device driver configuration', 'Alt+K')
    
    action_menu_clone_qaction = 'KoboTouch Config'
    dont_add_to = frozenset([
        'context-menu', 'context-menu-device', 
        'menubar-device', 'context-menu-cover-browser', 
        'context-menu-split'
    ])
    dont_remove_from = InterfaceAction.all_locations - dont_add_to
    action_type = 'current'

    def genesis(self):
        # Set up the action
        self.qaction.triggered.connect(self.show_kobo_config)

        # Set up the icon
        common_icons.set_plugin_icon_resources(self.name, {})
        self.rebuild_icon()

        # Set up theme change listener
        app = Application.instance()
        if app:
            app.palette_changed.connect(self.rebuild_icon)

    def rebuild_icon(self):
        icon = common_icons.get_icon('images/iconplugin.png')
        self.qaction.setIcon(icon)

    def show_kobo_config(self):
        # Open the KoboTouch device config dialog directly
        from calibre.customize.ui import is_disabled
        from calibre.devices.kobo.driver import KOBOTOUCH
        try:
            from calibre_plugins.kobotouch_extended.device.driver import KOBOTOUCHEXTENDED
            driver_to_configure = KOBOTOUCHEXTENDED if not is_disabled(KOBOTOUCHEXTENDED) else KOBOTOUCH
        except Exception as e:
            driver_to_configure = KOBOTOUCH

        driver_to_configure = driver_to_configure(None)
        driver_to_configure.do_user_config(self.gui)
