#!/usr/bin/env python

from calibre import prints

try:
	prints("Comfy Reminders::ui.py - loading translations")
	load_translations()
except NameError:
	prints("Comfy Reminders::ui.py - exception when loading translations")
	pass

from qt.core import (
	Qt,
	QDialog,
	QVBoxLayout,
	QHBoxLayout,
	QPushButton,
	QTableWidget,
	QTableWidgetItem,
	QLabel,
	QLineEdit,
	QPlainTextEdit,
	QDateTimeEdit,
	QDateTime,
	QCalendarWidget,
	QWidget,
	QTabWidget,
	QComboBox,
	QGroupBox,
	QTimeEdit,
	QFormLayout,
	QSize,
	QColor,
	QMenu,
	QEvent)
from calibre.gui2.widgets2 import Dialog
from calibre_plugins.comfy_reminders import plugin_prefs, ComfyRemindersPlugin
from calibre.utils.date import now, is_date_undefined, safeyear
from calibre.utils.iso8601 import utc_tz, local_tz, UNDEFINED_DATE
from calibre.gui2 import gprefs
from . import common_icons
from calibre.utils.config import config_dir
import datetime
import uuid
import traceback
import sys
import os
import json
from .config import ConfigDialog

# Default format is 24h European style
DEFAULT_TIMESTAMP_FORMAT = 'dd/MM/yyyy HH:mm'
AVAILABLE_TIMESTAMP_FORMATS = {
	_('24h European (31/12/2024 23:59)'): 'dd/MM/yyyy HH:mm',
	_('12h American (12/31/2024 11:59 PM)'): 'MM/dd/yyyy hh:mm AP',
	_('ISO (2024-12-31 23:59)'): 'yyyy-MM-dd HH:mm'
}

# Define reminder status constants
REMINDER_STATUS = {
	'ACTIVE': 'active',         # Reminder is active and set for the future
	'TRIGGERED': 'triggered',   # Reminder has triggered (time has passed)
	'ACKNOWLEDGED': 'acknowledged',  # Reminder was seen but task not completed
	'CLEARED': 'cleared',       # Reminder was manually cleared by user
	'NEVER_SET': 'never_set'    # Reminder was never set
}


def qt_to_dt(qdate_or_qdatetime, as_utc=True):
	"""Convert a QDateTime to a datetime object. For Calibre 6.x compatibility."""
	o = qdate_or_qdatetime
	if o is None:
		return UNDEFINED_DATE
	if hasattr(o, 'toUTC'):  # QDateTime
		def c(o, tz=utc_tz):
			d, t = o.date(), o.time()
			try:
				return datetime.datetime(
					safeyear(
						d.year()),
					d.month(),
					d.day(),
					t.hour(),
					t.minute(),
					t.second(),
					t.msec() * 1000,
					tz)
			except ValueError:
				return datetime.datetime(
					safeyear(
						d.year()),
					d.month(),
					1,
					t.hour(),
					t.minute(),
					t.second(),
					t.msec() * 1000,
					tz)

		spec = o.timeSpec()
		if spec == Qt.TimeSpec.LocalTime:
			ans = c(o, local_tz)
		elif spec == Qt.TimeSpec.UTC:
			ans = c(o, utc_tz)
		else:
			ans = c(o.toUTC(), utc_tz)
		return ans.astimezone(utc_tz if as_utc else local_tz)

	try:
		dt = datetime.datetime(
			safeyear(
				o.year()),
			o.month(),
			o.day()).replace(
			tzinfo=local_tz)
	except ValueError:
		dt = datetime.datetime(
			safeyear(
				o.year()),
			o.month(),
			1).replace(
			tzinfo=local_tz)
	return dt.astimezone(utc_tz if as_utc else local_tz)


def qt_from_dt(d, assume_utc=False):
	"""Convert a datetime object to QDateTime. For Calibre 6.x compatibility."""
	if is_date_undefined(d):
		from calibre.gui2 import UNDEFINED_QDATETIME
		return UNDEFINED_QDATETIME
	if d.tzinfo is None:
		d = d.replace(tzinfo=utc_tz if assume_utc else local_tz)
	d = d.astimezone(local_tz)
	# not setting a time zone means this QDateTime has timeSpec() ==
	# LocalTime which is what we want for display/editing.
	ans = QDateTime(d.year, d.month, d.day, d.hour, d.minute,
					d.second, int(d.microsecond / 1000))
	return ans


class NotesDialog(Dialog):
	def __init__(self, parent=None, notes=''):
		self.notes = notes  # Store notes before parent init
		Dialog.__init__(self, _('Notes'), 'notes-dialog', parent)
		icon = common_icons.get_icon('images/iconplugin')
		if icon and not icon.isNull():
			self.setWindowIcon(icon)

	def setup_ui(self):
		layout = QVBoxLayout(self)
		self.notes_edit = QPlainTextEdit(self)
		self.notes_edit.setPlainText(self.notes)
		layout.addWidget(self.notes_edit)

		# Add OK/Cancel buttons
		btn_layout = QHBoxLayout()
		save_btn = QPushButton(_('Save'), self)
		save_btn.clicked.connect(self.accept)
		cancel_btn = QPushButton(_('Cancel'), self)
		cancel_btn.clicked.connect(self.reject)
		btn_layout.addWidget(cancel_btn)
		btn_layout.addWidget(save_btn)
		layout.addLayout(btn_layout)

	def get_notes(self):
		return self.notes_edit.toPlainText()

	def sizeHint(self):
		return QSize(400, 300)


class CategoryDialog(Dialog):
	def __init__(self, parent=None, categories=None):
		self.categories = categories or []  # Store categories before parent init
		Dialog.__init__(self, _('Edit Categories'), 'category-dialog', parent)
		icon = common_icons.get_icon('images/iconplugin')
		if icon and not icon.isNull():
			self.setWindowIcon(icon)

	def setup_ui(self):
		layout = QVBoxLayout(self)

		# Categories list
		self.categories_list = QTableWidget(self)
		self.categories_list.setColumnCount(2)
		self.categories_list.setHorizontalHeaderLabels(
			[_('Category'), _('Actions')])
		self.categories_list.horizontalHeader().setStretchLastSection(True)
		self.categories_list.setColumnWidth(0, 180)
		layout.addWidget(self.categories_list)

		# Load categories
		self.load_categories(self.categories)

		# Add category section
		add_layout = QHBoxLayout()
		self.category_input = QLineEdit(self)
		self.category_input.setPlaceholderText(_('New category name'))
		self.category_input.returnPressed.connect(self.add_category)
		add_layout.addWidget(self.category_input)

		add_btn = QPushButton(_('Add'), self)
		add_btn.clicked.connect(self.add_category)
		add_layout.addWidget(add_btn)
		layout.addLayout(add_layout)

		# Save/Cancel buttons
		btn_layout = QHBoxLayout()
		save_btn = QPushButton(_('Save'), self)
		save_btn.clicked.connect(self.accept)
		cancel_btn = QPushButton(_('Cancel'), self)
		cancel_btn.clicked.connect(self.reject)
		btn_layout.addWidget(cancel_btn)
		btn_layout.addWidget(save_btn)
		layout.addLayout(btn_layout)

	def load_categories(self, categories):
		self.categories_list.setRowCount(len(categories))
		for row, category in enumerate(categories):
			# Category name
			self.categories_list.setItem(row, 0, QTableWidgetItem(category))

			# Delete button
			delete_btn = QPushButton(_('Delete'), self)
			delete_btn.clicked.connect(
				lambda checked, r=row: self.delete_category(r))
			self.categories_list.setCellWidget(row, 1, delete_btn)

	def add_category(self):
		category = self.category_input.text().strip()
		if category:
			# Check if category already exists
			for row in range(self.categories_list.rowCount()):
				if self.categories_list.item(row, 0).text() == category:
					return

			# Add new category
			current_row = self.categories_list.rowCount()
			self.categories_list.setRowCount(current_row + 1)
			self.categories_list.setItem(
				current_row, 0, QTableWidgetItem(category))

			delete_btn = QPushButton(_('Delete'), self)
			delete_btn.clicked.connect(
				lambda checked, r=current_row: self.delete_category(r))
			self.categories_list.setCellWidget(current_row, 1, delete_btn)

			self.category_input.clear()

	def delete_category(self, row):
		self.categories_list.removeRow(row)

	def get_categories(self):
		categories = []
		for row in range(self.categories_list.rowCount()):
			categories.append(self.categories_list.item(row, 0).text())
		return categories

	def sizeHint(self):
		return QSize(600, 400)  # Wider and taller initial size

	def accept(self):
		# Get current categories from the list
		self.categories = self.get_categories()
		# Call parent accept() to close dialog
		Dialog.accept(self)


class BackupDialog(QDialog):
	"""Dialog for managing and selecting backups to restore"""

	def __init__(self, parent=None):
		QDialog.__init__(self, parent)
		self.setWindowTitle(_('Manage Backups'))
		self.resize(600, 400)

		layout = QVBoxLayout(self)

		# Instructions label
		info_label = QLabel(
			_('Select a backup to restore. The most recent backups are kept.'))
		info_label.setWordWrap(True)
		layout.addWidget(info_label)

		# Backup list
		self.backup_list = QTableWidget()
		self.backup_list.setColumnCount(3)
		self.backup_list.setHorizontalHeaderLabels(
			[_('Date'), _('Time'), _('Filename')])
		self.backup_list.setSelectionBehavior(
			QTableWidget.SelectionBehavior.SelectRows)
		self.backup_list.setSelectionMode(
			QTableWidget.SelectionMode.SingleSelection)
		self.backup_list.horizontalHeader().setStretchLastSection(True)
		layout.addWidget(self.backup_list)

		# Load backups
		self.load_backups()

		# Buttons
		button_layout = QHBoxLayout()
		restore_btn = QPushButton(_('Restore Selected'))
		restore_btn.clicked.connect(self.accept)
		cancel_btn = QPushButton(_('Cancel'))
		cancel_btn.clicked.connect(self.reject)
		button_layout.addWidget(cancel_btn)
		button_layout.addWidget(restore_btn)
		layout.addLayout(button_layout)

		# Double click to restore
		self.backup_list.itemDoubleClicked.connect(self.accept)

		self.selected_backup = None

	def load_backups(self):
		"""Load available backups into the list"""
		backup_dir = os.path.join(
			config_dir,
			'plugins',
			'Comfy Reminders',
			'backups')
		if not os.path.exists(backup_dir):
			return

		backup_files = sorted([f for f in os.listdir(
			backup_dir) if f.startswith('reminders_backup_')])
		self.backup_list.setRowCount(len(backup_files))

		for row, filename in enumerate(backup_files):
			try:
				# Parse timestamp from filename
				# (reminders_backup_YYYYMMDD_HHMMSS.json)
				timestamp_str = filename.replace(
					'reminders_backup_', '').replace(
					'.json', '')
				timestamp = datetime.datetime.strptime(
					timestamp_str, '%Y%m%d_%H%M%S')

				# Display date and time separately
				date_item = QTableWidgetItem(timestamp.strftime('%Y-%m-%d'))
				time_item = QTableWidgetItem(timestamp.strftime('%H:%M:%S'))
				file_item = QTableWidgetItem(filename)

				self.backup_list.setItem(row, 0, date_item)
				self.backup_list.setItem(row, 1, time_item)
				self.backup_list.setItem(row, 2, file_item)
			except BaseException:
				# If filename parsing fails, just show the raw filename
				self.backup_list.setItem(
					row, 0, QTableWidgetItem(
						_('Unknown')))
				self.backup_list.setItem(
					row, 1, QTableWidgetItem(
						_('Unknown')))
				self.backup_list.setItem(row, 2, QTableWidgetItem(filename))

		# Auto-adjust columns
		self.backup_list.resizeColumnsToContents()

		# Select most recent by default
		if self.backup_list.rowCount() > 0:
			self.backup_list.selectRow(self.backup_list.rowCount() - 1)

	def get_selected_backup(self):
		"""Get the selected backup filename"""
		selected = self.backup_list.selectedItems()
		if selected:
			return selected[2].text()  # Get filename from third column
		return None


class TodoDialog(Dialog):
	def __init__(self, gui, parent=None):
		# Store gui reference
		self.gui = gui

		# Get timestamp format before parent init
		self.timestamp_format = plugin_prefs.get(
			'timestamp_format', DEFAULT_TIMESTAMP_FORMAT)

		# Initialize state BEFORE base Dialog init
		self.initialize_state()

		# Initialize base Dialog - this will call setup_ui for us
		Dialog.__init__(
			self,
			_('Reminders and Notebook'),
			'reminders-dialog',
			parent)

		# Set window icon
		icon = common_icons.get_icon('images/iconplugin')
		if icon and not icon.isNull():
			self.setWindowIcon(icon)

		# Fix any inconsistencies in reminder states before loading tables
		self.fix_reminder_status()

		# Migrate log entries to new structure
		self.migrate_log_entries()

		# Create a backup on startup
		self.create_backup()

		# Initialize tables
		self.load_todos()
		self.load_logs()

		# Restore window geometry and column states
		self.restore_state()

		# Save state when closing
		self.finished.connect(self.save_state)

		# Connect tab change handler
		self.tabs.currentChanged.connect(self.on_tab_changed)

		# Connect todo table cell change handler
		self.todo_table.cellChanged.connect(self.on_todo_cell_changed)

		# Connect todo table key press event for DEL key handling
		self.todo_table.keyPressEvent = self.todo_table_key_press_event

		# Connect table selection changed signal
		self.log_table.itemSelectionChanged.connect(
			self.on_log_selection_changed)

		# Connect sorting handler
		self.todo_table.horizontalHeader().sortIndicatorChanged.connect(self.on_sort_changed)

		# Initialize sorting state
		self.current_sort_column = None
		self.sort_orders = {}

		# Install event filter for title input
		self.title_input.installEventFilter(self)

	def eventFilter(self, obj, event):
		if obj == self.title_input and event.type() == QEvent.Type.KeyPress:
			key = event.key()
			if key in (Qt.Key.Key_Enter, Qt.Key.Key_Return):
				self.add_todo()
				return True
		return super().eventFilter(obj, event)

	def initialize_state(self):
		"""Initialize and validate reminder states before any UI operations"""
		try:
			if 'todos' not in plugin_prefs:
				plugin_prefs['todos'] = []
			if 'notification_times' not in plugin_prefs:
				plugin_prefs['notification_times'] = {}
			if 'notification_history' not in plugin_prefs:
				plugin_prefs['notification_history'] = {}
			if 'reminder_status' not in plugin_prefs:
				plugin_prefs['reminder_status'] = {}

			current_time = now().astimezone(datetime.timezone.utc)
			modified = False

			# Copy all state data to work with
			times = plugin_prefs.get('notification_times', {}).copy()
			history = plugin_prefs.get('notification_history', {}).copy()
			status = plugin_prefs.get('reminder_status', {}).copy()

			# Process each reminder
			for todo_id, time_str in list(times.items()):
				try:
					reminder_time = datetime.datetime.fromisoformat(time_str)
					# Check if reminder is overdue or triggered
					if reminder_time <= current_time:
						# Move to history but keep original time
						history[todo_id] = time_str
						status[todo_id] = REMINDER_STATUS['TRIGGERED']
						del times[todo_id]
						modified = True
				except (ValueError, TypeError) as e:
					print(f"Error processing reminder {todo_id}: {str(e)}")
					# Clean up invalid entries
					if todo_id in times:
						del times[todo_id]
						modified = True

			# Save all changes atomically if modified
			if modified:
				plugin_prefs['notification_times'] = times
				plugin_prefs['notification_history'] = history
				plugin_prefs['reminder_status'] = status

		except Exception as e:
			print(f"Error in initialize_state: {str(e)}")
			print(traceback.format_exc())

	def todo_table_key_press_event(self, event):
		"""Handle key press events in the todo table"""
		# Check if DEL key is pressed
		if event.key() == Qt.Key.Key_Delete:
			self.delete_selected_todos()
		else:
			# Call the original keyPressEvent method for other keys
			QTableWidget.keyPressEvent(self.todo_table, event)

	def accept(self):
		# Override accept to prevent Enter key from closing the dialog
		pass

	def save_state(self):
		"""Save window geometry and column widths"""
		# Save window geometry
		geom = bytearray(self.saveGeometry())
		gprefs['comfy_reminders_dialog_geometry'] = geom

		# Save todo table column widths
		todo_widths = {}
		for col in range(self.todo_table.columnCount()):
			todo_widths[str(col)] = self.todo_table.columnWidth(col)
		gprefs['comfy_reminders_todo_widths'] = todo_widths

		# Save log table column widths
		log_widths = {}
		for col in range(self.log_table.columnCount()):
			log_widths[str(col)] = self.log_table.columnWidth(col)
		gprefs['comfy_reminders_log_widths'] = log_widths

		# Save current tab index
		gprefs['comfy_reminders_last_tab'] = self.tabs.currentIndex()

		# Save sorting preferences - convert Qt enum to int to avoid TypeError
		gprefs['comfy_reminders_sort_column'] = self.current_sort_column
		gprefs['comfy_reminders_sort_orders'] = self.sort_orders

	def restore_state(self):
		"""Restore window geometry and column widths"""
		# Restore window geometry
		geom = gprefs.get('comfy_reminders_dialog_geometry', None)
		if (geom is not None) and (len(geom) > 0):
			self.restoreGeometry(geom)
		else:
			self.resize(950, 600)   # NOT 800

		# Restore todo table column widths
		todo_widths = gprefs.get('comfy_reminders_todo_widths', {})
		for col, width in todo_widths.items():
			self.todo_table.setColumnWidth(int(col), width)

		# Restore log table column widths
		log_widths = gprefs.get('comfy_reminders_log_widths', {})
		for col, width in log_widths.items():
			self.log_table.setColumnWidth(int(col), width)

		# Restore last active tab
		last_tab = gprefs.get(
			'comfy_reminders_last_tab',
			0)  # Default to first tab
		self.tabs.setCurrentIndex(last_tab)

		# Restore sort settings
		self.tabs.setCurrentIndex(last_tab)

		# Restore sort settings
		self.current_sort_column = gprefs.get(
			'comfy_reminders_sort_column', None)
		self.sort_orders = gprefs.get('comfy_reminders_sort_orders', {})

	def setup_ui(self):
		"""Called by Dialog.__init__"""
		# Create main layout using parent layout
		layout = QVBoxLayout()
		self.setLayout(layout)

		# Set minimum dialog width
		self.setMinimumWidth(950)  # Prevent dialog from becoming too narrow

		# Add format selector at the top with version label
		format_layout = QHBoxLayout()
		# Add format selector
		format_layout.addWidget(QLabel(_('Timestamp format:')))
		self.format_combo = QComboBox()
		for label in AVAILABLE_TIMESTAMP_FORMATS:
			self.format_combo.addItem(label)
		# Set current format
		current_format = self.timestamp_format
		for i, format_str in enumerate(AVAILABLE_TIMESTAMP_FORMATS.values()):
			if format_str == current_format:
				self.format_combo.setCurrentIndex(i)
				break
		self.format_combo.currentIndexChanged.connect(self.on_format_changed)
		format_layout.addWidget(self.format_combo)

		# Add stretch to push version label to the right
		format_layout.addStretch()

		# Add version label to the right
		from . import ComfyRemindersPlugin
		version_str = '.'.join(map(str, ComfyRemindersPlugin.version))
		version_label = QLabel(f'v{version_str}')
		version_label.setStyleSheet("""
			QLabel {
			color: #AABBCC;
			font-size: 9pt;
			padding: 4px;
			}
			""")
		format_layout.addWidget(version_label)
		layout.addLayout(format_layout)

		# Add tabs
		self.tabs = QTabWidget(self)
		self.todo_tab = QWidget()
		self.log_tab = QWidget()
		self.tabs.addTab(self.todo_tab, _("Reminders"))
		self.tabs.addTab(self.log_tab, _("Notebook"))
		layout.addWidget(self.tabs)

		# Set up individual tabs
		self.setup_todo_tab()
		self.setup_log_tab()

		# Enable multiple selection in todo table
		self.todo_table.setSelectionMode(
			QTableWidget.SelectionMode.ExtendedSelection)
		self.todo_table.setSelectionBehavior(
			QTableWidget.SelectionBehavior.SelectRows)

		# Configure table to prevent horizontal scrolling
		self.todo_table.horizontalHeader().setStretchLastSection(True)
		self.todo_table.horizontalHeader().setSectionResizeMode(
			0, self.todo_table.horizontalHeader().ResizeMode.Interactive)
		self.todo_table.horizontalHeader().setSectionResizeMode(
			1, self.todo_table.horizontalHeader().ResizeMode.Interactive)
		self.todo_table.horizontalHeader().setSectionResizeMode(
			2, self.todo_table.horizontalHeader().ResizeMode.Interactive)
		self.todo_table.horizontalHeader().setSectionResizeMode(
			3, self.todo_table.horizontalHeader().ResizeMode.Interactive)
		self.todo_table.horizontalHeader().setSectionResizeMode(
			4, self.todo_table.horizontalHeader().ResizeMode.Interactive)

	def setup_reminders_ui(self):
		"""Setup reminders UI elements"""

		# Get column data for both built-in and custom date columns
		date_columns = gprefs.get('date_columns', {})
		db = self.gui.current_db.new_api
		field_metadata = db.field_metadata

		# Check both built-in and custom date columns
		for key, value in field_metadata.items():
			if isinstance(value, dict):
				if value.get('datatype') == 'datetime':
					# Handle both built-in and custom columns
					display_name = value.get('name', key)
					if key.startswith('#'):
						display_name = f"{display_name} (Custom)"
					else:
						display_name = f"{display_name} (Built-in)"
					date_columns[key] = display_name

	def on_format_changed(self, index):
		format_str = AVAILABLE_TIMESTAMP_FORMATS[self.format_combo.currentText(
		)]
		plugin_prefs['timestamp_format'] = format_str
		self.timestamp_format = format_str
		# Refresh displays to show new format
		self.load_todos()
		self.load_logs()

	def on_tab_changed(self, index):
		"""Handle tab change event"""
		gprefs['comfy_reminders_last_tab'] = index

	def on_sort_changed(self, logical_index, order):
		"""Handle sorting when the sort indicator changes"""
		# Skip Notes and Actions columns
		if logical_index in [2, 4]:
			return

		# Sort the data
		todos = plugin_prefs['todos']
		if todos:
			# Choose sort key based on column
			key = None
			if logical_index == 0:  # Title
				def key(x): return x.get('title', '').lower()
			elif logical_index == 1:  # Reminder
				def get_reminder_time(todo):
					todo_id = str(todo.get('id'))
					times = plugin_prefs.get('notification_times', {})
					return times.get(todo_id, '')
				key = get_reminder_time
			elif logical_index == 3:  # Created
				def key(x): return x.get('created', '')

			if key:
				# Sort using Qt's order indicator
				todos.sort(
					key=key, reverse=(
						order == Qt.SortOrder.DescendingOrder))
				plugin_prefs['todos'] = todos
				self.load_todos()

	def setup_todo_tab(self):
		layout = QVBoxLayout(self.todo_tab)
		layout.setSpacing(6)

		# Add todo section
		add_layout = QHBoxLayout()
		add_layout.setSpacing(10)

		# DateTime and title section on the left
		left_side = QVBoxLayout()
		left_side.setSpacing(4)

		# Title section using form layout
		title_form = QFormLayout()
		title_form.setSpacing(4)
		title_form.setContentsMargins(0, 0, 0, 0)
		title_label = QLabel(_('Title:'))
		title_label.setStyleSheet('font-weight: bold;')
		self.title_input = QLineEdit(self)
		self.title_input.setPlaceholderText(_('What do you want to remember?'))
		self.title_input.setMinimumWidth(300)
		# Increase height for better visibility
		self.title_input.setMinimumHeight(30)
		# Add styling to make it more prominent in dark mode
		self.title_input.setStyleSheet("""
			QLineEdit {
			border: 1px solid #5B5B5B;
			border-radius: 3px;
			padding: 4px 8px;
			background-color: palette(base);
			color: palette(text);
			font-size: 12pt;
			}
			QLineEdit:focus {
			border: 2px solid #0078D4;
			}
			""")
		self.title_input.returnPressed.connect(self.add_todo)
		title_form.addRow(title_label, self.title_input)
		left_side.addLayout(title_form)

		# DateTime section with left-aligned title
		datetime_group = QGroupBox(_("Set Reminder Time"))
		datetime_group.setStyleSheet("""
			QGroupBox {
				font-weight: bold;
				text-align: left;
			}
			QGroupBox::title {
				subcontrol-origin: margin;
				subcontrol-position: top left;
				padding: 0 5px;
				margin-top: 3px;
			}
		""")
		datetime_layout = QVBoxLayout()

		# Calendar widget - removed custom styling
		calendar = QCalendarWidget(self)
		calendar.setGridVisible(True)
		calendar.setFirstDayOfWeek(Qt.DayOfWeek.Monday)
		calendar.setVerticalHeaderFormat(
			QCalendarWidget.VerticalHeaderFormat.NoVerticalHeader)
		calendar.setMinimumDate(QDateTime.currentDateTime().date())
		datetime_layout.addWidget(calendar)

		# Time edit and Add button in same row
		action_layout = QHBoxLayout()
		action_layout.setSpacing(4)  # Reduce spacing
		action_layout.setContentsMargins(0, 0, 0, 0)

		time_label = QLabel(_("Time:"))
		action_layout.addWidget(time_label)

		time_edit = QTimeEdit(self)
		time_edit.setDisplayFormat("HH:mm")
		time_edit.setTime(QDateTime.currentDateTime().time())
		time_edit.setButtonSymbols(QTimeEdit.ButtonSymbols.UpDownArrows)
		action_layout.addWidget(time_edit)

		action_layout.addStretch()

		self.add_button = QPushButton(_('Add reminder'), self)
		self.add_button.clicked.connect(self.add_todo)
		self.add_button.setMinimumHeight(32)  # Increase height
		self.add_button.setMinimumWidth(95)
		action_layout.addWidget(self.add_button)

		datetime_layout.addLayout(action_layout)
		datetime_group.setLayout(datetime_layout)
		left_side.addWidget(datetime_group)

		# Create a container for the left side
		left_container = QWidget()
		left_container.setLayout(left_side)
		left_container.setContentsMargins(0, 0, 10, 0)
		add_layout.addWidget(left_container)

		# Quick Reminders section on the right
		quick_group = QGroupBox(_("Quick Reminders"))
		quick_group.setStyleSheet("""
			QGroupBox {
			font-weight: bold;
			text-align: center;
			}
			QGroupBox::title {
			subcontrol-origin: margin;
			subcontrol-position: top center;
			padding: 0 5px;
			}
			""")
		quick_layout = QVBoxLayout()
		quick_layout.setSpacing(4)

		# Preset buttons in rows
		presets = [(_("Minutes"),
					{_('1m'): 1,
					 _('3m'): 3,
					 _('5m'): 5,
					 _('10m'): 10,
					 _('15m'): 15,
					 _('25m'): 25,
					 _('30m'): 30,
					 _('45m'): 45,
					 }),
				   (_("Hours"),
					{_('1h'): 60,
					   _('2h'): 120,
					   _('4h'): 240,
					   _('6h'): 360,
					   _('8h'): 480,
					   _('12h'): 720,
					   _('16h'): 960,
					   _('18h'): 1080,
					 }),
				   (_("Days"),
					{_('24h'): 1440,
					   _('36h'): 2160,
					   _('48h'): 2880,
					   _('72h'): 4320,
					   _('7d'): 10080,
					   _('15d'): 21600,
					   _('21d'): 30240,
					   _('28d'): 40320,
					 }),
				   (_("Months"),
					{_('1M'): 43200,
					   _('2M'): 86400,
					   _('3M'): 129600,
					   _('4M'): 172800,
					   _('6M'): 259200,
					   _('8M'): 345600,
					   _('9M'): 388800,
					   _('12M'): 525600,
					 }),
				   (_("Years"),
					{_('2y'): 1051200,
					   _('3y'): 1576800,
					   _('4y'): 2102400,
					   _('5y'): 2628000,
					   _('6y'): 3153600,
					   _('7y'): 3679200,
					   _('8y'): 4204800,
					   _('10y'): 5256000,
					 })]

		for label, buttons in presets:
			row = QHBoxLayout()
			row.setSpacing(4)
			row.addWidget(QLabel(label))
			for btn_text, minutes in buttons.items():
				btn = QPushButton(btn_text)
				btn.setFixedWidth(40)
				# Fix the quick reminder callback method
				btn.clicked.connect(
					lambda checked,
					min_value=minutes: self.add_quick_reminder(min_value))
				row.addWidget(btn)
			row.addStretch()
			quick_layout.addLayout(row)

		quick_group.setLayout(quick_layout)
		add_layout.addWidget(quick_group)

		# Store calendar and time edit as instance variables
		self.alarm_calendar = calendar
		self.alarm_time_edit = time_edit

		layout.addLayout(add_layout)

		# Add filter box section and batch action buttons just above the
		# reminder list
		filter_batch_layout = QVBoxLayout()
		filter_batch_layout.setSpacing(8)

		# Add filter box with bold label
		filter_layout = QHBoxLayout()
		filter_label = QLabel(_("Filter reminders:"))
		filter_label.setStyleSheet('font-weight: bold;')
		filter_layout.addWidget(filter_label)
		self.filter_input = QLineEdit(self)
		self.filter_input.setPlaceholderText(
			_("Type to filter by reminder title..."))
		# Connect the filter function to textChanged event
		self.filter_input.textChanged.connect(self.filter_todos)
		filter_layout.addWidget(self.filter_input)
		filter_batch_layout.addLayout(filter_layout)

		# Add batch action buttons with label
		batch_layout = QHBoxLayout()
		batch_label = QLabel(_("Batch actions:"), self)
		batch_label.setStyleSheet("font-weight: bold;")
		batch_layout.addWidget(batch_label)

		select_all_btn = QPushButton(_('Select All'), self)
		select_all_btn.clicked.connect(self.select_all_todos)
		batch_layout.addWidget(select_all_btn)

		clear_selection_btn = QPushButton(_('Clear Selection'), self)
		clear_selection_btn.clicked.connect(self.clear_todo_selection)
		batch_layout.addWidget(clear_selection_btn)

		batch_layout.addStretch()

		batch_complete_btn = QPushButton(_('Complete Selected'), self)
		batch_complete_btn.clicked.connect(self.complete_selected_todos)
		batch_layout.addWidget(batch_complete_btn)

		batch_delete_btn = QPushButton(_('Delete Selected'), self)
		batch_delete_btn.clicked.connect(self.delete_selected_todos)
		batch_layout.addWidget(batch_delete_btn)

		filter_batch_layout.addLayout(batch_layout)
		layout.addLayout(filter_batch_layout)

		# Reminders table
		self.todo_table = QTableWidget(self)
		self.todo_table.setColumnCount(5)
		self.todo_table.setHorizontalHeaderLabels([
			_('Title'),
			_('Reminder time'),
			_('Notes'),
			_('Created'),
			_('Actions')
		])

		# Basic header setup with minimal configuration
		header = self.todo_table.horizontalHeader()
		header.setStretchLastSection(False)
		header.setSectionsClickable(True)
		header.setSortIndicatorShown(True)

		# Set column widths - keeping your preferred widths
		self.todo_table.setColumnWidth(0, 250)  # Title
		self.todo_table.setColumnWidth(1, 200)  # Reminder
		self.todo_table.setColumnWidth(2, 80)   # Notes
		self.todo_table.setColumnWidth(3, 150)  # Created
		self.todo_table.setColumnWidth(4, 160)  # Actions

		# Set ALL columns to Interactive mode for proper resize handles
		for col in range(self.todo_table.columnCount()):
			header.setSectionResizeMode(col, header.ResizeMode.Interactive)

		# Connect sorting directly to the sortIndicatorChanged signal
		header.sortIndicatorChanged.connect(self.on_sort_changed)

		layout.addWidget(self.todo_table)
		# Export buttons
		button_layout = QHBoxLayout()
		button_layout.setSpacing(4)

		# Export menu button
		export_btn = QPushButton(_('Export'))
		export_menu = QMenu(self)
		export_xlsx_action = export_menu.addAction(_('Export to Excel'))
		export_csv_action = export_menu.addAction(_('Export to CSV'))
		export_rtf_action = export_menu.addAction(_('Export to RTF'))
		export_btn.setMenu(export_menu)
		export_btn.setMinimumWidth(120)
		export_xlsx_action.triggered.connect(self.export_all_to_xlsx)
		export_csv_action.triggered.connect(self.export_all_to_ods)
		export_rtf_action.triggered.connect(self.export_all_to_rtf)
		button_layout.addWidget(export_btn)

		# ICS Calendar button
		ics_btn = QPushButton(_('ICS Calendar'))
		ics_menu = QMenu(self)
		export_ics_action = ics_menu.addAction(_('Export to ICS'))
		import_ics_action = ics_menu.addAction(_('Import from ICS'))
		ics_btn.setMenu(ics_menu)
		ics_btn.setMinimumWidth(120)
		export_ics_action.triggered.connect(self.export_all_to_ics)
		import_ics_action.triggered.connect(self.import_ics)
		button_layout.addWidget(ics_btn)
		json_btn = QPushButton(_('Backup'))
		json_menu = QMenu(self)
		export_json_action = json_menu.addAction(_('Save Backup'))
		import_json_action = json_menu.addAction(_('Restore from Backup'))
		json_btn.setMenu(json_menu)
		json_btn.setMinimumWidth(120)
		export_json_action.triggered.connect(self.export_json)
		import_json_action.triggered.connect(self.import_json)
		button_layout.addWidget(json_btn)

		restore_btn = QPushButton(_('Restore automatic backup'))
		restore_btn.setMinimumWidth(120)
		restore_btn.clicked.connect(self.restore_backup)
		button_layout.addWidget(restore_btn)

		settings_btn = QPushButton(_('Settings'))
		settings_btn.setMinimumWidth(120)
		settings_btn.clicked.connect(self.show_config_dialog)
		button_layout.addWidget(settings_btn)

		button_layout.addStretch()
		layout.addLayout(button_layout)

	def create_quick_reminder_callback(self, minutes):
		"""Create callback function for quick reminder button"""
		def callback():
			self.add_quick_reminder(minutes)
		return callback

	def setup_log_tab(self):
		layout = QVBoxLayout(self.log_tab)

		# Add log entry section
		add_layout = QHBoxLayout()

		# Category section with edit button
		category_layout = QHBoxLayout()
		self.category_combo = QComboBox()
		self.categories = plugin_prefs.get('categories', [
			_('Activity'),
			_('Notable Quotes'),
			_('Weight'),
			_('Observation'),
			_('Health'),
			_('Mood'),
			_('Custom')
		])
		self.category_combo.addItems(self.categories)
		category_layout.addWidget(QLabel(_('Category:')))
		category_layout.addWidget(self.category_combo)

		edit_categories_btn = QPushButton(_('Edit Categories'))
		edit_categories_btn.clicked.connect(self.edit_categories)
		category_layout.addWidget(edit_categories_btn)
		add_layout.addLayout(category_layout)

		# Value input with Enter key support
		self.value_input = QLineEdit()
		self.value_input.setPlaceholderText(
			_('add activity, observation etc.'))
		self.value_input.returnPressed.connect(self.add_log_entry)
		# Changed from 'Value' to 'Note'
		add_layout.addWidget(QLabel(_('Note:')))
		add_layout.addWidget(self.value_input)

		# Add button
		add_btn = QPushButton(_('Add Log Entry'))
		add_btn.clicked.connect(self.add_log_entry)
		add_layout.addWidget(add_btn)

		layout.addLayout(add_layout)

		# Log entries table
		self.log_table = QTableWidget()
		# Reduced from 5 to 4 (removed Notes column)
		self.log_table.setColumnCount(4)
		self.log_table.setHorizontalHeaderLabels([_('Timestamp'), _(
			'Category'), _('Note'), _('Actions')])  # Changed Value to Note
		self.log_table.horizontalHeader().setStretchLastSection(
			False)  # Changed to false to control action column width
		self.log_table.setColumnWidth(0, 150)  # Timestamp
		self.log_table.setColumnWidth(1, 100)  # Category
		# Note - made wider since we removed Notes column
		self.log_table.setColumnWidth(2, 300)
		# Actions - just enough for delete icon
		self.log_table.setColumnWidth(3, 160)
		layout.addWidget(self.log_table)

		# Connect double-click on Note column to in-place editing
		self.log_table.cellChanged.connect(self.on_log_cell_changed)

		# Add scrollable detail panel at the bottom
		detail_layout = QVBoxLayout()

		# Create info section with metadata
		info_layout = QHBoxLayout()
		self.detail_timestamp_label = QLabel("")
		self.detail_timestamp_label.setStyleSheet("font-weight: bold;")
		info_layout.addWidget(self.detail_timestamp_label)

		info_layout.addStretch()

		self.detail_category_label = QLabel("")
		self.detail_category_label.setStyleSheet("font-weight: bold;")
		info_layout.addWidget(self.detail_category_label)

		detail_layout.addLayout(info_layout)

		# Create editable text area for displaying and editing note content
		# only
		self.detail_text = QPlainTextEdit()
		self.detail_text.setReadOnly(False)  # Make editable
		# Set minimum height for visibility
		self.detail_text.setMinimumHeight(100)
		self.detail_text.setStyleSheet("""
			QPlainTextEdit {
			background-color: palette(base);
			color: palette(text);
			border: 1px solid #5B5B5B;
			border-radius: 3px;
			padding: 4px;
			}
			""")
		detail_layout.addWidget(self.detail_text)

		# Add Save Changes button
		save_layout = QHBoxLayout()
		save_layout.addStretch()
		save_detail_btn = QPushButton(_("Save Changes"))
		save_detail_btn.clicked.connect(self.save_entry_details)
		save_layout.addWidget(save_detail_btn)
		detail_layout.addLayout(save_layout)

		# Create a group box to contain everything
		detail_group = QGroupBox(_("Selected Entry Details (Editable)"))
		detail_group.setStyleSheet("""
			QGroupBox {
			font-weight: bold;
			text-align: center;
			}
			QGroupBox::title {
			subcontrol-origin: margin;
			subcontrol-position: top center;
			padding: 0 5px;
			}
			""")
		detail_group.setLayout(detail_layout)
		layout.addWidget(detail_group)

		# Connect table selection changed signal
		self.log_table.itemSelectionChanged.connect(
			self.on_log_selection_changed)
		# Export buttons section
		btn_layout = QHBoxLayout()
		btn_layout.setSpacing(4)  # Consistent spacing

		# Export menu button
		export_btn = QPushButton(_('Export'))
		export_menu = QMenu(self)
		export_xlsx_action = export_menu.addAction(_('Export to Excel'))
		export_csv_action = export_menu.addAction(_('Export to CSV'))
		export_rtf_action = export_menu.addAction(_('Export to RTF'))
		export_btn.setMenu(export_menu)
		export_btn.setMinimumWidth(120)
		export_xlsx_action.triggered.connect(self.export_all_to_xlsx)
		export_csv_action.triggered.connect(self.export_all_to_ods)
		export_rtf_action.triggered.connect(self.export_all_to_rtf)
		btn_layout.addWidget(export_btn)

		# ICS Calendar button
		ics_btn = QPushButton(_('ICS Calendar'))
		ics_menu = QMenu(self)
		export_ics_action = ics_menu.addAction(_('Export to ICS'))
		import_ics_action = ics_menu.addAction(_('Import from ICS'))
		ics_btn.setMenu(ics_menu)
		ics_btn.setMinimumWidth(120)
		export_ics_action.triggered.connect(self.export_all_to_ics)
		import_ics_action.triggered.connect(self.import_ics)
		btn_layout.addWidget(ics_btn)
		# JSON menu button
		json_btn = QPushButton(_('Backup'))
		json_menu = QMenu(self)
		export_json_action = json_menu.addAction(_('Save Backup'))
		import_json_action = json_menu.addAction(_('Restore from Backup'))
		json_btn.setMenu(json_menu)
		json_btn.setMinimumWidth(120)
		export_json_action.triggered.connect(self.export_json)
		import_json_action.triggered.connect(self.import_json)
		btn_layout.addWidget(json_btn)

		restore_btn = QPushButton(_('Restore automatic backup'))
		restore_btn.setMinimumWidth(120)
		restore_btn.clicked.connect(self.restore_backup)
		btn_layout.addWidget(restore_btn)

		settings_btn = QPushButton(_('Settings'))
		settings_btn.setMinimumWidth(120)
		settings_btn.clicked.connect(self.show_config_dialog)
		btn_layout.addWidget(settings_btn)

		btn_layout.addStretch()
		layout.addLayout(btn_layout)

	def toggle_completed(self, row):
		try:
			todos = plugin_prefs['todos']
			if 0 <= row < len(todos):
				todo = todos[row]
				is_completed = not todo.get(
					'completed', False)  # Toggle completed state
				todo['completed'] = is_completed
				plugin_prefs['todos'] = todos

				# Get reminder info for display
				todo_id = str(todo['id'])
				notification_times = plugin_prefs.get('notification_times', {})
				notification_history = plugin_prefs.get(
					'notification_history', {})
				reminder_status = plugin_prefs.get('reminder_status', {})

				# Update visual states for title and reminder time
				title_item = self.todo_table.item(row, 0)
				if title_item:
					font = title_item.font()
					font.setStrikeOut(is_completed)
					title_item.setFont(font)

				# Update checkmark button
				done_btn = self.todo_table.cellWidget(row, 3)
				if done_btn:
					done_btn.setText(_('✓') if is_completed else _('⬜'))
					if is_completed:
						done_btn.setStyleSheet(
							'QPushButton { font-weight: bold; color: white; background-color: #28a745; }')
					else:
						done_btn.setStyleSheet('QPushButton { color: #666; }')

				# Force immediate visual refresh
				self.todo_table.viewport().update()

				# Refresh the whole display to update reminder display
				self.load_todos()

		except Exception as e:
			print(f"Error in toggle_completed: {str(e)}")
			print(traceback.format_exc())

	def edit_notes(self, row):
		"""Open notes dialog for editing reminder notes"""
		try:
			todos = plugin_prefs['todos']
			if 0 <= row < len(todos):
				todo = todos[row]
				dialog = NotesDialog(self, todo.get('notes', ''))
				if dialog.exec() == QDialog.DialogCode.Accepted:
					todo['notes'] = dialog.get_notes()
					plugin_prefs['todos'] = todos

					# Update the Notes button appearance
					notes_btn = self.todo_table.cellWidget(
						row, 2)  # Notes column
					if notes_btn:
						notes_btn.setText(
							_('📝 Notes') if todo['notes'] else _('Notes'))
						notes_btn.setStyleSheet(
							'QPushButton { font-weight: bold; }' if todo['notes'] else '')

					# Force immediate visual refresh
					self.todo_table.viewport().update()
		except Exception as e:
			print(f"Error in edit_notes: {str(e)}")
			print(traceback.format_exc())

	def update_notes_button(self, button, has_notes):
		if has_notes:
			button.setText(_('📝 Notes'))
			button.setStyleSheet('QPushButton { font-weight: bold; }')
		else:
			button.setText(_('Notes'))
			button.setStyleSheet('')

	def load_todos(self):
		try:
			# Get all state data
			todos = plugin_prefs.get('todos', [])
			times = plugin_prefs.get('notification_times', {})
			history = plugin_prefs.get('notification_history', {})
			status = plugin_prefs.get('reminder_status', {})

			# Clear and reset table
			self.todo_table.setRowCount(0)
			self.todo_table.setRowCount(len(todos))

			current_time = now().astimezone(datetime.timezone.utc)
			row = 0

			for todo in todos:
				todo_id = str(todo.get('id'))

				# Title with completion status
				title_item = QTableWidgetItem(todo.get('title', ''))
				if todo.get('completed', False):
					font = title_item.font()
					font.setStrikeOut(True)
					title_item.setFont(font)
				self.todo_table.setItem(row, 0, title_item)

				# Reminder time
				reminder_item = QTableWidgetItem()
				self.todo_table.setItem(row, 1, reminder_item)
				self.update_reminder_display(reminder_item, todo_id)

				# Notes button in Notes column
				notes_btn = QPushButton(
					_('📝 Notes') if todo.get('notes') else _('Notes'), self)
				notes_btn.clicked.connect(
					lambda checked, r=row: self.edit_notes(r))
				self.todo_table.setCellWidget(row, 2, notes_btn)

				# Created timestamp
				created = datetime.datetime.fromisoformat(todo['created'])
				created_qt = qt_from_dt(created)
				self.todo_table.setItem(
					row, 3, QTableWidgetItem(
						created_qt.toString(
							self.timestamp_format)))
				# Actions layout with tighter spacing
				actions_widget = QWidget()
				actions_layout = QHBoxLayout(actions_widget)
				actions_layout.setSpacing(2)  # Keep minimal spacing
				actions_layout.setContentsMargins(
					2, 0, 2, 0)  # Minimal side margins

				# Complete button
				complete_btn = QPushButton(
					_('✓') if todo.get('completed', False) else _('✓'), self)
				# Set name for identification
				complete_btn.setObjectName('complete_btn')
				complete_btn.setToolTip(_('Mark as completed'))
				# Fixed width for all action buttons
				complete_btn.setFixedWidth(50)
				if todo.get('completed', False):
					complete_btn.setStyleSheet(
						'QPushButton { font-weight: bold; color: white; background-color: #28a745; }')
				else:
					# Style for unchecked to ensure visibility
					complete_btn.setStyleSheet('''
						QPushButton { 
							color: palette(text);
							border: 1px solid palette(mid);
							border-radius: 3px;
							font-weight: normal;
							background-color: transparent;
						}
						QPushButton:hover {
							background-color: palette(alternate-base);
						}
					''')
				complete_btn.clicked.connect(
					lambda checked, r=row: self.toggle_completed(r))
				actions_layout.addWidget(complete_btn)

				# Edit button for reminder time
				edit_btn = QPushButton(_('⏰'), self)
				# Fixed width for all action buttons
				edit_btn.setFixedWidth(50)
				edit_btn.setToolTip(_('Edit Reminder'))
				edit_btn.clicked.connect(
					lambda checked, r=row: self.edit_alarm(r))
				actions_layout.addWidget(edit_btn)

				# Delete button
				delete_btn = QPushButton(_('❌'), self)
				# Fixed width for all action buttons
				delete_btn.setFixedWidth(50)
				delete_btn.setToolTip(_('Delete'))
				delete_btn.clicked.connect(
					lambda checked, r=row: self.delete_todo(r))
				actions_layout.addWidget(delete_btn)

				self.todo_table.setCellWidget(row, 4, actions_widget)

				row += 1

		except Exception as e:
			print(f"Error in load_todos: {str(e)}")
			print(traceback.format_exc())

	def get_reminder_time(self, reminder_id):
		"""Get the correct reminder time, prioritizing notification_times over created timestamp"""
		if reminder_id in plugin_prefs.get('notification_times', {}):
			return QDateTime.fromString(
				plugin_prefs['notification_times'][reminder_id], Qt.ISODate)
		elif reminder_id in plugin_prefs.get('notification_history', {}):
			return QDateTime.fromString(plugin_prefs.get(
				'notification_history')[reminder_id], Qt.ISODate)
		else:
			# Fallback to created time if no reminder time is found
			todo = next(
				(t for t in plugin_prefs.get(
					'todos',
					[]) if t['id'] == reminder_id),
				None)
			if todo:
				return QDateTime.fromString(todo['created'], Qt.ISODate)
		return None

	def update_reminder_display(self, reminder_item, reminder_id):
		"""Update the reminder display based on stored reminder time"""
		try:
			notification_times = plugin_prefs.get("notification_times", {})
			notification_history = plugin_prefs.get("notification_history", {})
			reminder_status = plugin_prefs.get("reminder_status", {})
			current_time = now().astimezone(datetime.timezone.utc)

			# Get the reminder time from either active or history
			reminder_time = None
			is_active = False			# First check active reminders
			if reminder_id in notification_times:
				reminder_time = datetime.datetime.fromisoformat(
					notification_times[reminder_id])
				is_active = True
			# Then check historical reminders
			elif reminder_id in notification_history:
				reminder_time = datetime.datetime.fromisoformat(
					notification_history[reminder_id])
				is_active = False

			if reminder_time:
				reminder_qt = qt_from_dt(reminder_time)
				reminder_str = reminder_qt.toString(self.timestamp_format)

				if is_active:
					# Active reminder
					if reminder_time > current_time:
						# Calculate time until due
						delta = reminder_time - current_time
						time_until = self.format_overdue_time(delta.total_seconds())
						reminder_item.setText(_('⏰ Due %s - %s to go') % (reminder_str, time_until))
						reminder_item.setToolTip(_('Due at %s') % reminder_str)
					else:
						# Overdue but still active
						delta = current_time - reminder_time
						overdue_text = self.format_overdue_time(delta.total_seconds())
						reminder_item.setText(_('🔴 Overdue by %s') % overdue_text)
						reminder_item.setToolTip(_('Was due at %s') % reminder_str)
				else:
					# Historical reminder
					status = reminder_status.get(
						reminder_id, REMINDER_STATUS["TRIGGERED"])
					delta = current_time - reminder_time
					delta_text = self.format_overdue_time(
						delta.total_seconds())

					if status == REMINDER_STATUS["CLEARED"]:
						reminder_item.setText(
							_('❌ Cleared (%s ago)') % delta_text)
						reminder_item.setToolTip(
							_('Was due at %s') %
							reminder_str)
					elif status == REMINDER_STATUS["TRIGGERED"]:
						reminder_item.setText(
							_('🔴 Overdue by %s') %
							delta_text)
						reminder_item.setToolTip(
							_('Was due at %s') %
							reminder_str)
					else:
						reminder_item.setText(
							_('⚡ Previous reminder (%s)') % reminder_str)
						reminder_item.setToolTip(
							_('Was due at %s') %
							reminder_str)
			else:
				# No reminder time found
				status = reminder_status.get(
					reminder_id, REMINDER_STATUS["NEVER_SET"])
				if status == REMINDER_STATUS["NEVER_SET"]:
					reminder_item.setText(_("⏰ No reminder set"))
					reminder_item.setToolTip(_("No reminder set"))
				else:
					# Changed from "Set reminder..."
					reminder_item.setText(_("⏰ No reminder set"))
					reminder_item.setToolTip(_("No reminder set"))

		except Exception as e:
			print(f"Error in update_reminder_display: {str(e)}")
			print(traceback.format_exc())

	def format_overdue_time(self, seconds_overdue):
		"""Format seconds into a human-readable overdue time string"""
		if seconds_overdue <= 0:
			return _("now")

		# Always round up to at least 1 minute
		# Round up by adding 59 seconds
		minutes = max(1, int((seconds_overdue + 59) // 60))
		hours = minutes // 60
		days = hours // 24

		if days > 365:
			years = days // 365
			remaining_days = days % 365
			if remaining_days > 0:
				return _("%sy %sd") % (years, remaining_days)
			return _("%sy") % years
		elif days > 30:
			months = days // 30
			remaining_days = days % 30
			if remaining_days > 0:
				return _("%smo %sd") % (months, remaining_days)
			return _("%smo") % months
		elif days > 0:
			remaining_hours = hours % 24
			if remaining_hours > 0:
				return _("%sd %sh") % (days, remaining_hours)
			return _("%sd") % days
		elif hours > 0:
			remaining_minutes = minutes % 60
			if remaining_minutes > 0:
				return _("%sh %sm") % (hours, remaining_minutes)
			return _("%sh") % hours
		else:
			return _("%sm") % minutes

	def add_todo(self):
		"""Add a new reminder with proper time handling"""
		try:
			title = self.title_input.text().strip()
			if not title:
				from calibre.gui2 import error_dialog
				error_dialog(self, _('Missing Title'),
							 _('Please enter a title for the reminder.'),
							 show=True)
				return

			current_time = now().astimezone(datetime.timezone.utc)
			todo_id = str(uuid.uuid4())

			# Create todo with creation time
			todo = {
				'id': todo_id,
				'title': title,
				'created': current_time.isoformat()
			}

			# Insert todo at the beginning of the list
			todos = plugin_prefs.get('todos', [])
			todos.insert(0, todo)  # Insert at position 0 instead of appending
			plugin_prefs['todos'] = todos

			# Get selected reminder time
			selected_date = self.alarm_calendar.selectedDate()
			selected_time = self.alarm_time_edit.time()
			reminder_dt = QDateTime(selected_date, selected_time)
			reminder_dt.setTimeSpec(Qt.TimeSpec.LocalTime)
			reminder_time = qt_to_dt(reminder_dt, as_utc=True)  # Force UTC

			# Store the reminder time
			self.set_reminder_time(todo_id, reminder_time)

			# Clear input
			self.title_input.clear()
			current_dt = QDateTime.currentDateTime()
			current_dt.setTimeSpec(Qt.TimeSpec.LocalTime)
			self.alarm_calendar.setSelectedDate(current_dt.date())
			self.alarm_time_edit.setTime(current_dt.time())

			# Refresh table
			self.load_todos()

		except Exception as e:
			print(f"Error in add_todo: {str(e)}")
			print(traceback.format_exc())

	def add_quick_reminder(self, minutes):
		"""Add a todo with reminder set to current time + specified minutes"""
		try:
			# Generate a descriptive title if none provided
			base_title = self.title_input.text().strip()
			if not base_title:
				# Create a descriptive title using the minutes value
				time_suffix = ""
				if minutes < 60:
					time_suffix = _("(%sm)") % minutes
				elif minutes < 1440:
					time_suffix = _("(%sh)") % (minutes // 60)
				elif minutes < 43200:
					time_suffix = _("(%sd)") % (minutes // 1440)
				else:
					time_suffix = _("(%sM)") % (minutes // 43200)
				base_title = _("Quick Reminder %s") % time_suffix

			# Create todo
			todo_id = str(uuid.uuid4())
			current_time = now().astimezone(datetime.timezone.utc)
			todo = {
				'id': todo_id,
				'title': base_title,
				'created': current_time.isoformat()
			}

			# Set reminder time to current time + minutes, ensuring UTC
			reminder_time = current_time + datetime.timedelta(minutes=minutes)

			# Save todo and reminder time
			todos = plugin_prefs.get('todos', [])
			notification_times = plugin_prefs.get('notification_times', {})
			notification_history = plugin_prefs.get('notification_history', {})
			reminder_status = plugin_prefs.get('reminder_status', {})

			# Insert at beginning instead of appending
			todos.insert(0, todo)

			if reminder_time > current_time:
				# Add to active reminders
				notification_times[todo_id] = reminder_time.isoformat()
				reminder_status[todo_id] = REMINDER_STATUS['ACTIVE']
			else:
				# Already passed, add to history instead
				notification_history[todo_id] = reminder_time.isoformat()
				reminder_status[todo_id] = REMINDER_STATUS['TRIGGERED']

			plugin_prefs['todos'] = todos
			plugin_prefs['notification_times'] = notification_times
			plugin_prefs['notification_history'] = notification_history
			plugin_prefs['reminder_status'] = reminder_status

			# Clear input
			self.title_input.clear()

			# Refresh table
			self.load_todos()

		except Exception as e:
			print(f"Error in add_quick_reminder: {str(e)}")
			print(traceback.format_exc())

	def set_reminder_time(self, todo_id, reminder_time, clear=False):
		"""Set a reminder time for a todo, handling active vs historical state"""
		try:
			current_time = now().astimezone(datetime.timezone.utc)
			notification_times = plugin_prefs.get('notification_times', {})
			notification_history = plugin_prefs.get('notification_history', {})
			reminder_status = plugin_prefs.get('reminder_status', {})

			# If there's an existing reminder, move it to history
			if todo_id in notification_times:
				old_reminder = notification_times[todo_id]
				notification_history[todo_id] = old_reminder

			if clear:
				# Clear the reminder
				if todo_id in notification_times:
					notification_history[todo_id] = notification_times[todo_id]
					reminder_status[todo_id] = REMINDER_STATUS['CLEARED']
					del notification_times[todo_id]
			else:
				# Set the new reminder based on whether it's in the future or
				# past
				if reminder_time > current_time:
					# Future reminder
					notification_times[todo_id] = reminder_time.isoformat()
					reminder_status[todo_id] = REMINDER_STATUS['ACTIVE']
				else:
					# Past reminder - store directly in history
					notification_history[todo_id] = reminder_time.isoformat()
					reminder_status[todo_id] = REMINDER_STATUS['TRIGGERED']

			# Save changes
			plugin_prefs['notification_times'] = notification_times
			plugin_prefs['notification_history'] = notification_history
			plugin_prefs['reminder_status'] = reminder_status

		except Exception as e:
			print(f"Error in set_reminder_time: {str(e)}")
			print(traceback.format_exc())

	def set_reminder_state(
			self,
			todo_id,
			notification_time=None,
			is_triggered=False):
		"""Atomically update reminder state to prevent loss during transitions"""
		try:
			notification_times = plugin_prefs.get(
				'notification_times', {}).copy()
			notification_history = plugin_prefs.get(
				'notification_history', {}).copy()
			reminder_status = plugin_prefs.get('reminder_status', {}).copy()
			current_status = reminder_status.get(todo_id)

			if is_triggered:
				# If it's a triggered reminder, preserve its state
				if todo_id in notification_times:
					# Move from active to history
					notification_history[todo_id] = notification_times[todo_id]
					del notification_times[todo_id]
					reminder_status[todo_id] = REMINDER_STATUS['TRIGGERED']
				elif current_status == REMINDER_STATUS['TRIGGERED']:
					# Already triggered - ensure it's in history
					if todo_id not in notification_history:
						# Try to recover from todos if needed
						todos = plugin_prefs.get('todos', [])
						todo = next(
							(t for t in todos if str(
								t['id']) == todo_id), None)
						if todo:
							notification_history[todo_id] = todo['created']
			elif notification_time:
				# New active reminder
				if current_status in [
						REMINDER_STATUS['TRIGGERED'],
						REMINDER_STATUS['CLEARED']]:
					# Keep history of the old reminder
					if todo_id in notification_times:
						notification_history[todo_id] = notification_times[todo_id]
					# Set new active reminder
					notification_times[todo_id] = notification_time.isoformat()
					reminder_status[todo_id] = REMINDER_STATUS['ACTIVE']
					# Simple active reminder update
					notification_times[todo_id] = notification_time.isoformat()
					reminder_status[todo_id] = REMINDER_STATUS['ACTIVE']

			# Save all changes atomically
			plugin_prefs['notification_times'] = notification_times
			plugin_prefs['notification_history'] = notification_history
			plugin_prefs['reminder_status'] = reminder_status

		except Exception as e:
			print(f"Error in set_reminder_state: {str(e)}")
			print(traceback.format_exc())

	# Add a utility function to fix any reminders with inconsistent state
	def fix_reminder_status(self):
		"""Check for and fix inconsistencies in reminder status"""
		try:
			todos = plugin_prefs.get('todos', [])
			notification_times = plugin_prefs.get('notification_times', {})
			notification_history = plugin_prefs.get('notification_history', {})
			reminder_status = plugin_prefs.get('reminder_status', {})
			current_time = now().astimezone(datetime.timezone.utc)

			fixed_count = 0

			# Check for reminders that are missing proper status
			for todo in todos:
				todo_id = str(todo['id'])

				# Check if a reminder exists in notification_times and should
				# be moved to history
				if todo_id in notification_times:
					reminder_time = datetime.datetime.fromisoformat(
						notification_times[todo_id])
					if reminder_time <= current_time:
						# Store reminder in history without modifying the time
						history[todo_id] = notification_times[todo_id]
						reminder_status[todo_id] = REMINDER_STATUS['TRIGGERED']
						del notification_times[todo_id]
						fixed_count += 1
						continue

				# Check if reminder is triggered but missing from history
				current_status = reminder_status.get(todo_id)
				if current_status == REMINDER_STATUS['TRIGGERED'] and todo_id not in notification_history:
					# Try to recover from created date if available
					created_time = datetime.datetime.fromisoformat(
						todo['created'])
					notification_history[todo_id] = created_time.isoformat()
					fixed_count += 1
					continue

				# Only reset status to NEVER_SET if it's not already in a valid
				# terminal state
				if (todo_id not in notification_times and
						todo_id not in notification_history and
						current_status not in [
						REMINDER_STATUS['NEVER_SET'],
						REMINDER_STATUS['TRIGGERED'],
						REMINDER_STATUS['CLEARED']
						]):
					reminder_status[todo_id] = REMINDER_STATUS['NEVER_SET']
					fixed_count += 1

			# Save changes if any fixes were made
			if fixed_count > 0:
				plugin_prefs['notification_times'] = notification_times
				plugin_prefs['notification_history'] = notification_history
				plugin_prefs['reminder_status'] = reminder_status
				return True

			return False
		except Exception as e:
			print(f"Error in fix_reminder_status: {str(e)}")
			print(traceback.format_exc())
			return False

	def delete_todo(self, row):
		try:
			todos = plugin_prefs['todos']
			if 0 <= row < len(todos):
				# Add confirmation dialog
				from calibre.gui2 import question_dialog
				if not question_dialog(
						self,
						_('Confirm Delete'),
						_('Are you sure you want to delete this reminder?'),
						show_copy_button=False):
					return

				todo = todos[row]
				todo_id = str(todo['id'])

				# Remove todo
				todos.pop(row)
				plugin_prefs['todos'] = todos

				# Remove any associated notification data
				notification_times = plugin_prefs.get('notification_times', {})

				# Remove any associated notification data
				notification_times = plugin_prefs.get('notification_times', {})
				notification_history = plugin_prefs.get(
					'notification_history', {})
				reminder_status = plugin_prefs.get('reminder_status', {})

				if todo_id in notification_times:
					del notification_times[todo_id]

				if todo_id in notification_history:
					del notification_history[todo_id]

				if todo_id in reminder_status:
					del reminder_status[todo_id]

				plugin_prefs['notification_times'] = notification_times
				plugin_prefs['notification_history'] = notification_history
				plugin_prefs['reminder_status'] = reminder_status

				# Refresh table
				self.load_todos()
		except Exception as e:
			print(f'TodoDialog: Error deleting todo: {str(e)}')
			print(traceback.format_exc())

	def delete_selected_todos(self):
		"""Delete all selected todos"""
		from calibre.gui2 import question_dialog

		selected_rows = self.get_selected_rows()
		if not selected_rows:
			return

		if not question_dialog(
				self,
				_('Confirm Delete'),
				_('Are you sure you want to delete the selected reminders?'),
				show_copy_button=False):
			return

		todos = plugin_prefs['todos']
		notification_times = plugin_prefs.get('notification_times', {})
		notification_history = plugin_prefs.get('notification_history', {})
		reminder_status = plugin_prefs.get('reminder_status', {})

		# Process in reverse order to avoid index shifting issues
		for row in reversed(selected_rows):
			if 0 <= row < len(todos):
				todo = todos[row]
				todo_id = str(todo['id'])

				# Remove todo
				todos.pop(row)

				# Clean up associated data
				if todo_id in notification_times:
					del notification_times[todo_id]
				if todo_id in notification_history:
					del notification_history[todo_id]
				if todo_id in reminder_status:
					del reminder_status[todo_id]

		plugin_prefs['todos'] = todos
		plugin_prefs['notification_times'] = notification_times
		plugin_prefs['notification_history'] = notification_history
		plugin_prefs['reminder_status'] = reminder_status

		# Refresh display
		self.load_todos()

	def complete_selected_todos(self):
		"""Mark all selected todos as completed"""
		selected_rows = self.get_selected_rows()
		if not selected_rows:
			return

		todos = plugin_prefs['todos']
		modified = False

		for row in selected_rows:
			if 0 <= row < len(todos):
				todo = todos[row]
				todo['completed'] = True
				modified = True

				# Update visual state
				title_item = self.todo_table.item(row, 0)
				if title_item:
					font = title_item.font()
					font.setStrikeOut(True)
					title_item.setFont(font)

				actions_widget = self.todo_table.cellWidget(
					row, 4)  # Actions column
				if actions_widget:
					complete_btn = actions_widget.findChild(
						QPushButton, name='complete_btn')
					if complete_btn:
						complete_btn.setText(_('✓'))
						complete_btn.setStyleSheet(
							'QPushButton { font-weight: bold; color: white; background-color: #28a745; }')

		if modified:
			plugin_prefs['todos'] = todos
		# Fully refresh the display to ensure all visual states are updated
		self.load_todos()

	def load_logs(self):
		try:
			# Initialize life_logs if not present
			if 'life_logs' not in plugin_prefs:
				plugin_prefs['life_logs'] = []

			logs = plugin_prefs['life_logs']
			self.log_table.setRowCount(len(logs))

			for row, log in enumerate(logs):  # Fixed loop syntax here
				# Timestamp - make read-only
				timestamp = datetime.datetime.fromisoformat(log['timestamp'])
				timestamp_qt = qt_from_dt(timestamp)
				timestamp_item = QTableWidgetItem(
					timestamp_qt.toString(self.timestamp_format))
				timestamp_item.setFlags(
					timestamp_item.flags() & ~Qt.ItemFlag.ItemIsEditable)
				self.log_table.setItem(row, 0, timestamp_item)

				# Category - make read-only but add category combobox in
				# Actions
				category_item = QTableWidgetItem(log['category'])
				category_item.setFlags(
					category_item.flags() & ~Qt.ItemFlag.ItemIsEditable)
				self.log_table.setItem(row, 1, category_item)

				# Note (formerly Value) - Make this row higher if content is
				# large
				value_item = QTableWidgetItem(log['value'])
				self.log_table.setItem(row, 2, value_item)

				# Set row height based on content
				metrics = self.log_table.fontMetrics()
				text_width = metrics.boundingRect(log['value']).width()
				cell_width = self.log_table.columnWidth(2)
				if text_width > cell_width:
					lines = (text_width / cell_width) + \
						1  # Add 1 for partial lines
					row_height = int(metrics.height() *
									 min(lines, 3))  # Cap at 3 lines
					self.log_table.setRowHeight(
						row, max(row_height, self.log_table.rowHeight(row)))

				# Actions layout with category change and delete
				actions_widget = QWidget()
				actions_layout = QHBoxLayout(actions_widget)
				actions_layout.setSpacing(2)
				actions_layout.setContentsMargins(2, 0, 2, 0)

				# Category change combobox
				category_combo = QComboBox()
				category_combo.addItems(self.categories)
				category_combo.setCurrentText(log['category'])
				category_combo.currentTextChanged.connect(
					lambda text, r=row: self.change_log_category(r, text))
				category_combo.setFixedWidth(100)
				actions_layout.addWidget(category_combo)

				# Delete button
				delete_btn = QPushButton(_('🗑'), self)
				delete_btn.setToolTip(_('Delete'))
				delete_btn.clicked.connect(
					lambda checked, r=row: self.delete_log(r))
				delete_btn.setFixedWidth(30)
				actions_layout.addWidget(delete_btn)

				self.log_table.setCellWidget(row, 3, actions_widget)

		except Exception as e:
			print(f"Error in load_logs: {str(e)}")
			print(traceback.format_exc())

	def change_log_category(self, row, new_category):
		"""Change the category of a log entry"""
		try:
			logs = plugin_prefs.get('life_logs', [])
			if 0 <= row < len(logs):
				logs[row]['category'] = new_category
				plugin_prefs['life_logs'] = logs

				# Update the category display in the table
				category_item = self.log_table.item(row, 1)
				if category_item:
					category_item.setText(new_category)

				# Update category label if this entry is selected
				selected_items = self.log_table.selectedItems()
				if selected_items and selected_items[0].row() == row:
					self.detail_category_label.setText(
						_("Category: %s") % new_category)

		except Exception as e:
			print(f"Error in change_log_category: {str(e)}")
			print(traceback.format_exc())

	def add_log_entry(self):
		try:
			value = self.value_input.text().strip()
			if not value:
				return

			category = self.category_combo.currentText()

			log_entry = {
				'id': str(uuid.uuid4()),
				'timestamp': now().astimezone(datetime.timezone.utc).isoformat(),  # Add timestamp in UTC
				'category': category,
				'value': value,
			}

			# Save log entry
			logs = plugin_prefs.get('life_logs', [])
			logs.append(log_entry)
			plugin_prefs['life_logs'] = logs

			# Clear input
			self.value_input.clear()

			# Refresh table
			self.load_logs()

		except Exception as e:
			print(f"Error in add_log_entry: {str(e)}")
			print(traceback.format_exc())

	def edit_log_notes(self, row):
		try:
			logs = plugin_prefs['life_logs']
			if 0 <= row < len(logs):
				log = logs[row]
				notes_dialog = NotesDialog(self, log.get('notes', ''))
				if notes_dialog.exec() == QDialog.DialogCode.Accepted:
					log['notes'] = notes_dialog.get_notes()
					plugin_prefs['life_logs'] = logs
					# Update notes button appearance
					notes_btn = self.log_table.cellWidget(row, 3)
					if notes_btn:
						self.update_notes_button(
							notes_btn, bool(log.get('notes')))
					# Force immediate visual refresh
					self.log_table.viewport().update()
		except Exception as e:
			pass

	def delete_log(self, row):
		try:
			logs = plugin_prefs['life_logs']
			if 0 <= row < len(logs):
				logs.pop(row)
				plugin_prefs['life_logs'] = logs
				self.load_logs()
		except Exception as e:
			pass

	def edit_categories(self):
		dialog = CategoryDialog(self, self.categories)
		if dialog.exec() == QDialog.DialogCode.Accepted:
			self.categories = dialog.get_categories()
			plugin_prefs['categories'] = self.categories

			# Update combobox
			current_category = self.category_combo.currentText()
			self.category_combo.clear()
			self.category_combo.addItems(self.categories)

			# Try to restore previous selection
			index = self.category_combo.findText(current_category)
			if (index >= 0):
				self.category_combo.setCurrentIndex(index)

	def edit_alarm(self, row):
		try:
			todos = plugin_prefs['todos']
			if 0 <= row < len(todos):
				todo = todos[row]
				todo_id = str(todo['id'])

				# Create reminder edit dialog
				dialog = QDialog(self)
				dialog.setWindowTitle(_('Edit Reminder'))
				layout = QVBoxLayout(dialog)

				# DateTime section with left-aligned title
				datetime_group = QGroupBox(_("Set Reminder Time"))
				datetime_group.setStyleSheet("""
					QGroupBox {
						font-weight: bold;
						text-align: left;
					}
					QGroupBox::title {
						subcontrol-origin: margin;
						subcontrol-position: top left;
						padding: 0 5px;
						margin-top: 3px;
					}
				""")
				datetime_layout = QVBoxLayout()

				# Calendar widget
				calendar = QCalendarWidget(dialog)
				calendar.setGridVisible(True)
				calendar.setFirstDayOfWeek(Qt.DayOfWeek.Monday)
				calendar.setVerticalHeaderFormat(
					QCalendarWidget.VerticalHeaderFormat.NoVerticalHeader)
				calendar.setMinimumDate(
					QDateTime.currentDateTime().date().addYears(-1))
				datetime_layout.addWidget(calendar)

				# Time edit with spinners
				time_layout = QHBoxLayout()
				time_layout.addWidget(QLabel(_("Time:")))
				time_edit = QTimeEdit(dialog)
				time_edit.setDisplayFormat("HH:mm")
				time_edit.setCalendarPopup(True)
				time_edit.setTime(QDateTime.currentDateTime().time())
				time_edit.setButtonSymbols(
					QTimeEdit.ButtonSymbols.UpDownArrows)
				time_layout.addWidget(time_edit)
				time_layout.addStretch()
				datetime_layout.addLayout(time_layout)

				datetime_group.setLayout(datetime_layout)
				layout.addWidget(datetime_group)

				# Set current date/time if there's an existing reminder
				current_dt = QDateTime.currentDateTime()
				notification_times = plugin_prefs.get('notification_times', {})
				notification_history = plugin_prefs.get(
					'notification_history', {})

				# Get the most recent reminder time (from either active or
				# history)
				reminder_time = None
				if todo_id in notification_times:
					reminder_time = datetime.datetime.fromisoformat(
						notification_times[todo_id])
				elif todo_id in notification_history:
					reminder_time = datetime.datetime.fromisoformat(
						notification_history[todo_id])

				if reminder_time:
					reminder_qt = qt_from_dt(reminder_time)
					calendar.setSelectedDate(reminder_qt.date())
					time_edit.setTime(reminder_qt.time())

				# Replace button box with simple buttons
				btn_layout = QHBoxLayout()
				save_btn = QPushButton(_('Save'), dialog)
				save_btn.clicked.connect(dialog.accept)
				clear_btn = QPushButton(_('Clear Reminder'), dialog)
				clear_btn.clicked.connect(dialog.reject)
				btn_layout.addWidget(clear_btn)
				btn_layout.addWidget(save_btn)
				layout.addLayout(btn_layout)

				result = dialog.exec()

				if result == QDialog.DialogCode.Accepted:
					# Combine date from calendar and time from time edit
					selected_date = calendar.selectedDate()
					selected_time = time_edit.time()
					reminder_dt = QDateTime(selected_date, selected_time)
					reminder_dt.setTimeSpec(Qt.TimeSpec.LocalTime)
					reminder_time = qt_to_dt(
						reminder_dt, as_utc=True)  # Force UTC conversion

					# Set the reminder time using our new function
					self.set_reminder_time(todo_id, reminder_time)

				# Refresh display
				self.load_todos()

		except Exception as e:
			print(f'TodoDialog: Error editing alarm: {str(e)}')
			print(traceback.format_exc())

	def export_all_to_xlsx(self):
		try:
			from calibre.gui2.qt_file_dialogs import choose_save_file
			from calibre.gui2 import error_dialog, info_dialog
			import sys
			import os

			# Add plugin's directory to path temporarily to find openpyxl
			plugin_dir = os.path.dirname(os.path.realpath(__file__))
			try:
				sys.path.insert(0, plugin_dir)
				from openpyxl import Workbook
				sys.path.pop(0)  # Remove temporary path
			except ImportError as e:
				error_dialog(
					self,
					_('Missing dependency'),
					_('Could not import openpyxl: %s\nPlugin directory: %s') %
					(str(e),
					 plugin_dir),
					show=True)
				return			# Get current timestamp for filename
			timestamp = datetime.datetime.now().strftime('_%Y-%m-%d, %H-%M-%S')
			filename = choose_save_file(
				self, 'export-all-xlsx',
				_('Export All Data'),
				filters=[(_('XLSX files'), ['xlsx'])],
				all_files=False, initial_filename=f'reminders_and_notebook{timestamp}.xlsx'
			)

			if not filename:
				return

			wb = Workbook()

			# Reminders sheet
			ws_todos = wb.active
			ws_todos.title = _("Reminders")
			ws_todos.append([_('Title'), _('Created'), _(
				'Reminder'), _('Status'), _('Notes'), _('Completed')])

			todos = plugin_prefs.get('todos', [])
			notification_times = plugin_prefs.get('notification_times', {})
			notification_history = plugin_prefs.get('notification_history', {})
			reminder_status = plugin_prefs.get('reminder_status', {})

			for todo in todos:
				# Format timestamps using Qt
				created = datetime.datetime.fromisoformat(todo['created'])
				created_qt = qt_from_dt(created)
				created_str = created_qt.toString(self.timestamp_format)

				# Get reminder details
				todo_id = str(todo['id'])
				reminder_str = _('No reminder set')
				status_str = _('No reminder')

				if todo_id in notification_times:
					reminder_time = datetime.datetime.fromisoformat(
						notification_times[todo_id])
					reminder_qt = qt_from_dt(reminder_time)
					reminder_str = reminder_qt.toString(self.timestamp_format)
					status_str = _('Active')
				elif todo_id in notification_history:
					reminder_time = datetime.datetime.fromisoformat(
						notification_history[todo_id])
					reminder_qt = qt_from_dt(reminder_time)
					reminder_str = reminder_qt.toString(self.timestamp_format)

					# Get the status for better display
					status = reminder_status.get(
						todo_id, REMINDER_STATUS['TRIGGERED'])
					if status == REMINDER_STATUS['TRIGGERED']:
						status_str = _('Triggered')
					elif status == REMINDER_STATUS['CLEARED']:
						status_str = _('Manually cleared')
					else:
						status_str = _('Historical')

				ws_todos.append([
					todo['title'],
					created_str,
					reminder_str,
					status_str,
					todo.get('notes', ''),
					_('Yes') if todo.get('completed', False) else _('No')
				])

			# Notebook entries sheet
			ws_logs = wb.create_sheet(title="Notebook")
			ws_logs.append(['Timestamp', 'Category', 'Note'])

			logs = plugin_prefs.get('life_logs', [])
			for log in logs:
				# Format timestamp using Qt
				timestamp = datetime.datetime.fromisoformat(log['timestamp'])
				timestamp_qt = qt_from_dt(timestamp)
				time_str = timestamp_qt.toString(self.timestamp_format)

				ws_logs.append([
					time_str,
					log['category'],
					log['value']  # Just export the main note text
				])

			# Auto-adjust columns in both sheets
			for ws in [ws_todos, ws_logs]:
				for col in ws.columns:
					max_length = 0
					column = col[0].column_letter
					for cell in col:
						try:
							max_length = max(
								max_length, len(str(cell.value or '')))
						except BaseException:
							pass
					adjusted_width = (max_length + 2)
					ws.column_dimensions[column].width = adjusted_width

			# Save with proper error handling
			try:
				wb.save(filename)
				info_dialog(self, _('Export successful'),
							_('Data has been exported to %s') % filename,
							show=True)
			except PermissionError:
				error_dialog(
					self,
					_('Export failed'),
					_('Could not save file. Please ensure you have write permissions and the file is not open in another program.'),
					show=True)
			except Exception as e:
				error_dialog(self, _('Export failed'),
							 _('Failed to save XLSX file: %s') % str(e),
							 show=True,
							 det_msg=traceback.format_exc())

		except Exception as e:
			error_dialog(self, _('Export error'),
						 _('Failed to export to XLSX: %s') % str(e),
						 show=True,
						 det_msg=traceback.format_exc())

	def export_all_to_ods(self):
		try:
			from calibre.gui2.qt_file_dialogs import choose_save_file
			from calibre.gui2 import error_dialog, info_dialog
			# Get current timestamp for filename
			timestamp = datetime.datetime.now().strftime('_%Y-%m-%d, %H-%M-%S')
			filename = choose_save_file(
				self, 'export-all-csv',
				_('Export All Data'),
				filters=[(_('CSV files'), ['csv'])],
				all_files=False, initial_filename=f'reminders_and_notebook{timestamp}.csv'
			)

			if not filename:
				return            # Write data to CSV file
			with open(filename, 'w', newline='', encoding='utf-8') as csvfile:
				import csv
				writer = csv.writer(csvfile)

				# Write reminders section
				writer.writerow([_('=== REMINDERS ===')])
				writer.writerow([])  # Add blank row for better readability
				writer.writerow([_('Title'), _('Created'), _(
					'Reminder'), _('Status'), _('Notes'), _('Completed')])

				# Get all reminders data
				todos = plugin_prefs.get('todos', [])
				notification_times = plugin_prefs.get('notification_times', {})
				notification_history = plugin_prefs.get(
					'notification_history', {})
				reminder_status = plugin_prefs.get('reminder_status', {})

				for todo in todos:
					# Format timestamps using Qt
					created = datetime.datetime.fromisoformat(todo['created'])
					created_qt = qt_from_dt(created)
					created_str = created_qt.toString(self.timestamp_format)

					# Get reminder details
					todo_id = str(todo['id'])
					reminder_str = _('No reminder')
					status_str = _('Never set')

					# First check for active reminders
					if todo_id in notification_times:
						reminder_time = datetime.datetime.fromisoformat(
							notification_times[todo_id])
						reminder_qt = qt_from_dt(reminder_time)
						reminder_str = reminder_qt.toString(
							self.timestamp_format)
						status_str = _('Active')
					# Then check for historical reminders
					elif todo_id in notification_history:
						reminder_time = datetime.datetime.fromisoformat(
							notification_history[todo_id])
						reminder_qt = qt_from_dt(reminder_time)
						reminder_str = reminder_qt.toString(
							self.timestamp_format)

						# Get the status for better display
						status = reminder_status.get(
							todo_id, REMINDER_STATUS['TRIGGERED'])
						if status == REMINDER_STATUS['TRIGGERED']:
							status_str = _('Triggered')
						elif status == REMINDER_STATUS['CLEARED']:
							status_str = _('Manually cleared')
						else:
							status_str = _('Historical')

					writer.writerow([
						todo['title'],
						created_str,
						reminder_str,
						status_str,
						todo.get('notes', ''),
						_('Yes') if todo.get('completed', False) else _('No')
					])
					# Add separator and notebook entries
				writer.writerow([])  # Add blank row for separation
				writer.writerow([])  # Add another blank row
				writer.writerow([_('=== NOTEBOOK ENTRIES ===')])
				writer.writerow([])  # Add blank row after section header
				# Removed Notes column, renamed Value to Note
				writer.writerow([_('Timestamp'), _('Category'), _('Note')])

				logs = plugin_prefs.get('life_logs', [])
				for log in logs:
					# Format timestamp using Qt
					timestamp = datetime.datetime.fromisoformat(
						log['timestamp'])
					timestamp_qt = qt_from_dt(timestamp)
					time_str = timestamp_qt.toString(self.timestamp_format)

					writer.writerow([
						time_str,
						log['category'],
						log['value']  # Just export the main note text
					])

				info_dialog(self, _('Export successful'),
							_('Data has been exported to %s') % filename,
							show=True)

		except Exception as e:
			error_dialog(self, _('Export error'),
						 _('Failed to export to CSV: %s') % str(e),
						 show=True,
						 det_msg=traceback.format_exc())

	def export_all_to_rtf(self):
		try:
			from calibre.gui2.qt_file_dialogs import choose_save_file
			# Get current timestamp for filename
			timestamp = datetime.datetime.now().strftime('_%Y-%m-%d, %H-%M-%S')
			filename = choose_save_file(
				self, 'export-all-rtf',
				_('Export All Data'),
				filters=[(_('RTF files'), ['rtf'])],
				all_files=False, initial_filename=f'reminders_and_notebook{timestamp}.rtf'
			)

			if not filename:
				return

			# Helper function to encode non-ASCII characters
			def encode_rtf(text):
				result = []
				for char in text:
					if ord(char) < 128:
						result.append(char)
					else:
						result.append(f'\\u{ord(char)}?')
				return ''.join(result)

			# Create RTF content
			rtf_content = []
			rtf_content.append('{\\rtf1\\ansi\\ansicpg1252\\deff0\n')
			rtf_content.append('{\\fonttbl\\f0\\fnil\\fcharset0 Arial;}\n')
			rtf_content.append('\\f0\\fs24\n')
			rtf_content.append(
				'\\b\\fs32 Reminders and Notebook\\b0\\fs24\\par\\par\n')

			# Reminders section
			rtf_content.append('\\b\\fs28 Reminders:\\b0\\fs24\\par\\par\n')
			todos = plugin_prefs.get('todos', [])
			notification_times = plugin_prefs.get('notification_times', {})
			notification_history = plugin_prefs.get('notification_history', {})
			reminder_status = plugin_prefs.get('reminder_status', {})

			for todo in todos:
				# Title with completion status
				completed = todo.get('completed', False)
				title_format = '\\strike ' if completed else ''
				rtf_content.append(
					f'{title_format}{encode_rtf(todo["title"])}\\strike0\\par\n')

				# Created date - using Qt formatting
				created = datetime.datetime.fromisoformat(todo['created'])
				created_qt = qt_from_dt(created)
				created_str = created_qt.toString(self.timestamp_format)
				rtf_content.append(
					f'Created: {encode_rtf(created_str)}\\par\n')

				# Get reminder details
				todo_id = str(todo['id'])

				# First check for active reminders
				if todo_id in notification_times:
					reminder_time = datetime.datetime.fromisoformat(
						notification_times[todo_id])
					reminder_qt = qt_from_dt(reminder_time)
					reminder_str = reminder_qt.toString(self.timestamp_format)
					rtf_content.append(
						f'Reminder: {encode_rtf(reminder_str)} (Active)\\par\n')
				# Then check for historical reminders
				elif todo_id in notification_history:
					reminder_time = datetime.datetime.fromisoformat(
						notification_history[todo_id])
					reminder_qt = qt_from_dt(reminder_time)
					reminder_str = reminder_qt.toString(self.timestamp_format)

					# Get the status for better display
					status = reminder_status.get(
						todo_id, REMINDER_STATUS['TRIGGERED'])
					if status == REMINDER_STATUS['TRIGGERED']:
						rtf_content.append(
							f'Reminder: {encode_rtf(reminder_str)} (Triggered)\\par\n')
					elif status == REMINDER_STATUS['CLEARED']:
						rtf_content.append(
							f'Reminder: {encode_rtf(reminder_str)} (Manually cleared)\\par\n')
					else:
						rtf_content.append(
							f'Reminder: {encode_rtf(reminder_str)} (Historical)\\par\n')
				else:
					rtf_content.append(_('No reminder set') + '\\par\n')

				# Notes
				if todo.get('notes'):
					rtf_content.append(
						f'Notes: {encode_rtf(todo["notes"])}\\par\n')

				rtf_content.append('\\par\n')

			# Notebook section
			rtf_content.append(
				'\\b\\fs28 Notebook Entries:\\b0\\fs24\\par\\par\n')
			logs = plugin_prefs.get('life_logs', [])
			for log in logs:
				# Timestamp - using Qt formatting
				timestamp = datetime.datetime.fromisoformat(log['timestamp'])
				timestamp_qt = qt_from_dt(timestamp)
				time_str = timestamp_qt.toString(self.timestamp_format)
				rtf_content.append(f'Time: {encode_rtf(time_str)}\\par\n')
				rtf_content.append(f'Category: {encode_rtf(log["category"])}\\par\n')
				rtf_content.append(f'Note: {encode_rtf(log["value"])}\\par\n')
				if log.get('notes'):
					rtf_content.append(
						f'Notes: {encode_rtf(log["notes"])}\\par\n')
				rtf_content.append('\\par\n')
			
			rtf_content.append('}')

			# Write to file in binary mode with proper encoding
			with open(filename, 'wb') as f:
				f.write(''.join(rtf_content).encode('utf-8', 'replace'))
			from calibre.gui2 import info_dialog
			info_dialog(self, _('Export successful'),
						_('Data has been exported to %s') % filename,
						show=True)

		except Exception as e:
			print(f'TodoDialog: Error exporting to RTF: {str(e)}')
			print(traceback.format_exc())
			
	def export_all_to_ics(self):
		try:
			from calibre.gui2.qt_file_dialogs import choose_save_file
			from calibre.gui2 import error_dialog, info_dialog
			import sys
			import os
			import datetime

			# Add plugin's directory to path temporarily to find icalendar
			plugin_dir = os.path.dirname(os.path.realpath(__file__))
			sys.path.insert(0, plugin_dir)
			import icalendar
			sys.path.remove(plugin_dir)  # Remove from path after import

			# Get current timestamp for filename
			timestamp = datetime.datetime.now().strftime('_%Y-%m-%d, %H-%M-%S')
			filename = choose_save_file(
				self, 'export-all-ics',
				_('Export All Data'),
				filters=[(_('ICS files'), ['ics'])],
				all_files=False,
				initial_filename=f'reminders{timestamp}.ics'
			)

			if not filename:
				return

			# Create calendar
			cal = icalendar.Calendar()
			cal.add('prodid', '-//Comfy Reminders Plugin//calibre//EN')
			cal.add('version', '2.0')

			todos = plugin_prefs.get('todos', [])
			notification_times = plugin_prefs.get('notification_times', {})

			# Add todos as calendar events
			for todo in todos:
				event = icalendar.Event()

				# Basic event properties
				event.add('summary', todo['title'])
				event.add('description', todo.get('notes', ''))

				# Created timestamp
				created = todo.get('created')
				if created:
					dt_created = datetime.datetime.fromisoformat(created)
					event.add('created', dt_created)

				# Set reminder time if available
				todo_id = todo['id']
				if todo_id in notification_times:
					start_time = datetime.datetime.fromisoformat(
						notification_times[todo_id])
					event.add('dtstart', start_time)
					event.add(
						'duration', datetime.timedelta(
							hours=1))  # Default 1 hour duration

					# Add alarm
					alarm = icalendar.Alarm()
					alarm.add('action', 'DISPLAY')
					alarm.add(
						'trigger', datetime.timedelta(
							minutes=0))  # Trigger at event time
					alarm.add('description', todo['title'])
					event.add_component(alarm)

				# Status
				if todo.get('completed'):
					event.add('status', 'COMPLETED')
				else:
					event.add('status', 'CONFIRMED')

				cal.add_component(event)

			# Write to file
			with open(filename, 'wb') as f:
				f.write(cal.to_ical())

			info_dialog(self, _('Export successful'),
						_('Data has been exported successfully.'),
						show=True)

		except Exception as e:
			error_dialog(self, _('Export error'),
						 _('Failed to export to ICS: %s') % str(e),
						 show=True,
						 det_msg=traceback.format_exc())
		 
	def import_ics(self):
		"""Import reminders from an ICS file"""
		try:
			from calibre.gui2.qt_file_dialogs import choose_files
			from calibre.gui2 import error_dialog, info_dialog, question_dialog
			import sys, os

			# Add plugin's directory to path temporarily to find icalendar
			plugin_dir = os.path.dirname(os.path.realpath(__file__))
			try:
				sys.path.insert(0, plugin_dir)
				import icalendar
				from datetime import datetime, timedelta
				sys.path.pop(0)  # Remove temporary path
			except ImportError as e:
				error_dialog(self, _('Missing dependency'),
					_('Could not import icalendar: %s\nPlugin directory: %s') % (str(e), plugin_dir),
					show=True)
				return

			files = choose_files(self, 'import-ics',
				_('Import Calendar'),
				filters=[(_('ICS files'), ['ics'])],
				all_files=False,
				select_only_single_file=True)

			if not files:
				return

			# Create backup before import
			self.create_backup()

			with open(files[0], 'rb') as f:
				cal = icalendar.Calendar.from_ical(f.read())			# Ask if user wants to import all events or just those with reminders
			import_all = question_dialog(self, _('Import Options'),
				_('Import all calendar events? Selecting No will import only events with reminders.'),
				show_copy_button=False)
				
			current_time = now()
			imported_count = 0
			skipped_count = 0
			
			# Get existing todos for duplicate checking
			existing_todos = plugin_prefs.get('todos', [])
			notification_times = plugin_prefs.get('notification_times', {})
			
			# Helper function to check for duplicates
			def is_duplicate(summary, start_time, created_time):
				for todo in existing_todos:
					# Check title match
					if todo['title'] == summary:
						todo_id = str(todo['id'])
						# Check reminder time match if it exists
						if todo_id in notification_times:
							todo_time = datetime.datetime.fromisoformat(notification_times[todo_id])
							if abs((todo_time - start_time).total_seconds()) < 60:  # Within 1 minute
								return True
						# If no reminder time, check creation time
						todo_created = datetime.datetime.fromisoformat(todo['created'])
						if abs((todo_created - created_time).total_seconds()) < 60:  # Within 1 minute
							return True
				return False
				
			for component in cal.walk():
				if component.name == "VEVENT":
					summary = str(component.get('summary', ''))
					if not summary:
						continue

					# Get event start time
					start = component.get('dtstart')
					if start:
						start = start.dt
						if not isinstance(start, datetime):
							# Convert date to datetime
							start = datetime.combine(start, datetime.min.time())
					else:
						continue

					# Check if event has an alarm/reminder
					has_reminder = False
					for alarm in component.walk('VALARM'):
						has_reminder = True
						break					# Skip if no reminder and not importing all
					if not has_reminder and not import_all:
						continue

					# Check for duplicates
					created = component.get('created')
					if created:
						created_time = created.dt
					else:
						created_time = start

					# Skip if duplicate found
					if next((todo for todo in plugin_prefs.get('todos', []) 
							if todo['title'] == summary
							and abs((datetime.fromisoformat(todo['created']) - created_time).total_seconds()) < 60), None):
						skipped_count += 1
						continue

					# Create reminder
					todo_id = str(uuid.uuid4())
					description = str(component.get('description', ''))

					todo = {
						'id': todo_id,
						'title': summary,
						'created': current_time.isoformat(),
						'notes': description
					}

					if component.get('status') == 'COMPLETED':
						todo['completed'] = True

					# Add to reminders
					todos = plugin_prefs.get('todos', [])
					todos.append(todo)
					plugin_prefs['todos'] = todos

					# Set reminder time
					notification_times = plugin_prefs.get('notification_times', {})
					notification_history = plugin_prefs.get('notification_history', {})
					reminder_status = plugin_prefs.get('reminder_status', {})

					if start > current_time:
						notification_times[todo_id] = start.isoformat()
						reminder_status[todo_id] = REMINDER_STATUS['ACTIVE']
					else:
						notification_history[todo_id] = start.isoformat()
						reminder_status[todo_id] = REMINDER_STATUS['TRIGGERED']

					plugin_prefs['notification_times'] = notification_times
					plugin_prefs['notification_history'] = notification_history
					plugin_prefs['reminder_status'] = reminder_status

					imported_count += 1			# Refresh display
			self.load_todos()

			info_dialog(self, _('Import Complete'),
				_('Successfully imported %d events.\nSkipped %d duplicate events.') % (imported_count, skipped_count),
				show=True)

		except Exception as e:
			error_dialog(self, _('Import Error'),
				_('Failed to import ICS file: %s') % str(e),
				show=True,
				det_msg=traceback.format_exc())

	def select_all_todos(self):
		"""Select all rows in the todo table"""
		self.todo_table.selectAll()

	def clear_todo_selection(self):
		"""Clear the current selection in the todo table"""
		self.todo_table.clearSelection()

	def get_selected_rows(self):
		"""Get unique selected row indices"""
		return sorted(set(item.row()
					  for item in self.todo_table.selectedItems()))

	def filter_todos(self, text):
		"""Filter reminders based on the text in the filter box"""
		try:
			filter_text = text.lower()

			# Get all todos
			todos = plugin_prefs.get('todos', [])

			# If filter is empty, show all rows
			if not filter_text:
				for row in range(len(todos)):
					self.todo_table.showRow(row)
				return

			# Otherwise, hide/show rows based on the filter text
			for row in range(len(todos)):
				todo = todos[row]
				# Show row if title contains the filter text
				if filter_text in todo['title'].lower():
					self.todo_table.showRow(row)
				else:
					self.todo_table.hideRow(row)

		except Exception as e:
			print(f"Error in filter_todos: {str(e)}")
			print(traceback.format_exc())

	def create_backup(self):
		"""Create a backup of the current data"""
		try:
			# Create backup directory if it doesn't exist
			backup_dir = os.path.join(
				config_dir, 'plugins', 'Comfy Reminders', 'backups')
			if not os.path.exists(backup_dir):
				os.makedirs(backup_dir)

			# Create backup filename with timestamp
			timestamp = datetime.datetime.now().strftime('%Y%m%d_%H%M%S')
			backup_file = os.path.join(
				backup_dir, f'reminders_backup_{timestamp}.json')

			# Save current data to backup file
			backup_data = {
				'todos': plugin_prefs.get(
					'todos', []), 'life_logs': plugin_prefs.get(
					'life_logs', []), 'notification_times': plugin_prefs.get(
					'notification_times', {}), 'notification_history': plugin_prefs.get(
					'notification_history', {}), 'reminder_status': plugin_prefs.get(
						'reminder_status', {})}

			with open(backup_file, 'w', encoding='utf-8') as f:
				json.dump(backup_data, f, indent=4, ensure_ascii=False)

			# Keep only the last 5 backups
			backup_files = sorted([f for f in os.listdir(
				backup_dir) if f.startswith('reminders_backup_')])
			if len(backup_files) > 5:
				for old_file in backup_files[:-5]:
					os.remove(os.path.join(backup_dir, old_file))

			return backup_file
		except Exception as err:
			print(f'Failed to create backup: {str(err)}')
			return None

	def export_json(self):
		"""Export the reminders and logs to a JSON file"""
		try:
			from calibre.gui2.qt_file_dialogs import choose_save_file			# Get current timestamp for filename
			timestamp = datetime.datetime.now().strftime('_%Y-%m-%d, %H-%M-%S')
			fname = choose_save_file(self, 'export-reminders',
									 _('Export Reminders and Notes'),
									 filters=[(_('Reminders Data'), ['json'])],
									 all_files=False,
									 initial_filename=f'reminders_export{timestamp}.json')

			if not fname:
				return

			# Prepare export data
			export_data = {
				'todos': plugin_prefs.get(
					'todos', []), 'life_logs': plugin_prefs.get(
					'life_logs', []), 'notification_times': plugin_prefs.get(
					'notification_times', {}), 'notification_history': plugin_prefs.get(
					'notification_history', {}), 'reminder_status': plugin_prefs.get(
						'reminder_status', {})}

			with open(fname, 'w', encoding='utf-8') as f:
				json.dump(export_data, f, indent=4)

			from calibre.gui2 import info_dialog
			info_dialog(self, _('Export Complete'),
						_('Data has been exported successfully.'),
						show=True)

		except Exception as err:
			from calibre.gui2 import error_dialog
			error_dialog(self, _('Error exporting data'),
						 str(err), show=True)

	def import_json(self):
		error_dialog(self, _('Error exporting data'),
					 str(err), show=True)

	def import_json(self):
		"""Import reminders and logs from a JSON file"""
		try:
			from calibre.gui2.qt_file_dialogs import choose_files

			files = choose_files(self, 'import-reminders',
								 _('Import Reminders and Notes'),
								 filters=[(_('Reminders Data'), ['json'])],
								 all_files=False,
								 select_only_single_file=True)

			if not files:
				return

			# First check if file is empty
			if os.path.getsize(files[0]) == 0:
				from calibre.gui2 import error_dialog
				error_dialog(self, _('Error importing data'),
							 _('The selected file is empty'), show=True)
				return

			with open(files[0], 'r', encoding='utf-8') as f:
				import_data = json.load(f)

			# Validate structure
			required_keys = ['todos', 'life_logs']
			if not all(key in import_data for key in required_keys):
				raise ValueError(_('Invalid data structure in import file'))

			# Create backup before replacing data
			self.create_backup()

			from calibre.gui2 import question_dialog
			# Update preferences with imported data
			plugin_prefs['todos'] = import_data['todos']
			plugin_prefs['life_logs'] = import_data['life_logs']
			plugin_prefs['notification_times'] = import_data.get(
				'notification_times', {})
			plugin_prefs['notification_history'] = import_data.get(
				'notification_history', {})
			plugin_prefs['reminder_status'] = import_data.get(
				'reminder_status', {})

			# Refresh displays
			self.load_todos()
			self.load_logs()

			from calibre.gui2 import info_dialog
			info_dialog(self, _('Import Complete'),
						_('Data imported successfully.'),
						show=True)

		except json.JSONDecodeError as err:
			from calibre.gui2 import error_dialog
			error_dialog(self, _('Error importing data'),
						 _('Invalid JSON file: %s') % str(err), show=True)
		except ValueError as err:
			from calibre.gui2 import error_dialog
			error_dialog(self, _('Error importing data'),
						 str(err), show=True)
		except Exception as err:
			from calibre.gui2 import error_dialog
			error_dialog(self, _('Error importing data'),
						 str(err), show=True)

	def restore_backup(self):
		"""Restore data from a selected backup"""
		try:
			# Find backup directory
			backup_dir = os.path.join(
				config_dir, 'plugins', 'Comfy Reminders', 'backups')
			if not os.path.exists(backup_dir):
				from calibre.gui2 import error_dialog
				error_dialog(self, _('No Backups'),
							 _('No backup files were found.'),
							 show=True)
				return

			# Show backup selection dialog
			dialog = BackupDialog(self)
			if dialog.exec() != QDialog.DialogCode.Accepted:
				return

			selected_backup = dialog.get_selected_backup()
			if not selected_backup:
				return

			backup_path = os.path.join(backup_dir, selected_backup)

			# Load and validate backup
			with open(backup_path, 'r', encoding='utf-8') as f:
				backup_data = json.load(f)

			# Validate structure
			required_keys = ['todos', 'life_logs']
			if not all(key in backup_data for key in required_keys):
				from calibre.gui2 import error_dialog
				error_dialog(self, _('Invalid Backup'),
							 _('The backup file appears to be corrupted.'),
							 show=True)
				return

			# Update preferences
			plugin_prefs['todos'] = backup_data['todos']
			plugin_prefs['life_logs'] = backup_data['life_logs']
			plugin_prefs['notification_times'] = backup_data.get(
				'notification_times', {})
			plugin_prefs['notification_history'] = backup_data.get(
				'notification_history', {})
			plugin_prefs['reminder_status'] = backup_data.get(
				'reminder_status', {})

			# Refresh displays
			self.load_todos()
			self.load_logs()

			from calibre.gui2 import info_dialog
			info_dialog(
				self,
				_('Restore Complete'),
				_('Data restored successfully from backup:\n%s') %
				selected_backup,
				show=True)

		except Exception as err:
			from calibre.gui2 import error_dialog
			error_dialog(self, _('Error Restoring Backup'),
						 str(err), show=True)

	def on_log_selection_changed(self):
		"""Handle log table selection changes to update detail panel"""
		selected_items = self.log_table.selectedItems()
		if not selected_items:
			self.detail_text.clear()
			self.detail_timestamp_label.setText("")
			self.detail_category_label.setText("")
			return

		# Get the row of the first selected item
		row = selected_items[0].row()

		# Get the log entry for this row
		logs = plugin_prefs.get('life_logs', [])
		if 0 <= row < len(logs):
			log = logs[row]

			# Format timestamp for display
			timestamp = datetime.datetime.fromisoformat(log['timestamp'])
			timestamp_qt = qt_from_dt(timestamp)
			time_str = timestamp_qt.toString(self.timestamp_format)

			# Update the read-only metadata labels
			self.detail_timestamp_label.setText(_("Time: %s") % time_str)
			self.detail_category_label.setText(
				_("Category: %s") %
				log['category'])

			# Set just the note content in the editable text area
			self.detail_text.setPlainText(log['value'])

	def save_entry_details(self):
		"""Save changes made in the detail panel to the selected log entry"""
		selected_items = self.log_table.selectedItems()
		if not selected_items:
			return

		# Get the row of the first selected item
		row = selected_items[0].row()

		# Get the log entry for this row
		logs = plugin_prefs.get('life_logs', [])
		if 0 <= row < len(logs):
			log = logs[row]

			# Simply set the value directly from the text edit
			log['value'] = self.detail_text.toPlainText().strip()

			# Save updated log entry
			plugin_prefs['life_logs'] = logs

			# Refresh display
			self.load_logs()

			# Reselect the row
			self.log_table.selectRow(row)

			# Refresh the detail panel with updated content
			self.on_log_selection_changed()

	def on_log_cell_changed(self, row, column):
		"""Handle changes to log table cells"""
		try:
			# We only want to handle changes to the Note column (column 2)
			if column != 2:
				return

			# Get the updated value from the table cell
			new_value = self.log_table.item(row, column).text().strip()

			# Get the log entry and update it
			logs = plugin_prefs.get('life_logs', [])
			if 0 <= row < len(logs):
				log = logs[row]
				log['value'] = new_value
				plugin_prefs['life_logs'] = logs

				# Update detail panel if this entry is currently selected
				selected_items = self.log_table.selectedItems()
				if selected_items and selected_items[0].row() == row:
					self.on_log_selection_changed()
		except Exception as e:
			print(f"Error in on_log_cell_changed: {str(e)}")
			print(traceback.format_exc())

	def on_todo_cell_changed(self, row, column):
		"""Handle changes to todo table cells"""
		try:
			# We only want to handle changes to the Title column (column 0)
			if column != 0:
				return

			# Get the updated value from the table cell
			new_title = self.todo_table.item(row, column).text().strip()

			# Don't allow empty titles
			if not new_title:
				todos = plugin_prefs.get('todos', [])
				if 0 <= row < len(todos):
					# Restore original title
					original_title = todos[row]['title']
					self.todo_table.item(row, column).setText(original_title)
					return

			# Get the todo item and update it
			todos = plugin_prefs.get('todos', [])
			if 0 <= row < len(todos):
				todo = todos[row]
				todo['title'] = new_title
				plugin_prefs['todos'] = todos
		except Exception as e:
			print(f"Error in on_todo_cell_changed: {str(e)}")
			print(traceback.format_exc())

	def migrate_log_entries(self):
		"""Migrate log entries to new structure"""
		try:
			modified = False
			logs = plugin_prefs.get('life_logs', [])

			for log in logs:
				# If entry has notes field, append it to value and remove notes
				# field
				if 'notes' in log:
					if log['notes']:  # If there was content in notes
						log['value'] = log['value'] + '\n\n' + log['notes']
						del log['notes']
						modified = True

			if modified:
				plugin_prefs['life_logs'] = logs

		except Exception as e:
			print(f"Error in migrate_log_entries: {str(e)}")
			print(traceback.format_exc())

	def create_reminder(self, title, delay_minutes, book_id=None):
		"""Create a reminder with the specified title and delay time
		Returns the reminder ID of the created reminder.
		If book_id is provided, it will be stored with the reminder for duplicate detection."""
		try:
			reminder_id = str(uuid.uuid4())
			created_time = now().astimezone(datetime.timezone.utc)
			notification_time = created_time + \
				datetime.timedelta(minutes=delay_minutes)

			# Create todo entry
			todo = {
				"id": reminder_id,
				"title": title,
				"created": created_time.isoformat()
			}

			# Store book_id if provided
			if book_id is not None:
				todo["book_id"] = book_id
				# Save todo at the beginning of the list
			todos = plugin_prefs.get('todos', [])
			todos.insert(0, todo)  # Insert at beginning instead of appending
			plugin_prefs['todos'] = todos

			# Save reminder time
			notification_times = plugin_prefs.get('notification_times', {})
			notification_times[reminder_id] = notification_time.isoformat()
			plugin_prefs['notification_times'] = notification_times

			# Set reminder status as active
			reminder_status = plugin_prefs.get('reminder_status', {})
			reminder_status[reminder_id] = REMINDER_STATUS['ACTIVE']
			plugin_prefs['reminder_status'] = reminder_status

			return reminder_id
		except Exception as e:
			print(f"Error in create_reminder: {str(e)}")
			print(traceback.format_exc())
			return None

	def has_reminder_for_book(self, book_id):
		try:
			todos = plugin_prefs.get('todos', [])
			for todo in todos:
				if todo.get('book_id') == book_id:
					return True, todo.get('title', _('Unknown title'))
			return False, None
		except Exception as e:
			print(f"Error in has_reminder_for_book: {str(e)}")
			print(traceback.format_exc())
			return False, None

	def export_dialog_state(self):
		"""Export the exact state of reminders as currently shown in the dialog"""
		try:
			from calibre.gui2.qt_file_dialogs import choose_save_file

			fname = choose_save_file(
				self,
				'export-dialog-state',
				_('Export Current Dialog State'),
				filters=[
					(_('Dialog State'),
					 ['json'])],
				initial_filename='reminders_dialog_state.json')

			if not fname:
				return

			# Capture current state as shown in the dialog
			dialog_state = {
				"todos": [],
				"reminder_display_info": {}
			}

			for row in range(self.todo_table.rowCount()):
				todo_id = plugin_prefs['todos'][row]['id']
				title = self.todo_table.item(row, 0).text()
				reminder_text = self.todo_table.item(row, 1).text()
				notes = plugin_prefs['todos'][row].get('notes', '')
				created = self.todo_table.item(row, 3).text()
				completed = plugin_prefs['todos'][row].get('completed', False)

				# Store the todo info
				todo_info = {
					"id": todo_id,
					"title": title,
					"reminder_display": reminder_text,  # Exact text shown in dialog
					"notes": notes,
					"created": created,
					"completed": completed
				}
				dialog_state["todos"].append(todo_info)

				# Store additional reminder state info
				reminder_info = {
					"notification_times": plugin_prefs.get(
						'notification_times', {}).get(todo_id), "notification_history": plugin_prefs.get(
						'notification_history', {}).get(todo_id), "reminder_status": plugin_prefs.get(
						'reminder_status', {}).get(todo_id)}
				dialog_state["reminder_display_info"][todo_id] = reminder_info

			# Add timestamp of export
			dialog_state["export_time"] = now().astimezone(
				datetime.timezone.utc).isoformat()
			dialog_state["timestamp_format"] = self.timestamp_format

			with open(fname, 'w', encoding='utf-8') as f:
				json.dump(dialog_state, f, indent=4, ensure_ascii=False)

			from calibre.gui2 import info_dialog
			info_dialog(self, _('Export Complete'),
						_('Dialog state has been exported successfully.'),
						show=True)

		except Exception as e:
			from calibre.gui2 import error_dialog
			error_dialog(self, _('Export Error'),
						 _('Failed to export dialog state: %s') % str(e),
						 show=True,
						 det_msg=traceback.format_exc())

	def show_config_dialog(self):
		"""Show the configuration dialog"""
		d = ConfigDialog(self.gui, self)
		d.exec()
