#!/usr/bin/env python

from calibre import prints
from calibre.utils.localization import get_lang

try:
    from qt.core import (QDialog, QVBoxLayout, QLabel, QDialogButtonBox,
                        QGroupBox, QGridLayout, QComboBox)
except ImportError:
    from PyQt5.Qt import (QDialog, QVBoxLayout, QLabel, QDialogButtonBox,
                         QGroupBox, QGridLayout, QComboBox)

# Load translations after importing calibre utilities
try:
    prints(_("Comfy Reminders::config.py - loading translations"))
    load_translations()
except Exception as e:
    prints(_("Comfy Reminders::config.py - exception when loading translations: {}").format(str(e)))

from calibre.gui2 import gprefs
from calibre.utils.config import JSONConfig
try:
    from calibre.utils.localization import get_lang
except ImportError:
    prints(_("Comfy Reminders::config.py - exception when importing localization functions"))

# Store plugin preferences 
plugin_prefs = JSONConfig('plugins/comfy_reminders')

# Store per-library settings namespace
PREFS_NAMESPACE = 'ComfyRemindersPlugin'
PREFS_KEY_SETTINGS = 'settings'

# Constants for settings
KEY_RELEASE_DATE_COLUMN = 'release_date_column'

# Set defaults for all preferences
plugin_prefs.defaults['todos'] = []
plugin_prefs.defaults['notification_times'] = {}
plugin_prefs.defaults['notification_history'] = {}
plugin_prefs.defaults['reminder_status'] = {}
plugin_prefs.defaults['life_logs'] = []
plugin_prefs.defaults['categories'] = [
    _('Activity'), 
    _('Notable Quotes'),
    _('Weight'), 
    _('Observation'), 
    _('Health'), 
    _('Mood'), 
    _('Custom')
]
plugin_prefs.defaults['timestamp_format'] = 'dd/MM/yyyy HH:mm'

DEFAULT_LIBRARY_VALUES = {
    KEY_RELEASE_DATE_COLUMN: ''
}

def get_library_config(db):
    """Get the library specific preferences"""
    library_config = db.prefs.get_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS,
                                         DEFAULT_LIBRARY_VALUES.copy())
    return library_config

def set_library_config(db, library_config):
    """Set the library specific preferences"""
    db.prefs.set_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS, library_config)

class CustomColumnComboBox(QComboBox):
    def __init__(self, parent, custom_columns={}, selected_column='', initial_items=['']):
        QComboBox.__init__(self, parent)
        self.populate_combo(custom_columns, selected_column, initial_items)

    def populate_combo(self, custom_columns, selected_column, initial_items=['']):
        self.clear()
        self.column_names = list(initial_items)
        if len(initial_items) > 0:
            self.addItems(initial_items)
        selected_idx = 0
        for idx, value in enumerate(initial_items):
            if value == selected_column:
                selected_idx = idx

        for key in sorted(custom_columns.keys()):
            self.column_names.append(key)
            self.addItem(_('%s (%s)') % (key, custom_columns[key]['name']))
            if key == selected_column:
                selected_idx = len(self.column_names) - 1
        self.setCurrentIndex(selected_idx)

    def get_selected_column(self):
        return self.column_names[self.currentIndex()]

class ConfigDialog(QDialog):
    def __init__(self, gui, parent=None):
        QDialog.__init__(self, parent)
        self.gui = gui
        self.db = gui.current_db
        self.setup_ui()
        self.restore_geometry()
        prints(_("Config dialog initialized"))

    def setup_ui(self):
        self.setWindowTitle(_('Comfy Reminders Configuration'))
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        # Columns group
        columns_group = QGroupBox(_('Date Column Settings'), self)
        layout.addWidget(columns_group)
        columns_layout = QGridLayout()
        columns_group.setLayout(columns_layout)

        # Release date column
        date_label = QLabel(_('Your date column:'), self)
        date_label.setToolTip(_('Select the column containing dates.\nThe column must be of type "date".'))
        columns_layout.addWidget(date_label, 0, 0)

        # Get date columns, excluding timestamp and last_modified
        date_columns = {k:v for k,v in self.get_date_columns().items() 
                       if k not in ['timestamp', 'last_modified']}
        
        library_config = get_library_config(self.db)
        self.date_column_combo = CustomColumnComboBox(self, date_columns, library_config[KEY_RELEASE_DATE_COLUMN])
        self.date_column_combo.setToolTip(date_label.toolTip())
        self.date_column_combo.setMinimumWidth(200) # Ensure enough width for longer translations
        columns_layout.addWidget(self.date_column_combo, 0, 1)

        # Dialog buttons
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        # Set minimum size
        self.resize(self.sizeHint())

    def get_date_columns(self):
        """Get both built-in and custom date columns"""
        date_columns = {}
        
        # Add built-in date columns
        field_metadata = self.gui.current_db.field_metadata
        for key, column in field_metadata.items():
            if isinstance(column, dict) and column.get('datatype') == 'datetime':
                if not key.startswith('#'):  # Not a custom column
                    date_columns[key] = {
                        'name': column.get('name', key),
                        'datatype': 'datetime',
                        'display': column.get('display', {})
                    }
                    prints(_("Added built-in datetime column: %s") % key)

        # Add custom date columns
        for key, column in field_metadata.custom_field_metadata().items():
            prints(_("Found custom column: %s, type: %s") % (key, column['datatype']))
            if column['datatype'] == 'datetime':
                date_columns[key] = column
                prints(_("Added custom datetime column: %s") % key)
                
        return date_columns

    def accept(self):
        # Save the current values
        selected = self.date_column_combo.get_selected_column()
        prints(_("Saving selected column: %s") % selected)
        library_config = get_library_config(self.gui.current_db)
        library_config[KEY_RELEASE_DATE_COLUMN] = selected
        set_library_config(self.gui.current_db, library_config)
        prints(_("Updated library preferences: %s") % library_config)
        self.save_geometry()
        return QDialog.accept(self)

    def reject(self):
        self.save_geometry()
        return QDialog.reject(self)

    def restore_geometry(self):
        geom = gprefs.get('comfy_reminders_config_dialog_geometry', None)
        if geom:
            self.restoreGeometry(geom)

    def save_geometry(self):
        gprefs['comfy_reminders_config_dialog_geometry'] = self.saveGeometry()