#!/usr/bin/env python

from calibre import prints

# Load translations first, before any other imports
try:
    prints("Comfy Reminders::action.py - loading translations")
    load_translations()
except NameError:
    prints("Comfy Reminders::action.py - exception when loading translations")
    pass

try:
    from qt.core import QAction, QIcon, QMenu, QTimer, QApplication
except ImportError:
    from PyQt5.Qt import QAction, QIcon, QMenu, QTimer, QApplication
    
from calibre.gui2.actions import InterfaceAction
from calibre_plugins.comfy_reminders.ui import TodoDialog
from calibre_plugins.comfy_reminders import plugin_prefs
from calibre.gui2 import Application, error_dialog, question_dialog, info_dialog
from calibre.utils.date import now, UNDEFINED_DATE
from calibre.gui2.notify import get_notifier
from calibre.utils.localization import get_lang
from calibre_plugins.comfy_reminders.config import KEY_RELEASE_DATE_COLUMN, get_library_config, set_library_config
from . import common_icons
import datetime
import traceback

# Reminder states
REMINDER_STATUS = {
    'NEVER_SET': 'never_set',
    'ACTIVE': 'active',
    'TRIGGERED': 'triggered',
    'CLEARED': 'cleared'
}

class TodoListAction(InterfaceAction):

    name = 'Comfy Reminders'
    action_spec = ('Comfy Reminders', None, _('Manage reminders'), 'Ctrl+Shift+T')
    _reminder_dialog = None  # Store dialog instance

    def genesis(self):
        try:
            # Initialize the icon using calibre's icon system
            common_icons.set_plugin_icon_resources(self.name, {})
            self.rebuild_icon()
              # Create menu
            self.menu = QMenu(self.gui)
            
            # Book reminders action - now first
            self.create_reminders_action = QAction(_("Create reminders from selected books"), self.gui)
            self.create_reminders_action.triggered.connect(self.create_reminders_from_books)
            self.menu.addAction(self.create_reminders_action)
            
            self.menu.addSeparator()
            
            # Main menu items
            self.create_menu_action(self.menu, 'show_dialog', _('Manage Reminders && Notes'),
                                icon=None, triggered=self.show_todo_dialog)
            self.menu.addSeparator()
            
            # Add quick reminder items
            self.create_menu_action(self.menu, 'add_5min', _('Add 5 minute reminder'),
                                icon=None, triggered=self.add_5min_reminder)
            self.create_menu_action(self.menu, 'add_30min', _('Add 30 minute reminder'),
                                icon=None, triggered=self.add_30min_reminder)
            self.create_menu_action(self.menu, 'add_1hour', _('Add 1 hour reminder'),
                                icon=None, triggered=self.add_1hour_reminder)
            self.create_menu_action(self.menu, 'add_1day', _('Add 1 day reminder'),
                                icon=None, triggered=self.add_1day_reminder)
            
            # Set menu for main button
            self.qaction.setMenu(self.menu)
            
            # Connect main button click directly
            self.qaction.triggered.connect(self.show_todo_dialog)
            
            # Start the notification timer
            self.notification_timer = QTimer(self.gui)
            self.notification_timer.timeout.connect(self.check_notifications)
            self.notification_timer.start(60000)  # Check every minute
            
            # Initialize notifier
            self.notifier = get_notifier()
            
        except Exception as e:
            raise  # Re-raise to show error in Calibre

        # Set up theme change listener
        app = QApplication.instance()
        if app:
            app.paletteChanged.connect(self.rebuild_icon)

    def rebuild_icon(self):
        """Update the icon based on current theme"""
        icon = common_icons.get_icon('images/iconplugin')  # Base name only - get_icon handles theme
        if icon and not icon.isNull():
            self.qaction.setIcon(icon)

    def add_5min_reminder(self):
        self.quick_add_reminder(_('Quick Reminder (5m)'), 5)
        
    def add_30min_reminder(self):
        self.quick_add_reminder(_('Quick Reminder (30m)'), 30)
    
    def add_1hour_reminder(self):
        self.quick_add_reminder(_('Quick Reminder (1h)'), 60)
        
    def add_1day_reminder(self):
        self.quick_add_reminder(_('Quick Reminder (1d)'), 1440)
    
    def quick_add_reminder(self, title, minutes):
        dialog = self.reminder_dialog()
        if dialog:
            dialog.add_quick_reminder(minutes)

    def show_todo_dialog(self):
        try:
            self._reminder_dialog = TodoDialog(self.gui)
            self._reminder_dialog.exec()
        except Exception as e:
            error_dialog(self.gui, _('Comfy Reminders Error'),
                        _('Error showing dialog: ') + str(e),
                        show=True,
                        det_msg=traceback.format_exc())
                        
    def check_notifications(self):
        try:
            current_time = now()
            notification_times = plugin_prefs.get('notification_times', {})
            notification_history = plugin_prefs.get('notification_history', {})
            reminder_status = plugin_prefs.get('reminder_status', {})
            todos = plugin_prefs.get('todos', [])
            modified = False
            
            for todo in todos:
                todo_id = str(todo.get('id'))
                if todo_id in notification_times:
                    notification_time = datetime.datetime.fromisoformat(notification_times[todo_id])
                    if current_time >= notification_time:
                        if self.notifier:
                            # Check if we're on Windows (has 'ok' attribute) or Linux (doesn't have it)
                            can_notify = hasattr(self.notifier, 'ok') and self.notifier.ok or not hasattr(self.notifier, 'ok')
                            if can_notify:
                                try:
                                    title = _('Reminder')
                                    msg = todo.get('title', '')
                                    self.notifier(msg, title, timeout=5000)
                                    
                                    notification_history[todo_id] = notification_times[todo_id]
                                    reminder_status[todo_id] = REMINDER_STATUS['TRIGGERED']
                                    del notification_times[todo_id]
                                    modified = True
                                except Exception as notify_error:
                                    prints(f"Notification error: {str(notify_error)}")
            
            # Save all state changes atomically if modified
            if modified:
                plugin_prefs['notification_times'] = notification_times
                plugin_prefs['notification_history'] = notification_history
                plugin_prefs['reminder_status'] = reminder_status
                            
        except Exception as e:
            error_dialog(self.gui, _('Comfy Reminders Error'),
                        _('Error checking notifications: ') + str(e),
                        show=True,
                        det_msg=traceback.format_exc())

    def create_reminders_from_books(self):
        """Create reminders based on the configured date column for selected books"""
        try:
            db = self.gui.current_db
            library_config = get_library_config(db)
            date_column = library_config.get(KEY_RELEASE_DATE_COLUMN, '')
            
            if not date_column:
                error_dialog(self.gui, _('No date column configured'),
                           _('Please configure a date column in Settings first.'),
                           show=True)
                return

            # Get selected books
            rows = self.gui.library_view.selectionModel().selectedRows()
            if not rows:
                error_dialog(self.gui, _('No books selected'),
                           _('Please select some books first.'),
                           show=True)
                return

            success_count = 0
            error_count = 0
            duplicate_count = 0

            for row in rows:
                try:
                    book_id = self.gui.library_view.model().id(row)
                    title = db.get_metadata(book_id, index_is_id=True).title
                    
                    # Get date value based on column type
                    date_value = None
                    if date_column.startswith('#'):  # Custom column
                        # Get custom column value
                        date_value = db.get_custom(book_id, label=date_column[1:], index_is_id=True)
                    else:  # Built-in column
                        # Get built-in column value using appropriate method
                        if hasattr(db, 'new_api'):
                            date_value = db.new_api.field_for(date_column, book_id)
                        else:
                            date_value = db.get_field(book_id, date_column, index_is_id=True)
                    
                    if not date_value or date_value == UNDEFINED_DATE:
                        error_count += 1
                        continue

                    # Get dialog instance
                    dialog = self.reminder_dialog()
                    if not dialog:
                        continue

                    # Check if reminder already exists for this book
                    has_duplicate, existing_title = dialog.has_reminder_for_book(book_id)
                    
                    # If a duplicate exists, ask the user what to do
                    if has_duplicate:
                        proceed = question_dialog(
                            self.gui,
                            _('Duplicate Book Reminder'),
                            _('A reminder already exists for "{0}":\n{1}\n\nCreate another reminder anyway?').format(title, existing_title),
                            show_copy_button=False
                        )
                        if not proceed:
                            duplicate_count += 1
                            continue

                    # Construct reminder title with book info
                    reminder_title = _('📚 {0} by {1}').format(title, db.get_metadata(book_id, index_is_id=True).authors[0])
                    
                    # Create reminder with book_id reference
                    reminder_id = dialog.create_reminder(
                        title=reminder_title,
                        delay_minutes=0,  # We'll set exact date later
                        book_id=book_id  # Store book_id for duplicate detection
                    )
                    
                    if reminder_id:
                        # Ensure proper UTC datetime
                        if hasattr(date_value, 'tzinfo'):
                            if date_value.tzinfo is None:
                                date_value = date_value.replace(tzinfo=datetime.timezone.utc)
                        else:
                            # If it's just a date without time, set it to midnight UTC
                            date_value = datetime.datetime.combine(
                                date_value,
                                datetime.time(0, 0),
                                tzinfo=datetime.timezone.utc
                            )
                        
                        # Update the reminder time
                        notification_times = plugin_prefs.get('notification_times', {})
                        notification_times[reminder_id] = date_value.isoformat()
                        plugin_prefs['notification_times'] = notification_times
                        
                        # Set status to active
                        reminder_status = plugin_prefs.get('reminder_status', {})
                        reminder_status[reminder_id] = REMINDER_STATUS['ACTIVE']
                        plugin_prefs['reminder_status'] = reminder_status
                        
                        success_count += 1

                except Exception as e:
                    error_count += 1
                    print(f"Error creating reminder for book {book_id}: {str(e)}")
                    print(traceback.format_exc())

            # Show results with duplicate info
            result_message = _('Successfully created {0} reminders.\n').format(success_count)
            
            if duplicate_count > 0:
                result_message += _('Skipped {0} duplicate book reminders.\n').format(duplicate_count)
                
            if error_count > 0:
                result_message += _('Failed to create {0} reminders.').format(error_count)
            
            info_dialog(self.gui, _('Reminders created'), result_message, show=True)

            # Refresh reminders dialog if open
            if self._reminder_dialog:
                self._reminder_dialog.load_todos()

        except Exception as e:
            error_dialog(self.gui, _('Error'),
                        _('Failed to create reminders: {0}').format(str(e)),
                        show=True,
                        det_msg=traceback.format_exc())

    def reminder_dialog(self):
        """Get or create the reminder dialog instance"""
        if not self._reminder_dialog:
            self._reminder_dialog = TodoDialog(self.gui)
        return self._reminder_dialog

    def apply_settings(self):
        pass

    def apply_library_data(self, db):
        """For library data load"""
        # Get library specific config
        cfg = get_library_config(db)
        
        # Get the selected date column name
        date_column = cfg.get(KEY_RELEASE_DATE_COLUMN, '')
        
        if date_column:
            # Check both built-in and custom columns
            field_metadata = db.field_metadata
            
            # Handle built-in date columns
            if not date_column.startswith('#'):  # Not a custom column
                column = field_metadata.get(date_column)
                if column and column.get('datatype') == 'datetime':
                    return True
                    
            # Handle custom date columns 
            custom_columns = field_metadata.custom_field_metadata()
            if date_column in custom_columns:
                column = custom_columns[date_column]
                if column['datatype'] == 'datetime':
                    return True
                    
        return False