#!/usr/bin/env python

from __future__ import absolute_import
__license__ = 'GPL v3'
__copyright__ = '2023'
__docformat__ = 'restructuredtext en'

try:
    load_translations()
except NameError:
    pass

from calibre.customize import InterfaceActionBase
from calibre.utils.config import JSONConfig
from calibre.gui2 import must_use_qt

# Store plugin preferences
plugin_prefs = JSONConfig('plugins/comfy_reminders')

# Store per-library settings namespace
PREFS_NAMESPACE = 'ComfyRemindersPlugin'
PREFS_KEY_SETTINGS = 'settings'

# Constants for settings
KEY_RELEASE_DATE_COLUMN = 'release_date_column'

# Set defaults for all preferences
plugin_prefs.defaults['todos'] = []
plugin_prefs.defaults['notification_times'] = {}
plugin_prefs.defaults['notification_history'] = {}
plugin_prefs.defaults['reminder_status'] = {}
plugin_prefs.defaults['life_logs'] = []
plugin_prefs.defaults['categories'] = [
    _('Activity'), 
    _('Notable Quotes'),
    _('Weight'), 
    _('Observation'), 
    _('Health'), 
    _('Mood'), 
    _('Custom')
]
plugin_prefs.defaults['timestamp_format'] = _('dd/MM/yyyy HH:mm')

class ComfyRemindersPlugin(InterfaceActionBase):
    name                = 'Comfy Reminders'
    description         = _('Create and manage reminders with an integrated notebook')
    supported_platforms = ['windows', 'osx', 'linux']
    author             = 'Comfy.n'
    version            = (1, 3, 1)
    minimum_calibre_version = (7, 23, 0)
    actual_plugin       = 'calibre_plugins.comfy_reminders.action:TodoListAction'

    def is_customizable(self):
        '''
        Disable customization via Preferences->Plugins since we handle settings
        through the main dialog's Settings button
        '''
        return False

    def config_widget(self):
        """
        Implement this method and :meth:`save_settings` in your plugin to
        use a custom configuration dialog.

        This method, if implemented, must return a QWidget. The widget can have
        an optional method validate() that takes no arguments and is called
        immediately after the user clicks OK. Changes are applied if and only
        if the method returns True.

        If for some reason you cannot perform the configuration at this time,
        return a tuple of two strings (message, details), these will be
        displayed as a warning dialog to the user and the process will be
        aborted.

        The base class implementation of this method raises NotImplementedError
        so by default no user configuration is possible.
        """
        with must_use_qt():
            from calibre_plugins.comfy_reminders.config import ConfigDialog
            return ConfigDialog(self.gui)

    def get_allowed_date_columns(self, db):
        """Get date columns excluding timestamp and last_modified"""
        date_columns = {}
        field_metadata = db.field_metadata
        
        # Add built-in date columns (excluding timestamp and last_modified)
        for key, column in field_metadata.items():
            if isinstance(column, dict) and column.get('datatype') == 'datetime':
                if not key.startswith('#') and key not in ['timestamp', 'last_modified']:
                    date_columns[key] = {
                        'name': column.get('name', key),
                        'datatype': 'datetime',
                        'display': column.get('display', {})
                    }
                    
        # Add custom date columns
        custom_fields = field_metadata.custom_field_metadata()
        for key, column in custom_fields.items():
            if column['datatype'] == 'datetime':
                date_columns[key] = column
                
        return date_columns


