#!/usr/bin/env python
# -*- coding: utf-8 -*-

# plugin: ForeignWords
# author: Maciej Haudek
# file: plugin.py (main plugin file)
# Copyright (c) 2024-2025 Maciej Haudek


from __future__ import unicode_literals, division, absolute_import, print_function

import os, inspect, re
import sys
import locale
from dialogs import launch_gui
from plugin_utils import _t  # Translate

from config import LANGUAGES

#try:
#    from PySide6.QtCore import QT_TRANSLATE_NOOP
#except:
#    pass


# ----------------------------------
# detect OS
# ----------------------------------
isosx = sys.platform.startswith('darwin')
islinux = sys.platform.startswith('linux')
iswindows = sys.platform.startswith('win32')

prefs = {}


def myOS():
    if isosx:
        print("INFO: OSX")
    if islinux:
        print("INFO: Linux")
    if iswindows:
        import platform as p
        print("INFO: Windows " + p.release())


# ----------------------------------
# detect UI language
# ----------------------------------
def GetUserDefaultUILanguage():
    if iswindows:
        import ctypes
        return locale.windows_locale[ctypes.windll.kernel32.GetUserDefaultUILanguage()][0:2]
    else:
        language, codepage = locale.getdefaultlocale()
        if language:
            return language[0:2]
        else:
            return 'en'


# ----------------------------------
# detect plugin version
# ----------------------------------
def get_current_version(bk):
    _version_pattern = re.compile(r'<version>([^<]*)</version>')
    _installed_version = None

    ppath = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, "plugin.xml")
    with open(ppath,'rb') as f:
        data = f.read()
        data = data.decode('utf-8', 'ignore')
        m = _version_pattern.search(data)
        if m:
            _installed_version = m.group(1).strip()
    return _installed_version


# the plugin entry point
def run(bk):
    try:
        if bk.launcher_version() < 20190927:
            # QT_TRANSLATE_NOOP
            print(_t("plugin", "Requires Sigil 1.0 or later"))
            return -1
    except:
        # QT_TRANSLATE_NOOP
        print(_t("plugin", "Requires Sigil 1.0 or later"))
        return -1

    _current_version = get_current_version(bk)
    # QT_TRANSLATE_NOOP
    print(_t("plugin", 'Launching the') + ' ' + bk._w.plugin_name + ' ' + _t("plugin", 'plugin') + " (" + _t("plugin", 'version') + ': ' + _current_version + ')\n')


    global prefs
    global word_list

    #------------------------
    # get preference
    #------------------------
    prefs = bk.getPrefs()

    # write initial JSON file
    if prefs == {}:
        prefs["use_dict"] = True
        prefs["lang_search"] = 'en'
        prefs["dict_file"] = 'default'
        prefs["use_class"] = True
        prefs["class_name"] = 'english'
        prefs["ignore_class"] = True
        prefs["use_lang"] = True
        prefs["use_xml_lang"] = True
        prefs["lang"] = 'en'
        #prefs["sort_attributes"] = False
        prefs["combine_words"] = True
        prefs["merge_spans"] = False
        prefs["run_without_confirm"] = True
        prefs["debug"] = False
        prefs['lang_codes'] = LANGUAGES

        bk.savePrefs(prefs)

    debug = True
    if 'debug' in prefs:
        debug = prefs.get('debug', debug)
    if debug: print(f"debug: {debug}")


    if debug:
        # QT_TRANSLATE_NOOP
        print(_t("plugin", "INFO: Some basic information..."))
        myOS()

        python_version = ".".join(map(str, sys.version_info[:3]))
        # QT_TRANSLATE_NOOP
        print(_t("plugin", 'INFO: Python'), python_version, '\n')

        uiLanguage = GetUserDefaultUILanguage()
        # QT_TRANSLATE_NOOP
        print(_t("plugin", "INFO: System interface language:"), uiLanguage)

        sigilui = bk.sigil_ui_lang
        # QT_TRANSLATE_NOOP
        print(_t("plugin", "INFO: Sigil interface language:"), sigilui)


    #print('\nClick OK to close the Plugin Runner window.')
    #print("b1")
    bailOut = launch_gui(bk, prefs, _current_version)
    #print("b2")
    # Save prefs to back to json
    bk.savePrefs(prefs)

    # Setting the proper Return value is important.
    # 0 - means success
    # anything else means failure
    return 0


def main():
    print("I reached main when I should not have\n")
    return -1


if __name__ == "__main__":
    sys.exit(main())
