#!/usr/bin/env python
# -*- coding: utf-8 -*-

# plugin: ForeignWords
# author: Maciej Haudek
# file: dialogs.py (plugin configuration window)
# Copyright (c) 2024-2025 Maciej Haudek


from __future__ import unicode_literals, division, absolute_import, print_function

import os
import sys
import inspect
import re

from plugin_utils import Qt, QtCore, QtGui, QtWidgets, QAction
from plugin_utils import PluginApplication, iswindows, _t  # , Signal, Slot, loadUi
try:
    from PySide6.QtCore import QT_TRANSLATE_NOOP
except:
    pass

try:
    from sigil_bs4 import BeautifulSoup
except Exception:
    from bs4 import BeautifulSoup

from config import LANGUAGES

import xml.etree.ElementTree as ET

from xml.sax.saxutils import escape

from essential import find_expressions, GetDictPath, mark_expressions, read_words, merge_spans
# sort_span_attributes
# find_and_mark_expressions

# Beta => Stable
BETA = True

BAIL_OUT = False

PROCESSING_SELECTED_FILES = QT_TRANSLATE_NOOP("guiMain", "Processing selected files...")
PROCESSING_ALL_FILES = QT_TRANSLATE_NOOP("guiMain", "Processing all files...")
PLUGIN_USING_PYSIDE6 = QT_TRANSLATE_NOOP("guiMain", "Plugin using PySide6")
PLUGIN_USING_PYQT5 = QT_TRANSLATE_NOOP("guiMain", "Plugin using PyQt5")

divider  = "-----------------------------------------------------"
divider2 = "====================================================="


def extract_body_snippet_with_tags(xhtml, char_limit=500):
    soup = BeautifulSoup(xhtml, 'html.parser')
    body = soup.body
    if body:
        body_html = str(body)
        return body_html[:char_limit]
    else:
        return "No <body> section found in the XHTML."

def launch_gui(bk, prefs, plugin_version):
    global file_list
    global selectedallmessage
    global selectedallmessagekolor
    global group_qss
    global tag_color
    global attr_color
    global val_color


    selected_files = []
    all_files = []
    for file_name in list(bk.selected_iter()):
        if bk.id_to_mime(file_name[1]) == 'application/xhtml+xml':
            selected_files.append(("manifest", file_name[1]))

    # get all files
    for id_type, href in bk.text_iter():
         all_files.append(("manifest", id_type))

    if BETA:
        # Beta icon for beta version
        icon = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'beta.svg')
    else:
        # Netutral icon
        icon = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'config.svg')

    if not os.path.exists(icon):
        icon = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'plugin.svg')

    if not os.path.exists(icon):
        icon = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'plugin.png')

    if not os.path.exists(icon):
        if prefs["debug"]:
            print(_t("guiMain", "INFO: Missing images"))


    mdp = True if iswindows else False
    app = PluginApplication(sys.argv, bk, app_icon=icon, match_dark_palette=mdp,
                            dont_use_native_menubars=True)


    if bk.colorMode() == "dark":
        group_qss="""
 QGroupBox {border: 1px solid; border-color: #242424; margin-top: 27px; font-size: 14px; font-weight: bold;}
 QGroupBox::title {color: #fff; background-color: #242424; subcontrol-origin: margin; subcontrol-position: top center; padding: 5px 8000px 5px 8000px;}
        """
        tag_color = "#efef8f"
        attr_color = "#9fc28a"
        val_color = "#e89198"
    else:
        group_qss="""
 QGroupBox {border: 1px solid; border-color: #555; margin-top: 27px; font-size: 14px; font-weight: bold;}
 QGroupBox::title {color: #fff; background-color: #555; subcontrol-origin: margin; subcontrol-position: top center; padding: 5px 8000px 5px 8000px;}
        """
        tag_color = "#0000ff"
        attr_color = "#800000"
        val_color = "#008080"

    if selected_files != []:
        selectedallmessage = _t("guiMain", PROCESSING_SELECTED_FILES)
        selectedallmessagekolor = 'QLabel {color: #25D366; font-weight: bold;}'
        file_list = selected_files
    else:
        selectedallmessage = _t("guiMain", PROCESSING_ALL_FILES)
        selectedallmessagekolor = 'QLabel {color: #BE1055; font-weight: bold;}'
        file_list = all_files

    try:
        win = guiMain(bk, prefs, file_list, plugin_version)
        win.show()
    except Exception as e:
        print(f"Error: {e}")
    app.exec()
    return win.getAbort()


class guiMain(QtWidgets.QWidget):
    def __init__(self, bk, prefs, file_list, plugin_version):
        super(guiMain, self).__init__()
        self.bk = bk
        self.prefs = bk.getPrefs()
        self.file_list = file_list
        self.plugin_version = plugin_version
        self._ok_to_close = False

        self.initUI()

    def initUI(self):
        # Main Layout
        main_layout = QtWidgets.QVBoxLayout(self)

        # All/selected
        file_list_layout = QtWidgets.QHBoxLayout()
        main_layout.addLayout(file_list_layout)
        self.label = QtWidgets.QLabel()
        font = QtGui.QFont()
        font.setPointSize(14)
        self.label.setFont(font)
        self.label.setText(selectedallmessage)
        self.label.setStyleSheet(selectedallmessagekolor)
        file_list_layout.addWidget(self.label)

        main_layout.addSpacing(10)

        # Two columns
        columns_layout = QtWidgets.QHBoxLayout()
        main_layout.addLayout(columns_layout)

        # Lewa kolumna – grupa "Main"
        main_group = QtWidgets.QGroupBox(self.tr("Main"))
        main_group.setObjectName("groupMain")
        main_group.setStyleSheet(group_qss)
        main_group_layout = QtWidgets.QFormLayout()

        # Elements Main Group START


        self.searchLabel = QtWidgets.QLabel(self.tr("SEARCH:"))
        self.searchLabel.setStyleSheet("QLabel {font-weight: bold; color: #CD71BB; font-size: 8pt;}")
        main_group_layout.addRow(self.searchLabel)


        self.useDictCheckBox = QtWidgets.QCheckBox()
        self.useDictCheckBox.setChecked(self.prefs.get("use_dict", True))
        # QT_TRANSLATE_NOOP
        self.useDictCheckBox.setToolTip(self.tr("Use user dictionary (enabled) or existing span tags in the text (disabled).\nSearch for existing span tags with the selected lang attribute."))
        # QT_TRANSLATE_NOOP
        main_group_layout.addRow(QtWidgets.QLabel(self.tr("Use User Dictionary:")), self.useDictCheckBox)

        self.useDictCheckBox.stateChanged.connect(self.toggleDictFile)


        # Lang Search START

        # QT_TRANSLATE_NOOP
        self.langSearchLabel = QtWidgets.QLabel(self.tr("Look For This Language Code:"))
        self.langSearchComboBox = QtWidgets.QComboBox()
        #self.langComboBox.addItem("en")
        self.langSearchComboBox.addItems(self.prefs.get('lang_codes',["en"]))
        # QT_TRANSLATE_NOOP
        self.langSearchComboBox.setToolTip(self.tr("Look for this language code in all span tags."))
        try:
            self.langSearchComboBox.setCurrentIndex(self.prefs.get('lang_codes',["en"]).index(self.prefs.get("lang_search", "en")))
        except:
            if self.prefs["debug"]: print(self.tr("INFO: Missing language in the list of languages:") + " " + self.prefs.get("lang_search"))
            self.langSearchComboBox.setCurrentIndex(0)

        main_group_layout.addRow(self.langSearchLabel, self.langSearchComboBox)

        # Lang Search END


        # Find folder user_dictionaries
        current_file_path = os.path.abspath(inspect.getfile(inspect.currentframe()))
        plugin_path = os.path.dirname(current_file_path)
        tmp_path = os.path.dirname(os.path.dirname(plugin_path))
        dictionary_folder = os.path.join(tmp_path, 'user_dictionaries')

        # Read filenames in folder, filtering out files with extensions
        dict_files = [f for f in os.listdir(dictionary_folder) if os.path.isfile(os.path.join(dictionary_folder, f)) and not os.path.splitext(f)[1]]

        self.dictLabel = QtWidgets.QLabel(self.tr("User Dictionary File:"))
        self.dictFileComboBox = QtWidgets.QComboBox()
        # Load user dict list to combobox
        self.dictFileComboBox.addItems(dict_files)
        # QT_TRANSLATE_NOOP
        self.dictFileComboBox.setToolTip(self.tr("Select the user dictionary file to use.\nOnly files without an extension."))
        #self.dictFileComboBox.setCurrentText(self.prefs.get("dict_file", "default"))
        # QT_TRANSLATE_NOOP
        main_group_layout.addRow(self.dictLabel, self.dictFileComboBox)
        #self.dictFileComboBox.setCurrentIndex(dict_files.index(self.prefs.get("dict_file", "default")))
        try:
            index = dict_files.index(self.prefs.get("dict_file", "default"))
        except ValueError:
            index = 0
            QtWidgets.QMessageBox.warning(self, self.tr("Warning"), self.tr("The user's dictionary folder lacks the file indicated in the configuration, the first file will be selected."))
        self.dictFileComboBox.setCurrentIndex(index)


        self.replaceLabel = QtWidgets.QLabel(self.tr("REPLACE:"))
        self.replaceLabel.setStyleSheet("QLabel {font-weight: bold; color: #CD71BB; font-size: 8pt;}")
        main_group_layout.addRow(self.replaceLabel)

        self.ignoreClassCheckBox = QtWidgets.QCheckBox()
        self.ignoreClassCheckBox.setChecked(self.prefs.get("ignore_class", True))
        # QT_TRANSLATE_NOOP
        self.ignoreClassCheckBox.setToolTip(self.tr("If there is a class attribute in the source span tag, leave it."))
        # QT_TRANSLATE_NOOP
        main_group_layout.addRow(QtWidgets.QLabel(self.tr("Ignore CSS Class:")), self.ignoreClassCheckBox)

        # Connect Ignore Class
        self.ignoreClassCheckBox.stateChanged.connect(self.toggleUseClassAndClassName)

        self.useClassLabel = QtWidgets.QLabel(self.tr("Use CSS Class:"))
        self.useClassCheckBox = QtWidgets.QCheckBox()
        self.useClassCheckBox.setChecked(self.prefs.get("use_class", True))
        # QT_TRANSLATE_NOOP
        self.useClassCheckBox.setToolTip(self.tr("Use a CSS class to style the marked text."))
        # QT_TRANSLATE_NOOP
        main_group_layout.addRow(self.useClassLabel, self.useClassCheckBox)

        # Connect Class
        self.useClassCheckBox.stateChanged.connect(self.toggleClassNameField)

        self.classNameLabel = QtWidgets.QLabel(self.tr("CSS Class Name:"))
        self.classNameLineEdit = QtWidgets.QLineEdit()
        self.classNameLineEdit.setText(self.prefs.get("class_name", "english"))
        # QT_TRANSLATE_NOOP
        self.classNameLineEdit.setToolTip(self.tr("Enter the name of the CSS class to use."))
        # QT_TRANSLATE_NOOP
        main_group_layout.addRow(self.classNameLabel, self.classNameLineEdit)

        # Initial stare for classNameLineEdit
        self.classNameLabel.setEnabled(self.useClassCheckBox.isChecked())
        self.classNameLineEdit.setEnabled(self.useClassCheckBox.isChecked())

        # Initial stare for useClass and classNameLineEdit
        self.classNameLabel.setEnabled(not self.ignoreClassCheckBox.isChecked())
        self.classNameLineEdit.setEnabled(not self.ignoreClassCheckBox.isChecked())
        self.useClassLabel.setEnabled(not self.ignoreClassCheckBox.isChecked())
        self.useClassCheckBox.setEnabled(not self.ignoreClassCheckBox.isChecked())

        self.useLangCheckBox = QtWidgets.QCheckBox()
        self.useLangCheckBox.setChecked(self.prefs.get("use_lang", True))
        # QT_TRANSLATE_NOOP
        self.useLangCheckBox.setToolTip(self.tr("Use the lang attribute to specify the language."))
        # QT_TRANSLATE_NOOP
        main_group_layout.addRow(QtWidgets.QLabel(self.tr("Use Lang Attribute:")), self.useLangCheckBox)

        self.useXmlLangCheckBox = QtWidgets.QCheckBox()
        self.useXmlLangCheckBox.setChecked(self.prefs.get("use_xml_lang", True))
        # QT_TRANSLATE_NOOP
        self.useXmlLangCheckBox.setToolTip(self.tr("Use the xml:lang attribute for XML compatibility."))
        # QT_TRANSLATE_NOOP
        main_group_layout.addRow(QtWidgets.QLabel(self.tr("Use xml:lang Attribute:")), self.useXmlLangCheckBox)

        # QT_TRANSLATE_NOOP
        self.langLabel = QtWidgets.QLabel(self.tr("Language Code:"))
        self.langComboBox = QtWidgets.QComboBox()
        #self.langComboBox.addItem("en")
        self.langComboBox.addItems(self.prefs.get('lang_codes',["en"]))
        # QT_TRANSLATE_NOOP
        self.langComboBox.setToolTip(self.tr("Select the language code to use."))
        try:
            self.langComboBox.setCurrentIndex(self.prefs.get('lang_codes',["en"]).index(self.prefs.get("lang", "en")))
        except:
            if self.prefs["debug"]: print(self.tr("INFO: Missing language in the list of languages:") + " " + self.prefs.get("lang"))
            self.langComboBox.setCurrentIndex(0)

        # Settings button
        self.settingsButton = QtWidgets.QPushButton()  # Utwórz przycisk bez tekstu
        self.settingsButton.clicked.connect(self.open_settings)

        settings_icon = os.path.join(plugin_path, 'config.svg')
        if os.path.exists(settings_icon):
            self.settingsButton.setIcon(QtGui.QIcon(settings_icon))
            icon_size = QtCore.QSize(16, 16)
            self.settingsButton.setFixedSize(icon_size)
            self.settingsButton.setIconSize(icon_size)
            self.settingsButton.setStyleSheet("QPushButton {border: none; padding: 0px;}")
        else:
            if self.prefs["debug"]:
                print(self.tr("INFO: Missing settings icon"))
            self.settingsButton.setText("Settings")

        self.settingsButton.setToolTip(self.tr("Edit the list of language codes."))


        combo_box_layout = QtWidgets.QHBoxLayout()
        combo_box_layout.addWidget(self.langComboBox)
        combo_box_layout.addWidget(self.settingsButton)
        main_group_layout.addRow(self.langLabel, combo_box_layout)

        # Initial state for combobox with label
        self.langComboBox.setEnabled(self.useLangCheckBox.isChecked() or self.useXmlLangCheckBox.isChecked())
        self.langLabel.setEnabled(self.useLangCheckBox.isChecked() or self.useXmlLangCheckBox.isChecked())

        # Connect Langs
        self.useLangCheckBox.stateChanged.connect(self.toggleLangComboBox)
        self.useXmlLangCheckBox.stateChanged.connect(self.toggleLangComboBox)

        # Elements Main Group END

        main_group.setLayout(main_group_layout)
        columns_layout.addWidget(main_group, stretch=1)

        # Prawa kolumna – pionowy układ dla "Additional" i "Debugging"
        right_column_layout = QtWidgets.QVBoxLayout()
        columns_layout.addLayout(right_column_layout, stretch=1)

        # Grupa "Additional"
        additional_group = QtWidgets.QGroupBox(self.tr("Additional"))
        additional_group.setObjectName("groupAdditional")
        additional_group.setStyleSheet(group_qss)
        additional_layout = QtWidgets.QFormLayout()

        # ElementsAdditional Group START

        #self.sortAttributesCheckBox = QtWidgets.QCheckBox()
        #self.sortAttributesCheckBox.setChecked(self.prefs.get("sort_attributes", True))
        # QT_TRANSLATE_NOOP
        #self.sortAttributesCheckBox.setToolTip(self.tr("Sort attributes in span tags."))
        # QT_TRANSLATE_NOOP
        #additional_layout.addRow(QtWidgets.QLabel(self.tr("Sort Attributes:")), self.sortAttributesCheckBox)

        self.combineWordsLabel = QtWidgets.QLabel(self.tr("Combine Words:"))
        self.combineWordsCheckBox = QtWidgets.QCheckBox()
        self.combineWordsCheckBox.setChecked(self.prefs.get("combine_words", True))
        # QT_TRANSLATE_NOOP
        self.combineWordsCheckBox.setToolTip(self.tr("Combine nearby words into phrases."))
        # QT_TRANSLATE_NOOP
        additional_layout.addRow(self.combineWordsLabel, self.combineWordsCheckBox)

        self.mergeSpansLabel = QtWidgets.QLabel(self.tr("Merge Spans:"))
        self.mergeSpansCheckBox = QtWidgets.QCheckBox()
        self.mergeSpansCheckBox.setChecked(self.prefs.get("merge_spans", True))
        # QT_TRANSLATE_NOOP
        self.mergeSpansCheckBox.setToolTip(self.tr("Merge span tags with the same attributes."))
        # QT_TRANSLATE_NOOP
        additional_layout.addRow(self.mergeSpansLabel, self.mergeSpansCheckBox)

        self.runWithoutConfirmCheckBox = QtWidgets.QCheckBox()
        self.runWithoutConfirmCheckBox.setChecked(self.prefs.get("run_without_confirm", False))
        # QT_TRANSLATE_NOOP
        self.runWithoutConfirmCheckBox.setToolTip(self.tr("Run the plugin without confirmation."))
        # QT_TRANSLATE_NOOP
        additional_layout.addRow(QtWidgets.QLabel(self.tr("Run Without Confirmation:")), self.runWithoutConfirmCheckBox)


        # Initial state for merge spans checkbox with label
        #self.mergeSpansCheckBox.setEnabled(self.sortAttributesCheckBox.isChecked())
        #self.mergeSpansLabel.setEnabled(self.sortAttributesCheckBox.isChecked())

        # Connect sort + merge
        #self.sortAttributesCheckBox.stateChanged.connect(self.toggleMergeSpans)

        # Elements Additional Group END

        additional_group.setLayout(additional_layout)
        right_column_layout.addWidget(additional_group)

        # Grupa "Debugging"
        debug_group = QtWidgets.QGroupBox(self.tr("Debugging"))
        debug_group.setObjectName("groupDeb")
        debug_group.setStyleSheet(group_qss)
        debug_layout = QtWidgets.QFormLayout()

        # Elements Debugging Group START

        self.debugCheckBox = QtWidgets.QCheckBox()
        self.debugCheckBox.setChecked(self.prefs.get("debug", False))
        # QT_TRANSLATE_NOOP
        self.debugCheckBox.setToolTip(self.tr("Enable debug mode to see detailed logs."))
        # QT_TRANSLATE_NOOP
        debug_layout.addRow(QtWidgets.QLabel(self.tr("Debug Mode (for developers):")), self.debugCheckBox)

        normal_label = QtWidgets.QLabel("Normal QLabel")
        disabled_label = QtWidgets.QLabel("Disabled QLabel")
        disabled_label.setEnabled(False)  # Wyłączenie etykiety

        debug_layout.addWidget(normal_label)
        debug_layout.addWidget(disabled_label)


        # Elements Debugging Group END

        debug_group.setLayout(debug_layout)
        right_column_layout.addWidget(debug_group)

        main_layout.addSpacing(10)

        # SPAN preview
        self.span_preview_label = QtWidgets.QLabel(self)
        self.span_preview_label.setText("")  # Początkowo puste
        self.span_preview_label.setAlignment(QtCore.Qt.AlignCenter)
        main_layout.addWidget(self.span_preview_label)

        main_layout.addSpacing(10)

        # Buttons and copytight
        button_layout = QtWidgets.QHBoxLayout()
        main_layout.addLayout(button_layout)
        self.process_button = QtWidgets.QPushButton(_t('guiMain', 'Process'), self)
        self.process_button.setToolTip(_t('guiMain', 'Processing files with current settings'))
        self.process_button.setStyleSheet("QPushButton {padding: 10px 0; font-size: 14px; font-weight: bold;}")
        self.process_button.clicked.connect(lambda: self.applyConfig())
        button_layout.addWidget(self.process_button)

        self.quit_button = QtWidgets.QPushButton(_t('guiMain', 'Quit'), self)
        self.quit_button.setToolTip(_t('guiMain', 'Quit with no changes'))
        self.quit_button.setStyleSheet("QPushButton {padding: 10px 0; font-size: 14px; font-weight: bold;}")
        self.quit_button.clicked.connect(self._quit_clicked)
        button_layout.addWidget(self.quit_button)

        main_layout.addSpacing(10)

        self.copyright_label = QtWidgets.QLabel(self)
        copyright_text = "Copyright (C) 2024-2025 Maciej Haudek • " \
                         "<a href='https://www.mobileread.com/forums/showthread.php?t=366832'>" + \
                         self.tr("Support") + "</a>"
        self.copyright_label.setText(copyright_text)
        self.copyright_label.setOpenExternalLinks(True)
        self.copyright_label.setAlignment(QtCore.Qt.AlignCenter)
        main_layout.addWidget(self.copyright_label)

        # Window title
        self.setWindowTitle(self.tr("Foreign Words Plugin") + " " + self.plugin_version)

        # Conntects for span preview
        self.classNameLineEdit.textChanged.connect(self.update_span_preview)
        self.langComboBox.currentIndexChanged.connect(self.update_span_preview)
        self.ignoreClassCheckBox.stateChanged.connect(self.update_span_preview)
        self.useClassCheckBox.stateChanged.connect(self.update_span_preview)
        self.useLangCheckBox.stateChanged.connect(self.update_span_preview)
        self.useXmlLangCheckBox.stateChanged.connect(self.update_span_preview)

        # Span preview init
        self.update_span_preview()


        # Initial for dictFile
        self.dictLabel.setVisible(self.useDictCheckBox.isChecked())
        self.dictFileComboBox.setVisible(self.useDictCheckBox.isChecked())
        self.langSearchLabel.setVisible(not self.useDictCheckBox.isChecked())
        self.langSearchComboBox.setVisible(not self.useDictCheckBox.isChecked())


        #if not self.prefs["use_dict"]:
        #    self.langLabel.setText(self.tr("Look For This Language Code:"))
        #    self.langLabel.setStyleSheet("font-weight: bold;");
        #else:
        #    self.langLabel.setStyleSheet("font-weight: normal;");

        self.show()


    def toggleDictFile(self, state):
        if state == 2:  # Qt.Checked:
            self.dictLabel.setVisible(True)
            self.dictFileComboBox.setVisible(True)
            self.langSearchLabel.setVisible(False)
            self.langSearchComboBox.setVisible(False)
            #self.langLabel.setText(self.tr("Language Code:"))
            #self.langLabel.setStyleSheet("font-weight: normal;");
        else:
            self.dictLabel.setVisible(False)
            self.dictFileComboBox.setVisible(False)
            self.langSearchLabel.setVisible(True)
            self.langSearchComboBox.setVisible(True)
            #self.langLabel.setText(self.tr("Look For This Language Code:"))
            #self.langLabel.setStyleSheet("font-weight: bold;");


    def toggleClassNameField(self, state):
        if state == 2:  # Qt.Checked
            self.classNameLabel.setEnabled(True)
            self.classNameLineEdit.setEnabled(True)
        else:
            self.classNameLabel.setEnabled(False)
            self.classNameLineEdit.setEnabled(False)


    def toggleUseClassAndClassName(self, state):
        if state == 2:  # Qt.Checked
            self.useClassLabel.setEnabled(False)
            self.useClassCheckBox.setEnabled(False)
            self.classNameLabel.setEnabled(False)
            self.classNameLineEdit.setEnabled(False)
        else:
            self.useClassLabel.setEnabled(True)
            self.useClassCheckBox.setEnabled(True)
            self.classNameLabel.setEnabled(self.useClassCheckBox.isChecked())
            self.classNameLineEdit.setEnabled(self.useClassCheckBox.isChecked())





    def toggleLangComboBox(self, state):
        self.langComboBox.setEnabled(self.useLangCheckBox.isChecked() or self.useXmlLangCheckBox.isChecked())
        self.langLabel.setEnabled(self.useLangCheckBox.isChecked() or self.useXmlLangCheckBox.isChecked())
        #if not self.prefs["use_dict"]:
        #    if not self.useLangCheckBox.isChecked() and not self.useXmlLangCheckBox.isChecked():
        #        QtWidgets.QMessageBox.warning(self, self.tr("Warning"), self.tr("If you remove both language attributes then they will\nno longer be in the file!\nYou can do so, of course, but you have been warned."))


    #def toggleMergeSpans(self, state):
    #    if state == 2:  # Qt.Checked:
    #        self.mergeSpansLabel.setEnabled(True)
    #        self.mergeSpansCheckBox.setEnabled(True)
    #    else:
    #        self.mergeSpansLabel.setEnabled(False)
    #        self.mergeSpansCheckBox.setEnabled(False)
    #        self.mergeSpansCheckBox.setChecked(False)


    def updateLanguageList(self):
        self.langComboBox.clear()
        self.langComboBox.addItems(self.prefs['lang_codes'])
        self.langSearchComboBox.clear()
        self.langSearchComboBox.addItems(self.prefs['lang_codes'])

        # Lang Search
        try:
            lang_search = self.prefs.get("lang_search", "en") # Get lang from prefs
            indexSearch = self.prefs['lang_codes'].index(lang_search) # Find index
            self.langSearchComboBox.setCurrentIndex(indexSearch)
            #if self.prefs["debug"]: print(f"Ustawiono index ComboBox na: {index} (język: {lang})")
        except ValueError:
            self.langSearchComboBox.setCurrentIndex(0)
            #if self.prefs["debug"]: print(f"Nie znaleziono języka '{lang}' w ustawieniach, ustawiono index na 0")



        # Lang Codes
        try:
            lang = self.prefs.get("lang", "en") # Get lang from prefs
            index = self.prefs['lang_codes'].index(lang) # Find index
            self.langComboBox.setCurrentIndex(index)
            #if self.prefs["debug"]: print(f"Ustawiono index ComboBox na: {index} (język: {lang})")
        except ValueError:
            self.langComboBox.setCurrentIndex(0)
            #if self.prefs["debug"]: print(f"Nie znaleziono języka '{lang}' w ustawieniach, ustawiono index na 0")


    def update_span_preview(self):

        """Aktualizuje podgląd znacznika span."""
        class_name = self.classNameLineEdit.text()
        lang = self.langComboBox.currentText()
        ignore_class = self.ignoreClassCheckBox.isChecked()
        use_class = self.useClassCheckBox.isChecked()
        use_lang = self.useLangCheckBox.isChecked()
        use_xml_lang = self.useXmlLangCheckBox.isChecked()
        span_tag = f"<span style=\"color: {tag_color}\">&lt;span</span>"
        if use_class and class_name and not ignore_class:
            span_tag += f' <span style=\"color: {attr_color}\">class</span>=<span style=\"color: {val_color}\">&quot;{class_name}&quot;</span>'
        if use_lang and lang:
            span_tag += f' <span style=\"color: {attr_color}\">lang=</span><span style=\"color: {val_color}\">&quot;{lang}&quot;</span>'
        if use_xml_lang and lang:
            span_tag += f' <span style=\"color: {attr_color}\">xml:lang</span>=<span style=\"color: {val_color}\">&quot;{lang}&quot;</span>'
        span_tag += f"<span style=\"color: {tag_color}\">&gt;&lt;/span&gt;</span>"
        self.span_preview_label.setText(span_tag)


    def open_settings(self):
        """
        Open the settings dialog.
        """
        dialog = SettingsDialog(self, prefs=self.prefs)
        dialog.load_settings()
        dialog.open()
        result = dialog.exec()
        if result == QtWidgets.QDialog.Accepted:
                #Save changes on accept C:\Users\<username>\AppData\Local\sigil-ebook\sigil\plugins_prefs\<plugin-name>
                self.bk.savePrefs(self.prefs)
                self.updateLanguageList()


    def applyConfig(self):
        self.prefs["use_dict"] = self.useDictCheckBox.isChecked()
        self.prefs["lang_search"] = self.langSearchComboBox.currentText()
        self.prefs["dict_file"] = self.dictFileComboBox.currentText()
        self.prefs["use_class"] = self.useClassCheckBox.isChecked()
        self.prefs["class_name"] = self.classNameLineEdit.text()
        self.prefs["ignore_class"] = self.ignoreClassCheckBox.isChecked()
        self.prefs["use_lang"] = self.useLangCheckBox.isChecked()
        self.prefs["use_xml_lang"] = self.useXmlLangCheckBox.isChecked()
        self.prefs["lang"] = self.langComboBox.currentText()
        #self.prefs["sort_attributes"] = self.sortAttributesCheckBox.isChecked()
        self.prefs["combine_words"] = self.combineWordsCheckBox.isChecked()
        self.prefs["merge_spans"] = self.mergeSpansCheckBox.isChecked()
        self.prefs["run_without_confirm"] = self.runWithoutConfirmCheckBox.isChecked()
        self.prefs["debug"] = self.debugCheckBox.isChecked()

        try:
            self.bk.savePrefs(self.prefs)
        except Exception as e:
            # QT_TRANSLATE_NOOP
            print(self.tr(f"Error saving preferences: {e}"))

        if self.prefs["debug"]:
            # QT_TRANSLATE_NOOP
            print(self.tr("INFO: Configuration saved:"), self.prefs)

        # Additional information when the user has not entered a class name
        if self.prefs["use_class"] and self.prefs["class_name"] == "":
            if self.prefs["debug"]:
                print(self.tr("INFO: Empty class name"))


        if self.prefs["run_without_confirm"]:
            self.runPlugin()
            self.close()
        else:
            # QT_TRANSLATE_NOOP
            #confirm = QtWidgets.QMessageBox.question(self, self.tr("Confirmation"), self.tr("Run the plugin?"))
            confirm = self.confirm_run() #QtWidgets.QMessageBox.question(self, self.tr("Confirmation"), self.tr("Run the plugin?"))
            if confirm == QtWidgets.QMessageBox.Yes:
                self.runPlugin()
                self.close()
            else:
                # QT_TRANSLATE_NOOP
                if self.prefs['debug']:
                    print(self.tr('In the confirmation box, the "No" button was pressed.'))
        # Close window
        #self.close()


    def confirm_run(self):
        """Confirm with the user before running the plugin."""
        #print("cr1")
        message = self.tr("Are you sure you want to continue?")
        message += "\n\n"
        if self.prefs["use_dict"]:
            message += self.tr("The plugin will find all the words from the dictionary and surround them with tag:")
        else:
           message += self.tr("The plugin will find all span tags with 'lang' or 'xml:lang' attribute set to")
           message += "'" + self.prefs["lang_search"] + "'"
           message += self.tr("language and replace them according to the settings:")
           message += "\n"
        #print("cr3")
        if self.prefs["use_class"] or self.prefs["use_lang"] or self.prefs["use_xml_lang"]:
            message += " <span"
            if self.prefs["use_class"] and self.prefs["class_name"] and not self.prefs["ignore_class"]:
                message += f' class="{self.prefs["class_name"]}"'
            if self.prefs["use_lang"] and self.prefs["lang"]:
                message += f' lang="{self.prefs["lang"]}"'
            if self.prefs["use_xml_lang"] and self.prefs["lang"]:
                message += f' xml:lang="{self.prefs["lang"]}"'
            message += "></span>\n\n"
        else:
            message += " <span></span>\n\n"
        #print("cr4")
        if self.prefs["ignore_class"]:
            message += self.tr("The existing class attribute and all attributes other than 'lang' and 'xml:lang' will remain unchanged.")
        else:
            if self.prefs["use_class"]:
                message += self.tr("The existing class, lang, xml:lang attributes of the span tag for the found words/phrases will be replaced!")
            else:
                message += self.tr("The existing class, lang, xml:lang attributes of the span tag for the found words/phrases will be removed!")
        #print("cr5")
        message += "\n\n"
        message += self.tr("As a user of this plugin, I declare that I definitely know what I am doing and have a backup of my EPUB file.")
        #print("cr6")
        return QtWidgets.QMessageBox.question(self, self.tr("Confirmation"), message,
                                          QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.No)


    def _quit_clicked(self):
        self._ok_to_close = True
        self.close()


    def keyPressEvent(self, event):
        if event.key() == QtCore.Qt.Key_Escape:
            # print(self.tr("INFO: The Escape key is pressed, I close the window."))
            self.close()
        else:
            super().keyPressEvent(event)


    def getAbort(self):
        return BAIL_OUT


    def closeEvent(self, event):
        if self._ok_to_close:
            event.accept()  # let the window close
            #if self.prefs['debug']:
            print("\n" + self.tr("INFO: Quit button pressed."))
        else:
            self._quit_clicked()
            #if self.prefs['debug']:
            print("\n" + self.tr("INFO: The plugin window has been closed."))



    def runPlugin(self):

        def build_lang_regex(lang_search=None):
            lang_part = r'\bxml:lang="([^"]*)"'
            if lang_search:
                    lang_part = f'\\blang="{lang_search}"'
            regex = (
                rf'(<span(?:\s+[^>]*\bclass="[^"]*")?(?:\s+[^>]*\bxml:lang="([^"]*)")?\s+[^>]*{lang_part}[^>]*>)'
                r'(.*?)</span>'
            )
            # print(regex)

            return regex


        temp_dir = self.bk._w.ebook_root

        # sigil 1.x requires full paths
        # get the default opf path and the opf file name
        if self.bk.launcher_version() >= 20190927:
            opf_path = os.path.join(self.bk._w.ebook_root, self.bk.get_opfbookpath())
            opf_name = os.path.basename(opf_path)
        else:
            opf_name = self.bk._w.opfname
            opf_path = os.path.join(bk._w.ebook_root, 'OEBPS', opf_name)

        tree = ET.parse(opf_path)
        root = tree.getroot()

        # get epub version number
        if self.prefs['debug']:
            epubversion = self.bk.epub_version()
            if epubversion.startswith("3"):
                # QT_TRANSLATE_NOOP
                print(self.tr("INFO: EPUB3 file"))
            else:
                # QT_TRANSLATE_NOOP
                print(self.tr("INFO: EPUB2 file"))

        word_list = ""
        word_list = read_words(GetDictPath(self.prefs['dict_file']))
        if self.prefs['debug']: print(f"INFO: word_list: {word_list}")

        if word_list != "":
            #for (id, href) in bk.text_iter():
            for (typ, id) in self.file_list:

                href = self.bk.id_to_href(id)
                #if self.prefs['debug']:
                print(divider2)
                # QT_TRANSLATE_NOOP
                print(self.tr("INFO: Processed file: %s") % href)

                try:
                    xhtml_orig = self.bk.readfile(id)
                    xhtml = xhtml_orig

                    if 'cover' not in href:  # skip cover
                        if self.prefs['debug']:
                            print(divider)
                            # QT_TRANSLATE_NOOP
                            print(self.tr("INFO: XHTML Before:"))
                            print(extract_body_snippet_with_tags(xhtml_orig))
                            print(divider)


                        # START USE_DICT CONDITION

                        if self.prefs['use_dict']:
                            #print("debug 1")
                            # Znalezione wyrażenia
                            found_expressions = find_expressions(xhtml, word_list, self.prefs['combine_words'])
                            #print("debug 2")
                            if found_expressions:
                                #print("debug 3")
                                # Sortowanie od najdłuższego do najkrótszego
                                expressions = sorted(found_expressions, key=len, reverse=True)
                                #print("debug 4")
                                # Ponieważ tu wykorzystujemy user dictionary to ustawiamy lang_search na None
                                searchLang = None
                                xhtml = mark_expressions(xhtml, expressions,
                                                          searchLang,
                                                          self.prefs['use_class'],
                                                          self.prefs['class_name'],
                                                          self.prefs['ignore_class'],
                                                          self.prefs['use_lang'],
                                                          self.prefs['use_xml_lang'],
                                                          self.prefs['lang'],
                                                          self.prefs['combine_words'])

                                    #print("debug 5")
                                if self.prefs['debug']:
                                    # QT_TRANSLATE_NOOP
                                    print(self.tr("INFO: Found words/phrases:"))
                                    for expr in expressions:
                                        print(f'• {expr}')
                                #print("debug 6")
                                if self.prefs["merge_spans"]:

                                    if self.prefs['debug']:
                                        print(divider)
                                        # QT_TRANSLATE_NOOP
                                        print(self.tr("INFO: XHTML Before merging:"))
                                        print(extract_body_snippet_with_tags(xhtml))
                                        print(divider)
                                    # QT_TRANSLATE_NOOP
                                    print(self.tr("INFO: Merge spans..."))
                                    xhtml = merge_spans(xhtml)
                            else:
                                # QT_TRANSLATE_NOOP
                                print(self.tr("INFO: No words/phrases were found"))

                        # FILES ONLY
                        else:
                            expressions = []

                            #lang_regex = r"(<span(?:\s+[^>]*\bclass=\"[^\"]*\")?(?:\s+[^>]*\bxml:lang=\"[^\"]*\")?\s+[^>]*\blang=\"([^\"]*)\"[^>]*>)(.*?)</span>"
                            lang_regex = build_lang_regex(lang_search=self.prefs['lang_search'])
                            #print("x1")
                            matches = re.finditer(lang_regex, xhtml)
                            #print("x2")
                            for match in matches:
                                #print("x22")
                                if match:
                                    #print(match.group(3))
                                    expressions.append(escape(match.group(3)))
                            #print("x3")
                            if not self.prefs['use_dict']:
                                searchLang = self.prefs['lang_search']
                            else:
                                searchLang = None
                            #print("x4")
                            if expressions:
                                print(self.tr("INFO: Found words/phrases (in files):"))
                                for expr in expressions:
                                    print(f'• {expr}')
                                xhtml = mark_expressions(xhtml, expressions,
                                                          searchLang,
                                                          self.prefs['use_class'],
                                                          self.prefs['class_name'],
                                                          self.prefs['ignore_class'],
                                                          self.prefs['use_lang'],
                                                          self.prefs['use_xml_lang'],
                                                          self.prefs['lang'],
                                                          self.prefs['combine_words'])



                        #print("debug 7")
                        if self.prefs['debug']:
                            print(divider)
                            # QT_TRANSLATE_NOOP
                            print(self.tr("INFO: XHTML Final:"))
                            print(extract_body_snippet_with_tags(xhtml))
                            print(divider)
                        #print("debug 8")
                        if xhtml != xhtml_orig:
                            #self.bk.writefile(id, str(xhtml))
                            soup = BeautifulSoup(xhtml, "html.parser")
                            #self.bk.writefile(id, str(soup.prettyprint_xhtml(indent_chars="  ")))
                            # try Sigil bs4 pretty print
                            try:
                                self.bk.writefile(id, str(soup.prettyprint_xhtml(indent_chars="  ")))
                            except:
                                self.bk.writefile(id, str(soup))

                            # QT_TRANSLATE_NOOP
                            print(self.tr("INFO: Saving file..."))
                        else:
                            # QT_TRANSLATE_NOOP
                            print(self.tr("INFO: No changes to save"))
                    else:
                        # QT_TRANSLATE_NOOP
                        print(self.tr("INFO: Skipping cover"))

                except Exception as e:
                    CRITICAL_ERROR = QT_TRANSLATE_NOOP("guiMain", "Critical error during processing")
                    print(self.tr(CRITICAL_ERROR) + " {href}:")
                    print(str(e))
                    import traceback
                    traceback.print_exc()

        else:
            EMPTY_DICT_WARNING = QT_TRANSLATE_NOOP("guiMain", "Since the list of words in dictionary \"{dict_file}\" is empty, further action makes no sense.")
            print(f'\n{self.tr(EMPTY_DICT_WARNING).format(dict_file=prefs["dict_file"])}')


class SettingsDialog(QtWidgets.QDialog):

    def __init__(self, parent=None, prefs=None):
        """
        Initialize the SettingsDialog.

        This dialog allows users to modify plugin settings.

        Args:
            parent (QWidget, optional): The parent widget. Defaults to None.
            prefs (dict, optional): A dictionary containing the current preferences.
                                    This dictionary will be modified directly.
                                    Defaults to None.

        Note:
            The prefs dictionary is modified in-place. Changes made in this dialog
            will affect the original dictionary even if the dialog is cancelled.
            Be careful what you save
        """
        super().__init__(parent)
        self.parent = parent
        self.prefs = prefs if prefs else {}
        self.setWindowTitle(self.tr("Settings"))
        self.setMinimumWidth(300)
        self.setup_ui()

    def setup_ui(self):
        """
        Set up the user interface for the settings dialog.
        """
        layout = QtWidgets.QVBoxLayout(self)
        #layout.setContentsMargins(15, 15, 15, 15)

        # Language codes input
        self.lblLang = QtWidgets.QLabel(self.tr("Language Codes (comma-separated)"))
        self.lneLang = QtWidgets.QLineEdit()
        self.lneLang.setPlaceholderText(self.tr("Codes for languages"))
        self.lneLang.setToolTip(self.tr("Up to 15 codes. Leave empty to use defaults"))
        layout.addWidget(self.lblLang)
        layout.addWidget(self.lneLang)
        layout.addSpacing(20)

        # Buttons
        lytButtons = QtWidgets.QHBoxLayout()
        self.btnAccept = QtWidgets.QPushButton(self.tr("Accept"))
        self.btnCancel = QtWidgets.QPushButton(self.tr("Cancel"))
        lytButtons.addWidget(self.btnAccept)
        lytButtons.addWidget(self.btnCancel)
        layout.addLayout(lytButtons)

        # Connect buttons
        self.btnAccept.clicked.connect(self.accept_settings)
        self.btnCancel.clicked.connect(self.reject)


    def accept_settings(self):
        """
        Accept and save the current settings.
        """
        lang_codes = [code.strip() for code in self.lneLang.text().split(',') if code.strip()]
        valid_codes = []

        for code in lang_codes:
            if not re.match(r"^[a-zA-Z-]+$", code):
                QtWidgets.QMessageBox.warning(self, self.tr("Warning"),
                                    self.tr("Invalid language code: {code}. Must contain only letters and hyphen",  code=code))
                return
            if code not in valid_codes:
                valid_codes.append(code)

        if len(valid_codes) > 15:
            QtWidgets.QMessageBox.warning(self, self.tr("Warning"),
                            self.tr("Maximum 15 unique language codes allowed. Truncating list"))
            valid_codes = valid_codes[:15]

        if len(valid_codes) < len(lang_codes):
             QtWidgets.QMessageBox.information(self, self.tr("Information"),
                                self.tr("Duplicate codes were removed from the list"))

        #Update the preferences and pass control back to the main GUI
        self.prefs['lang_codes'] = valid_codes
        self.accept()


    def load_settings(self):
        """
        Load existing settings into the dialog.
        """
        if self.prefs is None:
            return

        # What sound does a pref make if there's no one to store it?
        if 'lang_codes' not in self.prefs:
                self.prefs['lang_codes'] = LANGUAGES

        # Load language codes
        str_lang_codes = ', '.join(self.prefs['lang_codes'])
        if str_lang_codes:
            self.lneLang.setText(str_lang_codes)


def main():
    return -1


if __name__ == "__main__":
    sys.exit(main())
