#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
from collections import OrderedDict, defaultdict

from calibre import prints, sanitize_file_name
from calibre.constants import DEBUG
from calibre.utils.date import now, as_local_time, as_utc, format_date

from calibre_plugins.action_chains.common_utils import safe_name
from calibre.gui2.dialogs.template_dialog import TemplateDialog

from calibre_plugins.action_chains.templates.functions.base import TemplateFunction

try:
    load_translations()
except NameError:
    prints("ActionsChain/templates/functions.py - exception when loading translations")

class SelectionCount(TemplateFunction):

    doc = _('Returns the number of books selected in the library view. '
            'This function is part of the action chain plugin and will not work elsewhere.')
    name = 'selection_count'
    arg_count = 0
    _is_builtin = True

    def __init__(self, plugin_action):
        TemplateFunction.__init__(self, plugin_action)
        self.use_cache = False
        self.cached_selection_count = None

    def init_cache(self, count=None):
        self.use_cache = True
        self.cached_selection_count = count

    def flush_cache(self):
        self.use_cache = False
        self.cached_selection_count = None

    def evaluate(self, formatter, kwargs, mi, locals):
        gui = self.plugin_action.gui
        if self.use_cache:
            if self.cached_selection_count is not None:
                return self.cached_selection_count
            else:
                self.cached_selection_count = len(gui.current_view().selectionModel().selectedRows())
                return self.cached_selection_count

        else:
            return len(gui.current_view().selectionModel().selectedRows())

class ContainingFolder(TemplateFunction):

    doc = _('Returns containing folder for the book. '
            'This function is part of the action chain plugin and will not work elsewhere.')
    name = 'containing_folder'
    arg_count = 0
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals):
        db = self.plugin_action.gui.current_db
        book_id = mi.id
        return db.abspath(book_id, index_is_id=True)

class BookVars(TemplateFunction):

    doc = _('Retrieve book specific variable assinged to `var_name` supplied '
            'as the first argument. '
            'If `var_name` is an empty string, all names/values  will '
            'be returned as comma separted list of name:value pairs. '
            'Comma can be replaced by providing a second arg containing '
            'an alternative separator. '
            'This function is part of the action chain plugin and will not work elsewhere.')
    name = 'book_vars'
    arg_count = -1
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals, *args):
        sep = ','
        if len(args) not in [1, 2]:
            raise ValueError('Incorrect number of arguments')
        if len(args) == 2:
            if args[0] != '':
                raise ValueError('Incorrect number of arguments')
            sep = args[1]
        var_name = args[0]
        if not var_name.replace('_', '').isalnum():
            raise ValueError(f'Invalid variable name: {var_name}')
        book_id = mi.id
        if book_id:
            vars_dict = formatter.global_vars.get('_book_vars', defaultdict(dict))
            book_vars = vars_dict.get(book_id, {})
            if var_name:
                val = book_vars.get(var_name, '')
                if isinstance(val, (list,set,tuple)):
                    return ','.join([x for x in val])
                elif val is True:
                    return '1'
                elif val is False:
                    return '0'
                elif val is None:
                    return ''
                else:
                    return str(val)
            else:
                csp = ''
                for k,v in book_vars.items():
                    csp += f'{sep}{k}:{v}'
                csp = csp.lstrip(sep)
                return csp

class SetBookVars(TemplateFunction):

    doc = _('Assign value `var_value` to `var_name` specific current book. '
            'This function is part of the action chain plugin and will not work elsewhere.')
    name = 'set_book_vars'
    arg_count = 2
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals, var_name, var_value):
        if not var_name.replace('_', '').isalnum():
            raise ValueError(f'Invalid variable name: {var_name}')
        book_id = mi.id
        if book_id:
            vars_dict = formatter.global_vars.get('_book_vars', defaultdict(dict))
            vars_dict[book_id][var_name] = var_value
        return ''

class ListBookVars(TemplateFunction):

    doc = _('Return a list for variables set for book using set_boo_var')
    name = 'list_book_vars'
    arg_count = 0
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals):
        book_id = mi.id
        if book_id:
            vars_dict = formatter.global_vars.get('_book_vars', defaultdict(dict))
            return ','.join(list(vars_dict[book_id].keys()))
        else:
            return ''

class CategoryItems(TemplateFunction):

    doc = _('Return all item names from `category_name`'
            'Usage: category_items(category_name, [book_ids])\n'
            '`category_name`: Name of the category e.g authors, tags, ...etc.\n'
            '`book_ids`: (optional) book_ids to restrict the seach in.')
    name = 'category_items'
    arg_count = -1
    _is_builtin = True

    def __init__(self, plugin_action):
        TemplateFunction.__init__(self, plugin_action)
        self.possible_cats = self.plugin_action.gui.current_db.new_api.get_categories().keys()

    def evaluate(self, formatter, kwargs, mi, locals, *args):
        gui = self.plugin_action.gui
        db = gui.current_db        
        if len(args) not in [1, 2]:
            raise ValueError('Incorrect number of arguments')
        if len(args) == 2:
            try:
                book_ids = [ x.strip() for x in args[1].split(',') ]
                book_ids = [int(book_id) for book_id in book_ids]
            except:
                raise ValueError('Invalid list of book_ids')
        else:
            book_ids = None
        col_name = args[0]
        try:
            sep = db.field_metadata.all_metadata()[col_name]['is_multiple']['list_to_ui']
        except:
            sep = ' ::: '
        if not col_name in self.possible_cats:
            raise ValueError(f'Column is not a valid category: {col_name}')
        if self.context == 'template_dialog_mode':
            l = [ f'{col_name}{x}' for x in range(1,6) ]
            return sep.join(l)
        cats = db.new_api.get_categories(book_ids=book_ids)
        items = [ t.name for t in cats[col_name] ]
        return sep.join(items)

class SanitizePath(TemplateFunction):

    doc = _('Remove illegal characters from file paths. Also convert slash to backslash in Windows.\n'
            'Usage: sanitize_path(sanitize_path, [substitute])\n'
            '`file_path`: Path of the file you want to sanitize.\n'
            '`substitute`: optional argument for the character used to replace illegal characters. '
            'If not specified, "_" will be used by default. If "None" no sanitization will take place '
            'and the function will return the normalized path (replacing backslash with a slash in Windows.)')
    name = 'sanitize_path'
    arg_count = -1
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals, *args):
        gui = self.plugin_action.gui
        db = gui.current_db
        substitute = '_'
        if len(args) not in [1, 2]:
            raise ValueError('Incorrect number of arguments')
        if len(args) == 2:
            substitute = args[1]
        file_path = args[0]
        normalized = os.path.normpath(file_path)
        normalized = os.path.expandvars(normalized)
        if substitute == 'None':
            return normalized
        # Exclude drive (e.g. 'C:') from sanitization as colons are interpeted as illegal characters
        drive, driveless = os.path.splitdrive(normalized)
        components = driveless.split(os.sep)
        sanitized = [sanitize_file_name(x, substitute=substitute) for x in components]
        sanitized_string = os.sep.join(sanitized)
        return os.path.join(drive, sanitized_string)


class CoverPath(TemplateFunction):

    doc = _('Return book cover if present, else return an empty string')
    name = 'cover_path'
    arg_count = 0
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals):
        gui = self.plugin_action.gui
        db = gui.current_db
        book_id = mi.id
        if not db.has_cover(book_id):
            return ''
        path_to_cover = os.path.join(db.library_path, db.path(book_id, index_is_id=True), 'cover.jpg')
        return path_to_cover

class LastModified(TemplateFunction):

    doc = _('Rertun the time where database was last modified.')
    name = 'last_modified'
    arg_count = 0
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals, *args):
        db = self.plugin_action.gui.current_db
        db_last_modified = as_local_time(db.last_modified())
        return format_date(db_last_modified, 'iso')

class ChainVariant(TemplateFunction):

    doc = _('Add a chain variant.\n'
            'Usage: add_chain_variant(menu_text, argument, [sub_menu_text, icon])\n'
            'is stored as a chain variable, and can be accessed using the globals() template function.\n'
            '`menu_text`: The text appearing in the menu entry for this chain variant. '
            'If value is "separator", a menu separator will be inserted.'
            '`argument`: Argument passed to the chain by this particular variant. This argument '
            '`sub_menu_text`: (optional) submenu for the chain variant. Multiple nested submenus '
            'are supported. They should be separated with ::: e.g. submenu1:::submenu2.\n'
            '`icon`: (optional) icon for the chain variant menu entry.\n'
            'Using this function you can add multiple menu entries per chain, with each entry '
            'passing a different argument to the chain, which can take different action based '
            'on this passed value.')
    name = 'add_chain_variant'
    arg_count = -1
    _is_builtin = True

    def __init__(self, plugin_action):
        TemplateFunction.__init__(self, plugin_action)
        self.chain_variants = OrderedDict()

    def evaluate(self, formatter, kwargs, mi, locals, *args):
        gui = self.plugin_action.gui
        db = gui.current_db
        if len(args) not in [2, 3, 4]:
            raise ValueError('Incorrect number of arguments')
        if not self.context == 'template_dialog_mode':
            menu_text = args[0]
            if menu_text == 'separator':
                menu_text = safe_name('__separator__', self.chain_variants.keys())
            self.chain_variants[menu_text] = {'menu_text': menu_text}
            rep = self.chain_variants[menu_text]
            if len(args) > 1:
                rep['argument'] = args[1]
            if len(args) > 2:
                rep['sub_menu_text'] = args[2]
            if len(args) > 3:
                rep['icon'] = args[3]
        return ''

class BookField(TemplateFunction):

    doc = _('book_field(book_id, lookup_name) -- returns the metadata field named by lookup_name')
    name = 'book_field'
    arg_count = 2
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals, book_id, field_name):
        gui = self.plugin_action.gui
        db = gui.current_db
        try:
            book_id = int(book_id)
        except:
            raise ValueError(f'Invalid book_id: {book_id}')
        # You must check we are not in template mode, in case book_ids come
        # from a function like from_search()
        # from_search() returns arbitrary numbers when in template mode
        if not self.context == 'template_dialog_mode' and not book_id in db.all_ids():
            raise ValueError(f'book_id: ({book_id}) not available')
        mi = db.new_api.get_proxy_metadata(book_id)
        template = f"program: field('{field_name}')"
        return formatter.__class__().safe_format(template, mi, 'TEMPLATE_ERROR', mi)

class BookRawField(TemplateFunction):

    doc = _('raw_book_field(book_id, lookup_name [, optional_default]) -- returns the metadata '
            'field named by lookup_name without applying any formatting. It evaluates and returns '
            'the optional second argument `default` if the field is undefined (None).')
    name = 'book_raw_field'
    arg_count = -1
    _is_builtin = True

    def evaluate(self, formatter, kwargs, mi, locals, *args):
        gui = self.plugin_action.gui
        db = gui.current_db
        if len(args) not in [2, 3]:
            raise ValueError('Incorrect number of arguments')
        book_id = args[0]
        field_name = args[1]
        default = None
        if len(args) == 3:
            default = args[2]
        try:
            book_id = int(book_id)
        except:
            raise ValueError(f'Invalid book_id: {book_id}')
        # You must check we are not in template mode, in case book_ids come
        # from a function like from_search()
        # from_search() returns arbitrary numbers when in template mode
        if not self.context == 'template_dialog_mode' and not book_id in db.all_ids():
            raise ValueError(f'book_id: {book_id} not available')
        mi = db.new_api.get_proxy_metadata(book_id)
        if default is not None:
            template = f"program: raw_field('{field_name}', '{default}')"
        else:
            template = f"program: raw_field('{field_name}')"
        return formatter.__class__().safe_format(template, mi, 'TEMPLATE_ERROR', mi)
