#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
import traceback

from qt.core import Qt, QDialog, QLineEdit

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.dialogs.template_dialog import TemplateDialog
from calibre.utils.formatter_functions import formatter_functions

from calibre_plugins.action_chains.common_utils import (get_icon, call_method,
                                                        has_non_inherited_attribute)
from calibre_plugins.action_chains.templates import get_metadata_object, check_template, TemplateFunction


try:
    load_translations()
except NameError:
    prints("ActionsChain/templates/dialogs.py - exception when loading translations")


class TemplateBox(TemplateDialog):
    def __init__(
        self,
        parent,
        plugin_action,
        template_text='',
        placeholder_text=_('Enter a template to test using data from the selected book'),
        mi=None,
        all_functions=None,
        global_vars={},
        show_buttonbox=True,
        dialog_is_st_editor=False
    ):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db

        self.all_functions = all_functions
        if not self.all_functions:        
            self.all_functions = self.plugin_action.template_functions

        builtin_functions = formatter_functions().get_builtins()
        for name, func in self.all_functions.items():
            if isinstance(func, TemplateFunction):
                if has_non_inherited_attribute(func, '_is_builtin'):
                    builtin_functions[name] = func
        
        if not mi:
            mi = get_metadata_object(self.gui)

        if not template_text:
            text = placeholder_text
            text_is_placeholder = True
            window_title = _('Add template')
        else:
            text = None
            text_is_placeholder = False
            window_title = _('Edit Template')

        # We must call this before intializing the template dialog
        # Call on_template_dialog_start for template functions   
        self.on_template_dialog_start(self)

        from calibre_plugins.action_chains.templates import ACTemplateContext
        TemplateDialog.__init__(self,
                                parent,
                                text,
                                mi=mi,
                                text_is_placeholder=text_is_placeholder,
                                all_functions=self.all_functions,
                                builtin_functions=builtin_functions,
                                global_vars=global_vars,
                                python_context_object=ACTemplateContext())

        self.setup_saved_template_editor(show_buttonbox=show_buttonbox,
                                        show_doc_and_name=dialog_is_st_editor)

        self.setWindowTitle(window_title)
        if template_text:
            self.textbox.insertPlainText(template_text)

        # On dialog exit we must call on_template_dialog_exit for template functions. We connect to finished()
        self.finished.connect(partial(self.on_template_dialog_exit, self))

    def on_template_dialog_start(self, dialog):
        call_method(self.plugin_action.template_functions, 'on_template_dialog_start', dialog)

    def on_template_dialog_exit(self, dialog):
        call_method(self.plugin_action.template_functions, 'on_template_dialog_exit', dialog)

    def accept(self):        
        self.template = str(self.textbox.toPlainText()).rstrip()
        chk = check_template(self.template, self.plugin_action, template_functions=self.all_functions)
        if chk is True:
            # accept and save_geometry
            TemplateDialog.accept(self)

    def reject(self):
        # TemplateDialog.reject() closes parent dialog. So we have to override it.
        QDialog.reject(self)


class TemplateLineEditor(QLineEdit):

    '''
    Extend the context menu of a QLineEdit to include more actions.
    '''

    def __init__(self, parent, plugin_action):
        QLineEdit.__init__(self, parent)
        self.plugin_action = plugin_action
        self.setClearButtonEnabled(True)

    def contextMenuEvent(self, event):
        menu = self.createStandardContextMenu()
        menu.addSeparator()

        action_clear_field = menu.addAction(_('Remove any template from the box'))
        action_clear_field.triggered.connect(self.clear_field)
        action_open_editor = menu.addAction(_('Open template editor'))
        action_open_editor.triggered.connect(self.open_editor)
        menu.exec_(event.globalPos())

    def clear_field(self):
        self.setText('')

    def open_editor(self):
        t = TemplateBox(
            self,
            self.plugin_action,
            template_text = self.text(),
            placeholder_text=_('Template should return a comma separated list of formats.'),
        )

        t.setWindowTitle(_('Edit template'))
        if t.exec_():
            self.setText(t.template)
