#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QHBoxLayout,
                     QLabel, QGroupBox, QRadioButton, QLineEdit, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.utils.search_query_parser import ParseException

from calibre_plugins.action_chains.scopes.base import ActionScope
from calibre_plugins.action_chains.database import is_search_valid, get_books_modified_since

class BasicScopeConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):
        l = self.l = QVBoxLayout()
        self.setLayout(l)

        vl_chk = self.vl_chk = QCheckBox(_('Exclude books not in current virtual library'))
        vl_chk.setChecked(False)
        l.addWidget(vl_chk)

        scope_groupbox = QGroupBox(_('Scope options'))
        scope_groupbox_layout = QVBoxLayout()
        scope_groupbox.setLayout(scope_groupbox_layout)
        selected_opt = self.selected_opt = QRadioButton(_('Act on books selected in library view'))
        scope_groupbox_layout.addWidget(selected_opt)
        selected_opt.setChecked(True)
        all_opt = self.all_opt = QRadioButton(_('Act on all books'))
        scope_groupbox_layout.addWidget(all_opt)
        book_vars_opt = self.book_vars_opt = QRadioButton(_('Act on books that have values in Book Vars'))
        scope_groupbox_layout.addWidget(book_vars_opt)
        chain_modified_opt = self.chain_modified_opt = QRadioButton(_('Act on books modified since the start of this chain'))
        scope_groupbox_layout.addWidget(chain_modified_opt)
        last_modified_opt = self.last_modified_opt = QRadioButton(_('Act on books modified by last action'))
        scope_groupbox_layout.addWidget(last_modified_opt)

        search_layout = QVBoxLayout()
        scope_groupbox_layout.addLayout(search_layout)
        search_opt = self.search_opt = QRadioButton(_('Act on books resulting from search'))
        searchbox = self.searchbox = QLineEdit(self)
        searchbox.textChanged.connect(self._on_searchbox_text_change)
        search_layout.addWidget(search_opt)
        search_layout.addWidget(searchbox)
        self.selected_restrict_chk = QCheckBox(_('Restrict to selected books'))
        self.selected_restrict_chk.setChecked(False)
        search_layout.addWidget(self.selected_restrict_chk)

        self.l.addWidget(scope_groupbox)
        self.l.addStretch(1)
        self.setMinimumSize(300,300)

    def _on_searchbox_text_change(self):
        text = self.searchbox.text()
        if text:
            ok = is_search_valid(self.db, text)
            ss = 'QLineEdit { border: 2px solid %s; border-radius: 3px }' % (
            '#50c878' if ok else '#FF2400')
            self.searchbox.setStyleSheet(ss)
        else:
            self.searchbox.setStyleSheet('')

    def load_settings(self, settings):
        if settings:
            self.vl_chk.setChecked(settings.get('vl', False))
            scope = settings.get('opt', 'selected_books')
            if scope == 'selected_books':
                self.selected_opt.setChecked(True)
            elif scope == 'all_books':
                self.all_opt.setChecked(True)
            elif scope == 'book_vars':
                self.book_vars_opt.setChecked(True)
            elif scope == 'last_action_modified':
                self.last_modified_opt.setChecked(True)
            elif scope == 'chain_modified':
                self.chain_modified_opt.setChecked(True)
            elif scope == 'search':
                self.search_opt.setChecked(True)
                search_text = settings.get('search_text', '')
                self.searchbox.setText(search_text)
                self.selected_restrict_chk.setChecked(settings.get('restrict_to_selection', False))

    def save_settings(self):
        settings = {}
        settings['vl'] = self.vl_chk.isChecked()
        if self.selected_opt.isChecked():
            settings['opt'] = 'selected_books'
        elif self.all_opt.isChecked():
            settings['opt'] = 'all_books'
        elif self.book_vars_opt.isChecked():
            settings['opt'] = 'book_vars'
        elif self.last_modified_opt.isChecked():
            settings['opt'] = 'last_action_modified'
        elif self.chain_modified_opt.isChecked():
            settings['opt'] = 'chain_modified'
        elif self.search_opt.isChecked():
            settings['opt'] = 'search'
            settings['search_text'] = self.searchbox.text()
            settings['restrict_to_selection'] = self.selected_restrict_chk.isChecked()
        return settings

class BasicScope(ActionScope):

    name = 'Basic Scopes'

    def get_book_ids(self, gui, settings, chain):
        db = gui.current_db
        scope = settings.get('opt', 'selected_books')
        if scope == 'selected_books':
            book_ids = ActionScope.get_book_ids(self, gui, settings, chain)
        elif scope == 'all_books':
            book_ids = gui.current_db.all_ids()
        elif scope == 'book_vars':
            book_ids = self.plugin_action.book_vars.keys()
        elif scope == 'last_action_modified':
            last_step = chain.last_step
            last_step_start = chain.chain_links[chain.last_step]['start_time']
            book_ids = get_books_modified_since(db, last_step_start)
        elif scope == 'chain_modified':
                chain_start = chain.start_time
                book_ids = get_books_modified_since(db, chain_start)
        elif scope == 'search':
            search_text = settings['search_text']
            try:
                book_ids = db.data.search_getting_ids(search_text, '', use_virtual_library=False)
                if settings.get('restrict_to_selection', False):
                    selected_ids = set(ActionScope.get_book_ids(self, gui, settings, chain))
                    book_ids = list(set(book_ids).intersection(selected_ids))
            except ParseException:
                book_ids = []

        # discard book_ids not present in library
        book_ids = list(set(gui.current_db.all_ids()).intersection(set(book_ids)))

        if settings.get('vl'):
            vl_ids = self.get_current_restriction_book_ids()
            book_ids = list(set(vl_ids).intersection(set(book_ids)))

        return book_ids

    def validate(self, settings):
        gui = self.plugin_action.gui
        db = gui.current_db
        if not settings:
            return (_('Settings Error'), _('You must configure this scope before running it'))
        if settings['opt'] == 'search':
            search_text = settings['search_text']
            if not is_search_valid(db, search_text):
                return (_('Invalid search'), _(f'Search "{search_text}" is not valid'))
        return True

    def config_widget(self):
        return BasicScopeConfigWidget
