#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import QModelIndex, QObject

from calibre import prints
from calibre.constants import DEBUG


class ActionScope(QObject):

    name = 'Action Scope'

    '''
    This is the base class for all scope managers
    '''
    def __init__(self, plugin_action):
        QObject.__init__(self)
        self.plugin_action = plugin_action

    def config_widget(self):
        '''
        If you want your scope manager to have settings dialog, implement this method
        This should return a Qwidget (not dialog) with the following methods:
        [*] __init__(self, plugin_action)
        [*] save_settings(settings)
                This method is used to save the settings from the widget
                it should return a dictionary containing all the settings
        [*] load_settings(self, settings)
                This method is used to load the saved settings into the
                widget
        '''
        return None

    def validate(self, settings):
        '''
        Validate settings dict. This is called when you press the OK button in config dialog.
        changes are applied only if the this methods returns True.
        It is called also when verifying the chain validity on multiple occasions: startup,
        library change, chain dialog initialization .....
        If the setting in the dictionary are not valid, return a tuple
        of two strings (message, details), these will be displayed as a warning dialog to
        the user and the process will be aborted.
        '''
        return True

    def default_settings(self):
        '''
        default settings to be used if no settings are configured
        '''
        return {}

    def get_book_ids(self, gui, settings, chain):
        rows = gui.current_view().selectionModel().selectedRows()
        book_ids = [ gui.library_view.model().db.id(row.row()) for row in rows ]
        return book_ids


    def get_current_book_id(self, gui, settings, chain):
        dex = gui.library_view.currentIndex()
        if dex.isValid():
            book_id = gui.library_view.model().id(dex)
        else:
            book_id = None
        return book_id

    def get_current_restriction_book_ids(self):
        model = self.plugin_action.gui.library_view.model()
        rows = list(range(model.rowCount(QModelIndex())))
        book_ids = list(map(model.id, rows))
        return book_ids
