#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~
__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
from collections import OrderedDict
from functools import partial
import itertools, operator
import copy
import inspect

from qt.core import (QApplication, Qt, QTableView, QAbstractItemView, QModelIndex,
                     QAbstractProxyModel, QItemSelection, QMenu, QDialog, QAction,
                     QInputDialog)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import question_dialog, error_dialog, info_dialog, open_local_file
from calibre.utils.config import tweaks

import calibre_plugins.action_chains.config as cfg
from calibre_plugins.action_chains.gui.delegates import (
    ComboDelegate, ButtonDelegate, TreeComboDelegate, ImageComboDelegate)
from calibre_plugins.action_chains.gui.models import UP, DOWN, get_actual_index
from calibre_plugins.action_chains.gui import SettingsWidgetDialog, ScopeWidgetDialog
from calibre_plugins.action_chains.common_utils import ViewLog, get_icon, safe_name
from calibre_plugins.action_chains.conditions import ConditionsEval
from calibre_plugins.action_chains.chains import chains_config_from_archive
from calibre_plugins.action_chains.templates.dialogs import TemplateBox
from calibre_plugins.action_chains.templates.functions import ChainVariant
from calibre_plugins.action_chains.export_import import export_chains

try:
    load_translations()
except NameError:
    prints("ActionsChain::gui/views.py - exception when loading translations")

class TableView(QTableView):

    def __init__(self, parent):
        QTableView.__init__(self, parent)
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.setMouseTracking(True)
        self.setProperty('highlight_current_item', 150)
        self.verticalHeader().setDefaultSectionSize(30)
        self.verticalHeader().setVisible(True)
        self.verticalHeader().setSectionsMovable(True)
        self.horizontalHeader().setSectionsMovable(True)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self.horizontalHeader().setStretchLastSection(True)
        #self.column_header = HeaderView(Qt.Orientation.Horizontal, self)
        self.column_header = self.horizontalHeader()

    def _set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def source_model(self):
        model = QTableView.model(self)
        if isinstance(model, QAbstractProxyModel):
            return model.sourceModel()
        else:
            return model

    def set_current_row(self, row=0, select=True, for_sync=False):
        if row > -1 and row < self.model().rowCount(QModelIndex()):
            h = self.horizontalHeader()
            logical_indices = list(range(h.count()))
            logical_indices = [x for x in logical_indices if not
                    h.isSectionHidden(x)]
            pairs = [(x, h.visualIndex(x)) for x in logical_indices if
                    h.visualIndex(x) > -1]
            if not pairs:
                pairs = [(0, 0)]
            pairs.sort(key=lambda x: x[1])
            i = pairs[0][0]
            index = self.model().index(row, i)
            if for_sync:
                selm = self.selectionModel()
                selm.setCurrentIndex(index, selm.NoUpdate)
            else:
                self.setCurrentIndex(index)
                if select:
                    selm = self.selectionModel()
                    selm.select(index, selm.SelectionFlag.ClearAndSelect|selm.Rows)

    def select_rows(self, rows, change_current=True, scroll=True):
        rows = {x.row() if hasattr(x, 'row') else x for x in
            rows}
        rows = list(sorted(rows))
        if rows:
            row = rows[0]
            if change_current:
                self.set_current_row(row, select=False)
            if scroll:
                self.scroll_to_row(row)
        selm = self.selectionModel()
        sel = QItemSelection()
        max_col = self.model().columnCount(QModelIndex()) - 1
        # Create a range based selector for each set of contiguous rows
        # as supplying selectors for each individual row causes very poor
        # performance if a large number of rows has to be selected.
        for k, g in itertools.groupby(enumerate(rows), lambda i_x:i_x[0]-i_x[1]):
            group = list(map(operator.itemgetter(1), g))
            sel.merge(QItemSelection(self.model().index(min(group), 0),
                self.model().index(max(group), max_col)), selm.SelectionFlag.Select)
        selm.select(sel, selm.SelectionFlag.ClearAndSelect)
        return rows

    def scroll_to_row(self, row):
        if row > -1:
            h = self.horizontalHeader()
            for i in range(h.count()):
                if not h.isSectionHidden(i) and h.sectionViewportPosition(i) >= 0:
                    self.scrollTo(self.model().index(row, i), self.PositionAtCenter)
                    break

    def add_row(self):
        self.setFocus()
        # We will insert a blank row below the currently selected row
        row = self.selectionModel().currentIndex().row() + 1
        self.model().insertRow(row)
        self.scroll_to_row(row)

    def delete_rows(self):
        self.setFocus()
        selrows = self.selectionModel().selectedRows()
        selrows = sorted(selrows, key=lambda x: x.row())
        if len(selrows) == 0:
            return
        message = _('<p>Are you sure you want to delete this action?')
        if len(selrows) > 1:
            message = _(f'<p>Are you sure you want to delete the selected {len(selrows)} actions?')
        if not question_dialog(self, _('Are you sure?'), message, show_copy_button=False):
            return
        first_sel_row = selrows[0].row()
        for selrow in reversed(selrows):
            self.model().removeRow(selrow.row())
        if first_sel_row < self.model().rowCount(QModelIndex()):
            self.setCurrentIndex(self.model().index(first_sel_row, 0))
            self.scroll_to_row(first_sel_row)
        elif self.model().rowCount(QModelIndex()) > 0:
            self.setCurrentIndex(self.model().index(first_sel_row - 1, 0))
            self.scroll_to_row(first_sel_row - 1)

    def move_rows(self, direction=DOWN):
        self.setFocus()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        current_index = selm.currentIndex()
        if len(selrows) == 0:
            return
        # make sure there is room to move UP/DOWN
        if direction == DOWN:
            boundary_selrow = selrows[-1].row()
            if boundary_selrow >= self.model().rowCount(QModelIndex()) - 1:
                return
        elif direction == UP:
            boundary_selrow = selrows[0].row()
            if boundary_selrow <= 0:
                return
        
        rows = [row.row() for row in selrows]
        ##self.source_model().layoutAboutToBeChanged.emit()
        self.source_model().move_rows(rows, direction=direction)
        
        # reset selections and scroll
        rows = [row+direction for row in rows]
        scroll_to_row = boundary_selrow + direction
        self.select_rows(rows)
        #self.set_current_row(scroll_to_row)
        self.scroll_to_row(scroll_to_row)

    def show_column_header_context_menu(self, pos):
        srcm = self.source_model()
        idx = self.column_header.logicalIndexAt(pos)
        col = None
        if idx > -1 and idx < len(srcm.col_map):
            col = srcm.col_map[idx]
            name = str(self.model().headerData(idx, Qt.Orientation.Horizontal, Qt.ItemDataRole.DisplayRole) or '')
            self.column_header_context_menu = self.create_header_context_menu(col, name)
            self.column_header_context_menu.popup(self.column_header.mapToGlobal(pos))

    def create_header_context_menu(self, col, name):
        ans = QMenu(self)
        srcm = self.source_model()
        handler = partial(self.column_header_context_handler, column=col)
        if col in srcm.optional_cols:
            ans.addAction(_('Hide column %s') % name, partial(handler, action='hide'))

        hidden_cols = {srcm.col_map[i]: i for i in range(self.column_header.count())
                       if self.column_header.isSectionHidden(i)}

        hidden_cols = {k:v for k,v in hidden_cols.items() if k in srcm.optional_cols}

        ans.addSeparator()
        if hidden_cols:
            m = ans.addMenu(_('Show column'))
            hcols = [(hcol, str(self.model().headerData(hidx, Qt.Orientation.Horizontal, Qt.ItemDataRole.DisplayRole) or ''))
                     for hcol, hidx in hidden_cols.items()]
            hcols.sort()
            for hcol, hname in hcols:
                m.addAction(hname, partial(handler, action='show', column=hcol))
        return ans


    def column_header_context_handler(self, action=None, column=None):
        if not action or not column:
            return
        try:
            idx = self.col_map.index(column)
        except:
            return
        h = self.column_header

        if action == 'hide':
            if h.hiddenSectionCount() >= h.count():
                return error_dialog(self, _('Cannot hide all columns'), _(
                    'You must not hide all columns'), show=True)
            h.setSectionHidden(idx, True)
        elif action == 'show':
            h.setSectionHidden(idx, False)
            if h.sectionSize(idx) < 3:
                sz = h.sectionSizeHint(idx)
                h.resizeSection(idx, sz)

    def get_state(self):
        h = self.column_header
        cm = self.source_model().col_map
        state = {}
        state['hidden_columns'] = [cm[i] for i in range(h.count())
                if h.isSectionHidden(i)]
        state['column_positions'] = {}
        state['column_sizes'] = {}
        for i in range(h.count()):
            name = cm[i]
            state['column_positions'][name] = h.visualIndex(i)
            state['column_sizes'][name] = h.sectionSize(i)
        return state

    def apply_state(self, state):
        h = self.column_header
        cmap = {}
        hidden = state.get('hidden_columns', [])
        for i, c in enumerate(self.source_model().col_map):
            cmap[c] = i
            h.setSectionHidden(i, c in hidden)

        positions = state.get('column_positions', {})
        pmap = {}
        for col, pos in positions.items():
            if col in cmap:
                pmap[pos] = col
        for pos in sorted(pmap.keys()):
            col = pmap[pos]
            idx = cmap[col]
            current_pos = h.visualIndex(idx)
            if current_pos != pos:
                h.moveSection(current_pos, pos)

        sizes = state.get('column_sizes', {})
        for col, size in sizes.items():
            if col in cmap:
                sz = sizes[col]
                if sz < 3:
                    sz = h.sectionSizeHint(cmap[col])
                h.resizeSection(cmap[col], sz)

        for i in range(h.count()):
            if not h.isSectionHidden(i) and h.sectionSize(i) < 3:
                sz = h.sectionSizeHint(i)
                h.resizeSection(i, sz)

class ActionsTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.doubleClicked.connect(self._on_double_clicked)
        self.column_header.setSectionsClickable(True)
        self.column_header.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.column_header.customContextMenuRequested.connect(partial(self.show_column_header_context_menu))
        #self.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.setContextMenuPolicy(Qt.ContextMenuPolicy.DefaultContextMenu)
        self.customContextMenuRequested.connect(self.show_context_menu)

    def set_model(self, _model):
        self.setModel(_model)
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        #self.actions_delegate = ComboDelegate(self, self.plugin_action.actions.keys())
        self.actions_delegate = TreeComboDelegate(self, self.action_tree())
        self.setItemDelegateForColumn(self.col_map.index('action_name'), self.actions_delegate)

        self.button_delegate = ButtonDelegate(self)
        self.setItemDelegateForColumn(self.col_map.index('action_settings'), self.button_delegate)
        self.button_delegate.clicked.connect(self._on_button_clicked)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def action_tree(self):
        '''
        Build dictionary containing action tree where:
        1. Builtin actions are top level nodes
        2. User Action are inside a node called User Actions
        3. Actions imported from another plugins are under node
           for each plugin that has action. And all those are 
           under a note called 'Other Plugins Actions'
        This is a dictionary of nested dictionaries. Even actions
        are nested dictionaries in the form of 'action_name': {}
        '''
        for action in self.plugin_action.user_actions.values():
            setattr(action, '_display_tree', ['User Actions'] + getattr(action, 'display_tree', []))

        for action in self.plugin_action.imported_actions.values():
            source_plugin = getattr(action, '_source_plugin')
            tree_path = ['Imported From Other Plugins'] + getattr(action, 'display_tree', [])
            if source_plugin:
                tree_path.insert(1, source_plugin)
            setattr(action, '_display_tree', tree_path)

        d = OrderedDict()
        for name, action in self.plugin_action.actions.items():
            parent = d
            tree_path = getattr(action, '_display_tree', []) + [name]
            for node in tree_path:
                if not parent.get(node):
                    parent[node] = {}
                parent = parent[node]
        return d            

    def copy_row(self):
        self.setFocus()
        srcm = self.source_model()
        selm = self.selectionModel()
        index = selm.currentIndex()
        self.model().layoutAboutToBeChanged.emit()
        new_row = copy.deepcopy(srcm.chain_links[index.row()])
        # We will insert the new row below the currently selected row
        srcm.chain_links.insert(index.row()+1, new_row)
        self.model().layoutChanged.emit()
        self.select_rows([index.row()+1])

    def contextMenuEvent(self, event):
        self.show_context_menu(event)

    def show_context_menu(self, event):
        model = self.model()
        self.context_menu = m = self.create_context_menu()
        m.popup(event.globalPos())

    def create_context_menu(self):
        all_have_conditions, all_scope_enabled, all_have_scopes = self.is_conditions_and_scopes(self.selectionModel().selectedRows())
        m = QMenu(self)
        add_scope = QAction(get_icon('images/scope.png'), _('Add/Modify scope...'), self)
        add_scope.triggered.connect(self.set_scopes_for_selection)
        remove_scope = QAction(get_icon('images/no_scope.png'), _('Remove scope...'), self)
        remove_scope.triggered.connect(self.remove_scopes_for_selection)
        add_conditions = QAction(get_icon('images/condition.png'), _('Add/Modify condition(s)...'), self)
        add_conditions.triggered.connect(self.set_conditions_for_selection)
        remove_conditions = QAction(get_icon('images/no_condition.png'), _('Remove condition(s)...'), self)
        remove_conditions.triggered.connect(self.remove_conditions_for_selection)
        if not all_scope_enabled:
            add_scope.setEnabled(False)
            remove_scope.setEnabled(False)
        if not all_have_scopes:
            remove_scope.setEnabled(False)
        if not all_have_conditions:
            remove_conditions.setEnabled(False)
        m.addAction(add_scope)
        m.addAction(remove_scope)
        m.addSeparator()
        m.addAction(add_conditions)
        m.addAction(remove_conditions)
        return m

    def is_conditions_and_scopes(self, rows):
        srcm = self.source_model()
        row_count = len(rows)
        conditions, scope_support, scope = 0, 0, 0
        for row in rows:
            chain_link = srcm.chain_links[row.row()]
            link_has_conditions, link_support_scopes, link_has_scope = srcm.scope_and_conditions(chain_link)
            if link_has_conditions:
                conditions += 1
            if link_support_scopes:
                scope_support += 1
            if link_has_scope:
                scope += 1
        all_have_conditions = row_count and (conditions == row_count)
        all_scope_enabled = row_count and (scope_support == row_count)
        all_have_scopes = row_count and (scope == row_count)
        return all_have_conditions, all_scope_enabled, all_have_scopes


    def _on_button_clicked(self, index):
        srcm = self.source_model()
        col_name = srcm.col_map[index.column()]
        if col_name == 'action_settings':
            chain_link = srcm.chain_links[index.row()]
            action_name = chain_link['action_name']
            if not action_name:
                # user clicking on setting before choosing action
                error_dialog(
                    self,
                    _('No Action Selected'),
                    _('You must choose an action first'),
                    show=True
                )
                return
            action = self.plugin_action.actions[action_name]
            action_settings = chain_link.get('action_settings')
            if not action_settings:
                action_settings = action.default_settings()
            config_widget = action.config_widget()
            if config_widget:
                name = f'ActionsChain::{action_name}'
                title = f'{action_name}'
                if issubclass(config_widget, QDialog):
                    # config_widget is a dialog
                    d = config_widget(self, self.plugin_action, action, name, title)
                else:
                    # config_widget is a qwidget
                    d = SettingsWidgetDialog(name, self, self.plugin_action, config_widget, action, srcm.chain_name, srcm.chains_config, title)
                # inject copy of chain data into the settings dialog, for internal use only
                d._chain_link = copy.deepcopy(chain_link)
                if action_settings:
                    d.load_settings(action_settings)
                if d.exec_() == d.Accepted:
                    chain_link['action_settings'] = d.settings
                    # reset any previous error if present
                    chain_link['errors'] = ''
                    self.model().dataChanged.emit(index, index)

    def _on_double_clicked(self, index):
        srcm = self.source_model()
        col_name = srcm.col_map[index.column()]
        if col_name == 'errors':
            chain_link = srcm.chain_links[index.row()]
            details = chain_link.get('errors', '')
            self._view_error_details(details)
        elif col_name == 'conditions':
            self.set_conditions_for_selection()
        elif col_name == 'scope':
            have_conditions, scope_enabled, have_scopes = self.is_conditions_and_scopes([index])
            if scope_enabled:
                self.set_scopes_for_selection()
            

    def _view_error_details(self, details):
        ViewLog(_('Errors details'), details, self)

    def set_conditions_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        if len(selrows) != 1:
            condition_settings = {}
        else:
            selrow = selrows[0]
            row = selrow.row()
            chain_link = srcm.chain_links[row]
            condition_settings = chain_link.get('condition_settings', {})
        d = ConditionsEval(self, self.plugin_action, condition_settings, source='action')
        if d.exec_() == d.Accepted:
            new_settings = {
                'template': d.template,
                'value': d.value,
                'cmp_type': d.cmp_type,
                'datatype': d.datatype,
                'icon': d.icon,
                'tooltip': d.tooltip,
                'affect_validation': d.affect_validation
            }
            for selrow in selrows:
                row = selrow.row()
                chain_link = srcm.chain_links[row]
                chain_link['condition_settings'] = new_settings
        self.model().layoutChanged.emit()

    def remove_conditions_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        for selrow in selrows:
            row = selrow.row()
            chain_link = srcm.chain_links[row]
            try:
                del chain_link['condition_settings']
            except:
                pass
        self.model().layoutChanged.emit()

    def set_scopes_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        if len(selrows) != 1:
            scope_settings = {}
            title = _('Scope settings')
        else:
            selrow = selrows[0]
            row = selrow.row()
            chain_link = srcm.chain_links[row]
            scope_settings = chain_link.get('scope_settings', {})
        d = ScopeWidgetDialog('Action Chains::scope-settings', self, self.plugin_action)
        d.load_settings(scope_settings)
        if d.exec_() == d.Accepted:
            for selrow in selrows:
                # check if action supports scope managers
                row = selrow.row()
                chain_link = srcm.chain_links[row]
                action = chain_link['action_name']
                if getattr(self.plugin_action.actions.get(action), 'support_scopes', False):
                    row = selrow.row()
                    chain_link = srcm.chain_links[row]
                    chain_link['scope_settings'] = d.settings
        self.model().layoutChanged.emit()

    def remove_scopes_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        for selrow in selrows:
            row = selrow.row()
            chain_link = srcm.chain_links[row]
            try:
                del chain_link['scope_settings']
            except:
                pass
        self.model().layoutChanged.emit()

class MenusTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.icon_cache = self.plugin_action.icon_cache
        self.column_header.setSectionsClickable(True)
        self.horizontalHeader().setStretchLastSection(False)
        self.column_header.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.column_header.customContextMenuRequested.connect(partial(self.show_column_header_context_menu))
        self.customContextMenuRequested.connect(self.show_context_menu)
        self.doubleClicked.connect(self._on_double_clicked)

    def set_model(self, proxy_model):
        self.setModel(proxy_model)
        self.proxy_model = proxy_model
        source_model = proxy_model.sourceModel()
        source_model.error.connect(lambda *args: error_dialog(self, *args, show=True))
        self.col_map = source_model.col_map

        # Hide columns
        for col_name in source_model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        self.button_delegate = ButtonDelegate(self)
        self.setItemDelegateForColumn(self.col_map.index('settings'), self.button_delegate)
        self.button_delegate.clicked.connect(self._on_button_clicked)

        #self.icon_delegate = ImageComboDelegate(self, self.icon_cache)
        #self.setItemDelegateForColumn(self.col_map.index('image'), self.icon_delegate)

        #for r in range(_model.rowCount(QModelIndex())):
            #res = self.openPersistentEditor(_model.index(r, self.col_map.index('image')))

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in source_model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    #def add_row(self):
        #self.setFocus()
        ## We will insert a blank row below the currently selected row
        #row = self.selectionModel().currentIndex().row() + 1
        #self.model().insertRow(row)
        #self.scroll_to_row(row)
        #self.openPersistentEditor(self.model().index(row, self.col_map.index('image')))

    def append_data(self, chains_config):
        added = []
        failed = []
        srcm = self.source_model()
        names = srcm.get_names()
        srcm.layoutAboutToBeChanged.emit()
        for chain_config in chains_config:
            chain_name = chain_config['menuText']
            if chain_config['menuText'] in names:
                failed.append(chain_name)
                if DEBUG:
                    prints(f'Action chains: cannot import chain ({chain_name}). Name already exists')
            else:
                srcm.chains_config.append(chain_config)
                added.append(chain_name)
        srcm.layoutChanged.emit()
        return added, failed

    def get_selected_data(self):
        chains_config = []
        srcm = self.source_model()
        for selrow in self.selectionModel().selectedRows():
            row = get_actual_index(selrow).row()
            chains_config.append(srcm.chains_config[row])
        return chains_config

    def copy_row(self):
        self.setFocus()
        srcm = self.source_model()
        selm = self.selectionModel()
        index = selm.currentIndex()
        actual_index = get_actual_index(index)
        srcm.layoutAboutToBeChanged.emit()
        new_row = copy.deepcopy(srcm.chains_config[actual_index.row()])
        # change name
        if new_row['menuText']:
            all_names = srcm.get_names()
            new_name = new_row['menuText'] + ' (copy)'
            new_name = safe_name(new_name, all_names)
        else:
            new_name = new_row['menuText']
        new_row['menuText'] = new_name
        # We will insert a blank row below the currently selected row
        srcm.chains_config.insert(actual_index.row()+1, new_row)
        srcm.layoutChanged.emit()
        self.select_rows([index.row()+1])

    def contextMenuEvent(self, event):
        self.show_context_menu(event)

    def show_context_menu(self, event):
        model = self.model()
        self.context_menu = m = self.create_context_menu()
        m.popup(event.globalPos())

    def create_context_menu(self):
        all_have_conditions, all_have_variants = self.has_conditions_and_variants(self.selectionModel().selectedRows())
        m = QMenu(self)

        act_add_image = QAction(get_icon('images/image_add.png'), _('&Add image...'), m)
        act_add_image.triggered.connect(self.plugin_action.icon_cache.pick_icon)
        m.addAction(act_add_image)
        act_open = QAction(get_icon('document_open.png'), _('&Open images folder'), m)
        act_open.triggered.connect(partial(self.open_images_folder, self.plugin_action.resources_dir))
        m.addAction(act_open)
        m.addSeparator()
        act_import = QAction(get_icon('images/import.png'), _('&Import...'), m)
        act_import.triggered.connect(self.import_menus)
        m.addAction(act_import)
        act_export = QAction(get_icon('images/export.png'), _('&Export...'), m)
        act_export.triggered.connect(self.export_chains)
        m.addAction(act_export)
        m.addSeparator()

        add_conditions = QAction(get_icon('images/condition.png'), _('Add/Modify condition(s)...'), self)
        add_conditions.triggered.connect(self.set_conditions_for_selection)
        remove_conditions = QAction(get_icon('images/no_condition.png'), _('Remove condition(s)...'), self)
        remove_conditions.triggered.connect(self.remove_conditions_for_selection)
        if not all_have_conditions:
            remove_conditions.setEnabled(False)
        m.addAction(add_conditions)
        m.addAction(remove_conditions)
        
        if tweaks.get('action_chains_experimental', False):
            m.addSeparator()
            chain_variations = QAction(get_icon('images/multiple.png'), _('Chain Variations...'), self)
            chain_variations.triggered.connect(self.set_variations_for_selection)
            chain_variations.setEnabled(len(self.selectionModel().selectedRows()) == 1)
            m.addAction(chain_variations)

            remove_chain_variations = QAction(get_icon('minus.png'), _('Remove Chain Variations...'), self)
            remove_chain_variations.triggered.connect(self.remove_variations_for_selection)
            remove_chain_variations.setEnabled(all_have_variants)
            m.addAction(remove_chain_variations)
        return m

    def has_conditions_and_variants(self, rows):
        srcm = self.source_model()
        row_count = len(rows)
        conditions = 0
        variants = 0
        for row in rows:
            actual_index = get_actual_index(row)
            chain_config = srcm.chains_config[actual_index.row()]
            link_has_conditions = bool(chain_config.get('condition_settings', {}).get('template'))
            link_has_variants = bool(chain_config.get('variation_settings', {}).get('template'))
            if link_has_conditions:
                conditions += 1
            if link_has_variants:
                variants += 1
        all_have_conditions = row_count and (conditions == row_count)
        all_have_variants = row_count and (variants == row_count)
        return all_have_conditions, all_have_variants

    def _on_button_clicked(self, index):
        srcm = self.source_model()
        chain_config = srcm.chains_config[index.row()]
        col_name = srcm.col_map[index.column()]
        if col_name == 'settings':
            from calibre_plugins.action_chains.gui.actions_dialog import ActionsDialog
            chain_name = chain_config['menuText']
            if chain_name:
                d = ActionsDialog(self, self.plugin_action, copy.deepcopy(srcm.chains_config), chain_name)
                if d.exec_() == d.Accepted:
                    chain_config['chain_settings'] = d.chain_settings
                    chain_config['image'] = d.icon
                    self.model().dataChanged.emit(self.model().index(index.row(), 0), index)

    def _on_double_clicked(self, index):
        srcm = self.source_model()
        col_name = srcm.col_map[index.column()]
        if col_name == 'conditions':
            self.set_conditions_for_selection()
        elif col_name == 'active':
            if tweaks.get('action_chains_experimental', False):
                # only open dialog if there is an icon
                if self.model().data(index, Qt.DecorationRole):
                    self.set_variations_for_selection()

    def set_conditions_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        if len(selrows) != 1:
            condition_settings = {}
        else:
            selrow = selrows[0]
            row = get_actual_index(selrow).row()
            chain_config = srcm.chains_config[row]
            condition_settings = chain_config.get('condition_settings', {})
        d = ConditionsEval(self, self.plugin_action, condition_settings)
        if d.exec_() == d.Accepted:
            new_settings = {
                'template': d.template,
                'value': d.value,
                'cmp_type': d.cmp_type,
                'datatype': d.datatype,
                'icon': d.icon,
                'tooltip': d.tooltip,
                'affect_menu': d.affect_menu
            }
            for selrow in selrows:
                row = get_actual_index(selrow).row()
                chain_config = srcm.chains_config[row]
                chain_config['condition_settings'] = new_settings
        self.model().layoutChanged.emit()

    def remove_conditions_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        for selrow in selrows:
            row = get_actual_index(selrow).row()
            chain_config = srcm.chains_config[row]
            try:
                del chain_config['condition_settings']
            except:
                pass
        self.model().layoutChanged.emit()

    def set_variations_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        if len(selrows) != 1:
            return
        else:
            selrow = selrows[0]
            row = get_actual_index(selrow).row()
            chain_config = srcm.chains_config[row]
            variant_config = chain_config.get('variation_settings', {})
            template = variant_config.get('template', '')

        # must be initialized here to create a new dictionary
        chain_variant_function = ChainVariant(self.plugin_action)

        # We will add a special function, so we copy first to avoid adding it other places in the plugin
        template_functions = copy.copy(self.plugin_action.template_functions)

        # add special function
        template_functions[chain_variant_function.name] = chain_variant_function

        tooltip = _('You can add multiple menu entries for this chain by using the add_chain_variant() '
                    'function. Each chain variant should have a different argument. The argument will '
                    'stored as a chain variable that can be retrieved with globals() function. Call '
                    'add_chain_variant() once for each new menu entry you want to add.')
        
        d = TemplateBox(self,
                        self.plugin_action,
                        template_text=template,
                        all_functions = template_functions,
                        placeholder_text=_('Use add_chain_variant() to create chain variants.'))
        d.textbox.setToolTip(tooltip)
        d.setWindowTitle(_('Add chain variant(s)'))
        if d.exec_():
            if d.template:
                variant_config['template'] = d.template
                chain_config['variation_settings'] = variant_config

    def remove_variations_for_selection(self):
        srcm = self.source_model()
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        for selrow in selrows:
            row = get_actual_index(selrow).row()
            chain_config = srcm.chains_config[row]
            try:
                del chain_config['variation_settings']
            except:
                pass
        self.model().layoutChanged.emit()

    def open_images_folder(self, path):
        if not os.path.exists(path):
            if not question_dialog(self, _('Are you sure?'), '<p>'+
                    _('Folder does not yet exist. Do you want to create it?<br>%s') % path,
                    show_copy_button=False):
                return
            os.makedirs(path)
        open_local_file(path)

    def import_menus(self):
        chains_config = chains_config_from_archive(self.plugin_action.gui, self.plugin_action, add_resources=True)
        if not chains_config:
            return
        # Now insert the menus into the table
        added, failed = self.append_data(chains_config)
        det = _('%d menu items added') % len(added)
        if len(failed) > 0:
            det += _(', failed to add %d menu items') % len(failed)
        info_dialog(self, _('Import completed'), det,
                    show=True, show_copy_button=False)

    def export_chains(self):
        chains_config = self.get_selected_data()
        export_chains(self.plugin_action, self.plugin_action.gui, chains_config)

    def keyPressEvent(self, ev):
        selm = self.selectionModel()
        selrows = selm.selectedRows()
        col = self.col_map.index('subMenu')
        if ev.key() == Qt.Key.Key_F2 and (selm.currentIndex().column() == col) and len(selrows) > 1:
            new_name, ok = QInputDialog.getText(self, _('New submenu:'),
                                                _('New submenu:'), text='')
            new_name = str(new_name).strip()
            if ok:
                for selrow in selrows:
                    row = selrow.row()
                    index = self.model().index(row, col, QModelIndex())
                    self.model().setData(index, new_name, Qt.EditRole)
            return
        return TableView.keyPressEvent(self, ev)

class EventMembersTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        #self.setShowGrid(False)

    def set_model(self, _model):
        self.setModel(_model)
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        all_chain_names = [chain_config['menuText'] for chain_config in cfg.get_chains_config()]

        self.members_delegate = ComboDelegate(self, all_chain_names)
        self.setItemDelegateForColumn(self.col_map.index('chain_name'), self.members_delegate)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

class EventsTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.horizontalHeader().setStretchLastSection(False)
        #self.setShowGrid(False)

    def set_model(self, _model):
        self.setModel(_model)
        _model.error.connect(lambda *args: error_dialog(self, *args, show=True))
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        #self.events_delegate = ComboDelegate(self, self.plugin_action.events.keys())
        self.events_delegate = TreeComboDelegate(self, self.event_tree())
        self.setItemDelegateForColumn(self.col_map.index('event_name'), self.events_delegate)

        self.button_delegate = ButtonDelegate(self)
        self.setItemDelegateForColumn(self.col_map.index('event_settings'), self.button_delegate)
        self.button_delegate.clicked.connect(self._on_button_clicked)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def event_tree(self):
        '''
        Build dictionary containing event tree where:
        1. Builtin events are top level nodes
        2. User Events are inside a node called User Actions
        3. Events imported from another plugins are under node
           for each plugin that has events. And all those are 
           under a note called 'Other Plugins Events'
        This is a dictionary of nested dictionaries. Even events
        are nested dictionaries in the form of 'event_name': {}
        '''
        for event in self.plugin_action.user_events.values():
            setattr(event, '_display_tree', ['User Events'] + getattr(event, 'display_tree', []))

        for event in self.plugin_action.imported_events.values():
            source_plugin = getattr(event, '_source_plugin')
            tree_path = ['Imported From Other Plugins'] + getattr(event, 'display_tree', [])
            if source_plugin:
                tree_path.insert(1, source_plugin)
            setattr(event, '_display_tree', tree_path)

        d = OrderedDict()
        for name, event in self.plugin_action.events.items():
            parent = d
            tree_path = getattr(event, '_display_tree', []) + [name]
            for node in tree_path:
                if not parent.get(node):
                    parent[node] = {}
                parent = parent[node]
        return d

    def _on_button_clicked(self, index):
        srcm = self.source_model()
        event_config = srcm.events_config[index.row()]
        col_name = srcm.col_map[index.column()]
        if col_name == 'event_settings':
            from calibre_plugins.action_chains.gui.events_dialogs import EventMembersDialog
            event_settings = copy.deepcopy(event_config)
            event_name = event_config['event_name']
            if event_name:
                d = EventMembersDialog(self, self.plugin_action, event_config)
                if d.exec_() == d.Accepted:
                    event_config['event_settings'] = d.event_settings
                    self.model().dataChanged.emit(self.model().index(index.row(), 0), index)

class EventVariantsTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.horizontalHeader().setStretchLastSection(False)
        self.doubleClicked.connect(self._on_double_clicked)

    def set_model(self, _model):
        self.setModel(_model)
        _model.error.connect(lambda *args: error_dialog(self, *args, show=True))
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        #events_supporting_variants = [event.name for event in self.plugin_action.events.values() if getattr(event, 'support_variants', False)]
        from calibre_plugins.action_chains.events import get_variant_classes
        events_supporting_variants = [ cls.name for cls in get_variant_classes(self.plugin_action) ]
        self.events_delegate = ComboDelegate(self, events_supporting_variants)
        self.setItemDelegateForColumn(self.col_map.index('parent_event'), self.events_delegate)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)


    def _on_double_clicked(self, index):
        srcm = self.source_model()
        col_name = srcm.col_map[index.column()]
        if col_name == 'errors':
            variant_config = srcm.variants_config[index.row()]
            details = variant_config.get('errors', '')
            self._view_error_details(details)

    def _view_error_details(self, details):
        ViewLog(_('Errors details'), details, self)
