#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

# original: src/calibre/gui2/custom_column_widgets.py
# [*] modified to deal with standard as well as custom columns
# [*] replaced polyglot with six for compatibility withe earlier calibre versions
# [*] replaced label_string
# [*] Added setters to BulkSeries, BulkText
# [*] Modify Text, BulkText to handle identifiers
# [*] Modified BulkText to handle language field
# [*] Added new classes: PredefinedDateTime, BulkSimpleText, BulkLongText, BulkComment, Marked, BulkMarked
# [*] removed non-used functions: field_sort_key, populate_metadata_page
# [*] make QWidget superclass for Base() and added initControls
# [*] remove the overriding gui_val from BulkBase and rely on the inherited method from Base to prevent caching gui value which affects cfg widgets when the validation fails and we try to change the values, the cached values are returned instead

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>,  2020 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
import copy
from functools import partial
from collections import OrderedDict

from qt.core import (QApplication, Qt, QComboBox, QLabel, QSpinBox, QDoubleSpinBox,
                     QDateTime, QGroupBox, QVBoxLayout, QSizePolicy, QGridLayout, QUrl,
                     QSpacerItem, QIcon, QCheckBox, QWidget, QHBoxLayout, QLineEdit,
                     QMessageBox, QToolButton, QPlainTextEdit, QRadioButton, QPushButton, QSize)

from calibre import prints
from calibre.utils.date import now, as_local_time, as_utc, internal_iso_format_string, parse_date, is_date_undefined
from calibre.gui2.complete2 import EditWithComplete
from calibre.gui2.comments_editor import Editor as CommentsEditor
from calibre.gui2 import UNDEFINED_QDATETIME, error_dialog, elided_text
from calibre.gui2.covers import CoverSettingsWidget
from calibre.gui2.dialogs.tag_editor import TagEditor
from calibre.gui2.widgets2 import Dialog
from calibre.utils.config import tweaks
from calibre.utils.icu import sort_key
from calibre.library.comments import comments_to_html
from calibre.gui2.library.delegates import ClearingDoubleSpinBox, ClearingSpinBox
from calibre.gui2.widgets2 import RatingEditor, DateTimeEdit as DateTimeEditBase
from calibre.gui2.languages import LanguagesEdit

from calibre_plugins.action_chains.common_utils import get_icon, qt_from_dt, qt_to_dt
from calibre_plugins.action_chains.database import (all_field_names, set_marked_for_book,
                    all_marked_values, bulk_modify_marked, bulk_modify_multiple, bulk_modify_identifiers)

try:
    load_translations()
except NameError:
    prints("ActionChains::gui/single_field_wigets.py - exception when loading translations")

def safe_disconnect(signal):
    try:
        signal.disconnect()
    except Exception:
        pass

label_string = lambda x: x

def get_tooltip(col_metadata, add_index=False):
    key = col_metadata['label'] + ('_index' if add_index else '')
    label = col_metadata['name'] + (_(' index') if add_index else '')
    description = col_metadata.get('display', {}).get('description', '')
    return f"{label} (#{key}){':' if description else ''} {description}".strip()

class Base(QWidget):

    def __init__(self, plugin_action, col_name, parent=None):
        QWidget.__init__(self)
        self.col_name = col_name
        self.plugin_action = plugin_action
        self.gui = self.plugin_action.gui
        self.db = self.gui.current_db
        self.col_metadata = self.db.field_metadata.all_metadata()[col_name]
        self.initial_val = self.widgets = None
        self.signals_to_disconnect = []
        #self.setup_ui(parent)
        self.setup_ui(self)
        self.initControls()
        description = get_tooltip(self.col_metadata)
        try:
            self.widgets[0].setToolTip(description)
            self.widgets[1].setToolTip(description)
        except:
            try:
                self.widgets[1].setToolTip(description)
            except:
                pass

    def initControls(self):
        layout = QVBoxLayout()
        self.setLayout(layout)
        for w in self.widgets:
            layout.addWidget(w)
        self.adjustSize()
        layout.addStretch(1)

    def initialize(self, book_id):
        val = self.db.new_api.field_for(self.col_name, book_id)
        val = self.normalize_db_val(val)
        self.setter(val)
        self.initial_val = self.current_val  # self.current_val might be different from val thanks to normalization

    @property
    def current_val(self):
        return self.normalize_ui_val(self.gui_val)

    @property
    def gui_val(self):
        return self.getter()

    def commit(self, book_id, notify=False):
        val = self.current_val
        if val != self.initial_val:
            return self.db.new_api.set_field(self.col_name, {book_id: val}, allow_case_change=True)
        else:
            return set()

    def apply_to_metadata(self, mi):
        mi.set(self.col_name, self.current_val)

    def normalize_db_val(self, val):
        return val

    def normalize_ui_val(self, val):
        return val

    def break_cycles(self):
        self.db = self.widgets = self.initial_val = None
        for signal in self.signals_to_disconnect:
            safe_disconnect(signal)
        self.signals_to_disconnect = []

    def connect_data_changed(self, slot):
        pass

    # Add new methods
    def is_multiple(self):
        if self.col_metadata['is_multiple']:
            return True
        return False

    def is_names(self):
        if self.col_name in ['authors']:
            return True
        if self.col_metadata['display'].get('is_names', False):
            return True
        return False

class SimpleText(Base):

    def setup_ui(self, parent):
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent),
                        QLineEdit(parent)]

    def setter(self, val):
        self.widgets[1].setText(str(val or ''))

    def getter(self):
        return self.widgets[1].text().strip()

    def connect_data_changed(self, slot):
        self.widgets[1].textChanged.connect(slot)
        self.signals_to_disconnect.append(self.widgets[1].textChanged)


class LongText(Base):

    def setup_ui(self, parent):
        self._box = QGroupBox(parent)
        self._box.setTitle(label_string(self.col_metadata['name']))
        self._layout = QVBoxLayout()
        self._tb = QPlainTextEdit(self._box)
        self._tb.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Minimum)
        self._layout.addWidget(self._tb)
        self._box.setLayout(self._layout)
        self.widgets = [self._box]

    def setter(self, val):
        self._tb.setPlainText(str(val or ''))

    def getter(self):
        return self._tb.toPlainText()

    def connect_data_changed(self, slot):
        self._tb.textChanged.connect(slot)
        self.signals_to_disconnect.append(self._tb.textChanged)


class Bool(Base):

    def setup_ui(self, parent):
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent)]
        w = QWidget(parent)
        self.widgets.append(w)

        l = QHBoxLayout()
        l.setContentsMargins(0, 0, 0, 0)
        w.setLayout(l)
        self.combobox = QComboBox(parent)
        l.addWidget(self.combobox)

        c = QToolButton(parent)
        c.setText(_('Yes'))
        l.addWidget(c)
        c.clicked.connect(self.set_to_yes)

        c = QToolButton(parent)
        c.setText(_('No'))
        l.addWidget(c)
        c.clicked.connect(self.set_to_no)

        if self.db.new_api.pref('bools_are_tristate'):
            c = QToolButton(parent)
            c.setText(_('Clear'))
            l.addWidget(c)
            c.clicked.connect(self.set_to_cleared)

        w = self.combobox
        items = [_('Yes'), _('No'), _('Undefined')]
        icons = [I('ok.png'), I('list_remove.png'), I('blank.png')]
        if not self.db.new_api.pref('bools_are_tristate'):
            items = items[:-1]
            icons = icons[:-1]
        for icon, text in zip(icons, items):
            w.addItem(QIcon(icon), text)

    def setter(self, val):
        val = {None: 2, False: 1, True: 0}[val]
        if not self.db.new_api.pref('bools_are_tristate') and val == 2:
            val = 1
        self.combobox.setCurrentIndex(val)

    def getter(self):
        val = self.combobox.currentIndex()
        return {2: None, 1: False, 0: True}[val]

    def set_to_yes(self):
        self.combobox.setCurrentIndex(0)

    def set_to_no(self):
        self.combobox.setCurrentIndex(1)

    def set_to_cleared(self):
        self.combobox.setCurrentIndex(2)

    def connect_data_changed(self, slot):
        self.combobox.currentTextChanged.connect(slot)
        self.signals_to_disconnect.append(self.combobox.currentTextChanged)


class Int(Base):

    def setup_ui(self, parent):
        self.was_none = False
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent),
                ClearingSpinBox(parent)]
        w = self.widgets[1]
        w.setRange(-1000000, 100000000)
        w.setSpecialValueText(_('Undefined'))
        w.setSingleStep(1)
        w.valueChanged.connect(self.valueChanged)

    def setter(self, val):
        if val is None:
            val = self.widgets[1].minimum()
        self.widgets[1].setValue(val)
        self.was_none = val == self.widgets[1].minimum()

    def getter(self):
        val = self.widgets[1].value()
        if val == self.widgets[1].minimum():
            val = None
        return val

    def valueChanged(self, to_what):
        if self.was_none and to_what == -999999:
            self.setter(0)
        self.was_none = to_what == self.widgets[1].minimum()

    def connect_data_changed(self, slot):
        self.widgets[1].valueChanged.connect(slot)
        self.signals_to_disconnect.append(self.widgets[1].valueChanged)


class Float(Int):

    def setup_ui(self, parent):
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent),
                ClearingDoubleSpinBox(parent)]
        w = self.widgets[1]
        w.setRange(-1000000., float(100000000))
        w.setDecimals(2)
        w.setSpecialValueText(_('Undefined'))
        w.setSingleStep(1)
        self.was_none = False
        w.valueChanged.connect(self.valueChanged)


class Rating(Base):

    def setup_ui(self, parent):
        allow_half_stars = self.col_metadata['display'].get('allow_half_stars', False)
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent),
                        RatingEditor(parent=parent, is_half_star=allow_half_stars)]

    def setter(self, val):
        val = max(0, min(int(val or 0), 10))
        self.widgets[1].rating_value = val

    def getter(self):
        return self.widgets[1].rating_value or None

    def connect_data_changed(self, slot):
        self.widgets[1].currentTextChanged.connect(slot)
        self.signals_to_disconnect.append(self.widgets[1].currentTextChanged)


class DateTimeEdit(DateTimeEditBase):

    def focusInEvent(self, x):
        self.setSpecialValueText('')
        DateTimeEditBase.focusInEvent(self, x)

    def focusOutEvent(self, x):
        self.setSpecialValueText(_('Undefined'))
        DateTimeEditBase.focusOutEvent(self, x)

    def set_to_today(self):
        self.setDateTime(qt_from_dt(now()))

    def set_to_clear(self):
        self.setDateTime(qt_from_dt(now()))
        self.setDateTime(UNDEFINED_QDATETIME)


class DateTime(Base):

    def setup_ui(self, parent):
        cm = self.col_metadata
        self.widgets = [QLabel(label_string(cm['name']), parent)]
        w = QWidget(parent)
        self.widgets.append(w)
        self.l = l = QHBoxLayout()
        l.setContentsMargins(0, 0, 0, 0)
        w.setLayout(l)
        self.dte = dte = DateTimeEdit(parent)
        format_ = cm['display'].get('date_format','')
        if not format_:
            format_ = 'dd MMM yyyy hh:mm'
        elif format_ == 'iso':
            format_ = internal_iso_format_string()
        dte.setDisplayFormat(format_)
        dte.setCalendarPopup(True)
        dte.setMinimumDateTime(UNDEFINED_QDATETIME)
        dte.setSpecialValueText(_('Undefined'))
        l.addWidget(dte)

        self.today_button = QToolButton(parent)
        self.today_button.setText(_('Today'))
        self.today_button.clicked.connect(dte.set_to_today)
        l.addWidget(self.today_button)

        self.clear_button = QToolButton(parent)
        self.clear_button.setIcon(get_icon('trash.png'))
        self.clear_button.clicked.connect(dte.set_to_clear)
        l.addWidget(self.clear_button)

    def setter(self, val):
        if val is None:
            val = self.dte.minimumDateTime()
        else:
            val = qt_from_dt(val)
        self.dte.setDateTime(val)

    def getter(self):
        val = self.dte.dateTime()
        if is_date_undefined(val):
            val = None
        else:
            val = qt_to_dt(val)
        return val

    def normalize_db_val(self, val):
        return as_local_time(val) if val is not None else None

    def normalize_ui_val(self, val):
        return as_utc(val) if val is not None else None

    def connect_data_changed(self, slot):
        self.dte.dateTimeChanged.connect(slot)
        self.signals_to_disconnect.append(self.dte.dateTimeChanged)

    def commit(self, book_id, notify=False):
        val = self.current_val
        if val != self.initial_val:
            if self.col_name == 'last_modified':
                return self.db.update_last_modified([book_id], now=val)
            else:
                return self.db.new_api.set_field(self.col_name, {book_id: val}, allow_case_change=True)
        else:
            return set()

class Comments(Base):

    def setup_ui(self, parent):
        self._box = QGroupBox(parent)
        self._box.setTitle(label_string(self.col_metadata['name']))
        self._layout = QVBoxLayout()
        self._tb = CommentsEditor(self._box, toolbar_prefs_name='metadata-comments-editor-widget-hidden-toolbars')
        self._tb.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Minimum)
        # self._tb.setTabChangesFocus(True)
        self._layout.addWidget(self._tb)
        self._box.setLayout(self._layout)
        self.widgets = [self._box]

    def initialize(self, book_id):
        path = self.db.abspath(book_id, index_is_id=True)
        if path:
            self._tb.set_base_url(QUrl.fromLocalFile(os.path.join(path, 'metadata.html')))
        return Base.initialize(self, book_id)

    def setter(self, val):
        if not val or not val.strip():
            val = ''
        else:
            val = comments_to_html(val)
        self._tb.html = val
        self._tb.wyswyg_dirtied()

    def getter(self):
        val = str(self._tb.html).strip()
        if not val:
            val = None
        return val

    @property
    def tab(self):
        return self._tb.tab

    @tab.setter
    def tab(self, val):
        self._tb.tab = val

    def connect_data_changed(self, slot):
        self._tb.data_changed.connect(slot)
        self.signals_to_disconnect.append(self._tb.data_changed)


class MultipleWidget(QWidget):

    def __init__(self, parent):
        QWidget.__init__(self, parent)
        layout = QHBoxLayout()
        layout.setSpacing(5)
        layout.setContentsMargins(0, 0, 0, 0)

        self.tags_box = EditWithComplete(parent)
        layout.addWidget(self.tags_box, stretch=1000)
        self.editor_button = QToolButton(self)
        self.editor_button.setToolTip(_('Open item editor'))
        self.editor_button.setIcon(get_icon('chapters.png'))
        layout.addWidget(self.editor_button)
        self.setLayout(layout)

    def get_editor_button(self):
        return self.editor_button

    def update_items_cache(self, values):
        self.tags_box.update_items_cache(values)

    def clear(self):
        self.tags_box.clear()

    def setEditText(self):
        self.tags_box.setEditText()

    def addItem(self, itm):
        self.tags_box.addItem(itm)

    def set_separator(self, sep):
        self.tags_box.set_separator(sep)

    def set_add_separator(self, sep):
        self.tags_box.set_add_separator(sep)

    def set_space_before_sep(self, v):
        self.tags_box.set_space_before_sep(v)

    def setSizePolicy(self, v1, v2):
        self.tags_box.setSizePolicy(v1, v2)

    def setText(self, v):
        self.tags_box.setText(v)

    def text(self):
        return self.tags_box.text()


def _save_dialog(parent, title, msg, det_msg=''):
    d = QMessageBox(parent)
    d.setWindowTitle(title)
    d.setText(msg)
    d.setStandardButtons(QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel)
    return d.exec()


class Text(Base):

    def setup_ui(self, parent):
        self.sep = self.col_metadata['is_multiple']
        if self.col_name == 'tags':
            self.key = None
        else:
            # must be custom column
            self.key = self.col_name
        self.parent = parent

        if self.is_multiple() and self.col_name != 'identifiers':
            w = MultipleWidget(parent)
            w.set_separator(self.sep['ui_to_list'])
            if self.sep['ui_to_list'] == '&':
                w.set_space_before_sep(True)
                w.set_add_separator(tweaks['authors_completer_append_separator'])
            w.get_editor_button().clicked.connect(self.edit)
            w.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        else:
            w = EditWithComplete(parent)
            w.set_separator(None)
            w.setSizeAdjustPolicy(QComboBox.SizeAdjustPolicy.AdjustToMinimumContentsLengthWithIcon)
            w.setMinimumContentsLength(25)
            if self.col_name == 'identifiers':
                w.set_separator(self.sep['ui_to_list'])
            
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent), w]

    def initialize(self, book_id):
        values = all_field_names(self.db, self.col_name)
        values.sort(key=sort_key)
        self.book_id = book_id
        self.widgets[1].clear()
        self.widgets[1].update_items_cache(values)
        val = self.db.new_api.field_for(self.col_name, book_id)
        if isinstance(val, list):
            if not self.is_names():
                val.sort(key=sort_key)
        val = self.normalize_db_val(val)

        if self.is_multiple():
            self.setter(val)
        else:
            self.widgets[1].setText(val)
        self.initial_val = self.current_val

    def setter(self, val):
        if self.is_multiple():
            if not val:
                val = []
            self.widgets[1].setText(self.sep['list_to_ui'].join(val))

    def getter(self):
        if self.is_multiple():
            val = str(self.widgets[1].text()).strip()
            ans = [x.strip() for x in val.split(self.sep['ui_to_list']) if x.strip()]
            if not ans:
                ans = None
            return ans
        val = str(self.widgets[1].currentText()).strip()
        if not val:
            val = None
        return val

    def edit(self):
        if (self.getter() != self.initial_val and (self.getter() or self.initial_val)):
            d = _save_dialog(self.parent, _('Values changed'),
                    _('You have changed the values. In order to use this '
                       'editor, you must either discard or apply these '
                       'changes. Apply changes?'))
            if d == QMessageBox.Cancel:
                return
            if d == QMessageBox.Yes:
                self.commit(self.book_id)
                self.db.commit()
                self.initial_val = self.current_val
            else:
                self.setter(self.initial_val)
        d = TagEditor(self.parent, self.db, self.book_id, self.key)
        if d.exec() == TagEditor.Accepted:
            self.setter(d.tags)

    def connect_data_changed(self, slot):
        if self.is_multiple():
            s = self.widgets[1].tags_box.currentTextChanged
        else:
            s = self.widgets[1].currentTextChanged
        s.connect(slot)
        self.signals_to_disconnect.append(s)

    def normalize_db_val(self, val):
        if self.col_name == 'identifiers':
            val = [f'{k}:{v}' for k,v in val.items()]
        return val

class Series(Base):

    def setup_ui(self, parent):
        w = EditWithComplete(parent)
        w.set_separator(None)
        w.setSizeAdjustPolicy(QComboBox.SizeAdjustPolicy.AdjustToMinimumContentsLengthWithIcon)
        w.setMinimumContentsLength(25)
        self.name_widget = w
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent), w]
        w.editTextChanged.connect(self.series_changed)

        w = QLabel(label_string(self.col_metadata['name'])+_(' index'), parent)
        w.setToolTip(get_tooltip(self.col_metadata, add_index=True))
        self.widgets.append(w)
        w = QDoubleSpinBox(parent)
        w.setRange(-10000., float(100000000))
        w.setDecimals(2)
        w.setSingleStep(1)
        self.idx_widget=w
        w.setToolTip(get_tooltip(self.col_metadata, add_index=True))
        self.widgets.append(w)

    def initialize(self, book_id):
        values = all_field_names(self.db, self.col_name)
        values.sort(key=sort_key)
        val = self.db.new_api.field_for(self.col_name, book_id)
        if self.col_metadata['is_custom']:
            s_index = self.db.get_custom_extra(book_id, label=self.col_metadata['label'], index_is_id=True)
        else:
            s_index = self.db.series_index(book_id, index_is_id=True)
        try:
            s_index = float(s_index)
        except (ValueError, TypeError):
            s_index = 1.0
        self.idx_widget.setValue(s_index)
        val = self.normalize_db_val(val)
        self.name_widget.blockSignals(True)
        self.name_widget.update_items_cache(values)
        self.name_widget.setText(val)
        self.name_widget.blockSignals(False)
        self.initial_val, self.initial_index = self.current_val

    def getter(self):
        n = str(self.name_widget.currentText()).strip()
        i = self.idx_widget.value()
        return n, i

    def series_changed(self, val):
        val, s_index = self.gui_val
        if tweaks['series_index_auto_increment'] == 'no_change':
            pass
        elif tweaks['series_index_auto_increment'] == 'const':
            s_index = 1.0
        else:
            if self.col_metadata['is_custom']:
                s_index = self.db.get_next_cc_series_num_for(val, label=self.col_metadata['label'])
            else:
                s_index = self.db.get_next_series_num_for(val)
        self.idx_widget.setValue(s_index)

    @property
    def current_val(self):
        val, s_index = self.gui_val
        val = self.normalize_ui_val(val)
        return val, s_index

    def commit(self, book_id, notify=False):
        val, s_index = self.current_val
        if val != self.initial_val or s_index != self.initial_index:
            if not val:
                val = s_index = None
            else:
                val = f'{val} [{s_index}]'
            return self.db.new_api.set_field(self.col_name, {book_id: val}, allow_case_change=True)
        else:
            return set()

    def apply_to_metadata(self, mi):
        val, s_index = self.current_val
        mi.set(self.col_name, val, extra=s_index)

    def connect_data_changed(self, slot):
        for s in self.widgets[1].editTextChanged, self.widgets[3].valueChanged:
            s.connect(slot)
            self.signals_to_disconnect.append(s)


class Enumeration(Base):

    def setup_ui(self, parent):
        self.parent = parent
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent),
                QComboBox(parent)]
        w = self.widgets[1]
        vals = self.col_metadata['display']['enum_values']
        w.addItem('')
        for v in vals:
            w.addItem(v)

    def initialize(self, book_id):
        val = self.db.new_api.field_for(self.col_name, book_id)
        val = self.normalize_db_val(val)
        idx = self.widgets[1].findText(val)
        if idx < 0:
            error_dialog(self.parent, '',
                    _('The enumeration "{0}" contains an invalid value '
                      'that will be set to the default').format(
                                            self.col_metadata['name']),
                    show=True, show_copy_button=False)

            idx = 0
        self.widgets[1].setCurrentIndex(idx)
        self.initial_val = self.current_val

    def setter(self, val):
        self.widgets[1].setCurrentIndex(self.widgets[1].findText(val))

    def getter(self):
        return str(self.widgets[1].currentText())

    def normalize_db_val(self, val):
        if val is None:
            val = ''
        return val

    def normalize_ui_val(self, val):
        if not val:
            val = None
        return val

    def connect_data_changed(self, slot):
        self.widgets[1].currentIndexChanged.connect(slot)
        self.signals_to_disconnect.append(self.widgets[1].currentIndexChanged)

class BulkBase(Base):

    def get_initial_value(self, book_ids):
        values = set()
        for book_id in book_ids:
            val = self.db.new_api.field_for(self.col_name, book_id)
            if isinstance(val, list):
                val = frozenset(val)
            values.add(val)
            if len(values) > 1:
                break
        ans = None
        if len(values) == 1:
            ans = next(iter(values))
        if isinstance(ans, frozenset):
            ans = list(ans)
        return ans

    def initialize(self, book_ids):
        self.initial_val = val = self.get_initial_value(book_ids)
        val = self.normalize_db_val(val)
        self.setter(val)

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        val = self.gui_val
        val = self.normalize_ui_val(val)
        return self.db.new_api.set_field(self.col_name, {book_id: val for book_id in book_ids}, allow_case_change=True)

    def make_widgets(self, parent, main_widget_class, add_tags_edit_button=False):
        w = QWidget(parent)
        self.widgets = [QLabel(label_string(self.col_metadata['name']), w), w]
        l = QHBoxLayout()
        l.setContentsMargins(0, 0, 0, 0)
        w.setLayout(l)
        self.main_widget = main_widget_class(w)
        l.addWidget(self.main_widget)
        l.setStretchFactor(self.main_widget, 10)
        if add_tags_edit_button:
            self.edit_tags_button = QToolButton(parent)
            self.edit_tags_button.setToolTip(_('Open item editor'))
            self.edit_tags_button.setIcon(get_icon('chapters.png'))
            l.addWidget(self.edit_tags_button)
        self.a_c_checkbox = QCheckBox(_('Apply changes'), w)
        l.addWidget(self.a_c_checkbox)
        self.ignore_change_signals = True

        # connect to the various changed signals so we can auto-update the
        # apply changes checkbox
        if hasattr(self.main_widget, 'editTextChanged'):
            # editable combobox widgets
            self.main_widget.editTextChanged.connect(self.a_c_checkbox_changed)
        if hasattr(self.main_widget, 'textChanged'):
            # lineEdit widgets
            self.main_widget.textChanged.connect(self.a_c_checkbox_changed)
        if hasattr(self.main_widget, 'currentIndexChanged'):
            # combobox widgets
            self.main_widget.currentIndexChanged.connect(self.a_c_checkbox_changed)
        if hasattr(self.main_widget, 'valueChanged'):
            # spinbox widgets
            self.main_widget.valueChanged.connect(self.a_c_checkbox_changed)
        if hasattr(self.main_widget, 'dateTimeChanged'):
            # dateEdit widgets
            self.main_widget.dateTimeChanged.connect(self.a_c_checkbox_changed)

    def a_c_checkbox_changed(self):
        if not self.ignore_change_signals:
            self.a_c_checkbox.setChecked(True)


class BulkBool(BulkBase, Bool):

    def get_initial_value(self, book_ids):
        value = None
        for book_id in book_ids:
            val = self.db.new_api.field_for(self.col_name, book_id)
            if not self.db.new_api.pref('bools_are_tristate') and val is None:
                val = False
            if value is not None and value != val:
                return None
            value = val
        return value

    def setup_ui(self, parent):
        self.make_widgets(parent, QComboBox)
        items = [_('Yes'), _('No')]
        if not self.db.new_api.pref('bools_are_tristate'):
            items.append('')
        else:
            items.append(_('Undefined'))
        icons = [I('ok.png'), I('list_remove.png'), I('blank.png')]
        self.main_widget.blockSignals(True)
        for icon, text in zip(icons, items):
            self.main_widget.addItem(QIcon(icon), text)
        self.main_widget.blockSignals(False)

    def getter(self):
        val = self.main_widget.currentIndex()
        if not self.db.new_api.pref('bools_are_tristate'):
            return {2: False, 1: False, 0: True}[val]
        else:
            return {2: None, 1: False, 0: True}[val]

    def setter(self, val):
        val = {None: 2, False: 1, True: 0}[val]
        self.main_widget.setCurrentIndex(val)
        self.ignore_change_signals = False

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        val = self.gui_val
        val = self.normalize_ui_val(val)
        if not self.db.new_api.pref('bools_are_tristate') and val is None:
            val = False
        return self.db.new_api.set_field(self.col_name, {book_id: val for book_id in book_ids}, allow_case_change=True)

    def a_c_checkbox_changed(self):
        if not self.ignore_change_signals:
            if not self.db.new_api.pref('bools_are_tristate') and \
                                    self.main_widget.currentIndex() == 2:
                self.a_c_checkbox.setChecked(False)
            else:
                self.a_c_checkbox.setChecked(True)


class BulkInt(BulkBase):

    def setup_ui(self, parent):
        self.was_none = False
        self.make_widgets(parent, QSpinBox)
        self.main_widget.setRange(-1000000, 100000000)
        self.main_widget.setSpecialValueText(_('Undefined'))
        self.main_widget.setSingleStep(1)
        self.main_widget.valueChanged.connect(self.valueChanged)

    def setter(self, val):
        if val is None:
            val = self.main_widget.minimum()
        self.main_widget.setValue(val)
        self.ignore_change_signals = False
        self.was_none = val == self.main_widget.minimum()

    def getter(self):
        val = self.main_widget.value()
        if val == self.main_widget.minimum():
            val = None
        return val

    def valueChanged(self, to_what):
        if self.was_none and to_what == -999999:
            self.setter(0)
        self.was_none = to_what == self.main_widget.minimum()


class BulkFloat(BulkInt):

    def setup_ui(self, parent):
        self.make_widgets(parent, QDoubleSpinBox)
        self.main_widget.setRange(-1000000., float(100000000))
        self.main_widget.setDecimals(2)
        self.main_widget.setSpecialValueText(_('Undefined'))
        self.main_widget.setSingleStep(1)
        self.was_none = False
        self.main_widget.valueChanged.connect(self.valueChanged)


class BulkRating(BulkBase):

    def setup_ui(self, parent):
        allow_half_stars = self.col_metadata['display'].get('allow_half_stars', False)
        self.make_widgets(parent, partial(RatingEditor, is_half_star=allow_half_stars))

    def setter(self, val):
        val = max(0, min(int(val or 0), 10))
        self.main_widget.rating_value = val
        self.ignore_change_signals = False

    def getter(self):
        return self.main_widget.rating_value or None


class BulkDateTime(BulkBase):

    def setup_ui(self, parent):
        cm = self.col_metadata
        self.make_widgets(parent, DateTimeEdit)
        self.l = l = self.widgets[1].layout()
        self.today_button = QToolButton(parent)
        self.today_button.setText(_('Today'))
        l.insertWidget(1, self.today_button)
        self.clear_button = QToolButton(parent)
        self.clear_button.setIcon(get_icon('trash.png'))
        l.insertWidget(2, self.clear_button)
        l.insertStretch(3)

        w = self.main_widget
        format_ = cm['display'].get('date_format','')
        if not format_:
            format_ = 'dd MMM yyyy'
        elif format_ == 'iso':
            format_ = internal_iso_format_string()
        w.setDisplayFormat(format_)
        w.setCalendarPopup(True)
        w.setMinimumDateTime(UNDEFINED_QDATETIME)
        w.setSpecialValueText(_('Undefined'))
        self.today_button.clicked.connect(w.set_to_today)
        self.clear_button.clicked.connect(w.set_to_clear)

    def setter(self, val):
        if val is None:
            val = self.main_widget.minimumDateTime()
        else:
            val = qt_from_dt(val)
        self.main_widget.setDateTime(val)
        self.ignore_change_signals = False

    def getter(self):
        val = self.main_widget.dateTime()
        if is_date_undefined(val):
            val = None
        else:
            val = qt_to_dt(val)
        return val

    def normalize_db_val(self, val):
        return as_local_time(val) if val is not None else None

    def normalize_ui_val(self, val):
        return as_utc(val) if val is not None else None

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        dt = self.gui_val
        normalized_dt = BulkDateTime.normalize_ui_val(self, dt)
        if self.col_name == 'last_modified':
            return self.db.update_last_modified(book_ids, now=normalized_dt)
        else:
            return self.db.new_api.set_field(self.col_name, {book_id: normalized_dt for book_id in book_ids}, allow_case_change=True)

# new class to be used in config dialog instead of runtime
class PredefinedDateTime(BulkDateTime):
    def setup_ui(self, parent):
        BulkDateTime.setup_ui(self, parent)
        self.today_button.hide()
        self.current_chk = QCheckBox(_('Set to current time'))
        self.current_chk.setChecked(False)
        self.current_chk.stateChanged.connect(self._on_current_chk)
        self.widgets.append(self.current_chk)
        self.main_widget.set_to_clear()

    def _on_current_chk(self):
        self.main_widget.setEnabled(not self.current_chk.isChecked())
        self.main_widget.set_to_clear()
        if self.current_chk.isChecked():
            self.a_c_checkbox.setChecked(True)

    def getter(self):
        val = BulkDateTime.getter(self)
        is_current_checked = self.current_chk.isChecked()
        if is_current_checked:
            val = None
        return val, is_current_checked       

    def normalize_ui_val(self, val):
        dt, is_current_checked = val
#        if is_current_checked:
#            dt = now()
        normalized_dt = BulkDateTime.normalize_ui_val(self, dt)
        return normalized_dt, is_current_checked

    def setter(self, val):
        dt, is_current_checked = val
        self.current_chk.setChecked(is_current_checked)
        if is_current_checked:
            self.main_widget.set_to_clear()
        else:
            BulkDateTime.setter(self, dt)

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        dt, is_current_checked = self.gui_val
        if is_current_checked:
            # we have to re-calculate the time because the stored value is old
            dt = now()
        normalized_dt = BulkDateTime.normalize_ui_val(self, dt)
        if self.col_name == 'last_modified':
            return self.db.update_last_modified(book_ids, now=normalized_dt)
        else:
            return self.db.new_api.set_field(self.col_name, {book_id: normalized_dt for book_id in book_ids}, allow_case_change=True)

class BulkSeries(BulkBase):

    def setup_ui(self, parent):
        self.make_widgets(parent, EditWithComplete)
        values = self.all_values = all_field_names(self.db, self.col_name)
        values.sort(key=sort_key)
        self.main_widget.setSizeAdjustPolicy(QComboBox.SizeAdjustPolicy.AdjustToMinimumContentsLengthWithIcon)
        self.main_widget.setMinimumContentsLength(25)
        self.widgets.append(QLabel('', parent))
        w = QWidget(parent)
        layout = QHBoxLayout(w)
        layout.setContentsMargins(0, 0, 0, 0)
        self.remove_series = QCheckBox(parent)
        self.remove_series.setText(_('Clear series'))
        layout.addWidget(self.remove_series)
        self.idx_widget = QCheckBox(parent)
        self.idx_widget.setText(_('Automatically number books'))
        self.idx_widget.setToolTip('<p>' + _(
            'If not checked, the series number for the books will be set to 1. '
            'If checked, selected books will be automatically numbered, '
            'in the order you selected them. So if you selected '
            'Book A and then Book B, Book A will have series number 1 '
            'and Book B series number 2.') + '</p>')
        layout.addWidget(self.idx_widget)
        self.force_number = QCheckBox(parent)
        self.force_number.setText(_('Force numbers to start with '))
        self.force_number.setToolTip('<p>' + _(
            'Series will normally be renumbered from the highest '
            'number in the database for that series. Checking this '
            'box will tell calibre to start numbering from the value '
            'in the box') + '</p>')
        layout.addWidget(self.force_number)
        self.series_start_number = QDoubleSpinBox(parent)
        self.series_start_number.setMinimum(0.0)
        self.series_start_number.setMaximum(9999999.0)
        self.series_start_number.setProperty("value", 1.0)
        layout.addWidget(self.series_start_number)
        self.series_increment = QDoubleSpinBox(parent)
        self.series_increment.setMinimum(0.00)
        self.series_increment.setMaximum(99999.0)
        self.series_increment.setProperty("value", 1.0)
        self.series_increment.setToolTip('<p>' + _(
            'The amount by which to increment the series number '
            'for successive books. Only applicable when using '
            'force series numbers.') + '</p>')
        self.series_increment.setPrefix('+')
        layout.addWidget(self.series_increment)
        layout.addItem(QSpacerItem(20, 10, QSizePolicy.Expanding, QSizePolicy.Minimum))
        self.widgets.append(w)
        self.idx_widget.stateChanged.connect(self.a_c_checkbox_changed)
        self.force_number.stateChanged.connect(self.a_c_checkbox_changed)
        self.series_start_number.valueChanged.connect(self.a_c_checkbox_changed)
        self.series_increment.valueChanged.connect(self.a_c_checkbox_changed)
        self.remove_series.stateChanged.connect(self.a_c_checkbox_changed)
        self.main_widget
        self.ignore_change_signals = False

    def a_c_checkbox_changed(self):
        def disable_numbering_checkboxes(idx_widget_enable):
            if idx_widget_enable:
                self.idx_widget.setEnabled(True)
            else:
                self.idx_widget.setChecked(False)
                self.idx_widget.setEnabled(False)
            self.force_number.setChecked(False)
            self.force_number.setEnabled(False)
            self.series_start_number.setEnabled(False)
            self.series_increment.setEnabled(False)

        if self.ignore_change_signals:
            return
        self.ignore_change_signals = True
        apply_changes = False
        if self.remove_series.isChecked():
            self.main_widget.setText('')
            self.main_widget.setEnabled(False)
            disable_numbering_checkboxes(idx_widget_enable=False)
            apply_changes = True
        elif self.main_widget.text():
            self.remove_series.setEnabled(False)
            self.idx_widget.setEnabled(True)
            apply_changes = True
        else:  # no text, no clear. Basically reinitialize
            self.main_widget.setEnabled(True)
            self.remove_series.setEnabled(True)
            disable_numbering_checkboxes(idx_widget_enable=False)
            apply_changes = False

        self.force_number.setEnabled(self.idx_widget.isChecked())
        self.series_start_number.setEnabled(self.force_number.isChecked())
        self.series_increment.setEnabled(self.force_number.isChecked())

        self.ignore_change_signals = False
        self.a_c_checkbox.setChecked(apply_changes)

    def initialize(self, book_id):
        self.idx_widget.setChecked(False)
        self.main_widget.set_separator(None)
        self.main_widget.update_items_cache(self.all_values)
        self.main_widget.setEditText('')
        self.a_c_checkbox.setChecked(False)

    def getter(self):
        n = str(self.main_widget.currentText()).strip()
        autonumber = self.idx_widget.isChecked()
        force = self.force_number.isChecked()
        start = self.series_start_number.value()
        remove = self.remove_series.isChecked()
        increment = self.series_increment.value()
        return n, autonumber, force, start, remove, increment

    def setter(self, val):
        n, autonumber, force, start, remove, increment = val
        self.main_widget.setCurrentText(n)
        self.idx_widget.setChecked(autonumber)
        self.force_number.setChecked(force)
        self.series_start_number.setValue(start)
        self.remove_series.setChecked(remove)
        self.series_increment.setValue(increment)

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        series_map = {}
        val, update_indices, force_start, at_value, clear, increment = self.gui_val
        val = None if clear else self.normalize_ui_val(val)
        if clear or val != '':
            #extras = []
            for book_id in book_ids:
                if clear:
                    #extras.append(None)
                    series_map[book_id] = None
                    continue
                if update_indices:
                    if force_start:
                        s_index = at_value
                        at_value += increment
                    elif tweaks['series_index_auto_increment'] != 'const':
                        if self.col_metadata['is_custom']:
                            s_index = self.db.get_next_cc_series_num_for(val, label=self.col_metadata['label'])
                        else:
                            s_index = self.db.get_next_series_num_for(val)
                    else:
                        s_index = 1.0
                else:
                    if self.col_metadata['is_custom']:
                        s_index = self.db.get_custom_extra(book_id, label=self.col_metadata['label'], index_is_id=True)
                    else:
                        s_index = self.db.series_index(book_id, index_is_id=True)
                #extras.append(s_index)
                if val:
                    series_for_book = f'{val} [{s_index}]'
                    series_map[book_id] = series_for_book
            return self.db.new_api.set_field(self.col_name, series_map, allow_case_change=True)


class BulkEnumeration(BulkBase, Enumeration):

    def get_initial_value(self, book_ids):
        value = None
        first = True
        dialog_shown = False
        for book_id in book_ids:
            val = self.db.new_api.field_for(self.col_name, book_id)
            if val and val not in self.col_metadata['display']['enum_values']:
                if not dialog_shown:
                    error_dialog(self.parent, '',
                            _('The enumeration "{0}" contains invalid values '
                              'that will not appear in the list').format(
                                                    self.col_metadata['name']),
                            show=True, show_copy_button=False)
                    dialog_shown = True
            if first:
                value = val
                first = False
            elif value != val:
                value = None
        if not value:
            self.ignore_change_signals = False
        return value

    def setup_ui(self, parent):
        self.parent = parent
        self.make_widgets(parent, QComboBox)
        vals = self.col_metadata['display']['enum_values']
        self.main_widget.blockSignals(True)
        self.main_widget.addItem('')
        self.main_widget.addItems(vals)
        self.main_widget.blockSignals(False)

    def getter(self):
        return str(self.main_widget.currentText())

    def setter(self, val):
        if val is None:
            self.main_widget.setCurrentIndex(0)
        else:
            self.main_widget.setCurrentIndex(self.main_widget.findText(val))
        self.ignore_change_signals = False


class RemoveTags(QWidget):

    def __init__(self, parent, values):
        QWidget.__init__(self, parent)
        layout = QHBoxLayout()
        layout.setSpacing(5)
        layout.setContentsMargins(0, 0, 0, 0)

        self.tags_box = EditWithComplete(parent)
        self.tags_box.update_items_cache(values)
        layout.addWidget(self.tags_box, stretch=3)
        self.remove_tags_button = QToolButton(parent)
        self.remove_tags_button.setToolTip(_('Open item editor'))
        self.remove_tags_button.setIcon(get_icon('chapters.png'))
        layout.addWidget(self.remove_tags_button)
        self.checkbox = QCheckBox(_('Remove all items'), parent)
        layout.addWidget(self.checkbox)
        layout.addStretch(1)
        self.setLayout(layout)
        self.checkbox.stateChanged[int].connect(self.box_touched)

    def box_touched(self, state):
        if state:
            self.tags_box.setText('')
            self.tags_box.setEnabled(False)
        else:
            self.tags_box.setEnabled(True)


class BulkText(BulkBase):

    def setup_ui(self, parent):
        values = self.all_values = all_field_names(self.db, self.col_name)
        values.sort(key=sort_key)
        if self.is_multiple():
            is_tags = not self.is_names()
            self.make_widgets(parent, EditWithComplete, add_tags_edit_button=is_tags)
            self.main_widget.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
            self.adding_widget = self.main_widget

            if is_tags:
                if self.col_name == 'identifiers':
                    self.edit_tags_button.hide()
                    w = RemoveTags(parent, self.db.get_all_identifier_types())
                    w.remove_tags_button.hide()
                else:
                    self.edit_tags_button.clicked.connect(self.edit_add)
                    w = RemoveTags(parent, values)
                    w.remove_tags_button.clicked.connect(self.edit_remove)
                l = QLabel(label_string(self.col_metadata['name'])+': ' +
                                           _('items to remove'), parent)
                tt = get_tooltip(self.col_metadata) + ': ' + _('items to remove')
                l.setToolTip(tt)
                self.widgets.append(l)
                w.setToolTip(tt)
                self.widgets.append(w)
                self.removing_widget = w
                self.main_widget.set_separator(',')
                w.tags_box.textChanged.connect(self.a_c_checkbox_changed)
                w.checkbox.stateChanged.connect(self.a_c_checkbox_changed)
            else:
                self.main_widget.set_separator('&')
                self.main_widget.set_space_before_sep(True)
                self.main_widget.set_add_separator(
                                tweaks['authors_completer_append_separator'])
        else:
            self.make_widgets(parent, EditWithComplete)
            self.main_widget.set_separator(None)
            self.main_widget.setSizeAdjustPolicy(
                        QComboBox.SizeAdjustPolicy.AdjustToMinimumContentsLengthWithIcon)
            self.main_widget.setMinimumContentsLength(25)
        self.ignore_change_signals = False
        self.parent = parent

    def initialize(self, book_ids):
        if not self.col_name == 'identifiers':
            self.main_widget.update_items_cache(self.all_values)
        if not self.is_multiple():
            val = self.get_initial_value(book_ids)
            self.initial_val = val = self.normalize_db_val(val)
            self.ignore_change_signals = True
            self.main_widget.blockSignals(True)
            self.main_widget.setText(val)
            self.main_widget.blockSignals(False)
            self.ignore_change_signals = False

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        label = self.col_metadata['label']
        if self.is_multiple():
            ism = self.col_metadata['is_multiple']
            if self.is_names():
                val = self.gui_val
                add = [v.strip() for v in val.split(ism['ui_to_list']) if v.strip()]
                return self.db.new_api.set_field(self.col_name, {book_id: add for book_id in book_ids}, allow_case_change=True)
            else:
                remove_all, adding, rtext = self.gui_val
                remove = set()
                if remove_all:
                    remove = all_field_names(self.db, self.col_name)
                else:
                    txt = rtext
                    if txt:
                        remove = {v.strip() for v in txt.split(ism['ui_to_list'])}
                txt = adding
                if txt:
                    add = {v.strip() for v in txt.split(ism['ui_to_list'])}
                else:
                    add = set()
                if self.col_metadata['is_custom']:
                    self.db.set_custom_bulk_multiple(book_ids, add=add,
                                            remove=remove, label=label)
                elif self.col_name == 'tags':
                    self.db.bulk_modify_tags(book_ids, add=add, remove=remove)
                elif self.col_name == 'languages':
                    bulk_modify_multiple(self.db, book_ids, self.col_name, add=add, remove=remove, remove_all=remove_all)
                elif self.col_name == 'identifiers':
                    bulk_modify_identifiers(self.db, book_ids, add=add, remove=remove, remove_all=remove_all)
        else:
            val = self.gui_val
            val = self.normalize_ui_val(val)
            return self.db.new_api.set_field(self.col_name, {book_id: val for book_id in book_ids}, allow_case_change=True)

    def getter(self):
        if self.is_multiple():
            if not self.is_names():
                return self.removing_widget.checkbox.isChecked(), \
                        str(self.adding_widget.text()), \
                        str(self.removing_widget.tags_box.text())
            return str(self.adding_widget.text())
        val = str(self.main_widget.currentText()).strip()
        if not val:
            val = None
        return val

    def setter(self, val):
        if self.is_multiple():
            if not self.is_names():
                remove_all, adding, removing = val
                self.removing_widget.checkbox.setChecked(remove_all)
                self.adding_widget.setText(adding)
                self.removing_widget.tags_box.setText(removing)
                return
        if val:
            self.main_widget.setCurrentText(val)

    def edit_remove(self):
        self.edit(widget=self.removing_widget.tags_box)

    def edit_add(self):
        self.edit(widget=self.main_widget)

    def edit(self, widget):
        if widget.text():
            d = _save_dialog(self.parent, _('Values changed'),
                    _('You have entered values. In order to use this '
                       'editor you must first discard them. '
                       'Discard the values?'))
            if d == QMessageBox.Cancel or d == QMessageBox.No:
                return
            widget.setText('')
        if self.col_name == 'tags':
            key = None
        else:
            # must be custom column
            key = self.col_name
        d = TagEditor(self.parent, self.db, key=key)
        if d.exec() == TagEditor.Accepted:
            val = d.tags
            if not val:
                val = []
            widget.setText(self.col_metadata['is_multiple']['list_to_ui'].join(val))

# new class
class BulkBase2(object):
    def initialize(self, book_ids):
        pass

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        val = self.gui_val
        val = self.normalize_ui_val(val)
        return self.db.new_api.set_field(self.col_name, {book_id: val for book_id in book_ids}, allow_case_change=True)

    def connect_to_apply_changes(self):
        self.ignore_change_signals = False
        # connect to the various changed signals so we can auto-update the
        # apply changes checkbox
        if hasattr(self.editor, 'editTextChanged'):
            # editable combobox widgets
            self.editor.editTextChanged.connect(self.a_c_checkbox_changed)
        if hasattr(self.editor, 'textChanged'):
            # lineEdit widgets
            self.editor.textChanged.connect(self.a_c_checkbox_changed)

    def a_c_checkbox_changed(self):
        if not self.ignore_change_signals:
            self.a_c_checkbox.setChecked(True)

# new class
class BulkComments(BulkBase2, Comments):

    def setup_ui(self, parent):
        self._box = QGroupBox(parent)
        self._box.setTitle(label_string(self.col_metadata['name']))
        self._layout = QVBoxLayout()
        self._tb = CommentsEditor(self._box, toolbar_prefs_name='metadata-comments-editor-widget-hidden-toolbars')
        #
        self.editor = self._tb.editor
        #
        self._tb.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Minimum)
        # self._tb.setTabChangesFocus(True)
        self._layout.addWidget(self._tb)
        self.a_c_checkbox = QCheckBox(_('Apply to all'))
        self._layout.addWidget(self.a_c_checkbox)
        self._box.setLayout(self._layout)
        self.widgets = [self._box]
        #
        self.connect_to_apply_changes()
        #

# new class
class WidgetDialog(Dialog):
    def __init__(self, parent, widget_cls, plugin_action, col_name):
        self.plugin_action = plugin_action
        self.db = self.plugin_action.gui.current_db
        self.widget = widget_cls(self.plugin_action, col_name, self)
        Dialog.__init__(self, 'Comments Dialog', f'action-chains-comments-dialog-{col_name}', parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)
        l.addWidget(self.widget)
        l.addWidget(self.bb)

    def accept(self):
        self.val = self.widget.current_val
        Dialog.accept(self)

# new class
class PredefinedComments(BulkBase):

    def setup_ui(self, parent):
        self._val = None
        self._button = QPushButton(_('Comments Editor'))
        self.a_c_checkbox = QCheckBox(_('Apply to all'))
        self.a_c_checkbox.setVisible(False)
        self._button.clicked.connect(self._button_clicked)
        self.widgets = [self._button, self.a_c_checkbox]

    def _button_clicked(self):
        d = WidgetDialog(self, BulkComments, self.plugin_action, self.col_name)
        d.widget.setter(self._val or '')
        if d.exec_() == d.Accepted:
            self._val = d.val

    def getter(self):
        return self._val

    def setter(self, val):
        self._val = val

# new class
class BulkLongText(BulkBase2, LongText):

    def setup_ui(self, parent):
        self._box = QGroupBox(parent)
        self._box.setTitle(label_string(self.col_metadata['name']))
        self._layout = QVBoxLayout()
        self._tb = self.editor = QPlainTextEdit(self._box)
        self._tb.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Minimum)
        self._layout.addWidget(self._tb)
        self._box.setLayout(self._layout)
        self.a_c_checkbox = QCheckBox(_('Apply to all'))
        self._layout.addWidget(self.a_c_checkbox)
        self._box.setLayout(self._layout)
        self.widgets = [self._box]
        #
        self.connect_to_apply_changes()
        #

# new class
class BulkSimpleText(BulkBase2, SimpleText):

    def setup_ui(self, parent):
        self.widgets = [QLabel(label_string(self.col_metadata['name']), parent),
                        QLineEdit(parent)]
        self.a_c_checkbox = QCheckBox(_('Apply to all'))
        self.widgets.append(self.a_c_checkbox)
        #
        self.editor = self.widgets[1]
        self.connect_to_apply_changes()
        #

# new class
class Marked(Base):

    def __init__(self, plugin_action, col_name, parent=None):
        QWidget.__init__(self)
        self.col_name = col_name
        self.plugin_action = plugin_action
        self.db = self.plugin_action.gui.current_db
        self.col_metadata = self.db.field_metadata.all_metadata()[col_name]
        self.initial_val = self.widgets = None
        self.signals_to_disconnect = []
        #self.setup_ui(parent)
        self.setup_ui(self)
        self.initControls()

    def setup_ui(self, parent):
        self.parent = parent

        w = EditWithComplete(parent)
        w.set_separator(', ')
        w.setSizeAdjustPolicy(QComboBox.SizeAdjustPolicy.AdjustToMinimumContentsLengthWithIcon)
        w.setMinimumContentsLength(25)
        self.widgets = [QLabel(_('marked')), w]

    def initialize(self, book_id):
        values = all_marked_values(self.plugin_action)
        values.sort(key=sort_key)
        self.book_id = book_id
        self.widgets[1].clear()
        self.widgets[1].update_items_cache(values)
        val = self.db.data.get_marked(book_id)
        if val is None:
            val = ''
        self.initial_val = val
        val = self.normalize_db_val(val)

        self.setter(val)

    def normalize_db_val(self, val):
        return [x.strip() for x in val.split(',') if x.strip()]

    def normalize_ui_val(self, val):
        return val

    def setter(self, val):
        if not val:
            val = []
        self.widgets[1].setText(', '.join(val))

    def getter(self):
        val = str(self.widgets[1].text()).strip()
        return val

    def commit(self, book_id, notify=False):
        val = self.current_val
        if val != self.initial_val:
            return set_marked_for_book(self.plugin_action, book_id, val)
        else:
            return set()

    def connect_data_changed(self, slot):
        s = self.widgets[1].tags_box.currentTextChanged
        s.connect(slot)
        self.signals_to_disconnect.append(s)

# new class
class BulkMarked(BulkBase):

    def __init__(self, plugin_action, col_name, parent=None):
        QWidget.__init__(self)
        self.col_name = col_name
        self.plugin_action = plugin_action
        self.db = self.plugin_action.gui.current_db
        self.col_metadata = self.db.field_metadata.all_metadata()[col_name]
        self.col_metadata['name'] = 'marked'
        self.col_metadata['display'] = {
            "description": "",
            "is_names": False
        }
        self.initial_val = self.widgets = None
        self.signals_to_disconnect = []
        #self.setup_ui(parent)
        self.setup_ui(self)
        self.initControls()

    def setup_ui(self, parent):
        values = self.all_values = all_marked_values(self.plugin_action)
        values.sort(key=sort_key)

        self.make_widgets(parent, EditWithComplete, add_tags_edit_button=False)
        self.main_widget.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Preferred)
        self.adding_widget = self.main_widget

        w = RemoveTags(parent, values)
        w.remove_tags_button.hide()
        l = QLabel(label_string('marked')+': ' +
                                   _('items to remove'), parent)
        tt = 'marked' + ': ' + _('items to remove')
        l.setToolTip(tt)
        self.widgets.append(l)
        w.setToolTip(tt)
        self.widgets.append(w)
        self.removing_widget = w
        self.main_widget.set_separator(', ')
        w.tags_box.textChanged.connect(self.a_c_checkbox_changed)
        w.checkbox.stateChanged.connect(self.a_c_checkbox_changed)

        self.ignore_change_signals = False
        self.parent = parent

    @property
    def gui_val(self):
        return self.getter()

    def initialize(self, book_ids):
        self.main_widget.update_items_cache(self.all_values)

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        remove_all, add, remove = self.current_val
        add = [item.strip() for item in add.split(',') if item.strip()] if add else []
        remove = [item.strip() for item in remove.split(',') if item.strip()] if remove else []
        return bulk_modify_marked(self.plugin_action, book_ids, add=add, remove=remove, remove_all=remove_all)

    def getter(self):
        remove_all = self.removing_widget.checkbox.isChecked()
        add = str(self.adding_widget.text())
        remove = str(self.removing_widget.tags_box.text())
        return remove_all, add, remove

    def setter(self, val):
        remove_all, add, remove = val
        self.removing_widget.checkbox.setChecked(remove_all)
        self.adding_widget.setText(add)
        self.removing_widget.tags_box.setText(remove)

# new class
class BulkFormats(BulkBase):

    def __init__(self, plugin_action, col_name, parent=None):
        QWidget.__init__(self)
        self.col_name = col_name
        self.plugin_action = plugin_action
        self.db = self.plugin_action.gui.current_db
        self.initial_val = self.widgets = None
        self.signals_to_disconnect = []
        #self.setup_ui(parent)
        self.setup_ui(self)
        self.initControls()

    def setup_ui(self, parent):
        self.all_opt = QRadioButton(_('Remove all formats'))
        self.all_opt.setChecked(True)

        self.include_opt = QRadioButton(_('Remove only specified formats (comma separated list)'))
        self.include_edit = QLineEdit()

        self.exclude_opt = QRadioButton(_('Remove all formats except specified (comma separated list)'))
        self.exclude_edit = QLineEdit()

        self.widgets = [self.all_opt, self.include_opt, self.include_edit, self.exclude_opt, self.exclude_edit]
        self.a_c_checkbox = QCheckBox(_('Apply to all'))
        self.widgets.append(self.a_c_checkbox)

    @property
    def gui_val(self):
        return self.getter()

    def initialize(self, book_ids):
        pass

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        settings = self.current_val
        if settings['action_opt'] == 'remove':
            for book_id in book_ids:
                fmts_to_delete = set()
                fmts_string = self.db.formats(book_id, index_is_id=True)
                if fmts_string:
                    available_fmts = [ fmt.strip().upper() for fmt in fmts_string.split(',') ]
                else:
                    available_fmts = []
                if settings['remove_opt'] == 'all':
                    if available_fmts:
                        fmts_to_delete = available_fmts
                elif settings['remove_opt'] == 'include':
                    fmts_to_delete = [ fmt.strip().upper() for fmt in settings.get('include', '').split(',') ]
                elif settings['remove_opt'] == 'exclude':
                    fmts_to_keep = set([ fmt.strip().upper() for fmt in settings.get('exclude', '').split(',') ])
                    fmts_to_delete = set(available_fmts).difference(fmts_to_keep)
                for fmt in fmts_to_delete:
                    fmt = fmt.upper()
                    if fmt in available_fmts:
                        self.db.remove_format(book_id, fmt, index_is_id=True, notify=False)

    def getter(self):
        settings = {}
        # in case we add other actions in the future
        settings['action_opt'] = 'remove'
        #
        if self.all_opt.isChecked():
            settings['remove_opt'] = 'all'
        elif self.include_opt.isChecked():
            settings['remove_opt'] = 'include'
            settings['include'] = self.include_edit.text()
        elif self.exclude_opt.isChecked():
            settings['remove_opt'] = 'exclude'
            settings['exclude'] = self.exclude_edit.text()
        return settings

    def setter(self, val):
        settings = val
        if settings:
            if settings['action_opt'] == 'remove':
                if settings['remove_opt'] == 'all':
                    self.all_opt.setChecked(True)
                elif settings['remove_opt'] == 'include':
                    self.include_opt.setChecked(True)
                    self.include_edit.setText(settings['include'])
                elif settings['remove_opt'] == 'exclude':
                    self.exclude_opt.setChecked(True)
                    self.exclude_edit.setText(settings['exclude'])

# new class

class CoverSettingsDialog(Dialog):
    def __init__(self, parent, settings=None):
        self.settings = settings
        Dialog.__init__(self, 'Configure Cover Generation Dialog', 'action-chains-configure-cover-generation-dialog', parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)
        self.widget = CoverSettingsWidget()
        l.addWidget(self.widget)
        if self.settings:
            self.load_settings(self.settings)

        l.addWidget(self.bb)

    def load_settings(self, settings):
        self.widget.apply_prefs(settings)

    def accept(self):
        self.settings = self.widget.current_prefs
        Dialog.accept(self)

class BulkCover(BulkBase):

    def __init__(self, plugin_action, col_name, parent=None):
        QWidget.__init__(self)
        self.col_name = col_name
        self.plugin_action = plugin_action
        self.gui = self.plugin_action.gui
        self.db = self.plugin_action.gui.current_db
        self.initial_val = self.widgets = None
        self.generate_cover_opts = None
        self.signals_to_disconnect = []
        #self.setup_ui(parent)
        self.setup_ui(self)
        self.initControls()

    def setup_ui(self, parent):
        self.set_opt = QRadioButton(_('Set cover from book'))
        self.set_opt.setChecked(True)
        self.generate_opt = QRadioButton(_('Generate cover'))
        self.trim_opt = QRadioButton(_('Trim cover'))
        self.configure_generation_button = QPushButton(_('Configure cover generation'))
        self.configure_generation_button.clicked.connect(self._on_configure_generation_button_clicked)
        
        widget = QWidget()
        l = QHBoxLayout()
        widget.setLayout(l)
        for w in [self.set_opt, self.generate_opt, self.trim_opt]:
            l.addWidget(w)

        self.widgets = [widget, self.configure_generation_button]
        self.a_c_checkbox = QCheckBox(_('Apply to all'))
        self.widgets.append(self.a_c_checkbox)

    def _on_configure_generation_button_clicked(self):
        d = CoverSettingsDialog(self, self.generate_cover_opts)
        if d.exec() == d.Accepted:
            self.generate_cover_opts = d.settings

    def set_cover_from_books(self, book_ids):
        from calibre.ebooks.metadata.meta import metadata_from_formats
        cache = self.db.new_api
        for book_id in book_ids:
            fmts = cache.formats(book_id, verify_formats=False)
            paths = list(filter(None, [cache.format_abspath(book_id, fmt) for fmt in fmts]))
            mi = metadata_from_formats(paths)
            if mi.cover_data:
                cdata = mi.cover_data[-1]
            if cdata is not None:
                cache.set_cover({book_id: cdata})

    def generate_covers(self, book_ids, generate_cover_opts):
        from calibre.ebooks.covers import generate_cover
        for book_id in book_ids:
            mi = self.db.get_metadata(book_id, index_is_id=True)
            cdata = generate_cover(mi, generate_cover_opts)
            self.db.new_api.set_cover({book_id:cdata})

    def trim_covers_callback(self, book_ids, pbar):

        from calibre.utils.img import (
            image_from_data, image_to_data, remove_borders_from_image
        )
        
        pbar.update_overall(len(book_ids))
        
        for book_id in book_ids:
            cdata = self.db.new_api.cover(book_id)
            if cdata:
                img = image_from_data(cdata)
                nimg = remove_borders_from_image(img)
                if nimg is not img:
                    cdata = image_to_data(nimg)
                    self.db.new_api.set_cover({book_id:cdata})
            msg = _(f'Trimming cover for book_id: {book_id}')
            pbar.update_progress(1, msg)

    def trim_covers(self, book_ids):
        from calibre_plugins.action_chains.common_utils import DoubleProgressDialog
        callback = partial(self.trim_covers_callback, book_ids)
        pd = DoubleProgressDialog(1, callback, self.gui, window_title=_('Trimming ...'))

        self.gui.tags_view.blockSignals(True)
        QApplication.setOverrideCursor(Qt.ArrowCursor)
        try:
            pd.exec()

            pd.thread = None

            if pd.error is not None:
                return error_dialog(self.gui, _('Failed'),
                        pd.error[0], det_msg=pd.error[1],
                        show=True)
        finally:
            QApplication.restoreOverrideCursor()
            self.gui.tags_view.recount()

    @property
    def gui_val(self):
        return self.getter()

    def initialize(self, book_ids):
        pass

    def commit(self, book_ids, notify=False):
        if not self.a_c_checkbox.isChecked():
            return
        settings = self.current_val
        action_opt = settings['action_opt']
        if action_opt == 'set_from_book':
            self.set_cover_from_books(book_ids)
        elif action_opt == 'generate_cover':
            self.generate_covers(book_ids, self.generate_cover_opts)
        elif action_opt == 'trim_cover':
            self.trim_covers(book_ids)

    def getter(self):
        settings = {}
        # in case we add other actions in the future
        settings['action_opt'] = 'remove'
        if self.set_opt.isChecked():
            settings['action_opt'] = 'set_from_book'
        elif self.generate_opt.isChecked():
            settings['action_opt'] = 'generate_cover'
        elif self.trim_opt.isChecked():
            settings['action_opt'] = 'trim_cover'
        settings['generate_cover_opts'] = self.generate_cover_opts
        return settings

    def setter(self, val):
        settings = val
        if settings:
            if settings['action_opt'] == 'set_from_book':
                self.set_opt.setChecked(True)
            elif settings['action_opt'] == 'generate_cover':
                self.generate_opt.setChecked(True)
            elif settings['action_opt'] == 'trim_cover':
                self.trim_opt.setChecked(True)
            self.generate_cover_opts = settings.get('generate_cover_opts')

#def comments_factory(plugin_action, col_name, parent):
    #db = plugin_action.gui.current_db
    #fm = db.field_metadata.all_metadata()[col_name]
    #ctype = fm.get('display', {}).get('interpret_as', 'html')
    #if fm['is_custom']:
        #if ctype == 'short-text':
            #return SimpleText(plugin_action, col_name, parent)
        #if ctype in ('long-text', 'markdown'):
            #return LongText(plugin_action, col_name, parent)
    #return Comments(plugin_action, col_name, parent)

#def bulk_comments_factory(plugin_action, col_name, parent):
    #db = plugin_action.gui.current_db
    #fm = db.field_metadata.all_metadata()[col_name]
    #ctype = fm.get('display', {}).get('interpret_as', 'html')
    #if fm['is_custom']:
        #if ctype == 'short-text':
            #return BulkSimpleText(plugin_action, col_name, parent)
        #if ctype in ('long-text', 'markdown'):
            #return BulkLongText(plugin_action, col_name, parent)
    #return BulkComments(plugin_action, col_name, parent)


single_widgets = {
        'bool' : Bool,
        'rating' : Rating,
        'int': Int,
        'float': Float,
        'datetime': DateTime,
        'text' : Text,
        'comments': Comments,
        'series': Series,
        'enumeration': Enumeration,
        'marked': Marked,
        'formats': BulkFormats,
        'cover': BulkCover,
        'short-text': SimpleText,
        'long-text': LongText
}


bw = [
    ('bool', BulkBool),
    ('rating', BulkRating),
    ('int', BulkInt),
    ('float', BulkFloat),
    ('datetime', BulkDateTime),
    ('text', BulkText),
    ('series', BulkSeries),
    ('enumeration', BulkEnumeration),
    ('comments', BulkComments),
    ('cover', BulkCover),
    ('formats', BulkFormats),
    ('long-text', BulkLongText),
    ('short-text', BulkSimpleText),
    ('marked', BulkMarked)
]


bulk_widgets = OrderedDict()
for k, v in bw:
    bulk_widgets[k] = v

predefined_widgets = copy.deepcopy(bulk_widgets)
predefined_widgets['datetime'] = PredefinedDateTime
predefined_widgets['comments'] = PredefinedComments

def get_metadata_widget(db, col_name, mode='single'):
    w = {
            'single': single_widgets,
            'bulk': bulk_widgets,
            'predefined': predefined_widgets
    }
    fm = db.field_metadata.all_metadata()[col_name]
    interpret_as = fm.get('display', {}).get('interpret_as')
    # comments column type in fm is text unlike custom comments columns
    # so we have to test for it separtely at first
    if col_name in ['marked','cover','formats','comments']:
        return w[mode][col_name]
    elif interpret_as == 'html':
        coltype = 'comments'
    if interpret_as == 'short-text':
        coltype = 'short-text'
    if interpret_as in ('long-text', 'markdown'):
        coltype = 'long-text'
    else:
        coltype = fm['datatype']
    return w[mode][coltype]

#def get_metadata_widget(db, col_name, mode='single'):
    ## comments column type in fm is text unlike custom comments columns
    ## so we have to test for it separtely at first
    #if col_name == 'comments':
        #if mode in ['bulk','predefined']:
            #return BulkComments
        #else:
            #return Comments
    #elif col_name == 'marked':
        #if mode in ['bulk','predefined']:
            #return BulkMarked
        #else:
            #return Marked
    #elif col_name == 'cover':
        #return BulkCover
    #elif col_name == 'formats':
        #return BulkFormats
    #else:
        #typ = db.field_metadata.all_metadata()[col_name]['datatype']
        #if mode == 'bulk':
            #return bulk_widgets[typ]
        #elif mode == 'predefined':
            #return predefined_widgets[typ]
        #else:
            #return widgets[typ]

def get_predefined_widget_default(plugin_action, col_name):
    db = plugin_action.gui.current_db
    widget_cls = get_metadata_widget(db, col_name, mode='predefined')
    w = widget_cls(plugin_action, col_name, None)
    w.resize(QSize(0, 0))
    val = w.current_val
    del w
    return val
