#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QAbstractTableModel, QModelIndex,
                     QSortFilterProxyModel, QAbstractProxyModel,
                     QImage, QBrush, pyqtSignal, QStyle)

from calibre import prints
from calibre.gui2 import error_dialog
from calibre.constants import DEBUG

from calibre_plugins.action_chains.common_utils import get_icon, get_pixmap, call_method
from calibre_plugins.action_chains.chains import Chain
from calibre_plugins.action_chains.scopes.scope_tools import validate_scope
import calibre_plugins.action_chains.config as cfg

from calibre.constants import numeric_version
if numeric_version >= (5,99,0):
    QT_CHECKED = Qt.CheckState.Checked.value
    QT_UNCHECKED = Qt.CheckState.Unchecked.value
else:
    QT_CHECKED = Qt.Checked
    QT_UNCHECKED = Qt.Unchecked

try:
    load_translations()
except NameError:
    prints("ActionsChain::gui/models.py - exception when loading translations")

DOWN    = 1
UP      = -1

def get_actual_index(index):
    model = index.model()
    if isinstance(model, QAbstractProxyModel):
        return model.mapToSource(index)
    else:
        return index

class ActionsModel(QAbstractTableModel):

    def __init__(self, plugin_action, chains_config, chain_name):
        # We are passing the chains_config and chain_name and extracting the chain_links from them.
        # This is done because some actions like "Chain Caller" need to have access to the chain_name
        # and to chains_config.
        QAbstractTableModel.__init__(self)
        self.chains_config = chains_config
        self.chain_name = chain_name
        self.chain_config = cfg.get_chain_config(chain_name, chains_config=chains_config)
        self.chain_links = self.chain_config.get('chain_settings', {}).get('chain_links', [])
        self.plugin_action = plugin_action
        self.col_map = ['conditions','scope','action_name','action_settings','comment','errors']
        self.editable_columns = ['action_name','comment']
        #self.hidden_cols = ['errors']
        self.optional_cols = ['comment','conditions','scope']
        self.hidden_cols = []
        self.col_min_width = {
            'conditions': 25,
            'scope': 25,
            'action_name': 300,
            'action_settings': 50,
            'comment': 200,
            'errors': 250
        }
        all_headers = [_('Conditions'),_('Scope'),_('Action'),_('Settings'),_('Comment'),_('Errors')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.chain_links)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.chain_links):
            return None
        chain_link = self.chain_links[row]
        col_name = self.col_map[col]
        value = chain_link.get(col_name, '')
        error = chain_link.get('errors', '')
        is_conditions, support_scopes, is_scope = self.scope_and_conditions(chain_link)
        condition_settings = chain_link.get('condition_settings', {})
        scope_settings = chain_link.get('scope_settings', {})
        
        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name in ['action_settings']:
                pass
            elif col_name == 'errors':
                if error:
                    return _('Validation Error. Double click for details')
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'errors':
                if error:
                    return get_icon('dialog_error.png')
            elif col_name == 'conditions':
                if is_conditions:
                    icon = condition_settings.get('icon')
                    if not icon: icon = 'images/condition.png'
                else:
                    icon = 'images/no_condition.png'
                return get_icon(icon)
                    
            elif col_name == 'scope':
                if support_scopes:
                    if is_scope:
                        icon = scope_settings.get('icon')
                        if not icon: icon = 'images/scope.png'
                    else:
                        icon = 'images/no_scope.png'
                    return get_icon(icon)
                else:
                    icon = 'images/scope_unsupported.png'
                    return get_icon(icon)
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error
                    
            elif col_name == 'conditions':
                if is_conditions:
                    tooltip = condition_settings.get('tooltip')
                    if not tooltip: tooltip = _('Conditions set for running this action')
                else:
                    tooltip = _('No condition is set for this action')
                return tooltip

            elif col_name == 'comment':
                tooltip = chain_link.get('comment')
                return tooltip
                        
            elif col_name == 'scope':
                if support_scopes:
                    if is_scope:
                        tooltip = scope_settings.get('tooltip') 
                        if not tooltip: tooltip = _('Scope is set for this action')
                    else:
                        tooltip = _('No scope is set for this action')
                else:
                    tooltip = _('Action does not support scopes')
                return tooltip

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        chain_link = self.chain_links[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name == 'action_name':
                old_name = chain_link.get('action_name', '')
                chain_link['action_name'] = val
                if val != old_name:
                    # reset settings as they are not valid for this new action
                    chain_link['action_settings'] = {}
                    # reset any previous errors
                    chain_link['errors'] = ''
            elif col_name == 'settings':
                pass
            elif col_name in ['comment']:
                chain_link[col_name] = val
            done = True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            chain_link = self.chain_links[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
        return flags

    def button_state(self, index):
        visible = False
        enabled = False
        row, col = index.row(), index.column()
        chain_link = self.chain_links[row]
        action_name = chain_link.get('action_name')
        if action_name:
            visible = True
            action = self.plugin_action.actions.get(action_name)
            config_widget = None
            if action:
                config_widget = action.config_widget()
            if config_widget:
                enabled = True
        return visible, enabled
        

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            chain_link = {}
            chain_link['action_name'] = ''
            chain_link['action_settings'] = {}
            chain_link['comment'] = ''
            self.chain_links.insert(row + i, chain_link)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.chain_links.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        self.layoutAboutToBeChanged.emit()
        for row in srows:
            pop = self.chain_links.pop(row)
            self.chain_links.insert(row+direction, pop)
        self.layoutChanged.emit()   

    def scope_and_conditions(self, chain_link):
        condition_settings = chain_link.get('condition_settings', {})
        scope_settings = chain_link.get('scope_settings', {})
        is_conditions = condition_settings.get('template')
        support_scopes = False
        is_scope = False
        action_name = chain_link['action_name']
        action = self.plugin_action.actions.get(action_name)
        if action:
            support_scopes = getattr(action, 'support_scopes', False)
            if support_scopes:
                is_scope = scope_settings.get('scope_manager_name')
        return is_conditions, support_scopes, is_scope

    def validate_chain_link(self, row, chain_link):
        col = self.col_map.index('errors')
        action_name = chain_link['action_name']
        if action_name in self.plugin_action.actions.keys():
            action = self.plugin_action.actions[action_name]
            action_settings = chain_link.get('action_settings')
            if not action_settings:
                action_settings = action.default_settings()
            is_action_valid = action.validate(action_settings)
            if is_action_valid is not True:
                msg, details = is_action_valid
                chain_link['errors'] = details
                index = self.index(row, col, QModelIndex())
                self.dataChanged.emit(index, index)
                return is_action_valid
            # validate scope settings only if scopes are enabled
            is_scope_valid = validate_scope(self.plugin_action, chain_link)
            if is_scope_valid is not True:
                msg, details = is_scope_valid
                chain_link['errors'] = details
                index = self.index(row, col, QModelIndex())
                self.dataChanged.emit(index, index)
                return is_scope_valid
        else:
            details = _(f'Action ({action_name}) is not currently available')
            chain_link['errors'] = details
            index = self.index(row, col, QModelIndex())
            self.dataChanged.emit(index, index)
            return (_('Action unavailable'), details)
        return True

    def validate(self):
        is_chain_valid = True
        for row, chain_link in enumerate(self.chain_links):
            is_action_valid = self.validate_chain_link(row, chain_link)
            if is_action_valid is not True:
                is_chain_valid = False
        return is_chain_valid

class MenusModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, chains_config, plugin_action):
        QAbstractTableModel.__init__(self)
        self.chains_config = chains_config
        self.plugin_action = plugin_action
        self.col_map = [
            'active',
            'conditions',
            'menuText',
            'subMenu',
            'settings',
            #'image'
        ]
        self.editable_columns = [
            'menuText',
            'subMenu',
            #'image'
        ]
        self.hidden_cols = []
        self.optional_cols = ['conditions']
        self.col_min_width = {
            'active': 10,
            'conditions': 25,
            'subMenu': 50,
            'menuText': 200,
            'settings': 30,
            #'image': 200
        }
        all_headers = [
            '',
            _('Conditions'),
            _('Title'),
            _('Submenu'),
            _('Settings'),
            #_('Icon')
        ]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.chains_config)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.chains_config):
            return None
        chain_config = self.chains_config[row]
        col_name = self.col_map[col]
        value = chain_config.get(col_name, '')
        error = chain_config.get('errors', '')
        condition_settings = chain_config.get('condition_settings', {})
        is_conditions = condition_settings.get('template')

        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name in ['active','settings','conditions']:
                pass
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'menuText':
                icon_name = chain_config.get('image', '')
                if icon_name:
                    return self.plugin_action.icon_cache.get_icon(icon_name)
            #if col_name == 'image':
                #if value:
                    #return self.plugin_action.icon_cache.get_icon(value)
            elif col_name == 'errors':
                if error:
                    return get_icon('dialog_error.png')
            elif col_name == 'conditions':
                if is_conditions:
                    icon = condition_settings.get('icon')
                    if not icon: icon = 'images/condition.png'
                else:
                    icon = 'images/no_condition.png'
                return get_icon(icon)
            elif col_name == 'active':
                variant_config = chain_config.get('variation_settings')
                if variant_config:
                    icon = 'images/multiple.png'
                    return get_icon(icon)
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error
            elif col_name == 'conditions':
                if is_conditions:
                    tooltip = condition_settings.get('tooltip')
                    if not tooltip: tooltip = _('Conditions set for running this chain')
                else:
                    tooltip = _('No condition is set for this chain')
                return tooltip
            elif col_name == 'active':
                has_variants = chain_config.get('variation_settings', {}).get('template', '')
                if has_variants:
                    tooltip = _('Chain variants set for this chain')
                    return tooltip

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                is_checked = chain_config[col_name]
                state = QT_CHECKED if is_checked else QT_UNCHECKED
                return state

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        chain_config = self.chains_config[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name in ['active','settings']:
                pass
            # make sure the name is unique
            elif col_name == 'menuText':
                old_name = self.data(index, Qt.ItemDataRole.DisplayRole)
                names = self.get_names()
                if old_name in names:
                    names.remove(old_name)
                if val in names:
                    msg = _('Name must be unique')
                    details = _(f'Name ({val}) is already used by another menu')
                    self.error.emit(msg, details)
                else:
                    chain_config[col_name] = val
            else:
                chain_config[col_name] = val
            done = True

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                state = True if (value == QT_CHECKED) else False
                chain_config[col_name] = state
            return True
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            chain_config = self.chains_config[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
            elif col_name == 'subMenu':
                menu_col = self.col_map.index('menuText')
                menu_index = self.index(index.row(), menu_col, QModelIndex())
#                if self.data(menu_index, Qt.ItemDataRole.DisplayRole):
#                    flags |= Qt.ItemFlag.ItemIsEditable
                # Even if no menu name is given, submenu should be editable to enable adding separators
                # to submenus
                flags |= Qt.ItemFlag.ItemIsEditable
            if col_name == 'active':
                flags |=  Qt.ItemFlag.ItemIsUserCheckable
        return flags

    def button_state(self, index):
        visible = False
        enabled = True
        row, col = index.row(), index.column()
        chain_config = self.chains_config[row]
        chain_name = chain_config.get('menuText')
        if chain_name:
            visible = True
        return visible, enabled

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            chain_config = {}
            chain_config['active'] = True
            chain_config['menuText'] = ''
            chain_config['subMenu'] = ''
            chain_config['image'] = ''
            chain_config['chain_settings'] = {}
            self.chains_config.insert(row + i, chain_config)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.chains_config.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        self.layoutAboutToBeChanged.emit()
        for row in srows:
            pop = self.chains_config.pop(row)
            self.chains_config.insert(row+direction, pop)
        self.layoutChanged.emit()

    def get_names(self):
        names = []
        col = self.col_map.index('menuText')
        for row in range(self.rowCount(QModelIndex())):
            index = self.index(row, col, QModelIndex())
            name = self.data(index, Qt.ItemDataRole.DisplayRole)
            # empty name belong to separators, dont include
            if name:
                names.append(name)
        return names

class MenusFilterModel(QSortFilterProxyModel):

    def __init__(self, parent):
        QSortFilterProxyModel.__init__(self, parent)
        self.setSortRole(Qt.UserRole)
        self.setSortCaseSensitivity(Qt.CaseInsensitive)
        self.filter_text = ''
        self.match_tier = None

    def filterAcceptsRow(self, sourceRow, sourceParent):
        index = self.sourceModel().index(sourceRow, 0, sourceParent)
        chain_config = self.sourceModel().chains_config[index.row()]
        chain_name = chain_config['menuText']
        if self.filter_text.lower() in chain_name.lower():
            return True
        else:
            return False

    def set_filter_text(self, filter_text):
        self.filter_text = filter_text
        self.invalidateFilter()

class EventMembersModel(QAbstractTableModel):

    def __init__(self, event_members, plugin_action):
        QAbstractTableModel.__init__(self)
        self.event_members = event_members
        self.plugin_action = plugin_action
        self.col_map = ['icons','chain_name','errors']
        self.editable_columns = ['chain_name']
        #self.hidden_cols = ['errors']
        self.hidden_cols = []
        self.col_min_width = {
            'chain_name': 300,
            'icons': 25
        }
        all_headers = ['', _('Chain'),_('Errors')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.event_members)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.event_members):
            return None
        event_member = self.event_members[row]
        col_name = self.col_map[col]
        value = event_member.get(col_name, '')
        error = event_member.get('errors', '')

        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name == 'errors':
                if error:
                    return _('Chain error. Open chain dialog for more details')
            elif col_name in ['active','settings']:
                pass
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'errors':
                if error:
                    return get_icon('dialog_error.png')
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)
        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        event_member = self.event_members[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name == 'settings':
                pass
            else:
                event_member[col_name] = val
            done = True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            event_member = self.event_members[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
        return flags

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            event_member = {}
            event_member['chain_name'] = ''
            self.event_members.insert(row + i, event_member)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.event_members.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        self.layoutAboutToBeChanged.emit()
        for row in srows:
            pop = self.event_members.pop(row)
            self.event_members.insert(row+direction, pop)
        self.layoutChanged.emit()

    def validate_event_member(self, row, event_member):
        col = self.col_map.index('errors')
        chain_name = event_member['chain_name']
        chains_config = cfg.get_chains_config()
        all_chain_names = [chain_config['menuText'] for chain_config in chains_config]
        if chain_name in all_chain_names:
            chain_config = cfg.get_chain_config(chain_name)
            chain = Chain(self.plugin_action, chain_config)
            is_chain_valid = chain.validate(use_cache=False)
            if is_chain_valid is not True:
                msg, details = is_chain_valid
                event_member['errors'] = details
                index = self.index(row, col, QModelIndex())
                self.dataChanged.emit(index, index)
                return is_chain_valid
        else:
            details = _(f'Chain ({chain_name}) is not currently available')
            event_member['errors'] = details
            index = self.index(row, col, QModelIndex())
            self.dataChanged.emit(index, index)
            return (_('Chain unavailable'), details)
        return True

    def validate(self):
        is_chain_valid = True

        # init_cache for template functions, we do it here instead of in the chain.validate()
        # to avoid flusing it after each chain is checked
        call_method(self.plugin_action.template_functions, 'init_cache')
        try:
            for row, event_member in enumerate(self.event_members):
                is_chain_valid = self.validate_event_member(row, event_member)
                if is_chain_valid is not True:
                    is_chain_valid = False
        finally:
            call_method(self.plugin_action.template_functions, 'flush_cache')
        return is_chain_valid

class EventsModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, events_config, plugin_action):
        QAbstractTableModel.__init__(self)
        self.events_config = events_config
        self.plugin_action = plugin_action
        self.col_map = ['active','event_name','event_settings','errors']
        self.editable_columns = ['event_name']
        #self.hidden_cols = ['errors']
        self.hidden_cols = []
        self.col_min_width = {
            'active': 25,
            'event_name': 300,
            'event_settings': 25
        }
        all_headers = ['', _('Event'), _('Settings'), _('errors')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.events_config)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.events_config):
            return None
        event_config = self.events_config[row]
        col_name = self.col_map[col]
        value = event_config.get(col_name, '')
        error = event_config.get('errors', '')

        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name == 'errors':
                if error:
                    return error
            elif col_name in ['active','settings']:
                pass
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'errors':
                if error:
                    return get_icon('dialog_error.png')
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                is_checked = event_config[col_name]
                state = QT_CHECKED if is_checked else QT_UNCHECKED
                return state

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        event_config = self.events_config[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name == 'settings':
                pass
            # make sure no duplicate event entries
            elif col_name == 'event_name':
                old_name = self.data(index, Qt.ItemDataRole.DisplayRole)
                names = self.get_names()
                if old_name in names:
                    names.remove(old_name)
                if val in names:
                    msg = _('Duplicate event')
                    details = _(f'Name ({val}) is used in more than one entry')
                    self.error.emit(msg, details)
                else:
                    event_config[col_name] = val
            else:
                event_config[col_name] = val
            done = True

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                state = True if (value == QT_CHECKED) else False
                event_config[col_name] = state
            return True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            event_config = self.events_config[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
            if col_name == 'active':
                flags |=  Qt.ItemFlag.ItemIsUserCheckable
        return flags

    def button_state(self, index):
        visible = False
        enabled = True
        row, col = index.row(), index.column()
        event_config = self.events_config[row]
        event_name = event_config.get('event_name')
        if event_name:
            visible = True
        return visible, enabled

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            event_config = {}
            event_config['active'] = True
            event_config['event_name'] = ''
            event_config['event_settings'] = {}
            self.events_config.insert(row + i, event_config)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.events_config.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        self.layoutAboutToBeChanged.emit()
        for row in srows:
            pop = self.events_config.pop(row)
            self.events_config.insert(row+direction, pop)
        self.layoutChanged.emit()

    def get_names(self):
        names = []
        col = self.col_map.index('event_name')
        for row in range(self.rowCount(QModelIndex())):
            index = self.index(row, col, QModelIndex())
            name = self.data(index, Qt.ItemDataRole.DisplayRole)
            # empty name belong to separators, dont include
            if name:
                names.append(name)
        return names

    def validate(self):
        for event_config in self.events_config:
            event_name = event_config['event_name']
            if event_name not in self.plugin_action.events.keys():
                event_config['errors'] = _('Event is not available')


class EventVariantsModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, variants_config, plugin_action):
        QAbstractTableModel.__init__(self)
        self.variants_config = variants_config
        self.plugin_action = plugin_action
        self.col_map = ['variant_name','parent_event','errors']
        self.editable_columns = ['variant_name','parent_event']
        #self.hidden_cols = ['errors']
        self.hidden_cols = []
        self.col_min_width = {
            'variant_name': 300,
            'parent_event': 300
        }
        all_headers = [_('Variant Name'), _('Parent Event'), _('errors')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.variants_config)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.variants_config):
            return None
        variant_config = self.variants_config[row]
        col_name = self.col_map[col]
        value = variant_config.get(col_name, '')
        error = variant_config.get('errors', '')

        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name == 'errors':
                if error:
                    return error
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'errors':
                if error:
                    return get_icon('dialog_error.png')
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        variant_config = self.variants_config[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name == 'variant_name':
                old_name = self.data(index, Qt.ItemDataRole.DisplayRole)
                names = self.get_names()
                if old_name in names:
                    names.remove(old_name)
                if val in names:
                    msg = _('Duplicate name')
                    details = _(f'There already an event with the name: {val}. Choose another name')
                    self.error.emit(msg, details)
                else:
                    variant_config[col_name] = val
            else:
                variant_config[col_name] = val
            done = True            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            variant_config = self.variants_config[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
        return flags

    def button_state(self, index):
        visible = False
        enabled = True
        row, col = index.row(), index.column()
        variant_config = self.variants_config[row]
        event_name = variant_config.get('event_name')
        if event_name:
            visible = True
        return visible, enabled

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            variant_config = {}
            variant_config['variant_name'] = ''
            variant_config['parent_event'] = ''
            self.variants_config.insert(row + i, variant_config)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.variants_config.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        self.layoutAboutToBeChanged.emit()
        for row in srows:
            pop = self.variants_config.pop(row)
            self.variants_config.insert(row+direction, pop)
        self.layoutChanged.emit()

    def get_names(self):
        names = [event_name for event_name in self.plugin_action.events.keys()]
        return names

    def validate(self):
        names = self.get_names()
        for variant_config in self.variants_config:
            parent_event_name = variant_config['parent_event']
            if parent_event_name not in names:
                variant_config['errors'] = _('Parent event is not available')
            else:
                parent_event = self.plugin_action.events[parent_event_name]
                if not getattr(parent_event, 'support_variants', False):
                    variant_config['errors'] = _('Parent event does not support variants')
