#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QToolButton, QPushButton, QSizePolicy, QSpacerItem, 
                     QFont, QIcon, QModelIndex, QCheckBox, QGroupBox, QMenu, QComboBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, question_dialog
from calibre.gui2.widgets2 import Dialog
from calibre.utils.config import tweaks

from calibre_plugins.action_chains.common_utils import get_icon, get_pixmap
from calibre_plugins.action_chains.gui.views import ActionsTable
from calibre_plugins.action_chains.gui.models import ActionsModel, UP, DOWN
from calibre_plugins.action_chains.gui import ImageComboBox
import calibre_plugins.action_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("ActionsChain::gui/actions_dialog.py - exception when loading translations")

class ActionsDialog(Dialog):
    def __init__(self, parent, plugin_action, chains_config, chain_name):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.chain_name = chain_name
        self.chains_config = chains_config
        self.chain_config = cfg.get_chain_config(chain_name, chains_config=chains_config)
        self.chain_settings = self.chain_config['chain_settings']
        self.icon_cache = parent.icon_cache
        Dialog.__init__(self, 'Actions Dialog', 'action-chains-action-dialog', parent)

    def setup_ui(self):
        self.setWindowTitle(_('Add actions'))
        l = QVBoxLayout()
        self.setLayout(l)

        hl1 = QHBoxLayout()
        l.addLayout(hl1)
        image_lbl = QLabel(_('Chain icon'))
        hl1.addWidget(image_lbl)
        image_combobox = self.image_combobox = ImageComboBox(self, self.icon_cache)
        image_combobox.setSizeAdjustPolicy(
            QComboBox.AdjustToMinimumContentsLengthWithIcon)
        hl1.addWidget(image_combobox, stretch=1)
        icon = self.chain_config.get('image')
        if icon:
            image_combobox.set_current_text(icon)
        
        settings_groupbox = QGroupBox()
        l.addWidget(settings_groupbox)
        settings_groupbox_layout = QGridLayout()
        settings_groupbox.setLayout(settings_groupbox_layout)
        refresh_chk = self.refresh_chk = QCheckBox(_('Refresh books after running chain'))
        refresh_chk.setChecked(self.chain_settings.get('refresh_gui', True))
        settings_groupbox_layout.addWidget(refresh_chk, 1, 0, 1, 1)
        show_progress = self.show_progress = QCheckBox(_('Show a progress bar for this chain'))
        show_progress.setChecked(self.chain_settings.get('show_progress', True))
        settings_groupbox_layout.addWidget(show_progress, 1, 1, 1, 1)

        if tweaks.get('action_chains_experimental', False):
            jobify_chk = self.jobify_chk = QCheckBox(_('Run chain as a job (experimental)'))
            jobify_chk.setToolTip(
                _('This option will not work with all chains. '
                'The Presence of some actions in the chain e.g. Calibre Actions\n'
                'will make the chain not run in a job'))
            jobify_chk.setChecked(self.chain_settings.get('jobify', False))
            settings_groupbox_layout.addWidget(jobify_chk, 2, 0, 1, 1)

        #icon_groupbox = QGroupBox(_('Icon'))
        #icon_groupbox_layout = QVBoxLayout()
        #icon_groupbox.setLayout(icon_groupbox_layout)
        #settings_l.addWidget(icon_groupbox, 0, 1, 1, 1)
        #image_combobox = self.image_combobox = ImageComboBox(self, self.plugin_action.icon_cache)
        #icon_groupbox_layout.addWidget(image_combobox)
        #icon = self.chain_config.get('image')
        #if icon:
            #image_combobox.set_current_text(icon)
        #icon_groupbox_layout.addWidget(image_combobox)
        #icon_groupbox_layout.addWidget(QLabel(''))

        table_groupbox = QGroupBox(_('Chain links'))
        table_layout = QHBoxLayout()
        table_groupbox.setLayout(table_layout)
        l.addWidget(table_groupbox)
        
        self._table = ActionsTable(self)
        table_layout.addWidget(self._table)
        
        actions_model = ActionsModel(self.plugin_action, self.chains_config, self.chain_name)
        actions_model.validate()
        self._table.set_model(actions_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)
        
        # Restore table state
        state = cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_ACTIONS_TABLE_STATE]
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        table_layout.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(get_icon('arrow-up.png'))
        button_layout.addWidget(move_up_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add row'))
        add_button.setIcon(get_icon('plus.png'))
        button_layout.addWidget(add_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        copy_button = self.copy_button = QToolButton(self)
        copy_button.setToolTip(_('Duplicate action'))
        copy_button.setIcon(get_icon('edit-copy.png'))
        button_layout.addWidget(copy_button)
        spacerItem3 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem3)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(get_icon('minus.png'))
        button_layout.addWidget(delete_button)
        spacerItem4 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem4)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(get_icon('arrow-down.png'))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)
        copy_button.clicked.connect(self._table.copy_row)
        
        l.addWidget(self.bb)
        
        self._on_table_selection_change()

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.copy_button.setEnabled(selection_count == 1)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def save_table_state(self):
        # Save table state
        cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_ACTIONS_TABLE_STATE] = self._table.get_state()

    def reject(self):
        self.save_table_state()
        Dialog.reject(self)

    def accept(self):
        self.save_table_state()
        
        if self._table.model().rowCount(QModelIndex()) < 1:
            error_dialog(
                self,
                _('Chain invalid'),
                _('Chain must contain at least one action'),
                show=True
            )
            return
        is_valid = self._table.source_model().validate()
        if is_valid is not True:
            message = _('Chain contain some invalid actions. '
                        'Sometimes settings for an action can be valid in one library but not in another. '
                        'Are you sure you want to proceed with saving them anyway?')
            if not question_dialog(self, _('Are you sure?'), message, show_copy_button=False):
                return

        self.chain_settings = {}        
        chain_links = self._table.source_model().chain_links
        # Remove error keys from chain_links
        for chain_link in chain_links:
            try:
                del chain_link['errors']
            except:
                pass
        self.chain_settings['chain_links'] = chain_links
        self.chain_settings['refresh_gui'] = self.refresh_chk.isChecked()
        self.chain_settings['show_progress'] = self.show_progress.isChecked()
        if tweaks.get('action_chains_experimental', False):
            self.chain_settings['jobify'] = self.jobify_chk.isChecked()
        self.icon = self.image_combobox.current_text()
        Dialog.accept(self)
