#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, pyqtSignal, QTimer, QWidget, QVBoxLayout, QGroupBox, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.utils.date import now

from calibre_plugins.action_chains.events.base import ChainEvent
from calibre_plugins.action_chains.events.event_tools import EventLoop

try:
    load_translations()
except NameError:
    prints("ActionChains::events/vl_tab_changed.py - exception when loading translations")

ALL_BOOKS = '_ALL_BOOKS'

class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        group_box = QGroupBox('Options')
        group_box_l = QVBoxLayout()
        group_box.setLayout(group_box_l)
        l.addWidget(group_box)
        self.invisible_chk = QCheckBox(_('Event is active event when tabs are hidden'))
        self.invisible_chk.setChecked(True)
        group_box_l.addWidget(self.invisible_chk)
        
        l.addStretch(1)

        self.setMinimumSize(300,300)

    def load_settings(self, settings):
        if settings:
            self.invisible_chk.setChecked(settings['active_when_tabs_hidden'])

    def save_settings(self):
        settings = {}
        settings['active_when_tabs_hidden'] = self.invisible_chk.isChecked()
        return settings

class VLTabChangedEvent(ChainEvent):

    name = 'VL Tab Changed'

    # signal arguments: current_vl, previous_vl
    vl_tab_changed = pyqtSignal(str, str)

    def __init__(self, plugin_action):
        ChainEvent.__init__(self, plugin_action)
        self.vl_cache = []
        self.vl_time = None
        self.last_vl = str(self.gui.vl_tabs.tabData(self.gui.vl_tabs.currentIndex()) or '').strip() or ALL_BOOKS
        self.plugin_action.gui.vl_tabs.currentChanged.connect(self._on_vl_tab_change)

    def _on_vl_tab_change(self, idx):
        vl = str(self.gui.vl_tabs.tabData(idx) or '').strip() or ALL_BOOKS
        self.vl_cache.append(vl)
        self.vl_time = now()
        # When a VL that does not have a tab is applied (either through Virtual Library
        # dropdown menu or View Manager menu), tabs are re-ordered, and consequently
        # gui.vl_tabs.currentChanged is emitted for each tab, which would lead to the event
        # being run multiple times, affecting performance.
        # We cache all the siganls and only proceed with the last one.
        QTimer.singleShot(250, self.emit_signal)

    def emit_signal(self):
        if self.vl_cache:
            duration = now() - self.vl_time
            if duration.seconds > 0.25:
                vl = self.vl_cache[-1]
                options = self.get_options()
                if not options.get('active_when_tabs_hidden', True) and not self.gui.vl_tabs.isVisible():
                    pass
                else:
                    # In case tab is being dragged
                    if not vl == self.last_vl:
                        self.vl_tab_changed.emit(vl, self.last_vl)
                        if DEBUG:
                            prints('Action Chains: Signal vl_tab_changed emitted')
                self.vl_cache = []
                self.vl_time = None
                self.last_vl = vl
            else:
                QTimer.singleShot(500, self.emit_signal)         
    
    def get_event_signal(self):
        return self.vl_tab_changed

    def config_widget(self):
        return ConfigWidget
