#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QAbstractItemView, pyqtSignal)

from calibre import prints
from calibre.constants import DEBUG
from calibre.utils.config import tweaks

from calibre_plugins.action_chains.events.base import ChainEvent
from calibre_plugins.action_chains.events.event_tools import EventLoop

try:
    load_translations()
except NameError:
    prints("ActionChains::events/book_list_double_clicked.py - exception when loading translations")

class BookListDoubleClickedEvent(ChainEvent):

    name = 'Book List Double Clicked'

    book_list_double_clicked = pyqtSignal(int, str)

    def __init__(self, plugin_action):
        ChainEvent.__init__(self, plugin_action)
        lv = plugin_action.gui.library_view
        # Allow editing on single edit
        doubleclick_on_library_view = tweaks.get('doubleclick_on_library_view')
        single_click_edit = tweaks.get('action_chains_single_click_edit', False)
        if doubleclick_on_library_view not in [ 'open_viewer', 'show_book_details' ]:
            if single_click_edit is True:
                lv.setEditTriggers(QAbstractItemView.EditTrigger.SelectedClicked|lv.editTriggers())
        self.plugin_action.gui.library_view.doubleClicked.connect(self._on_book_list_double_clicked)
        self.plugin_action.gui.grid_view.doubleClicked.connect(self._on_book_list_double_clicked)

    def _on_book_list_double_clicked(self, *args):
        gui = self.plugin_action.gui
        index = gui.library_view.currentIndex()
        book_id = gui.library_view.model().id(index)
        if gui.grid_view_button.isChecked():
            column_name = ''
        else:
            model = index.model()
            column_map = model.column_map
            column_name = column_map[index.column()]
        self.book_list_double_clicked.emit(book_id, column_name)
    
    def get_event_signal(self):
        return self.book_list_double_clicked

    def run(self, *args):
        event_args = args
        # We turn off the qt_processEvents flag because without it the active cell enters
        # into edit mode when double clicked
        event = EventLoop(self.plugin_action, self, event_args, qt_processEvents=False)
        event._run_loop()
        del event
