#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
from collections import OrderedDict
import inspect

from qt.core import (QApplication, Qt)

from calibre import prints
from calibre.constants import DEBUG

import calibre_plugins.action_chains.config as cfg
from calibre_plugins.action_chains.chains import Chain
from calibre_plugins.action_chains.gui.events_dialogs import EventTimer
from calibre_plugins.action_chains.common_utils import call_method
from calibre_plugins.action_chains.events.base import ChainEvent
from calibre_plugins.action_chains.events.library_changed import LibraryChangedEvent
from calibre_plugins.action_chains.events.location_selected import LocationSelectedEvent
from calibre_plugins.action_chains.events.device_connected import DeviceConnectedEvent
from calibre_plugins.action_chains.events.device_disconnected import DeviceDisconnectedEvent
from calibre_plugins.action_chains.events.book_list_double_clicked import BookListDoubleClickedEvent
from calibre_plugins.action_chains.events.book_list_enter_key_pressed import BookListEnterKeyPressedEvent
from calibre_plugins.action_chains.events.calibre_initialized import CalibreInitializedEvent
from calibre_plugins.action_chains.events.timer import TimerEvent
from calibre_plugins.action_chains.events.vl_tab_changed import VLTabChangedEvent
from calibre_plugins.action_chains.events.shutting_down import ShuttingDownEvent
from calibre_plugins.action_chains.events.library_about_to_change import LibraryAboutToChangeEvent




def reorder_dictionary(dictionary, keys):
    new_dict = OrderedDict()
    for key in keys:
        new_dict[key] = dictionary[key]
    return new_dict

def add_to_events_dictionary(event, events_dictionary, signals_cache):
    if event.name in events_dictionary.keys():
        if DEBUG:
            print(f'Action Chains: Event with name ({event.name}) already exists. Skipping ...')
    else:
        signal = event.get_event_signal()
        if signal in signals_cache:
            if DEBUG:
                prints(f'Action Chains: Signal ({signal}) already registered. Skipping ...')
        else:
            events_dictionary[event.name] = event
            signals_cache.add(signal)

def add_event_and_variants(plugin_action, event_obj, events_dictionary, signals_cache):
    if isinstance(event_obj, ChainEvent):
        event = event_obj
        add_to_events_dictionary(event, events_dictionary, signals_cache)
    elif issubclass(event_obj, ChainEvent):
        event = event_obj(plugin_action)
        add_to_events_dictionary(event, events_dictionary, signals_cache)
        event_variants = cfg.get_event_variants(event.name)
        for variant_name in event_variants:
            variant = event_obj(plugin_action)
            variant.name = variant_name
            variant.support_variants = False
            add_to_events_dictionary(variant, events_dictionary, signals_cache)

def get_imported_event_objects(plugin_action):
    # Note: imported events can either be class or instantiated object
    all_imported_objects = {}
    for plugin_name, imported_resources in plugin_action.imported_resources.items():
        imported_objects = imported_resources.get('events', [])
        for imported_event in imported_objects:
            name = imported_event.name
            # must define a name attribute, must be set and not clash with builtin names
            # which can be imported into the module manager by custom events
            if name in ['', 'Chain Event']:
                continue
            imported_event._source_plugin = plugin_name
            all_imported_objects[name] = imported_event          
    return all_imported_objects

def get_user_event_classes(plugin_action):
    all_classes = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[ChainEvent]):
        name = cls.name
        # must define a name attribute
        if name in ['', 'Chain Event']:
            continue
        all_classes[name] = cls     
    return all_classes

def get_builtin_event_classes():
    builtin_events_classes = [
        LibraryAboutToChangeEvent,
        LibraryChangedEvent,
        LocationSelectedEvent,
        DeviceConnectedEvent,
        DeviceDisconnectedEvent,
        BookListDoubleClickedEvent,
        BookListEnterKeyPressedEvent,
        CalibreInitializedEvent,
        TimerEvent,
        VLTabChangedEvent,
        ShuttingDownEvent
    ]
    return builtin_events_classes

def get_variant_classes(plugin_action):
    classes = []
    builtin_events_classes = get_builtin_event_classes()
    imported_objects = get_imported_event_objects(plugin_action)
    imported_classes = [x for x in imported_objects if inspect.isclass(x)]
    user_event_classes = get_user_event_classes(plugin_action)
    for cls in builtin_events_classes + imported_classes + list(user_event_classes.values()):
        if getattr(cls, 'support_variants', False):
            classes.append(cls)
    return classes

def get_all_events(plugin_action):
    builtin_events_classes = get_builtin_event_classes()

    # cache events signals to prevent duplication
    signals_cache = set()

    all_events = OrderedDict()
    builtin_events = OrderedDict()
    user_events = OrderedDict()
    imported_events = OrderedDict()
    
    imported_objects = get_imported_event_objects(plugin_action)

    user_event_classes = get_user_event_classes(plugin_action)

    # Order is important, we start with builtins first to register its signals before
    # imported and user events
    for event_cls in builtin_events_classes:
        add_event_and_variants(plugin_action, event_cls, builtin_events, signals_cache)

    # Note: imported events can either be class or instantiated object
    for event_name, event_obj in imported_objects.items():
        try:
            add_event_and_variants(plugin_action, event_obj, imported_events, signals_cache)
        except TypeError as e:
            # TypeError: issubclass() arg 1 must be a class
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported event: Un-reconized object: {event_obj}\n{traceback.format_exc()}')            
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported event: {event_name}\n{traceback.format_exc()}')

    for event_name, event_cls in user_event_classes.items():
        try:
            add_event_and_variants(plugin_action, event_cls, user_events, signals_cache)
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing user event: {event_name}\n{traceback.format_exc()}')

    # Order is important to prevent overwriting builtin events
    all_events = dict(list(user_events.items()) + list(imported_events.items()) + list(builtin_events.items()))
    all_events = reorder_dictionary(all_events,
                                    list(builtin_events.keys()) + list(imported_events.keys()) + list(user_events.keys()))
    return all_events, builtin_events, user_events, imported_events

