#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.ui import get_gui
from calibre.utils.filenames import make_long_path_useable
from calibre.utils.search_query_parser import ParseException

def get_last_modified_map(db, book_ids):
    return {book_id: db.new_api.field_for('last_modified', book_id) for book_id in book_ids }

def get_books_modified_since(db, reference_timestamp):
    last_modified_map = get_last_modified_map(db, db.new_api.all_book_ids())
    return [book_id for book_id, lm in last_modified_map.items() if ( lm and (lm > reference_timestamp) )]

def is_search_valid(db, search_text):
    # search in a vl of only one book for performance
    res = True
    try:
        book_id = db.all_ids()[0]
        #restriction = f'id:={book_id}'
        book_ids = set((db.all_ids()[0],))
    except:
        #restriction = ''
        book_ids = None
    try:
        #db.data.search_getting_ids(search_text, restriction)
        db.new_api.search(search_text, book_ids=book_ids)
    except ParseException:
        res = False
    return res

def is_saved_search_valid(db, saved_search_name):
    search_text = f'search:"={saved_search_name}"'
    return is_search_valid(db, search_text)

def is_vl_valid(db, vl_name):
    search_text = db.prefs['virtual_libraries'][vl_name]
    return is_search_valid(db, search_text)

def get_valid_vls(db):
    res = []
    vls = db.prefs.get('virtual_libraries', {})
    for vl in vls.keys():
        if is_vl_valid(db, vl):
            res.append(vl)
        else:
            if DEBUG:
                prints(f'Action Chains: virtual library "{vl}" is invalid')
    return res

def get_valid_saved_searches(db):
    res = []
    saved_searches = db.saved_search_names()
    for saved_search in saved_searches:
        if is_saved_search_valid(db, saved_search):
            res.append(saved_search)
        else:
            if DEBUG:
                prints(f'Action Chains: saved search "{saved_search}" is invalid')
    return res

def all_field_names(db, col_name):
    try:
        return list(db.new_api.all_field_names(col_name))
    except:
        return []

def set_marked_ids(plugin_action, marked_id_map):
    db = plugin_action.gui.current_db
    db.set_marked_ids(marked_id_map)
    plugin_action.marked_id_map = marked_id_map
    

def get_marked_id_map(plugin_action):
    '''
    get map of book_id: marked_text
    '''
    db = plugin_action.gui.current_db
    marked_ids = dict()
    marked_books = db.search_getting_ids('marked:true', None)
    #marked_books = db.data.marked_ids.copy()
    for book_id in marked_books:
        marked_id_map = db.data.get_marked(book_id)
        marked_ids[book_id] = marked_id_map
    return marked_ids

def set_marked_for_book(plugin_action, book_id, val):
    db = plugin_action.gui.current_db
    marked_id_map = get_marked_id_map(plugin_action)
    if not val:
        try:
            del marked_id_map[book_id]
        except KeyError:
            pass
    else:
        marked_id_map[book_id] = val
    set_marked_ids(plugin_action, marked_id_map)
    return set([book_id])

def all_marked_values(plugin_action, sep=','):
    vals = set()
    marked_id_map = get_marked_id_map(plugin_action)
    for book_id, text in marked_id_map.items():
        for item in text.split(sep):
            vals.add(item.strip())
    return list(vals)
        
def bulk_modify_marked(plugin_action, book_ids, add=[], remove=[], remove_all=[], sep=','):
    db = plugin_action.gui.current_db
    if not add and not remove and not remove_all:
        return set()
    changed = set()
    marked_id_map = get_marked_id_map(plugin_action)
    for book_id in book_ids:
        original_val = marked_id_map.get(book_id, '')
        if original_val:
            val = set([item.strip() for item in original_val.split(sep)])
        else:
            val = set()
        if remove:
            val = val.difference(set(remove))
        if remove_all:
            val = set()
        if add:
            val = val.union(set(add))
            
        val = sep.join(list(val))

        if val != original_val:
            changed.add(book_id)
        if val:
            marked_id_map[book_id] = val
        else:
            try:
                del marked_id_map[book_id]
            except KeyError:
                pass
            
    set_marked_ids(plugin_action, marked_id_map)
    return changed

def bulk_modify_multiple(db, book_ids, col_name, add=[], remove=[], remove_all=None):
    col_metadata = db.field_metadata.all_metadata()[col_name]
    if not col_metadata['is_multiple']:
        return set()
    if not add and not remove and not remove_all:
        return set()
    col_id_map = {}
    for book_id in book_ids:
        val = db.new_api.field_for(col_name, book_id)
        val = set(val)
        if remove:
            val = val.difference(set(remove))
        if remove_all:
            val = set()
        if add:
            val = val.union(set(add))
        col_id_map[book_id] = val
    return db.new_api.set_field(col_name, col_id_map)

def bulk_modify_identifiers(db, book_ids, add=[], remove=[], remove_all=None):
    if not add and not remove and not remove_all:
        return set()
    col_id_map = {}
    for book_id in book_ids:
        identifiers = db.new_api.field_for('identifiers', book_id)
        if remove:
            for id_type in remove:
                # empty strings '' would raise a KeyError
                if id_type:
                    try:
                        del identifiers[id_type]
                    except:
                        pass
        if remove_all:
            identifiers = {}
        if add:
            for csp in add:
                id_type, id_val = csp.split(':')
                identifiers[id_type] = id_val
        col_id_map[book_id] = identifiers
    return db.new_api.set_field('identifiers', col_id_map)

def add_cover_from_file(db, book_id, cover_path, print_error=False):
    gui = get_gui
    if cover_path:
        cover_path = make_long_path_useable(os.path.abspath(cover_path))
        if not os.access(cover_path, os.R_OK):
            msg = _('Cannot read')
            details = _('You do not have permission to read the file: ') + cover_path
            if print_error:
                error_dialog(gui, msg, details, show=True)
            else:
                if DEBUG:
                    prints(details)
            return
        cover = None
        try:
            with open(cover_path, "rb") as f:
                cover = f.read()
        except IOError as e:
            msg = _('Error reading file')
            details = _("<p>There was an error reading from file: <br /><b>") + cover_path + "</b></p><br />"+str(e)
            if print_error:
                error_dialog(gui, msg, details, show=True)
            else:
                if DEBUG:
                    prints(details)
        if cover:
            try:
                db.new_api.set_cover({book_id: cover})
            except:
                if DEBUG:
                    prints(f'Action Chains: Failed to set image from file: {cover_path}')
        else:
            if print_error:
                error_dialog(gui, msg, details, show=True)
            else:
                if DEBUG:
                    prints(details)
