#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict
import traceback
import operator

from qt.core import (Qt, QApplication, QLineEdit, QLabel, QPlainTextEdit, QCheckBox,
                     QComboBox, QStackedLayout, QDoubleSpinBox, QFrame)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2.dialogs.template_dialog import TemplateDialog
from calibre.gui2.widgets2 import DateTimeEdit
from calibre.utils.date import now, as_local_time, as_utc, internal_iso_format_string, parse_date

from calibre_plugins.action_chains.common_utils import qt_from_dt, qt_to_dt
from calibre_plugins.action_chains.gui import ImageComboBox
from calibre_plugins.action_chains.templates import check_template, get_metadata_object, TemplateFunction, TEMPLATE_ERROR, get_template_output
from calibre_plugins.action_chains.templates.dialogs import TemplateBox

try:
    load_translations()
except NameError:
    prints("ActionsChain/conditions.py - exception when loading translations")

COMPARISON_OPERATORS = OrderedDict()
for x, y in (
    ('=', operator.eq),
    ('>', operator.gt),
    ('>=', operator.ge),
    ('<', operator.lt),
    ('<=', operator.le),
    ('!=', operator.ne) ):
    COMPARISON_OPERATORS[x] = y

TYPES = OrderedDict()
for x, y in (
    ('text', str),
    ('number', float),
    ('datetime', parse_date) ):
    TYPES[x] = y

def check_conditions(plugin_action, condition_settings, mi=None, template_functions={}, global_vars={}):
    if not mi:
        mi = get_metadata_object(plugin_action.gui)
    if not template_functions:
        template_functions = plugin_action.template_functions
    datatype = condition_settings['datatype']
    convert = TYPES[datatype]
    cmp_type = condition_settings['cmp_type']
    cmp_operator = COMPARISON_OPERATORS[cmp_type]
    value = condition_settings['value']
    template = condition_settings['template']
    template_output = get_template_output(template, mi, TEMPLATE_ERROR, mi,
                                          global_vars=global_vars, template_functions=template_functions)
    if template_output.startswith(TEMPLATE_ERROR):
        return False
    try:
        template_output = convert(template_output)
    except Exception as e:
        if DEBUG:
            prints(f'Action Chains: Error converting template output ({template_output}) | value ({value}) to type: {datatype}\n    template: {template}')
        return False
    if cmp_operator(template_output, value):
        return True
    else:
        return False

class ConditionsEval(TemplateBox):

    prefix_chain = _('Chain')
    prefix_action = _('Action')
    placeholder=_(
        'will run only if the result of the template entered '
        'here matches the value in the condition Value field below.')

    def __init__(self, parent, plugin_action, settings={}, global_vars={}, source='chain', mi=None):
        
        prefix = self.prefix_chain if source == 'chain' else self.prefix_action
        self.placeholder_text = prefix + ' ' + self.placeholder
        
        TemplateBox.__init__(
            self,
            parent,
            plugin_action,
            template_text='',
            placeholder_text=self.placeholder_text,
            mi=mi,
            global_vars=global_vars
        )

        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.setWindowTitle('Evaluate Condition')

        self.type_box = QComboBox(self)
        self.type_box.activated.connect(self.switch_editor)

        self.comparison_box = QComboBox(self)
        
        self.text_editor = QLineEdit(self)
        self.number_editor = QDoubleSpinBox(self)
        self.number_editor.setMinimum(-100000)
        self.number_editor.setMaximum(100000)
        self.datetime_editor = DateTimeEdit(self)
        self.datetime_editor.setDateTime(qt_from_dt(now()))
        self.sl = QStackedLayout()
        self.sl.addWidget(self.text_editor)
        self.sl.addWidget(self.number_editor)
        self.sl.addWidget(self.datetime_editor)

        self.menu_chk = QCheckBox(_('Disable menu item for chain if conditions are not met'))
        self.menu_chk.setChecked(False)
        
        self.action_validation_chk = QCheckBox(_('Disable action validation if conditions are not met'))
        self.action_validation_chk.setChecked(False)

        self.icon_combobox = ImageComboBox(self, self.plugin_action.icon_cache)
        self.tooltip_box = QLineEdit(self)

        self.help_label = QLabel(self)
        self.help_label.setWordWrap(True)
        self.help_label.setOpenExternalLinks(True)
        self.help_label.setObjectName("heading")
        self.help_label.setText('<a href="https://www.mobileread.com/forums/showpost.php?p=4073182&postcount=172">Conditions evaluator help</a>')
        self.help_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
        self.help_label.setAlignment(Qt.AlignLeft)

        self.user_label_1.setVisible(True)
        self.user_label_1.setText(_('Datatype'))
        self.user_layout_1.addWidget(self.type_box)

        self.user_label_2.setVisible(True)
        self.user_label_2.setText(_('Comparison'))
        self.user_layout_2.addWidget(self.comparison_box)

        self.user_label_3.setVisible(True)
        self.user_label_3.setText(_('Condition Value'))
        self.user_layout_3.addLayout(self.sl)

        self.user_label_4.setVisible(True)
        self.user_label_4.setText(_('Icon'))
        self.user_label_4.setToolTip(_('Icon chosen here will replace the default icon displayed for items with condtions'))
        self.user_layout_4.addWidget(self.icon_combobox)

        self.user_label_5.setVisible(True)
        self.user_label_5.setText(_('Tooltip'))
        self.user_label_5.setToolTip(_('Set the tooltip to be displayed when hovering on the conditions\'s icon'))
        self.user_layout_5.addWidget(self.tooltip_box)

        self.user_layout_6.addWidget(self.menu_chk)
        self.user_layout_6.addWidget(self.action_validation_chk)
        self.user_layout_6.addWidget(self.help_label)

        if source == 'chain':
            self.action_validation_chk.hide()
        else:
            self.menu_chk.hide()
        
        self.comparison_box.addItems(COMPARISON_OPERATORS.keys())
        self.comparison_box.setCurrentText('=')

        self.type_box.addItems(TYPES.keys())
        self.type_box.setCurrentText('text')
        
        self.load_settings(settings)

    def load_settings(self, settings):
        template = settings.get('template', '')
        cmp_type = settings.get('cmp_type', '=')
        datatype = settings.get('datatype', 'text')
        value = settings.get('value', '')
        icon = settings.get('icon', '')
        tooltip = settings.get('tooltip', '')
        self.textbox.insertPlainText(template)
        self.type_box.setCurrentText(datatype)
        self.comparison_box.setCurrentText(cmp_type)
        self.set_editor_value(value)
        self.icon_combobox.set_current_text(icon)
        self.tooltip_box.setText(tooltip)
        self.menu_chk.setChecked(settings.get('affect_menu', False))
        self.action_validation_chk.setChecked(settings.get('affect_validation', False))        
        if not template:
            self.textbox.setPlaceholderText(self.placeholder_text)
            self.textbox.clear()
        self.switch_editor()
        self.textbox_changed()

    def switch_editor(self):
        self.sl.setCurrentIndex(self.type_box.currentIndex())

    def editor_value(self):
        editor_type = self.type_box.currentText()
        if editor_type == 'text':
            return self.text_editor.text()
        elif editor_type == 'number':
            return self.number_editor.value()
        elif editor_type == 'datetime':
            value = self.datetime_editor.dateTime()
            # we need a json serializable format
            value = qt_to_dt(value)
            value = as_utc(value)
            return value

    def set_editor_value(self, value):
        editor_type = self.type_box.currentText()
        if editor_type == 'text':
            self.text_editor.setText(value)
        elif editor_type == 'number':
            self.number_editor.setValue(value)
        elif editor_type == 'datetime':
            value = as_local_time(value)
            return self.datetime_editor.setDateTime(qt_from_dt(value))

    def run_check(self):
        conditions_settings = {
            'datatype': self.type_box.currentText(),
            'cmp_type': self.comparison_box.currentText(),
            'value': self.editor_value(),
            'template': self.textbox.toPlainText()
        }
        return check_conditions(self.plugin_action,
                                conditions_settings,
                                mi=self.mi,
                                template_functions=self.all_functions,
                                global_vars=self.global_vars)

    def accept(self):
        self.template = str(self.textbox.toPlainText()).rstrip()
        chk = check_template(self.template, self.plugin_action)
        self.value = self.editor_value()
        self.datatype = self.type_box.currentText()
        self.cmp_type = self.comparison_box.currentText()
        self.icon = self.icon_combobox.currentText()
        self.tooltip = self.tooltip_box.text()
        self.affect_menu = self.menu_chk.isChecked()
        self.affect_validation = self.action_validation_chk.isChecked()
        if chk is True:
            # accept and save_geometry
            TemplateDialog.accept(self)


